/*
 *  linux/arch/arm/mach-pxa/dma.c
 *
 *  PXA DMA registration and IRQ dispatching
 *
 *  Author:	Nicolas Pitre
 *  Created:	Nov 15, 2001
 *  Copyright:	MontaVista Software Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>

#include <asm/system.h>
#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/dma.h>


static struct dma_channel {
	char *name;
	void (*irq_handler)(int, void *, struct pt_regs *);
	void *data;
} dma_channels[16];


int pxa_request_dma (char *name, pxa_dma_prio prio,
			 void (*irq_handler)(int, void *, struct pt_regs *),
		 	 void *data)
{
	unsigned long flags;
	int i, found = 0;

	/* basic sanity checks */
	if (!name || !irq_handler)
		return -EINVAL;

	local_irq_save(flags);

	/* try grabbing a DMA channel with the requested priority */
	for (i = prio; i < prio + (prio == DMA_PRIO_LOW) ? 8 : 4; i++) {
		if (!dma_channels[i].name) {
			found = 1;
			break;
		}
	}

	if (!found) {
		/* requested prio group is full, try hier priorities */
		for (i = prio-1; i >= 0; i--) {
			if (!dma_channels[i].name) {
				found = 1;
				break;
			}
		}
	}
		
	if (found) {
		DCSR(i) = DCSR_STARTINTR|DCSR_ENDINTR|DCSR_BUSERR;
		dma_channels[i].name = name;
		dma_channels[i].irq_handler = irq_handler;
		dma_channels[i].data = data;
	} else {
		printk (KERN_WARNING "No more available DMA channels for %s\n", name);
		i = -ENODEV;
	}

	local_irq_restore(flags);
	return i;
}

void pxa_free_dma (int dma_ch)
{
	unsigned long flags;

	if (!dma_channels[dma_ch].name) {
		printk (KERN_CRIT 
			"%s: trying to free channel %d which is already freed\n", __FUNCTION__,
			dma_ch);
		return;
	}

	local_irq_save(flags);
	DCSR(dma_ch) = DCSR_STARTINTR|DCSR_ENDINTR|DCSR_BUSERR;
	dma_channels[dma_ch].name = NULL;
	local_irq_restore(flags);
}

static void dma_irq_handler(int irq, void *dev_id, struct pt_regs *regs)
{
	int i, dint = DINT;

	for (i = 0; i < 16; i++) {
		if (dint & (1 << i)) {
			struct dma_channel *channel = &dma_channels[i];
			if (channel->name && channel->irq_handler) {
				channel->irq_handler(i, channel->data, regs);
			} else {
				/*
				 * IRQ for an unregistered DMA channel:
				 * let's clear the interrupts and disable it.
				 */
				printk (KERN_WARNING "spurious IRQ for DMA channel %d\n", i);
				DCSR(i) = DCSR_STARTINTR|DCSR_ENDINTR|DCSR_BUSERR;
			}
		}
	}
}

static int __init pxa_dma_init (void)
{
	int ret;

	ret = request_irq (IRQ_DMA, dma_irq_handler, 0, "DMA", NULL);
	if (ret)
		printk (KERN_CRIT "Wow!  Can't register IRQ for DMA\n");
	return ret;
}

__initcall(pxa_dma_init);

EXPORT_SYMBOL(pxa_request_dma);
EXPORT_SYMBOL(pxa_free_dma);

#ifdef CONFIG_PM
/* Drivers should call this from their PM callback function */

int pxa_dma_sleep(dmach_t channel)
{
#if 0
        pxa_dma_t *dma = &dma_chan[channel];
	int orig_state;

	if ((unsigned)channel >= MAX_PXA_DMA_CHANNELS || !dma->in_use)
		return -EINVAL;

	orig_state = dma->stopped;
	pxa_dma_stop(channel);
	dma->regs->ClrDCSR = DCSR_RUN | DCSR_IE | DCSR_STRTA | DCSR_STRTB;
	dma->stopped = orig_state;
	dma->spin_ref = 0;
#endif
	return 0;
}

int pxa_dma_wakeup(dmach_t channel)
{
#if 0
        pxa_dma_t *dma = &dma_chan[channel];
	dma_regs_t *regs;
	int flags;

	if ((unsigned)channel >= MAX_PXA_DMA_CHANNELS || !dma->in_use)
		return -EINVAL;

	regs = dma->regs;
	regs->ClrDCSR =
		(DCSR_DONEA | DCSR_DONEB | DCSR_STRTA | DCSR_STRTB |
		 DCSR_IE | DCSR_ERROR | DCSR_RUN);
	regs->DDAR = dma->device;
	local_irq_save(flags);
	process_dma(dma);
	local_irq_restore(flags);
#endif
	return 0;
}

EXPORT_SYMBOL(pxa_dma_sleep);
EXPORT_SYMBOL(pxa_dma_wakeup);

#endif
