/*
* Driver interface to the ASIC Companion chip on the iPAQ H5400
*
* Copyright © 2003 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author:  Keith Packard <keith.packard@hp.com>
*          May 2003
*/

#include <linux/module.h>
#include <linux/version.h>
#include <linux/config.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/mtd/mtd.h>
#include <linux/ctype.h>
#include <linux/delay.h>

#include <asm/arch/hardware.h>
#include <asm/arch-sa1100/h3600_hal.h>
#include <asm/irq.h>
#include <asm/io.h>

#include <asm/arch-pxa/h5400-gpio.h>
#include <asm/arch-pxa/h5400-asic.h>
#include <asm/arch-pxa/h5400-irqs.h>
#include "asm/arch-sa1100/ipaq-mtd-asset.h"
#include "h5400_asic_io.h"
#include "../../../drivers/video/mq1100.h"


// Define this to see all suspend/resume init/cleanup messages
#define DEBUG_INIT()  \
        if (0) printk(KERN_NOTICE "enter %s\n", __FUNCTION__)
#define DEBUG_FINI() \
        if (0) printk(KERN_NOTICE "leave %s\n", __FUNCTION__)
#define DEBUG_OUT(f,a...) \
	if (0) printk(KERN_NOTICE "%s:" f, __FUNCTION__, ##a)
#define DEBUG_ISR_INIT()  \
        if (0) printk(KERN_NOTICE "enter %s\n", __FUNCTION__)
#define DEBUG_ISR_FINI() \
        if (0) printk(KERN_NOTICE "leave %s\n", __FUNCTION__)
#define DEBUG_ISR_OUT(f,a...) \
	if (0) printk(KERN_NOTICE "%s:" f, __FUNCTION__, ##a)

#define PDEBUG(format,arg...) printk(KERN_DEBUG __FILE__ ":%s - " format "\n", __FUNCTION__, ## arg)
#define PALERT(format,arg...) printk(KERN_ALERT __FILE__ ":%s - " format "\n", __FUNCTION__, ## arg)
#define PERROR(format,arg...) printk(KERN_ERR __FILE__ ":%s - " format "\n", __FUNCTION__, ## arg)

#define ADC_DATA_MASK	0x3ff


/***********************************************************************************
 *   ADC support (stolen from the H3900 code)
 ***********************************************************************************/

enum adc_state {
	ADC_STATE_IDLE,
	ADC_STATE_TOUCHSCREEN,  // Servicing the touchscreen
	ADC_STATE_USER          // Servicing a user-level request
};

static struct adc_data {
	enum adc_state     state;
	struct semaphore   lock;      // Mutex for access from user-level
	wait_queue_head_t  waitq;     // Waitq for user-level access (waits for interrupt service)
	int                last;      // Return value for user-level acces
	int                user_mux;  // Requested mux for the next user read
	int                ts_mux;    // Requested mux for the next touchscreen read
	int              (*ts_callback)(int);  // Touchscreen callback
	unsigned long      shared;    // Shared resources
} g_adcdev;

enum touchscreen_state {
	TS_STATE_WAIT_PEN_DOWN,   // Waiting for a PEN interrupt
	TS_STATE_ACTIVE_SAMPLING  // Actively sampling ADC
};

static struct touchscreen_data {
	enum touchscreen_state state;
	struct timer_list      timer;
} g_touch;

static void h5400_asic_touchscreen_next_record( struct touchscreen_data *touch );

#define ADC_SAMPLE_PERIOD	10  /* Sample every 10 milliseconds */
static unsigned long adc_sample_period   = ADC_SAMPLE_PERIOD;
static unsigned long adc_spins_max       = 1000;
static unsigned long touchscreen_delay   = 0;
static unsigned int  clock_multiplier    = 2;
static unsigned long adc_prescaler       = 19;
static unsigned long adc_delay           = 500;
static unsigned long adc_pen_delay       = 1000;
static unsigned long adc_check_srcpnd    = 1;
static unsigned long adc_interrupt_mode  = 0;

MODULE_PARM(adc_check_srcpnd, "i");
MODULE_PARM(adc_sample_period, "i");
MODULE_PARM(adc_spins_max, "i");
MODULE_PARM(adc_delay, "i");
MODULE_PARM(adc_pen_delay, "i");
MODULE_PARM(touchscreen_delay, "i");
MODULE_PARM(adc_interrupt_mode, "i");

/* Jamey says: 
 *
 * initialization:
 * adc control: enable prescaler, use prescaler value 19, sel=3 (ain3?)
 * use adc delay=1000
 * touchscreen control: initialize to 0xd3
 *  H5400_ASIC_ADC_TSC_XYPST_INTERRUPT, H5400_ASIC_ADC_TSC_XP_SEN, 
 *  H5400_ASIC_ADC_TSC_YP_SEN, H5400_ASIC_ADC_TSC_YM_SEN
 */

#define ASIC_ADC_CONTROL_INIT	(H5400_ASIC_ADC_CONTROL_PRESCALER_ENABLE | \
				 ((clock_multiplier * adc_prescaler) << H5400_ASIC_ADC_CONTROL_PRESCALER_SHIFT) | \
				 H5400_ASIC_ADC_CONTROL_SEL_AIN3)

#define ASIC_ADC_DELAY		(clock_multiplier * adc_delay)

#define ASIC_ADC_TSC_WAIT	(H5400_ASIC_ADC_TSC_XYPST_INTERRUPT | \
				 H5400_ASIC_ADC_TSC_XP_SEN | \
				 H5400_ASIC_ADC_TSC_YP_SEN | \
				 H5400_ASIC_ADC_TSC_YM_SEN)
/* Jamey says:
 * when getting touch coordinage,
 *  - set adctsc to ((1<<3)|(1<<2)) -- pull up disable, auto mode
 *  - set bit 0 of adc control
 *  - set back to 0xd3 after getting coordinate
 */

#define ASIC_ADC_TSC_SAMPLE	(H5400_ASIC_ADC_TSC_AUTO_MODE | \
				 H5400_ASIC_ADC_TSC_PULL_UP)

#define ASIC_ADC_CONTROL_SAMPLE	(ASIC_ADC_CONTROL_INIT|H5400_ASIC_ADC_CONTROL_FORCE_START)

static void h5400_asic_touchscreen_start_record (struct touchscreen_data *);
static void h5400_asic_touchscreen_record (struct touchscreen_data *);

static void h5400_asic_adc_isr( int irq, void *dev_id, struct pt_regs *regs )
{
	DEBUG_ISR_INIT ();

	h5400_asic_touchscreen_start_record( &g_touch );

	DEBUG_ISR_FINI ();
}

static void h5400_asic_interrupt_mode ( void )
{
	DEBUG_ISR_INIT ();

	h5400_asic_write_register (H5400_ASIC_ADC_Delay, adc_pen_delay);
	h5400_asic_write_register (H5400_ASIC_ADC_Control, ASIC_ADC_CONTROL_INIT);
	h5400_asic_write_register (H5400_ASIC_ADC_TouchScreenControl, ASIC_ADC_TSC_WAIT);

	DEBUG_ISR_FINI ();
}

static void h5400_asic_adc_up( struct adc_data *adc )
{
	int half_clk;
	DEBUG_INIT ();
	
	/* Enable the ADC clock */
	h5400_asic_clock_enable (H5400_ASIC_CPM_CLKCON_ADC_CLKEN, 1);
	half_clk = h5400_asic_read_register (H5400_ASIC_CPM_ClockSleep) & H5400_ASIC_CPM_CLKSLEEP_HALF_CLK;
	if (half_clk) {
		clock_multiplier = 2;
	} else {
		clock_multiplier = 1;
	}

	h5400_asic_interrupt_mode ();
	
	DEBUG_FINI ();
}

static void h5400_asic_adc_down( struct adc_data *adc )
{
	DEBUG_INIT();
	
	/* Place the ADC in standby mode */
	DEBUG_OUT ("ADC_Control: %X\n", H5400_ASIC_ADC_Control);
	H5400_ASIC_SET_BIT (H5400_ASIC_ADC_Control, H5400_ASIC_ADC_CONTROL_STANDBY);
	DEBUG_OUT ("Write ADC_Control: %X\n", H5400_ASIC_ADC_Control);

	/* Disable the ADC clock */
	h5400_asic_clock_enable (H5400_ASIC_CPM_CLKCON_ADC_CLKEN, 0);
	
	if (adc->ts_mux != 0) {
		// Clear any current touchscreen requests
		adc->ts_mux = 0;
	}
	adc->state  = ADC_STATE_IDLE;
	DEBUG_FINI ();
}

int h5400_asic_adc_suspend( void )
{
	DEBUG_INIT();
	down(&g_adcdev.lock);  // No interruptions
	h5400_asic_adc_down( &g_adcdev );
	disable_irq( IRQ_H5400_ADCTS );
	DEBUG_FINI ();
	return 0;
}

void h5400_asic_adc_resume( void )
{
	DEBUG_INIT();
	enable_irq( IRQ_H5400_ADCTS );
	h5400_asic_adc_up( &g_adcdev );
	up(&g_adcdev.lock);
	DEBUG_FINI();
}

int h5400_asic_adc_init( void )
{
	int result;

	DEBUG_INIT();
	init_MUTEX(&g_adcdev.lock);
	init_waitqueue_head( &g_adcdev.waitq );

	result = request_irq(IRQ_H5400_ADCTS, h5400_asic_adc_isr, SA_SAMPLE_RANDOM, "h5400_adcts", NULL );

	if ( result )
	{
		printk(KERN_CRIT "%s: unable to grab ADCTS IRQ %d error=%d\n", __FUNCTION__, 
		       IRQ_H5400_ADC, result);
		return result;
	}

	h5400_asic_adc_up( &g_adcdev );

	DEBUG_FINI ();

	return result;
}

void h5400_asic_adc_cleanup( void )
{
	DEBUG_INIT();
	h5400_asic_adc_down( &g_adcdev );
	free_irq( IRQ_H5400_ADCTS, NULL );
	DEBUG_FINI ();
}

	
/***********************************************************************************
 *   Touchscreen support (stolen from the H3900 code)
 ***********************************************************************************/
	
/* Ask for the next sample */
static void h5400_asic_adc_start_touchscreen( struct touchscreen_data *touch )
{
	u32     spins = 0;
	u16	TouchScreenControl;
	u16	Control;

	DEBUG_ISR_INIT ();

	h5400_asic_write_register (H5400_ASIC_ADC_Delay, adc_delay);

	TouchScreenControl = ASIC_ADC_TSC_SAMPLE;
	if (adc_interrupt_mode) {
	  TouchScreenControl |= H5400_ASIC_ADC_TSC_XYPST_INTERRUPT;
	}

	h5400_asic_write_register (H5400_ASIC_IC_SRCPND, IRQ_H5400_ASIC_ADC_MASK);

	DEBUG_ISR_OUT( "Write ADC_TouchScreenControl: %X\n", TouchScreenControl );
	h5400_asic_write_register (H5400_ASIC_ADC_TouchScreenControl, TouchScreenControl);

	Control = ASIC_ADC_CONTROL_SAMPLE;
	DEBUG_ISR_OUT( "Write ADC_Control: %X\n", Control );
	h5400_asic_write_register (H5400_ASIC_ADC_Control, Control);
	DEBUG_ISR_FINI ();

	if (1) {
		if (adc_check_srcpnd)
			while (((h5400_asic_read_register (H5400_ASIC_IC_SRCPND) & (IRQ_H5400_ASIC_ADC_MASK)) == 0)
			       && (spins++ < adc_spins_max)) {
				/* wait for the interrupt that occurs when the ADC is complete */
			}
		else
			while (((h5400_asic_read_register (H5400_ASIC_ADC_Control) & H5400_ASIC_ADC_CONTROL_CONVERSION_END) == 0)
			       && (spins++ < adc_spins_max)) {
				/* wait for the interrupt that occurs when the ADC is complete */
			}
	}
	if (h5400_asic_read_register (H5400_ASIC_ADC_Control) & H5400_ASIC_ADC_CONTROL_CONVERSION_END) {
		h5400_asic_touchscreen_record (touch);
	} else {
		if (spins > 0)
			printk("touchscreen adc timed out: adc_control=%08lx tsc_control=%08lx spin=%d\n",
			       h5400_asic_read_register (H5400_ASIC_ADC_Control),
			       h5400_asic_read_register (H5400_ASIC_ADC_TouchScreenControl),
			       spins);
		/* try again */
		h5400_asic_touchscreen_next_record( touch );
	}
}

/* Pen is up, stop sample collection process */
static void h5400_asic_touchscreen_stop_record( struct touchscreen_data *touch )
{
	DEBUG_ISR_INIT ();
	touch->state = TS_STATE_WAIT_PEN_DOWN;
	h5400_asic_interrupt_mode ();
	DEBUG_ISR_FINI ();
}

/* Pen is down, start sample collection process */
static void h5400_asic_touchscreen_start_record( struct touchscreen_data *touch )
{
	DEBUG_ISR_INIT ();
	if (touch->state == TS_STATE_WAIT_PEN_DOWN)
	{
		touch->state = TS_STATE_ACTIVE_SAMPLING;

		h5400_asic_adc_start_touchscreen( touch );		
	}
	DEBUG_ISR_FINI ();
}


static void h5400_asic_touchscreen_timer_callback( unsigned long nr )
{
	DEBUG_INIT ();
	/* ask for another sample */
	h5400_asic_adc_start_touchscreen (&g_touch);
	DEBUG_FINI ();
}

static void h5400_asic_touchscreen_next_record( struct touchscreen_data *touch )
{
	unsigned long inc = (adc_sample_period * HZ) / 1000;
	if (!inc) inc = 1;

	/* 
	 * Set touch screen control back to 0xd3 so that the next
	 * sample will have the PEN_UP bit set correctly
	 */
	h5400_asic_write_register (H5400_ASIC_ADC_TouchScreenControl, ASIC_ADC_TSC_WAIT);
	h5400_asic_write_register (H5400_ASIC_ADC_Delay, adc_delay);

	/* queue the timer to get another sample */
	mod_timer (&touch->timer, jiffies + inc);
}

/* Read a sample, then queue a timer if the pen is still down */
static void h5400_asic_touchscreen_record( struct touchscreen_data *touch )
{
	u16	Data0, Data1;
	int	x, y;

	DEBUG_ISR_INIT ();
	
	if (touchscreen_delay)
		mdelay(touchscreen_delay);
	Data0 = h5400_asic_read_register (H5400_ASIC_ADC_Data0);
	Data1 = h5400_asic_read_register (H5400_ASIC_ADC_Data1);

	if (Data0 & H5400_ASIC_ADC_DATA0_PEN_UP) {
		DEBUG_ISR_OUT ("pen up\n");
		h3600_hal_touchpanel( 0, 0, 0 );
		h5400_asic_touchscreen_stop_record( touch );
	} else {
		x = Data0 & ADC_DATA_MASK;	/* flip X axis */
		y = Data1 & ADC_DATA_MASK;	/* flip Y axis */
		DEBUG_ISR_OUT ("pen down %d, %d\n", x, y);
		h3600_hal_touchpanel( x, y, 1 );
		h5400_asic_touchscreen_next_record( touch );
	}
	DEBUG_ISR_FINI ();
}
	
int  h5400_asic_touchscreen_init(void)
{
	DEBUG_INIT ();
	printk("h5400_touchscreen_init: adc_sample_period=%ld adc_spins_max=%ld adc_pen_delay=%ld adc_delay=%ld touchscreen_delay=%ld\n",
	       adc_sample_period, adc_spins_max, adc_pen_delay, adc_delay, touchscreen_delay);
	g_touch.timer.function = h5400_asic_touchscreen_timer_callback;
	g_touch.timer.data = (unsigned long) NULL;
	g_touch.state = TS_STATE_WAIT_PEN_DOWN;
	init_timer (&g_touch.timer);
	DEBUG_FINI();
	return 0;
}

void h5400_asic_touchscreen_cleanup(void)
{
	DEBUG_INIT ();
	del_timer_sync (&g_touch.timer);
	DEBUG_FINI ();
}

int  h5400_asic_touchscreen_suspend(void)
{
	DEBUG_INIT ();
	/* XXX */
	DEBUG_FINI ();
	return 0;
}

void h5400_asic_touchscreen_resume(void)
{
	DEBUG_INIT ();
	/* XXX */
	g_touch.state = TS_STATE_WAIT_PEN_DOWN;
	DEBUG_FINI ();
}



/***********************************************************************************
 *   Front panel button support
 ***********************************************************************************/

struct h5400_button_data
{
	int irq;
	int keycode;

	unsigned long bitmask;
};

static struct h5400_button_data h5400_buttons[] = {
	{ IRQ_H5400_ASIC_BUTTON_1,  H3600_KEYCODE_CALENDAR,	1 << 0 },
	{ IRQ_H5400_ASIC_BUTTON_2,  H3600_KEYCODE_CONTACTS,	1 << 1 },
	{ IRQ_H5400_ASIC_BUTTON_3,  H3600_KEYCODE_Q,		1 << 2 },
	{ IRQ_H5400_ASIC_BUTTON_4,  H3600_KEYCODE_START,	1 << 3 },
	{ IRQ_H5400_GPIO_JOYSTICK1, H3600_KEYCODE_UP,		1 << 4 },
	{ IRQ_H5400_GPIO_JOYSTICK2, H3600_KEYCODE_UP,		1 << 5 },
	{ IRQ_H5400_GPIO_JOYSTICK3, H3600_KEYCODE_UP,		1 << 6 },
	{ IRQ_H5400_GPIO_JOYSTICK4, H3600_KEYCODE_UP,		1 << 7 },
	{ IRQ_H5400_GPIO_JOYSTICK5, H3600_KEYCODE_ACTION,	1 << 8 }
};

static void
h5400_key_isr (int isr, void *dev_id, struct pt_regs *regs)
{
	struct h5400_button_data *b = (struct h5400_button_data *)dev_id;
	int down;

	down = (h5400_asic_read_register (H5400_ASIC_GPIO_GPA_DAT) & b->bitmask) ? 0 : 1;

	h3600_hal_keypress (H3600_MAKEKEY (b->keycode, down));
}

int
h5400_asic_key_init (void)
{
	int i;

	DEBUG_INIT();

	/* set all buttons to edge trigger, both edges, filter on */
	h5400_asic_write_register (H5400_ASIC_GPIO_INT1, 0xffffffff);
	h5400_asic_write_register (H5400_ASIC_GPIO_INT2, 0x7777ffff);

	/* IRQs in ENINT1 are masked in the interrupt control block.
	   Enable them permanently in the GPIO block.  */
	h5400_asic_write_register (H5400_ASIC_GPIO_ENINT1, 0x1ff);

	for (i = 0; i < ARRAY_SIZE (h5400_buttons); i++) {
		request_irq (h5400_buttons[i].irq, h5400_key_isr,
			     SA_SAMPLE_RANDOM, "h5400 buttons",
			     &h5400_buttons[i]);
	}

	DEBUG_FINI();

	return 0;
}

void
h5400_asic_key_cleanup (void)
{
	int i;

	for (i = 0; i < ARRAY_SIZE (h5400_buttons); i++) {
		free_irq (h5400_buttons[i].irq, &h5400_buttons[i]);
	}
}

/***********************************************************************************
 *   Backlight
 *
 ***********************************************************************************/

int h5400_asic_backlight_control (enum flite_pwr power, unsigned char level)
{
	if (0) PDEBUG("power=%d level=%d", power, level);

	mq1100_backlight_level(level);
	
	switch (power) {
	case FLITE_PWR_OFF:
		SET_H5400_ASIC_GPIO (GPB, BACKLIGHT_POWER_ON, 0);
		break;
	case FLITE_PWR_ON:
		SET_H5400_ASIC_GPIO (GPB, BACKLIGHT_POWER_ON, 1);
		break;
	}
	return 0;
}

int h5400_asic_backlight_init (void)
{
	return 0;
}

void h5400_asic_backlight_cleanup (void)
{
	h5400_asic_backlight_control (FLITE_PWR_OFF, 0);
}
