/*
* Driver interface to the ASIC Companion chip on the iPAQ H1900
* Based off of said interface for the iPAQ H5400
*
* Copyright 2003 Joshua Wise
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* Author:  Joshua Wise <joshua at joshuawise.com>
*          June 2003
*/

#include <linux/module.h>
#include <linux/version.h>
#include <linux/config.h>

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/interrupt.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/sysctl.h>
#include <linux/proc_fs.h>
#include <linux/delay.h>

#include <asm/arch/hardware.h>
#include <asm/arch-sa1100/h3600_hal.h>
#include <asm/irq.h>

#include <asm/mach/irq.h>
#include <asm/arch-pxa/h1900-gpio.h>
#include <asm/hardware/ipaq-asic3.h>

#define H3600_ASIC_PROC_DIR     "asic"
#define H3600_ASIC_PROC_STATS   "stats"
#define REG_DIRNAME "registers"

MODULE_AUTHOR("Joshua Wise");
MODULE_DESCRIPTION("Hardware abstraction layer for the iPAQ H1900");

#ifndef MODULE /* hack so we can compile this into the kernel */
static struct module __this_module;
#undef THIS_MODULE
#define THIS_MODULE		(&__this_module)
#endif

/* Statistics */
//struct asic_statistics g_h3600_asic_statistics;

/***********************************************************************************/
/*      Standard entry points for HAL requests                                     */
/***********************************************************************************/

int h1900_asic_get_version( struct h3600_ts_version *result )
{
	return -EINVAL;
}

int h1900_asic_eeprom_read( unsigned short address, unsigned char *data, unsigned short len )
{
	return -EINVAL;
}

int h1900_asic_eeprom_write( unsigned short address, unsigned char *data, unsigned short len )
{
	return -EINVAL;
}

int h1900_asic_get_thermal_sensor( unsigned short *result )
{
	return -EINVAL;
}

int h1900_asic_set_notify_led( unsigned char mode, unsigned char duration, 
			       unsigned char ontime, unsigned char offtime )
{
	return -EINVAL;
}

int h1900_asic_read_light_sensor( unsigned char *result )
{
	return -EINVAL;
}

int h1900_asic_spi_read( unsigned short address, unsigned char *data, unsigned short len )
{
	return -EINVAL;
}

int h1900_asic_spi_write( unsigned short address, unsigned char *data, unsigned short len )
{
	return -EINVAL;
}


int h1900_asic_codec_control( unsigned char command, unsigned char level)
{
	return -EINVAL;
}

int h1900_asic_get_option_detect( int *result )
{
	return -EINVAL;
}

int h1900_asic_audio_clock( long samplerate )
{
	return -EINVAL;
}

int h1900_asic_audio_power( long samplerate )
{
	return -EINVAL;
}

int h1900_asic_audio_mute( int mute )
{
	return -EINVAL;
}

int h1900_asic_set_ebat( void )
{
	return -EINVAL;
}

int h1900_asic_battery_read( struct h3600_battery *query )
{
	query->battery_count=0;
	return 0;
};

static struct h3600_hal_ops h1900_asic_ops = {
	get_version         : h1900_asic_get_version,
	eeprom_read         : h1900_asic_eeprom_read,
	eeprom_write        : h1900_asic_eeprom_write,
	get_thermal_sensor  : h1900_asic_get_thermal_sensor,
	set_notify_led      : h1900_asic_set_notify_led,
	read_light_sensor   : h1900_asic_read_light_sensor,
	get_battery         : h1900_asic_battery_read,
	spi_read            : h1900_asic_spi_read,
	spi_write           : h1900_asic_spi_write,
	get_option_detect   : h1900_asic_get_option_detect,
	audio_clock         : h1900_asic_audio_clock,
	audio_power         : h1900_asic_audio_power,
	audio_mute          : h1900_asic_audio_mute,
#if 0
	backlight_control   : h1900_asic_backlight_control,
#endif
	asset_read          : NULL,
	set_ebat            : h1900_asic_set_ebat,
        owner               : THIS_MODULE,
};

static struct proc_dir_entry   *asic_proc_dir;

struct simple_proc_entry {
	char *        name;
	read_proc_t * read_proc;
};

const struct simple_proc_entry sproc_list[] = {
};

static int __init h1900_asic_register_procfs( void )
{
	int i;

	asic_proc_dir = proc_mkdir(H3600_ASIC_PROC_DIR, NULL);
	if ( !asic_proc_dir ) {
		printk(KERN_ALERT  
		       "%s: unable to create proc entry %s\n", __FUNCTION__, H3600_ASIC_PROC_DIR);
		return -ENOMEM;
	}

	for ( i = 0 ; i < ARRAY_SIZE(sproc_list) ; i++ )
		create_proc_read_entry(sproc_list[i].name, 0, asic_proc_dir, 
				       sproc_list[i].read_proc, NULL);

	return 0;
}

static void  h1900_asic_unregister_procfs( void )
{
	int i;
	if ( asic_proc_dir ) {
		for (i=0 ; i<ARRAY_SIZE(sproc_list) ; i++ )
			remove_proc_entry(sproc_list[i].name, asic_proc_dir);

		remove_proc_entry(H3600_ASIC_PROC_DIR, NULL );
		asic_proc_dir = NULL;
	}
}



#define MAKEKEY(index, down)  ((down) ? (index) : ((index) | 0x80))

struct kpad {
	int keyno;
	int gpio;
	int irq;
};

#define KPTAB(sc,gp) {sc,gp,IRQ_GPIO(gp)}
static struct kpad gpiopadtab[] = {
	KPTAB(H3600_KEYCODE_UP,GPIO_NR_H1900_UP_BUTTON_N),
	KPTAB(H3600_KEYCODE_RIGHT,GPIO_NR_H1900_RIGHT_BUTTON_N),
	KPTAB(H3600_KEYCODE_LEFT,GPIO_NR_H1900_LEFT_BUTTON_N),
	KPTAB(H3600_KEYCODE_DOWN,GPIO_NR_H1900_DOWN_BUTTON_N),
	KPTAB(H3600_KEYCODE_ACTION,GPIO_NR_H1900_ACTION_BUTTON_N),
	KPTAB(H3600_KEYCODE_SUSPEND,GPIO_NR_H1900_POWER_BUTTON_N),
	{-1,-1,-1}
};

static void h1900_keypad(int irq, void* data, struct pt_regs *regs)
{
	int button;
	int gpiono;
	int ispushed;
	
	/* look up the keyno */
	for (button = 0; gpiopadtab[button].irq != -1; button++)
		if (gpiopadtab[button].irq == irq)
			break;
	
	if (gpiopadtab[button].irq == -1)
	{
		printk("keypad: unhandled irq %d (%d-%d)\n", irq, irq-22, GPLR(irq-22) & GPIO_bit(irq-22));
		return;
	};
	
	gpiono = gpiopadtab[button].gpio;
	button = gpiopadtab[button].keyno;
	
	ispushed = GPLR(gpiono) & GPIO_bit(gpiono);
	ispushed = !ispushed;
	
	h3600_hal_keypress( H3600_MAKEKEY( (unsigned char)button, (unsigned char)ispushed ) );
};

/**********************************************************************************
 *      Keypad handling (asic)
 **********************************************************************************/

#define BIT_RECORD   0x04
#define BIT_HOME     0x08
#define BIT_MAIL     0x10
#define BIT_CONTACTS 0x20
#define BIT_CALENDAR 0x40

static struct kpad bittokey[] = {
	{BIT_RECORD,H3600_KEYCODE_RECORD,0},
	{BIT_HOME,H3600_KEYCODE_START,0},
	{BIT_MAIL,H3600_KEYCODE_Q,0},
	{BIT_CONTACTS,H3600_KEYCODE_CONTACTS,0},
	{BIT_CALENDAR,H3600_KEYCODE_CALENDAR,0},
	{-1,-1,-1}
};

static void h1900_asickey(int bit)
{
	int pressed;
	int i;

	if (bit == 0)
		return;
		
	pressed = H3900_ASIC3_GPIO_OFFSET(unsigned int,D,Status) & bit;
	pressed = !pressed;
	
	H3900_ASIC3_GPIO_OFFSET(unsigned int,D,EdgeTrigger) = ((pressed) ? (H3900_ASIC3_GPIO_OFFSET(unsigned int,D,EdgeTrigger) | bit) : (H3900_ASIC3_GPIO_OFFSET(unsigned int,D,EdgeTrigger) & ~bit));
	
	for (i=0;bittokey[i].keyno != -1;i++)
		if (bittokey[i].keyno == bit)
			break;
			
	if (bittokey[i].keyno == -1)
	{
		printk("h1900_asickey: unhandled bit 0x%02X!\n", bit);
		return;
	};

	h3600_hal_keypress( H3600_MAKEKEY( (unsigned char)bittokey[i].gpio, (unsigned char)pressed ) );
};

static void h1900_asic(int irq, void* data, struct pt_regs *regs)
{
	unsigned int gpiostat = H3900_ASIC3_GPIO_OFFSET(unsigned int,D,IntStatus);
	
	gpiostat &= 0x7C;
	if (gpiostat)
	{
		H3900_ASIC3_GPIO_OFFSET(unsigned int,D,IntStatus) = 0x0;
		
		h1900_asickey(gpiostat&BIT_RECORD);
		h1900_asickey(gpiostat&BIT_HOME);
		h1900_asickey(gpiostat&BIT_MAIL);
		h1900_asickey(gpiostat&BIT_CONTACTS);
		h1900_asickey(gpiostat&BIT_CALENDAR);
	};
}

/***********************************************************************************
 *      Generic IRQ handling
 ***********************************************************************************/

#define buttonirq(x) 	set_GPIO_IRQ_edge(x, GPIO_BOTH_EDGES); \
			request_irq(IRQ_GPIO(x), h1900_keypad, SA_SAMPLE_RANDOM, "h1900_keypad_act", NULL);


static int h1900_asic_init_isr( void )
{
	set_GPIO_IRQ_edge(GPIO_NR_H1900_ASIC_IRQ_1_N, GPIO_BOTH_EDGES);
	set_GPIO_IRQ_edge(GPIO_NR_H1900_ASIC_IRQ_2_N, GPIO_BOTH_EDGES);
	request_irq(IRQ_GPIO(GPIO_NR_H1900_ASIC_IRQ_1_N), h1900_asic, SA_INTERRUPT, "h1900_asic", NULL);
	request_irq(IRQ_GPIO(GPIO_NR_H1900_ASIC_IRQ_2_N), h1900_asic, SA_INTERRUPT, "h1900_asic", NULL);
	
	buttonirq(GPIO_NR_H1900_ACTION_BUTTON_N)
	buttonirq(GPIO_NR_H1900_POWER_BUTTON_N)
	buttonirq(GPIO_NR_H1900_UP_BUTTON_N)
	buttonirq(GPIO_NR_H1900_DOWN_BUTTON_N)
	buttonirq(GPIO_NR_H1900_LEFT_BUTTON_N)
	buttonirq(GPIO_NR_H1900_RIGHT_BUTTON_N)

	return 0;
}

static void  h1900_asic_release_isr( void )
{
}


/***********************************************************************************
 *   Sub-module support
 ***********************************************************************************/

struct asic_system_handler { 
	char *name;
	int  (*init)( void );
	void (*cleanup)( void );
	int  (*suspend)( void );
	void (*resume)( void );
};

/* 
   We initialize and resume from top to bottom,
   cleanup and suspend from bottom to top 
*/
   
const struct asic_system_handler h1900_asic_system_handlers[] = {
#if 0
#if defined(CONFIG_MMC) || defined(CONFIG_MMC_MODULE)
	{
		name:    "mmc",
		init:    h1900_asic_mmc_init,
		cleanup: h1900_asic_mmc_cleanup,
		suspend: h1900_asic_mmc_suspend,
		resume:  h1900_asic_mmc_resume
	}
#endif
#endif
};

#define SYS_HANDLER_SIZE   (sizeof(h1900_asic_system_handlers)/sizeof(struct asic_system_handler))

static int h1900_asic_suspend_handlers( void )
{
	int i;
	int result;

	for ( i = SYS_HANDLER_SIZE - 1 ; i >= 0 ; i-- ) {
		if ( h1900_asic_system_handlers[i].suspend ) {
			if ((result = h1900_asic_system_handlers[i].suspend()) != 0 ) {
				while ( ++i < SYS_HANDLER_SIZE )
					if ( h1900_asic_system_handlers[i].resume )
						h1900_asic_system_handlers[i].resume();
				return result;
			}
		}
	}
	return 0;
}

static void h1900_asic_resume_handlers( void )
{
	int i;

	for ( i = 0 ; i < SYS_HANDLER_SIZE ; i++ )
		if ( h1900_asic_system_handlers[i].resume )
			h1900_asic_system_handlers[i].resume();
}

static int __init h1900_asic_init_handlers( void )
{
	int i;
	int result;

	for ( i = 0 ; i < SYS_HANDLER_SIZE ; i++ ) {
		if ( h1900_asic_system_handlers[i].init ) {
			if ( (result = h1900_asic_system_handlers[i].init()) != 0 ) {
				while ( --i >= 0 )
					if ( h1900_asic_system_handlers[i].cleanup )
						h1900_asic_system_handlers[i].cleanup();
				return result;
			}
		}
	}
	return 0;
}

static void  h1900_asic_cleanup_handlers( void )
{
	int i;

	for ( i = SYS_HANDLER_SIZE - 1 ; i >= 0 ; i-- )
		if ( h1900_asic_system_handlers[i].cleanup )
			h1900_asic_system_handlers[i].cleanup();
}

/***********************************************************************************
 *   Power management
 *
 *   On sleep, if we return anything other than "0", we will cancel sleeping.
 *
 *   On resume, if we return anything other than "0", we will put the iPAQ
 *     back to sleep immediately.
 ***********************************************************************************/

static int h1900_asic_pm_callback(pm_request_t req)
{
	int result = 0;

	switch (req) {
	case PM_SUSPEND:
		result = h1900_asic_suspend_handlers();
//		h3600_asic_initiate_sleep ();
		break;

	case PM_RESUME:
//		result = h3600_asic_check_wakeup ();
		if ( !result ) {
			h1900_asic_resume_handlers();
		}
		break;
	}
	return result;
}

static int h1900_asic_reset_handler(ctl_table *ctl, int write, struct file * filp,
				    void *buffer, size_t *lenp)
{
	MOD_INC_USE_COUNT;
	h1900_asic_pm_callback( PM_SUSPEND );
	h1900_asic_pm_callback( PM_RESUME );
	MOD_DEC_USE_COUNT;

	return 0;
}

/***********************************************************************************
 *   Initialization code
 ***********************************************************************************/

static  void h1900_asic_cleanup( void )
{
	h3600_unregister_pm_callback( h1900_asic_pm_callback );
	h1900_asic_unregister_procfs();
	h3600_hal_unregister_interface( &h1900_asic_ops );
	h1900_asic_cleanup_handlers();
	h1900_asic_release_isr();
//	ipaq_mtd_asset_cleanup();
}

int h1900_asic_init( void )
{
	int result;
	int line;

	if ( !machine_is_h1900() ) {
		printk("%s: unknown iPAQ model %s\n", __FUNCTION__, h3600_generic_name() );
		return -ENODEV;
	}

//	h3600_asic_register_battery_ops (&h1900_battery_ops);

//	result = ipaq_mtd_asset_init();
//	if ( result ) { line = __LINE__; goto init_fail; };

	result = h1900_asic_init_isr();  
	if ( result ) { line = __LINE__; goto init_fail; };

	result = h1900_asic_init_handlers();
	if ( result ) { line = __LINE__; goto init_fail; };

	result = h3600_hal_register_interface( &h1900_asic_ops );
	if ( result ) { line = __LINE__; goto init_fail; };

	result = h1900_asic_register_procfs();
	if ( result ) { line = __LINE__; goto init_fail; };

	result = h3600_register_pm_callback( h1900_asic_pm_callback );
	if ( result ) { line = __LINE__; goto init_fail; };

	return 0;

init_fail:
	printk("%s:%d: %s: FAILURE!  result=%d. Exiting...\n", __FILE__, line, __FUNCTION__, result);
	h1900_asic_cleanup();
	return result;
}

void  h1900_asic_exit( void )
{
	h1900_asic_cleanup();
}

module_init(h1900_asic_init)
module_exit(h1900_asic_exit)
