/*
    This file is part of KOrganizer.
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

#include <qwidget.h>
#include <qtooltip.h>
#include <qlayout.h>
#include <qvbox.h>
#include <qbuttongroup.h>
#include <qvgroupbox.h>
#include <qwidgetstack.h>
#include <qdatetime.h>
#include <qfile.h>


#include <kglobal.h>
#include <kdebug.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kfiledialog.h>
#include <kstandarddirs.h>

#include <libkcal/todo.h>
#include <libkcal/event.h>
#include <libkdepim/categoryselectdialog.h>
#include <libkdepim/kdateedit.h>

#include "koprefs.h"
#include "koglobals.h"

#include "koeditorgeneral.h"
#include "kolocationbox.h"
#include "koeditorgeneral.moc"
#ifndef DESKTOP_VERSION
#include <qpe/qpeapplication.h>
#else
#include <qapplication.h>
#endif

KOEditorGeneral::KOEditorGeneral(QObject* parent, const char* name) :
    QObject( parent, name)
{
}

KOEditorGeneral::~KOEditorGeneral()
{
}

void KOEditorGeneral::initHeader(QWidget *parent,QBoxLayout *topLayout)
{
    QGridLayout *headerLayout = new QGridLayout(topLayout);

#if 0
    mOwnerLabel = new QLabel(i18n("Owner:"),parent);
    headerLayout->addMultiCellWidget(mOwnerLabel,0,0,0,1);
#endif

    QLabel *summaryLabel = new QLabel(i18n("Summary:"),parent);
    headerLayout->addWidget(summaryLabel,1,0);

    mSummaryEdit = new KOLocationBox(TRUE,parent, 10);
    mSummaryEdit->setSizePolicy( QSizePolicy(  QSizePolicy::Preferred  ,QSizePolicy::MinimumExpanding  ,FALSE) );
    //mSummaryEdit->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)5 ,(QSizePolicy::SizeType)3 ,FALSE) );
    //qDebug("h %d %d ", summaryLabel->sizeHint().height(),mSummaryEdit->sizeHint().height() );
    int hei =  (summaryLabel->sizeHint().height() + mSummaryEdit->sizeHint().height())/2;
    if ( QApplication::desktop()->width() > 320 ) 
        mSummaryEdit->setMaximumHeight( hei +6 );
    //qDebug("%d %d %d %d %d %d ", QSizePolicy::Fixed ,  QSizePolicy::Minimum ,   QSizePolicy:: Maximum ,   QSizePolicy:: Preferred  ,    QSizePolicy::  MinimumExpanding , QSizePolicy::Expanding   );
    // SizeType { Fixed = 0, Minimum = MayGrow, Maximum = MayShrink, Preferred = MayGrow|MayShrink, MinimumExpanding = Minimum|ExpMask, Expanding = MinimumExpanding|MayShrink }
    //   mSummaryEdit = new QLineEdit(parent);
    headerLayout->addWidget(mSummaryEdit,1,1);
    connect ( mSummaryEdit->lineEdit(), SIGNAL ( returnPressed() ), this, SIGNAL (allAccepted () ) );
  
    QLabel *locationLabel = new QLabel(i18n("Location:"),parent);
    headerLayout->addWidget(locationLabel,2,0);

    mLocationEdit = new KOLocationBox(TRUE,parent,10); 
    mLocationEdit->setSizePolicy( QSizePolicy( QSizePolicy::Preferred  ,QSizePolicy::MinimumExpanding  ,FALSE) );
    if ( QApplication::desktop()->width() > 320 ) 
        mLocationEdit->setMaximumHeight( hei + 6);

    //   mLocationEdit = new QLineEdit(parent);
    connect ( mLocationEdit->lineEdit(), SIGNAL ( returnPressed() ), this, SIGNAL (allAccepted () ) );
    headerLayout->addWidget(mLocationEdit,2,1);
    headerLayout->setColStretch( 1, 10);
}
void KOEditorGeneral::setFocusOn( int i )
{
    qApp->processEvents();
    if ( i == 1 ) {
        mDescriptionEdit->setFocus();
        mDescriptionEdit->setCursorPosition( mDescriptionEdit->numLines (), 333);
    } 
    if ( i == 2 ) {
        mSummaryEdit->setFocus();
    }

}
void KOEditorGeneral::editCategories()
{
    // qDebug("KOEditorGeneral::editCategories() ");
    KPIM::CategorySelectDialog* csd = new KPIM::CategorySelectDialog( KOPrefs::instance(), 0 );
    connect(csd,SIGNAL(categoriesSelected(const QString &)), this ,SLOT(setCategories(const QString &)));
    //KOGlobals::fitDialogToScreen( csd );
    csd->setSelected( QStringList::split (",", mCategoriesLabel->text()) );
    csd->exec();
    delete csd;
}
void KOEditorGeneral::initCategories(QWidget *parent, QBoxLayout *topLayout)
{
    QBoxLayout *categoriesLayout = new QHBoxLayout( topLayout );

    mCategoriesButton = new QPushButton(parent);
    mCategoriesButton->setText(i18n("Categories..."));
    connect(mCategoriesButton,SIGNAL(clicked()),this, SLOT(editCategories() ));
    categoriesLayout->addWidget(mCategoriesButton);

    mCategoriesLabel = new QLabel(parent);
    mCategoriesLabel->setFrameStyle(QFrame::Panel|QFrame::Sunken);
    categoriesLayout->addWidget(mCategoriesLabel,1);
}

void KOEditorGeneral::initSecrecy(QWidget *parent, QBoxLayout *topLayout)
{
    QBoxLayout *secrecyLayout = new QHBoxLayout( topLayout );

    QLabel *secrecyLabel = new QLabel(i18n("Access:"),parent);
    mCancelBox = new QCheckBox ( i18n("Cancelled"), parent);
    secrecyLayout->addWidget(mCancelBox);
    secrecyLayout->addWidget(secrecyLabel);

    mSecrecyCombo = new QComboBox(parent);
    mSecrecyCombo->insertStringList(Incidence::secrecyList());
    secrecyLayout->addWidget(mSecrecyCombo);
}

void KOEditorGeneral::initDescription(QWidget *parent,QBoxLayout *topLayout)
{
    mDescriptionEdit = new KTextEdit(parent);
    mDescriptionEdit->setFont(KOPrefs::instance()->mEditBoxFont );
    mDescriptionEdit->append("");
    mDescriptionEdit->setReadOnly(false);
    mDescriptionEdit->setOverwriteMode(false);
    mDescriptionEdit->setWordWrap( KTextEdit::WidgetWidth ); 
    topLayout->addWidget(mDescriptionEdit); 
#ifndef DESKTOP_VERSION
    QPEApplication::setStylusOperation( mDescriptionEdit, QPEApplication::RightOnHold );
#endif
 
}

void KOEditorGeneral::initAlarm(QWidget *parent,QBoxLayout *topLayout)
{
    QBoxLayout *alarmLayout = new QHBoxLayout(topLayout);

    //mAlarmBell = new QLabel(parent);
    //mAlarmBell->setPixmap(SmallIcon("bell"));
    //alarmLayout->addWidget(mAlarmBell);
    if ( QApplication::desktop()->width() < 320 ) 
        mAlarmButton = new QCheckBox(i18n("Rem."),parent);
    else
        mAlarmButton = new QCheckBox(i18n("Reminder:"),parent);
      
    connect(mAlarmButton, SIGNAL(toggled(bool)), SLOT(enableAlarmEdit(bool)));
    alarmLayout->addWidget(mAlarmButton);

    mAlarmTimeEdit = new QSpinBox ( 0, 9999, 1, parent, "mAlarmTimeEdit  "  ) ;
    alarmLayout->addWidget(mAlarmTimeEdit);
    mAlarmIncrCombo = new QComboBox(false, parent);
    if ( QApplication::desktop()->width() < 320 ) {
        mAlarmIncrCombo->insertItem(i18n("min"));
        mAlarmIncrCombo->insertItem(i18n("hou"));
        mAlarmIncrCombo->insertItem(i18n("day"));
        mAlarmTimeEdit->setMaximumWidth( mAlarmTimeEdit->sizeHint().width() );
        mAlarmIncrCombo->setMaximumWidth( mAlarmIncrCombo->sizeHint().width() );
    } else {
        mAlarmIncrCombo->insertItem(i18n("minute(s)"));
        mAlarmIncrCombo->insertItem(i18n("hour(s)"));
        mAlarmIncrCombo->insertItem(i18n("day(s)"));
    }

    //  mAlarmIncrCombo->setMinimumHeight(20);
    alarmLayout->addWidget(mAlarmIncrCombo);
    mAlarmSoundButton = new QPushButton(parent);
    mAlarmSoundButton->setPixmap(SmallIcon("playsound"));
    mAlarmSoundButton->setToggleButton(true);
    QToolTip::add(mAlarmSoundButton, i18n("No sound set"));
    connect(mAlarmSoundButton, SIGNAL(clicked()), SLOT(pickAlarmSound()));
    alarmLayout->addWidget(mAlarmSoundButton);

    mAlarmProgramButton = new QPushButton(parent);
    mAlarmProgramButton->setPixmap(SmallIcon("run"));
    mAlarmProgramButton->setToggleButton(true);
    QToolTip::add(mAlarmProgramButton, i18n("No program set"));
    connect(mAlarmProgramButton, SIGNAL(clicked()), SLOT(pickAlarmProgram()));
    alarmLayout->addWidget(mAlarmProgramButton);
    mAlarmSoundButton->setMaximumWidth( mAlarmSoundButton->sizeHint().width() + 4 );
    mAlarmProgramButton->setMaximumWidth(mAlarmProgramButton->sizeHint().width() + 4 );
    //   if ( KOPrefs::instance()->mCompactDialogs ) {
    //     mAlarmSoundButton->hide();
    //     mAlarmProgramButton->hide();
    //   }
}

void KOEditorGeneral::pickAlarmSound()
{

    //QString prefix = mAlarmSound;
    if (!mAlarmSoundButton->isOn()) {
        //mAlarmSound = "";
        QToolTip::remove(mAlarmSoundButton);
        QToolTip::add(mAlarmSoundButton, i18n("No sound set"));
        mAlarmProgramButton->setOn(true);
        mAlarmSoundButton->setOn(false);
    } else {
        QString fileName(KFileDialog::getOpenFileName(mAlarmSound,
                                                      i18n("*.wav|Wav Files"), 0));
        if (!fileName.isEmpty()) {
            mAlarmSound = fileName;
            QToolTip::remove(mAlarmSoundButton);
            QString dispStr = i18n("Playing '%1'").arg(fileName);
            QToolTip::add(mAlarmSoundButton, dispStr);
            mAlarmProgramButton->setOn(false);
            mAlarmSoundButton->setOn(true);
        } else {
            mAlarmProgramButton->setOn(true);
            mAlarmSoundButton->setOn(false);
           
        }
    }
   
    if (mAlarmProgramButton->isOn())
        ((QWidget*)parent())->topLevelWidget()->setCaption(i18n("Proc.Al.: ") + mAlarmProgram );
    if ( mAlarmSoundButton->isOn())
        ((QWidget*)parent())->topLevelWidget()->setCaption(i18n("Audio.Al.: ") + mAlarmSound );
}

void KOEditorGeneral::pickAlarmProgram()
{
    if (!mAlarmProgramButton->isOn()) {
        //mAlarmProgram = "";
        QToolTip::remove(mAlarmProgramButton);
        QToolTip::add(mAlarmProgramButton, i18n("No program set"));
        mAlarmProgramButton->setOn(false);
        mAlarmSoundButton->setOn(true);
    } else {
        QString fileName(KFileDialog::getOpenFileName(mAlarmProgram,i18n("Procedure Alarm.: ")  , 0));
        if (!fileName.isEmpty()) {
            mAlarmProgram = fileName;
            QToolTip::remove(mAlarmProgramButton);
            QString dispStr = i18n("Running '%1'").arg(fileName);
            QToolTip::add(mAlarmProgramButton, dispStr);
            mAlarmSoundButton->setOn(false);
            mAlarmProgramButton->setOn(true);
        }  else {
            mAlarmProgramButton->setOn(false);
            mAlarmSoundButton->setOn(true);
        }
    }
    if (mAlarmProgramButton->isOn())
        ((QWidget*)parent())->topLevelWidget()->setCaption(i18n("Proc.Al.: ") + mAlarmProgram );
    if ( mAlarmSoundButton->isOn())
        ((QWidget*)parent())->topLevelWidget()->setCaption(i18n("Sound.Al.: ") + mAlarmSound );
}



void KOEditorGeneral::enableAlarmEdit(bool enable)
{
    if (  enable ) {
        if (!mAlarmProgramButton->isOn() && !mAlarmSoundButton->isOn()) {
            mAlarmSoundButton->setOn( true );
            if ( mAlarmSound.isEmpty() )
                mAlarmSound = KOPrefs::instance()->mDefaultAlarmFile;
            else {
                if ( ! QFile::exists( mAlarmSound ) )
                    mAlarmSound = KOPrefs::instance()->mDefaultAlarmFile;
            }
        }
        if (mAlarmProgramButton->isOn())
            ((QWidget*)parent())->topLevelWidget()->setCaption(i18n("Proc.Al.: ") + mAlarmProgram );
        if (!mAlarmSound.isEmpty() && mAlarmSoundButton->isOn())
            ((QWidget*)parent())->topLevelWidget()->setCaption(i18n("Sound.Al.: ") + mAlarmSound );
    }
    else {
        ((QWidget*)parent())->topLevelWidget()->setCaption(i18n("Alarm disabled"));

    }
    mAlarmTimeEdit->setEnabled(enable);
    mAlarmSoundButton->setEnabled(enable);
    mAlarmProgramButton->setEnabled(enable);
    mAlarmIncrCombo->setEnabled(enable);
}

void KOEditorGeneral::disableAlarmEdit(bool disable)
{
    enableAlarmEdit( !disable );
}

void KOEditorGeneral::enableAlarm( bool enable )
{
    enableAlarmEdit( enable );
}

void KOEditorGeneral::alarmDisable(bool disable)
{
    if (!disable) {
        //mAlarmBell->setEnabled(true);
        mAlarmButton->setEnabled(true);
    } else {
        //mAlarmBell->setEnabled(false);
        mAlarmButton->setEnabled(false);
        mAlarmButton->setChecked(false);
        mAlarmTimeEdit->setEnabled(false);
        mAlarmSoundButton->setEnabled(false);
        mAlarmProgramButton->setEnabled(false);
        mAlarmIncrCombo->setEnabled(false);
    }
}

void KOEditorGeneral::setCategories(const QString &str)
{
    mCategoriesLabel->setText(str);
}

void KOEditorGeneral::setDefaults(bool allDay)
{
#if 0
    mOwnerLabel->setText(i18n("Owner: ") + KOPrefs::instance()->fullName());
#endif

    enableAlarmEdit( !allDay );

    // TODO: Implement a KPrefsComboItem to solve this in a clean way.
    int alarmTime;
    int a[] = { 1,5,10,15,30,60,180, 1440 };
    int index = KOPrefs::instance()->mAlarmTime;
    if (index < 0 || index > 7) {
        alarmTime = 15;
    } else {
        alarmTime = a[index];
    }
    mAlarmButton ->setChecked( false );
    mAlarmTimeEdit->setValue(alarmTime);
    mAlarmIncrCombo->setCurrentItem(0);
    enableAlarmEdit( false );
    //alarmDisable (false);
    mSecrecyCombo->setCurrentItem(Incidence::SecrecyPublic);
    mCancelBox->setChecked( false ); 
    mSummaryEdit->setEditText("");
    mLocationEdit->setEditText("");
    mDescriptionEdit->setText("");
    mAlarmSound = KOPrefs::instance()->mDefaultAlarmFile;
    setCategories("");
}
void KOEditorGeneral::setSecrecy( int num )
{
    mSecrecyCombo->setCurrentItem(num);
}
void KOEditorGeneral::readIncidence(Incidence *event)
{

    mAlarmIncrCombo->setCurrentItem(0);
    mSummaryEdit->setEditText(event->summary());
    mLocationEdit->setEditText(event->location());
    mDescriptionEdit->setText(event->description());

#if 0
    // organizer information
    mOwnerLabel->setText(i18n("Owner: ") + event->organizer());
#endif
  
    enableAlarmEdit( event->isAlarmEnabled() );
    //qDebug("KOEditorGeneral::readIncidence(Incidence *event) ");
    if(!event->isAlarmEnabled()) {
        // TODO: Implement a KPrefsComboItem to solve this in a clean way.
        int alarmTime;
        int a[] = { 1,5,10,15,30,60,180, 1440 };
        int index = KOPrefs::instance()->mAlarmTime;
        if (index < 0 || index > 7) {
            alarmTime = 15;
        } else {
            alarmTime = a[index];
        }
        mAlarmTimeEdit->setValue(alarmTime);
    }
    mAlarmButton->setChecked( event->isAlarmEnabled() );
    mSecrecyCombo->setCurrentItem(event->secrecy());
    mCancelBox->setChecked( event->cancelled() ); 
    mAlarmProgramButton->setOn(false);
    mAlarmSoundButton->setOn(false);

    // set up alarm stuff
    QPtrList<Alarm> alarms = event->alarms();
    Alarm* alarm;
    mAlarmIncrCombo->setCurrentItem(0);
    for ( alarm = alarms.first(); alarm; alarm = alarms.next() ) {
        int offset;
        if ( alarm->hasTime() ) {
            QDateTime t = alarm->time();
            offset = event->dtStart().secsTo( t );
        } else {
            offset = alarm->startOffset().asSeconds();
        }
        if ( offset != 0 ) {
            offset = offset / -60; // make minutes
            if (offset % 60 == 0) { // divides evenly into hours?
                offset = offset / 60;
                mAlarmIncrCombo->setCurrentItem(1);
                if (offset % 24 == 0) { // divides evenly into days?
                    offset = offset / 24;
                    mAlarmIncrCombo->setCurrentItem(2);
                }
            }
        }
        mAlarmTimeEdit->setValue( offset );
        if (alarm->type() == Alarm::Procedure) {
      
            mAlarmProgram = alarm->programFile();
            mAlarmProgramButton->setOn(true);
            QString dispStr = i18n("Running '%1'").arg(mAlarmProgram);
            QToolTip::add(mAlarmProgramButton, dispStr);
        }
        else if (alarm->type() == Alarm::Audio) {
            mAlarmSound = alarm->audioFile(); 
            if ( ! QFile::exists( mAlarmSound ) )
                mAlarmSound = KOPrefs::instance()->mDefaultAlarmFile;
            mAlarmSoundButton->setOn(true);
            QString dispStr = i18n("Playing '%1'").arg(mAlarmSound);
            QToolTip::add(mAlarmSoundButton, dispStr);
        }
        mAlarmButton->setChecked(alarm->enabled());
        enableAlarmEdit( alarm->enabled() );
        //qDebug("nableAlarmEdit( alarm->enabled() )********* ");
        // TODO: Deal with multiple alarms
        break; // For now, stop after the first alarm
    }

    setCategories(event->categoriesStr());
}

void KOEditorGeneral::writeIncidence(Incidence *event)
{
    //  kdDebug() << "KOEditorGeneral::writeEvent()" << endl;
    mLocationEdit->save(KOLocationBox::LOCATION);
    event->setSummary(mSummaryEdit->currentText());
    event->setLocation(mLocationEdit->currentText());
    event->setDescription(mDescriptionEdit->text());
    event->setCategories(mCategoriesLabel->text());
    event->setSecrecy(mSecrecyCombo->currentItem());
    event->setCancelled(mCancelBox->isChecked() );; 
    // alarm stuff
    if (mAlarmButton->isChecked()) {
        if (event->alarms().count() == 0) 
            event->newAlarm();
        QPtrList<Alarm> alarms = event->alarms();
        Alarm *alarm;
        for (alarm = alarms.first(); alarm; alarm = alarms.next() ) {
            alarm->setEnabled(true);
            int j = mAlarmTimeEdit->value()* -60;
            if (mAlarmIncrCombo->currentItem() == 1)
                j = j * 60;
            else if (mAlarmIncrCombo->currentItem() == 2)
                j = j * (60 * 24);
            alarm->setStartOffset( j );

            if (!mAlarmProgram.isEmpty() && mAlarmProgramButton->isOn()) {
                alarm->setProcedureAlarm(mAlarmProgram);
            }
            else if (!mAlarmSound.isEmpty() && mAlarmSoundButton->isOn())
                alarm->setAudioAlarm(mAlarmSound);
            else
                alarm->setType(Alarm::Invalid);
            //alarm->setAudioAlarm("default");
            // TODO: Deal with multiple alarms
            break; // For now, stop after the first alarm
        }
    } else {
        Alarm* alarm = event->alarms().first();
        if ( alarm ) {
            alarm->setEnabled(false);
            alarm->setType(Alarm::Invalid);
        }
    }
}
