/*
 * btscanner - Displays the output of Bluetooth scans
 * Copyright (C) 2003 Pentest Limited
 * 
 * Written 2003 by Tim Hurman <timh at pentest.co.uk>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation;
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT OF THIRD PARTY
 * RIGHTS.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) AND AUTHOR(S) BE LIABLE
 * FOR ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY
 * DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 * AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * ALL LIABILITY, INCLUDING LIABILITY FOR INFRINGEMENT OF ANY PATENTS,
 * COPYRIGHTS, TRADEMARKS OR OTHER RIGHTS, RELATING TO USE OF THIS SOFTWARE
 * IS DISCLAIMED.
 */

/*
 * ouidb.c: Reads in an oui database and stores it in memory
 */

#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <regex.h>

#include "btscanner.h"
#define OUI_RE "^([0-9a-zA-Z]{6}) +\\(base 16\\)[ \t]+(.*)[\n\r \t]+$"

/* convert XXXXXX to a ba addr */
int oui2ba(bdaddr_t *ba, char *p, size_t len)
{
	uint32_t oui;

	if (len != 6)
		return 1;
	oui = strtol(p, (char**)NULL, 16);
	memset(ba, 0, sizeof(bdaddr_t));
	ba->b[3] = oui & 0xff;
	ba->b[4] = (oui>>8) & 0xff;
	ba->b[5] = (oui>>16) & 0xff;

	return 0;
}

/* initalise the gdbm file and read the oui list */
#define OUIDB_SZ 128
int ouidb_init (struct proc_info *pi, const char *filename)
{
	FILE *f;
	char buf[OUIDB_SZ];
	regex_t preg;
	regmatch_t pmatch[4];
	int i;
	bdaddr_t ba;
	datum k,v;

	memset(&ba, 0, sizeof(ba));
	i = regcomp(&preg, OUI_RE, REG_EXTENDED);
	if (i != 0) {
		fprintf(stderr, "Unable to compile re: \"%s\"\n", OUI_RE);
		return 1;
	}

	pi->ouidb_filename = (char*)malloc(sizeof(char)*32);
	if (NULL == pi->ouidb_filename) {
		perror("ouidb_init::malloc():");
		return 1;
	}
	strncpy(pi->ouidb_filename, "/tmp/ouidbXXXXXX", 32);
	umask(022);
	if(-1 == mkstemp(pi->ouidb_filename)) {
		perror("ouidb_init::mkstemp():");
		return 1;
	}

	pi->ouidb = gdbm_open(pi->ouidb_filename, 0, GDBM_WRCREAT, 0600, 0);
	if (NULL == pi->ouidb) {
		fprintf(stdout, "gdbm_open(): failed\n");
		return 1;
	}

	fprintf(stdout, "Opening the OUI database\n");
	f = fopen(filename, "r");
	if (NULL == f) {
		fprintf(stdout, "Error opening the OUI database: %s\n",
		  strerror(errno));
		return 1;
	}

	fprintf(stdout, "Reading the OUI database\n");

	while (NULL != fgets(buf, OUIDB_SZ, f)) {
		if(regexec(&preg, buf, 4, pmatch, 0))
			continue;

		buf[pmatch[1].rm_eo] = 0;
		buf[pmatch[2].rm_eo] = 0;
		if (oui2ba(&ba, buf+pmatch[1].rm_so, pmatch[1].rm_eo-pmatch[1].rm_so))
			continue;

		k.dptr = (char*)&ba;
		k.dsize = sizeof(ba);
		v.dptr = buf+pmatch[2].rm_so;
		v.dsize = (pmatch[2].rm_eo-pmatch[2].rm_so)+1;

		if (v.dsize == 0) {
			v.dptr = "Private";
			v.dsize = strlen(v.dptr)+1;
		}

		gdbm_store(pi->ouidb, k, v, GDBM_INSERT);
	}

	fprintf(stdout, "Finished reading the OUI database\n");
	gdbm_sync(pi->ouidb);
	fclose(f);
	regfree(&preg);
	return 0;
}

/* query a device */
char *ouidb_query(struct proc_info *pi, bdaddr_t *ba)
{
	bdaddr_t batmp;
	datum key, val;

	bacpy(&batmp, ba);
	batmp.b[0] = batmp.b[1] = batmp.b[2] = 0;
	key.dptr = (char*)&batmp;
	key.dsize = sizeof(batmp);

	val = gdbm_fetch(pi->ouidb, key);
	return val.dptr;
}


/* close down and cleanup */
int ouidb_close (struct proc_info *pi)
{
	if (NULL != pi->ouidb) {
		gdbm_close(pi->ouidb);
		pi->ouidb = NULL;
	}

	if (NULL != pi->ouidb_filename) {
		unlink(pi->ouidb_filename);
		free(pi->ouidb_filename);
		pi->ouidb_filename = NULL;
	}
	return 0;
}

