/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2001  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2001  Haavard Kvaalen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "pluginenum.h"
#include "dynamic_lib.h"

#include <glib.h>
#include <glib/gprintf.h>
#include <string.h>

#include "main.h"
#include "playlist.h"
#include "controlsocket.h"

#include "effect.h"
#include "general.h"
#include "input.h"
#include "output.h"
#include "visualization.h"

const gchar *plugin_dir_list[] = {
    PLUGINSUBS,
    NULL
};

extern struct InputPluginData *ip_data;
extern struct OutputPluginData *op_data;
extern struct EffectPluginData *ep_data;
extern struct GeneralPluginData *gp_data;
extern struct VisPluginData *vp_data;

extern Config cfg;

static gint d_iplist_compare(gconstpointer a, gconstpointer b)
{
    return strcmp((const gchar *) a, (const gchar *) b);
}

static gint inputlist_compare_func(gconstpointer a, gconstpointer b)
{
    const InputPlugin *ap = a, *bp = b;
    return strcasecmp(ap->description, bp->description);
}

static int outputlist_compare_func(gconstpointer a, gconstpointer b)
{
    const OutputPlugin *ap = a, *bp = b;
    return strcasecmp(ap->description, bp->description);
}

static int effectlist_compare_func(gconstpointer a, gconstpointer b)
{
    const EffectPlugin *ap = a, *bp = b;
    return strcasecmp(ap->description, bp->description);
}

static int generallist_compare_func(gconstpointer a, gconstpointer b)
{
    const GeneralPlugin *ap = a, *bp = b;
    return strcasecmp(ap->description, bp->description);
}

static int vislist_compare_func(gconstpointer a, gconstpointer b)
{
    const VisPlugin *ap = a, *bp = b;
    return strcasecmp(ap->description, bp->description);
}

static void plugin_return_tag_value_by_name_dummy(gchar * a, gchar ** b,
						  gchar ** c, gchar ** d,
						  gchar ** e, guint * x)
{
    /* dummy function */
}

static int plugin_check_duplicate(char *filename)
{
    GList *l;
    const gchar *base_filename = g_basename(filename);
    /*
     * erg.. gotta check 'em all, surely there's a better way
     *                                                 - Zinx
     */

    for (l = ip_data->input_list; l; l = l->next)
	if (!strcmp(base_filename,
		    g_basename(((InputPlugin *) l->data)->filename)))
	    return 1;

    for (l = op_data->output_list; l; l = l->next)
	if (!strcmp(base_filename,
		    g_basename(((OutputPlugin *) l->data)->filename)))
	    return 1;

    for (l = ep_data->effect_list; l; l = l->next)
	if (!strcmp(base_filename,
		    g_basename(((EffectPlugin *) l->data)->filename)))
	    return 1;

    for (l = gp_data->general_list; l; l = l->next)
	if (!strcmp(base_filename,
		    g_basename(((GeneralPlugin *) l->data)->filename)))
	    return 1;

    for (l = vp_data->vis_list; l; l = l->next)
	if (!strcmp(base_filename,
		    g_basename(((VisPlugin *) l->data)->filename)))
	    return 1;

    return 0;
}

static void add_plugin(gchar * filename)
{
    gpointer h;
    gpointer(*gpi) (void);

    if (plugin_check_duplicate(filename))
	return;

    if ((h = dynamic_lib_open(filename)) == NULL) {
	g_fprintf(stderr, "Failed to load plugin! (%s)\n",
		  dynamic_lib_error());
	return;
    }

    if ((gpi = dynamic_lib_find_symbol(h, "get_iplugin_info"))) {
	InputPlugin *p = gpi();
	p->handle = h;
	p->filename = g_strdup(filename);
/*		p->get_vis_type = input_get_vis_type; */
	if (p->return_tag_value_by_name == NULL) {
	    p->return_tag_value_by_name =
		plugin_return_tag_value_by_name_dummy;
	}
	p->add_vis_pcm = input_add_vis_pcm;
	p->set_info = playlist_set_info;
	p->set_info_text = (void (*)(char *)) input_set_info_text;

	ip_data->input_list = g_list_prepend(ip_data->input_list, p);
    } else if ((gpi = dynamic_lib_find_symbol(h, "get_oplugin_info"))) {
	OutputPlugin *p = gpi();
	p->handle = h;
	p->filename = g_strdup(filename);
	op_data->output_list = g_list_prepend(op_data->output_list, p);
    } else if ((gpi = dynamic_lib_find_symbol(h, "get_eplugin_info"))) {
	EffectPlugin *p = gpi();
	p->handle = h;
	p->filename = g_strdup(filename);
	ep_data->effect_list = g_list_prepend(ep_data->effect_list, p);
    } else if ((gpi = dynamic_lib_find_symbol(h, "get_gplugin_info"))) {
	GeneralPlugin *p = gpi();
	p->handle = h;
	p->filename = g_strdup(filename);
	p->xmms_session = ctrlsocket_get_session_id();
	gp_data->general_list = g_list_prepend(gp_data->general_list, p);
    } else if ((gpi = dynamic_lib_find_symbol(h, "get_vplugin_info"))) {
	VisPlugin *p = gpi();
	p->handle = h;
	p->filename = g_strdup(filename);
	p->xmms_session = ctrlsocket_get_session_id();
	p->disable_plugin = vis_disable_plugin;
	vp_data->vis_list = g_list_prepend(vp_data->vis_list, p);
    } else
	dynamic_lib_close(h);
}

static void bmp_scan_plugins(const gchar * dirname)
{
    gchar *filename, *ext;
    const gchar *entry;
    GDir *dir;

    if (!(dir = g_dir_open(dirname, 0, NULL)))
	return;

    while ((entry = g_dir_read_name(dir))) {
	filename = g_build_filename(dirname, entry);
	if (g_file_test(filename, G_FILE_TEST_IS_REGULAR) &&
	    (ext = strrchr(entry, '.')) != NULL)
	    if (!strcmp(ext, SHARED_LIB_EXT))
		add_plugin(filename);
	g_free(filename);
    }

    g_dir_close(dir);
}


void init_plugins(void)
{
    gchar *dir, *temp, *temp2;
    GList *node;
    GList *disabled_iplugin_names = NULL;
    OutputPlugin *op;
    InputPlugin *ip;
    gint dirsel = 0;

    if (cfg.disabled_iplugins) {
	temp = cfg.disabled_iplugins;
	while ((temp2 = strchr(temp, ','))) {
	    (*temp2) = '\0';
	    disabled_iplugin_names =
		g_list_append(disabled_iplugin_names, g_strdup(temp));
	    temp = temp2 + 1;
	}
	disabled_iplugin_names =
	    g_list_append(disabled_iplugin_names, g_strdup(temp));

	g_free(cfg.disabled_iplugins);
	cfg.disabled_iplugins = NULL;
    }

    ip_data = g_malloc0(sizeof(struct InputPluginData));
    op_data = g_malloc0(sizeof(struct OutputPluginData));
    ep_data = g_malloc0(sizeof(struct EffectPluginData));
    gp_data = g_malloc0(sizeof(struct GeneralPluginData));
    vp_data = g_malloc0(sizeof(struct VisPluginData));


#ifndef DISABLE_USER_PLUGIN_DIR
    dir = g_build_filename(g_get_home_dir(), BMP_RCPATH, "plugins");
    bmp_scan_plugins(dir);
    g_free(dir);

    /*
     * Having directories below ~/.beep/Plugins is deprecated and
     * might be removed at some point.
     */

    /*
     * This is in a separate loop so if the user puts them in the
     * wrong dir we'll still get them in the right order (home dir
     * first)                                                - Zinx
     */
    while (plugin_dir_list[dirsel]) {
	dir = g_strconcat(g_get_home_dir(), "/", BMP_RCPATH, "/Plugins/",
			  plugin_dir_list[dirsel++], NULL);
	bmp_scan_plugins(dir);
	g_free(dir);
    }
    dirsel = 0;
#endif

    while (plugin_dir_list[dirsel]) {
	dir = g_strconcat(PLUGIN_DIR, "/",
			  plugin_dir_list[dirsel++], NULL);
	bmp_scan_plugins(dir);
	g_free(dir);
    }

    op_data->output_list =
	g_list_sort(op_data->output_list, outputlist_compare_func);
    if (!op_data->current_output_plugin
	&& g_list_length(op_data->output_list))
	op_data->current_output_plugin = op_data->output_list->data;
    ip_data->input_list =
	g_list_sort(ip_data->input_list, inputlist_compare_func);
    ep_data->effect_list =
	g_list_sort(ep_data->effect_list, effectlist_compare_func);
    ep_data->enabled_list = NULL;
    gp_data->general_list =
	g_list_sort(gp_data->general_list, generallist_compare_func);
    gp_data->enabled_list = NULL;
    vp_data->vis_list =
	g_list_sort(vp_data->vis_list, vislist_compare_func);
    vp_data->enabled_list = NULL;
    general_enable_from_stringified_list(cfg.enabled_gplugins);
    vis_enable_from_stringified_list(cfg.enabled_vplugins);
    effect_enable_from_stringified_list(cfg.enabled_eplugins);

    g_free(cfg.enabled_gplugins);
    cfg.enabled_gplugins = NULL;
    g_free(cfg.enabled_vplugins);
    cfg.enabled_vplugins = NULL;
    g_free(cfg.enabled_eplugins);
    cfg.enabled_eplugins = NULL;

    node = op_data->output_list;
    while (node) {
	op = (OutputPlugin *) node->data;
	/*
	 * Only test basename to avoid problems when changing
	 * prefix.  We will only see one plugin with the same
	 * basename, so this is usually what the user want.
	 */
	if (!strcmp(g_basename(cfg.outputplugin),
		    g_basename(op->filename)))
	    op_data->current_output_plugin = op;
	if (op->init)
	    op->init();
	node = node->next;
    }

    node = ip_data->input_list;

    while (node) {
	ip = (InputPlugin *) node->data;
	temp = g_path_get_basename(ip->filename);
	if (g_list_find_custom
	    (disabled_iplugin_names, temp, d_iplist_compare))
	    disabled_iplugins = g_list_append(disabled_iplugins, ip);

	if (ip->init)
	    ip->init();

	g_free(temp);
	node = node->next;
    }

    node = disabled_iplugin_names;
    while (node) {
	g_free(node->data);
	node = node->next;
    }
    g_list_free(disabled_iplugin_names);

}

void cleanup_plugins(void)
{
    InputPlugin *ip;
    OutputPlugin *op;
    EffectPlugin *ep;
    GeneralPlugin *gp;
    VisPlugin *vp;
    GList *node, *next;

    if (get_input_playing())
	input_stop();

    if (disabled_iplugins)
	g_list_free(disabled_iplugins);
    node = get_input_list();
    while (node) {
	ip = (InputPlugin *) node->data;
	if (ip && ip->cleanup) {
	    ip->cleanup();
	    GDK_THREADS_LEAVE();
	    while (g_main_iteration(FALSE));
	    GDK_THREADS_ENTER();

	}
	dynamic_lib_close(ip->handle);
	node = node->next;
    }
    if (ip_data->input_list)
	g_list_free(ip_data->input_list);
    g_free(ip_data);

    node = get_output_list();
    while (node) {
	op = (OutputPlugin *) node->data;
	dynamic_lib_close(op->handle);
	node = node->next;
    }
    if (op_data->output_list)
	g_list_free(op_data->output_list);
    g_free(op_data);

    node = get_effect_list();
    while (node) {
	ep = (EffectPlugin *) node->data;
	if (ep && ep->cleanup) {
	    ep->cleanup();
	    GDK_THREADS_LEAVE();
	    while (g_main_iteration(FALSE));
	    GDK_THREADS_ENTER();

	}
	dynamic_lib_close(ep->handle);
	node = node->next;
    }
    if (ep_data->effect_list)
	g_list_free(ep_data->effect_list);
    g_free(ep_data);

    node = get_general_enabled_list();
    while (node) {
	gp = (GeneralPlugin *) node->data;
	next = node->next;
	enable_general_plugin(g_list_index(gp_data->general_list, gp),
			      FALSE);
	node = next;
    }
    if (gp_data->enabled_list)
	g_list_free(gp_data->enabled_list);

    GDK_THREADS_LEAVE();
    while (g_main_iteration(FALSE));
    GDK_THREADS_ENTER();

    node = get_general_list();
    while (node) {
	gp = (GeneralPlugin *) node->data;
	dynamic_lib_close(gp->handle);
	node = node->next;
    }
    if (gp_data->general_list)
	g_list_free(gp_data->general_list);

    node = get_vis_enabled_list();
    while (node) {
	vp = (VisPlugin *) node->data;
	next = node->next;
	enable_vis_plugin(g_list_index(vp_data->vis_list, vp), FALSE);
	node = next;
    }
    if (vp_data->enabled_list)
	g_list_free(vp_data->enabled_list);

    GDK_THREADS_LEAVE();
    while (g_main_iteration(FALSE));
    GDK_THREADS_ENTER();

    node = get_vis_list();
    while (node) {
	vp = (VisPlugin *) node->data;
	dynamic_lib_close(vp->handle);
	node = node->next;
    }
    if (vp_data->vis_list)
	g_list_free(vp_data->vis_list);
    g_free(vp_data);

}
