/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2003  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2003  Haavard Kvaalen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*  A note about Pango and some funky spacey fonts: Weirdly baselined fonts, or fonts
 *  with weird ascents or descents _will_ display a little bit weird in the playlist
 *  widget, but the display engine won't make it look too bad, just a little deranged.
 *  I honestly don't think it's worth fixing (around...), it doesn't have to be perfectly
 *  fitting, just the general look has to be ok, which it IMHO is.
 *
 *  A second note: The numbers aren't perfectly aligned, but in the end it looks better when
 *  using a single Pango layout for each number.
 *  For who ever wants to see it done different, link playlist_list_pango_new.c to playlist.c
 *  and recompile. It monospaces the digits in any font (probably not technically, but basically
 *  similar to e.g. gnome-terminal) so that they align much better.
 *  But... for one: it's slower. And 2ndly, the numbers, when using some non straight font
 *  (see above), look much worse.
 */

/* FIXME: This is all still too slow. */

#include "beep.h"

#ifdef HAVE_WCHAR_H
#include <wchar.h>
#endif
#include <X11/Xatom.h>

#include <ft2build.h>
#include FT_FREETYPE_H

#include <pango/pangoft2.h>

PangoFontDescription *playlist_list_font = NULL;
guint ascent, descent, width_delta_digit_one;
gboolean has_slant;


// I know this is possible with pango_font_metrics_get_approximate_char_width(),
// but at this point of time i simply don't know how to use that. I had to work
// from include files only since i have no internet conn. at home. I'll check into
// it as soon as possible and fix it, but i needed something to work with.

guint width_approx_letters;
guint width_colon, width_colon_third;
guint width_approx_digits, width_approx_digits_half;

static int playlist_list_auto_drag_down_func(gpointer data)
{
    PlayList_List *pl = data;

    if (pl->pl_auto_drag_down) {
	playlist_list_move_down(pl);
	pl->pl_first++;
	playlistwin_update_list();
	return TRUE;

    }
    return FALSE;
}

static int playlist_list_auto_drag_up_func(gpointer data)
{
    PlayList_List *pl = data;

    if (pl->pl_auto_drag_up) {
	playlist_list_move_up(pl);
	pl->pl_first--;
	playlistwin_update_list();
	return TRUE;

    }
    return FALSE;
}

void playlist_list_move_up(PlayList_List * pl)
{
    GList *list;

    PLAYLIST_LOCK();
    if ((list = get_playlist()) == NULL) {
	PLAYLIST_UNLOCK();
	return;
    }
    if (((PlaylistEntry *) list->data)->selected) {
	/* We are at the top */
	PLAYLIST_UNLOCK();
	return;
    }
    while (list) {
	if (((PlaylistEntry *) list->data)->selected)
	    glist_moveup(list);
	list = g_list_next(list);
    }
    PLAYLIST_UNLOCK();
    if (pl->pl_prev_selected != -1)
	pl->pl_prev_selected--;
    if (pl->pl_prev_min != -1)
	pl->pl_prev_min--;
    if (pl->pl_prev_max != -1)
	pl->pl_prev_max--;
}

void playlist_list_move_down(PlayList_List * pl)
{
    GList *list;

    PLAYLIST_LOCK();
    if ((list = g_list_last(get_playlist())) == NULL) {
	PLAYLIST_UNLOCK();
	return;
    }
    if (((PlaylistEntry *) list->data)->selected) {
	/* We are at the bottom */
	PLAYLIST_UNLOCK();
	return;
    }
    while (list) {
	if (((PlaylistEntry *) list->data)->selected)
	    glist_movedown(list);
	list = g_list_previous(list);
    }
    PLAYLIST_UNLOCK();
    if (pl->pl_prev_selected != -1)
	pl->pl_prev_selected++;
    if (pl->pl_prev_min != -1)
	pl->pl_prev_min++;
    if (pl->pl_prev_max != -1)
	pl->pl_prev_max++;
}

void playlist_list_button_press_cb(GtkWidget * widget,
				   GdkEventButton * event,
				   PlayList_List * pl)
{
    if (event->button == 1 && pl->pl_fheight &&
	inside_widget(event->x, event->y, &pl->pl_widget)) {
	int nr, y;

	y = event->y - pl->pl_widget.y;
	nr = (y / pl->pl_fheight) + pl->pl_first;
	if (nr >= get_playlist_length())
	    nr = get_playlist_length() - 1;
	if (!(event->state & GDK_CONTROL_MASK))
	    playlist_select_all(FALSE);

	if (event->state & GDK_SHIFT_MASK && pl->pl_prev_selected != -1) {
	    playlist_select_range(pl->pl_prev_selected, nr, TRUE);
	    pl->pl_prev_min = pl->pl_prev_selected;
	    pl->pl_prev_max = nr;
	    pl->pl_drag_pos = nr - pl->pl_first;
	} else {
	    if (playlist_select_invert(nr)) {
		if (event->state & GDK_CONTROL_MASK) {
		    if (pl->pl_prev_min == -1) {
			pl->pl_prev_min = pl->pl_prev_selected;
			pl->pl_prev_max = pl->pl_prev_selected;
		    }
		    if (nr < pl->pl_prev_min)
			pl->pl_prev_min = nr;
		    else if (nr > pl->pl_prev_max)
			pl->pl_prev_max = nr;
		} else
		    pl->pl_prev_min = -1;
		pl->pl_prev_selected = nr;
		pl->pl_drag_pos = nr - pl->pl_first;
	    }
	}
	if (event->type == GDK_2BUTTON_PRESS) {
	    /*
	     * Ungrab the pointer to prevent us from
	     * hanging on to it during the sometimes slow
	     * playlist_play().
	     */
	    gdk_pointer_ungrab(GDK_CURRENT_TIME);
	    gdk_flush();
	    playlist_set_position(nr);
	    if (!get_input_playing())
		playlist_play();
	}


	pl->pl_dragging = TRUE;
	playlistwin_update_list();
    }
}

int playlist_list_get_playlist_position(PlayList_List * pl, int x, int y)
{
    int iy, length;

    if (!inside_widget(x, y, pl) || !pl->pl_fheight)
	return -1;

    if ((length = get_playlist_length()) == 0)
	return -1;
    iy = y - pl->pl_widget.y;

    return (MIN((iy / pl->pl_fheight) + pl->pl_first, length - 1));
}

void playlist_list_motion_cb(GtkWidget * widget, GdkEventMotion * event,
			     PlayList_List * pl)
{
    gint nr, y, off, i;

    if (pl->pl_dragging) {
	y = event->y - pl->pl_widget.y;
	nr = (y / pl->pl_fheight);
	if (nr < 0) {
	    nr = 0;
	    if (!pl->pl_auto_drag_up) {
		pl->pl_auto_drag_up = TRUE;
		pl->pl_auto_drag_up_tag =
		    gtk_timeout_add(100, playlist_list_auto_drag_up_func,
				    pl);
	    }
	} else if (pl->pl_auto_drag_up)
	    pl->pl_auto_drag_up = FALSE;

	if (nr >= pl->pl_num_visible) {
	    nr = pl->pl_num_visible - 1;
	    if (!pl->pl_auto_drag_down) {
		pl->pl_auto_drag_down = TRUE;
		pl->pl_auto_drag_down_tag =
		    gtk_timeout_add(100, playlist_list_auto_drag_down_func,
				    pl);
	    }
	} else if (pl->pl_auto_drag_down)
	    pl->pl_auto_drag_down = FALSE;

	off = nr - pl->pl_drag_pos;
	if (off) {
	    for (i = 0; i < abs(off); i++) {
		if (off < 0)
		    playlist_list_move_up(pl);
		else
		    playlist_list_move_down(pl);

	    }
	    playlistwin_update_list();
	}
	pl->pl_drag_pos = nr;
    }
}

void playlist_list_button_release_cb(GtkWidget * widget,
				     GdkEventButton * event,
				     PlayList_List * pl)
{
    pl->pl_dragging = FALSE;
    pl->pl_auto_drag_down = FALSE;
    pl->pl_auto_drag_up = FALSE;
}

#define playlist_list_draw_string_wc playlist_list_draw_string

void playlist_list_draw_string(PlayList_List * pl,
			       PangoFontDescription * font, gint line,
			       gint width, gchar * text, guint ppos)
{
    int len;
    int pix_len;
    char *tmp;
    guint padding, plist_length_int;
    PangoLayout *layout;

    if (cfg.convert_underscore)
	while ((tmp = strchr(text, '_')) != NULL)
	    *tmp = ' ';

    if (cfg.convert_twenty)
	while ((tmp = strstr(text, "%20")) != NULL) {
	    char *tmp2 = tmp + 3;
	    *(tmp++) = ' ';
	    while (*tmp2)
		*(tmp++) = *(tmp2++);
	    *tmp = '\0';
	}

    len = strlen(text);

    pix_len = (width_approx_letters * len);

    while (pix_len > width && len > 4) {
	len--;
	text[len] = '\0';
	pix_len = (width_approx_letters * len);
    }

    if (cfg.show_numbers_in_pl) {


	gchar *pos_string = g_strdup_printf("%d", ppos);
	plist_length_int =
	    strlen(g_strdup_printf("%d", get_playlist_length_nolock())) +
	    1;

	padding = plist_length_int;
	padding = ((padding + 1) * width_approx_digits);

	layout =
	    gtk_widget_create_pango_layout(GTK_WIDGET(playlistwin),
					   g_locale_to_utf8(pos_string, -1,
							    NULL, NULL,
							    NULL));
	pango_layout_set_font_description(layout, playlist_list_font);
	pango_layout_set_width(layout, plist_length_int * 100);

	if (atoi(pos_string) == 1) {
	    pango_layout_set_alignment(layout, PANGO_ALIGN_LEFT);
	    gdk_draw_layout(pl->pl_widget.parent, pl->pl_widget.gc,
			    pl->pl_widget.x + (width_delta_digit_one) +
			    (width_approx_digits *
			     (-1 + plist_length_int - strlen(pos_string))),
			    pl->pl_widget.y + (line - 1) * pl->pl_fheight +
			    ascent + abs(descent), layout);
	} else {
	    pango_layout_set_alignment(layout, PANGO_ALIGN_LEFT);
	    gdk_draw_layout(pl->pl_widget.parent, pl->pl_widget.gc,
			    pl->pl_widget.x +
			    (width_approx_digits *
			     (-1 + plist_length_int - strlen(pos_string))),
			    pl->pl_widget.y + (line - 1) * pl->pl_fheight +
			    ascent + abs(descent), layout);
	}
	g_object_unref(layout);
    } else {
	padding = 3;
    }

    layout =
	gtk_widget_create_pango_layout(GTK_WIDGET(playlistwin),
				       g_locale_to_utf8(text, -1, NULL,
							NULL, NULL));

    pango_layout_set_font_description(layout, playlist_list_font);
    gdk_draw_layout(pl->pl_widget.parent, pl->pl_widget.gc,
		    pl->pl_widget.x + padding,
		    pl->pl_widget.y + (line - 1) * pl->pl_fheight +
		    ascent + abs(descent), layout);

    g_object_unref(layout);
}

void playlist_list_draw(Widget * w)
{
    PlayList_List *pl = (PlayList_List *) w;
    GList *list;
    GdkGC *gc;
    GdkPixmap *obj;
    PangoLayout *layout;
    gchar *title;
    gint width, height;
    gint i, tw, max_first;
    guint padding;
    guint max_time_len = 0;

    gc = pl->pl_widget.gc;
    width = pl->pl_widget.width;
    height = pl->pl_widget.height;

    obj = pl->pl_widget.parent;

    gdk_gc_set_foreground(gc, get_skin_color(SKIN_PLEDIT_NORMALBG));
    gdk_draw_rectangle(obj, gc, TRUE, pl->pl_widget.x, pl->pl_widget.y,
		       width, height);

    if (playlist_list_font == NULL) {
	g_log(NULL, G_LOG_LEVEL_CRITICAL, "Couldn't open playlist font");
	return;
    }


    PLAYLIST_LOCK();
    list = get_playlist();

    pl->pl_fheight = (ascent + abs(descent));
    pl->pl_num_visible = height / pl->pl_fheight;

    max_first = g_list_length(list) - pl->pl_num_visible;
    if (max_first < 0)
	max_first = 0;
    if (pl->pl_first >= max_first)
	pl->pl_first = max_first;
    if (pl->pl_first < 0)
	pl->pl_first = 0;
    for (i = 0; i < pl->pl_first; i++)
	list = g_list_next(list);

    for (i = pl->pl_first;
	 list && i < pl->pl_first + pl->pl_num_visible;
	 list = list->next, i++) {
	char qstr[20] = "", length[40] = "";
	int pos;

	PlaylistEntry *entry = (PlaylistEntry *) list->data;
	if (entry->selected) {
	    gdk_gc_set_foreground(gc,
				  get_skin_color(SKIN_PLEDIT_SELECTEDBG));
	    gdk_draw_rectangle(obj, gc, TRUE, pl->pl_widget.x,
			       pl->pl_widget.y +
			       ((i - pl->pl_first) * pl->pl_fheight),
			       width, pl->pl_fheight);
	}
	if (i == __get_playlist_position())
	    gdk_gc_set_foreground(gc, get_skin_color(SKIN_PLEDIT_CURRENT));
	else
	    gdk_gc_set_foreground(gc, get_skin_color(SKIN_PLEDIT_NORMAL));

	if (entry->title)
	    title = g_strdup(entry->title);
	else
	    title = g_path_get_basename(entry->filename);

	if (!g_locale_to_utf8(title, -1, NULL, NULL, NULL))
	    title = g_path_get_basename(entry->filename);

	pos = playlist_get_queue_position(entry);

	if (pos != -1)
	    sprintf(qstr, "|%d|%s", pos + 1,
		    entry->length != -1 ? " " : "");

	if (entry->length != -1)
	    sprintf(length, "%d:%-2.2d", entry->length / 60000,
		    (entry->length / 1000) % 60);

	if (pos != -1 || entry->length != -1) {
	    int x, y;
	    gchar *tail;
	    guint t_width;
	    guint len_tail;
	    guint len;

	    tail = g_strdup_printf("%s%s", qstr, length);

	    if (strlen(tail) > max_time_len)
		max_time_len = strlen(tail);

	    /* FIXME: This is just an approximate alignment, maybe
	       something still fast, but exact could be done */

	    len_tail = strlen(tail);
	    len_tail -= 4;

	    t_width =
		((4 + len_tail) * width_approx_digits) +
		(width_approx_digits_half) + 2;
	    tw = width - t_width - 5;

	    if (i == __get_playlist_position())
		gdk_gc_set_foreground(gc,
				      get_skin_color(SKIN_PLEDIT_CURRENT));
	    else
		gdk_gc_set_foreground(gc,
				      get_skin_color(SKIN_PLEDIT_NORMAL));
	    playlist_list_draw_string(pl, playlist_list_font,
				      i - pl->pl_first, tw, title, i + 1);

	    x = pl->pl_widget.x + width - width_approx_digits * 2;
	    y = pl->pl_widget.y + ((i - pl->pl_first) -
				   1) * pl->pl_fheight + ascent;

	    if (entry->selected) {
		gdk_gc_set_foreground(gc,
				      get_skin_color
				      (SKIN_PLEDIT_SELECTEDBG));
	    } else {
		gdk_gc_set_foreground(gc,
				      get_skin_color
				      (SKIN_PLEDIT_NORMALBG));
	    }

	    /* This isn't very cool, but i don't see a way to calculate row widths with Pango fast enough here */
	    gdk_draw_rectangle(obj, gc, TRUE,
			       pl->pl_widget.x + pl->pl_widget.width -
			       (width_approx_digits * 6), y + abs(descent),
			       (width_approx_digits * 6),
			       pl->pl_fheight - 1);

	    if (i == __get_playlist_position())
		gdk_gc_set_foreground(gc,
				      get_skin_color(SKIN_PLEDIT_CURRENT));
	    else
		gdk_gc_set_foreground(gc,
				      get_skin_color(SKIN_PLEDIT_NORMAL));

	    len = strlen(tail);



	    gchar **frags = g_strsplit(tail, ":", 0);

	    layout =
		gtk_widget_create_pango_layout(GTK_WIDGET(playlistwin),
					       g_locale_to_utf8(frags[1],
								-1, NULL,
								NULL,
								NULL));
	    pango_layout_set_font_description(layout, playlist_list_font);
	    pango_layout_set_width(layout, strlen(tail) * 100);
	    pango_layout_set_alignment(layout, PANGO_ALIGN_LEFT);
	    gdk_draw_layout(obj, gc, x - (0.25 * width_approx_digits),
			    y + abs(descent), layout);
	    g_object_unref(layout);

	    layout =
		gtk_widget_create_pango_layout(GTK_WIDGET(playlistwin),
					       g_locale_to_utf8(g_strconcat
								(frags[0],
								 ":",
								 NULL), -1,
								NULL, NULL,
								NULL));
	    pango_layout_set_font_description(layout, playlist_list_font);
	    pango_layout_set_width(layout, strlen(tail) * 100);
	    pango_layout_set_alignment(layout, PANGO_ALIGN_RIGHT);
	    gdk_draw_layout(obj, gc, x - (0.5 * width_approx_digits),
			    y + abs(descent), layout);
	    g_object_unref(layout);



	} else {
	    tw = width;

	    if (i == __get_playlist_position())
		gdk_gc_set_foreground(gc,
				      get_skin_color(SKIN_PLEDIT_CURRENT));
	    else
		gdk_gc_set_foreground(gc,
				      get_skin_color(SKIN_PLEDIT_NORMAL));

	    playlist_list_draw_string(pl, playlist_list_font,
				      i - pl->pl_first, tw, title, i + 1);
	}

	g_free(title);
    }

    /*
     * Drop target hovering over the playlist, so draw some hint where the
     * drop will occur.
     *
     * This is (currently? unfixably?) broken when dragging files from Qt/KDE apps,
     * probably due to DnD signaling problems (actually i have no clue).
     *
     */

    if (pl->pl_drag_motion) {

	unsigned int pos, x, y, plx, ply, plength;

	/* We already hold the mutex and have the playlist locked, so call
	   the non-locking function. */
	plength = get_playlist_length_nolock();

	x = pl->drag_motion_x;
	y = pl->drag_motion_y;

	plx = pl->pl_widget.x;
	ply = pl->pl_widget.y;

	if ((x > pl->pl_widget.x) && !(x > pl->pl_widget.width)) {

	    if ((y > pl->pl_widget.y)
		&& !(y > (pl->pl_widget.height + ply))) {

		pos =
		    ((y - ((Widget *) pl)->y) / pl->pl_fheight) +
		    pl->pl_first;
		if (pos > (plength)) {
		    pos = plength;
		}

		gdk_gc_set_foreground(gc,
				      get_skin_color(SKIN_PLEDIT_CURRENT));

		gdk_draw_line(obj, gc,
			      pl->pl_widget.x,
			      pl->pl_widget.y +
			      ((pos - pl->pl_first) * pl->pl_fheight),
			      pl->pl_widget.width + pl->pl_widget.x,
			      pl->pl_widget.y +
			      ((pos - pl->pl_first) * pl->pl_fheight));
	    }

	}

	/* When dropping on the borders of the playlist, outside the text area,
	 * files get appended at the end of the list. Show that too.
	 */

	if ((y < ply) || (y > pl->pl_widget.height + ply)) {
	    if ((y >= 0) || (y <= (pl->pl_widget.height + ply))) {
		pos = plength;
		gdk_gc_set_foreground(gc,
				      get_skin_color(SKIN_PLEDIT_CURRENT));

		gdk_draw_line(obj, gc,
			      pl->pl_widget.x,
			      pl->pl_widget.y +
			      ((pos - pl->pl_first) * pl->pl_fheight),
			      pl->pl_widget.width + pl->pl_widget.x,
			      pl->pl_widget.y +
			      ((pos - pl->pl_first) * pl->pl_fheight));
	    }
	}


    }

    gdk_gc_set_foreground(gc, get_skin_color(SKIN_PLEDIT_NORMAL));

    if (cfg.show_numbers_in_pl) {
	padding =
	    -1 +
	    strlen(g_strdup_printf("%d", get_playlist_length_nolock())) +
	    1;
	padding =
	    (padding * width_approx_digits) + (width_approx_digits / 1.3);
	if (get_playlist_length_nolock() <= 0)
	    padding = 3;

	// For italic or oblique fonts we add another half of the approximate width
	if (has_slant)
	    padding += width_approx_digits_half;

	gdk_draw_line(obj, gc,
		      pl->pl_widget.x + padding,
		      pl->pl_widget.y,
		      pl->pl_widget.x + padding,
		      (pl->pl_widget.y + pl->pl_widget.height - 1)
	    );

    }


/*	guint xpos = pl->pl_widget.x + width - (width_approx_digits*max_time_len)+width_colon_third-8-width_approx_digits_half;
	gdk_draw_line(obj, gc, 
			xpos,
			pl->pl_widget.y,
			xpos,
			(pl->pl_widget.y + pl->pl_widget.height-1)
	);
*/

    PLAYLIST_UNLOCK();
}

PlayList_List *create_playlist_list(GList ** wlist, GdkPixmap * parent,
				    GdkGC * gc, gint x, gint y, gint w,
				    gint h)
{
    PlayList_List *pl;

    pl = (PlayList_List *) g_malloc0(sizeof(PlayList_List));
    pl->pl_widget.parent = parent;
    pl->pl_widget.gc = gc;
    pl->pl_widget.x = x;
    pl->pl_widget.y = y;
    pl->pl_widget.width = w;
    pl->pl_widget.height = h;
    pl->pl_widget.visible = TRUE;
    pl->pl_widget.button_press_cb =
	(void (*)(GtkWidget *, GdkEventButton *, gpointer))
	playlist_list_button_press_cb;
    pl->pl_widget.button_release_cb =
	(void (*)(GtkWidget *, GdkEventButton *, gpointer))
	playlist_list_button_release_cb;
    pl->pl_widget.motion_cb =
	(void (*)(GtkWidget *, GdkEventMotion *, gpointer))
	playlist_list_motion_cb;
    pl->pl_widget.draw = playlist_list_draw;
    pl->pl_prev_selected = -1;
    pl->pl_prev_min = -1;
    pl->pl_prev_max = -1;
    add_widget(wlist, pl);
    return pl;
}

void playlist_list_set_font(char *font)
{

    gchar *font_lower;
    guint width_temp;
    guint width_temp_0;

    playlist_list_font = pango_font_description_from_string(font);

    /* hack uber extreme pt. 2 */
    text_get_extents(font,
		     "AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz ",
		     &width_approx_letters, NULL, &ascent, &descent);
    width_approx_letters = (width_approx_letters / 53);

    // Experimental: We don't weigh the 1 into total because it's width is almost always
    // very different from the rest
    text_get_extents(font, "023456789", &width_approx_digits, NULL, NULL,
		     NULL);
    width_approx_digits = (width_approx_digits / 9);

    // Precache some often used calculations
    width_approx_digits_half = width_approx_digits / 2;

    // FIXME: We assume that any other number is broader than the "1"
    text_get_extents(font, "1", &width_temp, NULL, NULL, NULL);
    text_get_extents(font, "2", &width_temp_0, NULL, NULL, NULL);

    if (abs(width_temp_0 - width_temp) < 2) {
	width_delta_digit_one = 0;
    } else {
	width_delta_digit_one = ((width_temp_0 - width_temp) / 2) + 2;
    }

    text_get_extents(font, ":", &width_colon, NULL, NULL, NULL);
    width_colon_third = width_colon / 4;

    font_lower = g_utf8_strdown(font, strlen(font));
    // This doesn't take any i18n into account, but i think there is none with TTF fonts
    // FIXME: This can probably be retrieved trough Pango too
    has_slant = g_strstr_len(font_lower, strlen(font_lower), "oblique")
	|| g_strstr_len(font_lower, strlen(font_lower), "italic");

}
