/*
 * Here should go the usual boiler plate.
 * The stuff I've added if Copyright Jean Tourrilhes 2002.
 * As most of it is cut'n'paste from part of the SDP library, this code
 * is of course GPL...
 * Jean II
 */

/************************** DOCUMENTATION **************************/
/*
 * These function add attributes to SDP records. You can add simple
 * attributes, or simple data sequence of them (not yet implemented).
 * There is not much checking, and you may do bad thing, so please
 * use with caution...
 * Jean II
 */

/***************************** INCLUDES *****************************/

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <getopt.h>

#include "sdp.h"
#include "sdp_lib.h"
/* We need to access the low level structure of the attributes */
#include "sdp_internal.h"

/* This should be in some header */
extern int optind,opterr,optopt;
extern char *optarg;

/****************************** TYPES ******************************/

/**************************** CONSTANTS ****************************/

/**************************** PROTOTYPES ****************************/

#define for_each_opt(opt, long, short) while ((opt=getopt_long(argc, argv, short ? short:"+", long, NULL)) != -1)

/************************** SINGLE VALUES **************************/
/*
 * Set attributes with single values in SDP record
 * Jean II
 */
int set_attrib(sdp_session_t *sess, uint32_t handle, uint16_t attrib, char *value) 
{
	sdp_list_t *attrid_list;
	uint32_t range = 0x0000ffff;
	sdp_record_t *rec;
		
	/* Get the old SDP record */
	attrid_list = sdp_list_append(NULL, &range);
	rec = sdp_service_attr_req(sess, handle, SDP_ATTR_REQ_RANGE, attrid_list);

	if (!rec) {
		printf("Service get request failed.\n");
		return -1;
	}

	/* Check the type of attribute */
	if (!strncasecmp(value, "u0x", 3)) {
		/* UUID16 */
		uint16_t value_int = 0;
		uuid_t value_uuid;
		value_int = strtoul(value + 3, NULL, 16);
		sdp_uuid16_create(&value_uuid, value_int);
		printf("Adding attrib 0x%X uuid16 0x%X to record 0x%X\n",
		       attrib, value_int, handle);

		sdp_attr_add_new(rec, attrib, SDP_UUID16, &value_uuid.value.uuid16);
	} else if (!strncasecmp(value, "0x", 2)) {
		/* Int */
		uint32_t value_int;  
		value_int = strtoul(value + 2, NULL, 16);
		printf("Adding attrib 0x%X int 0x%X to record 0x%X\n",
		       attrib, value_int, handle);

		sdp_attr_add_new(rec, attrib, SDP_UINT32, &value_int);
	} else {
		/* String */
		printf("Adding attrib 0x%X string \"%s\" to record 0x%X\n",
		       attrib, value, handle);

		/* Add/Update our attribute to the record */
		sdp_attr_add_new(rec, attrib, SDP_TEXT_STR8, value);
	}

	/* Update on the server */
	if (sdp_record_update(sess, rec)) {
		printf("Service Record update failed (%d).\n", errno);
		return -1;
	}
	return 0;
}

static struct option set_options[] = {
	{"help",    0,0, 'h'},
	{0, 0, 0, 0}
};

static char *set_help = 
	"Usage:\n"
	"\tget record_handle attrib_id attrib_value\n";

/* ------------------------------------------------------------------- */
/*
 * Add an attribute to an existing SDP record on the local SDP server
 */
int cmd_setattr(int argc, char **argv)
{
	int opt, status;
	uint32_t handle;
	uint16_t attrib;
	sdp_session_t *sess;

	for_each_opt(opt, set_options, NULL) {
		switch(opt) {
		default:
			printf(set_help);
			return -1;
		}
	}
	argc -= optind;
	argv += optind;

	if (argc < 3) {
		printf(set_help);
		return -1;
	}

	/* Convert command line args */
	handle = strtoul(argv[0], NULL, 16);
	attrib = strtoul(argv[1], NULL, 16);

	/* Do it */
	sess = sdp_connect(BDADDR_ANY, BDADDR_LOCAL, 0);
	if (!sess)
		return -1;
	status = set_attrib(sess, handle, attrib, argv[2]);
	sdp_close(sess);
	return status;
}

/************************** DATA SEQUENCES **************************/
/*
 * We do only simple data sequences. Sequence of sequences is a pain ;-)
 * Jean II
 */
int set_attribseq(sdp_session_t *session, uint32_t handle, uint16_t attrib, int argc, char **argv)
{
	sdp_list_t *attrid_list;
	uint32_t range = 0x0000ffff;
	sdp_record_t *rec;
	sdp_data_t *pSequenceHolder = NULL;
	void **dtdArray;
	void **valueArray;
	void **allocArray;
	uint8_t uuid16 = SDP_UUID16;
	uint8_t uint32 = SDP_UINT32;
	uint8_t str8 = SDP_TEXT_STR8;
	int i;

	/* Get the old SDP record */
	attrid_list = sdp_list_append(NULL, &range);
	rec = sdp_service_attr_req(session, handle, SDP_ATTR_REQ_RANGE, attrid_list);

	if (!rec) {
		printf("Service get request failed.\n");
		return -1;
	}

	/* Create arrays */
	dtdArray = (void **)malloc(argc * sizeof(void *));
	valueArray = (void **)malloc(argc * sizeof(void *));
	allocArray = (void **)malloc(argc * sizeof(void *));

	/* Loop on all args, add them in arrays */
	for (i = 0; i < argc; i++) {
		/* Check the type of attribute */
		if (!strncasecmp(argv[i], "u0x", 3)) {
			/* UUID16 */
			uint16_t value_int = strtoul((argv[i]) + 3, NULL, 16);
			uuid_t *value_uuid = (uuid_t *)malloc(sizeof(uuid_t));
			allocArray[i] = value_uuid;
			sdp_uuid16_create(value_uuid, value_int);

			printf("Adding uuid16 0x%X to record 0x%X\n",
			       value_int, handle);
			dtdArray[i] = &uuid16;
			valueArray[i] = &value_uuid->value.uuid16;
		} else if (!strncasecmp(argv[i], "0x", 2)) {
			/* Int */
			uint32_t *value_int = (int *)malloc(sizeof(int));
			allocArray[i] = value_int;
			*value_int = strtoul((argv[i]) + 2, NULL, 16);

			printf("Adding int 0x%X to record 0x%X\n",
			       *value_int, handle);
			dtdArray[i] = &uint32;
			valueArray[i] = value_int;
		} else {
			/* String */
			printf("Adding string \"%s\" to record 0x%X\n",
			       argv[i], handle);
			dtdArray[i] = &str8;
			valueArray[i] = argv[i];
		}
	}

	/* Add this sequence to the attrib list */
	pSequenceHolder = sdp_seq_alloc(dtdArray, valueArray, argc);
	if (pSequenceHolder) {
		sdp_attr_replace(rec, attrib, pSequenceHolder);

		/* Update on the server */
		if (sdp_record_update(session, rec)) {
			printf("Service Record update failed (%d).\n", errno);
			return -1;
		}
	} else {
		printf("Failed to create pSequenceHolder\n");
	}

	/* Cleanup */
	for (i = 0; i < argc; i++)
		free(allocArray[i]);
	free(dtdArray);
	free(valueArray);

	return 0;
}

static struct option seq_options[] = {
	{"help",    0,0, 'h'},
	{0, 0, 0, 0}
};

static char *seq_help = 
	"Usage:\n"
	"\tget record_handle attrib_id attrib_values\n";

/* ------------------------------------------------------------------- */
/*
 * Add an attribute sequence to an existing SDP record
 * on the local SDP server
 */
int cmd_setseq(int argc, char **argv)
{
	int opt, status;
	uint32_t handle;
	uint16_t attrib;
	sdp_session_t *sess;

	for_each_opt(opt, seq_options, NULL) {
		switch(opt) {
		default:
			printf(seq_help);
			return -1;
		}
	}
	argc -= optind;
	argv += optind;

	if (argc < 3) {
		printf(seq_help);
		return -1;
	}

	/* Convert command line args */
	handle = strtoul(argv[0], NULL, 16);
	attrib = strtoul(argv[1], NULL, 16);

	argc -= 2;
	argv += 2;

	/* Do it */
	sess = sdp_connect(BDADDR_ANY, BDADDR_LOCAL, 0);
	if (!sess)
		return -1;
	status = set_attribseq(sess, handle, attrib, argc, argv);
	sdp_close(sess);
	return status;
}
