/*
  BlueNIC - Bluetooth PAN profile implementation for BlueZ
  Copyright (C) 2002 Sony Corporation

  Author: Johannes Loebbert <loebbert@sony.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License, version 2, as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
*/

/*
  ChangeLog:
  2001/05/29 created by Johannes Loebbert
*/

/*
 * $Id: pantest.c,v 1.2 2002/08/04 21:35:46 maxk Exp $
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <linux/sockios.h>
#include <net/if.h>
#include <sys/ioctl.h>

#include <bluetooth/bluetooth.h>

#include "bnep_common.h"

#define IOCTL_TARGET_NINF 0
#define IOCTL_TARGET_BNEP 1


#ifdef BNEP_TEST
/*********************************************/
/*********************************************/

int ioctl_packet_field;
int ioctl_packet_content;
int ioctl_packet_length;
int ioctl_packet_header_length;
int ioctl_packet_protocol;
int ioctl_packet_802_1p;
int ioctl_packet_type;
int ioctl_packet_switch;
int ioctl_packet_single_value;
int ioctl_packet_message;
char eth_interface[8];
bdaddr_t destination_address;
int source_uuid;
int destination_uuid;


static void usage(void)
{
	printf("pantest - PAN testing tool for IoP/UPF\n");
	printf("Usage:\n");
	printf("\tpantest -i <ethX> reset\n");
	printf("\tpantest -i <ethX> add -f <field> -c <content> -l <length>\n");
	printf("\t\t-h <header_length> -v <value>\n");
	printf("\t\t[-s source-service-uuid] [-d destination-service-uuid]\n");
	printf("\tpantest -i <ethX> send -D <dst_addr> -p <protocol> [-P]\n");
	printf("\t\t[-t bnep-packet-type]\n");
	printf("\tpantest -i <ethX> ignore -m <message_type> [-S <on|off>]\n");
	printf("\nSee man pantest for more information.\n");
}


/*********************************************/
/*********************************************/
int atoi_hex_aware(const char *opt)
{
	if (strlen(opt) > 2 && strncmp(opt, "0x", 2) == 0)
		return strtol(opt, (char **) NULL, 16);
	else
		return atoi(opt);
}

/*********************************************/
/*********************************************/
int bnept_ioctl_call(int command, void *opt_args, int opt_args_length, int target)
{
	struct bnep_test_ctrl_struct bnep_test_ctrl;

	int if_fd = 0;
	struct ifreq ifr;

	//Prepare IOCTL
	bnep_test_ctrl.test_command = command;
	bnep_test_ctrl.opt_args = opt_args;
	strcpy(ifr.ifr_name, eth_interface);
	ifr.ifr_data = (void *) &bnep_test_ctrl;

	if (target == IOCTL_TARGET_NINF) {
		strcpy(ifr.ifr_name, eth_interface);
		ifr.ifr_data = (void *) &bnep_test_ctrl;

		if ((if_fd = socket(AF_PACKET, SOCK_DGRAM, 0)) < 0) {
			perror("Can't get socket");
			exit(0);
		}

		if (ioctl(if_fd, SIOC_TESTCASE, &ifr)) {
			perror("Can't execute socket call");
			exit(0);
		}
	} else {
		if ((if_fd = socket(AF_PACKET, SOCK_RAW, 0)) < 0) {
			perror("Can't get socket");
			exit(1);
		}
		if (ioctl(if_fd, SIOC_TESTCASE, &ifr)) {
			perror("Can't execute socket call");
			exit(0);
		}
	}

	close(if_fd);
	return 0;
}

/*********************************************/
/*********************************************/
int bnept_ioctl_send(int argc, char *argv[], int optind)
{
	struct bnep_test_ioctl_send_struct bnep_test_ioctl_send;

	bacpy(&bnep_test_ioctl_send.destination_address, &destination_address);
	bnep_test_ioctl_send.packet_type = ioctl_packet_type;
	bnep_test_ioctl_send.protocol = ioctl_packet_protocol;
	bnep_test_ioctl_send.h802_1p = ioctl_packet_802_1p;

	//Send command has to be sent to a specific device in the next release
	return bnept_ioctl_call(TESTCASE_SEND, &bnep_test_ioctl_send, sizeof(struct bnep_test_ioctl_send_struct), IOCTL_TARGET_BNEP);
}

/*********************************************/
/*********************************************/
int bnept_ioctl_ignore_messages(int argc, char *argv[], int optind)
{
	struct bnep_test_ioctl_ignore_msg_struct bnep_test_ioctl_ignore_msg;
	bnep_test_ioctl_ignore_msg.msg_type = ioctl_packet_message;
	bnep_test_ioctl_ignore_msg.switch_value = ioctl_packet_switch;
	return bnept_ioctl_call(TESTCASE_IGNORE, &bnep_test_ioctl_ignore_msg, sizeof(struct bnep_test_ioctl_ignore_msg_struct), IOCTL_TARGET_BNEP);
}


/*********************************************/
/*********************************************/
int bnept_ioctl_reset_packet(int argc, char *argv[], int optind)
{
	return bnept_ioctl_call(TESTCASE_RESET, NULL, 0, IOCTL_TARGET_BNEP);
}

/*********************************************/
/*********************************************/
int setfilter_protocol(int argc, char *argv[], int optind, unsigned char *data)
{
	int value_counter = 0;
	uint16_t *protocol;
	uint8_t tmp;
	int value_count = argc - optind;

	for (value_counter = 0; value_counter < value_count; value_counter++) {
		protocol = (uint16_t *) data;
		*protocol = atoi_hex_aware(argv[optind + value_counter]);
		tmp = (uint8_t) data[1];
		data[1] = data[0];
		data[0] = tmp;
		data += 2;
	}
	return 0;
}

/*********************************************/
/*********************************************/
int setfilter_multicast(int argc, char *argv[], int optind, unsigned char *data)
{
	int value_counter = 0;
	int value_count = argc - optind;

	for (value_counter = 0; value_counter < value_count; value_counter++) {
		bacpy((bdaddr_t *) data, strtoba(argv[optind + value_counter]));
		data += sizeof(bdaddr_t);
	}
	return 0;
}

/*********************************************/
/*********************************************/
int bnept_ioctl_add(int argc, char *argv[], int optind)
{

	struct bnep_test_ioctl_add_field_struct bnep_test_ioctl_add_field;

	//Prepare common ioctl fields
	bnep_test_ioctl_add_field.content = ioctl_packet_content;
	bnep_test_ioctl_add_field.field = ioctl_packet_field;

	switch (ioctl_packet_content) {
	case TESTCASE_CONTENT_DATA:{
			//Prepare data for data payload field
			struct bnep_test_ioctl_data_opt_struct bnep_test_ioctl_data_opt;
			bnep_test_ioctl_add_field.add_opts_length = sizeof(struct bnep_test_ioctl_data_opt_struct);
			bnep_test_ioctl_data_opt.length = ioctl_packet_length;
			bnep_test_ioctl_data_opt.header_length = ioctl_packet_header_length;
			bnep_test_ioctl_add_field.add_opts = (void *) &bnep_test_ioctl_data_opt;
			break;
		}
	case TESTCASE_CONTENT_CONNECT:{
			//Prepare data for connect message in payload
			struct bnep_test_ioctl_connect_opt_struct bnep_test_ioctl_connect_opt;
			bnep_test_ioctl_connect_opt.source_uuid = source_uuid;
			bnep_test_ioctl_connect_opt.destination_uuid = destination_uuid;
			bnep_test_ioctl_add_field.add_opts = (void *) &bnep_test_ioctl_connect_opt;
			break;
		}
	case TESTCASE_CONTENT_CONTROL:{
			//Prepare data for control message
			struct bnep_test_ioctl_control_opt_struct bnep_test_ioctl_control_opt;
			bnep_test_ioctl_control_opt.control_value = ioctl_packet_single_value;
			bnep_test_ioctl_control_opt.length = ioctl_packet_length;
			bnep_test_ioctl_add_field.add_opts = (void *) &bnep_test_ioctl_control_opt;
			break;
		}
	case TESTCASE_CONTENT_UNKNOWN_HEADER:{
			//Prepare data for unknown extension header
			struct bnep_test_ioctl_unknown_header_opt_struct bnep_test_ioctl_unknown_header_opt;
			bnep_test_ioctl_add_field.field = TESTCASE_ADD_EXTENSION;
			bnep_test_ioctl_unknown_header_opt.control_value = ioctl_packet_single_value;
			bnep_test_ioctl_unknown_header_opt.length = ioctl_packet_length;
			bnep_test_ioctl_add_field.add_opts = (void *) &bnep_test_ioctl_unknown_header_opt;
			break;
		}
	case TESTCASE_CONTENT_PROTOCOL_FILTER:{
			struct filter_request_struct filter_request;
			int value_count = argc - optind;
			if ((value_count % 2)) {
				printf("Odd filter parameter\n");
				return 10;
			}
			filter_request.filter_type = FILTER_TYPE_PROTOCOL;
			filter_request.data_length = sizeof(uint16_t) * value_count;
			filter_request.data = (unsigned char *) malloc(filter_request.data_length);
			setfilter_protocol(argc, argv, optind, filter_request.data);
			bnep_test_ioctl_add_field.add_opts = (void *) &filter_request;
			break;
		}
	case TESTCASE_CONTENT_MULTICAST_FILTER:{
			struct filter_request_struct filter_request;
			int value_count = argc - optind;
			if ((value_count % 2)) {
				printf("Odd filter parameter\n");
				return 10;
			}
			filter_request.filter_type = FILTER_TYPE_MULTICAST;
			filter_request.data_length = sizeof(bdaddr_t) * value_count;
			filter_request.data = (unsigned char *) malloc(filter_request.data_length);
			setfilter_multicast(argc, argv, optind, filter_request.data);
			bnep_test_ioctl_add_field.add_opts = (void *) &filter_request;
			break;
		}
	}

	bnept_ioctl_call(TESTCASE_ADD, &bnep_test_ioctl_add_field, sizeof(struct bnep_test_ioctl_add_field_struct), IOCTL_TARGET_BNEP);
	return 0;
}

extern int optind, opterr, optopt;


/*********************************************/

int main(int argc, char *argv[])
{
	register int opt;

	//Prepare minimal packet info
	ioctl_packet_802_1p = H802_1P_OFF;
	ioctl_packet_type = BNEP_PACKET_TYPE_AUTO;
	ioctl_packet_content = TESTCASE_CONTENT_DATA;
	ioctl_packet_field = TESTCASE_ADD_PAYLOAD;
	ioctl_packet_length = 0;
	ioctl_packet_header_length = 0;
	ioctl_packet_switch = TESTCASE_IGNORE_MESSAGE_OFF;
	strcpy(eth_interface, "eth1");

	while ((opt = getopt(argc, argv, "f:c:l:h:p:Pt:s:S:v:m:i:d:D:")) != EOF) {
		switch (opt) {
		case 'f':
			if (strcasecmp(optarg, "PAYLOAD") == 0)
				ioctl_packet_field = TESTCASE_ADD_PAYLOAD;
			else if (strcasecmp(optarg, "EXTENSION") == 0)
				ioctl_packet_field = TESTCASE_ADD_EXTENSION;
			else {
				usage();
				exit(1);
			}
			break;
		case 'c':
			if (strcasecmp(optarg, "DATA") == 0)
				ioctl_packet_content = TESTCASE_CONTENT_DATA;
			else if (strcasecmp(optarg, "MULTICAST_FILTER") == 0)
				ioctl_packet_content = TESTCASE_CONTENT_MULTICAST_FILTER;
			else if (strcasecmp(optarg, "PROTOCOL_FILTER") == 0)
				ioctl_packet_content = TESTCASE_CONTENT_PROTOCOL_FILTER;
			else if (strcasecmp(optarg, "CONNECT") == 0)
				ioctl_packet_content = TESTCASE_CONTENT_CONNECT;
			else if (strcasecmp(optarg, "CONTROL") == 0)
				ioctl_packet_content = TESTCASE_CONTENT_CONTROL;
			else if (strcasecmp(optarg, "UNKNOWN") == 0)
				ioctl_packet_content = TESTCASE_CONTENT_UNKNOWN_HEADER;
			else {
				usage();
				exit(1);
			}
			break;

		case 'l':
			ioctl_packet_length = atoi_hex_aware(optarg);
			break;
		case 'D':
			bacpy(&destination_address, strtoba(optarg));
			break;
		case 'h':
			ioctl_packet_header_length = atoi_hex_aware(optarg);
			break;
		case 'p':
			ioctl_packet_protocol = atoi_hex_aware(optarg);
			break;
		case 'P':
			ioctl_packet_802_1p = H802_1P_ON;
			break;
		case 't':
			if (strcasecmp(optarg, "BNEP_GENERAL_ETHERNET") == 0)
				ioctl_packet_type = BNEP_GENERAL_ETHERNET;
			else if (strcasecmp(optarg, "BNEP_COMPRESSED_ETHERNET") == 0)
				ioctl_packet_type = BNEP_COMPRESSED_ETHERNET;
			else if (strcasecmp(optarg, "BNEP_COMPRESSED_ETHERNET_SOURCE_ONLY") == 0)
				ioctl_packet_type = BNEP_COMPRESSED_ETHERNET_SOURCE_ONLY;
			else if (strcasecmp(optarg, "BNEP_COMPRESSED_ETHERNET_DEST_ONLY") == 0)
				ioctl_packet_type = BNEP_COMPRESSED_ETHERNET_DEST_ONLY;
			else {
				usage();
				exit(1);
			}
			break;
		case 'S':
			if (strcasecmp(optarg, "ON") == 0)
				ioctl_packet_switch = TESTCASE_IGNORE_MESSAGE_ON;
			else if (strcasecmp(optarg, "OFF") == 0)
				ioctl_packet_switch = TESTCASE_IGNORE_MESSAGE_OFF;
			else {
				usage();
				exit(1);
			}
			break;
		case 'v':
			ioctl_packet_single_value = atoi_hex_aware(optarg);
			break;
		case 's':
			source_uuid = atoi_hex_aware(optarg);
			break;
		case 'd':
			destination_uuid = atoi_hex_aware(optarg);
			break;
		case 'i':
			strcpy(eth_interface, optarg);
			break;
		case 'm':
			if (strcasecmp(optarg, "CONNECT") == 0)
				ioctl_packet_message = TESTCASE_IGNORE_CONNECT;
			else if (strcasecmp(optarg, "PROTOCOL_FILTER") == 0)
				ioctl_packet_message = TESTCASE_IGNORE_NETFILTER;
			else if (strcasecmp(optarg, "MULTICAST_FILTER") == 0)
				ioctl_packet_message = TESTCASE_IGNORE_MULTICASTFILTER;
			else {
				usage();
				exit(1);
			}
			break;
		default:
			usage();
		}
	}

	if (!(argc - optind)) {
		usage();
		exit(1);
	}

	if (strcasecmp(argv[optind], "SEND") == 0) {
		bnept_ioctl_send(argc, argv, optind);
	} else if (strcasecmp(argv[optind], "ADD") == 0) {
		++optind;
		bnept_ioctl_add(argc, argv, optind);
	} else if (strcasecmp(argv[optind], "IGNORE") == 0) {
		bnept_ioctl_ignore_messages(argc, argv, optind);
	} else if (strcasecmp(argv[optind], "RESET") == 0) {
		bnept_ioctl_reset_packet(argc, argv, optind);
	} else {
		usage();
		exit(1);
	}

	exit(0);
}
#else
int main(int argc, char *argv[])
{
	printf("This command is not available.\n");
	printf("You need to configure with \"--enable-test\".\n");
	exit(0);
}
#endif
