/*
 *   MediaMVP Server Widget Library
 *
 *   (C) 2004 Dave Kelly
 *
 *   $Id: mvpwindow.h,v 1.13 2004/08/30 11:21:05 ravelox Exp $
 *   $Date: 2004/08/30 11:21:05 $
 *
 *   Window handling
 *
 */


#ifndef MVPWINDOW_H
#define MVPWINDOW_H

#ifdef __cplusplus
extern "C" {
#endif

#include "libmvp.h"
#include "mvpfont.h"

#include <stdio.h>     /* We need NULL */


typedef struct _window window_t;

#define FLAG_DIRTY		0x00000001
#define FLAG_FOCUS		0x00000002
#define FLAG_MODAL		0x00000004
#define FLAG_VISIBLE	0x00000008

#define WINDOW_ROOT		0
#define WINDOW_BOX		1
#define WINDOW_LABEL	2
#define WINDOW_MENU		3
#define WINDOW_DIALOG	4

#define EVENT_HANDLED	1

typedef enum {
	event_null = 0,
	event_select, 		/* OK Pressed */
	event_focus,		/* Window gets focus */
	event_blur,			/* Window loses focus */
	event_highlight,	/* Cursor key in menus*/
	event_keypress,
    event_delete,       /* Window is being deleted */
    event_draw,         /* Window is being redrawn, this event is called
                           before the contents are drawn 
                        */
    event_position,     /* Position on screen has changed */
    event_size,         /* Window size has changed */
	event_yes,			/* Yes option selected on dialog box */
	event_no,			/* No option selected on dialog box */
	event_cancel,		/* Cancel option selected on dialog box */
} event_t;


typedef	void (*event_cb_t)(window_t *window, event_t event, void *cb_val, void *param);


/* Basic Application level API */
extern window_t *new_root(int width, int height, int bpp);
extern window_t *new_window(window_t *parent, int type, int x, int y, int width, int height);
extern void      delete_window(window_t *window);

extern void      window_register_callback(window_t *window, event_t event, event_cb_t event_callback, void *cb_ptr);


/* Window base property API */
extern void      window_set_fill_col(window_t *window, uint32_t fill_col);
extern void      window_set_border_col(window_t *window, uint32_t border_col);
extern void      window_set_border_size(window_t *window, int border_size);
extern void      window_set_font(window_t *window, font_t *font);
extern void      window_set_fgcol(window_t *window, uint32_t fgcol);
extern void      window_set_bgcol(window_t *window, uint32_t bgcol);
extern uint32_t  window_get_fill_col(window_t *window);
extern uint32_t  window_get_border_col(window_t *window);
extern int       window_get_border_size(window_t *window);
extern font_t   *window_get_font(window_t *window);
extern uint32_t  window_get_fgcol(window_t *window);
extern uint32_t  window_get_bgcol(window_t *window);
extern void		 window_set_focus(window_t *window);

extern void      window_set_name(window_t *window, char *name);
extern char     *window_get_name(window_t *window);


#ifdef TESTING
extern void      window_dump_window(window_t *window);
#endif

extern surface_t *window_get_draw_surface(window_t *window);
extern void       window_set_draw_surface(window_t *window, surface_t *surface);

/* Window size API */
extern int       window_get_xpos(window_t *window);
extern void      window_set_xpos(window_t *window, int x);
extern int       window_get_ypos(window_t *window);
extern void      window_set_ypos(window_t *window, int y);
extern void      window_set_pos(window_t *window, int x, int y);
extern int       window_get_height(window_t *window);
extern void      window_set_height(window_t *window, int height);
extern int       window_get_width(window_t *window);
extern void      window_set_width(window_t *window, int width);
extern void      window_set_dimension(window_t *window, int width, int height);

/* Box */

extern window_t  *new_box(window_t *parent, int x, int y, int width, int height);

/* Label */


#define LABEL_ALIGN_LEFT	0x00000001
#define LABEL_ALIGN_CENTRE	0x00000002
#define LABEL_ALIGN_RIGHT	0x00000004

#define LABEL_VALIGN_TOP	0x00000001
#define LABEL_VALIGN_MIDDLE	0x00000002
#define LABEL_VALIGN_BOTTOM 0x00000004

extern window_t  *new_label(window_t *parent, int x, int y, int width, int height, char *text);

extern void       label_set_align(window_t *label, int align);
extern void       label_set_valign(window_t *label, int valign);
extern void       label_set_text(window_t *label, char *text);

/* Menu */
extern window_t  *new_menu(window_t *parent, int x, int y, int width, int height);
extern void       menu_add_item(window_t *window, char *item);
extern void       menu_set_item_sel_fgcol(window_t *window, uint32_t fgcol);
extern void       menu_set_item_sel_bgcol(window_t *window, uint32_t bgcol);
extern void       menu_set_item_unsel_fgcol(window_t *window, uint32_t fgcol);
extern void       menu_set_item_unsel_bgcol(window_t *window, uint32_t bgcol);
extern int        menu_get_num_items(window_t *window);
extern int        menu_get_sel(window_t *window);
extern void       menu_set_sel(window_t *menu, int sel, bool_t wrap);
extern void       menu_set_title(window_t *menu, char *title);
extern void       menu_set_colour_text(window_t *menu, char *red, char *green, char *yellow, char *blue);

/* Dialog */
#define DIALOG_YES		0x01
#define DIALOG_NO		0x02
#define DIALOG_CANCEL	0x04
extern window_t *new_dialog(window_t *parent, int x, int y, int width, int height, char *text, int flags);

/* Hooks into the window api for writing additional widgets */

/* Definitions for basic attrib function callbacks */
typedef struct  {
	int type;
	void (*set_fill_col)(window_t *window, uint32_t fill_col); 
	void (*set_border_col)(window_t *window, uint32_t border_col); 
	void (*set_border_size)(window_t *window, int border_size);
	void (*set_font)(window_t *window, font_t *font);
	void (*set_fgcol)(window_t *window, uint32_t fgcol);
	void (*set_bgcol)(window_t *window, uint32_t bgcol);
	uint32_t (*get_fill_col)(window_t *window);
	uint32_t (*get_border_col)(window_t *window);
	int (*get_border_size)(window_t *window);
	font_t *(*get_font)(window_t *window);
	uint32_t (*get_fgcol)(window_t *window);
	uint32_t (*get_bgcol)(window_t *window);
} attrib_fns_t;


extern void       window_set_attrib_fns(window_t *window, attrib_fns_t *attrib_fns);

extern void       window_set_data(window_t *window, void *data);
extern void      *window_get_data(window_t *window);
extern void       window_set_draw_func(window_t *window, void (*draw_func)(window_t *this));
extern void       window_set_focus_func(window_t *window, void (*draw_func)(window_t *this));
extern void       window_set_delete_func(window_t *window, void (*delete_func)(window_t *this));
extern void       window_set_event_func(window_t *window, int (*event_func)(window_t *this, int keycode, void *appdata));

extern void       window_set_type(window_t *window, int type);
extern int        window_get_type(window_t *window);

extern int        window_get_xoffset(window_t *window);
extern void       window_set_xoffset(window_t *window, int xoffset);
extern int        window_get_yoffset(window_t *window);
extern void       window_set_yoffset(window_t *window, int yoffset);
extern void       window_set_offset(window_t *window, int xoffset, int yoffset);

extern void       window_add_child(window_t *parent, window_t *child);
extern int        window_draw(window_t *window);
extern surface_t *window_get_contents(window_t *window);
extern void       window_set_pixel(window_t *window, int x, int y, uint32_t col);
extern void       window_clear(window_t *window, uint32_t col);
extern void       window_mark_dirty(window_t *window);
extern void       window_mark_clean(window_t *window);
extern int        window_is_dirty(window_t *window);

extern void       window_set_modal(window_t *window, int modal);
extern int        window_is_modal(window_t *window);

extern int        window_is_visible(window_t *window);
extern void       window_set_visible(window_t *window, int visible);
extern void       window_dispatch_event(window_t *window, int keycode);
extern void       window_raise_event(window_t *window, event_t event, void *param);

#ifdef __cplusplus
}
#endif


#endif /* MVPWINDOW_H */
