/*
 *   MediaMVP Server Widget Library
 *
 *   (C) 2004 Dave Kelly
 *
 *   $Id: dialog.c,v 1.1 2004/08/30 08:55:24 ravelox Exp $
 *   $Date: 2004/08/30 08:55:24 $
 *
 *   Window handling
 *
 */


#include "mvpwindow.h"

static void      dialog_set_border_size(window_t *dialog, int border_size);
static void      dialog_set_border_col(window_t *dialog, uint32_t border_col);
static void      dialog_set_fill_col(window_t *dialog, uint32_t fill_col);
static void      dialog_set_font(window_t *dialog, font_t *font);

static uint32_t  dialog_get_fill_col(window_t *window);
static uint32_t  dialog_get_border_col(window_t *window);
static int       dialog_get_border_size(window_t *window);
static font_t   *dialog_get_font(window_t *window);

static void      draw_dialog(window_t *dialog);
static void      delete_dialog_data(window_t *dialog);
static int       dialog_handle_event(window_t *window, int keycode, void *appdata);

static attrib_fns_t dialog_fns = {
	0,
	dialog_set_fill_col,
	dialog_set_border_col,
	dialog_set_border_size,
	dialog_set_font,
	NULL,
	NULL,
	dialog_get_fill_col,
	dialog_get_border_col,
	dialog_get_border_size,
	dialog_get_font,
	NULL,
	NULL
};

typedef struct {
	int flags;
	font_t *display_font;
	window_t *box;
	window_t *dialog_label;
	int num_options;
	window_t **option_labels;
} dialog_t;

window_t *new_dialog(window_t *parent, int x, int y, int width, int height, char *text, int flags)
{
	window_t *window;
	dialog_t *dialog;
	int font_height;
	int dialog_label_height = 0;
	int dialog_option_width = 0;
	int border_size = 0;
	int i;

	window = new_window(parent, WINDOW_DIALOG, x, y, width, height);
	window_set_attrib_fns(window, &dialog_fns);

	dialog = (dialog_t *)malloc(sizeof(dialog_t));

	dialog->flags = flags;

	dialog->display_font = font_read(default_font);

	window_set_name(window, "%internal% dialog");
	window_set_data(window, dialog);

	window_set_draw_func(window, draw_dialog);
	window_set_delete_func(window, delete_dialog_data);
	window_set_event_func(window, dialog_handle_event);

	dialog->box = new_box(window,0,0, width, height);
	window_set_name(dialog->box, "%internal% dialog box");
	window_set_border_size(dialog->box, 2);
	window_set_border_col(dialog->box, 0xFFFFFF);
	window_set_fill_col(dialog->box, 0x0000FF);

	font_height = font_get_height(dialog->display_font);

	border_size = window_get_border_size(dialog->box);

	dialog_label_height = height - (2 * border_size) - (flags > 0 ? font_height : 0);

	dialog->dialog_label = new_label(dialog->box, border_size, border_size, (width - (2 * border_size)), dialog_label_height, text);
	label_set_align(dialog->dialog_label, LABEL_ALIGN_CENTRE);
	label_set_valign(dialog->dialog_label, LABEL_VALIGN_MIDDLE);

	dialog->num_options = 0;
	dialog->option_labels = NULL;

	if(flags & DIALOG_YES) {
		dialog->option_labels = (window_t **)realloc(dialog->option_labels, (dialog->num_options + 1) * sizeof(window_t *));
		dialog->option_labels[dialog->num_options] = new_label(dialog->box, 0 ,dialog_label_height + border_size, 0, font_height, "Yes");
		window_set_bgcol(dialog->option_labels[dialog->num_options], 0x00FF00);
		window_set_fgcol(dialog->option_labels[dialog->num_options], 0x000000);
		dialog->num_options++;
	}

	if(flags & DIALOG_NO) {
		dialog->option_labels = (window_t **)realloc(dialog->option_labels, (dialog->num_options + 1) * sizeof(window_t *));
		dialog->option_labels[dialog->num_options] = new_label(dialog->box, 0,dialog_label_height + border_size, 0, font_height, "No");
		window_set_bgcol(dialog->option_labels[dialog->num_options], 0xFF0000);
		window_set_fgcol(dialog->option_labels[dialog->num_options], 0x000000);
		dialog->num_options++;
	}

	if(flags & DIALOG_CANCEL) {
		dialog->option_labels = (window_t **)realloc(dialog->option_labels, (dialog->num_options + 1) * sizeof(window_t *));
		dialog->option_labels[dialog->num_options] = new_label(dialog->box, 0,dialog_label_height + border_size, 0, font_height, "Cancel");
		window_set_bgcol(dialog->option_labels[dialog->num_options], 0xFFFF00);
		window_set_fgcol(dialog->option_labels[dialog->num_options], 0x000000);
		dialog->num_options++;
	}

	dialog_option_width = (width - (2 * border_size)) / dialog->num_options;
	for(i = 0 ; i < dialog->num_options; i++) {
		window_set_width(dialog->option_labels[i], dialog_option_width);
		window_set_xpos(dialog->option_labels[i], border_size + (i * dialog_option_width));
		label_set_align(dialog->option_labels[i], LABEL_ALIGN_CENTRE);
		label_set_valign(dialog->option_labels[i], LABEL_VALIGN_MIDDLE);
	}

	window_mark_dirty(dialog->box);
	window_mark_dirty(window);

	return window;
}

static void dialog_set_border_size(window_t *dialog, int border_size)
{
	dialog_t *data;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return;
	}

	window_set_border_size(data->box, border_size);
}

static void dialog_set_border_col(window_t *dialog, uint32_t border_col)
{
	dialog_t *data;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return;
	}

	window_set_border_col(data->box , border_col);
}

static void dialog_set_fill_col(window_t *dialog, uint32_t fill_col)
{
	dialog_t *data;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return;
	}

	window_set_fill_col(data->box, fill_col);
}

static void dialog_set_font(window_t *dialog, font_t *font)
{
	dialog_t *data;
	int i=0;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return;
	}

	if(! font) {
		return;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return;
	}

	data->display_font = font;

	for(i=0 ; i < data->num_options; i++) {
		window_set_font(data->option_labels[i], data->display_font);
	}

	window_set_font(data->dialog_label, data->display_font);

	window_mark_dirty(dialog);
}

static uint32_t dialog_get_border_col(window_t *dialog)
{
	dialog_t *data;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return 0;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return 0;
	}

	return window_get_border_col(data->box);
}

static uint32_t dialog_get_fill_col(window_t *dialog)
{
	dialog_t *data;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return 0;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return 0;
	}

	return window_get_fill_col(data->box);
}

static int dialog_get_border_size(window_t *dialog)
{
	dialog_t *data;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return 0;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return 0;
	}

	return window_get_border_size(data->box);
}

static font_t *dialog_get_font(window_t *dialog)
{
	dialog_t *data;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return NULL;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return NULL;
	}

	return data->display_font;
}

static void delete_dialog_data(window_t *dialog)
{
	dialog_t *data;

	if(window_get_type(dialog) != WINDOW_DIALOG) {
		return;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return;
	}

	free(data->option_labels);

	return;
}

static void draw_dialog(window_t *dialog)
{
	dialog_t *data;

	fprintf(stderr, "draw_dialog\n");
	if( window_get_type(dialog) != WINDOW_DIALOG) {
		return;
	}

	if(! (data = (dialog_t *)window_get_data(dialog)) ) {
		return;
	}
}

static int dialog_handle_event(window_t *window, int keycode, void *appdata)
{
	dialog_t *data;
	int return_state = 0;

	if(window_get_type(window) != WINDOW_DIALOG) {
		return return_state;
	}


	if(! (data = (dialog_t *)window_get_data(window))) {
		return return_state;
	}

	switch(keycode) {
		case keyGreen:
			if(data->flags & DIALOG_YES) {
				window_raise_event(window, event_yes, NULL);
				return_state = EVENT_HANDLED;
			}
			break;
		case keyRed:
			if(data->flags & DIALOG_NO) {
				window_raise_event(window, event_no, NULL);
				return_state = EVENT_HANDLED;
			}
			break;
		case keyYellow:
			if(data->flags & DIALOG_CANCEL) {
				window_raise_event(window, event_cancel, NULL);
				return_state = EVENT_HANDLED;
			}
			break;
	}

	return return_state;
}
