/*
 *   Small .ini file reader for z88dk/anything
 *
 *   Will parse those Windows style files, with basic functionality -
 *   handles booleans, ints and strings
 *
 *   (C) D.J.Morris <dom@jadara.org.uk> 7/8/2001
 *
 */



#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "libini.h"


/* Maximum length of line to read in from the file*/

#define LINE_MAX 128


struct _option {
        char               *group;
        char               *word;
        int             id;
        unsigned char        type;
        void               *value;
        struct _option *next;
};

typedef struct _option option_t;

static option_t   *list;
static char          *current_section;
static int         current_id;

/* Internal functions */

static char       *skpws(char *ptr);
static void           strip_ws(char *ptr);
static void        get_section(char *ptr);
static void        option_parse(char *option, char *text);
static void        option_set(char *opt, char *it);
static void        option_do_set(option_t *option, char *value);
static void        iniread_file(FILE *fp);



void iniparse_cleanup()
{
    option_t *next;
    option_t *opt = list;

    while ( opt != NULL ) {
        free(opt->group);
        free(opt->word);
        next = opt->next;
        free(opt);
        opt = next;
    }
}


int iniparse_init()
{
    current_section = strdup("");
    current_id = 0;
    list = NULL;

    return 0;
}

int iniparse_file(char *name)
{
    FILE        *fp;

    fp = fopen(name,"r");

    if ( fp == NULL ) {
        return (-1);
    }
    iniread_file(fp);
    fclose(fp);
    return 0;
}

int iniparse_add(char *key2, unsigned char type, void *data)
{
    option_t *option;
    char         *word;
    char         *key = strdup(key2);
    
    word = strchr(key,':');
    
    if ( word == NULL ) {
        free(key);
        return -1;
    }
    
    *word++ = 0;
    
    option = malloc(sizeof(option_t));
    
    option->next = list;
    list = option;


    option->group  = strdup(key);
    option->word   = strdup(word);
    option->type   = type;
    option->value  = data;
#if 0
    switch ( type ) {
    case OPT_BOOL:
        *(char *)data = 0;
        break;
    case OPT_INT:
        *(int *) data = 0;
        break;
    case OPT_STR:
        *(char **) data = NULL;
        break;
    }
#endif
    free(key);
    return 0;
}

void iniread_file(FILE *fp)
{
    char        buffer[LINE_MAX];
    char   *start,*ptr;
    
    while (fgets(buffer,sizeof(buffer),fp) != NULL ) {
        start = skpws(buffer);
        if ( *start == ';' || *start =='#' || *start ==0)
            continue;
        if ( *start == '[' ) {
            get_section(start);
        } else {        /* Must be an option! */
            ptr = start;
            while ( isalnum(*ptr) )
                ptr++;
            *ptr++ = 0;
            if ( strlen(start) ) 
                option_parse(start,ptr);
        }
    }
}

void option_parse(char *option, char *text)
{
    char        *ptr;
    
    ptr = strchr(text,'=');
    if ( ptr == NULL)
        return;
    ptr++;
    ptr = skpws(ptr);        /* ptr now points to text */
    if ( *ptr == ';' || *ptr =='#' || *ptr ==0)
        return;
    /* Now we have some valid stuff */
    option_set(option,ptr);
}

void option_set(char *opt, char *it)
{
    option_t    *option;
    
    option = list;

    while ( option != NULL ) {
        if ( strcmp(option->group,current_section) == 0 &&
             strcmp(option->word,opt) == 0 ) {
            option_do_set(option,it);
            return;
        }
        option = option->next;
    }
}
        
void option_do_set(option_t *option, char *value)
{
    int         val;
    char        c;
    char       *ptr;
        char *temp;

    /* Handle comments after the value */
    ptr = strchr(value,';');
    if ( ptr )
        *ptr = 0;
    switch ( option->type ) {
    case OPT_INT:
        val = atoi(value);
        *(int *)(option->value) = val;
        return;
    case OPT_STR:
        strip_ws(value);
        temp = strdup(value);
        *(char **)(option->value) = strdup(value);
        return;
    case OPT_BOOL:
        c = toupper(value[0]);
        val = 0;
        switch (c) {
        case 'Y':
        case 'T':
        case '1':
            val = 1;
        }
        *(char *)(option->value) = val;
    }
}



/* enters in with ptr pointing to the '[' */

void get_section(char *ptr)
{
    char *end; 
    end = strchr(ptr,']');
    
    if ( end == NULL )
        return;
    *end = 0;
   
    free(current_section);
    current_section = strdup(ptr+1);
}


void strip_ws(char *value)
{
    value = value+strlen(value)-1;
    while (*value && isspace(*value)) 
        *value-- = 0;
}

char *skpws(char *ptr)
{
        while (*ptr && isspace(*ptr) )
                ptr++;
        return (ptr);
}


#ifdef TEST
#ifdef SMALL_C
#define HPSIZE 15000
HEAPSIZE(HPSIZE)
#endif
int main()
{
    char *ptr;
    int   i,ret;
#ifdef SMALL_C
    heapinit(HPSIZE);
#endif
    iniparse_init();
    iniparse_add("test:blah",OPT_STR,&ptr);
    iniparse_add("test:count",OPT_INT,&i);
    ret = iniparse_file("test.ini");
    iniparse_cleanup();
    printf("%d <%s> %d\n",ret,ptr,i);

}
#endif

