/*
 *   MediaMVP Server
 *
 *   (C) 2003 Dominic Morris
 *
 *   $Id: gui.c,v 1.23 2003/11/24 21:52:00 dom Exp $
 *   $Date: 2003/11/24 21:52:00 $
 *
 *
 *   Handles the GUI...
 */

#include "mediamvp.h"
#include "gui.h"

#define NEED_MSGS
#include "msgs.h"

typedef struct {
    void    *param;
    int    (*ack_func)(int acktype, void *param, uint8_t *buf, int len);
} ackhandle_t;

typedef struct {
    void    *param;
    int    (*key_func)(void *menu,int keycode); /* Key press callback */
} keyhandle_t;

struct _gui {
    dongle_t     *dongle;
    struct event  event;
    int           fd;
    int           state;
    uint8_t       send_update;      /* Send an update out when necessary */
    render_t     *render;           /* Rendering surface */
    keyhandle_t   menus;
    keyhandle_t   media;
    ackhandle_t   menus_ack;
    ackhandle_t   media_ack;
    app_t        *app;              /* Controlling application */
    char          inbuf[1024];
    int           inbufstart;
};


static void       gui_accept(int fd, short event, void *arg);
static void       gui_read(int fd, short event, void *arg);
static void       rfb_send_data(gui_t *gui);
static int        rfb_running(gui_t *gui, unsigned char *buf, int len);
static void       rfb_send_screen(gui_t *gui);
static int        deflateit(Byte *uncompr, uLong uncomprLen, Byte *compr, uLong comprLen) ;



static struct event listen_event;

static int        c_width        = RFB_WIDTH;
static int        c_depth        = RFB_DEPTH;
static int        c_totwidth     = RFB_WIDTH;
static int        c_totdepth     = RFB_DEPTH;

int               c_gui_port     = 5906;
char             *c_gui_host     = NULL;
uint32_t          c_gui_hostip;

void gui_config()
{
    iniparse_add("gui:port",OPT_INT,&c_gui_port);
    iniparse_add("gui:host",OPT_STR,&c_gui_host);
    iniparse_add("gui:width",OPT_INT,&c_width);
    iniparse_add("gui:depth",OPT_INT,&c_depth);
    iniparse_add("gui:screenwidth",OPT_INT,&c_totwidth);
    iniparse_add("gui:screendepth",OPT_INT,&c_totdepth);
}


void gui_init()
{
    int   sock;

    if ( c_gui_host ) {
        c_gui_hostip = host_resolve(c_gui_host);
        return;
    } else {
        c_gui_hostip = main_interfaceip;
    }


    if ( (sock = tcp_listen(NULL,c_gui_port,0) ) == -1 ) {
        perror("gui");
        exit(1);
    }

    event_set(&listen_event,sock,EV_READ,gui_accept,&listen_event);

    event_add(&listen_event,NULL);

}

gui_t *new_gui(dongle_t *dongle)
{
    gui_t  *gui;

    gui = malloc(sizeof(*gui));
    dongle->gui = gui;

    gui->dongle = dongle;
    gui->state  = VNC_VERSION;
    gui->render = new_render(c_width,c_depth);
    gui->app    = new_app(dongle,gui->render,gui);
    gui->inbufstart = 0;


    return gui;
}

void delete_gui(gui_t *gui)
{
    event_del(&gui->event);
    close(gui->fd);
    shutdown(gui->fd,SHUT_RDWR);
    delete_render(gui->render);
    delete_app(gui->app);
    free(gui);
}

static void gui_accept(int fd, short event, void *arg)
{
    struct event  *ev = arg;
    dongle_t      *dongle;
    gui_t         *gui;
    char          *hostname;
    int            cfd;

    /* Reschedule event */
    event_add(ev,NULL);


    if ( ( cfd = tcp_accept(fd,&hostname) ) == -1 ) {
        return;
    }
    printf("Accept GUI\n");
    dongle = dongle_return(hostname);

    gui = new_gui(dongle);

    gui->fd = cfd;

    event_set(&gui->event,cfd,EV_READ,gui_read,gui);
    event_add(&gui->event,NULL);
    rfb_send_data(gui);      /* Send out the protocol info */
}



static void gui_read(int fd, short event, void *arg)
{
    gui_t       *gui = arg;
    int          r,diff = 0 ;

    /* Reschedule event */
    event_add(&gui->event,NULL);

    r = read(fd,gui->inbuf + gui->inbufstart,sizeof(gui->inbuf)-gui->inbufstart);

    /* Socket closed/error */
    if ( r == 0 || r == -1 ) {
        dongle_close(gui->dongle);
        return;
    }
  
    /* We only expect to get one message a time */

    gui->inbufstart += r;

    while ( gui->inbufstart > 0 ) {
        switch( gui->state ) {
        case VNC_VERSION:
        case VNC_VERSION2:
            if ( gui->inbufstart >= 12 ) {
                gui->state = VNC_AUTH;
                diff = 12;
            } else {
                diff = 0;
            }	   
            break;
        case VNC_AUTH:
        case VNC_AUTH2:
            diff = 1;     /* Ignore auth stuff */
            gui->state = VNC_INIT;
            break;
        case VNC_RUNNING:
            diff = rfb_running(gui,gui->inbuf,gui->inbufstart);
            break;
        default:
        }
        if ( diff <= 0 ) {
            break;
        } else if ( diff > 0 ) {
            int  rem = gui->inbufstart - diff;
            if ( rem >= 0 ) {
                memmove(gui->inbuf,gui->inbuf+diff,rem);
                gui->inbufstart = rem;
            } else {
                gui->inbufstart = 0;
            }
        }          
    }

    /* Send out pending messages */
    if ( diff != -1 ) {
        rfb_send_data(gui);
    }


}

/** \brief Handle incoming RFB messages from the client
 *
 * \param gui Handle to the gui channel
 * \param buf Incoming message buffer
 * \param len Length of the incoming message buffer
 *
 * \return Number of bytes processed
 */ 
static int rfb_running(gui_t *gui, unsigned char *buf, int len)
{
    uint8_t  *ptr = buf + 2;
    int       diff = 0;

    switch ( buf[0] ) {
    case RFB_SET_PIXEL_FORMAT:
        if ( len >= 20 ) {
            uint8_t   bpp;
            uint8_t   depth;
            uint8_t   endian;
            uint8_t   truecolour;
            uint16_t  redmax;
            uint16_t  greenmax;
            uint16_t  bluemax;
            uint8_t   redshift;
            uint8_t   greenshift;
            uint8_t   blueshift;
	
            ptr += 2;  /* Skip padding */

            bpp = *ptr++;
            depth = *ptr++;
            endian = *ptr++;
            truecolour = *ptr++;
            BUF_TO_INT16(redmax,ptr);
            BUF_TO_INT16(greenmax,ptr);
            BUF_TO_INT16(bluemax,ptr);
            redshift = *ptr++;
            greenshift = *ptr++;
            blueshift = *ptr++;

            printf("Received SetPixelFormat bpp = %d depth = %d endian = %d truecolour = %d redmax = %d greenmax = %d bluemax = %d redshift = %d greenshift = %d blueshift = %d\n",
                   bpp,depth,endian,truecolour,
                   redmax,greenmax,bluemax,
                   redshift,greenshift,blueshift);
            diff = 20;
        }       
        break;
    case RFB_FIX_COLORMAP_ENTRIES:
        /* Shouldn't get here */
        break;
    case RFB_SET_ENCODINGS:
        if ( len >= 4 ) {
            uint16_t  num;
            uint32_t  type;
            int       i;
	    
            BUF_TO_INT16(num,ptr);
            printf("Client supports %d encodings\n",num);
            if ( len >= 4 + ( num * 4 ) ) {
                for ( i = 0 ; i < num; i++ ) {
                    BUF_TO_INT32(type,ptr);
                    printf("Accepts encoding type %d\n",type);
                }
                diff = 4 + ( num * 4);
            } else {
                diff = 0;
            }
        }
        break;
    case RFB_FB_UPDATE_REQ:
        if ( len >= 10 ) {
            uint16_t    width;
            uint16_t    depth;
            uint16_t    x;
            uint16_t    y;

            BUF_TO_INT16(x,ptr);
            BUF_TO_INT16(y,ptr);
            BUF_TO_INT16(width,ptr);
            BUF_TO_INT16(depth,ptr);

            printf("Received %s update request for (%d,%d) -> (%d,%d)\n",
                   buf[1] ? "incremental" :"",
                   x,y,
                   x+width,y+depth);
            diff = 10;
            gui->send_update = TRUE;
        }
        break;
    case RFB_KEY_EVENT:
        if ( len >= 8 ) {
            uint32_t   key;
            int        code;

            ptr = buf + 4;
            BUF_TO_INT32(key,ptr);	    

            code = key_code(key);


            if ( code != kNone ) {
                printf("Received IR key %s\n",key_string(key));

                /* If power key was pressed, shut everything down.. */
                if ( code == kPower ) {
                    dongle_close(gui->dongle);
                    return -1;
                }
		if (  gui->media.key_func ) {
                    /* Now pass the key onto the menu stuff if necessary */
                    if ( (gui->media.key_func)(gui->media.param,code) == 0 ) {
                        if ( gui->menus.key_func ) {
                            (gui->menus.key_func)(gui->menus.param,code);
                        }
                    }
                } else if ( gui->menus.key_func ) {
		    (gui->menus.key_func)(gui->menus.param,code);
		}
            }	    
        
            diff = 8;
        }
        break;
    case RFB_POINTER_EVENT:
        if ( len >= 6 ) {
            diff = 6;
        }
        break;
    case RFB_CLIENT_CUT_TEXT:
        if ( len >= 8 ) {
            uint32_t  tlen;

            ptr = buf + 4;

            BUF_TO_INT32(tlen,ptr);

            printf("Received client cuttext of length %d\n",tlen);

            if ( len >= 8 + tlen ) {
                diff = 8 + tlen;
            }
        }
        break;
    case RFB_RDC_ACK:     
        if ( len >= 10 ) {
            diff = 10;

            switch ( buf[1] ) {		          
            case RDC_SETTINGS:       /* This really is the only one we need to catch */
                if ( buf[7] != 0x00 ) {
                    if ( diff + buf[7] >= len ) {		   
                        diff += buf[7];
		    
                        /* Save those settings */
                        gui->dongle->tvmode = buf[11];
                        /* output mode seems to be supplied with -1 - quite why I've no idea! */
                        gui->dongle->videooutput = buf[12] - 1;
                        gui->dongle->flickermode = buf[13];
                        gui->dongle->aspectratio = buf[14];
                        printf("TVmode %d output = %d flicker = %d aspect = %d\n",
                               gui->dongle->tvmode,
                               gui->dongle->videooutput,
                               gui->dongle->flickermode,
                               gui->dongle->aspectratio);

                    } else {
                        diff = 0;  /* Not had all the data yet */
                    }		   
                } 
                break;
            default:
                /* Kick ack events up to a higher level so we can do multiple plays */
		if ( gui->media_ack.ack_func ) {
                    /* Now pass the key onto the menu stuff if necessary */
                    if ( (gui->media_ack.ack_func)(buf[1],gui->media_ack.param,buf,len) == 0 ) {
                        if ( gui->menus_ack.ack_func ) {
                            (gui->menus_ack.ack_func)(buf[1],gui->menus_ack.param,buf,len);
                        }
                    }
                } else if ( gui->menus_ack.ack_func ) {
		    (gui->menus_ack.ack_func)(buf[1],gui->menus_ack.param,buf,len);
		} else {
                    printf("Can't deal with MEDIA_ACK type %d\n",buf[1]);
                }
                break;
            }
        }
        break;
    case RFB_PING:
        if ( len >= 2 ) {
            diff = 2;
            gui_send_message(gui,&rfb_ping);
        }
        break;
    default:
        dump_hex(buf,len);
    
    }

    return diff;

}


static void rfb_send_data(gui_t *gui)
{
    unsigned char  buf[100];
    unsigned char *ptr;
    int   len;

    switch ( gui->state ) {
    case VNC_VERSION:
        len = sprintf(buf,"RFB %03d.%03d\012",RFB_MAJOR,RFB_MINOR);
        write(gui->fd,buf,len);
        break;
    case VNC_AUTH:
        buf[0] = 0;
        buf[1] = 0;
        buf[2] = 0;
        buf[3] = RFB_AUTH_NONE;
        write(gui->fd,buf,4);
        break;
    case VNC_INIT:
        ptr = buf;
        INT16_TO_BUF(c_totwidth,ptr);
        INT16_TO_BUF(c_totdepth,ptr);
        *ptr++ = 24;    /* bpp */
        *ptr++ = 24;    /* depth */
        *ptr++ = 0;     /* big endian */
        *ptr++ = 0;     /* true colour */
        INT16_TO_BUF(7,ptr);  /* red max */
        INT16_TO_BUF(7,ptr);  /* green max */
        INT16_TO_BUF(3,ptr);  /* blue max */
        *ptr++ = 0;     /* red shift */
        *ptr++ = 3;     /* green shift */
        *ptr++ = 6;     /* blue shift */
        *ptr++ = 0;     /* padding */
        *ptr++ = 0;     /* padding */
        *ptr++ = 0;     /* padding */
        len = strlen("rfbwindows");
        INT32_TO_BUF(len,ptr);
        strcpy(ptr,"rfbwindows");
        ptr += len;
        write(gui->fd,buf,ptr - buf);
        gui_send_settings(gui,RDC_SETTINGS_GET);
        gui->state = VNC_RUNNING;
        break;    
    case VNC_RUNNING:        
        if ( gui->send_update ) {
            rfb_send_screen(gui);
        }       
        break;
    }	
}

/** \brief Send a screen update to the client
 *
 * \param gui Handle to the gui channel
 *
 * We have to translate our rgb map into yuvy, compress it and then 
 * send it down the rfb channel
 */ 
static void rfb_send_screen(gui_t *gui)
{
    uint8_t        buf[100];
    uint8_t       *surface;
    uint8_t       *comp;
    uint8_t       *ptr = buf;
    int            yuvsize;
    int            gz;

    surface = render_rgb2yvuy(gui->render);

    if ( surface == NULL ) {
        return;
    }
    printf("Sending update\n");
    gui->send_update = FALSE;
    yuvsize = render_get_yuvsize(gui->render);
    comp = calloc(yuvsize,1);  

    *ptr++ = RFB_FB_UPDATE;
    *ptr++ = 0xCC;            /* Pad */
    INT16_TO_BUF(1,ptr);      /* Number rectangles */
    INT16_TO_BUF((c_totwidth - render_get_width(gui->render))/ 2,ptr);      /* x pos */
    INT16_TO_BUF((c_totdepth - render_get_depth(gui->render))/ 2,ptr);      /* y pos */

    INT16_TO_BUF(render_get_width(gui->render),ptr);      /* width */
    INT16_TO_BUF(render_get_depth(gui->render),ptr);      /* depth */
    INT32_TO_BUF(7,ptr);      /* The encoding type used by HCW... */


    gz = deflateit(surface,yuvsize,comp,yuvsize);
    INT32_TO_BUF(gz,ptr);
    INT16_TO_BUF(0,ptr);        /* colour format, 0 = yuv */
    INT16_TO_BUF(2,ptr);        /* bytes per pixel */
       
    write(gui->fd,buf,ptr - buf);
    write(gui->fd,comp,gz);
        
    free(comp);
    free(surface);
}

#define CHECK_ERR(err, msg) { \
    if (err != Z_OK) { \
        fprintf(stderr, "%s error: %d\n", msg, err); \
        exit(1); \
    } \
}

/** \brief deflate the yuvy buffer
 *
 *  \param uncompr The raw buffer
 *  \param uncomprLen Length of the raw buffer
 *  \param compr Buffer to place compressed raw buffer
 *  \param comprLen Length of destination buffer
 *
 *  Shamelessly stolen from a libz example
 */
static int deflateit(Byte *uncompr, uLong uncomprLen, Byte *compr, uLong comprLen) 
{
    z_stream c_stream; /* compression stream */
    int err;

    c_stream.zalloc = (alloc_func)0;
    c_stream.zfree = (free_func)0;
    c_stream.opaque = (voidpf)0;

    err = deflateInit(&c_stream, Z_DEFAULT_COMPRESSION);
    CHECK_ERR(err, "deflateInit");

    c_stream.next_out = compr;
    c_stream.avail_out = (uInt)comprLen;
  
    c_stream.next_in = uncompr;
    c_stream.avail_in = (uInt)uncomprLen;
    err = deflate(&c_stream, Z_FINISH);
    // CHECK_ERR(err, "deflate");
   
   

    err = deflateEnd(&c_stream);
    CHECK_ERR(err, "deflateEnd");

    return c_stream.total_out;
}


/** \brief Register key handling functions
 *
 *  \param gui Handle the the gui channel
 *  \param types REGISTER_MENU or REGISTER_MEDIA
 *  \param key_func Function to call back with key code
 *  \param param Generic pointer to pass back to the callback function
 *
 *  We have two sets of functions for handling keys - one whilst in the
 *  menu, and a second for playing media
 */
void gui_register_keys(gui_t *gui,int type, 
		       int (*key_func)(void *param,int keycode), void *param)
{
    if ( type == REGISTER_MENU ) {
        gui->menus.key_func = key_func;
        gui->menus.param = param;
    } else {
        gui->media.key_func = key_func;
        gui->media.param = param;
    }
}

/** \brief Register ack handling functions
 *
 *  \param gui Handle the the gui channel
 *  \param types REGISTER_MENU or REGISTER_MEDIA
 *  \param key_func Function to call back with the ack
 *  \param param Generic pointer to pass back to the callback function
 *
 *  We have two sets of functions for handling acks - one whilst in the
 *  menu, and a second for playing media
 */
void gui_register_ack(gui_t *gui, int type,
		      int    (*ack_func)(int acktype, void *param, unsigned char *buf, int len),
		      void  *param)
{
    if ( type == REGISTER_MENU ) {
        gui->menus_ack.ack_func = ack_func;
        gui->menus_ack.param = param;
    } else {
        gui->media_ack.ack_func = ack_func;
        gui->media_ack.param = param;
    }
}


/** \brief Send a filename to the client down the RFB channel
 *
 * \param gui Handle to the gui channel
 * \param name Filename to be send (may well be http:// form as well)
 *
 * The rfb_media_filename message is unusual in that we need to nadger
 * it before the client will accept it.
 *
 */  
void gui_send_play(gui_t *gui, char *name)
{
    unsigned  char  buf[FILENAME_MAX + 24];
    mvprfbmsg_t    *msg = &rfb_media_play;
    int             len = strlen(name);


    memcpy(buf,msg->msg,msg->len);

    buf[8] = len;

    strcpy(buf + msg->len,name);

    write(gui->fd,buf,len + msg->len);    
}

/** \brief Send a message to the client down the RFB channel
 *
 * \param gui Handle to the gui channel
 * \param msg Pointer to the message to send
 *
 */
void gui_send_message(gui_t *gui, mvprfbmsg_t *msg)
{
    write(gui->fd,msg->msg,msg->len);
}



/** \brief Send the settings to the client down the RFB channel
 *
 *  \param gui Handle to the gui channel
 */
void gui_send_settings(gui_t *gui, int subcommand)
{
    char    buf[12];

    gui_send_message(gui,&rfb_media_settings);

   
    buf[0] = subcommand;    /* 0,1 (or 03??) */
    buf[1] = gui->dongle->tvmode;
    buf[2] = gui->dongle->videooutput;
    buf[3] = gui->dongle->flickermode;
    buf[4] = gui->dongle->aspectratio;
    buf[5] = buf[6] = buf[7] = buf[8] = buf[9] = buf[10] = buf[11] = 0;

    write(gui->fd,buf,12);
}

/*
 * Local Variables:
 *  indent-tabs-mode:nil
 *  require-final-newline:t
 *  c-basic-offset: 4
 * End:
 */
