#/------------+----------+----------------------------------------------------\
#|  |\  /|    | Filename | TPDD.py                                            |
#|  | \/ |    |  Version | 1.4                                                |
#|  |    |    |   Author | Kurt McCullum                                      |
#|  |  /-+-\  +----------+-------------- DESCRIPTION -------------------------+
#|  |  | |    | This file contains the TPDD routines that will process all    |
#|     \---\  | responses to TPDD commands and return a response.             |
#|         |  |                                                               |
#|     \---/  |                                                               |
#| ---------- |                                                               |
#|  McCullum  |                                                               |
#|  Software  |                                                               |
#+------------+--------------------------- HISTORY ---------------------------+
#| 03/25/2020 | v1.0 Which only supports a single folder                      |
#| 03/26/2020 | v1.1 Added support for the Sardine dictionary                 |
#| 03/27/2020 | v1.2 Added support for directories                            |
#| 03/31/2020 | v1.3 Added the ability to turn off messages                   |
#\------------+---------------------------------------------------------------/

#/---------\
#| Modules |
#\----------------------------------------------------------------------------/
import os									# Needed to work with files

class TPDD:
	def __init__(self, BaseDir):
		self.SilentMode = False				# Turn off the console messages
		self.BaseDir = BaseDir				# Used as the base folder for files
		self.CurrentDir = BaseDir			# Current folder
		self.CurrentFilename = ""			# Current filename
		self.FileList = []					# List of files from folder
		self.DirectoryList = []				# List of directories in current folder
		self.FilePosition = 0				# Pointer used for FileList
		self.DirectoryPosition = 0			# Pointer used for DictionaryList
		self.FileBuffer = bytearray()		# Buffer for read/write files
		self.FilePosition = 0				# Pointer to position in FileBuffer
		self.FileMode = ""					# R=Read, W=Write, A=Append
		self.ReceivedM1 = 0					# Mode switch used for Sardine requests
		self.CRFeedback = ""				# Sardine feedback when initial CR received
		self.Dictionary = bytearray()		# Sardine dictionary buffer (104k)
		self.LoadDictionary()				# Load the dictionary if it exists

	#/------------\
	#| LogMessage |
	#+------------+---------------------------------------------------------------\
	#| Send log messages to stdio                                                 |
	#\----------------------------------------------------------------------------/
	def LogMessage(self, Message, EndCR):
		if(self.SilentMode == False):
			if(EndCR == True):
				print(Message)
			else:
				print(Message, sep=' ', end='',flush=True)


	#/--------------------\
	#| ProcessModeCommand |
	#+--------------------+-------------------------------------------------------\
	#| Process all standard mode (ZZ) commands                                    |
	#\----------------------------------------------------------------------------/
	def ProcessModeCommand(self, TPDD_Command):
		TPDD_Response = bytearray()			#Byte array to hold a standard response to the request
		if self.ValidCommand(TPDD_Command) == 0:
			return TPDD_Response

		if TPDD_Command[2] >= 0x40:			#If the request is for the TPDD2 bank 1
			TPDD_Command[2] = TPDD_Command[2] - 0x40 # Change it to bank 0

		if TPDD_Command[2] == 0x00: 		# Directory reference
			#print("Directory command")
			TPDD_Response = self.ProcessDirectoryCommand(TPDD_Command)

		elif TPDD_Command[2] == 0x01:		# Open file
			#print("Open file command")
			TPDD_Response = self.ProcessOpenFileCommand(TPDD_Command)

		elif TPDD_Command[2] == 0x02:		# Close file
			#print("Close file command")
			TPDD_Response = self.ProcessCloseFileCommand(TPDD_Command)

		elif TPDD_Command[2] == 0x03:		# Read from file
			#print("Read from file command")
			TPDD_Response = self.ProcessReadFromFileCommand(TPDD_Command)

		elif TPDD_Command[2] == 0x04:		# Write to file
			#print("Write to file command")
			TPDD_Response = self.ProcessWriteToFileCommand(TPDD_Command)

		elif TPDD_Command[2] == 0x05:		# Delete file
			#print("Delete file command")
			TPDD_Response = self.ProcessDeleteFileCommand(TPDD_Command)

		elif TPDD_Command[2] == 0x06:		# Format disk is not used. Return valid response.
			self.LogMessage("Format disk command (not supported)", True)
			TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

		elif TPDD_Command[2] == 0x07:		# Drive status (Send a valid response)
			self.LogMessage("Drive status", True)
			TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

		elif TPDD_Command[2] == 0x08:		# Change to FDC mode
			#print("FDC command")
			TPDD_Response = self.ProcessFDCCommand(TPDD_Command)

		elif TPDD_Command[2] == 0x0D:		# Rename file
			#print("Rename command")
			TPDD_Response = self.ProcessFileRenameCommand(TPDD_Command)

		if TPDD_Command[2] != 0x08 and TPDD_Command[2] != 0x07:
			self.ReceivedM1 = 0				# Reset the ReceivedM1 flag to false

		return TPDD_Response

	#/--------------------------\
	#| ProcessFileRenameCommand |
	#+--------------------------+-------------------------------------------------\
	#| Process TPDD rename file requests                                          |
	#\----------------------------------------------------------------------------/
	def ProcessFileRenameCommand(self, TPDD_Command):
		TPDD_Response = bytearray()
		baFilename = bytearray(24)
		for i in range(24):
			baFilename[i] = TPDD_Command[i+4]

		fnParts = baFilename.decode("ASCII").replace('*','^').split('.')
		fnParts[0] = fnParts[0].strip()
		newFilename = fnParts[0] + "."
		if len(fnParts) > 1:
			newFilename += fnParts[1].strip()

		try:
			if ".<>" in self.CurrentFilename: # This is a folder
				self.CurrentFilename = self.CurrentFilename[:-3] # Trim off the .<>

			if ".<>" in newFilename:
				newFilename = newFilename[:-3]

			self.LogMessage("Renaming " + self.CurrentDir + "/" + self.CurrentFilename + " to " + self.CurrentDir + "/" + newFilename, True)
			os.rename(self.CurrentDir + "/" + self.CurrentFilename, self.CurrentDir + "/" + newFilename)
			TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

		except:
			TPDD_Response = bytearray([0x12, 0x01, 0x50, 0x00]) # 0x50 Write protected error
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

		return TPDD_Response


	#/--------------------------\
	#| ProcessDeleteFileCommand |
	#+--------------------------+-------------------------------------------------\
	#| Process TPDD delete file requests                                          |
	#\----------------------------------------------------------------------------/
	def ProcessDeleteFileCommand(self, TPDD_Command):
		TPDD_Response = bytearray()
		try:
			if ".<>" in self.CurrentFilename: # This is a folder
				tFolder = ""
				self.CurrentFilename = self.CurrentFilename[:-3] # Trim off the .<>
				tFolder = self.CurrentDir + "/" + self.CurrentFilename
				self.LogMessage("Deleting folder " + tFolder, True)
				os.rmdir(tFolder)
				TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
				TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

			else:
				if os.path.isfile(self.CurrentDir + "/" + self.CurrentFilename):
					self.LogMessage("Deleting file " + self.CurrentDir + "/" + self.CurrentFilename, True)
					os.remove(self.CurrentDir + "/" + self.CurrentFilename)
					TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
					TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

				else: # File doesn't exist so no need to delete it
					self.LogMessage("File missing " + self.CurrentDir + "/" + self.CurrentFilename, True)
					TPDD_Response = bytearray([0x12, 0x01, 0x10, 0x00]) # 0x10 File not found
					TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

		except:	# If for some reason there was an error return an error
			TPDD_Response = bytearray([0x12, 0x01, 0x50, 0x00]) # 0x50 Write-protected error
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

		return TPDD_Response


	#/---------------------------\
	#| ProcessWriteToFileCommand |
	#+---------------------------+------------------------------------------------\
	#| Process TPDD write to file requests                                        |
	#\----------------------------------------------------------------------------/
	def ProcessWriteToFileCommand(self, TPDD_Command):
		TPDD_Response = bytearray()
		if len(self.FileBuffer) == 0:
			self.FileBuffer = bytearray(TPDD_Command[3])
			for i in range (TPDD_Command[3]):
				self.FileBuffer[i] = TPDD_Command[i + 4]

		else:
			for i in range (TPDD_Command[3]):
				self.FileBuffer.append(TPDD_Command[i + 4])

		self.LogMessage(".", False)
		TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
		TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
		return TPDD_Response


	#/-------------------\
	#| ProcessFDCCommand |
	#+-------------------+--------------------------------------------------------\
	#| Process TPDD FDC/Sardine requests                                          |
	#\----------------------------------------------------------------------------/
	def ProcessFDCCommand(self, TPDD_Command):
		TPDD_Response = bytearray()
		if self.ReceivedM1 == 1: # Request for current directory
			self.LogMessage("Sending current directory name", True)
			TPDD_Response = bytearray(14)
			TPDD_Response[0] = 0x12
			TPDD_Response[1] = 0x0B
			TPDD_Response[2] = 0x00
			for i in range(3, 13):
				TPDD_Response[i] = 0x20

			if self.CurrentDir == self.BaseDir: # Top level so put ROOT.<>
				TPDD_Response[3] = bytearray(b"R")[0]
				TPDD_Response[4] = bytearray(b"O")[0]
				TPDD_Response[5] = bytearray(b"O")[0]
				TPDD_Response[6] = bytearray(b"T")[0]
				TPDD_Response[9] = bytearray(b".")[0]
				TPDD_Response[10] = bytearray(b"<")[0]
				TPDD_Response[11] = bytearray(b">")[0]

			else:	# Show the current directory name
				tPos = 0
				tPos = self.CurrentDir.rfind("/")
				tPos = len(self.CurrentDir) - tPos - 1
				tFolder = self.CurrentDir[-tPos:]
				tBA = bytearray(tFolder,"ASCII")
				for i in range(len(tBA)):
					TPDD_Response[i+3] = tBA[i]

			TPDD_Response[13] = 0
			TPDD_Response[13] = self.CalcChecksum(TPDD_Response)
			self.ReceivedM1 = 0

		else: # This is a Sardine dictionary request
			if len(self.Dictionary) > 0:
				self.LogMessage("Sending Sardine response", True)
				TPDD_Response = bytearray("D0000000","ASCII")
				self.CRFeedback = bytearray("C1000000\r","ASCII")

		return TPDD_Response

	#/-------------------------\
	#| ProcessCloseFileCommand |
	#+-------------------------+--------------------------------------------------\
	#| Process TPDD close file requests                                           |
	#\----------------------------------------------------------------------------/
	def ProcessCloseFileCommand(self, TPDD_Command):
		TPDD_Response = bytearray()
		try:
			if self.FileMode == "W" or self.FileMode == "A":
				# Only write data if you are in write or append mode
				if os.path.isfile(self.CurrentDir + "/" + self.CurrentFilename):
					tFile = open(self.CurrentDir + "/" + self.CurrentFilename,"ab")
				else: # File doesn't exist so create a new file
					tFile = open(self.CurrentDir + "/" + self.CurrentFilename,"wb")

				tFile.write(self.FileBuffer)
				tFile.close()
				self.FileMode = "" # Clear the file mode
			self.LogMessage("\nClosing file " + self.CurrentDir + "/" + self.CurrentFilename, True)
			TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

		except:	# If for some reason there was an error return an error
			TPDD_Response = bytearray([0x12, 0x01, 0x50, 0x00]) # 0x50 Write-protected error
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

		self.FileBuffer = bytearray()						# Clear the file buffer
		return TPDD_Response


	#/----------------------------\
	#| ProcessReadFromFileCommand |
	#+----------------------------+-----------------------------------------------\
	#| Process TPDD read from file requests                                       |
	#\----------------------------------------------------------------------------/
	def ProcessReadFromFileCommand(self, TPDD_Command):
		TPDD_Response = bytearray()
		if self.FilePosition + 128 <= len(self.FileBuffer): # Send a full length packet
			TPDD_Response = bytearray(131)
			TPDD_Response[0] = 0x10
			TPDD_Response[1] = 128 # Regular packet length is 128 bytes
			for i in range(self.FilePosition, self.FilePosition + 128):
				TPDD_Response[i - self.FilePosition + 2] = self.FileBuffer[i]

			TPDD_Response[130]	# Zero out the checksum
			TPDD_Response[130] = self.CalcChecksum(TPDD_Response)
			self.FilePosition += 128

		else: # Send a partial packet with the remaining data
			TPDD_Response = bytearray(len(self.FileBuffer) - self.FilePosition + 3)
			TPDD_Response[0] = 0x10
			TPDD_Response[1] = len(self.FileBuffer) - self.FilePosition
			for i in range(self.FilePosition, len(self.FileBuffer)):
				TPDD_Response[i - self.FilePosition + 2] = self.FileBuffer[i]

			TPDD_Response[len(TPDD_Response) - 1]	# Zero out the checksum
			TPDD_Response[len(TPDD_Response) - 1] = self.CalcChecksum(TPDD_Response)
			self.FilePosition += len(self.FileBuffer) - self.FilePosition

		self.LogMessage(".",False)
		return TPDD_Response

	#/------------------------\
	#| ProcessOpenFileCommand |
	#+------------------------+---------------------------------------------------\
	#| Process TPDD open file requests                                            |
	#\----------------------------------------------------------------------------/
	def ProcessOpenFileCommand(self, TPDD_Command):
		TPDD_Response = bytearray()
		self.FileBuffer = bytearray()
		if TPDD_Command[4] == 0x01:	# Open file in write mode
			if ".<>" in self.CurrentFilename: # This is a folder
				# Create the new folder
				tFolder = ""
				self.CurrentFilename = self.CurrentFilename[:-3] # Trim off the .<>
				tFolder = self.CurrentDir + "/" + self.CurrentFilename
				self.LogMessage("Creating folder " + tFolder, True)
				try:
					os.mkdir(tFolder)
					TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
					TPDD_Response[3] = self.CalcChecksum(TPDD_Response)

				except:
					TPDD_Response = bytearray([0x12, 0x01, 0x50, 0x00]) # 0x50 Write-protected error
					TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
					self.FileMode = "" # Clear file mode

			else:
				self.LogMessage("Creating file " + self.CurrentDir + "/" + self.CurrentFilename, True)
				if os.path.isfile(self.CurrentDir + "/" + self.CurrentFilename):
					try:
						os.remove(self.CurrentDir + "/" + self.CurrentFilename)
						TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
						TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
						self.FileMode = "W" # Set file mode to write

					except:	# If for some reason there was an error return an error
						TPDD_Response = bytearray([0x12, 0x01, 0x50, 0x00]) # 0x50 Write-protected error
						TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
						self.FileMode = "" # Clear file mode

				else: # File doesn't exist so no need to delete it.
					TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
					TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
					self.FileMode = "W" # Set file mode to write

		elif TPDD_Command[4] == 0x02:  # Open file in append/write mode
			self.LogMessage("Append to file " + self.CurrentDir + "/" + self.CurrentFilename, True)
			TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
			self.FileMode = "A" # Set file mode to append

		elif TPDD_Command[4] == 0x03:  # Open file in read mode
			if ".<>" in self.CurrentFilename: # This is a folder
				# Select a new folder
				tFolder = ""
				if self.CurrentFilename == "PARENT.<>":
					tPos = 0
					tPos = self.CurrentDir.rfind("/")
					tFolder = self.CurrentDir[:tPos]
				else:
					self.CurrentFilename = self.CurrentFilename[:-3] # Trim off the .<>
					tFolder = self.CurrentDir + "/" + self.CurrentFilename
					self.CurrentFilename = ""

				if os.path.isdir(tFolder):
					self.LogMessage("Changing directory to" + tFolder, True)
					TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
					TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
					self.CurrentDir = tFolder

				else:
					TPDD_Response = bytearray([0x12, 0x01, 0x10, 0x00]) # 0x10 File not found
					TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
					self.FileMode = "" # Clear file mode


			else:
				if os.path.isfile(self.CurrentDir + "/" + self.CurrentFilename):
					try:
						self.LogMessage("Reading " + self.CurrentDir + "/" + self.CurrentFilename, True)
						self.FileBuffer = open(self.CurrentDir + "/" + self.CurrentFilename, "rb").read()
						TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
						TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
						self.FilePosition = 0 # Set the reading pointer to the beginning of the file
						self.FileMode = "R" # Set file mode to read

					except:
						TPDD_Response = bytearray([0x12, 0x01, 0x40, 0x00]) # 0x40 Read error
						TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
						self.FileMode = "" # Clear file mode

				elif os.path.isfile(os.getcwd() + "/" + self.CurrentFilename): # Check the application folder
					try:
						self.LogMessage("Reading " + os.getcwd() + "/" + self.CurrentFilename, True)
						self.FileBuffer = open(os.getcwd() + "/" + self.CurrentFilename, "rb").read()
						TPDD_Response = bytearray([0x12, 0x01, 0x00, 0x00]) # 0x00 Operation complete
						TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
						self.FilePosition = 0 # Set the reading pointer to the beginning of the file
						self.FileMode = "R" # Set file mode to read

					except:
						TPDD_Response = bytearray([0x12, 0x01, 0x40, 0x00]) # 0x40 Read error
						TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
						self.FileMode = "" # Clear file mode

				else: # The file should always exist but if not, send an error
					TPDD_Response = bytearray([0x12, 0x01, 0x10, 0x00]) # 0x10 File not found
					TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
					self.FileMode = "" # Clear file mode

		else: # Error, you should never get here. Send an error
			TPDD_Response = bytearray([0x12, 0x01, 0x30, 0x00]) # 0x30 Command parameter error
			TPDD_Response[3] = self.CalcChecksum(TPDD_Response)
			self.FileMode = "" # Clear file mode

		return TPDD_Response

	#/-------------------------\
	#| ProcessDirectoryCommand |
	#+-------------------------+--------------------------------------------------\
	#| Process the various TPDD directory requests                                |
	#\----------------------------------------------------------------------------/
	def ProcessDirectoryCommand(self, TPDD_Command):
		TPDD_Response = bytearray()	#Byte array to hold a standard response to the request
		if TPDD_Command[29] == 0x00: # Create a file reference
			baFilename = bytearray(24)
			for i in range(24):
				baFilename[i] = TPDD_Command[i+4]

			fnParts = baFilename.decode("ASCII").replace('*','^').split('.')
			fnParts[0] = fnParts[0].strip()
			self.CurrentFilename = fnParts[0] + "."
			if len(fnParts) > 1:
				self.CurrentFilename += fnParts[1].strip()

			TPDD_Response = bytearray(31)
			TPDD_Response[0] = 0x11
			TPDD_Response[1] = 0x1C
			if os.path.isfile(self.CurrentDir + "/" + self.CurrentFilename):
				self.LogMessage("Creating file reference " + self.CurrentDir + "/" + self.CurrentFilename, True)
				for i in range(4, 29):
					TPDD_Response[i - 2] = TPDD_Command[i]

				fnSize = 0
				fnSize = os.path.getsize(self.CurrentDir + "/" + self.CurrentFilename)
				TPDD_Response[27] = int(fnSize/256)	# Hi byte
				TPDD_Response[28] = fnSize % 256	# Lo byte
				TPDD_Response[29] = 0x4B			# Remaining sectors byte = 100k to TS-DOS (0x9D = 200k)
				TPDD_Response[30] = 0x00			# Make sure the checksub byte is 0x00
				TPDD_Response[30] = self.CalcChecksum(TPDD_Response)

			elif os.path.isdir(self.CurrentDir + "/" + fnParts[0]):
				self.LogMessage("Creating directory reference " + self.CurrentDir + "/" + fnParts[0], True)
				for i in range(4, 29):
					TPDD_Response[i - 2] = TPDD_Command[i]

				TPDD_Response[10] = 0x20
				TPDD_Response[11] = 0x20
				TPDD_Response[12] = 0x00
				fnSize = 0
				TPDD_Response[27] = int(fnSize/256)	# Hi byte
				TPDD_Response[28] = fnSize % 256	# Lo byte
				TPDD_Response[29] = 0x4B			# Remaining sectors byte = 100k to TS-DOS (0x9D = 200k)
				TPDD_Response[30] = 0x00			# Make sure the checksub byte is 0x00
				TPDD_Response[30] = self.CalcChecksum(TPDD_Response)

			elif os.path.isfile(os.getcwd() + "/" + self.CurrentFilename): # Check the application folder
				self.LogMessage("Creating directory reference " + os.getcwd() + "/" + self.CurrentFilename, True)
				for i in range(4, 29):
					TPDD_Response[i - 2] = TPDD_Command[i]

				fnSize = 0
				fnSize = os.path.getsize(os.getcwd() + "/" + self.CurrentFilename)
				TPDD_Response[27] = int(fnSize/256)	# Hi byte
				TPDD_Response[28] = fnSize % 256	# Lo byte
				TPDD_Response[29] = 0x4B			# Remaining sectors byte = 100k to TS-DOS (0x9D = 200k)
				TPDD_Response[30] = 0x00			# Make sure the checksub byte is 0x00
				TPDD_Response[30] = self.CalcChecksum(TPDD_Response)

			else: # File doesn't exist so send a blank response
				for i in range(2, 27):
					TPDD_Response[i] = 0x00	# Fill the filename location with nulls
				# set the size to zero
				TPDD_Response[27] = 0x00			# Hi byte
				TPDD_Response[28] = 0x00			# Lo byte
				TPDD_Response[29] = 0x4B			# Remaining sectors byte = 100k to TS-DOS (0x9D = 200k)
				TPDD_Response[30] = 0x00			# Make sure the checksub byte is 0x00
				TPDD_Response[30] = self.CalcChecksum(TPDD_Response)

		elif TPDD_Command[29] == 0x01: # Get the first file in the current directory
			# Start by creating a blank list of filenames and fill it with every
			# file in the CurrentDir
			self.FileList[:] = []
			lFiles = os.listdir(self.CurrentDir)
			lFiles.sort()
			if(self.CurrentDir != self.BaseDir):
				self.FileList.append(FileInfo("PARENT.<>",0))
			for tFile in lFiles:
				self.FileList.append(FileInfo(tFile,os.path.getsize(self.CurrentDir + "/" + tFile)))

			self.LogMessage("Sending first filename in directory", True)
			# Set the file position to the first file
			self.FilePosition = 0
			# Now get the file infomation for the selected FilePosition
			TPDD_Response = self.ProcessDirectoryFile()

		elif TPDD_Command[29] == 0x02: # Get the next file in the current directory
			# Now get the file infomation for the selected FilePosition
			self.LogMessage("Sending next filename in directory", True)
			TPDD_Response = self.ProcessDirectoryFile()

		return TPDD_Response

	#/----------------------\
	#| ProcessDirectoryFile |
	#+----------------------+-----------------------------------------------------\
	#| Return the next file in the file or directory list                         |
	#\----------------------------------------------------------------------------/
	def ProcessDirectoryFile(self):
		TPDD_Response = bytearray(31)	# 31 Byte array to hold a file response
		TPDD_Response[0] = 0x11
		TPDD_Response[1] = 0x1C

		if len(self.FileList) > self.FilePosition: # Take the next file in the list
			for i in range(2, 27):
				TPDD_Response[i] = 0x20 	# Fill the filename location with blanks

			fnParts = []
			fnParts = self.FileList[self.FilePosition].Filename.upper().replace('^','*').split('.')
			fnSize = 0
			fnSize = self.FileList[self.FilePosition].Filesize

			if len(fnParts) > 0: # Check to see if there is a filename to process
					# Put in the filename
					ba = bytearray(fnParts[0], encoding="ASCII")
					i = 0
					while i < len(ba) and i < 6:
						TPDD_Response[i + 2] = ba[i]
						i+=1
					# Put in the decimal
					TPDD_Response[8] = 0x2E # '.'

			if len(fnParts) > 1: # Check to see if there is a file extension to process
					# Put in the filename
					ba = bytearray(fnParts[1], encoding="ASCII")
					i = 0
					while i < len(ba) and i < 2:
						TPDD_Response[i + 9] = ba[i]
						i += 1

			else:	# This may be a directory so deal with it
				if os.path.isdir(self.CurrentDir + "/" + self.FileList[self.FilePosition].Filename):
					TPDD_Response[9] = 0x3C #'<'
					TPDD_Response[10] = 0x3E #'>'
					fnSize = 0						# Reset the file size to zero

			# Now deal with the file size and checksum
			TPDD_Response[26] = 0x46 			# 'F'
			TPDD_Response[27] = int(fnSize/256)	# Hi byte
			TPDD_Response[28] = fnSize % 256	# Lo byte
			TPDD_Response[29] = 0x4B			# Remaining sectors byte = 100k to TS-DOS (0x9D = 200k)
			TPDD_Response[30] = 0x00			# Make sure the checksub byte is 0x00
			TPDD_Response[30] = self.CalcChecksum(TPDD_Response)
		else: # Return a blank file
			for i in range(2, 27):
				TPDD_Response[i] = 0x00	# Fill the filename location with nulls

			# set the size to zero
			TPDD_Response[27] = 0x00			# Hi byte
			TPDD_Response[28] = 0x00			# Lo byte
			TPDD_Response[29] = 0x4B			# Remaining sectors byte = 100k to TS-DOS (0x9D = 200k)
			TPDD_Response[30] = 0x00			# Make sure the checksub byte is 0x00
			TPDD_Response[30] = self.CalcChecksum(TPDD_Response)
		# Return the file response
		self.FilePosition += 1
		return TPDD_Response

	#/--------------\
	#| CalcChecksum |
	#+--------------+-------------------------------------------------------------\
	#| Calculate the checksum of the TPDD_Command or TPDD_Response passed         |
	#\----------------------------------------------------------------------------/
	def CalcChecksum(self, TPDD_Data):
		i = 0
		Checksum = 0
		while i < len(TPDD_Data):
			Checksum += TPDD_Data[i]
			if Checksum > 255:
				Checksum -= 256
			i += 1
		Checksum = 255 - Checksum
		return Checksum

	#/--------------\
	#| ValidCommand |
	#+--------------+-------------------------------------------------------------\
	#| Check to see if this is a valid command by checking the checksum           |
	#\----------------------------------------------------------------------------/
	def ValidCommand(self, TPDD_Command):
		Counter = 2
		Checksum = 0
		while Counter <= TPDD_Command[3] + 3:
			Checksum = Checksum + TPDD_Command[Counter]
			if Checksum > 255:
				Checksum = Checksum - 256
			Counter += 1
		Checksum = 255 - Checksum
		if TPDD_Command[TPDD_Command[3] + 4] == Checksum:
			return 1
		else:
			return 0

	#/----------------\
	#| LoadDictionary |
	#+----------------+-----------------------------------------------------------\
	#| Load the dictionary used by Sardine                                        |
	#\----------------------------------------------------------------------------/
	def LoadDictionary(self):
		# Check to see if the dictionary file exists
		if os.path.isfile(os.getcwd() + "/mcomm.dic"):
			self.Dictionary = open(os.getcwd() + "/mcomm.dic", "rb").read()

		else:
			self.Dictionary = bytearray()

	#/-----------------\
	#| GetSardineTrack |
	#+-----------------+----------------------------------------------------------\
	#| Get the selected Sardine floppy disk track/sector                          |
	#\----------------------------------------------------------------------------/
	def GetSardineTrack(self, Track, Sector):
		if len(self.Dictionary) > 0:
			self.LogMessage("Sending Sardine track=" + str(Track) + " sector=" + str(Sector),True)
			TPDD_Response = bytearray(264)
			Offset = (Track * (264 * 5)) + (264 * (Sector - 1))
			for i in range(Offset, Offset + 264):
				TPDD_Response[i - Offset] = self.Dictionary[i]

		else: # Dictionary wasn't loaded so send an error
			TPDD_Response = bytearray()

		return TPDD_Response

class FileInfo(object):
	def __init__(self, Filename, Filesize):
		self.Filename = Filename
		self.Filesize = Filesize
