;;
;;      Name: SpDebug
;;
;;      Functions:  Debugger for the spelling rules       
;;
;;      Author: Alan W Black January 1987
;;
;;      Copyright Graeme Ritchie, Alan Black,
;;                Steve Pulman and Graham Russell  1987
;;
;;         ---------------------------------------------
;;         |    Not to be used for military purposes   |
;;         ---------------------------------------------
;;
;;      Description:
;;         This allows the user to debug spelling rules by
;;         giving two specific lexical and surface strings 
;;         These are then matched using the spelling rules.
;;         There are two forms of debugging.  INFO and STEP.
;;
;;         INFO summarises the debugging info.  It  gives information 
;;         about which rules complete and which rules block, also 
;;         information about terminal states (final or not).  All
;;         information is given in terms of the Rule names and none
;;         of this automata state rubbish.
;;         
;;         STEP gives a step by step analysis of the string by the 
;;         rules.  Printing the the actual paths searched and showing
;;         back tracking caused by the use of nulls.
;;
;;         This is an interactive debugger and designed to run as part
;;         of the dictionary command interpreter rather than an 
;;         independent lisp function like the dictionary access functions.
;;
;;    Restrictions:
;;       This is not written very well, I grew it out of the standard
;;       functions in spmoveau.  I am not happy with its contents but
;;       its function seems ok.  The main problem is I understand these
;;       form of rule VERY well and other people probably do not so 
;;       I am not sure if the information I present (or the way I
;;       present it) will be sensible to other people.  Sorry, complain
;;       to me though and I'll do something about it (i.e. ignore it - Ed).
;;

(declare
   (special
      D-TRANSITIONSLIST
      D-UNUSUALFEASIBLES
      D-TERMINATED-RULES
      D-FAILED-RULE 
      D-FAILEDLICENCE
      D-BLOCKED
      D-LICENCE-RULES
      D-TERMINATED-RULES-THIS-TIME 
   )
   (localf
      D-DebugInfoMode                 
      D-InitDebugGlobals              
      D-InitDebugGlobalsPerState      
      D-GetUsersSurfaceString         
      D-GetUsersLexicalString         
      D-GetUsersDebugMode             
      D-DebugStepMode                 
      D-AnalyseEndConfig              
      D-DisplayStep                   
      D-DisplayTape                   
      D-PlaceTapeHead                 
      D-DebugDescribeCurrentState     
      D-DebugAnalyseFailure           
      D-FindLicencePairs              
      D-FindCommitGroups              
      D-FindActiveRules               
      D-DebugCheckPairMatch           
      D-DebugMoveAllStates            
      D-DebugFindNextState            
      D-DebugCheckNewConfiguration    
      D-DebugMoveCommitStates         
      D-DebugIsFeasibleP              
      D-DebugFinal                    
      D-DebugTerminal                 
      D-DebugPause                    
   )
)

(include "keywords")
(include "macros")
(include "subrout")
(include "readatom")

(defun D-SpDebug ()
;;
;;   Koskenniemi Spelling Rule Debugger.  This is an interactive debugger
;;   designed to be called from the dictionary command interpreter.
;;
;;   It is designed to run with the alternative implementation of the 
;;   spelling rules, the one I did in July 1986.
;;
   (let ( lexicalstring surfacestring debugmode )
      (cond
	 ((not (assq 'sp D-LOADEDPARTS))
	    (error "No spelling rules are loaded - cannot debug")))
      (D-InitDebugGlobals)
      (setq lexicalstring (D-GetUsersLexicalString))
      (setq surfacestring (D-GetUsersSurfaceString))
      (cond
	 ((not (equal (length surfacestring) (length lexicalstring)))
	    (error "Lexical and surface strings MUST be of same length")
         )
      )
      (setq debugmode (D-GetUsersDebugMode))

      (cond
	 ((eq debugmode 'STEP)
	    (D-DebugStepMode
	       (append surfacestring (ncons (DK-ENDMARKER)))
	       (append lexicalstring (ncons (DK-ENDMARKER)))
	       nil                     ;; initial configuration of rules
	       (list                   ;; for tape displaying
		  surfacestring
		  lexicalstring)
	       1)                      ;; position of read head
         )
	 ((eq debugmode 'INFO)
	    (D-DebugInfoMode 
	       (append surfacestring (ncons (DK-ENDMARKER)))
	       (append lexicalstring (ncons (DK-ENDMARKER)))
	       nil)
         )
	 (t
	    (error (concat "Unknown debug mode " debugmode))
         )
      )
   )
)

(defun D-DebugInfoMode (surfacetape lexicaltape config)
;;
;;  Matches the two strings until it completes or fails, i.e.
;;  information is only displayed at the end rather than at each 
;;  step
;;
   (cond
      ((and (eq (car surfacetape) (DK-ENDMARKER))
	    (eq (car lexicaltape) (DK-ENDMARKER))
	    (D-DebugFinal config))
         (princ "Match OK") (terpri)
         (princ "   Rule(s) used were:")
	 (D-PList D-TERMINATED-RULES) (terpri)
      )
      ((and (eq (car surfacetape) (DK-ENDMARKER))
	    (eq (car lexicaltape) (DK-ENDMARKER)))
         (princ "Tapes have ended but: ") (terpri)
         (princ "   Rule(s) used were:")
	 (D-PList D-TERMINATED-RULES) (terpri)
	 (D-AnalyseEndConfig config))
      (t
	 (let ( (newconfig 
	    (D-DebugCheckPairMatch
	       (car lexicaltape) (car surfacetape) config)) )
	    (cond
	       ((eq 'ERROR newconfig)
		  (D-DebugAnalyseFailure)
	       )
	       (t
 		  (D-DebugInfoMode                ;; continue matching
		     (cdr surfacetape)
		     (cdr lexicaltape)
		     newconfig)
	       )
            )
	 )
      )
   )
)

(defun D-InitDebugGlobals ()
;;
;;   Initialises the global variables used in a debug match
;;
   (D-InitReader) 
   ;; Per match
   (setq D-TERMINATED-RULES nil)
   (setq D-FAILED-RULE nil)
   (setq D-FAILEDLICENCE nil)
   (setq D-BLOCKED nil)
   ;; Per state move
   (D-InitDebugGlobalsPerState)
)

(defun D-InitDebugGlobalsPerState ()
;;
;;  Initialises the global variables that are used each state
;;  mov as opposed to the those that are used per whole match
;;
   (setq D-LICENCE-RULES nil)
   (setq D-TERMINATED-RULES-THIS-TIME nil)
)

(defun D-GetUsersSurfaceString ()
;;
;;  gets the user to type in a surface string.  This is returned in
;;  an individual character form.
;;
   (let ( surfstring )
      (princ "Enter Surface string: ") (drain)
      (setq surfstring (D-ReadToEndOfLine 'EOF))
      (cond
	 ((null surfstring)
	    (error "Can't debug a null surface string"))
         (t
            (D-SplitIntoChars (car surfstring)))
      )
   )
)

(defun D-GetUsersLexicalString ()
;;
;;  gets the user to type in a lexical string.  This is returned in
;;  an individual character form.
;;
   (let ( lexstring )
      (princ "Enter Lexical string: ") (drain)
      (setq lexstring (D-ReadToEndOfLine 'EOF))
      (cond
	 ((null lexstring)
	    (error "Can't debug a null lexical string"))
         (t
	    (D-SplitIntoChars (car lexstring)))
      )
   )
)

(defun D-GetUsersDebugMode ()
;;
;;   returns the symbol STEP or INFO depending on the users reply
;;   STEP is the default
;;
   (let ( reply )
      (princ "Which mode ? ") (drain)
      (setq reply (D-ReadToEndOfLine 'EOF))
      (cond
	 ((null reply) 
	    (princ "Debug Mode is STEP") (terpri)
	    'STEP
         )
	 ((memq (car reply) (DK-SPDEBUG-INFO))
	    (princ "Debug Mode is INFO") (terpri)
	    'INFO
         )
	 (t
	    (princ "Debug Mode is STEP") (terpri)
	    'STEP
         )
      )
   )
)

(defun D-DebugStepMode (surfacetape lexicaltape config tapes pos)
;;
;;  Runs the spelling rules stepwise of the given surface and
;;  lexical tapes.  
;;
   (cond
      ((and (eq (car surfacetape) (DK-ENDMARKER))
	    (eq (car lexicaltape) (DK-ENDMARKER))
	    (D-DebugFinal config))
         (princ "Match OK") (terpri)
      )
      ((and (eq (car surfacetape) (DK-ENDMARKER))
	    (eq (car lexicaltape) (DK-ENDMARKER)))
         (princ "Tapes have ended but: ") (terpri)
	 (D-AnalyseEndConfig config))
      (t
	 (D-DisplayStep 
	    (D-DebugCheckPairMatch
	       (car lexicaltape) (car surfacetape) config)
	    (car lexicaltape) (car surfacetape)
	    (cdr lexicaltape) (cdr surfacetape)
	    tapes pos
         )
      )
   )
)

(defun D-AnalyseEndConfig (config)
;;
;;  Analyses why an end configuration is not final
;;
   (let ( (commits (D-FindCommitGroups config)) )
      (princ "Match Failed because:") (terpri)
      (mapcar
	 #'(lambda (commitgroup)
	    (princ "   The pair ") (princ (car commitgroup))
	    (princ " has not yet been accepted by")
	    (D-PList
	       (apply #'append (mapcar #'D-GetOrigins (cdr commitgroup))))
	    (terpri))
         commits)
   )
)

(defun D-DisplayStep (newconfig lex surf lexrest surfrest tapes pos)
;;
;;  This displays a step in the match giving the strings tested 
;;  waits for a response from the user then continues.
;;
   (let   ()
      (terpri)
      (D-DisplayTape tapes pos)
      
      (cond
	 ((eq 'ERROR newconfig)
	    (D-DebugAnalyseFailure)
         )
	 (t
            (D-DebugDescribeCurrentState (list lex surf) newconfig)
	    (cond
	       ((memq (D-DebugPause) (DK-COM-EXITS)) 
		  nil)          ;; stop debugging
	       (t
                  (D-DebugStepMode                ;; continue matching
		     surfrest lexrest newconfig 
		     tapes (add1 pos)) )
            )
	 )
      )
   )
)

(defun D-DisplayTape (tapes pos)
;;
;;  Displays the current configuration of the tapes
;;
   (princ "Lexical string : ") (D-PList (cadr tapes)) (terpri)
   (princ "                 ") (D-PlaceTapeHead pos) (terpri)
   (princ "Surface string : ") (D-PList (car tapes)) (terpri)
)

(defun D-PlaceTapeHead (pos)
;;
;;  cute function that prints the tape head at the appropriate 
;;  position on the screen.  This will not work if the characters in the 
;;  tape are more than two characters long and it depends on what D-PList
;;  does so it is a bit hacky.
;;
   (cond
      ((equal pos 1)
	 (princ " ^"))
      (t
	 (princ "  ")
         (D-PlaceTapeHead (- pos 1)))
   )
)

(defun D-DebugDescribeCurrentState (pair config)
;;
;;  Describes the current state of the recent successful match
;;
   (let ( (active (D-FindActiveRules config))
	  (commit (D-FindCommitGroups config)) )
      (princ "The pair ") (princ pair) (princ " was licensed by: ")
      (D-PList D-LICENCE-RULES) (terpri)
      (cond
	 (D-TERMINATED-RULES-THIS-TIME
            (princ "The following rule(s) have terminated:")
            (D-PList D-TERMINATED-RULES-THIS-TIME) (terpri)))
      (cond
	 (commit
	    (princ "Right hand side(s) pending for") (terpri)
	    (mapcar
	       #'(lambda (commitgroup)
		  (princ "   the rule(s)") 
		  (D-PList
		     (apply #'append
			(mapcar #'D-GetOrigins (cdr commitgroup))))
		  (princ " for the pair ")
		  (princ (car commitgroup)) (terpri))
               commit)))
      (cond
	 (active
	    (princ "Left hand side(s) active: ")
	    (D-PList (apply #'append (mapcar #'D-GetOrigins active)))
	    (terpri)))
      (D-InitDebugGlobalsPerState)   ;; ready for next transition
   )
)

(defun D-DebugAnalyseFailure ()
;;
;;  Print out reason for failure of match
;;
   (princ "The match failed because:") (terpri)
   (cond
      (D-BLOCKED            ;; a number of rules blocked
	 (princ "   The rule")
	 (D-PList D-BLOCKED)  ;; I don't when there could be list though
	 (princ " blocked")
	 (terpri))
      (D-FAILED-RULE
	 (princ "   The pair ")
	 (print (car D-FAILED-RULE))
	 (princ " was not accepted by:")
	 (D-PList (cdr D-FAILED-RULE)) (terpri))
      (D-FAILEDLICENCE
	 (princ "   No rule licenses the pair ")
	 (princ D-FAILEDLICENCE) (terpri))
      (t
	 (princ "   Can't tell - sorry (Please tell the author)") (terpri))
   )
)

(defun D-FindLicencePairs (config)
;;
;;  finds all the licence states in this config
;;  returns a list of names of their original rules
;;
   (mapcar
      #'(lambda (state)
	 (cond
	    ((eq (car state) 'COMMIT) nil)
	    ((D-LicenceP state)          ;; if a licence state
	       (D-GetOrigins state))
	    (t
	       nil)))
      config)
)

(defun D-FindCommitGroups (config)
;;
;;  finds all the commit groups in this config
;;  returns a list of names of their original rules
;;
   (mapcan
      #'(lambda (state)
	 (cond
	    ((eq (car state) 'COMMIT) (ncons (cdr state)))
	    (t
	       nil)))
      config)
)

(defun D-FindActiveRules (config)
;;
;;  finds all the commit groups in this config
;;  returns a list of names of their original rules
;;
   (mapcan
      #'(lambda (state)
	 (cond
	    ((eq (car state) 'COMMIT) nil)
	    (t (ncons state))))
      config)
)

(defun D-DebugCheckPairMatch (lex surf configuration)
;;
;;    moves on the next possible states, returns new config before post 
;;    processing of new state.
;;
   (catch
      (D-DebugCheckNewConfiguration
	 (list lex surf)                ;; the current pair
         (D-DebugMoveAllStates
            (concat 'D lex surf)
            (cons
	       D-TRANSITIONSLIST      ;; and restart it
	       configuration))))
)

(defun D-DebugMoveAllStates (pair configuration)
;;
;;   Moves each state to its next one.
;;
   (mapcan 
      #'(lambda (state)
	 (cond
	    ((eq (car state) 'COMMIT)   ;; a COMMIT group
	       (D-DebugMoveCommitStates pair state) ;; nil if it terminates 
	    )
	    (t
	       (D-DebugFindNextState pair state)))
      )
      configuration)
)

(defun D-DebugFindNextState (pair state)
;;
;;   finds the next state.  And returns the new and old state as a
;;   pair             
;;
   (let ( (newstate (assq pair (cdr state))) )
      (cond
	 ((null newstate) nil)
	 ((assq (DK-D-ERROR) (cdr newstate))
	    (setq D-BLOCKED (D-GetOrigins state))
	    (throw 'ERROR))
	 (t (mapcar #'(lambda (x) x) (cdr newstate)))) ;; copy list
   )
)
      

(defun D-DebugCheckNewConfiguration (pair newconfig)
;;
;;   checks firstly that this is non nil and also that at least
;;   one state is a licence state
;;   
;;   All licence states are collected together into a COMMIT group. Also
;;   all COMMIT groups are checked to see that they are non-nil
;;
   (let  (licencestates simplestates)
      (setq simplestates
	 (mapcan
	    #'(lambda (state)
	       (cond
		  ((eq (car state) 'COMMIT)
		     (ncons state))
                  ((D-LicenceP state)          ;; if a licence state
		     (setq licencestates
			(cons state licencestates))
                     nil) ; nil so licence state will appear in a new OR clause
                  ((D-TerminalP state) 
		     (setq D-TERMINATED-RULES-THIS-TIME
		       (cons (D-GetOrigins state) D-TERMINATED-RULES-THIS-TIME))
                     (setq D-TERMINATED-RULES   ;; for overall match
		       (cons (D-GetOrigins state) D-TERMINATED-RULES))
		     nil)    ;; no need to continue
                  (t
		     (ncons state))            ;; state is a boring one
               ))
            newconfig))
      (setq D-LICENCE-RULES
	 (apply #'append (mapcar #'D-GetOrigins licencestates)))
      (cond
	 ((null licencestates)   ;; no licence pair for this transition
	    (setq D-FAILEDLICENCE pair)  
	    (throw 'ERROR))
         ((D-DebugTerminal licencestates)  ;; these are true so no new OR clause
	    simplestates
         )
         (t                     ;; return new config with new OR clause
	    (cons
	       (cons 'COMMIT (cons pair licencestates))
	       simplestates)
         )
      )
   )
)

(defun D-DebugMoveCommitStates (pair orconfig)
;;
;;   Checks a commit group of states.  This has a slightly
;;   different interpretation than when moving normal states
;;   A new or state is returned, or nil if this one has been proved true.
;;   (returns a list of states)
;;   If it no new states can be found this throws 'ERROR
;;   
   (let ( newconfig )
      (setq newconfig
	 (mapcan
	    #'(lambda (state)
	       (D-DebugFindNextState pair state))
            (cddr orconfig)))
      (cond
	 ((null newconfig) 
	    (setq D-FAILED-RULE
	       (cons
		  (cadr orconfig)     ;; the pair this group was licencing
		  (apply #'append (mapcar #'D-GetOrigins (cddr orconfig)))))
	    (throw 'ERROR))  ;; not found true
	 ((D-DebugTerminal newconfig)
	    (let  ( (termrules
		 (apply #'append (D-DebugTerminal newconfig))) )
	    (setq D-TERMINATED-RULES
	       (append termrules D-TERMINATED-RULES))
	    (setq D-TERMINATED-RULES-THIS-TIME
	      (append termrules D-TERMINATED-RULES-THIS-TIME))
	       nil))      ;; has been found true
	 (t (ncons (cons 'COMMIT (cons
	    (cadr orconfig) newconfig))))    ;; still pending
      )
   )
)

(defun D-DebugIsFeasibleP (lex surf concatedpair)
;;
;;  This checks to see if the pair is feasible.  This has
;;  been written for speed purposes.  If the two symbols in
;;  the pair are equal then it is (probably feasible)
;;  so it is accepted and it is left upto the automata to through
;;  it away (which they will).  If they are no feasible then
;;  there is a memq down with the list of non-eq pairs whcih are 
;;  held in D-UNUSUALFEASIBLES
;;
   (cond
      ((eq lex surf) t)
      (t          ;; ok have to find out the hard way
	 (memq concatedpair D-UNUSUALFEASIBLES)
      )
   )
)

(defun D-DebugFinal (config)
;;
;;   Returns true if the given list of states contains at least one
;;   final state
;;
   (cond
      ((null config) t)       ;; all are final states
      ((eq (caar config) 'COMMIT) nil);;an incomplete COMMIT is nonfinal
      ((D-FinalP (car config))   ;; final state
         (D-DebugFinal (cdr config))
      )
      (t nil)        ;; mis match of final state
   )
)

(defun D-DebugTerminal (config)
;;
;;   Checks to see if any states in the config are terminal states
;;   returns a list of rules that have terminated        
;;
   (cond
      ((null config) nil)
      ((D-TerminalP (car config))
	 (cons                           ;; find ALL terminated rules
	    (D-GetOrigins (car config))
	    (D-DebugTerminal (cdr config))))
      (t (D-DebugTerminal (cdr config)))
   )
)

(defun D-DebugPause ()
;;
;;   prints message and waits for the user to press return.  
;;   Any atom typed before the cr is returned
;;
   (princ "Press RETURN key to continue")
   (terpri)
   (let ((reply (D-ReadToEndOfLine 'EOF)))
      (cond
	 ((null reply) nil)
	 ((memq (car reply) (DK-COM-EXITS)) (car reply))
	 (t reply)))
)

