package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.awt.*;
import java.util.*;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.UlcHashtable;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.ui.base.*;

/**
 * This class is used as the model for 'form-based' widgets :
 * @see UIFormComponent hierarchy.
 * This makes it possible to group all the changes made to the various
 * widgets associated with one FormModel, to be sent across as one request
 * to the application side.
 * @see ULCFormModel#saveInput()
 *
 * It implements the protocol for synchronizing changes
 * with the application via a simple getValue/setValue API.
 */
public class UIFormModel extends UIStandardFormModel implements IEnableListener {
	protected UlcHashtable fData = null;
	protected boolean fVetoChanges = false;
	protected UlcHashtable fChangedKeys;
	protected int fNotificationPolicy = FORM_NOTIFICATION_ON_REQUEST;
	protected Vector fEmptyStateListeners = null;
	protected static Boolean fgDummy = new Boolean(false);
	protected static Object fgNull = new Object();
	public static final boolean DEBUG = false;
	protected boolean fNotifiedUlcOfChanges = false;
public UIFormModel() {
	fData = new UlcHashtable();
	fChangedKeys = new UlcHashtable();
}
/**
 * Add a component who will be notified when the state 
 * of the receiver changes.
 */
public void addEnableListener(IEnableListenerTarget component) {
	Assert.isNotNull(component);
	if (fEmptyStateListeners == null)
		fEmptyStateListeners = new Vector();
	fEmptyStateListeners.addElement(component);
	component.setEnabled(shouldEnableListener());
}
/**
 * Cancel all current input
 */
protected void cancelInput() {
	fChangedKeys.clear();
	enableListeners();
}
/**
 * Invalidate cell range.
 * Typically called from ULC via "change".
 */
void change(Anything args) {
	// change notification from ULC

	if (DEBUG) {
		System.out.print("change: ");
		args.dump(System.out);
	}
	String key = args.get("key", null);
	if (key != null) {
		setValueAt2(InvalidCell.theInvalidCell(), key);
	} else { // everything
		fData.clear();
		cancelInput();
	}
	notify(args.get("type", FORM_MODEL_CHANGED), key);
	checkForInvalidFormValueUpdate();
}
void enableListeners() {
	boolean state = shouldEnableListener();
	if (fEmptyStateListeners != null) {
		Enumeration e = fEmptyStateListeners.elements();
		while (e.hasMoreElements()) {
			IEnableListenerTarget c = (IEnableListenerTarget) e.nextElement();
			c.setEnabled(state);
		}
	}
	if (state != fNotifiedUlcOfChanges) {
		fNotifiedUlcOfChanges = state;
		Anything a = new Anything();
		a.put("c", new Anything(state));
		sendOptionalEventULC("hasChanges", a);
	}
}
/**
 * Returns an attribute value for the record
 * with the unique identifier key.
 */
public Object getValueAt(String key) {
	Object o = fData.get(key);
	if (o instanceof InvalidCell) {
		o = PendingCell.thePendingCell();
		fData.put(o, key);
		requestData(key);
	} else
		if (o == null) {
			if (!fData.containsKey(key)) {
				o = PendingCell.thePendingCell();
				fData.put(o, key);
				requestData(key);
			}
		}
	if (o == fgNull) {
		return null;
	}
	return o;
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("setData")) { // data updates from ULC
		setData(args);
		return;
	}
	if (request.equals("change")) { // change notifications from ULC
		change(args);
		return;
	}
	if (request.equals("flush")) {
		notify(FORM_MODEL_ABOUT_TO_SAVE, null); // we do this only on a request from the ULC server
												// since some widgets may not have lost their focus as yet
		saveInput(args);
		return;
	}
	if (request.equals("setVeto")) {
		fVetoChanges = args.asBoolean(false);
		return;
	}
	if (request.equals("setNotificationPolicy")) {
		fNotificationPolicy = args.asInt(fNotificationPolicy);
		return;
	}	
	super.handleRequest(conn, request, args);
}
/**
 * remove the given component as listener from being informed
 * when the state of the receiver changes from enabled to disabled,
 * or vice-versa.
 *
 * @param component : The Component which will act as listener.
 */
public void removeEnableListener(IEnableListenerTarget component) {
	Assert.isNotNull(component);
	if (fEmptyStateListeners == null)
		return;
	fEmptyStateListeners.removeElement(component);
}
/**
 * The data corresponding to the given key is not available
 * in the receiver. Make a request for this data from the
 * application. Place dummy objects in this slot till the
 * reply is received (asynchronously).
 */
protected void requestData(String key) {
	Anything n = new Anything();
	if (fAttrNames != null) {
		for (int col = 0; col < fAttrNames.size(); col++) {
			String name = fAttrNames.slotName(col);
			Object o = fData.get(name);
			if (o == null || o instanceof InvalidCell) {
				setValueAt2(PendingCell.thePendingCell(), name);
				n.put(name, new Anything());
			}
		}
	}
	else {
		setValueAt2(PendingCell.thePendingCell(), key);
		n.put(key, new Anything());
	}
	if (n.size() > 0) {
		sendULC("getData", n);
		if (DEBUG) {
			System.out.print("getData: ");
			n.dump(System.out);
		}
	}
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	super.restoreState(conn, args);
	fVetoChanges = args.get("veto", false);
	fNotificationPolicy = args.get("notificationPolicy", FORM_NOTIFICATION_ON_REQUEST);
	Anything data = args.get("data");
	if ((data != null) && (!data.isNull()))
		setData(data);
}
void saveInput(Anything args) {
	if (fChangedKeys != null) {
		Enumeration e = fChangedKeys.keys();
		Anything a = new Anything();
		args.put("data", a);
		while (e.hasMoreElements()) {
			String key = (String) e.nextElement();
			if (key != null)
				a.put(key, convert(fData.get(key)));
		}
		fChangedKeys.clear();
		sendULC("setData", args);
		enableListeners();
		if (DEBUG) {
			System.out.print("setData: ");
			a.dump(System.out);
		}
	}
}
/**
 * Change table area immediately.
 * Typically called from ULC via "sendRange" and "change".
 */
public void setData(Anything columns) {
	if (DEBUG) {
		System.out.print("setData: ");
		columns.dump(System.out);
	}
	if (columns != null) {
		for (int col = 0; col < columns.size(); col++) {
			Object o = convert(fConnection, columns.get(col));
			String key = columns.slotName(col);
			setValueAt2(o, key);
			notify(FORM_MODEL_CHANGED, key);
		}
		checkForInvalidFormValueUpdate();
	}
}
/**
 * Change cell contents directly (fVeto == false)
 * or send a setData request to ULC (fVeto == true).
 * Typically called from widgets.
 */
public void setValueAt(Object value, String key, IFormModelListener sender, int notificationPolicy) {
	Object old = getValueAt(key);
	if ((old == null) || !old.equals(value)) {
		fData.put(key, value);
		fChangedKeys.put(key, fgDummy);
		if (fVetoChanges) {
			saveInput(new Anything(-1));
		}
		else {
			enableListeners();
			notify(FORM_MODEL_CHANGED, key, sender);
		}
		if (fNotificationPolicy == FORM_NOTIFICATION_IMMEDIATE) {
			saveInput(new Anything(-1));
		}
	}
}
/**
 * Change a single cell and clear its change status.
 * Typically called from ULC via "sendRange" and "change".
 */
void setValueAt2(Object cell, String key) {
	if (cell == null)
		cell = fgNull;
	fData.put(key, cell);
	if (fChangedKeys != null) {
		fChangedKeys.remove(key);
		enableListeners();
	}
}
/**
 * Return true if the component for which I am an enabler should be enabled.
 *
 */
public boolean shouldEnableListener() {
	boolean state = false;
	if (fChangedKeys != null)
		state = fChangedKeys.size() > 0;
	return state;
}
}
