package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997 Object Technology International Inc.
 *
 *
 * This class is used for handling the Button widget. It can have an icon
 * &/or a label associated with it, as well as a tool-tip and a mnemonic.
 */
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.ui.base.*;
public class UIButton extends UIComponent implements ActionListener {
	protected UiJButton fButton = null;
	protected boolean fDisableOnAction= false;
	protected UITrigger fTrigger= null;
/**
 * Notification that an action has been performed, i.e. the 
 * the button has been pressed.
 * 
 * @param event : The notification event.
 */
public void actionPerformed(ActionEvent event) {
	sendEventULC("action", "cmd", new Anything(event.getActionCommand()));
	if (fDisableOnAction)
		setEnabled(false);
	if (fTrigger != null)
		fTrigger.trigger(fConnection, TRIGGER_ON_ACTION, this, null);
}
/**
 * The receiver is being destroyed. Release all the 
 * associated resources.
 */
public void free() {
	if (fButton != null) {
		fButton.removeActionListener(this);
		fButton = null;
	}
	super.free();
}
/**
 * Answer the component associated with the
 * receiver.
 */
public Component getComponent() {
	return fButton;
}
/**
 * Answer the receiver's icon.
 */
public Icon getIcon() {
	return fButton.getIcon();
}
/**
 * Answer the receiver's label.
 */
public String getLabel() {
	return fButton.getText();
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("setIcon")) {
		UIIcon icon = null;
		if (!args.isNull())
			icon = (UIIcon) getManaged(UIIcon.class, conn, args);
		setIcon(icon);
		return;
	}
	if (request.equals("setMnemonic")) {
		setMnemonic(args.asInt(-1));
		return;
	}
	if (request.equals("setDisableOnAction")) {
		setDisableOnAction(args.asBoolean(fDisableOnAction));
		return;
	}
	if (request.equals("setTrigger")) {
		if (args.isNull())
			fTrigger = null;
		else
			fTrigger = (UITrigger) getManaged(UITrigger.class, conn, args);
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	String label = args.get("label", "");
	UiLabelAndMnemonic u = internalParseLabelForMnemonic(label);
	label = u.fLabel;
	UIIcon icon = (UIIcon) getManaged(UIIcon.class, conn, args.get("icon"));
	if ((icon != null) && (icon.getImage(this, "setIcon") != null)) {
		ImageIcon imageIcon = new ImageIcon(icon.getImage(this, "setIcon")); // 041
		if (label.equals(""))
			fButton = new UiJButton(imageIcon);
		else
			fButton = new UiJButton(label, imageIcon);
	} else {
		fButton = new UiJButton(label);
	}
	if (u.fMnemonic != null)
		setMnemonic(u.fMnemonic.charValue());
	if (args.isDefined("mnemonic"))
		setMnemonic(args.get("mnemonic", -1));
	super.restoreState(conn, args);
	setDisableOnAction(args.get("disableOnAction", fDisableOnAction));
	fButton.addActionListener(this);
	fTrigger = (UITrigger) getManaged(UITrigger.class, conn, args.get("trigger"));
}
/**
 * Set the background of the receiver to the color defined
 * by the RGB int values <code>red, green, and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
public void setBackgroundColor(int red, int green, int blue, boolean refresh) {
	setBackgroundColorWithOpaque(red, green, blue, refresh);
}
/**
 * If the disable on action is set to true then the button is disabled on 
 * user click. The application is responsible for enabling the button either 
 * by explicitly calling setEnable or via an enabler.
 *
 * @param disableOnAction if true button gets disabled each time it is clicked.
 */
public void setDisableOnAction(boolean disableOnAction) {
	fDisableOnAction = disableOnAction;
}
/**
 * Set the given icon as the receiver's icon.
 *
 * @param icon . the UIIcon to be set.
 */
public void setIcon(UIIcon icon) {
	if (icon != null) {
		Image image = icon.getImage(this, "setIcon");
		if (image != null) {
			fButton.setIcon(new ImageIcon(image)); // 041
		}
		else {
			fButton.setIcon(null);
		}
	}
	else
		fButton.setIcon(null);
}
/**
 * Set the given label as the receiver's label.
 *
 * @param label . the String to be set.
 */
public void setLabel(String label) {
	fButton.setText(label);
}
/**
 * Set the given mnemonic as the mnemonic for activating
 * the receiver.
 *
 * @param keyCode . the menmonic to be set.
 */
protected void setMnemonic(int keyCode) {
	if (keyCode != -1)
		fButton.setMnemonic((char) keyCode);
}
public void updateCachedResource(String methodName, ICachedResource resource, Object userData) {
	if (methodName.equals("setIcon")) {
		UIIcon icon = (UIIcon) resource;
		setIcon(icon);
		getBasicComponent().invalidate();
		forceFramePack();
		return;
	}
	super.updateCachedResource(methodName, resource, userData);
}
}
