package com.ibm.ulc.examples.ManagementStructure;

import java.util.*;

/**
 * This class is used for modelling an Employee in an organization.
 * The organizational hierarchy is to be represented as a "tree" structure,
 * wherein the Employee is a "node" of the tree.
 * The employee's immediate boss is his "parent", and all the employees
 * directly reporting to an Employee are his "children" in the tree.
 */
public class Employee { 

	/**
	 * The name of the employee
	 */
	protected String 	fName;

	/**
	 * The employee's title in the organization
	 */	
	protected String 	fTitle;	

	/**
	 * The employee's rank in the organization.
	 * Rank 1 = CEO of the company. 
	 * Rank 2 = Direct reports to the CEO
	 * and so on.
	 */		
	protected int 		fRank; 

	/**
	 * The employees who report directly to the receiver.
	 */
	protected Vector 	fImmediateReports;

	/**
	 * The immediate boss of the receiver
	 */
	protected Employee 	fBoss;

	/**
	 * Is this employee expanded on the UI. This flag is not
	 * used for any business logic as yet.
	 */
	private boolean		fIsExpanded;

	/**
	 * Has this employee been expanded (at least) once on the UI. 
	 */
	private boolean		fHasBeenExpandedOnce;

	/**
	 * The name of the icon to be displayed for this employee
	 */
	private String		fIconName;		
/**
 * Constructing a new employee. By default, he has no
 * direct reports, and is a collapsed node on the UI.
 */
public Employee() {
	fImmediateReports= new Vector();
	setCollapsed();
}
/**
 * Constructing a new employee with the given name
 * and organizational title.
 */
public Employee(String employeeName, String title) {
	this();
	fName= employeeName;
	fTitle= title;
}
/**
 * Constructing a new employee with the given name,
 * organizational title and immediate boss.
 */
public Employee(String employeeName, String title, Employee boss) {
	this(employeeName, title);
	setBoss(boss);
}
/**
 * Add a new immediate-report to the receiver. This person
 * will have a rank one lower than that of the receiver.
 */
public void addImmediateReport(Employee employee) {
	fImmediateReports.addElement(employee);
	employee.setRank(fRank + 1);
}
/**
 * Add a new immediate-report to the receiver, at the given
 * index (in the receiver's collection of direct-reports). 
 * This person will have a rank one lower than that of the 
 * receiver.
 */
public void addImmediateReport(Employee employee, int index) {
	try {
		fImmediateReports.insertElementAt(employee, index);
	}
	catch (ArrayIndexOutOfBoundsException e) {
		fImmediateReports.addElement(employee);
	}
	employee.setRank(fRank + 1);
}
/**
 * Add a new immediate-report to the receiver, after the given
 * employee (in the receiver's collection of direct-reports). 
 * This person will have a rank one lower than that of the 
 * receiver.
 */
public void addImmediateReportAfter(Employee newEmployee, Employee existingEmployee) {
	int indexForInsertion= getImmediateReports().indexOf(existingEmployee) + 1; //immediately after
	addImmediateReport(newEmployee, indexForInsertion);
}
/**
 * Add a new immediate-report to the receiver, before the given
 * employee (in the receiver's collection of direct-reports). 
 * This person will have a rank one lower than that of the 
 * receiver.
 */
public void addImmediateReportBefore(Employee newEmployee, Employee existingEmployee) {
	int existingIndex= getImmediateReports().indexOf(existingEmployee);
	int indexForInsertion= existingIndex; //Our new node should get this index !
	addImmediateReport(newEmployee, indexForInsertion);
}
/**
 * When no employee is selected, create a dummy employee
 * to display in the UI.
 */
protected static Employee createDummyEmployee() {
	return new Employee("", "");
}
/**
 * The receiver deletes himself by removing himself from
 * his boss' list of reports. This also implies that all the
 * children of the receiver are also deleted.
 *
 * Note: The root node of the tree cannot be deleted. For that,
 * see ULCTreeModel#changed()
 *
 */
public boolean delete() {
	//if (getChildCount() > 0) return false; //One possible scenario
	fBoss.deleteImmediateReport(this);
	return true;
}
/**
 * Delete all immediate-reports from the receiver. 
 */
public void deleteAllImmediateReports() {
	fImmediateReports.removeAllElements();
}
/**
 * Delete the given immediate-report from the receiver. 
 */
public void deleteImmediateReport(Employee employee) {
	fImmediateReports.removeElement(employee);
}
/**
 * Answer the immediate boss of the receiver
 */
public Employee getBoss() {
	return fBoss;
}
/**
 * Answer the name of the immediate boss of the receiver
 */
public String getBossName() {
	if (fBoss == null) return "";
	else return fBoss.getName();
}
/**
 * Answer the child for the receiver at the given index.
 * Answer null if an incorrectindex has been specified.
 */
public Employee getChildAtIndex(int index) {
	try {
		return (Employee) (fImmediateReports.elementAt(index));
	}
	catch (ArrayIndexOutOfBoundsException e) {
		return null;
	}
}
/**
 * Answer the number of children for the receiver
 */
public int getChildCount() {
	return fImmediateReports.size();
}
/**
 * Answer the children for the receiver
 */
public Vector getChildren() {
	return fImmediateReports;
}
/**
 * Answer the icon to be used for representing the receiver
 * as a node in the organization hierarchy (tree).
 *
 * Any application-specific logic can be used in here...
 */
public String getIconName() {
	if (fIconName != null)
		return fIconName;
	//Else : Nothing has been set using setIcon()
	//Note that we do *not* set fIconName in such a case :
	if (isCEO())
		return "red-ball.gif";
	if (fRank == 2)
		return "blue-ball.gif";
	if (fRank == 3)
		return "magenta-ball.gif";
	if (fRank == 4)
		return "green-ball.gif";
	return "cyan-ball.gif";
}
/**
 * Answer the direct reports (immediate children) for the receiver.
 */
public Vector getImmediateReports() {
	return fImmediateReports;
}
/**
 * Add a new immediate-report to the receiver. This person
 * will have a rank one lower than that of the receiver.
 */
public int getIndexOf(Employee parent, Employee child) {
	return parent.getImmediateReports().indexOf(child);
}
/**
 * Answer the label to be used for representing the receiver
 * as a node in the organization hierarchy (tree).
 */
public String getLabel() {
	return fTitle;
}
/**
 * Answer the name of the receiver.
 */
public String getName() {
	return fName;
}
/**
 * Implementing the ULCTreeModel API:
 * Answer the parent for the given object.
 *
 */
public Object getParent() {
	return getBoss();
}
/**
 * Answer the organizational title of the receiver.
 */
public String getTitle() {
	return fTitle;
}
/**
 * Answer boolean whether the receiver has been expanded at least once
 * so far.
 */
public boolean hasBeenExpandedOnce() {
	return fHasBeenExpandedOnce;
}
/**
 * Answer boolean, whether the receiver is the CEO
 * of the company.
 */
public boolean isCEO() {
	return fRank == 1;
}
/**
 * Set the given employee as the immediate boss of the
 * receiver. Also add the receiver to this boss' list
 * of immediate reports.
 * Note that the CEO is his own boss, hence is not added
 * into this list.
 */
public void setBoss(Employee boss) {
	fBoss= boss;
	if (!(this.isCEO())) boss.addImmediateReport(this);
}
/**
 * Set the given employee as the immediate boss of the
 * receiver. Also add the receiver to this boss' list
 * of immediate reports, at the specified index.
 *
 * This index is important since the application also needs to keep it's
 * elements in the right order, if it wants to add elements in the UI at
 * specific indices.
 *
 */
public void setBoss(Employee boss, int index) {
	fBoss= boss;
	if (!(this.isCEO())) boss.addImmediateReport(this, index);
}
/**
 * Set the given employee as the immediate boss of the
 * receiver. Also add the receiver to this boss' list
 * of immediate reports, after the given existingEmployee.
 */
public void setBossAfter(Employee boss, Employee existingEmployee) {
	fBoss= boss;
	if (!(this.isCEO())) boss.addImmediateReportAfter(this, existingEmployee);
}
/**
 * Set the given employee as the immediate boss of the
 * receiver. Also add the receiver to this boss' list
 * of immediate reports, before the given existingEmployee.
 */
public void setBossBefore(Employee boss, Employee existingEmployee) {
	fBoss= boss;
	if (!(this.isCEO())) boss.addImmediateReportBefore(this, existingEmployee);
}
/**
 * The UI has set this node as collapsed. Keep the
 * receiver in synch with this state.
 * This state could be used for setting a different
 * icon for the receiver (say). Not used currently. 
 */
public void setCollapsed() {
	fIsExpanded= false;
}
/**
 * The UI has set this node as expanded. Keep the
 * receiver in synch with this state.
 * This state could be used for setting a different
 * icon for the receiver (say). Not used currently.
 */
public void setExpanded() {
	fIsExpanded= false;
}
/**
 * Set boolean whether the receiver has been expanded at least once
 * so far.
 */
public void setHasBeenExpandedOnce(boolean bool) {
	fHasBeenExpandedOnce= bool;
}
/**
 * Answer the icon to be used for representing the receiver
 * as a node in the organization hierarchy (tree).
 *
 * Any application-specific logic can be used in here...
 */
public void setIconName(String name) {
	fIconName = name;
}
/**
 * Set the receiver's label to a new label. This newLabel
 * is the string that is displayed on the UI side as this
 * object's label.
 */
public void setLabel(String newLabel) {
	setTitle(newLabel);
}
/**
 * Set the receiver's name to a new name.
 */
public void setName(String name) {
	fName= name;
}
/**
 * Set the receiver's name to a new name.
 * Note: Ranks are assigned internally, with the 
 * CEO as rank 1.
 */
public void setRank(int rank) {
	fRank= rank;
}
/**
 * Set the receiver's organizational title to a new title.
 */
public void setTitle(String title) {
	fTitle= title;
}
/**
 * Set the receiver's organizational title to a new title.
 */
public String toString() {
	return getName();
}
}
