package com.ibm.ulc.examples.CustomerInformation;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import com.ibm.ulc.application.*;

/**
 * This class is intended to adapt the account to
 * the requirements of being placed on a ULC Form,
 * i.e. of adhering to the ULCFormModel interface.
 */
public class AccountAdapter extends ULCFormModel {
	protected Account fAccount;
	private CustomerInformationEditor fEditor;			// for showing alert after validation
	protected boolean fValidationResult;				// the result of the validation process
/**
 * By default, no account is selected (when this receiver is
 * being created. Hence a dummy instance is created, which makes
 * it easier for the user to modify.
 */
public AccountAdapter() {
	setNewDummyAccount();
	fValidationResult= true;				// success by default
}
public AccountAdapter (Account selectedAccount) {
	setAccount(selectedAccount);
}
public AccountAdapter(CustomerInformationEditor editor) {
	this();
	fEditor= editor;
}
/**
 * Any changes made to the selected account, have been 
 * Cancelled. So the receiver now just needs to make
 * sure that the selected account's original data is
 * being displayed in the account-details.
 *
 * Note that at this point, the fAccount value has not
 * changed (since we did not flush any changes). So we
 * need to call the basicSetAccount(fAccount) method here,
 * and not just the setAccount(..) - the latter will not
 * do the required refreshing since it'll see no change !
 */
 
public void accountModificationCancelled () {
	basicSetAccount(fAccount);
}
/** 
 * Set the account to be displayed. This method is
 * required since at times we do not want to compare
 * the accounts before doing this setting.
 *
 * @see accountModificationCancelled()
 */
void basicSetAccount(Account newlySelectedAccount) {
	fAccount= newlySelectedAccount;
	notifyChange();
}
/**
 * Create a new account to be held by the receiver,
 * and put all the changes made by the user (in the account
 * details) into this new account.
 * Note some of the reasons for this method implementation :
 * 	The flush has to be done only After a clone has been
 *		set as fAccount. Else the currently selected account
 *		will get over-written with the new values !
 * And finally, we answer null if the account number is incorrect,
 * since this needs to stopthe account being added to the list :
 * @ see setValueAt(Object String)
 * @ see AccountsAdapter addAccount(Account)
 */
public Account copyChangesIntoNewAccount() {
	Account cloneAccount= (Account) fAccount.clone();
	fAccount= cloneAccount;
	validateAndSaveChanges();
	if (!fValidationResult) return null; // validation unsuccessful
	else return fAccount;
}
public Account getAccount() {
	return fAccount;
}
/**
 * This is the abstract method in the superclass, which needs to be
 * over-ridden. Do the necessary processing to answer the corresponding
 * value from the 'real' model. 
 *
 * @param key 	is the key for an attribute in the receiver. So this
 *					would have a value (say) "Number", "Type" etc.						
 */

public Object getValueAt(String key) {
	return fAccount.getValueForAttributeName(key);
}
/** 
 * Factored out to do notification, so that
 * it can be called by someone who doesn't need
 * to import the IDefaults constants.
 */
void notifyChange() {
	notify(FORM_MODEL_CHANGED, null);
}
/** The adapter keeps track of the currently-selected
 * account, and issues the change-notification for the
 * actual UI update.
 *
 * Check required : That there is indeed a different account selected. 
 * If not, we should not send any updates, events etc.
 * Note that this comparision of the accounts is done for identity,
 * not just equality. This is because it is the SAME instance of 
 * Account which is being checked for.
 */
 
void setAccount(Account newlySelectedAccount) {
	if (newlySelectedAccount != fAccount) 
		basicSetAccount(newlySelectedAccount);
}
/**
 * By default, no account is selected (when this receiver is
 * being created. Hence a dummy instance is created, which makes
 * it easier for the user to modify...
 */
 
public void setNewDummyAccount() {
	setAccount(Account.createInitialDummyAccount());
}
/**
 * This is again a superclass method in the superclass, although not
 * abstract. It provides the hook to update the model with any changes
 * made in the UI. 
 *
 * @param key 		is the identifier for an attribute in the receiver. So this
 *						would have a value (say) "Number", "Type" etc.						
 * @param newValue is the value changed by the user in the UI.			
 */

public void setValueAt(Object newValue, String key) {
	String stringValue = String.valueOf(newValue);
	if (key.equals("Number"))
		fAccount.setNumber(stringValue);
	if (key.equals("Type"))
		fAccount.setType(stringValue);
	if (key.equals("Currency"))
		fAccount.setCurrency(stringValue);
	if (key.equals("Charges"))
		fAccount.setCharges(new Double(stringValue).doubleValue());
	if (key.equals("Balance"))
		fAccount.setBalance(new Double(stringValue).doubleValue());
	if (key.equals("Limit"))
		fAccount.setLimit(new Double(stringValue).doubleValue());
}
/**
 * Backup the current state, and restore if there is an
 * invalid input.
 */
public void validateAndSaveChanges() {
	Account backupAccount= (Account) fAccount.clone();
	saveInput();
	fValidationResult= new AccountValidator(fAccount, fEditor).validate();
	if (!fValidationResult) fAccount.mergeWith(backupAccount);
}
}
