package com.ibm.ulc.comm;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.net.*;
import java.util.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.base.IDefaults;

/**
 * A connection to a ULC peer.
 * A RequestProcessor is used to pass requests coming from the peer to an event loop.
 * The send() method allows a caller to pass a request to the peer.
 * An instance of an UlcTransport is used for the actual communication.
 * Incoming and outgoing communication is asynchronous.
 *
 * @see IRequestProcessor
 * @see UlcTransport
 */
public abstract
class UlcConnection extends UlcObject implements IConnection {

	public static boolean fgIPAddr= false;
	public static boolean fgDebug= false;
	public static final int INVALIDPORT= -1;
	
	private static int fgCount= 0;
	
	protected boolean fDebug= false;
	
	/* Addressing */
	private static String fgLocalHost= null;
	private String fUrlString= null;

	/* Identification */
	private String fConnectionId= null;
	private String fClientData= null;

	/* Monitoring */
	private IRequestFilter fRequestFilter= null;
	private Vector fRequestListeners= null;

	/* Transport */
	private UlcTransport fTransport= null;

	/* Multi-threading */
	private IRequestProcessor fRequestProcessor= null;
	private boolean fTerminated= true;

	/**
	 * User defined error message for connection
	 * down errors
	 * @serial
	 */
	private String fConnDownErrorMsg= null;
/**
 * Creates a connection for the given transport, request processor,
 * and connection identification data.
 *
 * Will use the given transport.
 *
 * @param transport com.ibm.ulc.comm.UlcTransport
 * @param rp com.ibm.ulc.comm.IRequestProcessor
 * @param connId java.lang.String
 * @param clientData java.lang.String
 *
 * @see UlcTransport
 */
protected UlcConnection(UlcTransport transport, IRequestProcessor rp,
						String connId, String clientData) {
	this(transport.getUrlString(), rp, connId, clientData);
	fTransport= transport;
}
/**
 * Creates a connection for the given url, request processor,
 * and connection identification data.
 *
 * Will establish a transport based on the given URL.
 *
 * @param urlString java.lang.String
 * @param rp com.ibm.ulc.comm.IRequestProcessor
 * @param connId java.lang.String
 * @param clientData java.lang.String
 *
 * @see UlcTransport
 */
protected UlcConnection(String urlString, IRequestProcessor rp,
						String connId, String clientData) {
	fUrlString= urlString;
	fRequestProcessor= rp;
	fConnectionId= connId;
	fClientData= clientData;
	fDebug= fgDebug;
}
/**
 * Register an IRequestListener that will be notified
 * each time a request is sent or received.
 * @see IRequestListener
 */
public void addRequestListener(IRequestListener listener) {
	if (fRequestListeners == null)
		fRequestListeners= new Vector();
	fRequestListeners.addElement(listener);
}
private boolean checkTerminated(String where, String args) {
	if (fTerminated) {
		String msg;
		if (args != null)
			msg= "connection terminated; " + args;
		else
			msg= "connection terminated";
		trouble(where, msg);
	}
	
	return fTerminated;
}
/**
 * Closes the connection (including the transport)
 */
public synchronized void close() {
	close(null);
}
/**
 * Closes the connection (including the transport)
 * and passes the given exception on to connection controllers.
 */
synchronized void close(UlcTransportException ex) {
	if (fTerminated)	
		// already terminated
		return;
	if (fDebug)
		System.out.println("<UlcConnection.close");
	fTerminated = true;
	if (fTransport != null) {
		fTransport.close();
		fTransport = null;
	}
	fRequestProcessor = null;
	fRequestFilter = null;
	fRequestListeners = null;
	shutdownHook();
	Server.propagateConnectionEnded(this, ex);
	logConnectionRemoved(fDebug);
	if (fDebug)
		System.out.println(">UlcConnection.close");
}
/**
 * Creates a request. Overridde to define the request to
 * to be created.
 */
abstract public Request createRequest();
public String getApplicationName() {
	String urlFile;
	if (fTransport == null)
		urlFile= getFileFromUrlString(fUrlString);
	else
		urlFile= fTransport.getUrlFile();
	if (urlFile == null)
		urlFile= "***unknown***";
	return urlFile;
}
public String getClientData() {
	return fClientData;
}
/**
 * Return a user defined error message for connection
 * down errors.
 *
 * @since	R3.1 
 * @return	the error message as <code>String</code>
 */
public String getConnDownErrorMsg() {
	return fConnDownErrorMsg;
}
/**
 * Helper method to extract the 'file' part from a URL string.
 */
public static String getFileFromUrlString(String urlString) {
	// Parse the given URL string and pick out the file name.
	// Let the URL class do the work;
	String file= null;
	urlString= prepareUrlString(urlString);
	if (urlString != null) {
		try {
			URL url= new URL(urlString);
			file= url.getFile();
		}
		catch (MalformedURLException e) {
		}

		// remove '/' at beginning
		if (file != null)
			if (file.length() > 0 && file.charAt(0) == '/')
				file= file.substring(1);		
	}
	
	return file;
}
public String getHost() {
	String urlHost;
	if (fTransport == null)
		urlHost= getHostFromUrlString(fUrlString);
	else
		urlHost= fTransport.getHost();
	if (urlHost == null)
		urlHost= "***unknown***";
	return urlHost;
}
/**
 * Helper method to extract the 'host' part from a URL string.
 */
public static String getHostFromUrlString(String urlString) {
	// Parse the given URL string and pick out the host name.
	// Let the URL class do the work;
	String host= null;
	urlString= prepareUrlString(urlString);
	if (urlString != null) {
		try {
			URL url= new URL(urlString);
			host= url.getHost();
		}
		catch (MalformedURLException e) {
		}
	}
	
	return host;
}
public String getId() {
	return fConnectionId;
}
/**
 * Helper method to return the local host either as host name
 * or as IP address (dotted string) depending on preference set.
 */
public static synchronized String getLocalHost() {
	if (fgLocalHost == null)
		try {
			InetAddress localAddr= InetAddress.getLocalHost();
			if (fgIPAddr)
				fgLocalHost= localAddr.getHostAddress();
			else
				fgLocalHost= localAddr.getHostName();
		} catch (UnknownHostException uhe) {
			System.out.println("@@@@@ Connection.getLocalHost(): cannot get local host!?");
			fgLocalHost = "127.0.0.1";
		}
	
	return fgLocalHost;
}
/**
 * Return a URL string that identifies the local end point
 * of the connection.
 */
public String getLocalUrlString() {
	if (fTransport != null)
		return fTransport.getLocalUrlString();
	return null;
}
public int getPort() {
	int urlPort;
	if (fTransport == null)
		urlPort= getPortFromUrlString(fUrlString);
	else
		urlPort= fTransport.getPort();
	return urlPort;
}
/**
 * Helper method to extract the 'port' part from a URL string.
 */
public static int getPortFromUrlString(String urlString) {
	// Parse the given URL string and pick out the port number.
	// Let the URL class do the work;
	int port= INVALIDPORT;
	urlString= prepareUrlString(urlString);
	if (urlString != null) {
		try {
			URL url= new URL(urlString);
			port= url.getPort();
		}
		catch (MalformedURLException e) {
		}
	}
	
	return port;
}
/**
 * Helper method to extract the 'protocol' part from a URL string.
 */
public static String getProtocolFromUrlString(String urlString) {
	// Parse the given URL string and pick out the protocol name.
	String protocol= null;
	if (urlString != null) {
		int ind= urlString.indexOf(":");
		if (ind >= 0)
			protocol= urlString.substring(0,ind);
		else
			protocol= "http";
	}
	
	return protocol;
}
/**
 * Helper method to extract the 'reference' ('anchor') part from a URL string.
 */
public static String getReferenceFromUrlString(String urlString) {
	// Parse the given URL string and pick out the reference part.
	// Let the URL class do the work;
	String ref= null;
	urlString= prepareUrlString(urlString);
	if (urlString != null) {
		try {
			URL url= new URL(urlString);
			ref= url.getRef();
		}
		catch (MalformedURLException e) {
		}
	}
	
	return ref;
}
/**
 * Return a URL string that identifies the remote end point
 * of the connection.
 */
public String getRemoteUrlString() {
	if (fTransport != null)
		return fTransport.getRemoteUrlString();
	return null;
}
/**
 * Returns the IRequestFilter object that is currently
 * installed or null.
 */
public IRequestFilter getRequestFilter() {
	return fRequestFilter;
}
/**
 * Returns the vector of currently registered IRequestListener
 * objects or null.
 */
public Vector getRequestListeners() {
	return fRequestListeners;
}
public UlcTransport getTransport() {
	return fTransport;
}
/**
 * Return a URL string that identifies the remote end point
 * of the connection.
 */
public String getUrlString() {
	return fUrlString;
}
public boolean isTerminated() {
	return fTerminated;
}
static synchronized void logConnectionAdded(boolean debug) {
	fgCount++;
	if (debug)
		System.out.println("+UlcConnections: " + fgCount);
}
static synchronized void logConnectionRemoved(boolean debug) {
	fgCount--;
	if (debug)
		System.out.println("-UlcConnections: " + fgCount);
}
/**
 * Posts a request to the connection's request queue.
 * Bypasses receive filters and listeners.
 */
public void postRequest(Request r) {
	if (fRequestProcessor != null)
		fRequestProcessor.addRequest(r);
}
static synchronized String prepareUrlString(String urlString) {
	// Make sure a URL can be constructed from given string
	// (replace arbitrary protocol by "http")
	String validUrl= null;
	if (urlString != null) {
		int ind= urlString.indexOf(":");
		if (ind >= 0)
			validUrl= "http" + urlString.substring(ind);
	}
	return validUrl;
}
/**
 * Receives a request and posts it to the connection's request queue.
 * Passes the request through an installed request filter
 * and notifies request listeners.
 */
public final boolean receive(Request request) {
	if (fRequestProcessor == null)
		return false;
		
	// filter first
	if (fRequestFilter != null)
		request= fRequestFilter.filterReceive(this,request);
	if (request == null)
		return false;

	// actual receive
	fRequestProcessor.addRequest(request);

	// notify listeners
	if (fRequestListeners != null)
		for (Enumeration e= fRequestListeners.elements(); e.hasMoreElements(); ) {
			IRequestListener listener= (IRequestListener) e.nextElement();
			listener.receivedRequest(this,request);
		}
	
	return true;
}
/**
 * Remove a previously registered IRequestListener
 */
public void removeRequestListener(IRequestListener listener) {
	if (fRequestListeners != null)
		fRequestListeners.removeElement(listener);
}
/**
 * Sends a request out using the installed transport.
 * Requests are sent asynchronously.
 */
public final synchronized boolean send(Request request) {
	if (checkTerminated("send", "request \"" + request.getName() + "\" discarded"))
		return false;
		
	// filter first
	if (fRequestFilter != null)
		request= fRequestFilter.filterSend(this,request);
	if (request == null)
		return false;

	// actual send
	fTransport.send(request);

	// notify listeners
	if (fRequestListeners != null)
		for (Enumeration e= fRequestListeners.elements(); e.hasMoreElements(); ) {
			IRequestListener listener= (IRequestListener) e.nextElement();
			listener.sentRequest(this,request);
		}
		
	return true;
}
/**
 * Set a user defined error message for connection
 * down errors.
 *
 * @since	R3.1 
 * @param	errorMessage	The <code>String</code> which should be used as error message 
 *
 */
public void setConnDownErrorMsg(String errorMessage) {
	fConnDownErrorMsg= errorMessage;
}
/**
 * Set the IRequestFilter object that will receive all requests just
 * before they are sent and all requests just after they are recieved.
 * @see IRequestFilter
 */
public void setRequestFilter(IRequestFilter filter) {
	fRequestFilter= filter;
}
	/**
	 * A hook that can be overridden to add additional actions when
	 * the connection is shut down.
	 */
	abstract protected void shutdownHook();
/**
 * Starts the connection. After this method, requests can be sent
 * and are received using the installed transport.
 */
public synchronized void start() throws UlcTransportException {
	UlcTransport.fgDebug= fDebug;
	if (!fTerminated)   // already started
		return;

	// finish transport setup
	if (fTransport == null) {
		fTransport= UlcTransport.create(fUrlString);
		if (fTransport == null) {
			String msg= "UlcConnection.start(). Cannot create transport! URL=" + fUrlString;
			if (fDebug)
				System.out.println(msg);
			UlcTransportException ex=
				new UlcTransportException(msg,IDefaults.TRANSPORT_EX_INIT_CONNECTION_EXCEPTION,(Exception)null);
			throw ex;
		}
	}
	fTransport.setConnection(this);

	// ready at this point:
	// - a transport (to send/receive data to/from peer)
	// - a request queue (to post requests to event processing loop)
	fTerminated= false;

	// tell transport to start working (asynchronously reading and writing)
	try {
		fTransport.start();
	}
	catch(UlcTransportException ex) {
		fTerminated= true;
		throw ex;
	}
	logConnectionAdded(fDebug);
}
}
