package com.ibm.ulc.comm;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import java.io.*;
import java.net.*;
import com.ibm.ulc.util.UlcObject;
import com.ibm.ulc.util.DebugThread;

/**
 * The framework for a server. It dynamically creates a
 * TransportServer (depending on the protocol specified in a URL)
 * that listens for incoming connections. A UlcConnection is then
 * created for each accepted connection.
 * The Server framework is independent of the request format and
 * the transport kind.
 *
 * @see UlcConnection
 * @see UlcTransportServer
 * @see Request
 */

public class Server extends UlcObject implements Runnable {
	protected int fPort = 0;
	private UlcTransportServer fTransportServer = null;
	private static Vector fgConnectionControllers = null;
	protected String fName = null;
	protected static ISocketFactory fgSocketFactory = null;
	final protected static boolean DEBUG = false;
	// name of file into which the URL of the server should be placed
	private static String fgServerUrlFileName = null;
	private static Object fPropagateLock = new Object();
/**
 * Constructs a server.
 */
public Server() {
	fName = "Listener";
}
/**
 * Constructs a server.
 */
public Server(String name) {
	fName = name;
}
/**
 * Method to register a connection controller object.
 */
public static synchronized void addConnectionController(IConnectionController controller) {
	if (fgConnectionControllers == null)
		fgConnectionControllers = new Vector();
	//we check before adding because ULC Java can call this method twice in some situations
	if (!fgConnectionControllers.contains(controller))
		fgConnectionControllers.addElement(controller);
}
/**
 * Factory method for creating a connection from a given transport.
 * Override to define the connection to be created.
 * @param transport com.ibm.ulc.comm.UlcTransport
 */
public void createConnection(UlcTransport transport) {
	trouble("createConnection(UlcTransport)", "abstract method called");
}
public String getName() {
	return fName;
}
public static String getServerUrlFileName() {
	return fgServerUrlFileName;
}
/**
 * Retrieves the factory object for creating all sockets within ULC.
 * If a factory hasn't been explicitely set this method creates an instance of
 * com.ibm.ulc.comm.DefaultSocketFactory.
 */
public static ISocketFactory getSocketFactory() {
	if (fgSocketFactory == null)
		fgSocketFactory = new DefaultSocketFactory();
	return fgSocketFactory;
}
/**
 * return the string to indicate that this Server has started.
 */
public String getStartedString() {
	return getName() + " started; accepting connections on port ";
}
/**
 * Returns the Transport server or null
 */
protected UlcTransportServer getTransportServer() {
	return fTransportServer;
}
/**
 * Helper method for connection control.
 * Calls all registered connection controllers' connectionCreated()
 * method with the given connection argument.
 */
public static void propagateConnectionCreated(IConnection connection) {
	// notify controllers
	synchronized (fPropagateLock) {
		if (fgConnectionControllers != null) {
			Vector tmp = (Vector) fgConnectionControllers.clone();
			for (Enumeration e = tmp.elements(); e.hasMoreElements();) {
				IConnectionController controller = (IConnectionController) e.nextElement();
				controller.connectionCreated(connection);
			}
		}
	}
}
/**
 * Helper method for connection control.
 * Calls all registered connection controllers' connectionEnded()
 * method with the given connection and exception arguments.
 */
public static void propagateConnectionEnded(IConnection connection, UlcTransportException ex) {
	// notify controllers
	synchronized (fPropagateLock) {
		if (fgConnectionControllers != null) {
			Vector tmp = (Vector) fgConnectionControllers.clone();
			for (Enumeration e = tmp.elements(); e.hasMoreElements();) {
				IConnectionController controller = (IConnectionController) e.nextElement();
				controller.connectionEnded(connection, ex);
			}
		}
	}
}
/**
 * Helper method for connection control.
 * Calls all registered connection controllers' connectionStarted()
 * method with the given connection and exception arguments.
 */
public static void propagateConnectionStarted(IConnection connection, UlcTransportException ex) {
	// notify controllers
	synchronized (fPropagateLock) {
		if (fgConnectionControllers != null) {
			Vector tmp = (Vector) fgConnectionControllers.clone();
			for (Enumeration e = tmp.elements(); e.hasMoreElements();) {
				IConnectionController controller = (IConnectionController) e.nextElement();
				controller.connectionStarted(connection, ex);
			}
		}
	}
}
/**
 * Method to deregister a connection controller object.
 */
public static synchronized void removeConnectionController(IConnectionController controller) {
	if (fgConnectionControllers != null)
		fgConnectionControllers.removeElement(controller);
}
/**
 * Runs the server and starts accepting connections.
 * @deprecated
 */
public void run() {
	System.out.println("Server.run : Should not be called. use start(aUrlString) instead");
}
/**
 */
protected static void setServerUrlFileName(String name) {
	fgServerUrlFileName= name;
}
/**
 * Sets the factory object for creating all sockets within ULC.
 * Call this method before calling method <code>Server.start</code>.
 * If the factory is not explicitely set ULC will use an instance of
 * com.ibm.ulc.comm.DefaultSocketFactory.
 */
public static void setSocketFactory(ISocketFactory factory) {
	fgSocketFactory = factory;
}
/**
 * Starts the server by creating a new thread for accepting
 * connections on the address specified by the passed in URL.
 * @param urlString java.lang.String
 */
public int start(String urlString) {
	// create transport server
	fTransportServer = null;
	try {
		fTransportServer = UlcTransportServer.create(urlString, this);
		if (fTransportServer == null) {
			System.out.println("could not create transport server for url: " + urlString);
			return 1;
		}
	} catch (UlcTransportException ex) {
		System.out.println("Exception while creating transport server for url: " + urlString + ". ex=" + ex);
		return 1;
	}
	// start transport server
	try {
		fTransportServer.start();
	} catch (UlcTransportException ex) {
		System.out.println("Exception while starting transport server for url: " + urlString + ". ex=" + ex);
		return 1;
	}
	return 0;
}
}
