package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.Common;

/**
 * Represents the ULC menu.
 * A popup window containing ULCMenuItems that is displayed when the user selects an item on the ULCMenuBar.
 * In addition to ULCMenuItems, a ULCMenu can also contain ULCSeparators.
 *
 * @see ULCMenuItem
 * @see ULCSeparator
 */
public class ULCMenu extends ULCComponent implements IModelOwner {
	/**
	 * The <code>int</code> value of the character to be used as a mnemonic.
	 * @serial	 	 
	 */
	protected int fMnemonic = -1;
	protected IForm fFormModel = null;
	protected String fFormAttributeName = null;
	protected Object fFormAttributeValue = null;
public ULCMenu() {
}
/**
 * Creates a new ULCMenu with the supplied string as its title.
 *
 * @param text 	The <code>String</code> label to be displayed.
 */
public ULCMenu(String text) {
	super(text);
}
/**
 * Creates a new ULCMenu with the supplied string as its title and the given mnemonic keycode.
 *
 * @param text 	The <code>String</code> label to be displayed.
 * @param keyCode	The <code>int</code> character value of the key to be used as a mnemonic.
 */
public ULCMenu(String text, int mnemonic) {
	super(text);
	fMnemonic = mnemonic;
}
/**
 * Appends another menu as a submenu to the end of this menu.
 *
 * @param menu 	The <code>ULCMenu</code> to be added as a submenu.
 */
public void add(ULCMenu menu) {
	internalAdd(menu);
}
/**
 * Appends a menuitem to the end of this menu.
 *
 * @param menuItem 	The <code>ULCMenuItem</code> to be added to myself.
 */
public void add(ULCMenuItem menuItem) {
	internalAdd(menuItem);
}
/**
 * Creates a new ULCMenuItem with the given name and listener and appends it to the end of this menu.
 *
 * @param label The label of the <code>ULCMenuItem</code> to be added.
 * @param l		The listener who will receive the <code>ULCMenuItem</code> actionEvents.
 */
public void add(String label, IActionListener l) {
	internalAdd(new ULCMenuItem(label, l));
}
/**
 * Appends a collection of <code>ULCMenuItems</code> to the end of this menu.
 *
 * @param menuItems The collection of <code>ULCMenuItem</code> objects to be added.
 */
public void addMenuItems(Vector menuItems) {
	internalAdd(menuItems);
}
/**
 * Appends a collection of <code>ULCMenus</code> to the end of this menu.
 *
 * @param menus	The collection of <code>ULCMenu</code> objects to be added.
 */
public void addMenus(Vector menus) {
	internalAdd(menus);
}
/**
 * Add all attributes that should be preloaded with a form model to the UI to the specified Vector.
 *
 * @return Vector
 */
public void addPreloadFormAttributesInto(Vector vectorOfPreloadAttributes) {
}
/**
 * Add all table attributes that should be preloaded to the UI to the specified Vector.
 * This receiver has no table attributes to preload.
 *
 * @param vectorOfPreloadAttributes	Vector	into which the receiver adds the
 *											table attributes it needs to preload.
 */
public void addPreloadTableAttributesInto(Vector vectorOfPreloadAttributes) {
}
/**
 * Creates a new ULCSeparator and appends it to the end of this menu.
 */
public void addSeparator() {
	addSeparator(new ULCSeparator());
}
/**
 * Adds the ULCSeparator it to the end of this menu.
 */
public void addSeparator(ULCSeparator separator) {
	internalAdd(separator);
}
/**
 * Get the receiver's optional <code>formAttributeName</code>.
 * If the formModel, formAttributeValue and formAttributeName are specified then this
 * menu will be shown only if the value in the formModel at formAttributeName matches 
 * the value of formAttributeValue.
 */
public String getFormAttributeName() {
	return fFormAttributeName;
}
/**
 * Get the receiver's optional <code>formAttributeValue</code>.
 * If the formModel, formAttributeValue and formAttributeName are specified then this
 * menu will be shown only if the value in the formModel at formAttributeName matches 
 * the value of formAttributeValue.
 */
public Object getFormAttributeValue() {
	return fFormAttributeValue;
}
/**
 * Get the receiver's optional FormModel.
 * If the formModel, formAttributeValue and formAttributeName are specified then this
 * menu will be shown only if the value in the formModel at formAttributeName matches 
 * the value of formAttributeValue.
 */
public IForm getFormModel() {
	return fFormModel;
}
/**
 * Gets the menu's label.
 *
 * @return The <code>String</code> value of my label.
 */
public String getLabel() {
	return fLabel;
}
/**
 * Gets the collection of <code>ULCMenuItem's</code> in this menu.
 *
 * @return The <code>Vector</code> containing <code>ULCMenuItem's</code>.
 */
public Vector getMenuItems() {
	Vector components = new Vector();
	Vector v = internalGetComponents();
	for (int i = 0; i < v.size(); i++) {
		Object next = v.elementAt(i);
		if (next instanceof ULCMenuItem)
			components.addElement((ULCMenuItem) next);
	}
	return components;
}
/**
 * Gets the collection of <code>ULCMenu's</code> in this menu.
 *
 * @return The <code>Vector</code> containing <code>ULCMenu's</code>.
 */
public Vector getMenus() {
	Vector components = new Vector();
	Vector v = internalGetComponents();
	for (int i = 0; i < v.size(); i++) {
		Object next = v.elementAt(i);
		if (next instanceof ULCMenu)
			components.addElement((ULCMenu) next);
	}
	return components;
}
/**
 * Gets this widgets mnemonic code.
 *
 * @return keyCode The keyCode that represents the character to be used as a mnemonic
 */
public int getMnemonic() {
	return fMnemonic;
}
/**
 * Gets the collection of <code>ULCSeparators</code> in this toolBar.
 *
 * @return The <code>Vector</code> containing <code>ULCSeparators</code>.
 */
public Vector getSeparators() {
	Vector components = new Vector();
	Vector v = internalGetComponents();
	for (int i = 0; i < v.size(); i++) {
		Object next = v.elementAt(i);
		if (next instanceof ULCSeparator)
			components.addElement(next);
	}
	return components;
}
/**
 * Inserts a menu at the specified index.
 *
 * @param menu 	The <code>ULCMenu</code> to be added as a submenu.
 * @param index	The <code>int</code> index at which this submenu should be added.
 */
public void insert(ULCMenu menu, int index) {
	internalInsert(menu, index);
}
/**
 * Inserts a menuitem at the specified index.
 *
 * @param menuitem 	The <code>ULCMenuItem</code> to be inserted.
 * @param index		The <code>int</code> index at which this submenu should be added.
 */
public void insert(ULCMenuItem menuItem, int index) {
	internalInsert(menuItem, index);
}
/**
 * Inserts a menuitem at the specified index.
 *
 * @param separator 	The <code>ULCSeparator</code> to be inserted.
 * @param index		The <code>int</code> index at which this ULCSeparator should be added.
 */
public void insert(ULCSeparator separator, int index) {
	internalInsert(separator, index);
}
/**
 * Creates a new ULCSeparator and inserts it at the specified index of this menu.
 */
public void insertSeparator(int index) {
	internalInsert(new ULCSeparator(), index);
}
protected void internalUpdatePreloadAttributes() {
	if ((getFormModel() != null) && (getFormAttributeName() != null)) {
		getFormModel().addPreloadAttribute(getFormAttributeName());
	}
}
/**
 * Remove the specified <code>ULCMenu</code> from my list of menus.
 *
 * @param menu  The <code>ULCMenu</code> to be removed.
 */
public void remove(ULCMenu menu) {
	internalRemove(menu);
}
/**
 * Remove the specified <code>ULCMenuItem</code> from my list of menu items.
 *
 * @param menuItem  The <code>ULCMenuItem</code> to be removed.
 */
public void remove(ULCMenuItem menuItem) {
	internalRemove(menuItem);
}
/**
 * Remove the specified <code>ULCSeparator</code> from my list of items.
 *
 * @param separator  The <code>ULCSeparator</code> to be removed.
 */
public void remove(ULCSeparator separator) {
	internalRemove(separator);
}
/**
 * Remove all the specified <code>ULCMenuItem's</code> from my list of menu items.
 *
 * @param menuItems  The <code>Vector</code> of <code>ULCMenuItem's</code> to be removed.
 */
public void removeMenuItems(Vector menuItems) {
	internalRemove(menuItems);
}
/**
 * Remove all the specified <code>ULCMenu's</code> from my list of menus.
 *
 * @param menus  The <code>Vector</code> of <code>ULCMenu's</code> to be removed.
 */
public void removeMenus(Vector menus) {
	internalRemove(menus);
}
/**
 * Remove all the specified <code>ULCSeparators's</code> from my list of items.
 *
 * @param separators  The <code>Vector</code> of <code>ULCSeparators's</code> to be removed.
 */
public void removeSeparators(Vector separators) {
	internalRemove(separators);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fMnemonic != -1)
		a.put("mnemonic", fMnemonic);
	if (fFormModel != null) {
		a.put("formModel", fFormModel.getRef(fContext));
		a.put("key", fFormAttributeName);
		a.put("value", Common.convertToAnything(fFormAttributeValue));
	}
}
/**
 * Set the receiver's formAttributeName.
 * If the formModel, formAttributeValue and formAttributeName are specified then this
 * menu will be shown only if the value in the formModel at formAttributeName matches 
 * the value of formAttributeValue.
 * Update the UI.
 *
 * @param formAttributeName the <code>String</code> attributeName
 */
public void setFormAttributeName(String formAttributeName) {
	fFormAttributeName = formAttributeName;
	internalUpdatePreloadAttributes();
	sendUI("setFormAttributeName", new Anything(fFormAttributeName));
}
/**
 * Set the receiver's formAttributeValue.
 * If the formModel, formAttributeValue and formAttributeName are specified then this
 * menu will be shown only if the value in the formModel at formAttributeName matches 
 * the value of formAttributeValue.
 * Update the UI.
 *
 * @param formAttributeValue the <code>Object</code> attributeValue
 */
public void setFormAttributeValue(Object formAttributeValue) {
	fFormAttributeValue = formAttributeValue;
	sendUI("setFormAttributeValue", Common.convertToAnything(fFormAttributeValue));
}
/**
 * Set the receiver's  FormModel. 
 * If the formModel, formAttributeValue and formAttributeName are specified then this
 * menu will be shown only if the value in the formModel at formAttributeName matches 
 * the value of formAttributeValue.
 * Update the UI.
 *
 * @param formModel the <code>IForm</code> of the receiver.
 */
public void setFormModel(IForm formModel) {
	if (fFormModel != null)
		fFormModel.removeOwner(this);
	fFormModel = formModel;
	if (fFormModel != null) {
		fFormModel.addOwner(this);
		internalUpdatePreloadAttributes();
	}
	sendUI("setFormModel", fFormModel);
}
/**
 * Sets the widgets's <code>String</code> label.
 * Note: If the label contains the <code>IDefaults.MNEMONIC_INDICATOR</code> then
 * the character after this indicator is treated as the mnemonic character for this widget.
 *
 * @param label the label <code>String</code> of the widget.
 */
public void setLabel(String label) {
	internalSetLabel(label);
}
/**
 * Sets the collection of <code>ULCMenuItem's</code> in this menu.
 *
 * @param The <code>Vector</code> containing <code>ULCMenuItem's</code>.
 */
public void setMenuItems(Vector items) {
	removeMenuItems(getMenuItems());
	addMenuItems(items);
}
/**
 * Sets this widgets mnemonic code.
 *
 * @param keyCode The keyCode that represents the character to be used as a mnemonic
 */
public void setMnemonic(int keyCode) {
	if (keyCode != fMnemonic) {
		fMnemonic = keyCode;
		sendUI("setMnemonic", new Anything(fMnemonic));
	}
}
/**
 * Do nothing since Menu does not support a popupMenu
 *
 * @param 	popupMenu 
 *			The <code>ULCMenu</code> to be displayed when a popupRequest is received
 */
public void setPopupMenu(ULCMenu popupMenu) {
}
}
