package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.awt.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;


/**
 * ULCComponent is the base class for all widgets.
 * It provides support for enabling/disabling, a label, popupMenus and
 * help texts (tooltips).
 * Setters for these fields are protected and their names have
 * the prefix "internal". Subclasses have to explicitly rename these
 * methods and change the visibility to public if they want to make the
 * setters publicly accessible.
 */
public class ULCComponent extends ULCProxy {
	/**
	 * The label string of this widget. The interpretation of the label
	 * string is left to each individual widget.
	 * @serial
	 */
	protected String fLabel = null;
	/**
	 * The popupMenu for this widget if specified that is a <code>ULCMenu</code>.
	 * @serial	 
	 */
	protected ULCMenu fPopupMenu = null;
	/**
	 * The Text that will be displayed if the mouse lingers over this component.
	 * @serial	 
	 */
	protected String fToolTipText = null;
	/**
	 * The enabled state of this widget.
	 * @serial	 
	 */
	protected boolean fEnabled = true;
	/**
	 * The cursor to be displayed when the mouse is over this widget.
	 * @serial	 
	 */
	protected int fCursor = ULCCursor.DEFAULT_CURSOR;
	/**
	 * The foreground color for this widget.
	 * @serial	 
	 */
	protected Color fForeColor;
	/**
	 * The background color for this widget.
	 * @serial	 
	 */
	protected Color fBackColor;
	/**
	 * The optional font to be used in this widget.
	 * @serial	 
	 */
	protected ULCFont fFont = null;
	/**
	 * The widget that if specified will cause this widget to be enabled in the UI.
	 * @serial	 
	 */
	protected IEnabler fEnabler = null;
	/**
	 * If true this widget will be made visible when created.
	 * @serial	 
	 */
	protected boolean fVisible = true;
	/**
	 * The partial URL for the help page for this widget
	 * This Url is appended to the getHelpRoot URL of the ULCContext and then displayed.
	 */
	protected String fHelpUrl = null;
/**
 * Default Constructor for all ULCComponents and their subclasses
 *
 */
public ULCComponent() {
}
/**
 * Construct a ULCComponent with the specified label
 *
 * @param label The <code> String</code> label
 */
public ULCComponent(String label) {
	fLabel = label;
}
/**
 * This widget is being destroyed perform any cleanUp necessary.
 */
public void free() {
	super.free();
	fEnabler = null;
	fPopupMenu = null;
	fForeColor = null;
	fBackColor = null;
	fFont = null;
	fHelpUrl = null;
}
/**
 * Gets the background color for this widget or null if not specified.
 *
 * @return The <code>Color</code> that is used within this widget
 */
public Color getBackground() {
	return fBackColor;
}
/**
 * Gets the current cursor to be displayed when the mouse pointer is over this widget 
 * to be cursorType.
 * The new cursor will remain till the next setCursor call.
 *
 * @return  The integer cursorType.
 *  cursorType can have the values:
 * 		UlcCursor.DEFAULT_CURSOR
 * 		UlcCursor.CROSSHAIR_CURSOR
 * 		UlcCursor.TEXT_CURSOR
 * 		UlcCursor.WAIT_CURSOR
 * @see ULCCursor
 */
public int getCursor() {
	return fCursor;
}
/**
 * Returns Makes the enabled state of this widget dependent on the specified enabler widget.
 * If called more than once an assertion will be violated.
 *
 * @param c The component that will enable/disable this object.
 */
public IEnabler getEnabler() {
	return fEnabler;
}
/**
 * Gets the font used for this widget.
 *
 * @return The <code>UlcFont</code> that is used within this widget
 */
public ULCFont getFont() {
	return fFont;
}
/**
 * Gets the foreground color for this widget or null if not specified.
 *
 * @return  The <code>Color</code> that is used within this widget
 */
public Color getForeground() {
	return fForeColor;
}
/**
 * Return the help url for this component or null if not defined.
 * This Url is combined with the ULCHelpBrowserContext helpUrl to form
 * the complete URL.
 *
 * @see ULCBrowserContext
 * @see ULCApplication#setHelpContext
 */
public String getHelpUrl() {
	return fHelpUrl;
}
/**
 * Gets the popupMenu for this widget.
 *
 * @return 	popupMenu 
 *			The <code>ULCMenu</code> to be displayed when a popupRequest is received
 */
public ULCMenu getPopupMenu() {
	return fPopupMenu;
}
/**
 * returns this objects tooltip text.
 *
 * @return  The text to be displayed when the mouse lingers over this object.
 */
public String getToolTipText() {
	return fToolTipText;
}
/**
 * Set the current cursor to be displayed when the mouse pointer is over this widget 
 * to be cursorType.
 * The new cursor will remain till the next internalSetCursor call.
 *
 * @param type The integer cursorType.
 *  cursorType can have the values:
 * 		UlcCursor.DEFAULT_CURSOR
 * 		UlcCursor.CROSSHAIR_CURSOR
 * 		UlcCursor.TEXT_CURSOR
 * 		UlcCursor.WAIT_CURSOR
 * @see ULCCursor
 */
protected void internalSetCursor(int type) {
	if (fCursor != type) {
		fCursor = type;
		sendUI("setCursor", new Anything(fCursor));
	}
}
/**
 * Sets this widgets label.
 *
 * @param label the label <code> String</code> of this widget.
 */
protected void internalSetLabel(String label) {
	if (fLabel == null || !fLabel.equals(label)) {
		fLabel = label;
		sendUI("setLabel", new Anything(fLabel));
	}
}
/**
 * returns true if this widget is Enabled.
 * This value only reflects the state when the widget was uploaded.
 * If the state is changed after the upload using a enabler directly in
 * the UI then this value will not return the correct state.
 *
 */
public boolean isEnabled() {
	return fEnabled;
}
/**
 * Return true if this widget is visible in the UI.
 */
public boolean isVisible() {
	return fVisible;
}
/**
 * Send a request to the UI to request that this widget receives the focus.
 * If the widget has not been uploaded as yet write an error to the console and
 * do nothing.
 */
public void requestFocus() {
	if (isUploaded()) {
		sendUI("requestFocus");
	}
	else {
		System.out.println("Component.requestFocus requests are ignored until component is visible");
	}
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fLabel != null)
		a.put("label", fLabel);
	if (!fEnabled)
		a.put("enabled", fEnabled);
	if (fToolTipText != null)
		a.put("tooltip", fToolTipText);
	if (fCursor != ULCCursor.DEFAULT_CURSOR)
		a.put("cursor", fCursor);
	if (fPopupMenu != null)
		a.put("popupMenu", fPopupMenu.getRef(fContext));
	if (fForeColor != null)
		a.put("fc", internalConvertColor(fForeColor));
	if (fBackColor != null)
		a.put("bc", internalConvertColor(fBackColor));
	if (fFont != null)
		a.put("font", fFont.getRef(fContext));
	if (fEnabler != null)
		a.put("enabler", fEnabler.getRef(fContext));
	if (!fVisible)
		a.put("visible", fVisible);
	if (fHelpUrl != null)
		a.put("helpUrl", fHelpUrl);
}
/**
 * Sets the background color for this widget.
 *
 * @param color The <code>Color</code> that should be used within this widget
 */
public void setBackground(Color color) {
	fBackColor = color;
	if (color == null)
		sendUI("setBackground", new Anything());
	else
		sendUI("setBackground", internalConvertColor(fBackColor));
}
/**
 * Set the current cursor to be displayed when the mouse pointer is over this widget 
 * to be cursorType.
 * The new cursor will remain till the next setCursor call.
 *
 * @param type The integer cursorType.
 *  cursorType can have the values:
 * 		UlcCursor.DEFAULT_CURSOR
 * 		UlcCursor.CROSSHAIR_CURSOR
 * 		UlcCursor.TEXT_CURSOR
 * 		UlcCursor.WAIT_CURSOR
 * @see ULCCursor
 */
public void setCursor(int cursorType) {
	internalSetCursor(cursorType);
}
/**
 * Enables or disables this widget.
 *
 * @param enabled if true enable, otherwise disable this widget in the UI.
 */
public void setEnabled(boolean enabled) {
	/* Don't do the test (state of the UI could have
	   changed due to enablers (PR 1FQDG30)	*/
	fEnabled = enabled;
	sendUI("setEnabled", new Anything(fEnabled));
}
/**
 * Makes the enabled state of this widget dependent on the specified enabler widget.
 *
 * @param c The component that will enable/disable this object.
 */
public void setEnabler(IEnabler c) {
	fEnabler = c;
	if (fEnabler != null)
		sendUI("setEnabler", fEnabler);
	else
		sendUI("setEnabler", new Anything());
}
/**
 * Sets the font used for this widget.
 *
 * @param font The <code>UlcFont</code> that should be used within this widget
 */
public void setFont(ULCFont font) {
	fFont = font;
	if (font == null)
		sendUI("setFont", new Anything());
	else
		sendUI("setFont", font);
}
/**
 * Sets the foreground color for this widget.
 *
 * @param color The <code>Color</code> that should be used within this widget
 */
public void setForeground(Color color) {
	fForeColor = color;
	if (color == null)
		sendUI("setForeground", new Anything());
	else
		sendUI("setForeground", internalConvertColor(fForeColor));
}
/**
 * Set the partial help url for this component.
 * This Url is combined with the ULCHelpBrowserContext helpUrl to form
 * the complete URL.
 *
 * @see ULCBrowserContext
 * @see ULCApplication#setHelpContext
 */

public void setHelpUrl(String helpUrl) {
	fHelpUrl = helpUrl;
}
/**
 * Sets the popupMenu for this widget.
 *
 * @param 	popupMenu 
 *			The <code>ULCMenu</code> to be displayed when a popupRequest is received
 */
public void setPopupMenu(ULCMenu popupMenu) {
	fPopupMenu = popupMenu;
	if (popupMenu == null)
		sendUI("setPopupMenu", new Anything());
	else
		sendUI("setPopupMenu", fPopupMenu);
}
/**
 * Sets this widgets tooltip text.
 *
 * @param text The text to be displayed when the mouse lingers over this widget.
 */
public void setToolTipText(String text) {
	if (fToolTipText == null || !fToolTipText.equals(text)) {
		fToolTipText = text;
		Anything tt;
		if (text == null)
			tt = new Anything();
		else
			tt = new Anything(fToolTipText);
		sendUI("setToolTipText", tt);
	}
}
/**
 * Hides or shows this widget.
 *
 * @param visible if true show, otherwise hide this widget in the UI.
 */
public void setVisible(boolean visible) {
	if (visible != fVisible) {
		fVisible = visible;
		sendUI("setVisible", new Anything(fVisible));
	}
}
}
