package com.ibm.ulc.application;


/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import java.util.Vector;
import java.util.Enumeration;
import java.awt.Point;

/**
 * The box object provides the basic mechanism for 'containing'
 * widgets in order to control their layout. It therefore supports
 * horizontal as well as vertical alignment aspects
 * of it's contained widgets. 
 * <p>
 * <pre>
 * Format for alignment string:
 * 	'horizontal specifier''vertical specifier'
 *		'horizontal specifier':
 * 			e expand
 * 		 	l left
 * 		 	c center
 * 		 	r right
 *		'vertical specifier':
 * 		 	e expand
 * 			t top
 * 		 	c center
 * 			b bottom
 * </pre>
 */
public class ULCBox extends ULCComponent {
	/**
	 * Specify the horizontal and vertical gaps between adjacent cells within the box
 	 * @serial
	 */
	protected int fHGap = 3, fVGap = 3;
	/**
	 * Specify the margin between the box and its contained cells.
 	 * @serial	 
	 */
	protected int fMargin = 0;
	/**
	 * Set true if the contents of this box should be treated as a radio group.
	 * If this is set to true all the widgets within this box should be of type ULCToggleButton.
 	 * @serial	 
	 */
	protected boolean fRadioGroup = false;
	/**
	 * Specify the number of rows and columns within this box.
 	 * @serial	 
	 */
	protected int fRows = 0, fCols = 0;
	/**
	 * The index of the next component to be added within this box.
 	 * @serial	 
	 */
	protected int fNextIndex = 0;
	/**
	 * Specify the value to be used in <code>fRadioGroup</code> to enable radio group behavior.
 	 * @serial	 
	 */
	public static final boolean RADIO_GROUP = true;
/**
 * Construct a vertical or horizontal box.
 *
 */
public ULCBox() {
	super();
}
/**
 * Construct a box with the given number of rows and columns.
 *
 * @param rows The integer number of rows within this box.
 * @param cols The integer number of columns within this box.
 */
public ULCBox(int rows, int columns) {
	fRows = rows;
	fCols = columns;
}
/**
 * Construct a box with the given number of rows and columns and the specified gaps.
 *
 * @param rows The integer number of rows within this box.
 * @param cols The integer number of columns within this box.
 * @param hgap The integer gap to be used between each column.
 * @param vgap The integer gap to be used between each row.
 */
public ULCBox(int rows, int columns, int horizontalGap, int verticalGap) {
	fRows = rows;
	fCols = columns;
	fHGap = horizontalGap;
	fVGap = verticalGap;
}
/**
 * Construct a box with the given rows and columns.
 * If <code>radioGroup</code> is true, all subsequently added ULCRadioButtons
 * will have coordinated radio button behavior.
 *
 * @param rows The integer number of rows within this box.
 * @param cols The integer number of columns within this box.
 * @param radioGroup If radioGroup is true then enable radioGroup behavior.
 */
public ULCBox(int rows, int columns, boolean radioGroup) {
	this(rows, columns);
	fRadioGroup = radioGroup;
}
/**
 * Construct a vertical or horizontal box.
 *
 * @param	vertical	
 *			The boolean value which if true indicates that this is a vertical box.
 */
public ULCBox(boolean vertical) {
	this(vertical, 3);
}
/**
 * Construct a vertical or horizontal box and use the specified gap
 * for either the horizontal or vertical gap.
 *
 * @param	vertical	
 *			The boolean value which if true indicates that this is a vertical box.
 * @param	gap
 *			The integer gap between rows or columns.
 */
public ULCBox(boolean vertical, int gap) {
	if (vertical) {
		fRows = 0;
		fCols = 1;
		fVGap = gap;
	}
	else {
		fRows = 1;
		fCols = 0;
		fHGap = gap;
	}
}
/**
 * Construct a vertical or horizontal box.
 * If <code>radioGroup</code> is true, all subsequently added ULCRadioButtons
 * will have coordinated radio button behavior.
 * Note: if a Button is a member of a ButtonGroup you can't reset its state to false!
 * You have to set another Button's state to true.
 *
 * @param	vertical	
 *			The boolean value which if true indicates that this is a vertical box.
 * @param	radioGroup
 *			The boolean value which if true enables radioGroup behavior.
 */
public ULCBox(boolean vertical, boolean radioGroup) {
	this(vertical);
	fRadioGroup = radioGroup;
}
/**
 * Add a ULCComponent at a specific cell, with an hSpan and vSpan
 *
 * @param x		The x location of the cell
 * @param y		The y location of the cell
 * @param hSpan	The horizontal span of the component
 * @param vSpan	The vertical span of the component
 * @param c		The component to be added.
 */
public void add(int x, int y, int hSpan, int vSpan, ULCComponent c) {
	internalAdd(new BoxCellAttributes(c, x, y, hSpan, vSpan));
}
/**
 * Add a ULCComponent at a specific cell.
 *
 * @param x		The x location of the cell
 * @param y		The y location of the cell
 * @param hSpan	The horizontal span of the component
 * @param vSpan	The vertical span of the component
 * @param align The alignment string 
 * <p>
 * <pre>
 * Format for alignment string:
 * 	'horizontal specifier''vertical specifier'
 *		'horizontal specifier':
 * 			e expand
 * 		 	l left
 * 		 	c center
 * 		 	r right
 *		'vertical specifier':
 * 		 	e expand
 * 			t top
 * 		 	c center
 * 			b bottom
 * </pre>
 * @param c		The component to be added.
 */
public void add(int x, int y, int hSpan, int ySpan, String align, ULCComponent c) {
	internalAdd(new BoxCellAttributes(c, x, y, hSpan, ySpan, align));
}
/**
 * Add a ULCComponent at a specific cell, with an hSpan
 *
 * @param x		The x location of the cell
 * @param y		The y location of the cell
 * @param hSpan	The horizontal span of the component
 * @param c		The component to be added.
 */
public void add(int x, int y, int hSpan, ULCComponent c) {
	internalAdd(new BoxCellAttributes(c, x, y, hSpan));
}
/**
 * Add a ULCComponent at a specific cell.
 *
 * @param x		The x location of the cell
 * @param y		The y location of the cell
 * @param c		The component to be added.
 */
public void add(int x, int y, ULCComponent c) {
	internalAdd(new BoxCellAttributes(c, x, y));
}
/**
 * Add a ULCComponent at a specific cell.
 *
 * @param x		The x location of the cell
 * @param y		The y location of the cell
 * @param align The alignment string 
 * <p>
 * <pre>
 * Format for alignment string:
 * 	'horizontal specifier''vertical specifier'
 *		'horizontal specifier':
 * 			e expand
 * 		 	l left
 * 		 	c center
 * 		 	r right
 *		'vertical specifier':
 * 		 	e expand
 * 			t top
 * 		 	c center
 * 			b bottom
 * </pre>
 * @param hSpan	The horizontal span of the component
 * @param c		The component to be added.
 */
public void add(int x, int y, String align, int hSpan, ULCComponent c) {
	internalAdd(new BoxCellAttributes(c, x, y, align, hSpan));
}
/**
 * Add a ULCComponent at a specific cell.
 *
 * @param x		The x location of the cell
 * @param y		The y location of the cell
 * @param align The alignment string 
 * <p>
 * <pre>
 * Format for alignment string:
 * 	'horizontal specifier''vertical specifier'
 *		'horizontal specifier':
 * 			e expand
 * 		 	l left
 * 		 	c center
 * 		 	r right
 *		'vertical specifier':
 * 		 	e expand
 * 			t top
 * 		 	c center
 * 			b bottom
 * </pre>
 * @param c		The component to be added.
 */
public void add(int x, int y, String align, ULCComponent c) {
	internalAdd(new BoxCellAttributes(c, x, y, align));
}
/**
 * Add a ULCComponent at the next possible cell.
 * Cells are filled from left to right and top to bottom at the next available index.
 *
 * @param c The component to be added.
 */
public void add(ULCComponent c) {
	Point nextCell = getNextCell();
	internalAdd(new BoxCellAttributes(c, nextCell.x, nextCell.y));
}
/**
 * Add a ULCComponent at the next possible cell
 *
 * @param align The alignment string 
 * <p>
 * <pre>
 * Format for alignment string:
 * 	'horizontal specifier''vertical specifier'
 *		'horizontal specifier':
 * 			e expand
 * 		 	l left
 * 		 	c center
 * 		 	r right
 *		'vertical specifier':
 * 		 	e expand
 * 			t top
 * 		 	c center
 * 			b bottom
 * </pre>
 * @param c 	The component to be added.
 */
public void add(String align, ULCComponent c) {
	Point nextCell = getNextCell();
	internalAdd(new BoxCellAttributes(c, nextCell.x, nextCell.y, align));
}
/**
 * Columns. 
 *
 */
public int getColumns() {
	return fCols;
}
/**
 * Horizontal Gap. 
 *
 */
public int getHorizontalGap() {
	return fHGap;
}
/**
 * Margin. 
 *
 */
public int getMargin() {
	return fMargin;
}
/**
 * Get the next free cell to add into.
 * Cells are filled from left to right and top to bottom at the next available index.
 * Note calling this method increments the current index if the returned value is not used
 * the box will not layout correctly.
 *
 * @return Point - The next cell to add into.
 */
public Point getNextCell() {
	int x, y;
	if (fCols == 0) {
		x = fNextIndex;
		y = 0;
	}
	else {
		x = fNextIndex % fCols;
		y = fNextIndex / fCols;
	}
	fNextIndex++;
	return new Point(x, y);
}
/**
 * Rows. 
 *
 */
public int getRows() {
	return fRows;
}
/**
 * Vertical Gap. 
 *
 */
public int getVerticalGap() {
	return fVGap;
}
/**
 * Returns true if this box is set to behave like a RadioGroup.
 * 	Note: if a Button is a member of a RadioGroup you can't reset its state to false!
 * You have to set another Button's state to true.
 *
 * @return	set rg to true to enable radioGroup behavior
 */
public boolean isRadioGroup() {
	return fRadioGroup;

}
/**
 * Remove a ULCComponent if it already exists as a child.
 *
 * @param c The component to be removed.
 */
public void remove(ULCComponent c) {
	if (fList != null)
		for (int i = 0; i < fList.size(); i++) {
			BoxCellAttributes bc = (BoxCellAttributes) fList.elementAt(i);
			if (c == bc.getComponent()) {
				fList.removeElement(bc);
				internalRemove(c);
				fNextIndex--;
				return;
			}
		}
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	a.put("type", "Box");
	super.saveState(a);
	if (fHGap != 3)
		a.put("hg", fHGap);
	if (fVGap != 2)
		a.put("vg", fVGap);
	if (fMargin != 0)
		a.put("m", fMargin);
	if (fRadioGroup)
		a.put("group", fRadioGroup);
}
/**
 * Add the component c at grid position x and y.
 *
 * @param x	The integer column position
 * @param y	The integer row position
 * @param c	The component to be added
 */
public void set(int x, int y, ULCComponent c) {
	internalAdd(new BoxCellAttributes(c, x, y));
}
/**
 * Set the columns to be columns. This should be private
 * because it needs to be set before uploading. But
 * beaninfo won't let us have a private setter.
 *
 * @param columns	The integer columns 
 */
public void setColumns(int columns) {
	fCols = columns;
}
/**
 * Set the horizontal gap between columns to be hgap.
 *
 * @param horizontalGap	The integer gap 
 */
public void setHorizontalGap(int horizontalGap) {
	if (horizontalGap != fHGap) {
		fHGap = horizontalGap;
		sendUI("setHGap", new Anything(fHGap));
	}
}
/**
 * Set the margin between the box and its enclosed components.
 *
 * @param margin	The integer margin 
 */
public void setMargin(int margin) {
	if (margin != fMargin) {
		fMargin = margin;
		sendUI("setMargin", new Anything(fMargin));
	}
}
/**
 * Sets the RadioGroup behavior of a Box.
 * 	Note: if a Button is a member of a ButtonGroup you can't reset its state to false!
 * You have to set another Button's state to true.
 *
 * @param radioGroup set radioGroup to true to enable radioGroup behavior
 */
public void setRadioGroup(boolean radioGroup) {
	if (radioGroup != fRadioGroup) {
		fRadioGroup = radioGroup;
		sendUI("setRadioGroup", new Anything(fRadioGroup));
	}
}
/**
 * Set the rows to be rows. This should be private
 * because it needs to be set before uploading. But
 * beaninfo won't let us have a private setter.
 *
 * @param rows	The integer rows 
 */
public void setRows(int rows) {
	fRows = rows;
}
/**
 * Set the vertical gap between rows to be vgap.
 *
 * @param verticalGap	The integer gap 
 */
public void setVerticalGap(int verticalGap) {
	if (verticalGap != fVGap) {
		fVGap = verticalGap;
		sendUI("setVGap", new Anything(fVGap));
	}
}
/**
 * Increment the index at which the next component will be added by n.
 *
 * @param n The number of cells to be skipped over.
 */
public void skip(int n) {
	fNextIndex += n;
}
/**
 * Add the component c spanning hspan number of cells horizontally.
 *
 * @param hspan The number of cells to span this component over.
 */
public void span(int hspan, ULCComponent c) {
	int x, y;
	if (fCols == 0) {
		x = fNextIndex;
		y = 0;
	}
	else {
		x = fNextIndex % fCols;
		y = fNextIndex / fCols;
	}
	fNextIndex += hspan;
	internalAdd(new BoxCellAttributes(c, x, y, hspan));
}
/**
 * Add the component c specifying the horizontal spanning and alignment attribute.
 *
 * @param hspan The number of cells to span this component over.
 * @param attr The alignment attribute.
 * <p>
 * <pre>
 * Format for alignment string:
 * 	'horizontal specifier''vertical specifier'
 *		'horizontal specifier':
 * 			e expand
 * 		 	l left
 * 		 	c center
 * 		 	r right
 *		'vertical specifier':
 * 		 	e expand
 * 			t top
 * 		 	c center
 * 			b bottom
 * </pre>
 * @param c The component to be added.
 */
public void span(int hspan, String attr, ULCComponent c) {
	int x, y;
	if (fCols == 0) {
		x = fNextIndex;
		y = 0;
	}
	else {
		x = fNextIndex % fCols;
		y = fNextIndex / fCols;
	}
	fNextIndex += hspan;
	internalAdd(new BoxCellAttributes(c, x, y, attr, hspan));
}
}
