package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import java.util.*;

/**
 * An instance of class ULCApplication represents the root object for an ULC application.
 * This root object knows the connection to the UIEngine (ULCContext).
 * All ULC objects have to be (recursively) reachable from this class.
 * ULCApplication holds the ULCShells of the ULC application.
 */
public class ULCApplication extends ULCProxy {
	protected Vector fToBeRemovedShells = new Vector();
	protected ULCHelpBrowserContext fHelpContext = null;
	protected boolean fTerminateOnLastShellClose = true;
/**
 * Create a new ULCApplication with the specified ULCContext.
 * The context contains the connection to the UIEngine.
 *
 * @param context	The ULCContext upon which I should create myself.
 */
public ULCApplication(ULCContext context) {
	super(context);
	fContext.add(this);
}
/**
 * Add a ULCShell to my list of active shells.
 * Every shell that is part of the given application,
 * must be added here. All of these shells are then
 * controlled by the receiver.
 */
public void add(ULCShell shell) {
	shell.setContext(fContext);
	shell.setApplication(this);
	if (shell.isVisible())
		internalAdd(shell);
	else {
		if (!fToBeRemovedShells.contains(shell))
			fToBeRemovedShells.addElement(shell);
	}
}
/**
 * This widget is being destroyed perform any cleanUp necessary.
 */
public void free() {
	super.free();
	if (fToBeRemovedShells != null) {
		for (int i = 0; i < fToBeRemovedShells.size(); i++) {
			ULCProxy p = (ULCProxy) fToBeRemovedShells.elementAt(i);
			p.free();
		}
		fToBeRemovedShells.removeAllElements();
	}
	fHelpContext = null;
}
/**
 * Return the HelpBrowserContext for this ULCApplication and all its children.
 * The Help context is used when the user hits the F1 key.
 */
public ULCHelpBrowserContext getHelpContext() {
	return fHelpContext;
}
/**
 * Return true if this application will terminate itself when the last shell is closed.
 * If set to false then it is up to the developer to call terminate on this application
 * to force it to terminate itself.
 * (Default is true)
 */
public boolean getTerminateOnLastShellClose() {
	return fTerminateOnLastShellClose;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn	ORBConnection	The connection on which the reply should be sent.
 * @param args	Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("terminate")) {
		terminate();
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Convenience method to reconnect to the UIEngine.
 */
public void reconnectUI() {
	fContext.reconnectUI();
}
/**
 * Remove a ULCShell from my list of active shells.
 */
public void remove(ULCShell shell) {
	if (fList != null) {
		fList.removeElement(shell);
		if (!shell.isDestroyOnClose()) {
			sendUI("remove", shell);
		}
	}
}
/**
 * Convenience method to close the current connection to the UIEngine and start a new one.
 */
public void restartUI() {
	fContext.restartUI();
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fHelpContext != null)
		a.put("hc", fHelpContext.getRef(fContext));
}
/**
 * Register the HelpBrowserContext into the UI. This help context will be
 * used for all widgets within this ULCContext.
 * The Help context is used when the user hits the F1 key.
 *
 * @param conn		ORBConnection	The active connection.
 * @param args		Anything		The arguments associated with this request.
 */
public void setHelpContext(ULCHelpBrowserContext helpContext) {
	fHelpContext = helpContext;
	sendUI("setHelpContext", helpContext.getRef(getContext()));
}
/**
 * Set true if this application will terminate itself when the last shell is closed.
 * If set to false then it is up to the developer to call terminate on this application
 * to force it to terminate itself.
 * (Default is true)
 */
public void setTerminateOnLastShellClose(boolean terminateOnLastShellClose) {
	fTerminateOnLastShellClose = terminateOnLastShellClose;
}
/**
 * Handles all processing necessary when a shell has been closed.
 *
 * @param shell 	The <code> ULCShell</code> that is being closed.
 */
public void shellClosed(ULCShell shell) {
	remove(shell);
	if (!shell.isDestroyOnClose())
		fToBeRemovedShells.addElement(shell);
	if (getTerminateOnLastShellClose())
		terminateIfLastShell();
}
/**
 * Handles all processing necessary when a shell has been shown.
 *
 * @param shell 	The <code> ULCShell</code> that is being shown.
 */
public void shellShown(ULCShell shell) {
	if (fToBeRemovedShells.indexOf(shell) >= 0) {
		add(shell);
		fToBeRemovedShells.removeElement(shell);
	}
}
/**
 * This is the method to be used for terminating
 * an ULC application. It will send a request to the UI 
 * to destroy all objects associated with this object and will
 * result in all the shells (contained therein) to be destroyed.
 * Usually this is the last request sent to the UI.
 */
public void terminate() {
	sendUI("terminate");
	fContext.applicationTerminated(this);
}
/**
 * Check if there are no active ULCShells within my ULCApplication. 
 * If there are no active ULCShells close the connection
 */
protected void terminateIfLastShell() {
	Enumeration e = fList.elements();
	while (e.hasMoreElements()) {
		Object o = e.nextElement();
		if (o instanceof ULCShell) {
			return;
		}
	}
	terminate();
}
}
