/*
 * (c) Copyright 1996, KL GROUP INC.
 * ALL RIGHTS RESERVED
 *
 * This file is provided for demonstration and educational uses only.
 * Permission to use, copy, modify and distribute this file for
 * any purpose and without fee is hereby granted, provided that the
 * above copyright notice and this permission notice appear in all
 * copies, and that the name of KL Group not be used in advertising
 * or publicity pertaining to this material without the specific,
 * prior written permission of an authorized representative of
 * KL Group.
 *
 * KL GROUP MAKES NO REPRESENTATIONS AND EXTENDS NO WARRANTIES, EX-
 * PRESS OR IMPLIED, WITH RESPECT TO THE SOFTWARE, INCLUDING, BUT
 * NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR ANY PARTICULAR PURPOSE, AND THE WARRANTY AGAINST IN-
 * FRINGEMENT OF PATENTS OR OTHER INTELLECTUAL PROPERTY RIGHTS.  THE
 * SOFTWARE IS PROVIDED "AS IS", AND IN NO EVENT SHALL KL GROUP OR
 * ANY OF ITS AFFILIATES BE LIABLE FOR ANY DAMAGES, INCLUDING ANY
 * LOST PROFITS OR OTHER INCIDENTAL OR CONSEQUENTIAL DAMAGES
 * RELATING TO THE USE OF THIS SOFTWARE.
 */

//   RCSID -- $RCSfile: JCRowColLayout.java $ $Revision: 2.0 $
//            $Date: 1997/06/09 18:22:26 $  $Locker: $  KL Group Inc.

package jclass.contrib;
import java.awt.*;

/**
 * JCRowColLayout is an improved version of Java's AWT GridLayout. It
 * implements the LayoutManager interface, and lays out a
 * grid of rows and columns based on the attributes of the
 * individual rows and columns. Whereas GridLayout uses
 * the widest and tallest child to size each cell, JCRowColLayout
 * uses the widest element in a column to set the width of that
 * column, and the tallest element in a row to set the height of
 * that row. <P>
 * Constructing a JCRowColLayout with a negative number of rows
 * or columns tells JCRowColLayout that the number of rows or
 * columns is variable, and the other item is fixed.
 */
public class JCRowColLayout implements LayoutManager {
	/** The horiztonal gap between items. */
    int hgap;

	/** The vertical gap between items. */
    int vgap;

	/** The number of rows in the layout, as set by the user.
	 * This number may not correspond exactly to the number of
	 * rows in the layout.
	 */
    int rows;

	/** The number of columns in the layout, as set by the user.
	 * This number may not correspond exactly to the number of
	 * columns in the layout.
	 */
    int cols;

	/** An array of row heights.
	 * It is accurate only after a call to getRowColSizes();.
	 */
	int rowHeights[] = null;

	/** This is an array of column widths.
	 * It is accurate only after a call to getRowColSizes();.
	 */
	int colWidths[] = null;

/**
 * A simple constructor with no "gap" information. By default
 * the gap between items is set to 0.
 *
 * @param rows the number of rows in the layout
 * @param cols the number of colums in the layout
 */
public JCRowColLayout(int rows, int cols) {
	this(rows, cols, 0, 0);
}

/**
 * A constructor that includes "gap" information.
 *
 * @param rows the number of rows in the layout
 * @param cols the number of columns in the layout
 * @param hgap the horizontal gap between items
 * @param vgap the vertical gap between items
 */
public JCRowColLayout(int rows, int cols, int hgap, int vgap) {
	if ((rows == 0) && (cols == 0)) {
	    throw new IllegalArgumentException("invalid rows,cols");
	}

	this.rows = rows;
	this.cols = cols;
	this.hgap = hgap;
	this.vgap = vgap;
}

/**
 * An empty method that must appear (it is part of the
 * LayoutManager interface). It does not do anything.
 */
public void addLayoutComponent(String name, Component comp) {
}

/**
 * An empty method that must appear (it is part of the
 * LayoutManager interface). It does not do anything.
 */
public void removeLayoutComponent(Component comp) {
}

/**
 * Traverses the children and determines row heights
 * and column widths. The values are stored in rowHeights[]
 * and columWidths[] for use by other JCRowColLayout members.
 *
 * @param min if true, the minimum size is used. Otherwise,
 *            the preferred size is used.
 * @param parent the parent component. Used to access the
 *               children. (In AWT, container rather than layouts own the
 *               kids).
 */
private void getRowColSizes(boolean min, Container parent) {
	int ncomponents = parent.countComponents();
	int nrows = rows;
	int ncols = cols;
	if (nrows > 0) {
	    ncols = (ncomponents + nrows - 1) / nrows;
	} else {
	    nrows = (ncomponents + ncols - 1) / ncols;
	}
	// Initialize height and width data
	rowHeights = new int[nrows];
	colWidths = new int[ncols];
	for (int indx = 0; indx < nrows; indx++) {
		rowHeights[indx] = 0;
	}
	for (int indx = 0; indx < ncols; indx++) {
		colWidths[indx] = 0;
	}

	for (int i = 0 ; i < ncomponents ; i++) {
	    Component comp = parent.getComponent(i);
	    Dimension d = null;
		if (min) {
			d = comp.minimumSize();
		}
		else {
			d = comp.preferredSize();
		}

		int lrow = i / ncols;
		if (d.height > rowHeights[lrow]) {
			rowHeights[lrow] = d.height;
		}

		int lcol = i % ncols;
		if (d.width > colWidths[lcol]) {
			colWidths[lcol] = d.width;
		}
	}
}

/**
 * Adds together the items of an array and returns it as the
 * "total size". This is used to calculate total width and total height
 * values based on rowHeight[] and columnWidth[].
 *
 * @param array[] the array of ints to be totalled
 * @return int the total of all elements in the array
 */
final private int totalSize(int[] array) {
	int r = 0;
	if (array != null) {
		for (int i = 0; i < array.length; i++) {
			r += array[i];
		}
	}
	return r;
}

/**
 * Calculates the preferred size for this layout. It is part of the
 * LayoutManager interface.
 *
 * @param parent the parent of this layout
 */
public Dimension preferredLayoutSize(Container parent) {
	Insets insets = parent.insets();
	getRowColSizes(false, parent);
	int h = totalSize(rowHeights);
	int w = totalSize(colWidths);
	return new Dimension(insets.left + insets.right + w
						 + (colWidths.length+1)*hgap,
						 insets.top + insets.bottom + h
						 + (rowHeights.length+1)*vgap);
}

/**
 * Calculates the minimum size for this layout. It is part of the
 * LayoutManager interface.
 *
 * @param parent the parent of this layout
 */
public Dimension minimumLayoutSize(Container parent) {
	Insets insets = parent.insets();
	getRowColSizes(true, parent);
	int h = totalSize(rowHeights);
	int w = totalSize(colWidths);
	return new Dimension(insets.left + insets.right + w + (colWidths.length+1)*hgap,
			     insets.top + insets.bottom + h + (rowHeights.length+1)*vgap);
}

/**
 * Performs the layout of the children. It is part of the LayoutManager
 * interface. It calculates the number of actual rows and columns
 * based on the user's settings, retrieves row height and column
 * width information, then moves all the children to the appropriate
 * places.
 *
 * @param parent the parent of this layout
 */
public void layoutContainer(Container parent) {
	Insets insets = parent.insets();
	int ncomponents = parent.countComponents();
	getRowColSizes(false, parent);
	int nrows = rows;
	int ncols = cols;

	if (ncomponents == 0) {
	    return;
	}
	if (nrows > 0) {
	    ncols = (ncomponents + nrows - 1) / nrows;
	} else {
	    nrows = (ncomponents + ncols - 1) / ncols;
	}

	Dimension psize = parent.size();
	int w = 0;
	int h = 0;

	for (int c = 0, x = insets.left+hgap ; c < ncols ; c++) {
	    for (int r = 0, y = insets.top+vgap ; r < nrows ; r++) {
			int i = r * ncols + c;
			if (i < ncomponents) {
				w = Math.min(colWidths[c], psize.width-insets.right-x);
				if (w < 0) w = 0;
				h = Math.min(rowHeights[r], psize.height-insets.bottom-y);
				if (h < 0) h = 0;

				parent.getComponent(i).reshape(x, y, w, h);
			}
			y += rowHeights[r] + vgap;
	    }
		x += colWidths[c] + hgap;
	}
}

}

