/*
*********************************************************************

SAMPLE NAME:  mkmiglst.c

USAGE:        mkmiglst [-h|-?] [-T|-C] [-f|-c] [-o <outfile>] [-r <release>] ...

COMPILATION:  
              Unix:
                 unset LIBPATH
                 cc  -D__UNIX__  mkmiglst.c -o mkmiglst
              Intel:
                 icc -D__INTEL__ mkmiglst.c

ENVIRONMENT VARIABLES:
              To migrate from CMVC 2.3.1:
                 CMVC_FAMILY, CMVC_USER, CMVC_BECOME, CMVC_KEYS, CMVC_ALLCOMMON 
                 TC_FAMILY, TC_USER, TC_BECOME, TC_DBPATH, HOSTNAME

              To migrate from TeamConnection Version 2:
                 TC_FAMILY, TC_FAMILY_CLIENT, TC_USER, TC_BECOME, HOSTNAME

DESCRIPTION:  This utility creates a migration nput script for use by the
              VisualAge TeamConnection Version 3 migration utility:
              - migtc:   Migrate from TeamConnection Version 2
              - migcmvc: Migrate from CMVC Version 2.3.1.3 (or later)
              The generated migration script is input to these utilities.
              For example, a TeamConnection V2 to TeamConnection V3
              migration can be executed to populate a new V3 family
              using the following command:
              $ migtc < migtc.lst > migtc.out 2>&1
              This command will using the default TeamConnection V2 to
              V3 migration script (provided with TeamConnection V3)
              to migrate ALL data from a V2 family to a V3 family.

AUTHOR:       Originally written in Korn Shell by Clifford Meyers
              Converted to C, VATC2 support added, and Intel support
              added by Lee Perlov.  Additional updates by Mark Dunn

PREREQUISITE: Provide "rm.exe" and "tail.exe" and from aixlike tools.

*********************************************************************
*                IBM VisualAge TeamConnection
*                     Version 3 Release 2
*
*                          5622-717
*    (C) Copyright, IBM Corp., 1998. All Rights Reserved.
*             Licensed Materials - Property of IBM
*
*         US Government Users Restricted Rights
*     - Use, duplication or disclosure restricted by
*         GSA ADP Schedule Contract with IBM Corp.
*
*              IBM is a registered trademark of
*        International Business Machines Corporation
*********************************************************************
*
*           NOTICE TO USERS OF THE SOURCE CODE EXAMPLES
*
* INTERNATIONAL BUSINESS MACHINES CORPORATION PROVIDES THE SOURCE CODE
* EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS, "AS IS" WITHOUT
* WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT
* LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE
* OF THE SOURCE CODE EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS,
* IS WITH YOU.  SHOULD ANY PART OF THE SOURCE CODE EXAMPLES PROVE
* DEFECTIVE, YOU (AND NOT IBM OR AN AUTHORIZED DEALER) ASSUME THE ENTIRE
* COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
*
*********************************************************************
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef __INTEL__
   #include <direct.h>
#endif

extern int errno;

/* This is based on a limit used for actions in TeamC */
#define maxParmName 40

#define MAX_PATH 256
#define MAX_BUF  1000
#define MAX_RELS 20000

#define MIG_TEAMC 1
#define MIG_CMVC  2

#define MIG_FILE  1
#define MIG_CHNG  2

#define OS_ERROR  0
#define OS_UNIX   1
#define OS_INTEL  2

#ifdef __UNIX__
   #define DIR_SEP '/'
#endif
#ifdef __INTEL__
   #define DIR_SEP '\\'
#endif

#define REL_ALL   1
#define REL_LIST  2


/* Function Declarations */
int  getOpSys(void);
FILE *initFile(char *pFileName, char *pParms);
char *getEnvVal(char *inValue);
void usage(FILE *pFILE);
void userHelp(void);
int  genScript1(FILE* outFile, int fSource, int fParts, char *pReleases);
int  genScript2(FILE* outFile, int fSource, char *pReleases, char *pHomeDir,
     char *pSourceFamily, int fParts, int fOpSys);
int  genIntructions(FILE* outfile, char *pOutFile, int fSource, int fOpSys,
     char *pHomeDir);

/* Keyword global */
char keywords[]="$KW=@(#); $ChkD=1999/04/30 18:31:34; $FN=tools/mkmiglst.c; $Own=jhook; $Ver; $EKW;";

/*-----------------------------------------------------------------------------\
|  getOpsys:                                                                   |
\-----------------------------------------------------------------------------*/
int getOpSys(void)
{
   #ifdef __UNIX__
      return(OS_UNIX);
   #endif
   #ifdef __INTEL__
      return(OS_INTEL);
   #endif
   return(OS_ERROR);

}

/*-----------------------------------------------------------------------------\
|  initFile:                                                                   |
|  - Receive parameters to open file                                           |
|  - Return File Handle                                                        |
\-----------------------------------------------------------------------------*/
FILE *initFile(char *pFileName, char *pParms)
{
   FILE *pFile;
   pFile = fopen(pFileName, pParms);
   if (pFile == (FILE *)0)
   {
       fprintf(stderr,"Error: Could not open file \"%s\"\n", pFileName);
       return (FILE *)0;
   }

   return (pFile);
}

/*-----------------------------------------------------------------------------\
|  getEnvVal:                                                                  |
\-----------------------------------------------------------------------------*/
char *getEnvVal(char *inValue)
{
   char *outValue;

   outValue = getenv(inValue);
   if (outValue == NULL)
   {
      fprintf(stderr, "Error: %s environment variable must be set\n",
         inValue);
      return NULL;
   }
   return outValue;
}

/*-----------------------------------------------------------------------------\
|  usage:                                                                      |
\-----------------------------------------------------------------------------*/
void usage(FILE *pFILE)
{

   fprintf(pFILE,"\nmkmiglst usage:\n\
mkmiglst -T|-C|-h|-? -f|-c [-o <outfile>] [-a|-r <release>...]\n\
mkmiglst                       - This help message\n\
mkmiglst [-h|-?]               - Extended help\n\
Notes: Parameters must be used in order above,\n\
       Run command from family account home directory.\n");
   return;
}

/*-----------------------------------------------------------------------------\
|  userHelp:                                                                   |
\-----------------------------------------------------------------------------*/
void userHelp(void)
{

   usage(stdout);
   fprintf(stdout,"\n\
Version: %s\n\
\n\
Description:\n\
   This utility creates a migration script for use by the\n\
   VisualAge TeamConnection Version 3 migration utilities:\n\
   - migtc:   Migrate from TeamConnection Version 2\n\
   - migcmvc: Migrate from CMVC Version 2.3.1.3 (or later)\n\n\
   The generated migration script is input to these utilities.\n\
   Also generated is a file containing instructions for performing\n\
   the migration.\n\
   Example: Migrate all data in a TeamConnection V2 family to\n\
   a TeamConnection V3 family (generate script, list instructions, migrate):\n\
   $ mkmiglst -T -c -o migtc -a\n\
   $ more migtc.txt\n\
   $ migtc < migtc1.lst > migtc.out 2>&1\n\
   $ migtc < migtc2.lst >> migtc.out 2>&1\n\
   Example: CMVC 2.3.1 to TeamConnection V3 migration,\n\
   migrating snapshots of selected releases from CMVC 2.3.1 family to\n\
   a TeamConnection V3 family (generate script, list instructions, migrate):\n\
   $ mkmiglst -C -f -o migcmvc -r v209 -r v210\n\
   $ more migcmvc.txt\n\
   $ migcmvc < migcmvc1.lst 2>&1 | tee migcmvc1.out \n\
   $ migcmvc < migcmvc2.lst 2>&1 | tee migcmvc2.out \n\
   Example: CMVC 2.3.1.4 to TeamConnection V3 migration,\n\
   migrating snapshots of selected releases inputted from a file:\n\
   $ mkmiglst -C -f -o migcmvc < release.list\n\
   $ more migcmvc.txt\n\
   $ migcmvc < migcmvc1.lst 2>&1 | tee migcmvc1.out\n\
   $ migcmvc < migcmvc2.lst 2>&1 | tee migcmvc2.out\n", keywords);

   fprintf(stdout,"\
\nDetails:\n\
   Executing mkmiglst generates the following outputs (assume \"-o XXX\" used):\n\
   - XXX.txt: Instructions for performing migration, customized based on\n\
     the parameters used in mkmiglst.\n\
   - XXX1.lst: First file containing actual migration commands,\n\
     customized based on the parameters used in mkmiglst.  Commands in this\n\
     file migrate all non-part data.\n\
   - XXX2.lst: Second file containing actual migration commands,\n\
     customized based on the parameters used in mkmiglst.  Commands in this\n\
     file migrate all files/parts (snapshot of release or all change history).\n\
   These scripts do the following:\n\
   - Create an extract directory.\n\
   - Create a subdirectory for each release.\n\
   - Provide a commented entry that will allow for delete of directory\n\
     after the release is migrated.");

   fprintf(stdout,"\
\nOptions:\n\
   Migrate from CMVC, TeamConnection, or provide help:\n\
   -C: To migrate from CMVC Version 2.3.1 (CMVC 2.3.1.3 or later) required.\n\
   -T: To migrate from VisualAge TeamConnection Version 2.09\n\
   -h or -?: For this detailed help\n\
   Migrate snapshot of files or complete change history:\n\
   -f: Migrate FileView (snapshot of committed release)\n\
   -c: Migrate ChangeView (complete history of release)\n\
   Output file option:\n\
   -o: Specify filename for migration listfile.  Default=migrate.lst.\n\
   Migrate selected release or all releases:\n\
   -r ReleaseName: The release to be migrated.  Use -r for each release.\n\
   -a: To migrate the entire family (all releases)\n\
       Note: If no releases are specified, and -a is not used, then the user\n\
       is prompted to enter each release.\n\
\n\
\nNotes:\n\
   1. This tool and associated technical report is a supplement to the\n\
      VisualAge TeamConnection Version 3 Administrator's Guide.\n\
      Read the chapter on Migration before using this tool.\n\
   2. The output of this tool is determined by the experiences of\n\
      the VisualAge TeamConnection development, test and services teams.\n\
      Editing of the output may be desired, but should not be necessary.\n\
   3. Required environment variables\n\
      For CMVC 2.3.1:\n\
      To migrate from CMVC 2.3.1:\n\
         CMVC_FAMILY, CMVC_USER, CMVC_BECOME, CMVC_KEYS, CMVC_ALLCOMMON\n\
         TC_FAMILY, TC_USER, TC_BECOME, TC_DBPATH, HOSTNAME\n\
      To migrate from TeamConnection Version 2:\n\
         TC_FAMILY, TC_FAMILY_CLIENT, TC_USER, TC_BECOME, HOSTNAME\n\n");

   return;
}

/*-----------------------------------------------------------------------------\
|  genScript1:                                                                 |
|  - Print first set of instructions for CMVC or VATC 2 to VATC 3              |
\-----------------------------------------------------------------------------*/
int genScript1(FILE* outFile, int fSource, int fParts, char *pRelList)
{

   if (fSource == MIG_CMVC)
   {
      printf("Writing first set of instructions for CMVC to VATC3\n");
      fprintf(outFile, "# Writing first set of instructions for CMVC to VATC3\n");
   }
   else
   {
      printf("Writing first set of instructions for VATC2 to VATC3\n");
      fprintf(outFile, "# Writing first set of instructions for VATC2 to VATC3\n");
   }

   fprintf(outFile, "\
# If you use release.delete.ksh set maxerrors to 500\n\
# (allowing for duplicate records)\n\
set maxerrors 5000\n\
# This is the recommended batchsize (a smaller number reduces the\n\
# rollback on failure)\n\
set batchsize 100\n\
# Migrating individual tables\n\
migrate Users\n\
migrate HostView\n\
migrate Authority\n\
migrate Interest\n\
migrate Cfgcomproc\n\
migrate Cfgrelproc\n\
migrate Config\n\
migrate CompView where 1=1 order by addDate\n\
migrate bCompView where name = 'root'\n\
# Recommend migrating all Release records, due to references\n\
# from other objects\n\
migrate ReleaseView\n\
migrate EnvView\n\
migrate AccessView\n\
migrate NotifyView\n\
# Recommend migrating all defects and features, due to references\n\
# from other objects\n\
migrate DefectView\n\
migrate FeatureView\n");

   if (fSource == MIG_TEAMC)
   {
      fprintf(outFile, "\
migrate BuilderView\n\
migrate ParserView\n");

      if (fParts == MIG_FILE)
      {
         fprintf(outFile, "\
migrate DriverView where state in ('commit','complete')\n\
migrate WorkAreaView where state in ('commit','complete')\n\
migrate DriverMemberView order by WorkAreaName asc, committedVersion desc\n");
      }
      else /* Change */
      {
         fprintf(outFile, "\
migrate DriverView\n\
migrate WorkAreaView\n\
migrate DriverMemberView order by WorkareaName asc, committedVersion desc\n");
      }
   }
   else /* CMVC */
   {
      if (fParts == MIG_FILE)
      {
         fprintf(outFile, "\
migrate LevelView where state in ('commit','complete') order by commitDate\n\
migrate TrackView where state in ('commit','complete')\n\
migrate LevelMemberView\n");
      }
      else /* Change */
      {
         fprintf(outFile, "\
migrate LevelView order by commitDate\n\
migrate TrackView\n\
migrate LevelMemberView\n");
      }
   }

   fprintf(outFile, "\
migrate FixView\n\
migrate ApproverView\n\
migrate ApprovalView\n\
migrate SizeView\n\
migrate TestView\n\
migrate VerifyView\n\
migrate CoreqView\n\
migrate NoteView\n\
quit\n");

   return(0);
}

/*-----------------------------------------------------------------------------\
|  genScript2:                                                                 |
|  - Print second set of instructions for CMVC or VATC 2 to VATC 3             |
\-----------------------------------------------------------------------------*/
int genScript2(FILE* outFile, int fSource, char *pReleases, char *pHomeDir,
    char *pSourceFamily, int fParts, int fOpSys)
{
   char pFile[MAX_PATH];     /* File or Part */
   char pTrack[MAX_PATH];    /* Track or WorkArea */
   char pLevel[MAX_PATH];    /* Level or Driver */
   char pCurrRel[MAX_PATH];  /* Current Release */
   char *p, *pEnd;

   if (fSource == MIG_CMVC)
   {
      printf("Writing second set of instructions for CMVC to VATC3\n");
      fprintf(outFile, "# Writing second set of instructions for CMVC to VATC3\n");
      strcpy(pFile, "File");
      strcpy(pTrack, "Track");
      strcpy(pLevel, "Level");
   }
   else
   {
      printf("Writing second set of instructions for VATC2 to VATC3\n");
      fprintf(outFile, "# Writing second set of instructions for VATC2 to VATC3\n");
      strcpy(pFile, "Part");
      strcpy(pTrack, "WorkArea");
      strcpy(pLevel, "Driver");
   }

   fprintf(outFile, "\
# Few warnings should be reported migrating part data\n\
set maxerrors 100\n");

   fprintf(outFile, "\
# You can increase decache to increase performance if you do not approach\n\
# your paging space limit during tests\n\
set decache 1\n");

   fprintf(outFile, "\
# This is the recommended batchsize (a smaller number reduces\n\
# the rollback on failure)\n\
set batchsize 100\n\
# Migrating %s data\n", pFile);

   /* Process versionView */
   if (fParts == MIG_CHNG)
   {
      fprintf(outFile,"\
# Unset top for changeView migration\n\
set top\n");
   }

   /* Process versionView */
   if ((fParts == MIG_CHNG) && (fSource == MIG_TEAMC))
   {
      fprintf(outFile,"\
migrate versionView where 1=1 order by addDate\n\
migrate branchPoints\n");
   }

   /* Loop through Releases to be migrated */
   /* - Always use a list of releases */
   p = pReleases;
   for (pEnd = strchr(p, ' ');
        *p != NULL;
        p = pEnd + 1, pEnd = strchr(p, ' ')
       )
   {
      strncpy(pCurrRel, p, pEnd-p);
      pCurrRel[pEnd-p] = NULL;

      /* File migration options for all platforms (big if statement) */
      if (fParts == MIG_FILE)
      {
         if (fSource == MIG_CMVC)
         {
            fprintf(outFile,"\
# Extract directory must be exported for CMVC to mount\n");

            if (fOpSys == OS_UNIX)
            {
               fprintf(outFile,"\
set top %s/extract/%s\n\
! mkdir -p %s/extract/%s\n\
# Extract without expanding keywords\n\
! Release -extract %s -committed -nokeys -node $(hostname) -root %s/extract/%s -family %s\n\
migrate %sView where releaseName = '%s'\n\
# rm -rf %s/extract/%s\n",
               pHomeDir, pCurrRel, pHomeDir, pCurrRel, pCurrRel,
               pHomeDir, pCurrRel, pSourceFamily, pFile, pCurrRel,
               pHomeDir, pCurrRel);
            }
            else  /* OS_INTEL */
            {
               fprintf(outFile,"\
set top %s\\extract\\%s\n\
! mkdir %s\\extract\n\
! mkdir %s\\extract\\%s\n\
# Extract without expanding keywords and convert text files to Intel format\n\
! Release -extract %s -committed -nokeys -crlf -node %%HOSTNAME%% -root %s\\extract\\%s -family %s\n\
migrate %sView where releaseName = '%s'\n\
# rm -rf %s\\extract\\%s\n",
                pHomeDir, pCurrRel, pHomeDir, pHomeDir, pCurrRel, pCurrRel,
                pHomeDir, pCurrRel, pSourceFamily, pFile, pCurrRel, pHomeDir,
                pCurrRel);
            }
         }
         else /* TeamConnection V2 */
         {
            if(fOpSys == OS_UNIX)
            {
               fprintf(outFile,"\
set top %s/extract/%s\n\
! mkdir -p %s/extract/%s\n\
# Extract without expanding keywords and convert text files to Unix format\n\
! teamc release -extract %s -nokeys -crlf -root %s/extract/%s -family %s\n\
migrate %sView -release %s where dropDate is null\n\
# rm -rf %s/extract/%s\n",
               pHomeDir, pCurrRel, pHomeDir, pCurrRel, pCurrRel, pHomeDir,
               pCurrRel, pSourceFamily, pFile, pCurrRel, pHomeDir, pCurrRel);
            }
            else /* OS_INTEL */
            {
               fprintf(outFile,"\
set top %s\\extract\\%s\n\
! mkdir %s\\extract\n\
! mkdir %s\\extract\\%s\n\
# Extract without expanding keywords and convert text files to Intel format\n\
! teamc release -extract %s -nokeys -crlf -root %s\\extract\\%s -family %s\n\
migrate %sView -release %s where dropDate is null\n\
# rm -rf %s\\extract\\%s\n",
               pHomeDir, pCurrRel, pHomeDir, pHomeDir, pCurrRel, pCurrRel,
               pHomeDir, pCurrRel, pSourceFamily, pFile, pCurrRel, pHomeDir,
               pCurrRel);
            }
         }
      }
      else /* Change history loop */
      {
         if (fSource == MIG_CMVC)
         {
            /* All Parts/Files for change history */
            fprintf(outFile,"\
migrate changeView where releaseName = '%s' order by versionID\n\
migrate %sView where releaseName = '%s'\n", 
               pCurrRel, pFile, pCurrRel);
         }
         else
         {
            /* All Parts/Files for change history */
            fprintf(outFile,"\
migrate %sFullView where releaseName = '%s'\n\
migrate %sView where releaseName = '%s'\n", 
               pFile, pCurrRel, pFile, pCurrRel);
         }
      }
   } /* end loop */

   /* Loop through Releases to be migrated */
   /* - only for TeamConnection and Change View */
   if ((fParts == MIG_CHNG) && (fSource == MIG_TEAMC))
   {
      p = pReleases;
      for (pEnd = strchr(p, ' ');
           *p != NULL;
           p = pEnd + 1, pEnd = strchr(p, ' ')
          )
      {
         strncpy(pCurrRel, p, pEnd-p);
         pCurrRel[pEnd-p] = NULL;

         /* All Parts/Files for change history */
         fprintf(outFile,"\
migrate changeView where releaseName = '%s' order by versionID\n",
            pCurrRel);
      }
   }

   if (fParts == MIG_CHNG)
   {
      fprintf(outFile,"\
# Common Part migation requires all releases for guarenteed success\n\
# - Run separately if using a selected set of releases\n\
# - Warning: This is a very time comsuming command\n\
# migrate commonParts\n");
   }

   fprintf(outFile,"\
quit\n");

   return(0);
}

/*-----------------------------------------------------------------------------\
|  userDirections:                                                                   |
\-----------------------------------------------------------------------------*/
void genInstructions(FILE *outFile, char *pOutFile, int fSource, int fOpSys, char *pHomeDir)
{
   printf("Writing user instructions\n");


   if (fSource == MIG_CMVC)
   {
      fprintf(outFile, "\
PRE-MIGRATION RECOMMENDATIONS\n\
-----------------------------\n\
1. Upgrade CMVC family to CMVC 2.3.1.3 (or later), using latest FixPak and HotFix.\n\
   Technical reports addressing CMVC and TeamConnection migration issues\n\
   are available at http://www.software.ibm.com/ad/teamcon (select Library).\n\
2. Read the README.TXT file provided with the installation media for any \n\
   notes on migration.\n\
3. Create a superuser login and a host list entry for the\n\
   TeamConnection family account in the CMVC family to be migrated.\n\
4. Backup the CMVC family and database.\n\
5. Follow directions for migration in the VisualAge TeamConnection\n\
   Enterprise Server Version 3 Installation.  Use the on-line copy\n\
   delivered on CD-ROM, as the directions are more current.\n\n");
    }
    else
    {
      fprintf(outFile, "\
PRE-MIGRATION RECOMMENDATIONS\n\
-----------------------------\n\
1. Upgrade TeamConnection V2 family to FixPak 209, then add HotFix 3.\n\
   Technical reports addressing CMVC and TeamConnection migration issues\n\
   are available at http://www.software.ibm.com/ad/teamcon (select Library).\n\
2. Read the README.TXT file provided with the installation media for any \n\
   notes on migration.\n\
3. Create a superuser login and a host list entry for the\n\
   TeamConnection V3 family account in the TeamConnection V2\n\
   family to be migrated.\n\
4. Backup the VisualAge TeamConnection V2 family.\n\
5. Follow directions for migration in the VisualAge TeamConnection\n\
   Enterprise Server Version 3 Installation.  Use the on-line copy\n\
   delivered on CD-ROM, as the directions are more current.\n\n");
    }


   if (fSource == MIG_CMVC)
   {
      fprintf(outFile, "\
MIGRATION COMMANDS\n\
------------------\n\
Use the following commands, where the migration instructions call for\n\
executing migcmvc:\n\
1.  migcmvc < %s1.lst 2>&1 | tee %s1.out\n\
2.  For status: tail -f %s1.out\n\
    * tail.exe for OS/2 and Windows NT provided in migrate.zip \n\
      available from the IBM VisualAge TeamConnection ftp site:\n\
      ftp://ftp.boulder.ibm.com/ps/products/teamconnection/tools\n\
3.  Repeat for %s2.lst\n\
4.  Return to migration instructions for the following:\n\
    - Update starting Defect/Feature number in fhcsequence table\n\
    - Performance Tuning\n\
    - Backup of Database\n\
\n\
POST-MIGRATION RECOMMENDATIONS\n\
------------------------------\n\
1.  Remove authorities from CMVC family to block any further\n\
    work on the migrated CMVC release.\n\n",
         pOutFile, pOutFile, pOutFile, pOutFile);
   }
   else
   {
      fprintf(outFile, "\
MIGRATION COMMANDS\n\
------------------\n\
Use the following commands, where the migration instructions call for\n\
executing migtc:\n\
1.  migtc > %s1.out 2>&1 < %s1.lst\n\
2.  For status: tail -f %s1.out\n\
    * tail.exe for OS/2 and Windows NT provided in migrate.zip \n\
      available from the IBM VisualAge TeamConnection ftp site:\n\
      ftp://ftp.boulder.ibm.com/ps/products/teamconnection/tools\n\
3.  Repeat for %s2.lst\n\
4.  Return to migration instructions for the following:\n\
    - Update starting Defect/Feature number in fhcsequence table\n\
    - Performance Tuning\n\
    - Backup of Database\n\
\n\n\
POST-MIGRATION RECOMMENDATIONS\n\
------------------------------\n\
1.  Remove authorities from VATC2 family to block any further\n\
    work on the migrated VATC2 release.\n\n",
         pOutFile, pOutFile, pOutFile, pOutFile);
   }

   if (fSource == MIG_CMVC)
   {
      if (fOpSys == OS_UNIX)
      {
         fprintf(outFile, "\
OPTIONAL ACTIONS FOR CMVC MIGRATION\n\
-----------------------------------\n\
1. Cleanup the work in progress in the target release(s)\n\
   using the following shell scripts.\n\
      release_complete.ksh\n\
      reassign_work.ksh\n\
      release_delete.ksh\n\
\n\
VERIFICATION ACTIONS FOR CMVC MIGRATION\n\
---------------------------------------\n\
1. Change to directory with %s2.lst\n\
2. migVerify.ksh -C %s2.lst \n\
   migVerify.ksh will temporarily create $HOME/cmvc/extract\n\
   and $HOME/TCv300/extract directory structures. \n\
3. Review output file: summary.file.\n",
         pOutFile, pOutFile);
      }
      else
      {
         fprintf(outFile, "\
RECOMMENDED VERIFICATION ACTIONS\n\
---------------------------------------\n\
1. Extract the committed version of selected releases in\n\
   CMVC and VisualAge TeamConnection Version 3.\n\
2. Compare the results using available tools.\n");
      }
   }
   else
   {
      if (fOpSys == OS_UNIX)
      {
         fprintf(outFile, "\
VERIFICATION ACTIONS FOR TEAMCONNECTION V2 MIGRATION\n\
---------------------------------------\n\
1. Change to directory with %s2.lst\n\
2. migVerify.ksh -T %s2.lst \n\
   migVerify.ksh will temporarily create $HOME/TCv2/extract\n\
   and $HOME/TCv300/extract directory structures. \n\
3. Review output file: summary.file.\n",
         pOutFile, pOutFile);
      }
      else
         fprintf(outFile, "\
RECOMMENDED VERIFICATION ACTIONS\n\
---------------------------------------\n\
1. Extract the committed version of selected releases in\n\
   VisualAge TeamConnection Version 2 and\n\
   VisualAge TeamConnection Version 3.\n\
2. Compare the results using available tools.\n");
   }

   fprintf(outFile, "\n\
end of instructions\n\n");

   return;
}

/*-----------------------------------------------------------------------------\
|  main:                                                                       |
|  - Print standard set of arguments at beginning of parameter list:           |
|    UEprogram, UEparameter and EnvFile                                        |
|  - Print rest of arguments; the positional parameters                        |
|  - If EnvFile is not NULL, print the contents of the EnvFile                 |
\-----------------------------------------------------------------------------*/
int main(int argc, char *argv[])
{
   int  fSource;            /* CMVC or TeamConnection V2 */
   int  fRelAll;            /* Migrate All Releases */
   int  fParts;             /* Tip versions (Files) or Change History */
   int  fOpSys;             /* Operating system: Unix or Intel */
   int  nRelParm;           /* Parameter position of first release parameter */
   char pInRel[MAX_PATH];   /* Base name of currently input release name */
   char pReleases[MAX_RELS];/* List of all releases, space separated */
   char pRelList[MAX_RELS]; /* List of all releases, formatted for queries */
   char pHomeDir[MAX_PATH]; /* Name of current working directory */
   char pOutput[MAX_PATH];  /* Base name of output files */
   char pMigOut1[MAX_PATH]; /* Name of first migration file */
   char pMigOut2[MAX_PATH]; /* Name of second migration file */
   char pMigInst[MAX_PATH]; /* Name of instruction migration file */
   char *pMigRpt;           /* Release Report File */
   char *pSourceFamily;     /* Pointer to TC_CLIENT_FAMILY or CMVC_FAMILY */
   char *pTargetFamily;     /* Pointer to value of TC_FAMILY */
   FILE *pfMig1, *pfMig2, *pfMigI, *pfRpt;
   char pCommand[MAX_BUF];
   char pBuffer[MAX_BUF];
   char *p;
   int  i, rc = 0;

   /* If no parameters, print help text */
   if (argc == 1)
   {
      usage(stderr);
      exit (1);
   }

   /* If no parameters, print help text */
   if ((argc == 2) &&
       ((strcmp(argv[1],"-?")==0) ||
        (strcmp(argv[1],"-h")==0)
       )
      )
   {
      userHelp();
      exit (0);
   }

   /* Check operating system */
   fOpSys=getOpSys();
   if (fOpSys == OS_ERROR)
   {
      fprintf(stderr, "ERROR: Compile with operating system parameter \
-D__UNIX__ or -D__INTEL\n");
      exit (1);
   }

   /* If too parameters, print usage text */
   if (argc < 3)
   {
      fprintf(stderr, "ERROR: Too few parameters\n");
      usage(stderr);
      exit (1);
   }

   /* If parameter 1 is incorrect, print usage text */
   if ((strcmp(argv[1],"-T")!=0) && (strcmp(argv[1],"-C")!=0)
      )
   {
      fprintf(stderr, "ERROR: Parameter 1 must be -T or -C\n");
      usage(stderr);
      exit (1);
   }
   else
   {
      if (strcmp(argv[1],"-T")==0)
      {
         printf("Migrating from TeamConnection Version 2\n");
         fSource=MIG_TEAMC;
      }
      else
      {
         printf("Migrating from CMVC Version 2.3.1\n");
         fSource=MIG_CMVC;
      }
   }

   /* If parameter 2 is incorrect, print usage text */
   if ((strcmp(argv[2],"-f")!=0) && (strcmp(argv[2],"-c")!=0)
      )
   {
      fprintf(stderr, "ERROR: Parameter 2 must be -f or -c\n");
      usage(stderr);
      exit (1);
   }
   else
   {
      if (strcmp(argv[2],"-f")==0)
      {
         printf("Migrating snapshot of committed version of release(s)\n");
         fParts=MIG_FILE;
      }
      else
      {
         printf("Migrating complete change history of release(s)\n");
         fParts=MIG_CHNG;
      }
   }

   /* Check environment variables */
   if (fSource == MIG_TEAMC)
   {
      pSourceFamily = getEnvVal("TC_FAMILY_CLIENT");
   }
   else
   {
      pSourceFamily = getEnvVal("CMVC_FAMILY");
      /* Check for CMVC_KEYS only with CMVC */
      p = getEnvVal("CMVC_KEYS");
      if (p == NULL)
      {
         printf("WARNING: To migrate CMVC Keywords, set CMVC_KEYS=YES\n");
      }
      p = getEnvVal("CMVC_BECOME");
      if (p == NULL)
      {
         printf("WARNING: CMVC_FAMILY needs to be set prior to migration\n");
      }
   }
   pTargetFamily = getEnvVal("TC_FAMILY");

   if ((pSourceFamily == NULL) || (pTargetFamily == NULL))
   {
      exit (1);
   }
   printf("Source family: %s\n", pSourceFamily);
   printf("Target family: %s\n", pTargetFamily);

   /* If parameter 3 not correctly set up as output file, print usage text */
   if (argc > 3)
   {
      if (strcmp(argv[3],"-o")==0)
      {
         if (argv[4]==NULL)
         {
            fprintf(stderr, "ERROR: Output FileName required after -o\n");
            usage(stderr);
            exit (1);
         }
         else
         {
            strcpy(pOutput, argv[4]);
            nRelParm=5; /* Where will first release be */
         }
      }
      else
      {
         strcpy(pOutput, "migrate");
         printf("Using default for output filenames: %s\n", pOutput);
         nRelParm=3; /* Where will first release be */
      }
   }
   else
   {
      strcpy(pOutput, "migrate");
      printf("Using default for output filenames: %s\n", pOutput);
      nRelParm=3; /* Where will first release be */
   }
   sprintf(pMigOut1, "%s1.lst", pOutput);
   sprintf(pMigOut2, "%s2.lst", pOutput);
   sprintf(pMigInst, "%s.txt", pOutput);
   printf("Migration file 1: %s\n", pMigOut1);
   printf("Migration file 2: %s\n", pMigOut2);
   printf("Instruction file: %s\n", pMigInst);

   /* If releases specified, process... */
   strcpy (pReleases, ""); /* Initialize list */
   strcpy (pRelList, "('"); /* Initialize list */
   i = 0;
   if (argc > nRelParm)
   {
      if (strcmp(argv[nRelParm],"-a") == 0)
      {
         fRelAll = REL_ALL;
         printf("Migrating all releases\n");
         strcpy (pRelList, ""); /* Re-initialize list */
         if (argc > nRelParm+1)
         {
            fprintf(stderr, "ERROR: -a must be last parameter when used\n");
            usage(stderr);
            exit (1);
         }
         else
         {
            /* Generate list of releases in temporary file */
            pMigRpt = tmpnam(NULL);
            if (fSource == MIG_CMVC)
            {
               sprintf(pCommand, "Report -view releaseView -where \"dropDate is null\" -raw -family %s 2>&1 > %s 2>&1",
                  pSourceFamily, pMigRpt);
            }
            else /* TEAMC */
            {
               sprintf(pCommand, "teamc report -view releaseView -where \"dropDate is null\" -raw -family %s > %s 2>&1",
                  pSourceFamily, pMigRpt);
            }

            rc = system(pCommand);
            if (rc != 0)
            {
               printf("ERROR: The following command did not execute correctly\n\
->%s\n->OUTPUT of command:\n", pCommand);
               if (fOpSys == OS_INTEL)
               {
                  sprintf(pCommand, "type %s", pMigRpt);
               }
               else
               {
                  sprintf(pCommand, "cat %s", pMigRpt);
               }
               fflush(stdout);
               fflush(stderr);
               system(pCommand);
               exit(1);
            }

            /* Read file and generate release list */
            pfRpt = initFile(pMigRpt, "rb");
            if (pfRpt == (FILE *)0)
            {
               exit (1);
            }

            /* Put ReleaseName in pInRel and dump the rest in command */
            fgets(pBuffer, MAX_BUF, pfRpt);
            while (!feof(pfRpt))
            {
               p = strchr(pBuffer, '|');
               *p = NULL;
               strcpy(pInRel, pBuffer);
               strcat(pReleases, pInRel);
               strcat(pRelList, pInRel);
               strcat(pReleases, " ");
               strcat(pRelList, "','");
               i++;
               printf("Migrating release: %s\n", pInRel);
               nRelParm=nRelParm+2; /* Where will next release be */

               fgets(pBuffer, MAX_BUF, pfRpt);
            }
            fclose(pfRpt);
            remove(pMigRpt);
         }
      }
      else
      {
         fRelAll = REL_LIST;
         printf("Migrating list of releases\n");

         /* Loop while release parameters need to be parsed */
         while (argc > nRelParm)
         {
            if (strcmp(argv[nRelParm],"-r")==0)
            {
               if (argv[nRelParm+1]==NULL)
               {
                  fprintf(stderr, "ERROR: Release name missing after -r\n");
                  usage(stderr);
                  exit (1);
               }
               else
               {
                  strcat(pReleases, argv[nRelParm+1]);
                  strcat(pRelList, argv[nRelParm+1]);
                  strcat(pReleases, " ");
                  strcat(pRelList, "','");
                  i++;
                  printf("Migrating release: %s\n", argv[nRelParm+1]);
                  nRelParm=nRelParm+2; /* Where will next release be */
               }
            }
            else
            {
               fprintf(stderr, "ERROR: Parameter %d must be -r\n", nRelParm);
               usage(stderr);
               exit (1);
            }
         }
         p = pRelList + strlen(pRelList) - 2;
         strcpy(p, ")");
      }
   }
   else
   {
      /* Need to prompt for releases */
      fRelAll = REL_LIST;
      printf("Enter name of release(s) to migrate.  Finish by pressing <enter>.\n");
      printf("Enter release: ");
      p = gets(pInRel);
      while ((p != NULL) && (strlen(p) > 0))
      {
         strcat(pReleases, pInRel);
         strcat(pRelList, pInRel);
         strcat(pReleases, " ");
         strcat(pRelList, "','");
         i++;
         p = gets(pInRel);
      }
      p = pRelList + strlen(pRelList) - 2;
      strcpy(p, ")");
   }

   /* Extra validation */
   if (fRelAll == REL_LIST)
   {
      printf("Total number of releases in list: %d\n", i);
   }

   getcwd(pHomeDir, MAX_PATH);
   if (*(pHomeDir+strlen(pHomeDir)-1) == DIR_SEP)
   {
      *(pHomeDir+strlen(pHomeDir)-1) = NULL;
   }
   printf("Writing to root directory: %s\n", pHomeDir);

   /* Open Output files */
   pfMig1 = initFile(pMigOut1, "w");
   pfMig2 = initFile(pMigOut2, "w");
   pfMigI = initFile(pMigInst, "w");
   if ((pfMig1 == (FILE *)0) || (pfMig2 == (FILE *)0) ||
       (pfMigI == (FILE *)0))
   {
      exit (1);
   }

   /* Generate files for migration */
   genScript1(pfMig1, fSource, fParts,  pRelList);
   genScript2(pfMig2, fSource, pReleases, pHomeDir, pSourceFamily, fParts, fOpSys);
   genInstructions(pfMigI, pOutput, fSource, fOpSys, pHomeDir);

   fclose(pfMig1);
   fclose(pfMig2);
   fclose(pfMigI);

   printf("Processing complete: Output files generated.\n");
   return (0);
}

/* end of file */
