/*
****************************************************************************

NAME: modifyParserInParts releaseName defectORfeature parser inputFileRoot startFileNum endFileNum

EXAMPLE USAGE:

* To invoke this script and use the 5 partial files dfa-cobparse.lst.1 thru 
  dfa-cobparse.lst.5, with the appropriate release and parser names. Furthermore,
  the processing for each partial file will be done in a specific workarea name
  which is derived from the defectORfeature, for example ibm-mig-cobparse-1 thru
  ibm-mig-cobparse-5. Each workarea is integrated, committed and completed before
  the next workarea is created.
  
  rexx modifyParserInParts.rex dfarel1 ibm-mig cobparse dfa-cobparse.lst 1 5  
      
* If you want to simultaneously see the messages in the screen and capture them
  into a file for later viewing, you can invoke the command as follows:
  
  rexx modifyParserInParts.rex dfarel1 ibm-mig cobparse dfa-cobparse.lst 1 5  2>&1 | tee modify1.out

ENVIRONMENT VARIABLES:
      TC_FAMILY
      TC_BECOME

DESCRIPTION:
  This sample script changes the parser of the parts specified in the
  partial files (whose format is based on the inputFileRoot and whose number
  is specified by a batch counter, based on startFileNum and endFileNum) 
  that have a given parser and then issues for each part
  "Part -modify -parser parser". For example, if the inputFileRoot
  is dfa-cobparse.lst, then the partial files will be dfa-cobparse.lst.1 thru
  dfa-cobparse.lst.5, if the startFileNum is 1 and endFileNum is 5.
  
  The processing of each partial file is done within the context of a 
  workarea whose name is based on defectORfeature (which is a defect or feature
  in its ready state), the parser name and the index for the partial file.
  For example, if the feature is ibm-mig, then the workArea names will be
  ibm-mig-cobparse-1 thru ibm-mig-cobparse-5.
  
  The idea is that each workarea should not have a very large number of part
  changes. The expected default number of lines in the partial files is 500, thus,
  one workarea will be opened, then 500 changes will be done, then the workarea
  is integrated and completed, then the next workarea is opened, and the process
  is repeated.
  

****************************************************************************
*
*           NOTICE TO USERS OF THE SOURCE CODE EXAMPLES
*
* INTERNATIONAL BUSINESS MACHINES CORPORATION PROVIDES THE SOURCE CODE
* EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS, "AS IS" WITHOUT
* WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT
* LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
* PARTICULAR PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE
* OF THE SOURCE CODE EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS,
* IS WITH YOU.  SHOULD ANY PART OF THE SOURCE CODE EXAMPLES PROVE
* DEFECTIVE, YOU (AND NOT IBM OR AN AUTHORIZED DEALER) ASSUME THE ENTIRE
* COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
*
****************************************************************************
*/

/* This block of code gets the name of the command, for later use. */

parse source . . cmdName
cmdName = filespec("name",cmdName)
cmdName = substr(cmdName,1,lastPos('.',cmdName)-1)

/* Parse the command line and display usage, if needed. */

parse arg releaseName defectORfeature parser inputFileRoot startFileNum endFileNum .
if (releaseName = '') | (defectORfeature = '') | (parser = '') | (inputFileRoot = '') | (startFileNum = '') | (endFileNum = '') then
do
  say 'usage: 'cmdName' releaseName defectORfeature parser inputFileRoot startFileNum endFileNum'
  say 'This sample script will change the parser for the parts that have a parser'
  say 'and then issues "Part -modify -parser parser" for each part.'
  say 'The parts are read in batches of a maximum number of lines in order to avoid'
  say 'exceeding the recommended maximum of part changes in a workarea (around 500).'
  say 'The total number of batches is specified by startFileNum and endFileNum.'
  say 'Each batch is associated with both:'
  say '- an input file (whose name is based on inputFileRoot and batch counter) and '
  say '- a workarea name (whose name is based on defectORfeature and batch counter).'
  say 'Each workarea is created, the parts are modified, the workarea is integrated'
  say 'and completed; then the next workarea is proceesed in the same manner.'
  exit 1
end

say 'Executing:       ' cmdName
say 'releaseName:     ' releaseName
say 'defectORfeature: ' defectORfeature
say 'parser:          ' parser
say 'inputFileRoot:   ' inputFileRoot
say 'startFileNum:    ' startFileNum
say 'endFileNum:      ' endFileNum
parserAbbrev = substr(parser,1,3)
say 'parserAbbrev:    ' parserAbbrev    /* 3 letter abbreviation of the parser */
TC_DBPATH=VALUE( "TC_DBPATH", , "ENVIRONMENT" )
say 'TC_DBPATH:       ' TC_DBPATH

/* Processing the input files */

counterTotalLines = 0  /* Total number of processed lines, in all batch files */
counterBatch      = startFileNum  /* Index for the batch file and associated workarea */

do until ( counterBatch > endFileNum )  /* Read until the end of the input file */
  counterLines      = 1  /* Line counter for each file */
  
  say ''
  say 'Processing batch:  ' counterBatch
  /* Display the date and time stamp */
  'date /T'
  'time /T'

  /* Opening input file */

  tempFile = inputFileRoot || '.' || counterBatch
  rc_file =  STREAM( tempFile, C, 'open read' )
  if ( rc_file \= "READY:" ) then
  do
    say "Error while trying to open input file:      " tempFile
    say "the return code from opening the stream is: " rc_file
    say "Exiting due to error."
    exit 1
  end

  workAreaName = defectORfeature || '-' || parserAbbrev || '-' || counterBatch
  say ''
  say 'Creating workarea: ' workAreaName
  say "teamc workarea -create -name " workAreaName " -release " releaseName " -defect " defectORfeature " -verbose "
     "@teamc workarea -create -name " workAreaName " -release " releaseName " -defect " defectORfeature " -verbose "
  
  do until ( lines(tempFile) = 0 )  /* Read until the end of the temporary file */
    inputLine = LINEIN(tempFile)
    parse var inputLine baseName '|' releaseName '|' compName '|' committedVersion '|' ,
      addDate '|' dropDate '|' lastUpdate '|' pathName '|' currentVersion '|' ,
      nuAddDate '|' nuDropDate '|' nuPathName '|' userLogin '|' fmode '|' ,
      fileType '|' changeType '|' waName '|' partType '|' temporary '|' ,
      builderName '|' parserName '|' parentName '|' hasChildren '|',
      buildStatus '|' typeOfRelation '|' parameters '|',
      workareAreaChange '|' translation '|' transState '|' restricted .

    /* Modify parser if pathName is not null; that is, if null, skip and read another part */
    if (pathName <> '') then 
    do   
      say "Modifying the parser for part ("counterLines"): " pathName
      say "teamc Part -modify " pathName " -release " releaseName " -workarea " workareaName " -verbose -parser " parser
         "@teamc Part -modify " pathName " -release " releaseName " -workarea " workareaName " -verbose -parser " parser
      if ( RC \= 0 ) then
      do
        say "Failure to modify the parser for part: " pathName
        say "Exiting now."
        exit 1
      end
      counterLines = counterLines + 1
    end /* if pathName is not null */

  end /* do-until for each file */

  say ''
  say 'Cleaning up the %TC_DBPATH%\queue directory'  
  'del /f %TC_DBPATH%\queue\i*'
  'del /f %TC_DBPATH%\queue\m*'

  say ''
  say '***' 
  say 'Integrating and completing workarea: ' workAreaName
  /* Display the date and time stamp */
  'date /T'
  'time /T'

  say "teamc workarea -integrate " workAreaName " -release " releaseName " -verbose -force "
     "@teamc workarea -integrate " workAreaName " -release " releaseName " -verbose -force "
    if ( RC \= 0 ) then
    do
      say "Failure to integrate the workarea: " workAreaName
      say "Exiting now."
      exit 1
    end
  say '***' 
  say ''
  
  counterTotalLines = counterTotalLines + counterLines
  counterBatch = counterBatch + 1 
  rc = STREAM( tempFile, C, 'close' )
end  /* do-until for all batches */
  
say ""
say "Total number of parts that were modified: " counterTotalLines
/* Display the date and time stamp */
'date /T'
'time /T'

exit 0

/*                                                                   */
/*-------  the end  -------------------------------------------------*/
