/*************************************************************************
 *
 * PROGRAM:  FHBPLPRS.CMD
 *
 * IBM PL/I Sample Parser TeamConnection
 *
 *********************************************************************
 *              IBM TeamConnection for OS/2
 *                     Version 1 Release 0
 *
 *                       5622-717
 *    (C) Copyright, IBM Corp., 1995. All Rights Reserved.
 *             Licensed Materials - Property of IBM
 *
 *         US Government Users Restricted Rights
 *     - Use, duplication or disclosure restricted by
 *         GSA ADP Schedule Contract with IBM Corp.
 *
 *              IBM is a registered trademark of
 *        International Business Machines Corporation
 *********************************************************************
 *
 *           NOTICE TO USERS OF THE SOURCE CODE EXAMPLES
 *
 * INTERNATIONAL BUSINESS MACHINES CORPORATION PROVIDES THE SOURCE CODE
 * EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS, "AS IS" WITHOUT
 * WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING, BUT NOT
 * LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE.  THE ENTIRE RISK AS TO THE QUALITY AND PERFORMANCE
 * OF THE SOURCE CODE EXAMPLES, BOTH INDIVIDUALLY AND AS ONE OR MORE GROUPS,
 * IS WITH YOU.  SHOULD ANY PART OF THE SOURCE CODE EXAMPLES PROVE
 * DEFECTIVE, YOU (AND NOT IBM OR AN AUTHORIZED DEALER) ASSUME THE ENTIRE
 * COST OF ALL NECESSARY SERVICING, REPAIR OR CORRECTION.
 *
 *********************************************************************
 *
 * Simple include-file parser for PL/I
 *
 * Syntax:
 *    fhbplprs options... infile outfile
 *
 *    where:
 *      infile:      PL/I source file to be scanned
 *      outfile:     Output file containing list of include-files.  Optional.
 *                   Default is STDOUT:.
 *      opts:
 *        /lc        Folds include-filenames to lowercase
 *        /uc        Folds include-filenames to uppercase
 *                   The default is to use filenames exactly as they appear in
 *                   the source file.
 *        /ext fext  Uses fext as the file extension.  Default is inc.
 *
 * Example:
 *    fhbplprs /uc /ext CPY myfile.pli myfile.out
 *
 * Limitations:
 *    - Assumes all include-files have the same filename extension.
 *    - Does not handle INCLUDE directives split across lines.
 *
 * Return code:
 *   0  if no errors
 *   16 if errors
 *
 **************************************************************************/
   parse arg arguments

   call RxFuncAdd 'SysFileDelete', 'RexxUtil', 'SysFileDelete'
   call RxFuncAdd 'SysFileTree', 'RexxUtil', 'SysFileTree'

   /*
   **  Determine options
   */

   case = ''
   exten = ''
   fileargs = ''

   do until words(arguments) == 0
      arg1 = strip(word(arguments, 1), 'both')
      if translate(arg1) == '/LC' then
         case = 'LOWER'
      else if translate(arg1) == '/UC' then
         case = 'UPPER'
      else if translate(arg1) == '/EXT' then do
         arguments = subword(arguments, 2)
         exten = strip(word(arguments, 1), 'both')
      end
      else
        fileargs = fileargs arg1

      arguments = subword(arguments, 2)
   end

   parse var fileargs inarg outarg familyname releasename workareaname

   /*
   **  Determine output file name
   */
   outarg = strip(outarg)
   If outarg = ""
      then  outfile = "STDOUT:"
   else
      outfile = outarg

   /*
   ** Open the output file
   */
   status = 0
   if outfile <> "STDOUT:" then do
      call SysFileDelete outfile
      status = lineout(outfile,,1)
   end
   if status = '1'
   then do
      call abend "FHBPLPRS:  File" outfile "could not be opened."
   end

   /*
   **  Validate input file
   */
   inarg = strip(inarg)
   if inarg = ""
   then do
      call abend "FHBPLPRS:  Input file not specified."
   end

   infile = inarg

   call  SysFileTree infile, 'file', 'F'

   if  result <> "0"
   then do
      call abend "FHBPLPRS:  File" infile "not found."
   end

   if file.0 = 0
   then do
        call abend "FHBPLPRS:  File" infile "not found."
   end

   if infile = outfile
   then do
      call abend "FHBPLPRS:  Input stream same as output."
   end

   /*
   **  Determine extension
   */
   if exten = "" then
      exten = "inc"

   /*
   **  Scan the input file looking for %INCLUDE directives
   **  or EXEC SQL INCLUDE directives
   */
   call linein infile 1 0

   do until lines(infile) = 0

      newline = linein(infile)

      /*
      ** Determine what kind of line we've got
      */
      parse upper var newline word1 word2 word3 stuff
      if word1 = "%INCLUDE" then
         linetype = "INCLUDE"

      else if (word1 = "%") & ,
              (word2 = "INCLUDE")
      then
         linetype = "INCLUDE"

      else if (word1 = "EXEC") & ,
              (word2 = "SQL")  & ,
              (word3 = "INCLUDE")
      then
         linetype = "SQLINCLUDE"

      else
         linetype = "OTHER"

      select

         /*
         **  Handle PL/I include directive
         */
         when linetype = "INCLUDE" then do

            parse var newline stuff1 "%" include inclname ";" stuff2

            /*
            ** Handle this case:  syslib(filename)
            */
            parse var inclname libname "(" fname ")" stuff1
            if fname <> "" then
               inclname = fname
             else
                inclname = word(inclname, 1)

            /*
            ** If filename missing just ignore it.
            */
            inclname = strip(inclname)
            if inclname == "" then
              iterate

         end

         /*
         **  Handle EXEC SQL INCLUDE
         */
         when linetype = "SQLINCLUDE" then do

            parse var newline exec sql include inclname ";" stuff

         end

         otherwise
            iterate

      end /* select */

      /*
      **  If filename is a quoted string, remove the quotes
      */

      if substr(inclname, 1, 1) = "'" then
         inclname = strip(inclname, , "'")
      else if substr(inclname, 1, 1) = '"' then
         inclname = strip(inclname, , '"')

      else do

         /*
         **  Add the extension
         */
         inclname = inclname || "." || exten

         /*
         ** Fold case of filename if requested
         */
         if case = 'UPPER' then
            inclname = translate(inclname)
         else if case = 'LOWER' then
            inclname = translate(inclname, xrange("a", "z"), xrange("A", "Z"))
      end

      /*
      ** Write out the filename
      */
      call lineout outfile , inclname

   end  /*until i.e. reading of all file complete */

   exit 0

   /*
   *** Issue an error message and exit
   */
   abend:
      parse arg message
      say message
   exit 16

