//=============================================================================
// SDA_SampleAgent
//
// Used to simulate an intelligent agent which provides affinity information
// to Network Dispatcher's 'Server Directed Affinity' feature.
//
// Three classes are defined herein:
//  SDA_SampleAgent - main program which communicates with ND.
//  SDA_Info - holds affinity information in the format for transfer to ND.
//  SDA_Utils - miscellaneous utility routines to communicate with ND.
// These classes were designed so that you could use SDA_Info and SDA_Utils
// unchanged and intact, while replacing SDA_SampleAgent with your own code.
//
// This program opens a TCP connection with Network Dispatcher.
// It sends affinity records to ND, queries them, and deletes them.
//
//       Message Flows
// ---------------------------
//  ND   <- connect -   Agent
//       <- Auth ----
//       <- CIB -----
//       -- Auth --->
//
//       <- Auth ----
//       <- Affin----
//       -- Auth --->
//       -- Resp --->
//
//       <- Auth ----
//       <- Affin----
//       -- Auth --->
//       -- Resp --->
//
//       <- Auth ----
//       <- Affin----
//       -- Auth --->
//       -- Resp --->
//
//          etc...
//
//  The file must be located in the following Network Dispatcher directory:
//
//  - For dispatcher - nd/dispatcher/lib/SDA/ (nd\dispatcher\lib\SDA on NT)
//
//
// AD 03/14/1998 - for Java 1.1.4 (and above)
//=============================================================================
import java.io.*;
import java.net.*;
import java.util.*;
import java.util.Date;


//-----------------------------------------------------------------------------
// SDA_SampleAgent - A class to implement a simple SDA agent.
//-----------------------------------------------------------------------------
class SDA_SampleAgent
{
  //---------------------------------------------------------------------------
  // Version of this agent.
  //---------------------------------------------------------------------------
  //blic static String sVersion = "Program SDA_SampleAgent, version 1.0, March 14, 1998.";
  public static String sVersion = "Program SDA_SampleAgent, version 1.1, November 30, 2000.";


  //---------------------------------------------------------------------------
  // Object variables.
  //---------------------------------------------------------------------------
  // Variables used to communicate with ND.
  InetAddress      inaNetworkDispatcher;    // The IP address of ND.
  int              iNetworkDispatcherPort;  // Port number to connect to ND.
  Socket           soNetworkDispatcher;     // Socket to talk to ND.
  DataInputStream  disIn = null;            // Input stream from ND.
  DataOutputStream dosOut = null;           // Output stream to ND.

  // Variables used to define entries in the ND affinity table.
  InetAddress      inaSdaCluster;           // Cluster address under test.
  int              iSdaPort;                // Port number under test.
  InetAddress      inaSdaClient;            // Client address under test.
  InetAddress      inaSdaServer;            // Server address under test.


  //---------------------------------------------------------------------------
  // Function - Display the contents of this object (for debug only).
  //---------------------------------------------------------------------------
  public String toString() {
    String sRc = "SDA_SampleAgent object:\n" +
                 "------------------------------------------------\n" +
                 "inaNetworkDispatcher ..... " + inaNetworkDispatcher   + "\n" +
                 "iNetworkDispatcherPort ... " + iNetworkDispatcherPort + "\n" +
                 "soNetworkDispatcher ...... " + soNetworkDispatcher    + "\n" +
                 "disIn .................... " + disIn                  + "\n" +
                 "dosOut ................... " + dosOut                 + "\n" +
                 "inaSdaCluster ............ " + inaSdaCluster          + "\n" +
                 "iSdaPort ................. " + iSdaPort               + "\n" +
                 "inaSdaClient ............. " + inaSdaClient           + "\n" +
                 "inaSdaServer ............. " + inaSdaServer           + "\n" ;
    return sRc;
  }


  //---------------------------------------------------------------------------
  // Constructor.
  //---------------------------------------------------------------------------
  public SDA_SampleAgent() throws Exception {

    // Note: These values are samples only.
    // You must modify them to work in your environment.

    // Define the address and port to communicate with ND.
    inaNetworkDispatcher      = InetAddress.getByName("9.37.59.117");
    inaNetworkDispatcher      = InetAddress.getByName("9.37.52.217");
    iNetworkDispatcherPort    = 10005;

    // Define variables used as entries in the ND affinity table.
    inaSdaCluster             = InetAddress.getByName("9.37.61.44");
    iSdaPort                  = 80;
    inaSdaClient              = InetAddress.getByName("9.37.54.55");
    inaSdaServer              = InetAddress.getByName("9.37.52.115");
  }


  //---------------------------------------------------------------------------
  // Function - Open an authorized connection with ND and identify ourself.
  //---------------------------------------------------------------------------
  public void openND() throws Exception {

    // Open the socket.
    openSocket();

    // Send an auth string.
    SDA_Utils.sendAuth(dosOut);

    // Send our CIB string.
    SDA_Utils.sendCIB(dosOut);

    // Wait for an auth string.
    SDA_Utils.recvAuth(disIn);
  }


  //---------------------------------------------------------------------------
  // Function - Open a socket to ND on its SDA listen port.
  //---------------------------------------------------------------------------
  public void openSocket() throws Exception {

    // Tell the user...
    System.out.println( "\n\nSDA_Sample> OPENING CONNECTION WITH NETWORK DISPATCHER:\n" +
                        "            Address ........ " + inaNetworkDispatcher + "\n" +
                        "            Port ........... " + iNetworkDispatcherPort );
    try
    {
      // Open the socket.
      soNetworkDispatcher = new Socket(inaNetworkDispatcher, iNetworkDispatcherPort);

      // "Cast" the socket into streams for use reading and writing data.
      disIn = new DataInputStream( soNetworkDispatcher.getInputStream() );
      dosOut = new DataOutputStream( soNetworkDispatcher.getOutputStream() );

      // Set the socket receive timeout (arbitrarily) in milliseconds.
      soNetworkDispatcher.setSoTimeout(20000);

      // Tell the user...
      System.out.println( "SDA_Sample> Opened connection successfully." );
    }

    catch( Exception e )
    {
      // Tell the user...
      System.out.println( "SDA_Sample> Error: Caught a java exception while opening socket to ND:\n" + e );
      System.out.println( "SDA_Sample> Perhaps the ND's hostname is invalid?...");
      System.out.println( "SDA_Sample> Perhaps the ND's SDA listen port is invalid?...");
      throw e;
    }
  }


  //---------------------------------------------------------------------------
  // Function - Query the contents of the ND affinity table.
  //---------------------------------------------------------------------------
  public void queryTable() throws Exception {

    // Create a java object for use communicating with ND and define the command.
    SDA_Info sdai = new SDA_Info();
    sdai.iMessageVersion = 1;
    sdai.iCommand        = SDA_Info.SDA_CMD_QUERY;
    sdai.iResponse       = SDA_Info.SDA_RSP_SUCCESS;
    sdai.iClusterAddr    = SDA_Utils.inaToInt(inaSdaCluster);
    sdai.iPort           = iSdaPort;
    sdai.iNumAffinities  = 0;

    // Tell the user...
    System.out.println("\n\nSDA_Sample> ABOUT TO QUERY THE CONTENTS OF THE ND AFFINITY TABLE:");
    System.out.println(sdai);

    try {
      // Send the command to ND and get a response.
      SDA_Utils.talkToNd(sdai,disIn,dosOut);
    }
    catch(Exception e) {
      throw e;
    }

    // Bomb on error.
    if (SDA_Info.SDA_RSP_SUCCESS != sdai.iResponse) {
      throw new Exception("Error from ND querying the contents of the ND affinity table...");
    }
  }


  //---------------------------------------------------------------------------
  // Function - Query the contents of the ND affinity table for one client.
  //---------------------------------------------------------------------------
  public void queryOneClient() throws Exception {

    // Create a java object for use communicating with ND and define the command.
    SDA_Info sdai = new SDA_Info();
    sdai.iMessageVersion = 1;
    sdai.iCommand        = SDA_Info.SDA_CMD_QUERY;
    sdai.iResponse       = SDA_Info.SDA_RSP_SUCCESS;
    sdai.iClusterAddr    = SDA_Utils.inaToInt(inaSdaCluster);
    sdai.iPort           = iSdaPort;
    sdai.iNumAffinities  = 1;

    sdai.aiClientAddr[0] = SDA_Utils.inaToInt(inaSdaClient);
    sdai.aiServerAddr[0] = 0;
    sdai.aiResponse[0]   = SDA_Info.SDA_RSP_SUCCESS;


    // Tell the user...
    System.out.println("\n\nSDA_Sample> ABOUT TO QUERY THE TABLE FOR ONE CLIENT:");
    System.out.println(sdai);

    try {
      // Send the command to ND and get a response.
      SDA_Utils.talkToNd(sdai,disIn,dosOut);
    }
    catch(Exception e) {
      throw e;
    }

    // Bomb on error.
    if (SDA_Info.SDA_RSP_SUCCESS != sdai.iResponse) {
      throw new Exception("Error from ND querying one client in the ND affinity table...");
    }
  }


  //---------------------------------------------------------------------------
  // Function - Query the contents of the ND affinity table for one server.
  //---------------------------------------------------------------------------
  public void queryOneServer() throws Exception {

    // Create a java object for use communicating with ND and define the command.
    SDA_Info sdai = new SDA_Info();
    sdai.iMessageVersion = 1;
    sdai.iCommand        = SDA_Info.SDA_CMD_QUERY;
    sdai.iResponse       = SDA_Info.SDA_RSP_SUCCESS;
    sdai.iClusterAddr    = SDA_Utils.inaToInt(inaSdaCluster);
    sdai.iPort           = iSdaPort;
    sdai.iNumAffinities  = 1;

    sdai.aiClientAddr[0] = 0;
    sdai.aiServerAddr[0] = SDA_Utils.inaToInt(inaSdaServer);
    sdai.aiResponse[0]   = SDA_Info.SDA_RSP_SUCCESS;


    // Tell the user...
    System.out.println("\n\nSDA_Sample> ABOUT TO QUERY THE TABLE FOR ONE SERVER:");
    System.out.println(sdai);

    try {
      // Send the command to ND and get a response.
      SDA_Utils.talkToNd(sdai,disIn,dosOut);
    }
    catch(Exception e) {
      throw e;
    }

    // Bomb on error.
    if (SDA_Info.SDA_RSP_SUCCESS != sdai.iResponse) {
      throw new Exception("Error from ND querying one server in the ND affinity table...");
    }
  }


  //---------------------------------------------------------------------------
  // Function - Add a record to the ND affinity table.
  //---------------------------------------------------------------------------
  public void addRecord() throws Exception {

    // Create a java object for use communicating with ND and define the command.
    SDA_Info sdai = new SDA_Info();
    sdai.iMessageVersion = 1;
    sdai.iCommand        = SDA_Info.SDA_CMD_ADD;
    sdai.iResponse       = SDA_Info.SDA_RSP_SUCCESS;
    sdai.iClusterAddr    = SDA_Utils.inaToInt(inaSdaCluster);
    sdai.iPort           = iSdaPort;
    sdai.iNumAffinities  = 1;

    sdai.aiClientAddr[0] = SDA_Utils.inaToInt(inaSdaClient);
    sdai.aiServerAddr[0] = SDA_Utils.inaToInt(inaSdaServer);
    sdai.aiResponse[0]   = SDA_Info.SDA_RSP_SUCCESS;

    // Tell the user...
    System.out.println("\n\nSDA_Sample> ABOUT TO ADD A RECORD TO THE ND AFFINITY TABLE:");
    System.out.println(sdai);

    try {
      // Send the command to ND and get a response.
      SDA_Utils.talkToNd(sdai,disIn,dosOut);
    }
    catch(Exception e) {
      throw e;
    }

    // Bomb on error.
    if ((SDA_Info.SDA_RSP_SUCCESS != sdai.iResponse) ||
        (1 > sdai.iNumAffinities) ||
        (SDA_Info.SDA_RSP_SUCCESS != sdai.aiResponse[0])) {
      throw new Exception("Error from ND adding record to the ND affinity table...");
    }
  }


  //---------------------------------------------------------------------------
  // Function - Delete a record from the ND affinity table.
  //---------------------------------------------------------------------------
  public void deleteRecord() throws Exception {

    // Create a java object for use communicating with ND and define the command.
    SDA_Info sdai = new SDA_Info();
    sdai.iMessageVersion = 1;
    sdai.iCommand        = SDA_Info.SDA_CMD_DELETE;
    sdai.iResponse       = SDA_Info.SDA_RSP_SUCCESS;
    sdai.iClusterAddr    = SDA_Utils.inaToInt(inaSdaCluster);
    sdai.iPort           = iSdaPort;
    sdai.iNumAffinities  = 1;

    sdai.aiClientAddr[0] = SDA_Utils.inaToInt(inaSdaClient);
    sdai.aiServerAddr[0] = SDA_Utils.inaToInt(inaSdaServer);
    sdai.aiResponse[0]   = SDA_Info.SDA_RSP_SUCCESS;

    // Tell the user...
    System.out.println("\n\nSDA_Sample> ABOUT TO DELETE A RECORD FROM THE ND AFFINITY TABLE:");
    System.out.println(sdai);

    try {
      // Send the command to ND and get a response.
      SDA_Utils.talkToNd(sdai,disIn,dosOut);
    }
    catch(Exception e) {
      throw e;
    }

    // Bomb on error.
    if ((SDA_Info.SDA_RSP_SUCCESS != sdai.iResponse) ||
        (1 > sdai.iNumAffinities) ||
        (SDA_Info.SDA_RSP_SUCCESS != sdai.aiResponse[0])) {
      throw new Exception("Error from ND deleting record from the ND affinity table...");
    }
  }


  //---------------------------------------------------------------------------
  // Function - Delete all records from the ND affinity table.
  //---------------------------------------------------------------------------
  public void deleteAllRecords() throws Exception {

    // Create a java object for use communicating with ND and define the command.
    SDA_Info sdai = new SDA_Info();
    sdai.iMessageVersion = 1;
    sdai.iCommand        = SDA_Info.SDA_CMD_DELETE_ALL;
    sdai.iResponse       = SDA_Info.SDA_RSP_SUCCESS;
    sdai.iClusterAddr    = SDA_Utils.inaToInt(inaSdaCluster);
    sdai.iPort           = iSdaPort;
    sdai.iNumAffinities  = 0;

    // Tell the user...
    System.out.println("\n\nSDA_Sample> ABOUT TO DELETE ALL RECORDS FROM THE ND AFFINITY TABLE:");
    System.out.println(sdai);

    try {
      // Send the command to ND and get a response.
      SDA_Utils.talkToNd(sdai,disIn,dosOut);
    }
    catch(Exception e) {
      throw e;
    }

    // Bomb on error.
    if (SDA_Info.SDA_RSP_SUCCESS != sdai.iResponse) {
      throw new Exception("Error from ND deleting all records from the ND affinity table...");
    }
  }


  //---------------------------------------------------------------------------
  // Function - Main.
  //---------------------------------------------------------------------------
  public static void main( String[] sArgs )
  {
    SDA_SampleAgent sa = null;

    // Tell the user our version.
    System.out.println("SDA_Sample> " + sVersion);

    try {
      // Define the addresses and ports specifically for this sample program.
      sa = new SDA_SampleAgent();

      // Open a connection with ND and identify ourself.
      sa.openND();

      // Query the contents of the ND affinity table.
      sa.queryTable();

      // Delete all affinity records.
      sa.deleteAllRecords();

      // Query the contents of the ND affinity table.
      sa.queryTable();

      // Add a new affinity record.
      sa.addRecord();

      // Query the contents of the ND affinity table.
      sa.queryTable();

      // Query one client in the ND affinity table.
      sa.queryOneClient();

      // Query one server in the ND affinity table.
      sa.queryOneServer();

      // Delete the new affinity record.
      sa.deleteRecord();

      // Query the contents of the ND affinity table.
      sa.queryTable();
    }

    // Error handling.
    catch(Exception e) {
      // Tell the user.
      System.out.println("SDA_Sample> Caught exception in routine main: " + e);
    }

    // Close socket.
    finally {
      if (null != sa.soNetworkDispatcher) {
        System.out.println("SDA_Sample> Closing socket in routine main.");
        try { sa.soNetworkDispatcher.close(); }  catch(Exception x){;}
      }
    }
  }


}  // End - class SDA_SampleAgent.





//-----------------------------------------------------------------------------
// SDA_Utils - Miscellaneous utility routines to communicate with ND.
//-----------------------------------------------------------------------------
class SDA_Utils
{
  //---------------------------------------------------------------------------
  // Definition of the auth string.
  //---------------------------------------------------------------------------
  public static final byte[] abAuth = {
    // "MANAGER Copyright (C) International Business Machines 1996"
    0x4d, 0x41, 0x4e, 0x41, 0x47, 0x45, 0x52, 0x20,
    0x43, 0x6f, 0x70, 0x79, 0x72, 0x69, 0x67, 0x68,
    0x74, 0x20, 0x28, 0x43, 0x29, 0x20, 0x49, 0x6e,
    0x74, 0x65, 0x72, 0x6e, 0x61, 0x74, 0x69, 0x6f,
    0x6e, 0x61, 0x6c, 0x20, 0x42, 0x75, 0x73, 0x69,
    0x6e, 0x65, 0x73, 0x73, 0x20, 0x4d, 0x61, 0x63,
    0x68, 0x69, 0x6e, 0x65, 0x73, 0x20, 0x31, 0x39,
    0x39, 0x36
  };


  //---------------------------------------------------------------------------
  // Definition of the ConnectionInitBlock (CIB) for this agent.
  //---------------------------------------------------------------------------
  public static final byte[] abCIB = {
    // Interface protocol version number "01.00.00.00"
    0x30, 0x31, 0x2e, 0x30, 0x30, 0x2e, 0x30, 0x30, 0x2e, 0x30, 0x30, 0x00,
    // Name of this agent "SDA_Sample"
    0x53, 0x44, 0x41, 0x5F, 0x53, 0x61, 0x6d, 0x70, 0x6c, 0x65,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    // Reserved for future use. Must be zero.
    0x00, 0x00, 0x00, 0x00
  };


  //---------------------------------------------------------------------------
  // Function - Send an "Auth" string to ND.
  //---------------------------------------------------------------------------
  public static void sendAuth(DataOutputStream dosOut) throws Exception {

    // Tell the user...
    System.out.println( "SDA_Sample> About to send an auth string to ND...");

    try {
      // Send the "auth" string as an array of bytes.
      dosOut.write( abAuth, 0, abAuth.length );
      dosOut.flush();
    }

    catch( Exception e ) {
      // Tell the user...
      System.out.println( "SDA_Sample> Error 2 sending auth string to ND.");
      throw e;
    }
  }


  //---------------------------------------------------------------------------
  // Function - Receive an "Auth" string from ND.
  // This blocks until data is received or the socket timeout expires.
  //---------------------------------------------------------------------------
  public static void recvAuth(DataInputStream disIn) throws Exception {
    byte[] ab = new byte[abAuth.length];
    int iBytesRead = 0;

    // Tell the user...
    System.out.println( "SDA_Sample> Waiting for an auth string from ND...");

    try {
      // Read into temporary array ab.
      iBytesRead = disIn.read(ab);

      // Verify we received an auth.  Check the number of bytes.
      if (abAuth.length == iBytesRead) {
        // Check each byte.
        for (int i=0; i<iBytesRead; i++) {
          if (ab[i] != abAuth[i]) {
            // Tell the user.
            System.out.println( "SDA_Sample> Error in content of auth string received from ND.");
            throw new Exception("invalid content auth");
          }
        }
      }
      else {
        // Tell the user.
        System.out.println( "SDA_Sample> Error in length of auth string received from ND.");
        throw new Exception("invalid length auth");
      }

      // Tell the user...
      System.out.println( "SDA_Sample> Successfully received an auth string from ND...");
    }

    catch( Exception e ) {
      System.out.println( "SDA_Sample> Error waiting for auth string from ND.");
      throw e;
    }
  }


  //---------------------------------------------------------------------------
  // Function - Send a "CIB" (ConnectionInitBlock) string to ND.
  //---------------------------------------------------------------------------
  public static void sendCIB(DataOutputStream dosOut) throws Exception {

    // Tell the user...
    System.out.println( "SDA_Sample> About to send a CIB string to ND...");

    try {
      // Send the "CIB" string as an array of bytes.
      dosOut.write( abCIB, 0, abCIB.length );
      dosOut.flush();
    }

    catch( Exception e ) {
      // Tell the user...
      System.out.println( "SDA_Sample> Error sending CIB string to ND.");
      throw e;
    }
  }


  //---------------------------------------------------------------------------
  // Function - Send an SDA_Info command to ND and get a response.
  //---------------------------------------------------------------------------
  public static void talkToNd(SDA_Info sdai,
                              DataInputStream disIn,
                              DataOutputStream dosOut) throws Exception {

    try {
      // Send an Authorization string to ND.
      SDA_Utils.sendAuth(dosOut);

      // Send the affinity information command.
      sdai.sendCommand(dosOut);
    }
    catch(Exception e) {
      System.out.println("SDA_Sample> Error sending command to ND.");
      throw e;
    }

    try {
      // Wait for an auth string.
      SDA_Utils.recvAuth(disIn);

      // Receive the response.
      sdai.recvResponse(disIn);

      // Tell the user.
      System.out.println("SDA_Sample> Received response from ND:");
      System.out.println(sdai);
    }
    catch(Exception e) {
      System.out.println("SDA_Sample> Error receiving response from ND.");
      throw e;
    }
  }


  //---------------------------------------------------------------------------
  // Function - Convert a java InetAddress to an int.
  //---------------------------------------------------------------------------
  public static int inaToInt(InetAddress ina) {
    int i;
    int iRc;

    // Extract the address into an array of bytes.
    byte[] ab = ina.getAddress();

    // Move the bytes into an int.
    i = (ab[0] << 24) & 0xFF000000;  iRc  = i;
    i = (ab[1] << 16) & 0x00FF0000;  iRc += i;
    i = (ab[2] <<  8) & 0x0000FF00;  iRc += i;
    i = (ab[3]      ) & 0x000000FF;  iRc += i;
    return iRc;
  }


}  // End - class SDA_Utils.



//-----------------------------------------------------------------------------
// SDA_Info - A class to hold affinity information for transfer to and from ND
//-----------------------------------------------------------------------------
class SDA_Info extends Thread    // Extends thread only to use the sleep method.
{
  //---------------------------------------------------------------------------
  // Command and response constants.
  //---------------------------------------------------------------------------
  public static final int SDA_CMD_ADD                        = 1;
  public static final int SDA_CMD_DELETE                     = 2;
  public static final int SDA_CMD_DELETE_ALL                 = 3;
  public static final int SDA_CMD_QUERY                      = 4;

  public static final int SDA_RSP_SUCCESS                    = 0;
  public static final int SDA_RSP_NO_SUCH_SERVER             = -11;
  public static final int SDA_RSP_NO_SUCH_RECORD             = -26;
  public static final int SDA_RSP_NO_MEMORY_FOR_RECORD       = -27;
  public static final int SDA_RSP_RECORD_ALREADY_EXISTS      = -28;
  public static final int SDA_RSP_TOO_MANY_RECORDS           = -101;
  public static final int SDA_RSP_NO_MEMORY_FOR_REQUEST      = -102;
  public static final int SDA_RSP_NO_SUCH_CLUSTER            = -103;
  public static final int SDA_RSP_NO_SUCH_PORT               = -104;
  public static final int SDA_RSP_PORT_NOT_STICKY            = -105;
  public static final int SDA_RSP_IS_NULL                    = -106;
  public static final int SDA_RSP_NO_SUCH_COMMAND            = -107;
  public static final int SDA_RSP_STICKYTIME_INVALID_FOR_SDA = -108;
  public static final int SDA_RSP_EXECUTOR_NOT_RUNNING       = -109;


  //---------------------------------------------------------------------------
  // Misc Constants.
  //---------------------------------------------------------------------------
  public static final int SDA_MAX_AFFINITIES = 3000;
  public static final int SDA_MESSAGE_VERSION = 1;


  //---------------------------------------------------------------------------
  // Object Variables.
  //---------------------------------------------------------------------------
  public int iMessageVersion;
  public int iCommand;
  public int iResponse;
  public int iClusterAddr;
  public int iPort;
  public int iNumAffinities;
  public int[] aiClientAddr  = new int[SDA_MAX_AFFINITIES];
  public int[] aiServerAddr  = new int[SDA_MAX_AFFINITIES];
  public int[] aiResponse    = new int[SDA_MAX_AFFINITIES];


  //---------------------------------------------------------------------------
  // Constructor.
  //---------------------------------------------------------------------------
  public SDA_Info() {
    iMessageVersion = SDA_MESSAGE_VERSION;
    iResponse = SDA_RSP_SUCCESS;
  }


  //---------------------------------------------------------------------------
  // Function - Send an affinity message to an output stream.
  //---------------------------------------------------------------------------
  public void sendCommand(DataOutputStream dosOut) throws IOException
  {
    // Give ND a little time to do some other work.
    try { sleep(250); }  catch(Exception e) {;}

    // Write single values.
    dosOut.writeInt(iMessageVersion);
    dosOut.writeInt(iCommand);
    dosOut.writeInt(iResponse);
    dosOut.writeInt(iClusterAddr);
    dosOut.writeInt(iPort);
    dosOut.writeInt(iNumAffinities);

    // Write all Affinities.
    for (int i=0; i<iNumAffinities; i++) {
      // Get the values.
      dosOut.writeInt(aiResponse[i]);
      dosOut.writeInt(aiClientAddr[i]);
      dosOut.writeInt(aiServerAddr[i]);
    }
  }


  //---------------------------------------------------------------------------
  // Function - Read an affinity message from an input stream.
  //---------------------------------------------------------------------------
  public void recvResponse(DataInputStream disIn) throws IOException
  {
    // Read the message version.
    iMessageVersion = disIn.readInt();

    // Check the message version.
    if (SDA_MESSAGE_VERSION != iMessageVersion) {
      // We do not know how to interpret the received data.
      throw new IOException();
    }

    // Read single values from the message.
    iCommand        = disIn.readInt();
    iResponse       = disIn.readInt();
    iClusterAddr    = disIn.readInt();
    iPort           = disIn.readInt();
    iNumAffinities  = disIn.readInt();

    // Read all Affinities.
    for (int i=0; i<iNumAffinities; i++) {
      // Get the values.
      aiResponse[i]   = disIn.readInt();
      aiClientAddr[i] = disIn.readInt();
      aiServerAddr[i] = disIn.readInt();
    }
  }


  //---------------------------------------------------------------------------
  // Function - Display the contents of this object (for debug only).
  //---------------------------------------------------------------------------
  public String toString() {
    String sRc = "SDA_Info object:\n" +
                 "------------------------------------------------\n" +
                 "MessageVersion .... " + iMessageVersion + "\n" +
                 "Command ........... " + iCommand + " (" + printSdaCommand(iCommand) + ")\n" +
                 "Response .......... " + iResponse + " (" + printSdaResponse(iResponse) + ")\n" +
                 "ClusterAddr ....... " + printAddress(iClusterAddr) + "\n" +
                 "Port .............. " + iPort + "\n" +
                 "NumAffinities ..... " + iNumAffinities + "\n";
    for (int i=0; i<iNumAffinities; i++) {
      sRc += "Record " + i + ":\n" +
             "  Response ........ " + aiResponse[i] + " (" + printSdaResponse(aiResponse[i]) + ")\n" +
             "  ClientAddr ...... " + printAddress(aiClientAddr[i]) + "\n" +
             "  ServerAddr ...... " + printAddress(aiServerAddr[i]) + "\n" ;
    }
    return sRc;
  }


  //---------------------------------------------------------------------------
  // Function - convert address from an int to a dotted "a.b.c.d" string.
  //---------------------------------------------------------------------------
  public static String printAddress(int i) {
    String sRc = "";
    sRc += (( i >>> 24 ) & 0xFF ) + "." +
           (( i >>> 16 ) & 0xFF ) + "." +
           (( i >>>  8 ) & 0xFF ) + "." +
           (( i >>>  0 ) & 0xFF );
    return sRc;
  }

  //---------------------------------------------------------------------------
  // Function - Convert a command number into a string.
  //---------------------------------------------------------------------------
  public static String printSdaCommand(int i) {
    String sRc;
    switch (i) {
      case SDA_CMD_ADD        : sRc =  "SDA_CMD_ADD"           ; break;
      case SDA_CMD_DELETE     : sRc =  "SDA_CMD_DELETE"        ; break;
      case SDA_CMD_DELETE_ALL : sRc =  "SDA_CMD_DELETE_ALL"    ; break;
      case SDA_CMD_QUERY      : sRc =  "SDA_CMD_QUERY"         ; break;
      default                 : sRc =  "Unknown"               ; break;
    }
    return sRc;
  }


  //---------------------------------------------------------------------------
  // Function - Convert a response number into a string.
  //---------------------------------------------------------------------------
  public static String printSdaResponse(int i) {
    String sRc;
    switch (i) {
      case SDA_RSP_SUCCESS                    : sRc = "SDA_RSP_SUCCESS"                      ; break;
      case SDA_RSP_NO_SUCH_SERVER             : sRc = "SDA_RSP_NO_SUCH_SERVER"               ; break;
      case SDA_RSP_NO_SUCH_RECORD             : sRc = "SDA_RSP_NO_SUCH_RECORD"               ; break;
      case SDA_RSP_NO_MEMORY_FOR_RECORD       : sRc = "SDA_RSP_NO_MEMORY_FOR_RECORD"         ; break;
      case SDA_RSP_RECORD_ALREADY_EXISTS      : sRc = "SDA_RSP_RECORD_ALREADY_EXISTS"        ; break;
      case SDA_RSP_TOO_MANY_RECORDS           : sRc = "SDA_RSP_TOO_MANY_RECORDS"             ; break;
      case SDA_RSP_NO_MEMORY_FOR_REQUEST      : sRc = "SDA_RSP_NO_MEMORY_FOR_REQUEST"        ; break;
      case SDA_RSP_NO_SUCH_CLUSTER            : sRc = "SDA_RSP_NO_SUCH_CLUSTER"              ; break;
      case SDA_RSP_NO_SUCH_PORT               : sRc = "SDA_RSP_NO_SUCH_PORT"                 ; break;
      case SDA_RSP_PORT_NOT_STICKY            : sRc = "SDA_RSP_PORT_NOT_STICKY"              ; break;
      case SDA_RSP_IS_NULL                    : sRc = "SDA_RSP_IS_NULL"                      ; break;
      case SDA_RSP_NO_SUCH_COMMAND            : sRc = "SDA_RSP_NO_SUCH_COMMAND"              ; break;
      case SDA_RSP_STICKYTIME_INVALID_FOR_SDA : sRc = "SDA_RSP_STICKYTIME_INVALID_FOR_SDA"   ; break;
      case SDA_RSP_EXECUTOR_NOT_RUNNING       : sRc = "SDA_RSP_EXECUTOR_NOT_RUNNING"         ; break;
      default                                 : sRc = "Unknown"                              ; break;
    }
    return sRc;
  }

}  // End - class SDA_Info.
