-----------------------------------------------------------------------
--(c) Copyright IBM Corporation 2006  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_locktree.db2                                              --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Creates a graphical representation of current lock waits.         --
--                                                                   --
-----------------------------------------------------------------------


-----------------------------------------------------------------------
-- Create the table we're going to draw the lock tree into.
call db2perf_quiet_drop('TABLE db2perf_locktree')@
CREATE TABLE db2perf_locktree(line INTEGER,branch VARCHAR(200))@

-----------------------------------------------------------------------
-- Create a table to contain the basic lockwait data from the snapshot table function.

CREATE VIEW db2perf_lockwait_view AS SELECT * FROM TABLE( snapshot_lockwait( CAST(NULL AS VARCHAR(256)),-1 ) ) AS t@
call db2perf_quiet_drop('TABLE db2perf_lockwait')@
CREATE TABLE db2perf_lockwait LIKE db2perf_lockwait_view@
call db2perf_quiet_drop('VIEW db2perf_lockwait_view')@

-----------------------------------------------------------------------
-- We also want to get some basic application data to go with the lock wait,
-- so create a table to hold application info.
CREATE VIEW db2perf_appl_info_view AS SELECT * FROM TABLE( snapshot_appl_info( CAST(NULL AS VARCHAR(256)),-1 ) ) AS t@
call db2perf_quiet_drop('TABLE db2perf_appl_info')@
CREATE TABLE db2perf_appl_info LIKE db2perf_appl_info_view@
call db2perf_quiet_drop('VIEW db2perf_appl_info_view')@



CALL db2perf_quiet_drop('PROCEDURE db2perf_locktree( INTEGER, INTEGER, INTEGER, INTEGER)')@

-----------------------------------------------------------------------
-----------------------------------------------------------------------
-- db2perf_locktree
-- 
-- Recursive version, called by non-recursive version below.
-----------------------------------------------------------------------
-----------------------------------------------------------------------
CREATE PROCEDURE db2perf_locktree(
	IN lock_holder INTEGER, 	-- agent id holding lock
	IN level INTEGER,		-- number of levels deep in the tree 
	IN parent_line INTEGER, 	-- line number of the file that our parent is at
	INOUT current_line INTEGER )	-- tracks the line number of the file we're drawing at
LANGUAGE SQL
MODIFIES SQL DATA
BEGIN
   DECLARE at_end             INTEGER;
   DECLARE stmt               VARCHAR(256);
   DECLARE blanks             CHAR(150);
   DECLARE snapshot_timestamp TIMESTAMP;
   DECLARE waiter             INTEGER;
   DECLARE lock_mode          BIGINT;
   DECLARE lock_object_type   BIGINT;
   DECLARE lock_wait_time     DECIMAL(8,3);
   DECLARE lock_escallation   SMALLINT;
   DECLARE table_name         VARCHAR(64);
   DECLARE appl_name          VARCHAR(32);
   DECLARE starting_line      INTEGER;


   -----------------------------------------------------------------------
   -- Cursor to find all the agents waiting on this holder
   DECLARE waiters CURSOR FOR
     SELECT 
	agent_id,
	lock_mode,
	lock_object_type,
	snapshot_timestamp - lock_wait_start_time,
	lock_escallation,
	rtrim(substr(table_schema,1,32))||'.'||substr(table_name,1,32)
	FROM db2perf_lockwait
        WHERE agent_id_holding_lk = lock_holder;

   DECLARE CONTINUE HANDLER FOR NOT FOUND
     SET at_end = 1; 

   SET at_end = 0;
   SET blanks = '                                                                                                                                                      ';
   SET starting_line = current_line;


   
   -----------------------------------------------------------------------
   -- Build a dynamic SQL statement to call ourselves one level deeper (to
   -- process all the agents that are waiting on us).

   SET stmt = 'CALL db2perf_locktree( ?, ?+1, ?, ? )';
   PREPARE s1 FROM stmt;


   OPEN waiters;
   FETCH waiters INTO 
	waiter,lock_mode,lock_object_type,lock_wait_time,lock_escallation,table_name;


   -----------------------------------------------------------------------
   -- Loop to process all the agents waiting on us.
   WHILE at_end = 0 DO

     IF level > 0 THEN
        -----------------------------------------------------------------------
	-- If our current agent isn't at the root of the tree (i.e., if it is
	-- waiting on something else itself), then draw a line from the parent
	-- to it.
	-- We do this by inserting '|' characters into all the rows between the
	-- parent and self.   We use the level we're at to calculate the indentation,
	-- and we use substr() to split the row, insert the '|', and put it
	-- back together again.
	UPDATE db2perf_locktree
		SET branch = substr(branch,1,8*(level-1)) || '|' || substr(branch,8*(level-1)+2)
		WHERE line > parent_line AND line < current_line;
     END IF;

     -----------------------------------------------------------------------
     -- For each lock wait, there is an agent waiting on us.  So we use
     -- the application snapshot to try to get the application name, etc.
     -- It's possible that the application went away between the time the
     -- lock snapshot was taken and the time the application snapshot was
     -- taken, so we have to be prepare to report something like 'unknown
     -- application'.
     -- 
     -- When we're inserting new lock wait descriptions (application name, etc.)
     -- we need to indent.  We do that by taking a substr() of a long string
     -- of blanks.  The length of the substring (the amount of indentation we use)
     -- is based on the level we're at.

     SET appl_name = '<unknown appl name>';
     SELECT substr(appl_name,1,32) INTO appl_name FROM db2perf_appl_info
	WHERE agent_id = waiter;
     INSERT INTO db2perf_locktree VALUES 
	( current_line,
	  CASE level 
		WHEN 0 THEN '' 
	  	ELSE ( substr(blanks,1,8*(level-1)) || '+-------') 
	  END
	   || 'Waiter appl handle:     ' 
	   || char(waiter) 
	   || ' (' 
	   || rtrim(appl_name) 
	   || ')'
	);
     SET current_line=current_line+1;

     -----------------------------------------------------------------------
     -- Same thing, but for the lock holder this time, plus lock object type, etc.
	
     SET appl_name = '<unknown appl name>';
     SELECT substr(appl_name,1,32) INTO appl_name from db2perf_appl_info
	WHERE agent_id = lock_holder;
     INSERT INTO db2perf_locktree VALUES 
	( current_line,
	  substr(blanks,1,8*level) 
	  	|| '   Holder appl handle:  ' 
		|| char(lock_holder) 
		|| ' (' 
		|| rtrim(appl_name) 
		|| ')'
	);
     SET current_line=current_line+1;

     INSERT INTO db2perf_locktree VALUES 
	( current_line,
	  substr(blanks,1,8*level) 
		|| '   Lock object type:    ' 
		|| db2perf_lkobj2str(lock_object_type) 
	);
     SET current_line=current_line+1;

     INSERT INTO db2perf_locktree VALUES 
	( current_line,
	  substr(blanks,1,8*level) 
	  	|| '   Lock mode requested: ' 
		|| db2perf_lkmode2str(lock_mode) 
	);
     SET current_line=current_line+1;

     INSERT INTO db2perf_locktree VALUES 
	( current_line,
	  substr(blanks,1,8*level) 
	  	|| '   Lock wait time (ms): ' 
		|| char(cast(lock_wait_time * 1000 as integer)) );
     SET current_line=current_line+1;

     INSERT INTO db2perf_locktree VALUES 
	( current_line,
	  substr(blanks,1,8*level) 
	  	|| '   Lock escallation:    ' 
		|| coalesce( char(lock_escallation), 'N' ) );
     SET current_line=current_line+1;

     INSERT INTO db2perf_locktree VALUES 
	( current_line,
	  substr(blanks,1,8*level) 
	  	|| '   Table name:          ' 
		|| table_name );
     SET current_line=current_line+1;


     -----------------------------------------------------------------------
     -- Recursive SQL/PL stored procedures can only go 16 levels deep.
     -- We're counting from zero here, so if we get to 14 (plus the original 
     -- one the user called), we need to just say that we can't go any further
     -- down this chain.
     -- Otherwise, we call this routine again for each of the waiters on
     -- the current agent.  They will call down to their waiters, and so on,
     -- until they either go as deep as they can, or until there are no 
     -- more waiters down that path.  
     -- Then the calls return and we proceed down the path for the next waiter
     -- and so on.

     IF level < 14 THEN
	INSERT INTO db2perf_locktree VALUES 
	   ( current_line,' ' );
	SET current_line=current_line+1;

	-- Recurse down this path ...
	EXECUTE s1 INTO current_line USING waiter, level, starting_line, current_line ;

        SET starting_line = starting_line + 8;

     ELSE
        -----------------------------------------------------------------------
	-- Can't go any deeper

	INSERT INTO db2perf_locktree VALUES 
	   ( current_line,substr(blanks,1,8*level) || '   |' );
	INSERT INTO db2perf_locktree VALUES 
	   ( current_line,substr(blanks,1,8*level) || '   +------- ?  (reached maximum depth)' );

	SET current_line=current_line+2;

	INSERT INTO db2perf_locktree VALUES 
	   ( current_line,' ' );
	SET current_line=current_line+1;

        SET starting_line = starting_line + 10;

     END IF;


     -----------------------------------------------------------------------
     -- Get the next waiter on self, and loop back again.
     SET at_end = 0;
     FETCH waiters INTO 
	waiter,lock_mode,lock_object_type,lock_wait_time,lock_escallation,table_name;

   END WHILE;

   CLOSE waiters;

END @


CALL db2perf_quiet_drop('PROCEDURE db2perf_locktree()')@


-----------------------------------------------------------------------
-----------------------------------------------------------------------
-- db2perf_locktree
-- 
-- User-called version.  No arguments.  Sets things up and calls
-- the recursive one above to do the real work.
-----------------------------------------------------------------------
-----------------------------------------------------------------------

CREATE PROCEDURE db2perf_locktree( )
DYNAMIC RESULT SETS 1
LANGUAGE SQL
MODIFIES SQL DATA
BEGIN
   DECLARE at_end             INTEGER;
   DECLARE current_line       INTEGER;
   DECLARE parent_line        INTEGER;
   DECLARE root_holder        INTEGER;
   DECLARE lock_count         INTEGER DEFAULT 0;
   DECLARE snapshot_timestamp TIMESTAMP;


   -----------------------------------------------------------------------
   -- Cursor to get the 'roots' of the current lock waits.   Those agents
   -- which are being waited on, but are not waiting on anyone else.
   DECLARE roots CURSOR FOR
     SELECT 
	DISTINCT h.agent_id_holding_lk 
	FROM db2perf_lockwait h 
	WHERE NOT EXISTS 
		(SELECT w.agent_id 
		 FROM db2perf_lockwait w 
		 WHERE w.agent_id = h.agent_id_holding_lk);

   DECLARE locktree_cursor CURSOR WITH RETURN TO CALLER FOR
     SELECT branch
       FROM db2perf_locktree
       ORDER BY line;


   DECLARE CONTINUE HANDLER FOR NOT FOUND
     SET at_end = 1; 

   SET current_line = 0;
   SET parent_line = 0;
   SET root_holder = 0;

   -----------------------------------------------------------------------
   -- Clean out previous data, and get new snapshots.
   DELETE FROM db2perf_lockwait;
   DELETE FROM db2perf_appl_info;
   DELETE FROM db2perf_locktree;

   INSERT INTO db2perf_lockwait 
	SELECT * FROM TABLE( snapshot_lockwait( CAST( NULL AS varchar(256)),-1 ) ) AS t;

   INSERT INTO db2perf_appl_info 
	SELECT * FROM TABLE( snapshot_appl_info( CAST( NULL AS varchar(256)),-1 ) ) AS t;

   SET at_end = 0;


   OPEN roots;
   FETCH roots INTO root_holder;

   WHILE at_end = 0 DO

      -----------------------------------------------------------------------
      -- For each of the 'root' applications, call the recursive db2perf_locktree
      -- which will write the graphical tree into the report table.

      CALL db2perf_locktree( root_holder, 0, parent_line, current_line );

      SET parent_line = current_line;

      FETCH roots INTO root_holder;

   END WHILE;

   CLOSE roots;


   -----------------------------------------------------------------------
   -- Report statistics on how many lockewaits were found.

   IF current_line = 0 THEN
      INSERT INTO db2perf_locktree VALUES ( 0,'No lockwaits found' );
   ELSE
      SELECT count(*) 
	INTO lock_count 
	FROM db2perf_locktree 
	WHERE branch LIKE '%Waiter appl handle%';

      SELECT min(snapshot_timestamp) 
	INTO snapshot_timestamp
	FROM db2perf_lockwait;

      INSERT INTO db2perf_locktree 
	VALUES ( 0, rtrim(char(lock_count)) || ' lockwaits found at time ' || char(snapshot_timestamp) );
   END IF;

   DELETE FROM db2perf_lockwait;
   DELETE FROM db2perf_appl_info;

   OPEN locktree_cursor;
END@
