-----------------------------------------------------------------------
--(c) Copyright IBM Corporation 2006  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_utils.db2                                                 --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Utility routines to translate numeric codes in snapshot table     --
-- functions into words.    Also provides a function to do a         --
-- 'quiet' SQL DROP (i.e., don't report NOT FOUND errors.)           --
--                                                                   --
-----------------------------------------------------------------------


----------------------------------------------------------------------
-- db2perf_quiet_drop : performs a SQL DROP statement, 
--		        suppressing the 'not found' error if necessary
-- db2perf_lkobj2str  : translates a  LOCK OBJECT         code to a string
-- db2perf_lkmode2str : translates a  LOCK MODE           code to a string
-- db2perf_lkstat2str : translates a  LOCK STATE          code to a string
-- db2perf_tabtyp2str : translates a  TABLE TYPE          code to a string
-- db2perf_apstat2str : translates an APPLICATION STATE   code to a string
-- db2perf_tbstyp2str : translates a  TABLESPACE TYPE     code to a string
-- db2perf_tbscon2str : translates a  TABLESPACE CONTENT  code to a string
-- db2perf_op2str     : translates a  STATEMENT OPERATION code to a string
-- db2perf_type2str   : translates a  STATEMENT TYPE      code to a string
-- db2perf_crmsg      : creates the db2perf_msg table used by db2perf_bp
-- db2perf_trimlzero  : replaces leading zeroes in a numeric string with blanks
----------------------------------------------------------------------



----------------------------------------------------------------------
-- db2perf_quiet_drop
--
-- Executes an SQL 'DROP', suppressing the 'not found' error if it occurs.
-- Useful for avoiding spurious errors when doing DROPs in setup scripts.
-- Prepends the 'DROP' verb to the statement, so the statement passed in
-- is something like 
--     CALL db2perf_quiet_drop( 'table foo' )
-- which executes "DROP table foo"
----------------------------------------------------------------------

DROP PROCEDURE db2perf_quiet_drop( VARCHAR(1000) )@

ECHO Creating db2perf_quiet_drop@

CREATE PROCEDURE db2perf_quiet_drop( IN statement VARCHAR(1000) )
LANGUAGE SQL
BEGIN
   DECLARE SQLSTATE CHAR(5);
   DECLARE NotThere    CONDITION FOR SQLSTATE '42704';
   DECLARE NotThereSig CONDITION FOR SQLSTATE '42883';

   DECLARE EXIT HANDLER FOR NotThere, NotThereSig
      SET SQLSTATE = '     ';

   SET statement = 'DROP ' || statement;
   EXECUTE IMMEDIATE statement;
END@



----------------------------------------------------------------------
-- db2perf_crmsg
--
-- Creates a message table used for db2perf_bp.   Suppresses the 'already
-- exists' error if necessary.
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'procedure db2perf_crmsg' )@

ECHO Creating db2perf_crmsg@

CREATE PROCEDURE db2perf_crmsg
LANGUAGE SQL
BEGIN

   DECLARE statement VARCHAR(128);
   DECLARE SQLSTATE CHAR(5);
   DECLARE AlreadyThere CONDITION FOR SQLSTATE '42710';

   DECLARE EXIT HANDLER FOR AlreadyThere
      SET SQLSTATE = '     ';

   SET statement = 'CREATE TABLE db2perf_msg ( ts timestamp, severity char(8), metric char(32), value char(16), Comments char(40) )';
   EXECUTE IMMEDIATE statement;

END @


----------------------------------------------------------------------
-- db2perf_lkobj2str - translates a LOCK OBJECT to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_lkobj2str' )@

ECHO Creating db2perf_lkobj2str@

CREATE FUNCTION db2perf_lkobj2str(lock_object_type bigint)
RETURNS char(18)
DETERMINISTIC 
NO EXTERNAL ACTION 
CONTAINS SQL
BEGIN ATOMIC
  DECLARE retstr CHAR(18);

  IF lock_object_type IS NULL THEN
    RETURN NULL;
  END IF;
  SET retstr =
        CASE lock_object_type
                WHEN 1  THEN 'Table'
                WHEN 2  THEN 'Row'
                WHEN 3  THEN 'Internal lock'
                WHEN 4  THEN 'Tablespace'
                WHEN 5  THEN 'EOT Lock'
                WHEN 6  THEN 'Keyvalue Lock'
                WHEN 7  THEN 'Sysboot Lock'
                WHEN 8  THEN 'Internal P Lock'
                WHEN 9  THEN 'Internal V Lock'
                WHEN 10 THEN 'Internal S Lock'
                WHEN 11 THEN 'Internal J Lock'
                WHEN 12 THEN 'Internal L Lock'
                WHEN 13 THEN 'Internal C Lock'
                WHEN 14 THEN 'Internal B Lock'
                WHEN 15 THEN 'Internal O Lock'
                WHEN 16 THEN 'Internal T Lock'
                WHEN 17 THEN 'Internal Q Lock'
                WHEN 18 THEN 'Inplace Reorg Lock'
                WHEN 19 THEN 'Block Lock'
                ELSE NULL
        END;
  RETURN retstr;
END@


----------------------------------------------------------------------
-- db2perf_lkmode2str - translates a LOCK MODE code to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_lkmode2str' )@

ECHO Creating db2perf_lkmode2str@

CREATE FUNCTION db2perf_lkmode2str(lock_mode bigint)
RETURNS char(35)
DETERMINISTIC 
NO EXTERNAL ACTION 
CONTAINS SQL
BEGIN ATOMIC
  DECLARE retstr CHAR(35);

  IF lock_mode IS NULL THEN
    RETURN NULL;
  END IF;

  SET retstr =
        CASE lock_mode
                WHEN  1 THEN 'Intention Share Lock'
                WHEN  2 THEN 'Intention Exclusive Lock'
                WHEN  3 THEN 'Share Lock'
                WHEN  4 THEN 'Share with Intention Exclusive Lock'
                WHEN  5 THEN 'Exclusive Lock'
                WHEN  6 THEN 'Intent None (For Dirty Read)'
                WHEN  7 THEN 'Super Exclusive Lock'
                WHEN  8 THEN 'Update Lock'
                WHEN  9 THEN 'Next-key Share Lock'
                WHEN 10 THEN 'Next-key Exclusive Lock'
                WHEN 11 THEN 'Weak Exclusive Lock'
                WHEN 12 THEN 'Next-key Weak Exclusive Lock'
                ELSE NULL
	END;
  RETURN retstr;
END@


----------------------------------------------------------------------
-- db2perf_lkstat2str - translates a LOCK STATE code to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_lkstat2str' )@

ECHO Creating db2perf_lkstat2str@

CREATE FUNCTION db2perf_lkstat2str(lock_status bigint)
RETURNS char(10)
DETERMINISTIC 
NO EXTERNAL ACTION 
CONTAINS SQL
BEGIN ATOMIC
  DECLARE retstr CHAR(10);

  IF lock_status IS NULL THEN
    RETURN NULL;
  END IF;

  SET retstr =
        CASE lock_status
                WHEN  1 THEN 'Granted'
                WHEN  2 THEN 'Converting'
		ELSE NULL
	END;

  RETURN retstr;
END@


----------------------------------------------------------------------
-- db2perf_tabtyp2str - translates a TABLE TYPE code to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_tabtyp2str' )@

ECHO Creating db2perf_tabtyp2str@

CREATE FUNCTION db2perf_tabtyp2str(table_type bigint)
RETURNS char(20)
DETERMINISTIC 
NO EXTERNAL ACTION 
CONTAINS SQL
BEGIN ATOMIC
  DECLARE retstr CHAR(20);

  IF table_type IS NULL THEN
    RETURN NULL;
  END IF;

  SET retstr =
        CASE table_type
		WHEN  1 THEN 'User table'
		WHEN  2 THEN 'Dropped user table'
		WHEN  3 THEN 'Temporary table'
		WHEN  4 THEN 'System Catalog table'
		WHEN  5 THEN 'Reorg table'
		ELSE NULL
	END;

  RETURN retstr;
END@



----------------------------------------------------------------------
-- db2perf_apstat2str - translates an APPLICATION STATUS code to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_apstat2str' )@

ECHO Creating db2perf_apstat2str@

CREATE FUNCTION db2perf_apstat2str(app_status bigint)
RETURNS char(36)
DETERMINISTIC 
NO EXTERNAL ACTION 
CONTAINS SQL
BEGIN ATOMIC
  DECLARE retstr CHAR(36);

  IF app_status IS NULL THEN
    RETURN NULL;
  END IF;

  SET retstr =
        CASE app_status
		WHEN  1 THEN 'Connect pending'
		WHEN  2 THEN 'Connect completed'
		WHEN  3 THEN 'UOW executing'
		WHEN  4 THEN 'UOW waiting'
		WHEN  5 THEN 'Lock wait'
		WHEN  6 THEN 'Commit active'
		WHEN  7 THEN 'Rollback active'
		WHEN  8 THEN 'Recompiling a plan'
		WHEN  9 THEN 'Compiling a SQL statement'
		WHEN 10 THEN 'Request interrupted'
		WHEN 11 THEN 'Disconnect pending'
		WHEN 12 THEN 'Prepared transaction'
		WHEN 13 THEN 'Heuristically committed'
		WHEN 14 THEN 'Heuristically rolled back'
		WHEN 15 THEN 'Transaction ended'
		WHEN 16 THEN 'Creating Database'
		WHEN 17 THEN 'Restarting a Database'
		WHEN 18 THEN 'Restoring a Database'
		WHEN 19 THEN 'Performing a Backup'
		WHEN 20 THEN 'Performing a fast load'
		WHEN 21 THEN 'Performing a fast unload'
		WHEN 22 THEN 'Wait to disable tablespace'
		WHEN 23 THEN 'Quiescing a tablespace'
		WHEN 24 THEN 'Waiting for remote node'
		WHEN 25 THEN 'Pending results from remote request'
		WHEN 26 THEN 'App has been decoupled from coord'
		WHEN 27 THEN 'Rollback to savepoint'

		ELSE NULL
	END;

  RETURN retstr;
END@



----------------------------------------------------------------------
-- db2perf_tbstyp2str - translates a TABLE TYPE code to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_tbstyp2str' )@

ECHO Creating db2perf_tbstyp2str@

CREATE FUNCTION db2perf_tbstyp2str(tablespace_type bigint)
RETURNS char(3)
DETERMINISTIC
NO EXTERNAL ACTION
CONTAINS SQL
BEGIN ATOMIC
  DECLARE retstr CHAR(3);

  IF tablespace_type IS NULL THEN
    RETURN NULL;
  END IF;

  SET retstr =
        CASE tablespace_type
                WHEN  0 THEN 'DMS'
                WHEN  1 THEN 'SMS'
                ELSE NULL
        END;
  RETURN retstr;
END@




----------------------------------------------------------------------
-- db2perf_tbscon2str - translates a TABLESPACE CONTENT code to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_tbscon2str' )@

ECHO Creating db2perf_tbscon2str@

CREATE FUNCTION db2perf_tbscon2str(tablespace_content bigint)
RETURNS char(16)
DETERMINISTIC
NO EXTERNAL ACTION
CONTAINS SQL
BEGIN ATOMIC
  DECLARE retstr CHAR(16);

  IF tablespace_content IS NULL THEN
    RETURN NULL;
  END IF;

  SET retstr =
        CASE tablespace_content
                WHEN  0 THEN 'Any'
                WHEN  1 THEN 'Long'
                WHEN  2 THEN 'System temporary'
                WHEN  3 THEN 'User temporary'
                ELSE NULL
        END;
  RETURN retstr;
END@




----------------------------------------------------------------------
-- db2perf_op2str - translates a STATEMENT OPERATION code to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_op2str' )@

ECHO Creating db2perf_op2str@

CREATE FUNCTION db2perf_op2str(op bigint)
RETURNS char(8)
DETERMINISTIC 
NO EXTERNAL ACTION 
CONTAINS SQL
BEGIN ATOMIC
  DECLARE retstr CHAR(8);
  IF op IS NULL THEN
  RETURN NULL;
  END IF;
  set retstr =
  case op
    when 1  then 'PREPARE'
    when 2  then 'EXECUTE'
    when 3  then 'EXECIMM'
    when 4  then 'OPEN'
    when 5  then 'FETCH'
    when 6  then 'CLOSE'
    when 7  then 'DESCRIBE'
    when 8  then 'COMMIT'
    when 9  then 'ROLLBACK'
    when 10 then 'FREELOC'
    when 11 then 'PREPCOMM'
    when 12 then 'CALL'
    when 15 then 'SELECT'
    when 16 then 'PREPOPEN'
    when 17 then 'PREPEXEC'
    when 18 then 'COMPILE'
    when 19 then 'SET'
    when 20 then 'RUNSTATS'
    when 21 then 'REORG'
    when 22 then 'REBIND'
    when 23 then 'REDIST'
    when 24 then 'GETTA'
    when 25 then 'GETAA'
    else
      NULL
  end;
  RETURN retstr;
END@


----------------------------------------------------------------------
-- db2perf_type2str - translates a STATEMENT TYPE code to a string 
-- Based on constants in sqllib/include/sqlmon.h
----------------------------------------------------------------------

CALL db2perf_quiet_drop( 'function db2perf_type2str' )@

ECHO Creating db2perf_type2str@

CREATE FUNCTION db2perf_type2str(op bigint)
RETURNS CHAR(8)
DETERMINISTIC 
NO EXTERNAL ACTION 
CONTAINS SQL
BEGIN ATOMIC

  DECLARE retstr CHAR(8);

  IF op IS NULL THEN
    RETURN NULL;
  END IF;

  SET retstr =
  case op
    when 1 then 'STATIC'
    when 2 then 'DYNAMIC'
    when 3 then 'NON STMT'
    else
      NULL
  end;

  RETURN retstr;
END@



----------------------------------------------------------------------
-- db2perf_trimlzero
-- Trims leading zeroes from an input numeric statement
-- e.g. '0001.20' becomes '   1.20'
--
-- We use this because the CHAR() function on a DECIMAL uses leading
-- zeroes instead of leading blanks, and I think they look prettier with blanks.  :-)
----------------------------------------------------------------------

CALL db2perf_quiet_drop('FUNCTION db2perf_trimlzero')@

ECHO Creating db2perf_trimlzero@

CREATE FUNCTION db2perf_trimlzero( string VARCHAR(20) )
  RETURNS VARCHAR(20)
  DETERMINISTIC NO EXTERNAL ACTION CONTAINS SQL
  BEGIN ATOMIC

   DECLARE i           INTEGER;
   DECLARE blanks      CHAR(20) DEFAULT ' ';
   DECLARE sign        VARCHAR(1) DEFAULT '';
   DECLARE len         INTEGER;
   DECLARE work_string VARCHAR(20);

   IF string IS NULL THEN
     RETURN NULL;
   END IF;

   SET len = length(string);

   -- Set a flag to mark that we saw a sign
   SET i = 0;
   IF substr(string,1,1) = '-' THEN
     SET sign = '-';
     SET i=i+1;
   END IF;

   -- Scan the string, counting leading zeroes.   Look ahead for the decimal point too.
   WHILE i < length(string)
     AND substr(string,i+1,1) =  '0'
     AND substr(string,i+2,1) <> '.'
     AND substr(string,i+2,1) <> ' ' DO
     SET i = i+1;
   END WHILE;

   -- Now rebuild the string, with a sign as necessary
   IF sign <> '' THEN
     SET work_string = 
		substr(blanks,1, (CASE WHEN i>1 THEN i-2 ELSE 0 END)) 		-- leading blanks
	     || sign 								-- any sign we found
	     || right(string,length(string)-i) 					-- the meat of the number
	     || ' ';
   ELSE
     SET work_string = 
		substr(blanks,1,i) 
	     || right(string,length(string)-i);
   END IF;

   RETURN substr(blanks,1,len-length(rtrim(work_string))) || rtrim(work_string);
END@


