-----------------------------------------------------------------------
--(c) Copyright IBM Corporation 2006  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_snapdiff.db2                                              --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Compares snapshots taken across & between intervals, and reports  --
-- the largest differences.                                          --
--                                                                   --
--                                                                   --
-- 060505   Restrict to first row from db & dbm                      --
--                                                                   --
-----------------------------------------------------------------------

----------------------------------------------------------------------
----------------------------------------------------------------------
-- Create the report table where we'll write results, and
-- tables to hold snapshots (directly from views on the 
-- snapshot table functions.
--
-- Use db2perf_quiet_drop() from db2perf_utils because
-- it doesn't complain when the item being dropped is already
-- 'not there'.
----------------------------------------------------------------------
----------------------------------------------------------------------

CALL db2perf_quiet_drop('TABLE db2perf_snapdiff_report')@
CREATE TABLE db2perf_snapdiff_report( line INTEGER, message VARCHAR(120) )@

CALL db2perf_quiet_drop('VIEW db2perf_snapdbm_view')@
CALL db2perf_quiet_drop('TABLE db2perf_snapdbm')@
CREATE VIEW db2perf_snapdbm_view AS SELECT * FROM table(snapshot_dbm(-1)) as t@
CREATE TABLE db2perf_snapdbm LIKE db2perf_snapdbm_view@
CALL db2perf_quiet_drop('VIEW db2perf_snapdbm_view')@

CALL db2perf_quiet_drop('VIEW db2perf_snapdb_view')@
CALL db2perf_quiet_drop('TABLE db2perf_snapdb')@
CREATE VIEW db2perf_snapdb_view AS SELECT * FROM table(snapshot_database(CAST (NULL as varchar(256)),-1)) as t@
CREATE TABLE db2perf_snapdb LIKE db2perf_snapdb_view@
CALL db2perf_quiet_drop('VIEW db2perf_snapdb_view')@

CALL db2perf_quiet_drop('VIEW db2perf_snaptbs_view')@
CALL db2perf_quiet_drop('TABLE db2perf_snaptbs')@
CREATE VIEW db2perf_snaptbs_view AS SELECT * FROM table(snapshot_tbs(CAST (NULL as varchar(256)),-1)) as t@
CREATE TABLE db2perf_snaptbs LIKE db2perf_snaptbs_view@
CALL db2perf_quiet_drop('VIEW db2perf_snaptbs_view')@

CALL db2perf_quiet_drop('VIEW db2perf_snaptb_view')@
CALL db2perf_quiet_drop('TABLE db2perf_snaptb')@
CREATE VIEW db2perf_snaptb_view AS SELECT * FROM table(snapshot_table(CAST (NULL as varchar(256)),-1)) as t@
CREATE TABLE db2perf_snaptb LIKE db2perf_snaptb_view@
CALL db2perf_quiet_drop('VIEW db2perf_snaptb_view')@

CALL db2perf_quiet_drop('VIEW db2perf_snapbp_view')@
CALL db2perf_quiet_drop('TABLE db2perf_snapbp')@
CREATE VIEW db2perf_snapbp_view AS SELECT * FROM table(snapshot_bp(CAST (NULL as varchar(256)),-1)) as t@
CREATE TABLE db2perf_snapbp LIKE db2perf_snapbp_view@
CALL db2perf_quiet_drop('VIEW db2perf_snapbp_view')@

----------------------------------------------------------------------
-- The TOC table is the 'table of contents' used to map interval numbers to the timestamp
-- values for the start & stop.
----------------------------------------------------------------------

CALL db2perf_quiet_drop('TABLE db2perf_snapdiff_toc')@
CREATE TABLE db2perf_snapdiff_toc ( interval integer, start timestamp, stop timestamp, dbm char, db char, tbs char, tb char, bp char )@


----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_rowdiff()
--
-- This is the heart of the compare function.   It picks up 4 rows
-- from the table 'snap_table_name', each with one of the start/stop
-- timestamps for interval 1 or 2.
-- It proceeds through all numeric columns in that row, first calculating
-- the change in the numeric within the two intervals, and then the
-- difference between the intervals themselves.
----------------------------------------------------------------------
----------------------------------------------------------------------

CALL db2perf_quiet_drop('PROCEDURE db2perf_rowdiff')@

ECHO Creating stored procedure db2perf_rowdiff@
ECHO @

CREATE PROCEDURE db2perf_rowdiff( 
	IN snap_table_name VARCHAR(128),	-- table we're looking at - usually blank/null to mean all tables
	IN ts_colname VARCHAR(128),		-- name of the column which contains timestamps
	IN ts_start_1 TIMESTAMP,		-- ts value for start of interval 1
	IN ts_stop_1 TIMESTAMP,			-- ts value for stop of interval 1
	IN ts_start_2 TIMESTAMP,		-- ts value for start of interval 2
	IN ts_stop_2 TIMESTAMP,			-- ts value for stop of interval 2
	IN qual_col_expr VARCHAR(256),		-- expression to choose pairs of rows within an interval
	IN k_commit_sql_stmts_1 DOUBLE,		-- normalizing factor - # of transactions in interval 1
	IN k_commit_sql_stmts_2 DOUBLE,		-- normalizing factor - # of transactions in interval 2
	IN threshold_pct INTEGER,		-- what's the lowest % change we care about
	IN ignore_list VARCHAR(256),		-- an IN list of column names we don't care about for this table
	INOUT line INTEGER  )			-- the # of the line in the report file we're writing
LANGUAGE SQL
BEGIN
   DECLARE SQLSTATE CHAR(5);

   DECLARE colname  VARCHAR(128);
   DECLARE typename VARCHAR(128);
   
   DECLARE col_stmt VARCHAR(2048);
   DECLARE row_stmt VARCHAR(8192);

   DECLARE at_end INT DEFAULT 0;

   DECLARE ts          TIMESTAMP;
   DECLARE value_1     DOUBLE;
   DECLARE value_2     DOUBLE;
   DECLARE start_value DOUBLE;
   DECLARE stop_value  DOUBLE;
   DECLARE diff        DOUBLE DEFAULT 0;

   DECLARE i INTEGER;
   DECLARE blanks CHAR(20) DEFAULT ' ';

   DECLARE print_colname  CHAR(40);
   DECLARE print_value_1  CHAR(16);
   DECLARE print_value_2  CHAR(16);
   DECLARE print_diff     CHAR(16);

   DECLARE start_line     INTEGER;
  
   DECLARE col_curs CURSOR FOR col_stmt;
   DECLARE row_curs CURSOR FOR row_stmt;

   DECLARE CONTINUE HANDLER FOR NOT FOUND
     SET at_end = 1; 

   SET start_line = line;


   ----------------------------------------------------------------------
   -- Build the statement to pick up numeric columns from the named table.
   SET col_stmt = 
   	'SELECT colname,typename '
	|| 'FROM syscat.columns '
	|| 'WHERE '
	|| '   tabschema = CURRENT SCHEMA '
	|| '   AND tabname = ''' || snap_table_name || ''''
	|| '   AND typename in '
	|| '	(''SMALLINT'',''INTEGER'',''DECIMAL'',''DOUBLE'',''BIGINT'') '
	|| CASE WHEN ignore_list = '' THEN '' ELSE ' AND colname not in ' || ignore_list END
	|| ' ORDER BY colno';

   PREPARE col_stmt from col_stmt;

   
   ----------------------------------------------------------------------
   -- Insert a line into the report table indicating the start & stop timestamps of the 2 intervals.
   INSERT into db2perf_snapdiff_report VALUES 
		(line,
		  '                                                  ' 
			|| char( date(ts_start_1) ) || '       ' || char( date(ts_start_2) )  ),
		  (line+1,
		  '                                                    ' 
			|| char( time(ts_start_1) ) || '         ' || char( time(ts_start_2) ) ),
		  (line+2, 
		  '                                                  ' 
			|| '     to               to' ),
		  (line+3,
		  '                                                  ' 
			|| char( date(ts_stop_1) ) || '       ' || char( date(ts_stop_2) )  ),
		  (line+4,
		  '                                                    ' 
			|| char( time(ts_stop_1) ) || '         ' || char( time(ts_stop_2) ) );
   SET line = line+5;

   INSERT into db2perf_snapdiff_report VALUES 
		  (line,
		  '                                                  ----------       ----------' );
   SET line = line+1;

   SET at_end = 0;


   ----------------------------------------------------------------------
   -- If we're normalizing our data to the number of transactions in the interval, then report
   -- what that interval is.
   ----------------------------------------------------------------------
   IF k_commit_sql_stmts_1 <> 1 OR k_commit_sql_stmts_2 <> 1 THEN
	  SET diff       = 100*(k_commit_sql_stmts_2-k_commit_sql_stmts_1)/k_commit_sql_stmts_1;
          SET print_colname = 'Normalizing to 1000 Tx per Interval';
	  SET print_value_1 = db2perf_trimlzero( char(CAST(k_commit_sql_stmts_1 as DECIMAL(12,1))));
	  SET print_value_2 = db2perf_trimlzero( char(CAST(k_commit_sql_stmts_2 as DECIMAL(12,1))));
	  SET print_diff    = db2perf_trimlzero( char(cast(diff as DECIMAL(8,1)))) || '%';
	  INSERT into db2perf_snapdiff_report VALUES ( 
		  line, '      ' ||
		  print_colname  || ' ' ||
		  print_value_1  || ' ' ||
		  print_value_2  || ' ' ||
		  print_diff );
     SET line = line+1;
   END IF;


   ----------------------------------------------------------------------
   -- Open a cursor against syscat.columns and go through all the numeric columns for this snapshot table
   OPEN col_curs;
   FETCH col_curs INTO colname,typename;
   WHILE at_end = 0 DO

     ----------------------------------------------------------------------
     -- For this column, get the 4 values (start/stop interval 1, start/stop interval 2) we care about,
     -- cast to double.
     -- The ones we care about have the timestamps that were passed in, plus the 'qualifying expression'
     -- which might be something like TABNAME = 'FOO', so that we just pick up rows from the table
     -- snapshot which are about table FOO.
     -- We get them in ascending order of timestamp.
     -- Note, this means we do a lot of dynamic SQL in this script!

     SET row_stmt = 
     	   'SELECT '  || ts_colname || ',cast(' || colname || ' AS double) ' || 
	   ' FROM '      || CURRENT SCHEMA || '.' || snap_table_name || 
	   ' WHERE '     || ts_colname || ' IN ( ''' || char(ts_start_1)  || ''',''' || char(ts_stop_1)   || ''', ' ||
	                                        '''' || char(ts_start_2)  || ''',''' || char(ts_stop_2)   || ''') ' ||
	   CASE WHEN qual_col_expr = '' THEN '' ELSE 'AND (' || qual_col_expr || ')' END ||
	   ' ORDER BY '  || ts_colname ;

     PREPARE row_stmt FROM row_stmt;
     OPEN row_curs;

     SET start_value = 0;
     FETCH row_curs INTO ts,start_value;

     IF at_end = 0 THEN
	SET print_colname = substr(colname,1,40);
	SET diff = 0;

        ----------------------------------------------------------------------
        -- We make sure as we go that we're getting the rows that we think we are: interval 1 start, then 
	-- stop, then interval 2 start, then stop.
	IF ts = ts_start_1 THEN

	  SET stop_value = 0;
	  FETCH row_curs INTO ts,stop_value;

	  IF at_end = 0 AND ts = ts_stop_1 THEN
	    
            ----------------------------------------------------------------------
	    -- Calculate the difference in interval 1, normalizing to # of Tx if requested.
	    -- Call db2perf_trimlzero to replace leading zeros in the output of char(<decimal number>)
	    -- with blanks (looks nicer!)

	    SET value_1  = (stop_value - start_value) / k_commit_sql_stmts_1;
	    SET print_value_1 = db2perf_trimlzero( char(CAST(value_1 as DECIMAL(12,1))) );

	    SET print_value_2 = '            -';
	    SET print_diff = ' ';

	    SET start_value = 0;
	    FETCH row_curs INTO ts,start_value;
	    IF at_end = 0 THEN
	      IF ts = ts_start_2 THEN
		SET stop_value = 0;
		FETCH row_curs INTO ts,stop_value;

		IF at_end = 0 AND ts = ts_stop_2 THEN
		  SET value_2  = (stop_value - start_value) / k_commit_sql_stmts_2;
		  SET print_value_2 = db2perf_trimlzero( char(CAST(value_2 as DECIMAL(12,1))) );

                  ----------------------------------------------------------------------
		  -- We now have a value for interval 1 & one for interval 2
		  -- If there's no difference, then print blanks, else
		  -- figure out the % difference.

		  IF value_1 = value_2 THEN
		    SET print_diff      = ' ';
		  ELSE 
		    IF value_1 = 0.0 THEN
		      -- value1 is zero, no percents don't make any sense
		      SET print_diff    = '     Inf';
		    ELSE
		      SET diff       = 100*(value_2-value_1)/value_1;
		      SET print_diff = db2perf_trimlzero( char(cast(diff as DECIMAL(8,1)))) || '%';
		    END IF; 
		  END IF;

		END IF;
	      END IF;
	    END IF;
	  END IF;

	END IF;

        ----------------------------------------------------------------------
	-- Should we print this?   Is the difference over the threshold, and are the numbers big enough
	-- to worry about?  (Assume anything smaller than 5 is generally too small.)
	-- This is a big assumption but it seems to be generally true
	-- If the % difference is big enough, print some stars to attract attention ...

	IF  abs(diff) >= threshold_pct                    -- It's above our threshold percent
	AND ( threshold_pct = 0 OR   			  -- Only worry about real value thresholds if we have
	      ( abs(value_1) > 5  OR  abs(value_2) > 5) ) --   a % threshold too
	AND print_value_1 IS NOT NULL 
	AND print_value_2 IS NOT NULL THEN
	  INSERT into db2perf_snapdiff_report VALUES ( 
		  line,
		  CASE 
			WHEN abs(diff) > 100 THEN '***   ' -- Obviously you could change these stars to match
			WHEN abs(diff) >  33 THEN '**    ' --    the thresholds you find meaningful
			WHEN abs(diff) >  10 THEN '*     ' 
			ELSE '      ' END ||
		  print_colname || ' ' ||
		  print_value_1 || ' ' ||
		  print_value_2 || ' ' ||
		  print_diff );

	  SET line = line+1;
	END IF;
     END IF;

     CLOSE row_curs;

     -- Done with this column - go to the next one.

     SET at_end = 0;
     FETCH col_curs INTO colname,typename;

   END WHILE;
     
   ----------------------------------------------------------------------
   -- If we get to the end and nothing was printed, then go back & remove the header we put at the beginning
   IF line = start_line + 6 THEN
     SET line=start_line;
     DELETE from db2perf_snapdiff_report where line >= start_line;
   END IF;

END@



----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_get1kTxCount
-- Determines how many 'kilotransactions' were run in the two intervals.
-- We use these to normalize things to 'change in metric per 1000 Tx'.
----------------------------------------------------------------------
----------------------------------------------------------------------

CALL db2perf_quiet_drop('PROCEDURE db2perf_getTxCount')@

ECHO Creating stored procedure db2perf_get1kTxCount@
ECHO @

CREATE PROCEDURE db2perf_getTxCount( 
	IN ts_start_1  TIMESTAMP,
	IN ts_stop_1 TIMESTAMP,
	IN ts_start_2  TIMESTAMP,
	IN ts_stop_2 TIMESTAMP,
	INOUT line    INTEGER,
	OUT k_commit_sql_stmts_1 DOUBLE,
	OUT k_commit_sql_stmts_2 DOUBLE )
LANGUAGE SQL
BEGIN
   DECLARE at_end INT DEFAULT 0;

   DECLARE CONTINUE HANDLER FOR NOT FOUND
     SET at_end = 1; 

   ----------------------------------------------------------------------
   -- Pick up the transaction numbers from db2perf_snapdb, which contains the database snapshot data
   -- for the intervals in question.

   SELECT cast(abs(stop.commit_sql_stmts - 
                   ( SELECT max(start.commit_sql_stmts) 
         	     FROM db2perf_snapdb start WHERE snapshot_timestamp = ts_start_1
		   ) ) as double) / 1000
	INTO k_commit_sql_stmts_1
	FROM db2perf_snapdb stop
	WHERE snapshot_timestamp = ts_stop_1;

   SELECT cast(abs(stop.commit_sql_stmts - 
                   ( SELECT max(start.commit_sql_stmts) 
		     FROM db2perf_snapdb start WHERE snapshot_timestamp = ts_start_2
		   ) ) as double) / 1000
	INTO k_commit_sql_stmts_2
	FROM db2perf_snapdb stop
	WHERE snapshot_timestamp = ts_stop_2;

   -- Try to avoid any divide-by-zero problems
   IF k_commit_sql_stmts_1 = 0 THEN
     SET k_commit_sql_stmts_1 = 1;
   END IF;
   IF k_commit_sql_stmts_2 = 0 THEN
     SET k_commit_sql_stmts_2 = 1;
   END IF;

   -- Report if there are no transactions here.
   IF at_end <> 0 THEN
     INSERT INTO db2perf_snapdiff_report 
	VALUES ( line, 'No transaction rate information available; results not normalized' );
     SET line=line+1;
   END IF;
END@




----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_snapdiff
--
-- This is the main routine.  Depending on what operation is passed in,
-- this routine collects, diffs or deletes snapshot information
----------------------------------------------------------------------
----------------------------------------------------------------------


CALL db2perf_quiet_drop('PROCEDURE db2perf_snapdiff( VARCHAR(20), VARCHAR(128), INTEGER, INTEGER, CHAR(1), INTEGER )')@

ECHO Creating stored procedure db2perf_snapdiff@
ECHO @

CREATE PROCEDURE db2perf_snapdiff( 
	IN operation VARCHAR(20),		-- operation = start/stop/diff/list/delete
	IN snap_table_name VARCHAR(128),	-- which snapshot table?   Defaults to all
	IN interval_1 integer,			-- 'before' interval number
	IN interval_2 integer,			-- 'after' interval number
	IN normalize_to_Tx CHAR(1),		-- normalize to k Transactions - Y or N
	IN threshold_pct INTEGER )		-- pct below which we don't display differences
LANGUAGE SQL
BEGIN
   DECLARE SQLSTATE CHAR(5);

   DECLARE new_ts TIMESTAMP;
   DECLARE old_ts TIMESTAMP;

   DECLARE ts_start_1  		TIMESTAMP;
   DECLARE ts_stop_1 		TIMESTAMP;
   DECLARE ts_start_2  		TIMESTAMP;
   DECLARE ts_stop_2 		TIMESTAMP;
   DECLARE get_interval_1 	INTEGER DEFAULT 0;
   DECLARE get_interval_2 	INTEGER DEFAULT 0;
   DECLARE k_commit_sql_stmts_1 DOUBLE DEFAULT 1;
   DECLARE k_commit_sql_stmts_2 DOUBLE DEFAULT 1;

   DECLARE current_interval INTEGER;

   DECLARE line INTEGER DEFAULT 0;

   DECLARE switch_total INTEGER DEFAULT 0;

   DECLARE tablespace_name CHAR(128);
   DECLARE table_name CHAR(128);
   DECLARE bufferpool VARCHAR(128);
   DECLARE message VARCHAR(100);

   DECLARE dbm_snap CHAR;
   DECLARE db_snap  CHAR;
   DECLARE tbs_snap CHAR;
   DECLARE tb_snap  CHAR;
   DECLARE bp_snap  CHAR;

   ----------------------------------------------------------------------
   -- 'ignore lists' we pass to diff_row, to tell it which columns we don't care about too much.

   DECLARE dbm_ignore_list  VARCHAR(256) 
   	DEFAULT '(''DB2_STATUS'',''COMM_PRIVATE_MEM'')';

   DECLARE db_ignore_list  VARCHAR(256) 
   	DEFAULT '(''TOTAL_LOG_USED'',''TOTAL_LOG_AVAILABLE'',''LOCK_LIST_IN_USE'',''APPL_ID_OLDEST_XACT'',''SEC_LOG_USED_TOP'',''TOT_LOG_USED_TOP'',''LOCKS_WAITING'')';
   DECLARE tbs_ignore_list VARCHAR(256) DEFAULT '';
   DECLARE tb_ignore_list  VARCHAR(256) DEFAULT '(''TABLE_FILE_ID'',''TABLE_TYPE'')';
   DECLARE bp_ignore_list  VARCHAR(256) DEFAULT '';


   DECLARE at_end INT DEFAULT 0;

   ----------------------------------------------------------------------
   -- Cursors to get the lists of objects like tables & tablespaces that have data for the intervals
   -- We then use these table names, tablespace names, etc., to do the interval analysis in groups
   -- That is, we figure out the snapshot changes, etc. for table T1, then for table T2, etc.  We
   -- obviously don't want to mix these up.

   DECLARE tbs_cursor CURSOR FOR
   	SELECT DISTINCT tablespace_name
	FROM db2perf_snaptbs
	WHERE snapshot_timestamp = ts_start_1
	   OR snapshot_timestamp = ts_stop_1
	   OR snapshot_timestamp = ts_start_2
	   OR snapshot_timestamp = ts_stop_2;

   DECLARE tb_cursor CURSOR FOR
   	SELECT DISTINCT table_name
	FROM db2perf_snaptb
	WHERE (snapshot_timestamp = ts_start_1
	   OR snapshot_timestamp = ts_stop_1
	   OR snapshot_timestamp = ts_start_2
	   OR snapshot_timestamp = ts_stop_2)
	  AND table_schema = CURRENT SCHEMA;

   DECLARE bp_cursor CURSOR FOR
   	SELECT DISTINCT bp_name
	FROM db2perf_snapbp
	WHERE (snapshot_timestamp = ts_start_1
	   OR snapshot_timestamp = ts_stop_1
	   OR snapshot_timestamp = ts_start_2
	   OR snapshot_timestamp = ts_stop_2);

   ----------------------------------------------------------------------
   -- Declare a cursor for the 'list' operation
   
   DECLARE list_cursor CURSOR FOR
   	SELECT char(interval) || char(start) || '  ' || coalesce(char(stop),'                          ') || '    ' 
		|| coalesce(db2perf_trimlzero( char(cast(stop-start as decimal(8,1)))),'          ') 
		|| '   '    || dbm || '   ' || db || '   ' || tbs || '   ' || tb || '   ' || bp 
	FROM db2perf_snapdiff_toc
	ORDER BY interval DESC;

   ----------------------------------------------------------------------
   -- Before we leave the routine, we leave open a cursor on our report table
   -- so that the caller (typically CLP) can just pull that information back easily.

   DECLARE report_cursor CURSOR WITH RETURN TO CALLER FOR
   	SELECT message
	FROM db2perf_snapdiff_report
	ORDER BY line ASC;

   DECLARE CONTINUE HANDLER FOR NOT FOUND
     SET at_end = 1; 


   ----------------------------------------------------------------------
   -- Make sure the parameters are all OK.

   SET operation       = translate(operation);
   SET snap_table_name = coalesce(translate(snap_table_name),'');
   SET interval_1      = coalesce(interval_1,-1);
   SET interval_2      = coalesce(interval_2,-1);
   SET normalize_to_Tx = coalesce(translate(normalize_to_Tx),'Y');
   SET threshold_pct   = coalesce(threshold_pct,5);


   -- Clean out the report table

   DELETE FROM db2perf_snapdiff_report;


   -- Swap the interval numbers if they're in the 'wrong' order
  
   IF interval_1 > interval_2 THEN
     SET current_interval   = interval_1;
     SET interval_1         = interval_2;
     SET interval_2         = current_interval;
   END IF;


   ----------------------------------------------------------------------
   -- Write a header to the top of the report, indicating what we're going to do.

   INSERT INTO db2perf_snapdiff_report VALUES ( line, 'db2perf_snapdiff called at ' || char(current timestamp) );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '  operation:.............. ' || operation );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '  snap_table_name:........ ' || snap_table_name );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '  first interval:......... ' || char(interval_1) );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '  second interval:........ ' || char(interval_2) );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '  normalize to 1k Tx:..... ' || normalize_to_Tx );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '  threshold %:............ ' || char(threshold_pct) );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '  ' );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '====================================================================================================' );
   SET line=line+1;
   INSERT INTO db2perf_snapdiff_report VALUES ( line, '  ' );
   SET line=line+1;


   --------------------------------------
   -- Delete
   --
   -- Clean out the tables, including the table of contents.
   --------------------------------------
   IF operation = 'DELETE' THEN

     DELETE FROM db2perf_snapdbm;
     DELETE FROM db2perf_snapdb;
     DELETE FROM db2perf_snaptbs;
     DELETE FROM db2perf_snaptb;
     DELETE FROM db2perf_snapbp;
     DELETE FROM db2perf_snapdiff_toc;

     INSERT INTO db2perf_snapdiff_report VALUES ( line, 'Snapshot tables cleaned' );
     SET line=line+1;

   ELSE 
     --------------------------------------
     -- Start / Stop
     --
     -- Collect new snapshot rows
     --------------------------------------

     IF operation = 'START' OR operation = 'STOP' THEN

       ----------------------------------------------------------------------
       -- Snapshot table functions need to have the global switches turned on

       SELECT DFT_MON_BUFPOOL + DFT_MON_SORT + DFT_MON_STMT + DFT_MON_TABLE + DFT_MON_UOW 
	 INTO switch_total
	 FROM table( get_dbm_config() ) AS t
	 WHERE DBMCONFIG_TYPE = 0;

       IF switch_total <> 5 THEN
	 INSERT INTO db2perf_snapdiff_report VALUES ( line, 'DFT_MON_xxx switches should be on for best results' );
	 SET line=line+1;
       ELSE

	 VALUES CURRENT TIMESTAMP INTO new_ts;


         ----------------------------------------------------------------------
         -- Make sure we're not STARTing when we're already started, and not STOPping when already stopped.
	 -- Intervals are in the TOC and number starting from 1 & going up.  A zero interval indicates
	 -- one that's in the process of being captured.   On START, the interval number is set to zero
	 -- for the new row.  On STOP, the row with zero is now complete, so the interval is updated
	 -- from zero to be the new maximum number.

	 SET at_end = 0;
	 -- Are we between a start & a stop?
	 SELECT interval INTO current_interval FROM db2perf_snapdiff_toc WHERE interval = 0;
	 IF at_end = 1 AND operation = 'STOP' THEN
	   -- Yes we are - change it to a start
	   INSERT INTO db2perf_snapdiff_report VALUES ( line, 'Warning: no starting snapshot found, changing to START' );
	   SET operation = 'START';
	 ELSE
	   IF at_end = 0 AND operation = 'START' THEN
	     -- No '0' interval was found and we're supposed to be stopping - switch to start
	     INSERT INTO db2perf_snapdiff_report VALUES ( line, 'Warning: interval already started, changing to STOP' );
	     SET operation = 'STOP';
	   END IF;
	 END IF;


	 IF operation = 'START' THEN
	   -- We're (really) starting an interval, so write a '0' interval to the table of contents
	   INSERT INTO db2perf_snapdiff_toc VALUES ( 0, new_ts, NULL, 'N','N','N','N','N' );
	 ELSE
	   -- We're (really) stopping an interval, so replace the '0' interval number with a new highest value
	   SELECT max(interval)+1 INTO current_interval FROM db2perf_snapdiff_toc;

           SELECT dbm, db, tbs, tb, bp 
    	     INTO dbm_snap, db_snap, tbs_snap, tb_snap, bp_snap
  	     FROM db2perf_snapdiff_toc
	    WHERE interval = 0;

	   UPDATE db2perf_snapdiff_toc 
	      SET interval = current_interval, stop = new_ts 
	      WHERE interval = 0;

	   -- We want to make sure when we're doing a STOP, that we're taking snapshots on the same tables
	   -- we did the START on.   Replace the setting of snap_table_name with what we can derive from
	   -- the flag settings in the toc table.

           SET snap_table_name = CASE 
		WHEN dbm_snap = 'Y' AND db_snap = 'Y' AND tbs_snap = 'Y' AND tb_snap = 'Y' AND bp_snap = 'Y' 
			THEN ''
		WHEN dbm_snap = 'Y' 
			THEN 'DB2PERF_SNAPDBM'
		WHEN db_snap = 'Y' 
			THEN 'DB2PERF_SNAPDB'
		WHEN tbs_snap = 'Y' 
			THEN 'DB2PERF_SNAPTBS'
		WHEN tb_snap = 'Y' 
			THEN 'DB2PERF_SNAPTB'
		WHEN bp_snap = 'Y' 
			THEN 'DB2PERF_SNAPBP'
		ELSE
			''
		END;

	 END IF;


         ----------------------------------------------------------------------
         -- Go through the snapshot tables one by one to see which one(s) we should be collecting new snapshots for.

	 IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPDBM' THEN

           ----------------------------------------------------------------------
           -- We mark in the TOC table that we have DBM snapshot data for this interval
	   IF operation = 'START' THEN
	     UPDATE db2perf_snapdiff_toc 
	     	SET dbm = 'Y'
		WHERE interval = 0;
	   END IF;

           ----------------------------------------------------------------------
           -- Grab a row of snapshot data directly from the table function and save it away.
	   -- Update the timestamp so that all rows we collected for this interval have the exact
	   -- same one.

	   INSERT INTO db2perf_snapdbm 
	      SELECT * FROM table(snapshot_dbm(-1)) as t
	      FETCH FIRST 1 ROW ONLY;
	   SELECT max(snapshot_timestamp) INTO old_ts FROM db2perf_snapdbm;
	   UPDATE db2perf_snapdbm 
	      SET snapshot_timestamp = new_ts
	      WHERE snapshot_timestamp = old_ts;

	 END IF; -- Database manager snapshot


	 IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPDB' THEN

           ----------------------------------------------------------------------
           -- We mark in the TOC table that we have DB snapshot data for this interval
	   
	   IF operation = 'START' THEN
	     UPDATE db2perf_snapdiff_toc 
	     	SET db = 'Y'
		WHERE interval = 0;
	   END IF;

           ----------------------------------------------------------------------
           -- Grab a row of snapshot data directly from the table function and save it away.
	   -- Update the timestamp so that all rows we collected for this interval have the exact
	   -- same one.

	   INSERT INTO db2perf_snapdb
	      SELECT * FROM table(snapshot_database(CAST (NULL as varchar(256)),-1)) as t
	      FETCH FIRST 1 ROW ONLY;
	   SELECT max(snapshot_timestamp) INTO old_ts FROM db2perf_snapdb;
	   UPDATE db2perf_snapdb 
	      SET snapshot_timestamp = new_ts
	      WHERE snapshot_timestamp = old_ts;

	 END IF; -- database snapshot


	 IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPTBS' THEN

           ----------------------------------------------------------------------
           -- We mark in the TOC table that we have tablespace snapshot data for this interval

	   IF operation = 'START' THEN
	     UPDATE db2perf_snapdiff_toc 
	     	SET tbs = 'Y'
		WHERE interval = 0;
	   END IF;

           ----------------------------------------------------------------------
           -- Grab a row of snapshot data directly from the table function and save it away.
	   -- Update the timestamp so that all rows we collected for this interval have the exact
	   -- same one.

	   INSERT INTO db2perf_snaptbs
	      SELECT * FROM table(snapshot_tbs(CAST (NULL as varchar(256)),-1)) as t;
	   SELECT max(snapshot_timestamp) INTO old_ts FROM db2perf_snaptbs;
	   UPDATE db2perf_snaptbs 
	      SET snapshot_timestamp = new_ts
	      WHERE snapshot_timestamp = old_ts;

	 END IF; -- tablespace snapshot


	 IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPTB' THEN

           ----------------------------------------------------------------------
           -- We mark in the TOC table that we have table snapshot data for this interval

	   IF operation = 'START' THEN
	     UPDATE db2perf_snapdiff_toc 
	     	SET tb = 'Y'
		WHERE interval = 0;
	   END IF;

           ----------------------------------------------------------------------
           -- Grab a row of snapshot data directly from the table function and save it away.
	   -- Update the timestamp so that all rows we collected for this interval have the exact
	   -- same one.

	   INSERT INTO db2perf_snaptb
	      SELECT * FROM table(snapshot_table(CAST (NULL as varchar(256)),-1)) as t;
	   SELECT max(snapshot_timestamp) INTO old_ts FROM db2perf_snaptb;
	   UPDATE db2perf_snaptb 
	      SET snapshot_timestamp = new_ts
	      WHERE snapshot_timestamp = old_ts;

	 END IF; -- table snapshot



	 IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPBP' THEN

           ----------------------------------------------------------------------
           -- We mark in the TOC table that we have bufferpool snapshot data for this interval

	   IF operation = 'START' THEN
	     UPDATE db2perf_snapdiff_toc 
	     	SET bp = 'Y'
		WHERE interval = 0;
	   END IF;

           ----------------------------------------------------------------------
           -- Grab a row of snapshot data directly from the table function and save it away.
	   -- Update the timestamp so that all rows we collected for this interval have the exact
	   -- same one.

	   INSERT INTO db2perf_snapbp
	      SELECT * FROM table(snapshot_bp(CAST (NULL as varchar(256)),-1)) as t;
	   SELECT max(snapshot_timestamp) INTO old_ts FROM db2perf_snapbp;
	   UPDATE db2perf_snapbp 
	      SET snapshot_timestamp = new_ts
	      WHERE snapshot_timestamp = old_ts;

	 END IF;  -- bufferpool snapshot



         -- Report that we took a snapshot
	 INSERT INTO db2perf_snapdiff_report VALUES ( line, 
	 	'Snapshot ' || operation || ' at timestamp ' || char(new_ts) );
	 SET line=line+1;
       END IF;

     ELSE

       IF operation = 'COMPARE' OR operation = 'DIFF' THEN

	 --------------------------------------
	 -- Compare / Diff
	 --
	 -- Compare snapshot values across intervals
	 --------------------------------------

	 
         ----------------------------------------------------------------------
	 -- An interval # of -1 (either start or stop) means use the latest (or 2nd latest)

	 IF interval_1 = -1 THEN
	   SET get_interval_1 = 1;	-- set the flag that says we need to determine the real interval number
	 ELSE
	   SELECT start,stop INTO ts_start_1,ts_stop_1 -- otherwise, get the timestamp values for this interval
	      FROM db2perf_snapdiff_toc 
	      WHERE interval = interval_1;
	 END IF;
	   
	 IF interval_2 = -1 THEN
	   SET get_interval_2 = 1;
	 ELSE
	   SELECT start,stop INTO ts_start_2,ts_stop_2 FROM db2perf_snapdiff_toc 
	      WHERE interval = interval_2;
	 END IF;
	   


         ----------------------------------------------------------------------
	 -- Database manager snapshot
         ----------------------------------------------------------------------

         IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPDBM' THEN

           ----------------------------------------------------------------------
           -- Get the 2nd interval timestamps

	   IF get_interval_2 = 1 THEN
	     IF get_interval_1 = 0 THEN
               ----------------------------------------------------------------------
	       -- Only need to get the 2nd interval (the First one was provided).  
	       -- Get the one that follows the First one in the TOC.

	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT min(interval) 
			  FROM db2perf_snapdiff_toc 
			  WHERE interval > interval_1 AND dbm = 'Y'
			) 
		      AND dbm = 'Y';
	     ELSE
               ----------------------------------------------------------------------
	       -- We didn't get either interval, so we need to get both.  
	       -- Get the last (greatest) one as the 2nd interval.

	       SELECT interval,start,stop  
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT max(interval) 
			  FROM db2perf_snapdiff_toc 
			  WHERE dbm = 'Y' 
			) 
		      AND dbm = 'Y';
	     END IF;
	   END IF;


           ----------------------------------------------------------------------
           -- Now determine the 1st interval, if we need to

	   IF get_interval_1 = 1 THEN
	     SELECT interval,start,stop 
	       INTO interval_1,ts_start_1,ts_stop_1 
	       FROM db2perf_snapdiff_toc 
	      WHERE interval = 
	      		( SELECT max(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE interval < interval_2 AND dbm = 'Y'
			)
		    AND dbm = 'Y';
	   END IF;

	   IF  ts_start_1 IS NOT NULL 
	   AND ts_stop_1  IS NOT NULL 
	   AND ts_start_2 IS NOT NULL 
	   AND ts_stop_2  IS NOT NULL THEN

             
             ----------------------------------------------------------------------
	     -- If we get to here and all the timestamps are ok, then write out
	     -- the information to the report.

	     INSERT INTO db2perf_snapdiff_report VALUES ( line, 
		'Database Manager Snapshot (db2perf_snapdbm) --------------------------------------------------------');
	     INSERT INTO db2perf_snapdiff_report VALUES ( line+1, '' );
	     SET line = line+2;

	     IF normalize_to_Tx IN ('Y','T','1') THEN
	       -- We want to normalize to the number of transactions, so find out how many there are
	       -- in our intervals.
	       CALL db2perf_getTxCount( 
	       		ts_start_1, 
			ts_stop_1, 
			ts_start_2, 
			ts_stop_2, 
			line, 
			k_commit_sql_stmts_1, 
			k_commit_sql_stmts_2 );
	     END IF;

 
             ----------------------------------------------------------------------
	     -- Now compare the deltas within our two intervals

	     CALL db2perf_rowdiff( 'DB2PERF_SNAPDBM','snapshot_timestamp',ts_start_1,ts_stop_1,ts_start_2,ts_stop_2,'',
		  k_commit_sql_stmts_1,k_commit_sql_stmts_2,threshold_pct,dbm_ignore_list,line );

	     INSERT INTO db2perf_snapdiff_report VALUES ( line, '' );
	     SET line = line+1;

	   END IF;
	 END IF; -- comparing database manager snapshot



         IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPDB' THEN

	   IF get_interval_2 = 1 THEN
	     IF get_interval_1 = 0 THEN
	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT min(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE interval > interval_1 AND db = 'Y' 
			) 
		      AND db = 'Y';
	     ELSE
	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT max(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE db = 'Y'
			)
		      AND db = 'Y';
	     END IF;
	   END IF;

	   IF get_interval_1 = 1 THEN
	     SELECT interval,start,stop 
	       INTO interval_1,ts_start_1,ts_stop_1 
	       FROM db2perf_snapdiff_toc 
	      WHERE interval = 
	     	 	( SELECT max(interval) 
		      	    FROM db2perf_snapdiff_toc 
		           WHERE interval < interval_2 AND db = 'Y'
			) 
		    AND db = 'Y';
	   END IF;

	   IF ts_start_1 IS NOT NULL 
	   AND ts_stop_1 IS NOT NULL 
	   AND ts_start_2 IS NOT NULL 
	   AND ts_stop_2 IS NOT NULL THEN
	     INSERT INTO db2perf_snapdiff_report VALUES ( line, 
		'Database Snapshot (db2perf_snapdb) -----------------------------------------------------------------' );
	     INSERT INTO db2perf_snapdiff_report VALUES ( line+1, '' );
	     SET line = line+2;

	     IF normalize_to_Tx IN ('Y','T','1') THEN
	       CALL db2perf_getTxCount( 
	       		ts_start_1, 
			ts_stop_1, 
			ts_start_2, 
			ts_stop_2, 
			line, 
			k_commit_sql_stmts_1, 
			k_commit_sql_stmts_2 );
	     END IF;

	     CALL db2perf_rowdiff( 'DB2PERF_SNAPDB','snapshot_timestamp',ts_start_1,ts_stop_1,ts_start_2,ts_stop_2,'',
		  k_commit_sql_stmts_1,k_commit_sql_stmts_2,threshold_pct,db_ignore_list,line );

	     INSERT INTO db2perf_snapdiff_report VALUES ( line, '' );
	     SET line = line+1;
	   END IF;
	 END IF; -- comparing database snapshot




         IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPTBS' THEN

	   IF get_interval_2 = 1 THEN
	     IF get_interval_1 = 0 THEN
	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT min(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE interval > interval_1 AND tbs = 'Y'
			) 
		      AND tbs = 'Y';
	     ELSE
	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT max(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE tbs = 'Y') AND tbs = 'Y';
	     END IF;
	   END IF;

	   IF get_interval_1 = 1 THEN
	     SELECT interval,start,stop 
	       INTO interval_1,ts_start_1,ts_stop_1 
	       FROM db2perf_snapdiff_toc 
	      WHERE interval = 
	      		( SELECT max(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE interval < interval_2 AND tbs = 'Y'
			) 
		    AND tbs = 'Y';
	   END IF;


	   IF  ts_start_1 IS NOT NULL 
	   AND ts_stop_1  IS NOT NULL 
	   AND ts_start_2 IS NOT NULL 
	   AND ts_stop_2  IS NOT NULL THEN
	     INSERT INTO db2perf_snapdiff_report VALUES ( line, 
		'Tablespace Snapshot (db2perf_snaptbs) --------------------------------------------------------------' );
	     INSERT INTO db2perf_snapdiff_report VALUES ( line+1, '' );
	     SET line = line+2;

	     IF normalize_to_Tx IN ('Y','T','1') THEN
	       CALL db2perf_getTxCount( 
	       		ts_start_1, 
			ts_stop_1, 
			ts_start_2, 
			ts_stop_2, 
			line, 
			k_commit_sql_stmts_1, 
			k_commit_sql_stmts_2 );
	     END IF;

		
             ----------------------------------------------------------------------
	     -- In the case of tablespaces (and tables and bufferpools) we have pairs
	     -- of intervals to consider - so we have to match them up, and compare the 
	     -- data for each tablespace correctly.

	     SET at_end = 0;
	     OPEN tbs_cursor;
	     FETCH tbs_cursor INTO tablespace_name;

	     WHILE at_end = 0 DO
	       INSERT INTO db2perf_snapdiff_report VALUES ( line+1, '  Tablespace ' || tablespace_name );
	       INSERT INTO db2perf_snapdiff_report VALUES ( line+2, '' );
	       SET line = line+2;

	       CALL db2perf_rowdiff( 'DB2PERF_SNAPTBS','snapshot_timestamp',ts_start_1,ts_stop_1,ts_start_2,ts_stop_2,
		      'TABLESPACE_NAME = ''' || tablespace_name || '''',
		      k_commit_sql_stmts_1,k_commit_sql_stmts_2,threshold_pct,tbs_ignore_list,line );
	       INSERT INTO db2perf_snapdiff_report VALUES ( line, '' );
	       SET line = line+1;

	       FETCH tbs_cursor INTO tablespace_name;
	     END WHILE;
	     SET at_end = 0;
	   END IF;
	 END IF;  -- comparing tablespace snapshots




         IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPTB' THEN

	   IF get_interval_2 = 1 THEN
	     IF get_interval_1 = 0 THEN
	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT min(interval) 
			    FROM db2perf_snapdiff_toc 
			  WHERE interval > interval_1 AND tb = 'Y'
			) 
		      AND tb = 'Y';
	     ELSE
	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT max(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE tb = 'Y'
			) 
		      AND tb = 'Y';
	     END IF;
	   END IF;

	   IF get_interval_1 = 1 THEN
	     SELECT interval,start,stop 
	       INTO interval_1,ts_start_1,ts_stop_1 
	       FROM db2perf_snapdiff_toc 
	      WHERE interval = 
	     		( SELECT max(interval) 
			    FROM db2perf_snapdiff_toc 
		           WHERE interval < interval_2 AND tb = 'Y'
			) 
		     AND tb = 'Y';
	   END IF;

	   IF  ts_start_1 IS NOT NULL 
	   AND ts_stop_1  IS NOT NULL 
	   AND ts_start_2 IS NOT NULL 
	   AND ts_stop_2  IS NOT NULL THEN

	     INSERT INTO db2perf_snapdiff_report VALUES ( line+1, 
		'Table Snapshot (db2perf_snaptb) --------------------------------------------------------------------' );
	     INSERT INTO db2perf_snapdiff_report VALUES ( line+2, '' );
	     SET line = line+3;

	     IF normalize_to_Tx IN ('Y','T','1') THEN
	       CALL db2perf_getTxCount( 
	       		ts_start_1, 
			ts_stop_1, 
			ts_start_2, 
			ts_stop_2, 
			line, 
			k_commit_sql_stmts_1, 
			k_commit_sql_stmts_2 );
	     END IF;

	     -- As with Tablespaces and Bufferpools, we need to pair up comparisons for particular
	     -- tables.

	     SET at_end = 0;
	     OPEN tb_cursor;
	     FETCH tb_cursor INTO table_name;

	     WHILE at_end = 0 DO
	       INSERT INTO db2perf_snapdiff_report VALUES ( line+1, '  Table ' || table_name );
	       INSERT INTO db2perf_snapdiff_report VALUES ( line+2, '' );
	       SET line = line+2;

	       CALL db2perf_rowdiff( 'DB2PERF_SNAPTB','snapshot_timestamp',ts_start_1,ts_stop_1,ts_start_2,ts_stop_2,
		      'TABLE_NAME = ''' || table_name || ''' AND TABLE_SCHEMA = ''' || CURRENT SCHEMA || '''',
		      k_commit_sql_stmts_1,k_commit_sql_stmts_2,threshold_pct,tb_ignore_list,line );
	       INSERT INTO db2perf_snapdiff_report VALUES ( line, '' );
	       SET line = line+1;

	       FETCH tb_cursor INTO table_name;
	     END WHILE;
	     SET at_end = 0;
	   END IF;
	 END IF;  -- comparing table snapshots



         IF snap_table_name = '' OR snap_table_name = 'DB2PERF_SNAPBP' THEN

	   IF get_interval_2 = 1 THEN
	     IF get_interval_1 = 0 THEN
	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT min(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE interval > interval_1 AND bp = 'Y'
			) 
		      AND bp = 'Y';
	     ELSE
	       SELECT interval,start,stop 
	         INTO interval_2,ts_start_2,ts_stop_2 
		 FROM db2perf_snapdiff_toc 
	       	WHERE interval = 
			( SELECT max(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE bp = 'Y'
			) 
		      AND bp = 'Y';
	     END IF;
	   END IF;

	   IF get_interval_1 = 1 THEN
	     SELECT interval,start,stop 
	       INTO interval_1,ts_start_1,ts_stop_1 
	       FROM db2perf_snapdiff_toc 
	      WHERE interval = 
	      		( SELECT max(interval) 
			    FROM db2perf_snapdiff_toc 
			   WHERE interval < interval_2 AND bp = 'Y'
			) 
		    AND bp = 'Y';
	   END IF;

	   IF  ts_start_1 IS NOT NULL 
	   AND ts_stop_1  IS NOT NULL 
	   AND ts_start_2 IS NOT NULL 
	   AND ts_stop_2  IS NOT NULL THEN
	     INSERT INTO db2perf_snapdiff_report VALUES ( line+1, 
		'Buffer Pool Snapshot (db2perf_snapbp) --------------------------------------------------------------' );
	     INSERT INTO db2perf_snapdiff_report VALUES ( line+2, '' );
	     SET line = line+3;

	     IF normalize_to_Tx IN ('Y','T','1') THEN
	       CALL db2perf_getTxCount( 
	       		ts_start_1, 
			ts_stop_1, 
			ts_start_2, 
			ts_stop_2, 
			line, 
			k_commit_sql_stmts_1, 
			k_commit_sql_stmts_2 );
	     END IF;

	     SET at_end = 0;
	     OPEN bp_cursor;
	     FETCH bp_cursor INTO bufferpool;

	     WHILE at_end = 0 DO
	       INSERT INTO db2perf_snapdiff_report VALUES ( line+1, '  Bufferpool ' || bufferpool );
	       INSERT INTO db2perf_snapdiff_report VALUES ( line+2, '' );
	       SET line = line+2;

	       CALL db2perf_rowdiff( 'DB2PERF_SNAPBP','snapshot_timestamp',ts_start_1,ts_stop_1,ts_start_2,ts_stop_2,
		      'BP_NAME = ''' || bufferpool || '''',
		      k_commit_sql_stmts_1,k_commit_sql_stmts_2,threshold_pct,bp_ignore_list,line );
	       INSERT INTO db2perf_snapdiff_report VALUES ( line, '' );
	       SET line = line+1;

	       FETCH bp_cursor INTO bufferpool;
	     END WHILE;
	   END IF;
	 END IF; -- comparing bufferpool snapshots


       ELSE 

	 --------------------------------------
	 -- LIST
	 --
	 -- Lists the snapshots that we have in the table of contents
	 --------------------------------------
         IF operation = 'LIST' THEN

	   INSERT INTO db2perf_snapdiff_report VALUES ( line, 
	     'Interval   Start Time                  Stop Time                  Duration (s)   DBM  DB TBS  TB  BP' );

	   OPEN list_cursor;
	   SET at_end = 0;

 	   FETCH list_cursor INTO message;
	   WHILE at_end = 0 DO
	     INSERT INTO db2perf_snapdiff_report VALUES ( line, message );
	     SET line = line + 1;
 	     FETCH list_cursor INTO message;
	   END WHILE;

	 ELSE

	   INSERT INTO db2perf_snapdiff_report VALUES ( line, 'Unknown operation ''' || operation || '''' );
	   SET line=line+1;

	 END IF; -- unknown operation
       END IF; -- operation = diff
     END IF; -- operation = start/stop
   END IF; -- operation = delete

   ----------------------------------------------------------------------
   -- All done.    Open a cursor on the report table, and return
   OPEN report_cursor;
END@


----------------------------------------------------------------------
----------------------------------------------------------------------
-- Now we create a couple of 'fastpath' versions of our routine, 
-- that spare the user from having to enter all the parameters.
----------------------------------------------------------------------
----------------------------------------------------------------------

ECHO Creating stored procedure db2perf_snapdiff with defaulted parameters@
ECHO @


----------------------------------------------------------------------
-- db2perf_snapdiff() with just the OPERATION parameter
----------------------------------------------------------------------

CALL db2perf_quiet_drop('PROCEDURE db2perf_snapdiff( VARCHAR(20) )')@

CREATE PROCEDURE db2perf_snapdiff( IN operation VARCHAR(20))
DYNAMIC RESULT SETS 1
LANGUAGE SQL
BEGIN
   DECLARE report_cursor CURSOR WITH RETURN TO CALLER FOR
   	SELECT message
	FROM db2perf_snapdiff_report
	ORDER BY line ASC;

   CALL db2perf_snapdiff( operation,'',-1,-1,'Y',5 );

   OPEN report_cursor;
END@



----------------------------------------------------------------------
-- db2perf_snapdiff() with no parameters, upon which we dump out usage information.
----------------------------------------------------------------------

CALL db2perf_quiet_drop('PROCEDURE db2perf_snapdiff()')@

CREATE PROCEDURE db2perf_snapdiff()
DYNAMIC RESULT SETS 1
LANGUAGE SQL
BEGIN
   DECLARE status VARCHAR(120);

   DECLARE report_cursor CURSOR WITH RETURN TO CALLER FOR
   	SELECT message
	FROM db2perf_snapdiff_report
	ORDER BY line ASC;

   DELETE FROM db2perf_snapdiff_report;

   INSERT into db2perf_snapdiff_report VALUES 
   	(  0,'Usage:'),
	(  1,'  db2perf_snapdiff( <operation> )' ),
	(  2,'  ' ),
	(  3,'  db2perf_snapdiff( <operation>, '),
	(  4,'                    <snapdiff_table_name>, '),
	(  5,'                    <interval_1>, '),
	(  6,'                    <interval_2>, '),
	(  7,'                    <normalize>, '),
	(  8,'                    <threshold_pct> )' ),
	(  9,'    operation          = [start | stop | compare | diff | list | delete]'),
	( 10,'    snapdiff_tablename = [NULL | '''' | ' ),
	( 12,'                          db2perf_snapdbm | ' ),
	( 13,'                          db2perf_snapdb  | ' ),
	( 14,'                          db2perf_snaptbs | ' ),
	( 15,'                          db2perf_snaptb  | ' ),
	( 16,'                          db2perf_snapbp]'    ),
	( 17,'    interval_1          = [NULL | '''' | '),
	( 18,'                          <interval number of ''before'' rows in snapshot tables>]' ),
	( 19,'    interval_2          = [NULL | '''' | '),
	( 20,'                          <interval number of ''after'' rows in snapshot tables>]' ),
	( 21,'    normalize          = [NULL | '''' | Y|y|T|t]   |   [N|n|F|f '),
	( 22,'                               - normalize values to per 1,000 transactions' ),
	( 23,'    threshold_pct      = [NULL | '''' | 0]   |   <numeric> '),
	( 24,'                               - only report % differences larger than this value');

   OPEN report_cursor;
END@
