-----------------------------------------------------------------------
--(c) Copyright IBM Corporation 2006  All rights reserved.           --
--                                                                   --
--This sample program is owned by International Business Machines    --
--Corporation or one of its subsidiaries ("IBM") and is copyrighted  --
--and licensed, not sold.                                            --
--BY ACCESSING, COPYING, OR USING THIS SAMPLE PROGRAM, YOU AGREE TO  --
--THE TERMS OF THE AGREEMENT TITLED "International License Agreement --
--for Non-Warranted db2perf Programs" LOCATED IN THE FILE NAMED      --
--"license.txt".                                                     --
--                                                                   --
-- db2perf_evmon.db2                                                 --
-- Steve Rees - srees@ca.ibm.com                                     --
--                                                                   --
-- Analyzes statement event monitor data in DB2 for hot statements,  --
-- high commit & rollback time, etc.                                 --
--                                                                   --
-----------------------------------------------------------------------


----------------------------------------------------------------------
-- Our results are written to a table in the database.   Drop it if
-- it's here, and then recreate it.

CALL db2perf_quiet_drop('TABLE db2perf_evmon_report')@
CREATE TABLE db2perf_evmon_report( line INTEGER, message VARCHAR(150) )@


CALL db2perf_quiet_drop('PROCEDURE db2perf_evmon( VARCHAR(128), INTEGER)' )@


----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_evmon
----------------------------------------------------------------------
----------------------------------------------------------------------
CREATE PROCEDURE db2perf_evmon( 
	event_table 	VARCHAR(128), 	-- the name of the table in DB2 that contains the event monitor data
	top 		INTEGER )	-- the "Top N" statements we want reporting on
DYNAMIC RESULT SETS 1
BEGIN
  DECLARE line INTEGER DEFAULT 0;

  DECLARE stmt VARCHAR(2000);

  DECLARE start_time      TIMESTAMP;
  DECLARE stop_time       TIMESTAMP;
  DECLARE physical_reads  INTEGER;
  DECLARE rows_read       INTEGER;
  DECLARE rows_written    INTEGER;
  DECLARE sort_overflows  INTEGER;
  DECLARE total_sort_time INTEGER;
  DECLARE total_sorts     INTEGER;
  DECLARE elapsed_time    DECIMAL(8,1);
  DECLARE pkg_name        CHAR(8);
  DECLARE section         INTEGER;
  DECLARE count           INTEGER;
  DECLARE cpu_time        DECIMAL(9,3);
  DECLARE operation       BIGINT;
  DECLARE stmt_type       BIGINT;
  DECLARE stmt_text       VARCHAR(2000);

  DECLARE msg_buffer      VARCHAR(200);

  DECLARE at_end INT DEFAULT 0;

  DECLARE top10_physread CURSOR FOR top10_physread_stmt;
  DECLARE top10_elapsed  CURSOR FOR top10_elapsed_stmt;
  DECLARE top10_rows     CURSOR FOR top10_rows_stmt;
  DECLARE top10_sorts    CURSOR FOR top10_sorts_stmt;
  DECLARE stats_cursor   CURSOR FOR stats_stmt;
  DECLARE time_cursor    CURSOR FOR time_stmt;


  ----------------------------------------------------------------------
  -- We return the report in a result set via a cursor we leave open when we return.
  DECLARE report_cursor CURSOR WITH RETURN TO CALLER FOR
    SELECT message
    FROM db2perf_evmon_report
    ORDER BY line;

  DECLARE CONTINUE HANDLER FOR NOT FOUND
      SET at_end = 1;

  DELETE FROM db2perf_evmon_report;


  ----------------------------------------------------------------------
  -- We start with some overall statistics on the event monitor table
  -- number of statements, number of connections, time span, etc.

  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Statistics on event monitor table ' || event_table );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );

  ----------------------------------------------------------------------
  -- Because the name of the table holding the event monitor data is
  -- not known when this routine is written, we need to build the
  -- (dynamic) SQL on the fly.
  

  ----------------------------------------------------------------------
  -- Number of events

  SET stmt= 'SELECT count(*) from ' || event_table;
  PREPARE stats_stmt FROM stmt;
  OPEN stats_cursor;
  FETCH stats_cursor INTO count;
  CLOSE stats_cursor;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Number of events:............ ' || char(count) );
  SET line=line+1;


  ----------------------------------------------------------------------
  -- Number of connections

  SET stmt= 'SELECT count(distinct agent_id) from ' || event_table ;
  PREPARE stats_stmt FROM stmt;
  OPEN stats_cursor;
  FETCH stats_cursor INTO count;
  CLOSE stats_cursor;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Number of connections:....... ' || char(count) );
  SET line=line+1;


  ----------------------------------------------------------------------
  -- Number of committed transactions

  SET stmt= 'SELECT count(*) from ' || event_table || ' where db2perf_op2str(stmt_operation) = ''COMMIT''';
  PREPARE stats_stmt FROM stmt;
  OPEN stats_cursor;
  FETCH stats_cursor INTO count;
  CLOSE stats_cursor;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Number of transactions:...... ' || char(count) );
  SET line=line+1;


  ----------------------------------------------------------------------
  -- Number of rolled back transactions

  SET stmt= 'SELECT count(*) from ' || event_table || ' where db2perf_op2str(stmt_operation) = ''ROLLBACK''';
  PREPARE stats_stmt FROM stmt;
  OPEN stats_cursor;
  FETCH stats_cursor INTO count;
  CLOSE stats_cursor;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Number of rollbacks:......... ' || char(count) );
  SET line=line+1;


  ----------------------------------------------------------------------
  -- Time span of trace, from first event to last

  SET stmt= 'SELECT min(start_time),max(stop_time) from ' || event_table;
  PREPARE time_stmt FROM stmt;
  OPEN time_cursor;
  FETCH time_cursor INTO start_time,stop_time;
  CLOSE time_cursor;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Start / stop timestamps:..... ' || char(start_time) || ' to ' || char(stop_time) || 
    ' (' || ltrim(db2perf_trimlzero(char(CAST(stop_time-start_time AS DECIMAL(8,1))))) || ' seconds)' );
  SET line=line+1;


  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;


  ----------------------------------------------------------------------
  -- Now we look for 'Top 10' statements, first by elapsed time.
  -- We use GROUP BY to collapse individual events into summaries by
  -- package, section, stmt type and statement text
  ----------------------------------------------------------------------

  SET stmt=
  	'SELECT '
     || '  cast(sum(stop_time-start_time) as decimal(8,1)) as "Elapsed Time", '
     || '  package_name as "Package",'
     || '  cast(section_number as smallint) as "Section",'
     || '  count(*) as "# of events",'
     || '  cast(sum(system_cpu_time+user_cpu_time)/1000000 as decimal(9,3)) as "CPU seconds",'
     || '  stmt_type, '
     || '  cast(substr(stmt_text,1,2000) as VARCHAR(2000)) as "Stmt" '
     || 'FROM ' || event_table || ' '
     || 'WHERE db2perf_op2str(stmt_operation) IN (''PREPARE'',''EXECUTE'',''OPEN'',''FETCH'',''CLOSE'',''DESCRIBE'') '
     || 'GROUP BY package_name,section_number,stmt_type,cast(substr(stmt_text,1,2000) as varchar(2000))'
     || 'ORDER BY "Elapsed Time" desc '
     || 'FETCH FIRST ' || char(top) || ' rows only ';

  PREPARE top10_elapsed_stmt
	FROM stmt;

  OPEN top10_elapsed;

  SET at_end = 0;

  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Top ' || rtrim(char(top)) || ' statements by elapsed time' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Elapsed    Package   Section     # Events     CPU Time  Type     Statement' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    '------------------------------------------------------------------------------------------------------------------------------------------------------' );
  SET line=line+1;


  FETCH top10_elapsed
	INTO elapsed_time,
	     pkg_name,
	     section,
	     count,
	     cpu_time,
	     stmt_type,
	     stmt_text;

  WHILE at_end = 0 DO
    
    ----------------------------------------------------------------------
    -- If it's a static statement, we don't get the statement text from the
    -- event table, we get it from the catalog instead.

    IF db2perf_type2str(stmt_type) = 'STATIC' THEN
      SET stmt_text = '';
      SELECT substr(text,1,2000) 
        INTO stmt_text
      	FROM syscat.statements
      	WHERE pkgname = pkg_name
        AND sectno = section
	FETCH FIRST 1 ROW ONLY;

      SET at_end = 0;
    END IF;

   
    ----------------------------------------------------------------------
    -- Build the row we write out to the report table.

    SET msg_buffer = 
    	db2perf_trimlzero(CHAR(elapsed_time))   || ' '
	|| pkg_name 				|| ' '
	|| db2perf_trimlzero(char(section)) 	|| ' '
	|| db2perf_trimlzero(char(count)) 	|| ' '
	|| db2perf_trimlzero(char(cpu_time))	|| ' '
	|| db2perf_type2str(stmt_type)          || ' ';
    SET msg_buffer = msg_buffer ||
	substr( stmt_text,1,150-length(msg_buffer) );
    INSERT INTO db2perf_evmon_report VALUES ( line, msg_buffer );
    SET line=line+1;

    FETCH top10_elapsed
	  INTO elapsed_time,
	       pkg_name,
	       section,
	       count,
	       cpu_time,
	       stmt_type,
	       stmt_text;
  END WHILE;

  CLOSE top10_elapsed;



  ----------------------------------------------------------------------
  -- Next, top 10 by number of physical reads.  This gets us pretty close to hit ratio 
  -- by statement.
  ----------------------------------------------------------------------

  SET stmt=
  	'SELECT '
     || '  sum(pool_data_p_reads+pool_index_p_reads+pool_temp_data_p_reads+pool_temp_index_p_reads) as physreads, '
     || '  package_name,'
     || '  cast(section_number as smallint),'
     || '  count(*),'
     || '  stmt_type, '
     || '  CAST(substr(stmt_text,1,2000) as VARCHAR(2000))'
     || 'FROM ' || event_table || ' '
     || 'WHERE db2perf_op2str(stmt_operation) IN (''PREPARE'',''EXECUTE'',''OPEN'',''FETCH'',''CLOSE'',''DESCRIBE'') '
     || 'GROUP BY package_name,section_number,stmt_type,cast(substr(stmt_text,1,2000) as varchar(2000)) '
     || 'ORDER BY physreads DESC '
     || 'FETCH FIRST ' || char(top) || ' rows only ';

  PREPARE top10_physread_stmt
	FROM stmt;

  OPEN top10_physread;

  SET at_end = 0;

  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Top ' || rtrim(char(top)) || ' statements by total physical reads' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Physical');
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Reads       Package   Section     # Events   Type     Statement' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    '------------------------------------------------------------------------------------------------------------------------------------------------------' );
  SET line=line+1;


  FETCH top10_physread
	INTO physical_reads,
	     pkg_name,
	     section,
	     count,
	     stmt_type,
	     stmt_text;

  WHILE at_end = 0 DO

    ----------------------------------------------------------------------
    -- If it's a static statement, we don't get the statement text from the
    -- event table, we get it from the catalog instead.

    IF db2perf_type2str(stmt_type) = 'STATIC' THEN
      SET stmt_text = '';
      SELECT substr(text,1,2000) 
        INTO stmt_text
      	FROM syscat.statements
      	WHERE pkgname = pkg_name
        AND sectno = section
	FETCH FIRST 1 ROW ONLY;

      SET at_end = 0;
    END IF;

    SET msg_buffer =
    	db2perf_trimlzero(CHAR(physical_reads)) || ' '
	|| pkg_name 				|| ' '
	|| db2perf_trimlzero(char(section)) 	|| ' '
	|| db2perf_trimlzero(char(count)) 	|| ' '
	|| db2perf_type2str(stmt_type)          || ' ';
    SET msg_buffer = msg_buffer ||
	substr( stmt_text,1,150-length(msg_buffer) );
    INSERT INTO db2perf_evmon_report VALUES ( line, msg_buffer );
    SET line=line+1;

    FETCH top10_physread
	  INTO physical_reads,
	       pkg_name,
	       section,
	       count,
	       stmt_type,
	       stmt_text;
  END WHILE;

  CLOSE top10_physread;


  ----------------------------------------------------------------------
  -- Next, top 10 by number of rows read + rows written.  This helps us identify scans, and
  -- cross-references well with physical reads to show the big I/O hitters.
  ----------------------------------------------------------------------

  SET stmt=
  	'SELECT '
     || '  sum(rows_read),'
     || '  sum(rows_written),'
     || '  package_name,'
     || '  cast(section_number as smallint),'
     || '  count(*),'
     || '  stmt_type, '
     || '  cast(substr(stmt_text,1,2000) as varchar(2000))'
     || 'FROM ' || event_table || ' '
     || 'WHERE db2perf_op2str(stmt_operation) IN (''PREPARE'',''EXECUTE'',''OPEN'',''FETCH'',''CLOSE'',''DESCRIBE'') '
     || 'GROUP BY package_name,section_number,stmt_type,cast(substr(stmt_text,1,2000) as varchar(2000)) '
     || 'ORDER BY (sum(rows_read)+sum(rows_written)) DESC '
     || 'FETCH FIRST ' || char(top) || ' rows only ';

  PREPARE top10_rows_stmt
	FROM stmt;

  OPEN top10_rows;

  SET at_end = 0;

  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Top ' || rtrim(char(top)) || ' statements by rows read / written' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Rows Read   Rows Written Package   Section     # Events   Type     Statement' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    '------------------------------------------------------------------------------------------------------------------------------------------------------' );
  SET line=line+1;


  FETCH top10_rows
	INTO rows_read,
	     rows_written,
	     pkg_name,
	     section,
	     count,
	     stmt_type,
	     stmt_text;

  WHILE at_end = 0 DO
    IF db2perf_type2str(stmt_type) = 'STATIC' THEN
      SET stmt_text = '';
      SELECT substr(text,1,2000) 
        INTO stmt_text
      	FROM syscat.statements
      	WHERE pkgname = pkg_name
        AND sectno = section
	FETCH FIRST 1 ROW ONLY;

      SET at_end = 0;
    END IF;

    SET msg_buffer = 
    	db2perf_trimlzero(CHAR(rows_read))       || '  '
    	|| db2perf_trimlzero(CHAR(rows_written)) || ' '
	|| pkg_name 				 || ' '
	|| db2perf_trimlzero(char(section)) 	 || ' '
	|| db2perf_trimlzero(char(count)) 	 || ' '
	|| db2perf_type2str(stmt_type)           || ' ';
    SET msg_buffer = msg_buffer ||
	substr( stmt_text,1,150-length(msg_buffer) );
    INSERT INTO db2perf_evmon_report VALUES ( line, msg_buffer );
    SET line=line+1;

    FETCH top10_rows
	  INTO rows_read,
	       rows_written,
	       pkg_name,
	       section,
	       count,
	       stmt_type,
	       stmt_text;
  END WHILE;

  CLOSE top10_rows;



  ----------------------------------------------------------------------
  -- Next, top 10 by number of rows read + rows written.  This helps us identify scans, and
  -- cross-references well with physical reads to show the big I/O hitters.
  ----------------------------------------------------------------------

  SET stmt=
  	'SELECT '
     || '  sum(total_sort_time),'
     || '  sum(total_sorts),'
     || '  sum(sort_overflows),'
     || '  package_name,'
     || '  cast(section_number as smallint),'
     || '  count(*),'
     || '  stmt_type, '
     || '  cast(substr(stmt_text,1,2000) as varchar(2000))'
     || 'FROM ' || event_table || ' '
     || 'WHERE db2perf_op2str(stmt_operation) IN (''PREPARE'',''EXECUTE'',''OPEN'',''FETCH'',''CLOSE'',''DESCRIBE'') '
     || 'GROUP BY package_name,section_number,stmt_type,cast(substr(stmt_text,1,2000) as varchar(2000)) '
     || 'ORDER BY sum(total_sort_time) DESC '
     || 'FETCH FIRST ' || char(top) || ' rows only ';

  PREPARE top10_sorts_stmt
	FROM stmt;

  OPEN top10_sorts;

  SET at_end = 0;

  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Top ' || rtrim(char(top)) || ' statements by sort time' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,' ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Sort        Total        Sort' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    'Time        Sorts        Overflows   Package      Section    # Events   Type   Statement' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line,
    '------------------------------------------------------------------------------------------------------------------------------------------------------' );
  SET line=line+1;


  FETCH top10_sorts
	INTO total_sort_time,
	     total_sorts,
	     sort_overflows,
	     pkg_name,
	     section,
	     count,
	     stmt_type,
	     stmt_text;

  WHILE at_end = 0 DO

    IF db2perf_type2str(stmt_type) = 'STATIC' THEN
      SET stmt_text = '';
      SELECT substr(text,1,2000) 
        INTO stmt_text
      	FROM syscat.statements
      	WHERE pkgname = pkg_name
        AND sectno = section
	FETCH FIRST 1 ROW ONLY;

      SET at_end = 0;
    END IF;

    SET msg_buffer = 
    	db2perf_trimlzero(CHAR(total_sort_time))   || '  '
    	|| db2perf_trimlzero(CHAR(total_sorts))    || ' '
    	|| db2perf_trimlzero(CHAR(sort_overflows)) || ' '
	|| pkg_name 				   || ' '
	|| db2perf_trimlzero(char(section)) 	   || ' '
	|| db2perf_trimlzero(char(count)) 	   || ' '
	|| db2perf_type2str(stmt_type)             || ' ';
    SET msg_buffer = msg_buffer ||
	substr( stmt_text,1,150-length(msg_buffer) );
    INSERT INTO db2perf_evmon_report VALUES ( line, msg_buffer );
    SET line=line+1;

    FETCH top10_sorts
	  INTO total_sort_time,
	       total_sorts,
	       sort_overflows,
	       pkg_name,
	       section,
	       count,
	       stmt_type,
	       stmt_text;
  END WHILE;

  CLOSE top10_sorts;


  ----------------------------------------------------------------------
  -- The analysis is done, so open the cursor on the report table, and return.
  ----------------------------------------------------------------------

  OPEN report_cursor;

END@



----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_evmon
-- 	This version calls the main one, and just defaults to 'top 10'
----------------------------------------------------------------------
----------------------------------------------------------------------
CALL db2perf_quiet_drop('PROCEDURE db2perf_evmon( VARCHAR(128) )')@

CREATE PROCEDURE db2perf_evmon( event_table VARCHAR(128) )
DYNAMIC RESULT SETS 1
BEGIN
  DECLARE report_cursor CURSOR WITH RETURN TO CALLER FOR
    SELECT message
    FROM db2perf_evmon_report
    ORDER BY line;

  CALL db2perf_evmon( event_table, 10 );

  OPEN report_cursor;
END@



----------------------------------------------------------------------
----------------------------------------------------------------------
-- db2perf_evmon
-- 	This version takes no arguments and just prints out the usage instructions
----------------------------------------------------------------------
----------------------------------------------------------------------
CALL db2perf_quiet_drop('PROCEDURE db2perf_evmon( )')@

CREATE PROCEDURE db2perf_evmon( )
DYNAMIC RESULT SETS 1
BEGIN
  DECLARE line INTEGER DEFAULT 0;

  DECLARE report_cursor CURSOR WITH RETURN TO CALLER FOR
    SELECT message
    FROM db2perf_evmon_report
    ORDER BY line;

  DELETE FROM db2perf_evmon_report;

  INSERT INTO db2perf_evmon_report VALUES ( line, 'Usage: ' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line, 'db2perf_evmon( evmon-table-name [, number-of-statements ] )' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line, '    evmon-table-name: name of table with statement event monitor data' );
  SET line=line+1;
  INSERT INTO db2perf_evmon_report VALUES ( line, '    number-of-statements: the top N statements to report on (default: 10)' );
  SET line=line+1;

  OPEN report_cursor;
END@

