/***************************************************************************/
/*                                                                         */
/*  Licensed Materials - Property of IBM                                   */
/*                                                                         */
/*  IBM Visual Warehouse                                                   */
/*  Copyright (C) International Business Machines Corp., 1996              */
/*  All rights reserved                                                    */
/*                                                                         */
/***************************************************************************/

/*

        (C) Copyright IBM Corporation 1996, Visual Warehouse

        Description : Determine the sum of all the NET_INCOME
                      and trigger the award if the sum has exceeded
                      $550,000.
                      This is a simple example of trigger program.

        DISCLAIMER OF WARRANTIES.  The following [enclosed] code is
        sample code created by IBM Corporation. This sample code is
        not part of any standard IBM product and is provided to you
        solely for the purpose of assisting you in the development
        of your applications.  The code is provided "AS IS",
        without warranty of any kind. No IBM shall be liable for
        any damages arising out of your use of the sample code,
        even if they have been advised of the possibility of
        such damages.


        Linkage: The input is a single DB table/edition 

*/

//Parse upper arg StableName Iedt  '(' parms

// Developer's note:
// This file consists of several object class definitions which could be separated
// into smaller files.  User may modify this User Defined Program for double byte
// support or use ones own object classes.
// Methods of each object class are grouped together with the class definition
// for easy reading.

// include files --------------------------------------------------------------
#include <windows.h>
#include <iostream.h>
#include <tchar.h>
#include <sqlext.h>
#include <stdio.h>
#include <time.h>

// constants defined for this module ------------------------------------------
#define TRGAWARD_NUM_ARGS        7    // number of input arguments for TRGAWARD
#define TRGAWARD_ARG_TO_FOLLOW   5    // number of args to follow = TRGAWARD_NUM_ARGS - 2
#define TRGAWARD_MIN_INCOME  50000    // min amount of income to receive an award
#define VW_ODBC_SUCCESS    "00000"    // successful sqlstate
// error codes used -----------------------------------------------------------
#define VW_OK                    0  
#define VW_REACHED_AWARD_LEVEL 100       // award would be issued, rc will be reset to 0
#define VW_NO_AWARD            900       // no award would be issued
#define VW_ODBC_ERROR          -16       // database processing error
#define VW_ERR_NUM_ARGS      -1000       // invalid number of input arguments
#define VW_ERR_INV_ARGS      -1010       // invalid argument values
// class declarations ---------------------------------------------------------
class CommandLine;      // parser for command line arguments
class SqlRC;            // returned SQLCODE, SQLSTATE, SLQ Messages 
class SqlStmt;          // ODBC cursor object class
typedef SqlStmt   *PSQLSTMT;
class SqlOdbc;          // ODBC Object class
typedef SqlOdbc   *PSQLODBC;

// handle and cursor states
enum state { VW_NOTALLOCATED, VW_ALLOCATED, VW_INUSE, 
             VW_COMMITTED, VW_ROLLBACKED, VW_CLOSED};


// class definitions ----------------------------------------------------------
class CommandLine {
public:
      CommandLine( void ) {};
      long commandLineParser( const int argc,  TCHAR *argv[] );
      TCHAR * getTgtTableName ( void ) { return tgtTableName; }
      TCHAR * getConnectString( void );
      long getOutEdNum( void ) { return ( _ttol( pszOutEdition ));}
protected:
      TCHAR *DSName, *Userid, *Passwd;
      TCHAR *tgtTableName;
      TCHAR pszOutEdition[12];
      BOOL  numStrValidation( const char * digitStr );
};
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       CommandLine methods 
// Command Line Parser ********************************************************
long CommandLine::commandLineParser( const int argc,  TCHAR *argv[] )
{     
   // input: TRGAWARD.EXE 5 DSName UserID PassWD TtableName Oedt 
   // input has to be at least TRGAWARD_NUM_ARGS number of arguments; 
   //   including this exe name
   // the second parameter which is number of arguments to follow,
   // must equal to TRGAWARD_ARG_TO_FOLLOW
   // note: if user is going to add user specified paramters, this section 
   // needs to be modified
   long rc = (argc >= TRGAWARD_NUM_ARGS) ? VW_OK : VW_ERR_NUM_ARGS ;
   if ( !rc ) {
      // validate the input is numeric string before calling atol to convert to integer
      if ( numStrValidation( argv[1] )) {
         long lArgNum = _ttol( argv[1] );
         rc = ( lArgNum >= TRGAWARD_ARG_TO_FOLLOW ) ? VW_OK : VW_ERR_NUM_ARGS;
      }
      else rc = VW_ERR_INV_ARGS;    // first argument is not a digit
   }
   if ( !rc ) {
      for ( int i = 1; ( i < argc ); i++ ) {
         switch (i) {
            // In the original OS/2 version, all input arguments were uppercased.
            // In this NT version, DSN, USERID and PASSWORD are left exactly as they were 
            //    defined in VisualWarehouse.
            case 2:  DSName = argv[i];
                     break;
            case 3:  Userid = argv[i];
                     break;
            case 4:  Passwd = argv[i];
                     break;
            case 5:  tgtTableName = _tcsupr( argv[i] );
                     break;
            case 6:  strcpy( pszOutEdition, argv[i] );
                     break;
            default:
                     break;
         }
      }
      rc = numStrValidation( pszOutEdition ) ? VW_OK : VW_ERR_INV_ARGS;
   }
   return rc;
}
// Input numeric string verification ************************************************
BOOL CommandLine::numStrValidation( const char * digitStr )
{
   BOOL rc = TRUE;
   unsigned int i;
   unsigned int len = _tcslen( digitStr );
   // Make sure the number string contains only digits
   for ( i = 0; (( rc ) && ( i < len )); i++ )
      rc = _istdigit( digitStr[i] );
   return rc;
}
// retrieve connect string ****************************************************
TCHAR *CommandLine::getConnectString( void )
{ 
   TCHAR *pszStr = new TCHAR[SQL_MAX_OPTION_STRING_LENGTH]; 
   _tcscpy( pszStr, _TEXT("DSN="));
   _tcscat( pszStr, DSName );
   _tcscat( pszStr, _TEXT(";UID="));
   _tcscat( pszStr, Userid );
   _tcscat( pszStr, _TEXT(";PWD="));
   _tcscat( pszStr, Passwd );
   _tcscat( pszStr, _TEXT(";"));
   return ( pszStr );
}


class SqlRC {
public:  
   SqlRC();
   void print( void );
   SqlRC &set( const HENV &hEnv, const HDBC &hDbc, const HSTMT &hStmt );

   SQLINTEGER  getSQLCode ( void ) { return sqlCode; }
   TCHAR *     getSQLState( void ) { return (char *)sqlState; }
   // in dealing with ODBC drivers, sqlstate returned is more important than sqlcode.
   // sqlcode is meaningless until a connection to DB2 is established.
   long        getRC( void ) { return(
               (!_tcscmp( (const char *)sqlState, VW_ODBC_SUCCESS )) ? VW_OK : VW_ODBC_ERROR); }
private:
  	UCHAR       sqlMessage[SQL_MAX_MESSAGE_LENGTH];
	UCHAR       sqlState[SQL_SQLSTATE_SIZE + 1];
	SDWORD      sqlCode;
};
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       SqlRC methods 
// SqlRC constructor **********************************************************
SqlRC::SqlRC( void): sqlCode ( 0 )
{
	sqlMessage[0] = NULL; 
   _tcscpy( (char *)sqlState, VW_ODBC_SUCCESS );
}
// SqlRC Print ****************************************************************
void SqlRC::print( void)
{
   FILE * pfOutFile = NULL;
   pfOutFile = fopen( "TRGAWARD.LOG", "a+" );
   if ( pfOutFile ) {
      unsigned char date[10],time[10];
      fprintf( pfOutFile, "TRGAWARD: %s  %s\n", _strdate((char *)date ), _strtime((char *)time ));
      fprintf( pfOutFile, " ODBC returned SQLCODE = %d, SQLSTATE = %s, System Message: %s\n",
      sqlCode, sqlState, ( _tcslen((const TCHAR *)sqlMessage )) ? sqlMessage : NULL );
      fclose( pfOutFile );
   }
}
// SqlRC set sqlcode, sqlstate, sql message ***********************************
SqlRC &SqlRC::set( const HENV &hEnv, const HDBC &hDbc, const HSTMT &hStmt )
{
   UCHAR    ucSQLState[5];                   // local SQLSTATE
   SDWORD   dwSQLCode;                       // local SQLCODE
   UCHAR    ucErrMsg[SQL_MAX_MESSAGE_LENGTH];// local SQL MESSAGE 
   SWORD    wErrMsgLen;
   RETCODE  retcode = SQLError ( hEnv, hDbc, hStmt, sqlState, &sqlCode,
                                 sqlMessage, SQL_MAX_MESSAGE_LENGTH - 1, &wErrMsgLen );
   if (retcode != SQL_NO_DATA_FOUND) {
      while ( retcode == SQL_SUCCESS )
      {  // pops out all accumulated errors
         retcode = SQLError ( hEnv, hDbc, hStmt, ucSQLState, &dwSQLCode,
                      ucErrMsg, SQL_MAX_MESSAGE_LENGTH - 1, &wErrMsgLen );
      } // end of retriving all the errors accumulated
   }
   return *this;
}
// ODBC Connection object
class SqlOdbc {
public:
   SqlOdbc ( void );
   ~SqlOdbc( void );
   SqlOdbc ( SqlOdbc & Conn ) { hEnv = Conn.hEnv; eEnvState = Conn.eEnvState;
                                hDbc = Conn.hDbc; eDbcState = Conn.eDbcState; }
   virtual RETCODE connect( const TCHAR * connectStr, SqlRC & src );
   virtual SqlRC commit( void );
   virtual SqlRC rollback( void );
   virtual void  disconnect( void );
   virtual HDBC  getHDBC( void ) { return ( hDbc ); }
   virtual state getDBCSTATE( void ) { return ( eDbcState ); }
protected:
   HENV     hEnv;                // odbc environment handle
   HDBC     hDbc;                // odbc connection handle
   state    eEnvState;
   state    eDbcState;
};
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       SqlOdbc Methods 
// SqlOdbc constructor ********************************************************
SqlOdbc::SqlOdbc( void): hDbc(SQL_NULL_HDBC),hEnv(SQL_NULL_HENV),
eEnvState(VW_NOTALLOCATED),eDbcState(VW_NOTALLOCATED)
{
}
// SqlOdbc destructor *********************************************************
SqlOdbc::~SqlOdbc( void )
{
   if ( eDbcState != VW_NOTALLOCATED ) {
      // if process went down unexpectly, first rollback the transaction;   
      //  then drop the open cursor before disconnect from db.
      if ( eDbcState == VW_INUSE )   {
         SQLTransact( hEnv, hDbc, SQL_ROLLBACK );
      }
      SQLFreeConnect( hDbc );
   }
   SQLFreeEnv ( hEnv );
}
// SqlOdbc Connect ************************************************************
RETCODE SqlOdbc::connect( const TCHAR * connectStr, SqlRC & sRC )
{
   RETCODE  retcode;
   unsigned char szConnStrOut[ SQL_MAX_OPTION_STRING_LENGTH ];
   SWORD    cbConnStrOut;

   // allocate env handle if not already allocated
   if ( eEnvState == VW_NOTALLOCATED ) {
      retcode = SQLAllocEnv ( &hEnv );
      if ( retcode == SQL_SUCCESS )    {
         eEnvState = VW_ALLOCATED;
         // allocate connection handle
         retcode = SQLAllocConnect( hEnv, &hDbc );
         if ( retcode == SQL_SUCCESS )    {
            eDbcState = VW_ALLOCATED;
            // establish connection
            retcode = SQLDriverConnect ( 
                              hDbc,     // connection handle
                              NULL,     // no window handle
       (unsigned char *)connectStr,     // connect string
               _tcslen(connectStr),     // length of connect string
                      szConnStrOut,     // returned connect buffer
  SQL_MAX_OPTION_STRING_LENGTH - 1,     // len of ret conn buffer
                     &cbConnStrOut,     // actual len of conn buffer
               SQL_DRIVER_NOPROMPT      // Driver Mgr take it as is
                              );
            if ( retcode == SQL_SUCCESS ) {                                 
               eDbcState = VW_INUSE;
               // AUTOCOMMIT must be set off explicitly.
               // Otherwise SQLSTATE 25000 will result at the time of Disconnect, 
               // because SQLTransact calls would be all ignored.
               retcode =  SQLSetConnectOption( hDbc, 
                                     SQL_AUTOCOMMIT, SQL_AUTOCOMMIT_OFF );
            }
         }
      }
   }
   if ( retcode != SQL_SUCCESS ) sRC.set( hEnv, hDbc, SQL_NULL_HSTMT );
   return retcode;
}
// SqlOdbc Disconnect *********************************************************
void SqlOdbc::disconnect(void)
{
   if ( eDbcState != VW_NOTALLOCATED ) {
      RETCODE retcode;
      // if process went down unexpectly, first rollback the transaction;   
      //  then drop the open cursor before disconnect from db.
      if ( eDbcState == VW_INUSE )   {
         retcode = SQLTransact( hEnv, hDbc, SQL_ROLLBACK );
      }
      retcode = SQLDisconnect( hDbc );
      retcode = SQLFreeConnect( hDbc );
   }
}
// SqlOdbc commit *************************************************************
SqlRC SqlOdbc::commit(void)
{
   SqlRC sRC;
   if ( eDbcState == VW_INUSE )   {
      RETCODE retcode = SQLTransact( hEnv, hDbc, SQL_COMMIT );
      if ( retcode != SQL_SUCCESS ) sRC.set( hEnv, hDbc, SQL_NULL_HSTMT );
      else eDbcState = VW_COMMITTED;
   }
   return sRC;
}
// SqlOdbc rollback ***********************************************************
SqlRC SqlOdbc::rollback(void)
{
   SqlRC sRC;
   if ( eDbcState == VW_INUSE )   {
      RETCODE retcode = SQLTransact( hEnv, hDbc, SQL_ROLLBACK );
      if ( retcode != SQL_SUCCESS ) sRC.set( hEnv, hDbc, SQL_NULL_HSTMT );
      else eDbcState = VW_ROLLBACKED;
   }
   return sRC;
}


// ODBC Cursor object: derived from SqlOdbc
class SqlStmt : public SqlOdbc {
public:
   SqlStmt ( SqlOdbc & Conn );
   ~SqlStmt( void ); 

   state    getState( void ) { return eStmtState; }
   HSTMT    openCursor( void );   // open and return a cursor
   HSTMT    getCursor( void ) { return (eStmtState!=VW_NOTALLOCATED)?hStmt:SQL_NULL_HSTMT; }    
   SqlRC    prepare( UCHAR * instructions );
   SqlRC    execute( void );
   RETCODE  fetch( SqlRC & src );
private:
   HSTMT    hStmt;
   state    eStmtState;
};
// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       SqlStmt Methods 
// SqlStmt constructor ********************************************************
SqlStmt::SqlStmt( SqlOdbc & Conn)
:SqlOdbc( Conn ), hStmt(SQL_NULL_HSTMT),eStmtState(VW_NOTALLOCATED)
{
}
// SqlStmt destructor *********************************************************
SqlStmt::~SqlStmt( void )
{
   if ( eStmtState != VW_NOTALLOCATED ) {
     RETCODE  retcode =  SQLFreeStmt( hStmt, SQL_DROP );
     eStmtState = VW_NOTALLOCATED;
   }
}
// SqlStmt openCursor *********************************************************
HSTMT SqlStmt::openCursor( void )
{
   RETCODE  retcode = SQLAllocStmt ( hDbc, &hStmt );
   if ( retcode == SQL_SUCCESS )    {
      eStmtState = VW_ALLOCATED;
   }
   return ( (retcode == SQL_SUCCESS) ? hStmt : SQL_NULL_HSTMT );
}
// SqlStmt prepare ************************************************************
SqlRC SqlStmt::prepare( UCHAR * instructions ) 
{
   SqlRC    rc;
   RETCODE  retcode = SQLPrepare ( hStmt, instructions, _tcslen((const char*)instructions));
   if ( retcode == SQL_SUCCESS )    {
      eStmtState = eDbcState = VW_INUSE;
   }
   return ( (retcode == SQL_SUCCESS) ? rc : rc.set(hEnv, hDbc, hStmt) );
}
// SqlStmt execute  ***********************************************************
SqlRC SqlStmt::execute( void )
{
   SqlRC    rc;
   RETCODE  retcode = SQLExecute ( hStmt );
   return ( (retcode == SQL_SUCCESS) ? rc : rc.set(hEnv, hDbc, hStmt) );
}
// SqlStmt fetch **************************************************************
RETCODE SqlStmt::fetch( SqlRC & src )
{
   RETCODE  retcode = SQLFetch ( hStmt );
   if (( retcode == SQL_ERROR ) || ( retcode == SQL_SUCCESS_WITH_INFO ))
      src.set(hEnv, hDbc, hStmt);
   return ( retcode );
}

// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++ 
//                       MAIN           
// local function declarations ------------------------------------------------
void print( int rc );
#define Tsel  "SELECT NET_INCOME FROM "


// main ***********************************************************************
int main( int argc, TCHAR *argv[] )
{
   int         rc = VW_OK;
   SqlRC       tRC;
   CommandLine cmdArgs;

   cout << "Enter TrgAward" << endl;
   rc = cmdArgs.commandLineParser( argc, argv );
   if ( !rc )  {  // command line input was parsed and validated
      SqlOdbc DBConnection;
      rc = DBConnection.connect( cmdArgs.getConnectString(), tRC );
      if ( rc == VW_OK ) {  // connection established
         PSQLSTMT  psTgtCursor = new SqlStmt(DBConnection);
         HSTMT     hTgt;
         hTgt = psTgtCursor->openCursor();
         if ( psTgtCursor->getState() == VW_ALLOCATED ) // cursors opened
         {  
            // initialize SELECT statements
            UCHAR tgtStmt[200];
            _tcscpy( (char *) tgtStmt, Tsel );
            _tcscat( (char *) tgtStmt, cmdArgs.getTgtTableName() );
            tRC = psTgtCursor->prepare( tgtStmt );    
            // execute SELECT statement
            if ( (rc = tRC.getRC()) == VW_OK ) 
               tRC = psTgtCursor->execute();
            if ( (rc = tRC.getRC()) == VW_OK ) { //ready to fetch and calculate
               RETCODE  retcode;
               SDWORD   rowCount = 0;
               // Declare and initialize the accumulated income value
               SDWORD totalIncome = 0;
               // Declare storage locations for result set data 
               SDWORD lNet_Income;
               // Declare storage locations for bytes available to return */
               SDWORD cbNet_Income;
               // Bind columns in SELECT result set to storage locations 
               SQLBindCol(hTgt, 1, SQL_C_SLONG, &lNet_Income  ,  0, &cbNet_Income );
               // ready to fetch and calculate
               while  (( retcode = psTgtCursor->fetch( tRC )) == SQL_SUCCESS ) {
                  totalIncome += lNet_Income;
                  rowCount++;
               }
               if ( retcode != SQL_NO_DATA_FOUND ) {
                  // erred during fetch and insert
                  rc = tRC.getSQLCode();        // return execution error
               }
               else if ( totalIncome > 550000 )
                  rc = VW_REACHED_AWARD_LEVEL;
               else
                  rc = VW_NO_AWARD;
               // end if update was successful
            } // endif both source and target are prepared
         } // end if cursors opened successfully
         // free cursors 
         delete psTgtCursor;
         DBConnection.disconnect();
      } // end if connection was successful
   } // end if command line parsing was successful
   // show processing result
   if ( rc == VW_ODBC_ERROR ) rc = tRC.getSQLCode();
   tRC.print();
   print (rc); 
   // after message is printed, return 0 if award level is reached
   if ( rc == VW_REACHED_AWARD_LEVEL ) rc = VW_OK;
   return rc;
} // end of main


// local function print RC ****************************************************
void print( int rc ) 
{
   TCHAR message[200];
   switch ( rc ) {
   case VW_REACHED_AWARD_LEVEL: 
      _tcscpy( message, _TEXT("TrgAward completed successfully and target earning has been reached.") );
      break;
   case VW_NO_AWARD: 
      _tcscpy( message, _TEXT("TrgAward completed but target earning was not reached.") );
      break;
   case VW_ERR_NUM_ARGS:
      _tcscpy( message, _TEXT("ERROR: Invalid number of input arguments.") );
      break;
   case VW_ERR_INV_ARGS:
      _tcscpy( message, _TEXT("ERROR: Invalid argument values.") );
      break;
   default:
      _stprintf( message, _TEXT("SQL ERROR: SQLCODE = %d."), rc );
      break;
   }
   FILE * pfOutFile = NULL;
   pfOutFile = fopen( "TRGAWARD.LOG", "a+" );
   if ( pfOutFile ) {
      unsigned char date[10],time[10];
      _ftprintf( pfOutFile, "\n\nTRGAWARD: %s  %s\n", _strdate((char *)date ), _strtime((char *)time ));
      _ftprintf( pfOutFile, " %s\n", message);
      fclose( pfOutFile );
   }
}

