/**INC+***********************************************************************/
/*                                                                           */
/* Include: nof_c.h                                                          */
/*                                                                           */
/* Purpose: Standard include file for NOF library                            */
/*                                                                           */
/*  COPYRIGHT DATA CONNECTION LIMITED 1993-2006                             */
/*                                                                           */
/**INC-***********************************************************************/

#ifndef NOF_C_INCLUDED
#define NOF_C_INCLUDED

#include <values_c.h>

/*****************************************************************************/
/* Special handling for C++                                                  */
/*****************************************************************************/
#ifdef __cplusplus

extern "C" {

#endif

/*****************************************************************************/
/* NOF API Versions                                                          */
/*****************************************************************************/
#define SNAPIX_API_NOF_VERSION_1        1
#define SNAPIX_API_NOF_VERSION_2        2

/*****************************************************************************/
/* Latest versions of the SNAP-IX APIs.                                      */
/*****************************************************************************/
#define SNAPIX_API_NOF_CURRENT_VERSION      SNAPIX_API_NOF_VERSION_2

#define nof(X, Y)    nof_version(SNAPIX_API_NOF_CURRENT_VERSION, X, Y)

#define nof_async(W, X, Y, Z)   \
                  nof_async_version(SNAPIX_API_NOF_CURRENT_VERSION, W, X, Y, Z)

#if defined (__STDC__) || defined (__cplusplus)
/*****************************************************************************/
/* Do fully specified function declarations for ANSI 'C'                     */
/*****************************************************************************/

/*****************************************************************************/
/* Function type for NOF -> Application callback                             */
/*****************************************************************************/
typedef void (*VNV_CALLBACK) (AP_UINT32        handle,
                              void            *vcb,
                              AP_CORR          correlator,
                              AP_UINT32        indic_length);

/*****************************************************************************/
/* Synchronous entry point for NOF API library                               */
/*****************************************************************************/
extern void nof_version(AP_INT32        version,
                        AP_UINT32       handle,
                        void           *vcb);

/*****************************************************************************/
/* Asynchronous entry point for NOF API library                              */
/*****************************************************************************/
extern AP_UINT16      nof_async_version(AP_INT32        version,
                                        AP_UINT32       handle,
                                        void           *vcb,
                                        VNV_CALLBACK    callback_fn,
                                        AP_CORR         correlator);

#else
/*****************************************************************************/
/* Minimal, for K&R                                                          */
/*****************************************************************************/

/*****************************************************************************/
/* Function type for NOF -> Application callback                             */
/*****************************************************************************/
#define VNV_CALLBACK_VOID void
typedef VNV_CALLBACK_VOID (*VNV_CALLBACK) ();
#undef VNV_CALLBACK_VOID

/*****************************************************************************/
/* Synchronous entry point for NOF API library                               */
/*****************************************************************************/
extern void nof_version();

/*****************************************************************************/
/* Asynchronous entry point for NOF API library                              */
/*****************************************************************************/
extern AP_UINT16      nof_async_version();
#endif

/**STRUCT+********************************************************************/
/* Structure:   TG_DEFINED_CHARS                                             */
/*                                                                           */
/* Description: Defined fields of TG Characteristics Control Vector X'47'    */
/*              (see SNA Formats)                                            */
/*****************************************************************************/

typedef struct tg_defined_chars
{
  unsigned char effect_cap;                      /* effective capacity       */
  unsigned char reserve1[5];                     /* reserved                 */
  unsigned char connect_cost;                    /* connection cost          */
  unsigned char byte_cost;                       /* byte cost                */
  unsigned char reserve2;                        /* reserved                 */
  unsigned char security;                        /* security                 */
  unsigned char prop_delay;                      /* propagation delay        */
  unsigned char modem_class;                     /* modem class              */
  unsigned char user_def_parm_1;                 /* user-defined parameter 1 */
  unsigned char user_def_parm_2;                 /* user-defined parameter 2 */
  unsigned char user_def_parm_3;                 /* user-defined parameter 3 */
} TG_DEFINED_CHARS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: COS_NODE_STATUS                                                */
/*                                                                           */
/* Description: Node status description for COS node rows                    */
/*****************************************************************************/

typedef struct cos_node_status
{
  unsigned char rar;                          /* route additional resistance */
  unsigned char status;                       /* node status.                */
  unsigned char reserv1[2];                   /* reserved                    */
} COS_NODE_STATUS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: COS_NODE_ROW                                                   */
/*                                                                           */
/* Description: COS definition node row                                      */
/*****************************************************************************/

typedef struct cos_node_row
{
  COS_NODE_STATUS minimum;                                       /* minimum  */
  COS_NODE_STATUS maximum;                                       /* maximum  */
  unsigned char   weight;                                        /* weight   */
  unsigned char   reserv1;                                       /* reserved */
} COS_NODE_ROW;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: COS_TG_ROW                                                     */
/*                                                                           */
/* Description: COS definition TG row                                        */
/*****************************************************************************/

typedef struct cos_tg_row
{
  TG_DEFINED_CHARS minimum;                                      /* minimum  */
  TG_DEFINED_CHARS maximum;                                      /* maximum  */
  unsigned char    weight;                                       /* weight   */
  unsigned char    reserv1;                                      /* reserved */
} COS_TG_ROW;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* Maximum number of bytes in ANR labels and NCE labels                      */
/*****************************************************************************/
#define MAX_ANR_LABEL_SIZE              (0x08)

/*****************************************************************************/
/* Link Address                                                              */
/*****************************************************************************/
#define NAP_UNSPECIFIED      0x00
#define NAP_IP_ADDRESS_INFO  0x01

typedef unsigned char MAC_ADDRESS[6];
typedef unsigned char LSAP_ADDRESS;

/**STRUCT+********************************************************************/
/* Structure:   TR_ADDRESS                                                   */
/*                                                                           */
/* Description: Token Ring Address                                           */
/*****************************************************************************/

typedef struct tr_address
{
  unsigned char mac_address[6];                         /* MAC address       */
  unsigned char lsap_address;                           /* local SAP address */
} TR_ADDRESS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   FR_ADDRESS                                                   */
/*                                                                           */
/* Description: Frame Relay Address                                          */
/*****************************************************************************/

typedef struct fr_address
{
  AP_UINT16      dlci;                                  /* DLC Identifier    */
  unsigned char lsap_address;                           /* local SAP address */
} FR_ADDRESS;

/**STRUCT-********************************************************************/

#define MAX_LINK_ADDR_LEN 274

/**STRUCT+********************************************************************/
/* Structure:   IP_ADDRESS_INFO                                              */
/*                                                                           */
/* Description: Format of IP address                                         */
/*****************************************************************************/

typedef struct ip_address_info
{
  unsigned char lsap;                       /* Link Service Access Point     */
#define IP_VERSION_4          (0x40)
#define IP_VERSION_4_HOSTNAME (0x48)
#define IP_VERSION_6          (0x68)        /* Private between IPDLC and UDP */
#define IP_VERSION_6_HOSTNAME (0x60)
#define IP_VERSION_MASK       (0xF8)
  unsigned char version;
  unsigned char address[MAX_LINK_ADDR_LEN-2]; /* IP Address                  */

} IP_ADDRESS_INFO;

/**STRUCT-********************************************************************/

typedef unsigned char SDLC_ADDRESS;

typedef unsigned char X25_ADDRESS[15];

typedef unsigned char GDLC_CHANNEL_ADDRESS[20];

typedef unsigned char GDLC_MPC_ADDRESS[20];

typedef unsigned char SUBCHANNEL_ADDRESS;

/**STRUCT+********************************************************************/
/* Structure:   LINK_ADDRESS                                                 */
/*                                                                           */
/* Description: Link Address Structure                                       */
/*                                                                           */
/* NOTE TO MERGERS                                                           */
/* The length and reserved field have been swapped in this structure to make */
/* the field easier to set up within the Parsing Engine.  This is storing    */
/* up future problems and MUST NOT be taken as a precedent for swapping      */
/* fields in other APPN structures.                                          */
/*****************************************************************************/

typedef struct link_address
{
  unsigned char  format;          /* NAP_UNSPECIFIED or NAP_IP_ADDRESS_INFO  */
  unsigned char  reserve1;                                        /* pad     */
  AP_UINT16      length;                                          /* length  */
  unsigned char  address[MAX_LINK_ADDR_LEN];                      /* address */
} LINK_ADDRESS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   CN_DEF_DATA                                                  */
/*                                                                           */
/* Description: Defined Data for a Connection Network                        */
/*****************************************************************************/

typedef struct cn_def_data
{
  unsigned char    description[32];                 /* resource description  */
  unsigned char    reserve0[16];                    /* reserved              */
  unsigned char    num_ports;                       /* number of ports on CN */
  unsigned char    cn_type;                         /* type of the CN        */
  unsigned char    reserve1[15];                    /* reserved              */
  TG_DEFINED_CHARS tg_chars;                        /* TG characteristics    */
} CN_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   CN_PORT_NAME                                                 */
/*                                                                           */
/* Description: The port name on which is defined a particular CN.           */
/*****************************************************************************/

typedef struct cn_port_name

{
  unsigned char        port_name[8];                /* port name             */

} CN_PORT_NAME;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DEFINE_CN                                                    */
/*                                                                           */
/* Description: Message to add a new CN_CB or update existing one            */
/*****************************************************************************/

typedef struct define_cn
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  fqcn_name[17];                /* name of connection network */
  CN_DEF_DATA    def_data;                     /* CN defined data            */
  unsigned char  port_name[8][8];              /* port names                 */
} DEFINE_CN;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLC_DEF_DATA                                                 */
/*                                                                           */
/* Description: DLC defined data                                             */
/*****************************************************************************/

typedef struct dlc_def_data
{
  unsigned char  description[32];         /* resource description            */
  unsigned char  initially_active;        /* initially active                */
  unsigned char  reserv1[15];             /* reserved                        */
  unsigned char  dlc_type;                /* DLC type                        */
  unsigned char  neg_ls_supp;             /* negotiable link station support */
  unsigned char  port_types;              /* allowable port types            */
  unsigned char  hpr_only;                /* only support HPR?               */
  unsigned char  reserv3;                 /* reserved                        */
  unsigned char  retry_flags;             /* reserved                        */
  AP_UINT16      max_activation_attempts; /* reserved                        */
  AP_UINT16      activation_delay_timer;  /* reserved                        */
  unsigned char  reserv4[4];              /* reserved                        */
  AP_UINT16      dlc_spec_data_len;       /* Length of DLC specific data     */
} DLC_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DEFINE_DLC                                                   */
/*                                                                           */
/* Description: DEFINE_DLC verb control block                                */
/*****************************************************************************/

typedef struct define_dlc
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  dlc_name[8];                       /* name of DLC           */
  DLC_DEF_DATA   def_data;                          /* DLC defined data      */
} DEFINE_DLC;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* Port Number used by DLC to identify the port                              */
/*****************************************************************************/
#define NULL_PORT_NUMBER 0L

/**STRUCT+********************************************************************/
/* Structure:   PORT_DEF_DATA                                                */
/*                                                                           */
/* Description: Port defined data                                            */
/*****************************************************************************/

typedef struct port_def_data
{
  unsigned char    description[32];     /* resource description              */
  unsigned char    initially_active;    /* initially active                  */
  unsigned char    reserv2[15];         /* reserved                          */
  unsigned char    dlc_name[8];         /* DLC name associated with port     */
  unsigned char    port_type;           /* port type                         */
  unsigned char    port_attributes[4];  /* port attributes                   */
  unsigned char    implicit_uplink_to_en; /* implicit EN links up or down?   */
  unsigned char    implicit_appn_links_len; /* reserved                      */
  unsigned char    reserv3;             /* reserved                          */
  AP_UINT32        port_number;         /* port number                       */
  AP_UINT16        max_rcv_btu_size;    /* max receive BTU size              */
  AP_UINT16        tot_link_act_lim;    /* total link activation limit       */
  AP_UINT16        inb_link_act_lim;    /* inbound link activation limit     */
  AP_UINT16        out_link_act_lim;    /* outbound link activation limit    */
  unsigned char    ls_role;             /* initial link station role         */
  unsigned char    retry_flags;         /* reserved                          */
  AP_UINT16        max_activation_attempts; /* reserved                      */
  AP_UINT16        activation_delay_timer;  /* reserved                      */
  unsigned char    mltg_pacing_algorithm;   /* reserved                      */
  unsigned char    implicit_tg_sharing_prohibited; /* reserved               */
  unsigned char    link_spec_data_format; /* reserved                        */
  unsigned char    limit_enable;        /* reserved                          */
  unsigned char    reserv1[6];          /* reserved                          */
  unsigned char    implicit_dspu_template[8]; /* implicit dspu template      */
  AP_UINT16        implicit_ls_limit;   /* implicit ls limit                 */
  unsigned char    reserv4;             /* reserved                          */
  unsigned char    implicit_dspu_services; /* Implicit DSPU support          */
  AP_UINT16        implicit_deact_timer; /* Implicit link HPR link           */
                                         /* deactivation timer               */
  AP_UINT16        act_xid_exchange_limit; /* activation XID exchange limit  */
  AP_UINT16        nonact_xid_exchange_limit; /* non-act. XID exchange limit */
  unsigned char    ls_xmit_rcv_cap;     /* LS transmit-receive capability    */
  unsigned char    max_ifrm_rcvd;       /* maximum number of I-frames that   */
                                        /* can be received                   */
  AP_UINT16        target_pacing_count; /* Target pacing count               */
  AP_UINT16        max_send_btu_size;   /* Desired maximum send BTU size     */
  LINK_ADDRESS     dlc_data;            /* DLC data                          */
  LINK_ADDRESS     hpr_dlc_data;        /* reserved                          */
  unsigned char    implicit_cp_cp_sess_support; /* implicit links allow      */
                                                /*  CP-CP sessions           */
  unsigned char    implicit_limited_resource; /* implicit links are limited  */
                                              /*  resource                   */
  unsigned char    implicit_hpr_support;      /* Implicit links support HPR  */
  unsigned char    implicit_link_lvl_error;   /* Implicit links support HPR  */
                                              /* link-level error recovery   */
  unsigned char    retired1;            /* reserved                          */
  TG_DEFINED_CHARS default_tg_chars;    /* default TG chars                  */
  unsigned char    discovery_supported; /* reserved                          */
  AP_UINT16        port_spec_data_len;  /* length of port specification data */
  AP_UINT16        link_spec_data_len;  /* length of link specification data */
} PORT_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   LS_DEF_DATA                                                  */
/*                                                                           */
/* Description: Link Station defined data                                    */
/*****************************************************************************/

typedef struct ls_def_data
{
  unsigned char    description[32];          /* resource description         */
  unsigned char    initially_active;         /* initially active             */
  unsigned char    reserv2;                  /* reserved                     */
  AP_UINT16        react_timer;              /* Reactivation timer (in secs) */
  AP_UINT16        react_timer_retry;        /* Reactivation timer limit     */
  AP_UINT16        activation_count;         /* reserved                     */
  unsigned char    restart_on_normal_deact;  /* restart the link on any      */
                                             /* failure                      */
  unsigned char    reserv3[7];               /* reserved                     */
  unsigned char    port_name[8];             /* name of associated port      */
  unsigned char    adj_cp_name[17];          /* adjacent CP name             */
  unsigned char    adj_cp_type;              /* adjacent node type           */
  LINK_ADDRESS     dest_address;             /* destination address          */
  unsigned char    auto_act_supp;            /* auto-activate supported      */
  unsigned char    tg_number;                /* TG number                    */
  unsigned char    limited_resource;         /* limited resource             */
  unsigned char    solicit_sscp_sessions;    /* solicit SSCP sessions        */
  unsigned char    pu_name[8];               /* Local PU name (reserved if   */
                                             /* solicit_sscp_sessions is set */
                                             /* to AP_NO)                    */
  unsigned char    disable_remote_act;       /* disable remote activation    */
  unsigned char    dspu_services;            /* Services provided for        */
                                             /* downstream PU                */
  unsigned char    dspu_name[8];             /* Downstream PU name (reserved */
                                             /* if dspu_services is set to   */
                                             /* AP_NONE)                     */
  unsigned char    dlus_name[17];            /* reserved                     */
  unsigned char    bkup_dlus_name[17];       /* reserved                     */
  unsigned char    hpr_supported;            /* does the link support HPR?   */
  unsigned char    hpr_link_lvl_error;       /* does link use link-level     */
                                             /* recovery for HPR frms?       */
  AP_UINT16        link_deact_timer;         /* HPR link deactivation timer  */
  unsigned char    reserv1;                  /* reserved                     */
  unsigned char    default_nn_server;        /* auto-start LS to NN server?  */
  unsigned char    ls_attributes[4];         /* link station attributes      */
  unsigned char    adj_node_id[4];           /* node ID of adjacent node     */
  unsigned char    local_node_id[4];         /* local node ID                */
  unsigned char    cp_cp_sess_support;       /* CP-CP session support        */
  unsigned char    use_default_tg_chars;     /* Use the default tg_chars     */
  TG_DEFINED_CHARS tg_chars;                 /* TG characteristics           */
  AP_UINT16        target_pacing_count;      /* target pacing count          */
  AP_UINT16        max_send_btu_size;        /* maximum send BTU size        */
  unsigned char    ls_role;                  /* link station role            */
  unsigned char    max_ifrm_rcvd;            /* no. before acknowledgement   */
  AP_UINT16        dlus_retry_timeout;       /* seconds to  recontact a DLUS */
  AP_UINT16        dlus_retry_limit;         /* attempts to recontact a DLUS */
  unsigned char    conventional_lu_compression; /* compression required?     */
  unsigned char    conventional_lu_cryptography; /* reserved                 */
  unsigned char    null_address_meaning;     /* reserved                     */
  unsigned char    retry_flags;              /* reserved                     */
  AP_UINT16        max_activation_attempts;  /* reserved                     */
  AP_UINT16        activation_delay_timer;   /* reserved                     */
  unsigned char    branch_link_type;         /* is link an up or down link   */
  unsigned char    adj_brnn_cp_support;      /* adj CP allowed to be BrNN?   */
  unsigned char    mltg_pacing_algorithm;    /* reserved                     */
  unsigned char    reserv5;                  /* reserved                     */
  AP_UINT16        max_rcv_btu_size;         /* reserved                     */
  unsigned char    tg_sharing_prohibited;    /* reserved                     */
  unsigned char    link_spec_data_format;    /* reserved                     */
  unsigned char    pu_can_send_dddlu_offline;/* send DDDLU offline?          */
  unsigned char    reserv4[13];              /* reserved                     */
  AP_UINT16        link_spec_data_len;       /* length of link specific data */
} LS_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DEFINE_LS                                                    */
/*                                                                           */
/* Description: DEFINE_LS verb control block                                 */
/*****************************************************************************/

typedef struct define_ls
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  ls_name[8];                        /* name of link station  */
  LS_DEF_DATA    def_data;                          /* LS defined data       */
} DEFINE_LS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DEFINE_PORT                                                  */
/*                                                                           */
/* Description: DEFINE_PORT verb control block                               */
/*****************************************************************************/

typedef struct define_port
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  port_name[8];                      /* name of port          */
  PORT_DEF_DATA  def_data;                          /* port defined data     */
} DEFINE_PORT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DELETE_CN                                                    */
/*                                                                           */
/* Description: DELETE_CN verb control block                                 */
/*****************************************************************************/

typedef struct delete_cn
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  fqcn_name[17];                /* name of Connection Network */
  unsigned char  reserv1;                      /* reserved                   */
  AP_UINT16      num_ports;                    /* number of ports to delete  */
  unsigned char  port_name[8][8];              /* names of ports to delete   */
} DELETE_CN;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DELETE_DLC                                                   */
/*                                                                           */
/* Description: DELETE_DLC verb control block                                */
/*****************************************************************************/

typedef struct delete_dlc
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  dlc_name[8];                       /* name of DLC           */
} DELETE_DLC;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DELETE_LS                                                    */
/*                                                                           */
/* Description: DELETE_LS verb control block                                 */
/*****************************************************************************/

typedef struct delete_ls
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  ls_name[8];                        /* name of link station  */
} DELETE_LS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DELETE_PORT                                                  */
/*                                                                           */
/* Description: DELETE_PORT verb control block                               */
/*****************************************************************************/

typedef struct delete_port
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  port_name[8];                      /* name of port          */
} DELETE_PORT;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: ROUTING_DATA                                                   */
/*                                                                           */
/* Description: Routing data used in defining a local lu                     */
/*****************************************************************************/

typedef struct routing_data
{
  unsigned char sys_name[128];               /* Name of target system for TP */
  AP_INT32      timeout;                     /* timeout value in seconds     */
  unsigned char back_level;                  /* is LU back level (V4)?       */
  unsigned char reserved[59];                /* reserved                     */
} ROUTING_DATA;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* Minimum value for the timeout defined above.                              */
/*****************************************************************************/
#define V0_DLD_MIN_TIMEOUT 2


/**STRUCT+********************************************************************/
/* Structure:   LOCAL_LU_DEF_DATA                                            */
/*                                                                           */
/* Description: Data used in defining a local lu                             */
/*****************************************************************************/

typedef struct local_lu_def_data
{
  unsigned char  description[32];      /* resource description               */
  unsigned char  reserv1;              /* reserved                           */
  unsigned char  security_list_name[14]; /* Security Access List name        */
  unsigned char  reserv3;              /* reserved                           */
  unsigned char  lu_alias[8];          /* local LU alias                     */
  unsigned char  nau_address;          /* NAU address                        */
  unsigned char  syncpt_support;       /* syncpt supported?                  */
  AP_UINT16      lu_session_limit;     /* LU session limit                   */
  unsigned char  default_pool;         /* in default pool?                   */
  unsigned char  reserv2;              /* reserved                           */
  unsigned char  pu_name[8];           /* PU name                            */
  unsigned char  lu_attributes;        /* LU attributes                      */
  unsigned char  sscp_id[6];           /* SSCP id                            */
  unsigned char  disable;              /* disable?                           */
  ROUTING_DATA   attach_routing_data;  /* routing data for incoming attaches */
  unsigned char  reserv6;              /* reserved                           */
  unsigned char  reserv4[7];           /* reserved                           */
  unsigned char  reserv5[16];          /* reserved                           */
} LOCAL_LU_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_LOCAL_LU                                                */
/*                                                                           */
/* Description: DEFINE_LOCAL_LU verb control block                           */
/*****************************************************************************/

typedef struct define_local_lu
{
  AP_UINT16         opcode;                         /* verb operation code   */
  unsigned char     reserv2;
  unsigned char     format;
  AP_UINT16         primary_rc;                     /* primary return code   */
  AP_UINT32         secondary_rc;                   /* secondary return code */
  unsigned char     lu_name[8];                     /* local LU name         */
  LOCAL_LU_DEF_DATA def_data;                       /* defined data          */
} DEFINE_LOCAL_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_LU62_TIMEOUT                                            */
/*                                                                           */
/* Description: DEFINE_LU62_TIMEOUT verb control block                       */
/*****************************************************************************/
typedef struct define_lu62_timeout
{
  AP_UINT16         opcode;                         /* verb operation code   */
  unsigned char     reserv2;                        /* reserved              */
  unsigned char     format;
  AP_UINT16         primary_rc;                     /* primary return code   */
  AP_UINT32         secondary_rc;                   /* secondary return code */
  unsigned char     resource_type;                  /* resource type         */
  unsigned char     resource_name[17];              /* resource name         */
  AP_UINT16         timeout;                        /* timeout               */
} DEFINE_LU62_TIMEOUT;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   LU_0_TO_3_DEF_DATA                                           */
/*                                                                           */
/* Description: Data used to define a LU_0_TO_3                              */
/*****************************************************************************/

typedef struct lu_0_to_3_def_data
{
  unsigned char description[32];               /* resource description       */
  unsigned char reserv1[16];                   /* reserved                   */
  unsigned char nau_address;                   /* LU NAU address             */
  unsigned char pool_name[8];                  /* LU Pool name               */
  unsigned char pu_name[8];                    /* PU name                    */
  unsigned char priority;                      /* LU priority                */
  unsigned char lu_model;                      /* LU model type              */
  unsigned char sscp_id[6];                    /* SSCP id                    */
  AP_UINT16      timeout;                      /* Timeout                    */
  unsigned char app_spec_def_data[16];         /* application-specified data */
  unsigned char model_name[7];                 /* reserved                   */
  unsigned char term_method;                   /* session termination type   */
  unsigned char disconnect_on_unbind;          /* disconnect on UNBIND flag  */
  unsigned char reserv3[15];                   /* reserved                   */
} LU_0_TO_3_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_LU_0_TO_3                                               */
/*                                                                           */
/* Description: DEFINE_LU_0_TO_3 verb control block                          */
/*****************************************************************************/

typedef struct define_lu_0_to_3
{
  AP_UINT16          opcode;                        /* verb operation code   */
  unsigned char      reserv2;
  unsigned char      format;
  AP_UINT16          primary_rc;                    /* primary return code   */
  AP_UINT32          secondary_rc;                  /* secondary return code */
  unsigned char      lu_name[8];                    /* LU name               */
  LU_0_TO_3_DEF_DATA def_data;                      /* defined data          */
} DEFINE_LU_0_TO_3;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_LU_0_TO_3_RANGE                                         */
/*                                                                           */
/* Description: DEFINE_LU_0_TO_3_RANGE verb control block                    */
/*****************************************************************************/

typedef struct define_lu_0_to_3_range
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  base_name[6];                 /* Base name                  */
  unsigned char  description[32];              /* resource description       */
  unsigned char  reserv1[16];                  /* reserved                   */
  unsigned char  min_nau;                      /* Minimum NAU address        */
  unsigned char  max_nau;                      /* Maximum NAU address        */
  unsigned char  pool_name[8];                 /* LU Pool name               */
  unsigned char  pu_name[8];                   /* PU name                    */
  unsigned char  priority;                     /* LU priority                */
  unsigned char  lu_model;                     /* LU type                    */
  unsigned char  sscp_id[6];                   /* SSCP id                    */
  AP_UINT16      timeout;                      /* Timeout                    */
  unsigned char  app_spec_def_data[16];        /* application-specified data */
  unsigned char  reserv3[7];                   /* reserved                   */
  unsigned char  name_attributes;              /* Extension type             */
  unsigned char  base_number;                  /* First extension number     */
  unsigned char  term_method;                  /* session termination type   */
  unsigned char  disconnect_on_unbind;         /* disconnect on UNBIND flag  */
  unsigned char  reserv4[13];                  /* reserved                   */
} DEFINE_LU_0_TO_3_RANGE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_LU_POOL                                                 */
/*                                                                           */
/* Description: DEFINE_LU_POOL verb control block                            */
/*****************************************************************************/

typedef struct define_lu_pool
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  pool_name[8];                      /* LU pool name          */
  unsigned char  description[32];                   /* resource description  */
  unsigned char  reserv1[16];                       /* reserved              */
  unsigned char  reserv3[4];                        /* reserved              */
  AP_UINT16      num_lus;                           /* number of LUs to add  */
  unsigned char  lu_names[10][8];                   /* LU names              */
} DEFINE_LU_POOL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DOWNSTREAM_LU_DEF_DATA                                       */
/*                                                                           */
/* Description: Parameters used for defining downstream lus                  */
/*****************************************************************************/

typedef struct downstream_lu_def_data
{
  unsigned char description[32];                /* resource description      */
  unsigned char reserv1[16];                    /* reserved                  */
  unsigned char nau_address;                    /* downstream LU nau address */
  unsigned char dspu_name[8];                   /* Downstream PU name        */
  unsigned char host_lu_name[8];                /* Host LU or Pool name      */
  unsigned char allow_timeout;                  /* Allow timeouts?           */
  unsigned char delayed_logon;                  /* Delay logon?              */
  unsigned char reserv2[6];                     /* reserved                 */
} DOWNSTREAM_LU_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_DOWNSTREAM_LU                                           */
/*                                                                           */
/* Description: DEFINE_DOWNSTREAM_LU verb control block                      */
/*****************************************************************************/

typedef struct define_downstream_lu
{
  AP_UINT16              opcode;                    /* verb operation code   */
  unsigned char          reserv2;
  unsigned char          format;
  AP_UINT16              primary_rc;                /* primary return code   */
  AP_UINT32              secondary_rc;              /* secondary return code */
  unsigned char          dslu_name[8];              /* Downstream LU name    */
  DOWNSTREAM_LU_DEF_DATA def_data;                  /* Defined data          */
} DEFINE_DOWNSTREAM_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_DOWNSTREAM_LU_RANGE                                     */
/*                                                                           */
/* Description: DEFINE_DOWNSTREAM_LU_RANGE verb control block                */
/*****************************************************************************/

typedef struct define_downstream_lu_range
{
  AP_UINT16      opcode;                     /* verb operation code          */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                 /* primary return code          */
  AP_UINT32      secondary_rc;               /* secondary return code        */
  unsigned char  dslu_base_name[5];          /* Downstream LU base name      */
  unsigned char  reserv3;                    /* reserved                     */
  unsigned char  description[32];            /* resource description         */
  unsigned char  reserv1[16];                /* reserved                     */
  unsigned char  min_nau;                    /* Minimum NAU address in range */
  unsigned char  max_nau;                    /* Maximum NAU address in range */
  unsigned char  dspu_name[8];               /* Downstream PU name           */
  unsigned char  host_lu_name[8];            /* Host LU or Pool name         */
  unsigned char  allow_timeout;              /* Allow timeouts?              */
  unsigned char  delayed_logon;              /* Delay logon?                 */
  unsigned char  reserv4[6];                 /* reserved                     */
} DEFINE_DOWNSTREAM_LU_RANGE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PRIMARY_LINE_DEF_DATA                                          */
/*                                                                           */
/* Description: Parameters for defining LU0 primary lines                    */
/*****************************************************************************/

typedef struct primary_line_def_data
{
  unsigned char   description[32];
  unsigned char   reserv1[16];
  unsigned char   device_name[5];
  unsigned char   reserv2[3];
  AP_UINT16       max_ifield;
  unsigned char   remote_poll;
  unsigned char   local_poll;
  unsigned char   dlc_type;
  unsigned char   options;
  AP_UINT16       transmit_rate;
} PRIMARY_LINE_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_PRIMARY_LINE                                            */
/*                                                                           */
/* Description: DEFINE_PRIMARY_LINE verb control block                       */
/*****************************************************************************/

typedef struct define_primary_line
{
  AP_UINT16               opcode;
  unsigned char           reserv2;
  unsigned char           format;
  AP_UINT16               primary_rc;
  AP_UINT32               secondary_rc;
  unsigned char           line_name[14];
  PRIMARY_LINE_DEF_DATA   def_data;
} DEFINE_PRIMARY_LINE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_PRIMARY_LINE                                            */
/*                                                                           */
/* Description: DELETE_PRIMARY_LINE verb control block                       */
/*****************************************************************************/

typedef struct delete_primary_line
{
  AP_UINT16       opcode;
  unsigned char   reserv2;
  unsigned char   format;
  AP_UINT16       primary_rc;
  AP_UINT32       secondary_rc;
  unsigned char   line_name[14];
} DELETE_PRIMARY_LINE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_PRIMARY_LINE                                             */
/*                                                                           */
/* Description: QUERY_PRIMARY_LINE verb control block                        */
/*****************************************************************************/

typedef struct query_primary_line
{
  AP_UINT16       opcode;
  unsigned char   reserv2;
  unsigned char   format;
  AP_UINT16       primary_rc;
  AP_UINT32       secondary_rc;
  unsigned char   *buf_ptr;
  AP_UINT32       buf_size;
  AP_UINT32       total_buf_size;
  AP_UINT16       num_entries;
  AP_UINT16       total_num_entries;
  unsigned char   list_options;
  unsigned char   reserv3;
  unsigned char   line_name[14];
} QUERY_PRIMARY_LINE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PRIMARY_LINE_DATA                                              */
/*                                                                           */
/* Description: Overlay returned on QUERY_PRIMARY_LINE                       */
/*****************************************************************************/

typedef struct primary_line_data
{
  AP_UINT16       overlay_size;
  unsigned char   line_name[14];
  PRIMARY_LINE_DEF_DATA def_data;
} PRIMARY_LINE_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PRIMARY_LU0_DEF_DATA                                           */
/*                                                                           */
/* Description: Parameters for defining LU0 primary LUs                      */
/*****************************************************************************/

typedef struct primary_lu0_def_data
{
  unsigned char  description[32];
  unsigned char  reserv1[16];
  unsigned char  default_name[8];
  unsigned char  line_name[14];
  unsigned char  nau_address;
  unsigned char  init_self;
  unsigned char  fm_profile;
  unsigned char  ts_profile;
  unsigned char  ps_profile;
  unsigned char  reserv2;
  unsigned char  trans_protocols[6];
  unsigned char  fm_pri_protocols;
  unsigned char  fm_sec_protocols;
  unsigned char  fm_com_protocols[2];
  AP_UINT16      ps_protocols_len;
  unsigned char  ps_protocols[11];
  unsigned char  reserv3;
  AP_UINT16      user_data_len;
  unsigned char  user_data[24];
  unsigned char  plu_app_path[65];
  unsigned char  api_trace;
} PRIMARY_LU0_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_PRIMARY_LU0                                             */
/*                                                                           */
/* Description: DEFINE_PRIMARY_LU0 verb control block                        */
/*****************************************************************************/

typedef struct define_primary_lu0
{
  AP_UINT16             opcode;
  unsigned char         reserv2;
  unsigned char         format;
  AP_UINT16             primary_rc;
  AP_UINT32             secondary_rc;
  unsigned char         lu_name[8];
  PRIMARY_LU0_DEF_DATA  def_data;
} DEFINE_PRIMARY_LU0;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_PRIMARY_LU0                                             */
/*                                                                           */
/* Description: DELETE_PRIMARY_LU0 verb control block                        */
/*****************************************************************************/

typedef struct delete_primary_lu0
{
  AP_UINT16       opcode;
  unsigned char   reserv2;
  unsigned char   format;
  AP_UINT16       primary_rc;
  AP_UINT32       secondary_rc;
  unsigned char   lu_name[8];
} DELETE_PRIMARY_LU0;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_PRIMARY_LU0                                              */
/*                                                                           */
/* Description: QUERY_PRIMARY_LU0 verb control block                         */
/*****************************************************************************/

typedef struct query_primary_lu0
{
  AP_UINT16       opcode;
  unsigned char   reserv2;
  unsigned char   format;
  AP_UINT16       primary_rc;
  AP_UINT32       secondary_rc;
  unsigned char   *buf_ptr;
  AP_UINT32       buf_size;
  AP_UINT32       total_buf_size;
  AP_UINT16       num_entries;
  AP_UINT16       total_num_entries;
  unsigned char   list_options;
  unsigned char   reserv3;
  unsigned char   lu_name[8];
} QUERY_PRIMARY_LU0;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PRIMARY_LU0_DATA                                               */
/*                                                                           */
/* Description: Overlay returned on QUERY_PRIMARY_LU0                        */
/*****************************************************************************/

typedef struct primary_lu0_data
{
  AP_UINT16       overlay_size;
  unsigned char   lu_name[8];
  PRIMARY_LU0_DEF_DATA  def_data;
} PRIMARY_LU0_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PLU_CHARS                                                      */
/*                                                                           */
/* Description: Partner LU characteristics - passed in on DEFINE_PARTNER_LU  */
/*****************************************************************************/

typedef struct plu_chars
{
  unsigned char  fqplu_name[17];         /* fully qualified partner LU name  */
  unsigned char  plu_alias[8];           /* partner LU alias                 */
  unsigned char  description[32];        /* resource description             */
  unsigned char  reserv2[16];            /* reserved                         */
  unsigned char  plu_un_name[8];         /* partner LU uninterpreted name    */
  unsigned char  preference;             /* reserved                         */
  AP_UINT16      max_mc_ll_send_size;    /* maximum MC send LL size          */
  unsigned char  conv_security_ver;      /* conversation security supported? */
  unsigned char  parallel_sess_supp;     /* parallel sessions supported?     */
  unsigned char  reserv3[8];             /* reserved                         */
} PLU_CHARS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_PARTNER_LU                                              */
/*                                                                           */
/* Description: DEFINE_PARTNER_LU verb control block                         */
/*****************************************************************************/

typedef struct define_partner_lu
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  PLU_CHARS      plu_chars;                    /* partner LU characteristics */
} DEFINE_PARTNER_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: MODE_CHARS                                                     */
/*                                                                           */
/* Description: Mode characteristics - passed in on DEFINE_MODE              */
/*****************************************************************************/

typedef struct mode_chars
{
  unsigned char  description[32];        /* resource description             */
  unsigned char  reserv2[16];            /* reserved                         */
  AP_UINT16      max_ru_size_upp;        /* maximum RU size upper bound      */
  unsigned char  receive_pacing_win;     /* receive pacing window            */
  unsigned char  default_ru_size;        /* default RU size to maximize      */
                                         /* performance                      */
  AP_UINT16      max_neg_sess_lim;       /* maximum negotiable session limit */
  AP_UINT16      plu_mode_session_limit; /* LU-mode session limit            */
  AP_UINT16      min_conwin_src;         /* minimum source contention winner */
                                         /* sessions                         */
  unsigned char  cos_name[8];            /* class of service name            */
  unsigned char  cryptography;           /* cryptography (reserved)          */
  unsigned char  compression;            /* Compression options              */
  AP_UINT16      auto_act;               /* number of sessions to be         */
                                         /* activated automatically          */
  AP_UINT16      min_conloser_src;       /* minimum source contention loser  */
  AP_UINT16      max_ru_size_low;        /* maximum RU size lower bound      */
  AP_UINT16      max_receive_pacing_win; /* maximum receive pacing window    */
  unsigned char  max_compress_lvl;       /* maximum compression level        */
  unsigned char  max_decompress_lvl;     /* maximum decompression level      */
  unsigned char  comp_in_series;         /* reserved                         */
  unsigned char  reserv4[25];            /* reserved                         */
} MODE_CHARS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_MODE                                                    */
/*                                                                           */
/* Description: DEFINE_MODE verb control block                               */
/*****************************************************************************/

typedef struct define_mode
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  mode_name[8];                      /* mode name             */
  AP_UINT16      reserv3;                           /* reserved              */
  MODE_CHARS     mode_chars;                        /* mode characteristics  */
} DEFINE_MODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFAULT_CHARS                                                  */
/*                                                                           */
/* Description: Default characteristics - passed in on DEFINE_DEFAULTS       */
/*****************************************************************************/

typedef struct default_chars
{
  unsigned char   description[32];               /* resource description     */
  unsigned char   reserv2[16];                   /* reserved                 */
  unsigned char   mode_name[8];                  /* mode name                */
  unsigned char   implicit_plu_forbidden;        /* implicit plu forbidden   */
  unsigned char   specific_security_codes;       /* specific security codes  */
  AP_UINT16       limited_timeout;               /* limited resource timeout */
  unsigned char   reserv[244];                   /* reserved                 */
} DEFAULT_CHARS;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_DEFAULTS                                                */
/*                                                                           */
/* Description: DEFINE_DEFAULTS verb control block                           */
/*****************************************************************************/

typedef struct define_defaults
{
  AP_UINT16       opcode;                        /* verb operation code      */
  unsigned char   reserv2;                       /* reserved                 */
  unsigned char   format;                        /* format                   */
  AP_UINT16       primary_rc;                    /* primary return code      */
  AP_UINT32       secondary_rc;                  /* secondary return code    */
  DEFAULT_CHARS   default_chars;                 /* default characteristics  */
} DEFINE_DEFAULTS;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TP_CHARS                                                       */
/*                                                                           */
/* Description: TP characteristics - passed in on DEFINE_TP                  */
/*****************************************************************************/

typedef struct tp_chars
{
  unsigned char  description[32];       /* resource description              */
  unsigned char  security_list_name[14]; /* Security Access List name        */
  unsigned char  reserv1[2];            /* reserved                          */
  unsigned char  conv_type;             /* conversation type                 */
  unsigned char  security_rqd;          /* security support                  */
  unsigned char  sync_level;            /* synchronisation level support     */
  unsigned char  dynamic_load;          /* dynamic load (AP_YES)             */
  unsigned char  enabled;               /* is the TP enabled?                */
  unsigned char  pip_allowed;           /* program initialization parameters */
                                        /*   supported                       */
  unsigned char  reserv3[10];           /* reserved                          */
  AP_UINT16      tp_instance_limit;     /* limit on currently active TP      */
                                        /* instances                         */
  AP_UINT16      incoming_alloc_timeout; /* incoming allocation timeout      */
  AP_UINT16      rcv_alloc_timeout;     /* receive allocation timeout        */
  AP_UINT16      tp_data_len;           /* reserved                          */
  unsigned char  tp_data[120];          /* reserved                          */
} TP_CHARS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TP                                                      */
/*                                                                           */
/* Description: DEFINE_TP verb control block                                 */
/*****************************************************************************/

typedef struct define_tp
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  tp_name[64];                       /* TP name               */
  TP_CHARS       tp_chars;                          /* TP characteristics    */
} DEFINE_TP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   CP_CREATE_PARMS                                              */
/*                                                                           */
/* Description: CP_CREATE_PARMS structure passed in at start of day          */
/*****************************************************************************/

typedef struct cp_create_parms
{
  AP_UINT16      crt_parms_len;        /* length of CP_CREATE_PARMS          */
  unsigned char  description[32];      /* resource description               */
  unsigned char  reserv1[2];           /* reserved                           */
  unsigned char  ms_support;           /* Platform specific use              */
  unsigned char  queue_nmvts;          /* Platform specific use              */
  unsigned char  reserv3[12];          /* reserved                           */
  unsigned char  node_type;            /* node type                          */
  unsigned char  fqcp_name[17];        /* fully qualified CP name            */
  unsigned char  cp_alias[8];          /* CP alias                           */
  unsigned char  mode_to_cos_map_supp; /* mode to COS mapping support        */
  unsigned char  mds_supported;        /* MDS and MS capabilities            */
  unsigned char  node_id[4];           /* node ID                            */
  AP_UINT16      max_locates;          /* maximum locates node can process   */
  AP_UINT16      dir_cache_size;       /* directory cache size (reserved     */
                                       /* if not NN)                         */
  AP_UINT16      max_dir_entries;      /* maximum directory entries (zero    */
                                       /*  means unlimited)                  */
  AP_UINT16      locate_timeout;       /* locate timeout in seconds          */
  unsigned char  reg_with_nn;          /* register resources with NNS        */
  unsigned char  reg_with_cds;         /* register resources with CDS        */
  AP_UINT16      mds_send_alert_q_size; /* size of MDS send alert queue      */
  AP_UINT16      cos_cache_size;       /* number of cos definitions          */
  AP_UINT16      tree_cache_size;      /* Topology Database routing tree     */
                                       /* cache size                         */
  AP_UINT16      tree_cache_use_limit; /* number of times a tree can be used */
  AP_UINT16      max_tdm_nodes;        /* max number of nodes that can be    */
                                       /* stored in Topology Database        */
  AP_UINT16      max_tdm_tgs;          /* max number of TGs that can be      */
                                       /* stored in Topology Database        */
  AP_UINT32      max_isr_sessions;     /* maximum ISR sessions               */
  AP_UINT32      isr_sessions_upper_threshold;        /* upper threshold for */
                                                      /*  ISR sessions       */
  AP_UINT32      isr_sessions_lower_threshold;        /* lower threshold for */
                                                      /*  ISR sessions       */
  AP_UINT16      isr_max_ru_size;      /* max RU size for ISR                */
  AP_UINT16      isr_rcv_pac_window;   /* ISR receive pacing window size     */
  unsigned char  store_endpt_rscvs;    /* endpoint RSCV storage              */
  unsigned char  store_isr_rscvs;      /* ISR RSCV storage                   */
  unsigned char  store_dlur_rscvs;     /* DLUR RSCV storage                  */
  unsigned char  dlur_support;         /* is DLUR supported?                 */
  unsigned char  pu_conc_support;      /* is PU conc supported?              */
  unsigned char  nn_rar;               /* Route additional resistance        */
  unsigned char  hpr_support;          /* level of HPR support               */
  unsigned char  mobile;               /* reserved                           */
  unsigned char  discovery_support;    /* reserved                           */
  unsigned char  discovery_group_name[8];    /* reserved                     */
  unsigned char  implicit_lu_0_to_3;   /* reserved                           */
  unsigned char  default_preference;   /* reserved                           */
  unsigned char  anynet_supported;     /* reserved                           */
#define MAX_LS_EXCEPTION_EVENTS  200
  AP_UINT16      max_ls_exception_events;    /* Max # exception entries      */
  unsigned char  comp_in_series;       /* reserved                           */
  unsigned char  max_compress_lvl;     /* Max compression level (reserved)   */
  unsigned char  node_spec_data_len;   /* reserved                           */
  unsigned char  ptf[64];              /* program temporary fix array        */
  unsigned char  cos_table_version;    /* version of COS tables to be used   */
  unsigned char  send_term_self;       /* default PLU-SLU session term       */
  unsigned char  disable_branch_awareness; /* disable BrNN awareness         */
  unsigned char  cplu_syncpt_support;  /* syncpoint support on CP LU?        */
  unsigned char  cplu_attributes;      /* attributes for CP LU.              */
  unsigned char  maximum_locate_size;  /* maximum locate size in usits of K  */
  AP_UINT16      max_isr_pac_window;   /* max pacing window for ISR adap pac */
  unsigned char  reserved[92];         /* reserved                           */
} CP_CREATE_PARMS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TERM_NODE                                                      */
/*                                                                           */
/* Description: TERM_NODE verb control block                                 */
/*****************************************************************************/

typedef struct term_node
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  stop_type;                         /* stop type             */
} TERM_NODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_COS                                                     */
/*                                                                           */
/* Description: DEFINE_COS verb control block                                */
/*****************************************************************************/

typedef struct define_cos
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  cos_name[8];                       /* class of service name */
  unsigned char  description[32];                   /* resource description  */
  unsigned char  reserv1[16];                       /* reserved              */
  unsigned char  transmission_priority;             /* transmission priority */
  unsigned char  reserv3[9];                        /* reserved              */
  unsigned char  num_of_node_rows;                  /* number of node rows   */
  unsigned char  num_of_tg_rows;                    /* number of TG rows     */
} DEFINE_COS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_COS                                                     */
/*                                                                           */
/* Description: DELETE_COS verb control block                                */
/*****************************************************************************/

typedef struct delete_cos
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  cos_name[8];                       /* class of service name */
} DELETE_COS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_DIRECTORY_ENTRY                                         */
/*                                                                           */
/* Description: DEFINE_DIRECTORY_ENTRY verb control block                    */
/*****************************************************************************/

typedef struct define_directory_entry
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  resource_name[17];       /* network qualified resource name */
  unsigned char  reserv1a;                /* reserved                        */
  AP_UINT16      resource_type;           /* resource type                   */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv3[16];             /* reserved                        */
  unsigned char  parent_name[17];         /* fully qualified parent name     */
  unsigned char  reserv1b;                /* reserved                        */
  AP_UINT16      parent_type;             /* parent's resource type          */
  unsigned char  reserv4[8];              /* reserved                        */
} DEFINE_DIRECTORY_ENTRY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_DIRECTORY_ENTRY                                         */
/*                                                                           */
/* Description: DELETE_DIRECTORY_ENTRY verb control block                    */
/*****************************************************************************/

typedef struct delete_directory_entry
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  resource_name[17];         /* fully qualified resource name */
  unsigned char  reserv3;                   /* reserved                      */
  AP_UINT16      resource_type;             /* resource type                 */
} DELETE_DIRECTORY_ENTRY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_ADJACENT_LEN_NODE                                       */
/*                                                                           */
/* Description: Defines adjacent LEN node                                    */
/*****************************************************************************/

typedef struct define_adjacent_len_node
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  cp_name[17];                       /* CP name               */
  unsigned char  description[32];                   /* resource description  */
  unsigned char  reserv1[16];                       /* reserved              */
  unsigned char  num_of_lus;                        /* number of LUs         */
  unsigned char  wildcard_lus;                      /* wildcard LUs          */
  unsigned char  reserv3[8];                        /* reserved              */
  unsigned char  lu_names[10][8];                   /* LU names              */
} DEFINE_ADJACENT_LEN_NODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_ADJACENT_LEN_NODE                                       */
/*                                                                           */
/* Description:  Deletes an adjacent LEN node                                */
/*****************************************************************************/

typedef struct delete_adjacent_len_node
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  cp_name[17];                       /* CP name               */
  unsigned char  num_of_lus;                        /* number of LUs         */
  unsigned char  lu_names[10][8];                   /* LU names              */
  unsigned char  wildcard_lus;                      /* wildcard LUs          */
} DELETE_ADJACENT_LEN_NODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_LU_LU_PASSWORD                                          */
/*                                                                           */
/* Description: DEFINE_LU_LU_PASSWORD verb control block                     */
/*****************************************************************************/

typedef struct define_lu_lu_password
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  lu_name[8];              /* local LU name                   */
  unsigned char  lu_alias[8];             /* local LU alias                  */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  verification_protocol;   /* verification protocol           */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv1[16];             /* reserved                        */
  unsigned char  reserv3[8];              /* reserved                        */
  unsigned char  password[8];             /* password                        */
} DEFINE_LU_LU_PASSWORD;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LU_LU_PASSWORD                                           */
/*                                                                           */
/* Description: QUERY_LU_LU_PASSWORD verb control block                      */
/*****************************************************************************/
typedef struct query_lu_lu_password
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char *buf_ptr;                 /* buffer pointer                  */
  AP_UINT32      buf_size;                /* buffer length                   */
  AP_UINT32      total_buf_size;          /* returned total buffer length    */
  AP_UINT16      num_entries;             /* no of entries requested/ret'd   */
  AP_UINT16      total_num_entries;       /* returned total no of entries    */
  unsigned char  list_options;            /* list options                    */
  unsigned char  reserv3;                 /* reserved                        */
  unsigned char  lu_name[8];              /* LU name                         */
  unsigned char  lu_alias[8];             /* LU alias                        */
  unsigned char  plu_alias[8];            /* PLU alias                       */
  unsigned char  fqplu_name[17];          /* FQPLU name                      */
} QUERY_LU_LU_PASSWORD;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PASSWORD_INFO                                                  */
/*                                                                           */
/* Description: Information returned for QUERY_LU_LU_PASSWORD                */
/*****************************************************************************/

typedef struct password_info
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  unsigned char  plu_alias[8];                 /* PLU alias                  */
  unsigned char  fqplu_name[17];               /* FQPLU name                 */
  unsigned char  description[32];              /* resource description       */
  unsigned char  reserv1[16];                  /* reserved                   */
  unsigned char  password[8];                  /* password                   */
  unsigned char  protocol_defined;             /* protocol defined           */
  unsigned char  protocol_in_use;              /* protocol in use            */
} PASSWORD_INFO;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: USERID_PASSWORD_CHARS                                          */
/*                                                                           */
/* Description: User characteristics - passed in on DEFINE_USERID_PASSWORD   */
/*****************************************************************************/

typedef struct userid_password_chars
{
  unsigned char  description[32];                  /* resource description   */
  unsigned char  reserv2[16];                      /* reserved               */
  AP_UINT16      profile_count;                    /* number of profiles     */
  AP_UINT16      reserv1;                          /* reserved               */
  unsigned char  password[10];                     /* password               */
  unsigned char  profiles[10][10];                 /* profiles               */
} USERID_PASSWORD_CHARS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_USERID_PASSWORD                                         */
/*                                                                           */
/* Description: DEFINE_USERID_PASSWORD verb control block                    */
/*****************************************************************************/

typedef struct define_userid_password
{
  AP_UINT16             opcode;                  /* verb operation code      */
  unsigned char         reserv2;
  unsigned char         format;
  AP_UINT16             primary_rc;              /* primary return code      */
  AP_UINT32             secondary_rc;            /* secondary return code    */
  unsigned char         define_type;             /* what the define type is  */
  unsigned char         user_id[10];             /* user id                  */
  unsigned char         reserv3[8];              /* reserved                 */
  USERID_PASSWORD_CHARS password_chars;          /* password characteristics */
} DEFINE_USERID_PASSWORD;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_USERID_PASSWORD                                          */
/*                                                                           */
/* Description: QUERY_USERID_PASSWORD verb control block                     */
/*****************************************************************************/

typedef struct query_userid_password
{
  AP_UINT16      opcode;                 /* verb operation code              */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;             /* primary return code              */
  AP_UINT32      secondary_rc;           /* secondary return code            */
  unsigned char *buf_ptr;                /* buffer pointer                   */
  AP_UINT32      buf_size;               /* size of the buffer               */
  AP_UINT32      total_buf_size;         /* returned total buffer size       */
  AP_UINT16      num_entries;            /* no of entries req'd/ret'd        */
  AP_UINT16      total_num_entries;      /* total no of entries              */
  unsigned char  list_options;           /* list options                     */
  unsigned char  reserv3;                /* reserved                         */
  unsigned char  user_id[10];            /* user id                          */
} QUERY_USERID_PASSWORD;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: USERID_INFO                                                    */
/*                                                                           */
/* Description: Information returned for QUERY_USERID_PASSWORD               */
/*****************************************************************************/

typedef struct userid_info
{
  AP_UINT16             overlay_size;    /* overlay size                     */
  unsigned char         user_id[10];     /* user id                          */
  USERID_PASSWORD_CHARS password_chars;  /* user characteristics             */
} USERID_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: CPIC_SIDE_INFO                                                 */
/*                                                                           */
/* Description: CPI-C side information                                       */
/*****************************************************************************/

typedef struct cpic_side_info
{
  unsigned char partner_lu_name[17];      /* Fully qualified partner LU name */
  unsigned char reserved[3];              /* Reserved                        */
  AP_UINT32     tp_name_type;             /* TP name type                    */
  unsigned char tp_name[64];              /* TP name                         */
  unsigned char mode_name[8];             /* Mode name                       */
  AP_UINT32     conversation_security_type; /* Conversation security type    */
  unsigned char security_user_id[10];     /* User ID                         */
  unsigned char security_password[10];    /* Password                        */
  unsigned char lu_alias[8];              /* LU alias                        */
} CPIC_SIDE_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   CPIC_SIDE_INFO_DEF_DATA                                      */
/*                                                                           */
/* Description: Defined Data for CPI-C side information entry                */
/*****************************************************************************/

typedef struct cpic_side_info_def_data
{
  unsigned char  description[32];                  /* resource description   */
  unsigned char  reserv1[16];                      /* reserved               */
  CPIC_SIDE_INFO side_info;                        /* CPIC side info         */
  unsigned char  user_data[24];                    /* User defined data      */
} CPIC_SIDE_INFO_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DEFINE_CPIC_SIDE_INFO                                        */
/*                                                                           */
/* Description: DEFINE_CPIC_SIDE_INFO verb control block                     */
/*****************************************************************************/

typedef struct define_cpic_side_info
{
  AP_UINT16               opcode;               /* verb operation code       */
  unsigned char           reserv2;
  unsigned char           format;
  AP_UINT16               primary_rc;           /* primary return code       */
  AP_UINT32               secondary_rc;         /* secondary return code     */
  unsigned char           reserv2a[8];          /* reserved                  */
  unsigned char           sym_dest_name[8];     /* Symbolic destination name */
  CPIC_SIDE_INFO_DEF_DATA def_data;             /* defined data              */
} DEFINE_CPIC_SIDE_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_LOCAL_LU                                                */
/*                                                                           */
/* Description: DELETE_LOCAL_LU verb control block                           */
/*****************************************************************************/

typedef struct delete_local_lu
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  lu_name[8];                        /* local LU name         */
} DELETE_LOCAL_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_LU62_TIMEOUT                                            */
/*                                                                           */
/* Description: DELETE_LU62_TIMEOUT verb control block                       */
/*****************************************************************************/
typedef struct delete_lu62_timeout
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  resource_type;                     /* resource type         */
  unsigned char  resource_name[17];                 /* resource name         */
} DELETE_LU62_TIMEOUT;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_PARTNER_LU                                              */
/*                                                                           */
/* Description: DELETE_PARTNER_LU verb control block                         */
/*****************************************************************************/

typedef struct delete_partner_lu
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
} DELETE_PARTNER_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_LU_0_TO_3                                               */
/*                                                                           */
/* Description: DELETE_LU_0_TO_3 verb control block                          */
/*****************************************************************************/

typedef struct delete_lu_0_to_3
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  lu_name[8];                        /* LU name               */
} DELETE_LU_0_TO_3;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_LU_0_TO_3_RANGE                                         */
/*                                                                           */
/* Description: DELETE_LU_0_TO_3_RANGE verb control block                    */
/*****************************************************************************/

typedef struct delete_lu_0_to_3_range
{
  AP_UINT16      opcode;                     /* verb operation code          */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                 /* primary return code          */
  AP_UINT32      secondary_rc;               /* secondary return code        */
  unsigned char  base_name[6];               /* Base name                    */
  unsigned char  min_nau;                    /* Minimum NAU address in range */
  unsigned char  max_nau;                    /* Maximum NAU address in range */
  unsigned char  name_attributes;            /* Extension type               */
  unsigned char  base_number;                /* First extension number       */
  unsigned char  reserv5[16];                /* reserved                     */
} DELETE_LU_0_TO_3_RANGE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_LU_POOL                                                 */
/*                                                                           */
/* Description: DELETE_LU_POOL verb control block                            */
/*****************************************************************************/

typedef struct delete_lu_pool
{
  AP_UINT16      opcode;                          /* verb operation code     */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                      /* primary return code     */
  AP_UINT32      secondary_rc;                    /* secondary return code   */
  unsigned char  pool_name[8];                    /* LU pool name            */
  AP_UINT16      num_lus;                         /* Number of specified LUs */
  unsigned char  lu_names[10][8];                 /* LU names                */
} DELETE_LU_POOL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_DOWNSTREAM_LU                                           */
/*                                                                           */
/* Description: DELETE_DOWNSTREAM_LU verb control block                      */
/*****************************************************************************/

typedef struct delete_downstream_lu
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  dslu_name[8];                      /* Downstream LU name    */
} DELETE_DOWNSTREAM_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_DOWNSTREAM_LU_RANGE                                     */
/*                                                                           */
/* Description: DELETE_DOWNSTREAM_LU_RANGE verb control block                */
/*****************************************************************************/

typedef struct delete_downstream_lu_range
{
  AP_UINT16      opcode;                     /* verb operation code          */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                 /* primary return code          */
  AP_UINT32      secondary_rc;               /* secondary return code        */
  unsigned char  dslu_base_name[5];          /* LU base name                 */
  unsigned char  min_nau;                    /* Minimum NAU address in range */
  unsigned char  max_nau;                    /* Maximum NAU address in range */
} DELETE_DOWNSTREAM_LU_RANGE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_MODE                                                    */
/*                                                                           */
/* Description: DELETE_MODE verb control block                               */
/*****************************************************************************/

typedef struct delete_mode
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  mode_name[8];                      /* mode name             */
} DELETE_MODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_TP                                                      */
/*                                                                           */
/* Description: DELETE_TP verb control block                                 */
/*****************************************************************************/

typedef struct delete_tp
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  tp_name[64];                       /* TP name               */
} DELETE_TP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_LU_LU_PASSWORD                                          */
/*                                                                           */
/* Description: DELETE_LU_LU_PASSWORD verb control block                     */
/*****************************************************************************/

typedef struct delete_lu_lu_password
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  lu_name[8];              /* LU name                         */
  unsigned char  lu_alias[8];             /* local LU alias                  */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  reserv3;                 /* reserved                        */
} DELETE_LU_LU_PASSWORD;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_USERID_PASSWORD                                         */
/*                                                                           */
/* Description: DELETE_USERID_PASSWORD verb control block                    */
/*****************************************************************************/

typedef struct delete_userid_password
{
  AP_UINT16             opcode;                  /* verb operation code      */
  unsigned char         reserv2;
  unsigned char         format;
  AP_UINT16             primary_rc;              /* primary return code      */
  AP_UINT32             secondary_rc;            /* secondary return code    */
  AP_UINT16             delete_type;             /* type of delete           */
  unsigned char         user_id[10];             /* user id                  */
  USERID_PASSWORD_CHARS password_chars;          /* password characteristics */
} DELETE_USERID_PASSWORD;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DELETE_CPIC_SIDE_INFO                                        */
/*                                                                           */
/* Description: DELETE_CPIC_SIDE_INFO verb control block                     */
/*****************************************************************************/

typedef struct delete_cpic_side_info
{
  AP_UINT16      opcode;                        /* verb operation code       */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                    /* primary return code       */
  AP_UINT32      secondary_rc;                  /* secondary return code     */
  unsigned char  reserv2a[8];                   /* reserved                  */
  unsigned char  sym_dest_name[8];              /* Symbolic destination name */
} DELETE_CPIC_SIDE_INFO;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* List option variants                                                      */
/*****************************************************************************/
#define  AP_DETAIL                                (0x01)
#define  AP_SUMMARY                               (0x02)
#define  AP_LIST_INCLUSIVE                        (0x04)
#define  AP_LIST_FROM_NEXT                        (0x08)
#define  AP_FIRST_IN_LIST                         (0x10)
#define  AP_LIST_BY_ALIAS                         (0x20)

/**STRUCT+********************************************************************/
/* Structure: QUERY_NODE                                                     */
/*                                                                           */
/* Description: Used to query node details.  Note nn_* fields are reserved   */
/*              when querying an end node and en_* fields are reserved       */
/*              when querying a network node.                                */
/*                                                                           */
/*****************************************************************************/

typedef struct query_node
{
  AP_UINT16       opcode;             /* verb operation code                 */
  unsigned char   reserv2;
  unsigned char   format;
  AP_UINT16       primary_rc;         /* primary return code                 */
  AP_UINT32       secondary_rc;       /* secondary return code               */
  CP_CREATE_PARMS cp_create_parms;    /* create parameters                   */
  AP_UINT32       up_time;            /* time since node started             */
  AP_UINT32       mem_size;           /* reserved                            */
  AP_UINT32       mem_used;           /* reserved                            */
  AP_UINT32       mem_warning_threshold;  /* reserved                        */
  AP_UINT32       mem_critical_threshold; /* reserved                        */
  unsigned char   nn_functions_supported; /* NN functions supported          */
  unsigned char   functions_supported;    /* functions supported             */
  unsigned char   en_functions_supported; /* EN functions supported          */
  unsigned char   nn_status;          /* node status                         */
  AP_UINT32       nn_frsn;            /* NN flow reduction sequence number   */
  AP_UINT32       nn_rsn;             /* Resource sequence number            */
  AP_UINT32       def_ls_good_xids;   /* Good XIDS for defined link stations */
  AP_UINT32       def_ls_bad_xids;    /* Bad XIDS for defined link stations  */
  AP_UINT32       dyn_ls_good_xids;   /* Good XIDS for dynamic link stations */
  AP_UINT32       dyn_ls_bad_xids;    /* Bad XIDS for dynamic link stations  */
  unsigned char   dlur_release_level; /* DLUR release level                  */
  unsigned char   nns_dlus_served_lu_reg_supp; /*                            */
  unsigned char   nns_en_reg_diff_owning_cp;  /*                             */
  unsigned char   reserva[17];        /* reserved                            */
  unsigned char   fq_nn_server_name[17]; /* Fully Qualified NN server name   */
  AP_UINT32       current_isr_sessions; /* Number of ISR sessions            */
  unsigned char   nn_functions2;      /* further NN functions supported      */
  unsigned char   branch_ntwk_arch_version; /* level of BrNN support         */
  unsigned char   reservb[28];        /* reserved                            */
} QUERY_NODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_DEFAULTS                                               */
/*                                                                           */
/* Description: Used to query node defaults.                                 */
/*                                                                           */
/*****************************************************************************/
typedef struct query_defaults
{
  AP_UINT16        opcode;                  /* verb operation code           */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;              /* primary return code           */
  AP_UINT32        secondary_rc;            /* secondary return code         */
  DEFAULT_CHARS    default_chars;           /* default characteristics       */
} QUERY_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_HEADER                                                 */
/*                                                                           */
/* Description: Common header used by all queries which incorporate list     */
/*              options                                                      */
/*****************************************************************************/
typedef struct query_header
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
} QUERY_HEADER;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LOCAL_LU                                                 */
/*                                                                           */
/* Description: Used to obtain information on one or more local LUs          */
/*****************************************************************************/

typedef struct query_local_lu
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  lu_name[8];                   /* LU name                    */
  unsigned char  lu_alias[8];                  /* LU alias                   */
  unsigned char  pu_name[8];                   /* PU name filter             */
} QUERY_LOCAL_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LOCAL_LU_SUMMARY                                               */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_LOCAL_LU (above)          */
/*****************************************************************************/

typedef struct local_lu_summary
{
  AP_UINT16      overlay_size;                     /* overlay size           */
  unsigned char  lu_name[8];                       /* LU name                */
  unsigned char  lu_alias[8];                      /* LU alias               */
  unsigned char  description[32];                  /* resource description   */
  unsigned char  reserv1[16];                      /* reserved               */
} LOCAL_LU_SUMMARY;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: SESSION_STATS                                                  */
/*                                                                           */
/* Description: Session statistics returned in a QUERY_SESSION response      */
/*****************************************************************************/

typedef struct session_stats
{
  AP_UINT16      rcv_ru_size;          /* session receive RU size            */
  AP_UINT16      send_ru_size;         /* session send RU size               */
  AP_UINT16      max_send_btu_size;    /* maximum send BTU size              */
  AP_UINT16      max_rcv_btu_size;     /* maximum rcv BTU size               */
  AP_UINT16      max_send_pac_win;     /* maximum send pacing window size    */
  AP_UINT16      cur_send_pac_win;     /* current send pacing window size    */
  AP_UINT16      max_rcv_pac_win;      /* maximum receive pacing window size */
  AP_UINT16      cur_rcv_pac_win;      /* current receive pacing window size */
  AP_UINT32      send_data_frames;     /* number of data frames sent         */
  AP_UINT32      send_fmd_data_frames; /* num fmd data frames sent           */
  AP_UINT32      send_data_bytes;      /* number of data bytes sent          */
  AP_UINT32      rcv_data_frames;      /* number of data frames received     */
  AP_UINT32      rcv_fmd_data_frames;  /* num fmd data frames received       */
  AP_UINT32      rcv_data_bytes;       /* number of data bytes received      */
  unsigned char  sidh;                 /* session ID high byte (from LFSID)  */
  unsigned char  sidl;                 /* session ID low byte (from LFSID)   */
  unsigned char  odai;                 /* ODAI bit set                       */
  unsigned char  ls_name[8];           /* Link station name                  */
  unsigned char  pacing_type;          /* Pacing Type                        */
} SESSION_STATS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: EXTENDED_SESSION_STATS                                         */
/*                                                                           */
/* Description: Session statistics returned in a QUERY_SESSION response      */
/*****************************************************************************/

typedef struct extended_session_stats
{
  AP_UINT16      rcv_ru_size;          /* session receive RU size            */
  AP_UINT16      send_ru_size;         /* session send RU size               */
  AP_UINT16      max_send_btu_size;    /* maximum send BTU size              */
  AP_UINT16      max_rcv_btu_size;     /* maximum rcv BTU size               */
  AP_UINT16      max_send_pac_win;     /* maximum send pacing window size    */
  AP_UINT16      cur_send_pac_win;     /* current send pacing window size    */
  AP_UINT16      send_rpc;             /* send residual pacing count         */
  AP_UINT16      max_rcv_pac_win;      /* maximum receive pacing window size */
  AP_UINT16      cur_rcv_pac_win;      /* current receive pacing window size */
  AP_UINT16      rcv_rpc;              /* receive residual pacing count      */
  AP_UINT32      send_data_frames;     /* number of data frames sent         */
  AP_UINT32      send_fmd_data_frames; /* num fmd data frames sent           */
  AP_UINT32      send_data_bytes;      /* number of data bytes sent          */
  AP_UINT32      send_fmd_data_bytes;  /* number of fmd data bytes sent      */
  AP_UINT32      rcv_data_frames;      /* number of data frames received     */
  AP_UINT32      rcv_fmd_data_frames;  /* num fmd data frames received       */
  AP_UINT32      rcv_data_bytes;       /* number of data bytes received      */
  AP_UINT32      rcv_fmd_data_bytes;   /* number of fmd data bytes received  */
  unsigned char  sidh;                 /* session ID high byte (from LFSID)  */
  unsigned char  sidl;                 /* session ID low byte (from LFSID)   */
  unsigned char  odai;                 /* ODAI bit set                       */
  unsigned char  ls_name[8];           /* Link station name                  */
  unsigned char  pacing_type;          /* Pacing Type                        */
  unsigned char  reserv1[100];         /* reserved                           */
} EXTENDED_SESSION_STATS;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: LOCAL_LU_DET_DATA                                              */
/*                                                                           */
/* Description: Determined data for Dependent LU 6.2                         */
/*****************************************************************************/

typedef struct local_lu_det_data
{
  unsigned char    lu_sscp_sess_active;     /* Is the LU-SSCP session active?*/
  unsigned char    appl_conn_active;        /* Application is using LU       */
  unsigned char    reserv1[2];              /* reserved                      */
  SESSION_STATS    lu_sscp_stats;           /* LU-SSCP session statistics    */
  unsigned char    sscp_id[6];              /* SSCP id                       */
} LOCAL_LU_DET_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   LOCAL_LU_DETAIL                                              */
/*                                                                           */
/* Description: Overlay returned on QUERY_LOCAL_LU when queried for detail   */
/*              information                                                  */
/*****************************************************************************/

typedef struct local_lu_detail
{
  AP_UINT16         overlay_size;                  /* overlay size           */
  unsigned char     lu_name[8];                    /* LU name                */
  LOCAL_LU_DEF_DATA def_data;                      /* defined data           */
  LOCAL_LU_DET_DATA det_data;                      /* determined data        */
} LOCAL_LU_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LU62_TIMEOUT                                             */
/*                                                                           */
/* Description: Used to obtain information on one or more LU 6.2 timeout     */
/*              definitions                                                  */
/*****************************************************************************/

typedef struct query_lu62_timeout
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char *buf_ptr;                      /* buffer pointer             */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size          */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* list options               */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  resource_type;                /* resource type              */
  unsigned char  resource_name[17];            /* resource name              */
} QUERY_LU62_TIMEOUT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   LU62_TIMEOUT_DATA                                            */
/*                                                                           */
/* Description: Overlay returned on QUERY_LU62_TIMEOUT                       */
/*****************************************************************************/
typedef struct lu62_timeout_data
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  resource_type;           /* resource type                   */
  unsigned char  resource_name[17];       /* resource name                   */
  AP_UINT16      timeout;                 /* timeout                         */
} LU62_TIMEOUT_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_STATISTICS                                             */
/*                                                                           */
/* Description: Used to obtain DLC statistics for a port or link station     */
/*****************************************************************************/

typedef struct query_statistics
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  name[8];                           /* port/link station name*/
  unsigned char  stats_type;                        /* type of statistics    */
  unsigned char  table_type;                        /* type of table         */
  unsigned char  reset_stats;                       /* reset statistics ?    */
  unsigned char  dlc_type;                          /* type of DLC           */
  unsigned char  statistics[256];                   /* current statistics    */
  unsigned char  reserva[20];                       /* reserved              */
} QUERY_STATISTICS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LU_POOL                                                  */
/*                                                                           */
/* Description: Used to obtain information on one or more LU pools           */
/*****************************************************************************/

typedef struct query_lu_pool
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  pool_name[8];                 /* Pool name                  */
  unsigned char  lu_name[8];                   /* LU name                    */
} QUERY_LU_POOL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LU_POOL_SUMMARY                                                */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_LU_POOL (above)           */
/*****************************************************************************/

typedef struct lu_pool_summary
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  unsigned char  pool_name[8];                 /* Pool name                  */
  unsigned char  description[32];              /* resource description       */
  unsigned char  reserv1[16];                  /* reserved                   */
  AP_UINT16      num_active_lus;               /* number of active lus       */
  AP_UINT16      num_avail_lus;                /* number of available lus    */
} LU_POOL_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LU_POOL_DETAIL                                                 */
/*                                                                           */
/* Description: Detail overlay returned for QUERY_LU_POOL (above)            */
/*****************************************************************************/

typedef struct lu_pool_detail
{
  AP_UINT16      overlay_size;                  /* overlay size              */
  unsigned char pool_name[8];                   /* Pool name                 */
  unsigned char description[32];                /* resource description      */
  unsigned char reserv1[16];                    /* reserved                  */
  unsigned char lu_name[8];                     /* LU name                   */
  unsigned char lu_sscp_sess_active;            /* Is LU-SSCP session active */
  unsigned char appl_conn_active;               /* Is appl connection open   */
  unsigned char plu_sess_active;                /* Is PLU-SLU session active */
} LU_POOL_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_PU                                                       */
/*                                                                           */
/* Description: Used to obtain information on one or more PUs and the links  */
/*              associated with them                                         */
/*****************************************************************************/

typedef struct query_pu
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  pu_name[8];                   /* PU name                    */
  unsigned char  host_attachment;              /* reserved                   */
} QUERY_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PU_DATA                                                        */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_PU (above)                */
/*****************************************************************************/

typedef struct pu_data
{
  AP_UINT16      overlay_size;                  /* overlay size              */
  unsigned char pu_name[8];                     /* PU name                   */
  unsigned char description[32];                /* resource description      */
  unsigned char reserv1[16];                    /* reserved                  */
  unsigned char ls_name[8];                     /* LS name                   */
  unsigned char pu_sscp_sess_active;            /* Is PU-SSCP session active */
  unsigned char host_attachment;                /* Host attachment           */
  SESSION_STATS pu_sscp_stats;                  /* reserved                  */
  unsigned char sscp_id[6];                     /* sscp id                   */
  unsigned char conventional_lu_compression;    /* compression required?     */
  unsigned char conventional_lu_cryptography;   /* reserved                  */
  unsigned char dddlu_supported;                /* PU supports DDDLU         */
  unsigned char tcpcv_supported;                /* M/F supports TCP/IP CV 64 */
  unsigned char dddlu_offline_supported;        /* PU supports DDDLU offline */
  unsigned char reserva[9];                     /* reserved                  */
} PU_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TP                                                       */
/*                                                                           */
/* Description: Used for querying TP specific information                    */
/*****************************************************************************/

typedef struct query_tp
{
  AP_UINT16      opcode;                       /* Verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* Primary return code        */
  AP_UINT32      secondary_rc;                 /* Secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  lu_name[8];                   /* LU name                    */
  unsigned char  lu_alias[8];                  /* LU alias                   */
  unsigned char  tp_name[64];                  /* TP name                    */
} QUERY_TP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TP_DATA                                                        */
/*                                                                           */
/* Description: Overlay returned for QUERY_TP (above)                        */
/*****************************************************************************/

typedef struct tp_data
{
  AP_UINT16      overlay_size;                     /* overlay size           */
  unsigned char  tp_name[64];                      /* TP name                */
  unsigned char  description[32];                  /* resource description   */
  unsigned char  reserv1[16];                      /* reserved               */
  AP_UINT16      instance_limit;                   /* maximum instance count */
  AP_UINT16      instance_count;                   /* current instance count */
  AP_UINT16      locally_started_count;            /* reserved               */
  AP_UINT16      remotely_started_count;           /* reserved               */
  unsigned char  reserva[20];                      /* reserved               */
} TP_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_PARTNER_LU                                               */
/*                                                                           */
/* Description: Used to query partner LU information                         */
/*****************************************************************************/

typedef struct query_partner_lu
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  *buf_ptr;                /* pointer to buffer               */
  AP_UINT32      buf_size;                /* buffer size                     */
  AP_UINT32      total_buf_size;          /* total buffer size required      */
  AP_UINT16      num_entries;             /* number of entries               */
  AP_UINT16      total_num_entries;       /* total number of entries         */
  unsigned char  list_options;            /* listing options                 */
  unsigned char  reserv3;                 /* reserved                        */
  unsigned char  lu_name[8];              /* LU name                         */
  unsigned char  lu_alias[8];             /* LU alias                        */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  active_sessions;         /* active sessions only filter     */
} QUERY_PARTNER_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PLU_SUMMARY                                                    */
/*                                                                           */
/* Description: Summary overlay used for QUERY_PARTNER_LU (above)            */
/*****************************************************************************/

typedef struct plu_summary
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  reserv1;                 /* reserved                        */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv2[16];             /* reserved                        */
  AP_UINT16      act_sess_count;          /* currently active sessions count */
  unsigned char  partner_cp_name[17];     /* partner control point name      */
  unsigned char  partner_lu_located;      /* partner LU located yet?         */
} PLU_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PLU_DETAIL                                                     */
/*                                                                           */
/* Description: Detail overlay used for QUERY_PARTNER_LU (above)             */
/*****************************************************************************/

typedef struct plu_detail
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  reserv1;                 /* reserved                        */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv2[16];             /* reserved                        */
  AP_UINT16      act_sess_count;          /* currently active sessions count */
  unsigned char  partner_cp_name[17];     /* partner control point name      */
  unsigned char  partner_lu_located;      /* partner LU located yet?         */
  unsigned char  plu_un_name[8];          /* partner LU uninterpreted name   */
  unsigned char  parallel_sess_supp;      /* parallel sessions supported?    */
  unsigned char  conv_security;           /* conversation security           */
  AP_UINT16      max_mc_ll_send_size;     /* maximum send LL size for mapped */
                                          /* conversations                   */
  unsigned char  implicit;                /* implicit or explicit entry      */
  unsigned char  security_details;        /* session security details        */
  unsigned char  duplex_support;          /* duplex type supported           */
  unsigned char  preference;              /* preference                      */
  unsigned char  reserva[16];             /* reserved                        */
} PLU_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_MODE                                                     */
/*                                                                           */
/* Description: Used for querying mode information                           */
/*****************************************************************************/

typedef struct query_mode
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  *buf_ptr;                /* pointer to buffer               */
  AP_UINT32      buf_size;                /* buffer size                     */
  AP_UINT32      total_buf_size;          /* total buffer size required      */
  AP_UINT16      num_entries;             /* number of entries               */
  AP_UINT16      total_num_entries;       /* total number of entries         */
  unsigned char  list_options;            /* listing options                 */
  unsigned char  reserv3;                 /* reserved                        */
  unsigned char  lu_name[8];              /* LU name                         */
  unsigned char  lu_alias[8];             /* LU alias                        */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  mode_name[8];            /* mode name                       */
  unsigned char  active_sessions;         /* active sessions only filter     */
} QUERY_MODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: MODE_SUMMARY                                                   */
/*                                                                           */
/* Description: Summary overlay used for query_mode (above)                  */
/*****************************************************************************/

typedef struct mode_summary
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  mode_name[8];            /* mode name                       */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv2[16];             /* reserved                        */
  AP_UINT16      sess_limit;              /* current session limit           */
  AP_UINT16      act_sess_count;          /* currently active sessions count */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  reserv1[3];              /* reserved                        */
} MODE_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: MODE_DETAIL                                                    */
/*                                                                           */
/* Description: Detail overlay used for query_mode (above)                   */
/*****************************************************************************/

typedef struct mode_detail
{
  AP_UINT16      overlay_size;           /* overlay size                     */
  unsigned char  mode_name[8];           /* mode name                        */
  unsigned char  description[32];        /* resource description             */
  unsigned char  reserv2[16];            /* reserved                         */
  AP_UINT16      sess_limit;             /* session limit                    */
  AP_UINT16      act_sess_count;         /* currently active sessions count  */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  reserv1[3];              /* reserved                        */
  AP_UINT16      min_conwinners_source;  /* minimum conwinner sess limit     */
  AP_UINT16      min_conwinners_target;  /* minimum conloser sess limit      */
  unsigned char  drain_source;           /* drain source?                    */
  unsigned char  drain_partner;          /* drain partner?                   */
  AP_UINT16      auto_act;               /* auto activated conwinner         */
                                         /* session limit                    */
  AP_UINT16      act_cw_count;           /* active conwinner sessions count  */
  AP_UINT16      act_cl_count;           /* active conloser sessions count   */
  unsigned char  sync_level;             /* synchronization level            */
  unsigned char  default_ru_size;        /* default RU size to maximize      */
                                         /* performance                      */
  AP_UINT16      max_neg_sess_limit;     /* maximum negotiated session limit */
  AP_UINT16      max_rcv_ru_size;        /* maximum receive RU size          */
  AP_UINT16      pending_session_count;  /* pending sess count for mode      */
  AP_UINT16      termination_count;      /* termination count for mode       */
  unsigned char  implicit;               /* implicit or explicit entry       */
  unsigned char  reserva[15];            /* reserved                         */
} MODE_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_SESSION                                                  */
/*                                                                           */
/* Description: Used for querying session info                               */
/*****************************************************************************/

typedef struct query_session
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  *buf_ptr;                /* pointer to buffer               */
  AP_UINT32      buf_size;                /* buffer size                     */
  AP_UINT32      total_buf_size;          /* total buffer size required      */
  AP_UINT16      num_entries;             /* number of entries               */
  AP_UINT16      total_num_entries;       /* total number of entries         */
  unsigned char  list_options;            /* listing options                 */
  unsigned char  reserv3;                 /* reserved                        */
  unsigned char  lu_name[8];              /* LU name                         */
  unsigned char  lu_alias[8];             /* LU alias                        */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  mode_name[8];            /* mode name                       */
  unsigned char  session_id[8];           /* session ID                      */
} QUERY_SESSION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LU_0_TO_3                                                */
/*                                                                           */
/* Description: Used to obtain information on one or more LUs type 0 to 3    */
/*****************************************************************************/

typedef struct query_lu_0_to_3
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  pu_name[8];                   /* PU name filter             */
  unsigned char  lu_name[8];                   /* LU name                    */
  unsigned char  host_attachment;              /* reserved                   */
} QUERY_LU_0_TO_3;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LU_0_TO_3_SUMMARY                                              */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_LU_0_TO_3 (above)         */
/*****************************************************************************/

typedef struct lu_0_to_3_summary
{
  AP_UINT16      overlay_size;               /* overlay size                 */
  unsigned char pu_name[8];                  /* PU name                      */
  unsigned char lu_name[8];                  /* LU name                      */
  unsigned char description[32];             /* resource description         */
  unsigned char reserv1[16];                 /* reserved                     */
  unsigned char nau_address;                 /* NAU address                  */
  unsigned char lu_sscp_sess_active;         /* Is LU-SSCP session active    */
  unsigned char appl_conn_active;            /* Is connection to appl active */
  unsigned char plu_sess_active;             /* Is PLU-SLU session active    */
  unsigned char host_attachment;             /* LU's host attachment         */
} LU_0_TO_3_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SESSION_USER_INFO                                              */
/*                                                                           */
/* Description: Information about the user of this LU of type 0 to 3         */
/*              This field is mapped over the app_spec_det_data of           */
/*              LU_0_TO_3_DET_DATA                                           */
/*****************************************************************************/

typedef struct session_user_info
{
  unsigned char     user_name[32];                  /* 3270 user name        */
  unsigned char     system_name[128];               /* computer name         */
  AP_UINT32         user_pid;                       /* process ID            */
  AP_UINT32         user_type;                      /* user type             */
  AP_UINT32         user_uid;                       /* user ID               */
  AP_UINT32         user_gid;                       /* group ID              */
  unsigned char     user_gname[32];                 /* group name            */
  unsigned char     reserv4[32];                    /* Reserved              */
} SESSION_USER_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LUA_SESSION_USER_INFO                                          */
/*                                                                           */
/* Description: Information about LUA user mapped over the app_spec_data     */
/*****************************************************************************/

typedef struct lua_session_user_info
{
  unsigned char user_ip_address[40];     /* IP address of LUA client/server  */
  unsigned char user_host_address[256];  /* Host name of LUA client/server   */
  unsigned char reserved[24];            /* Reserved                         */
} LUA_SESSION_USER_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LU_0_TO_3_DET_DATA                                             */
/*                                                                           */
/* Description: Determined data for LUs of type 0 to 3                       */
/*****************************************************************************/

typedef struct lu_0_to_3_det_data
{
  unsigned char lu_sscp_sess_active;          /* Is LU-SSCP session active   */
  unsigned char appl_conn_active;             /* Application is using LU     */
  unsigned char plu_sess_active;              /* Is PLU-SLU session active   */
  unsigned char host_attachment;              /* Host attachment             */
  SESSION_STATS lu_sscp_stats;                /* reserved                    */
  SESSION_STATS plu_stats;                    /* reserved                    */
  unsigned char plu_name[8];                  /* PLU name                    */
  unsigned char session_id[8];                /* Internal ID of PLU-SLU sess */
  unsigned char app_spec_det_data[360];       /* application specified data  */
  unsigned char app_type;                     /* application type            */
  unsigned char sscp_id[6];                   /* sscp id                     */
  unsigned char bind_lu_type;                 /* LU type from BIND message   */
  unsigned char compression;                  /* level of compression in use */
  unsigned char cryptography;                 /* reserved                    */
  unsigned char reserva[10];                  /* reserved                    */
} LU_0_TO_3_DET_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LU_0_TO_3_DETAIL                                               */
/*                                                                           */
/* Description: Overlay returned for QUERY_LU_0_TO_3 when detailed data      */
/*              is requeseted                                                */
/*****************************************************************************/

typedef struct lu_0_to_3_detail
{
  AP_UINT16          overlay_size;                        /* overlay size    */
  unsigned char      lu_name[8];                          /* LU name         */
  unsigned char      reserv1[2];                          /* reserved        */
  LU_0_TO_3_DET_DATA det_data;                            /* Determined data */
  LU_0_TO_3_DEF_DATA def_data;                            /* Defined data    */
} LU_0_TO_3_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DOWNSTREAM_PU                                            */
/*                                                                           */
/* Description: Used for querying DLUR and PU Concentration supported        */
/*              downstream PUs                                               */
/*****************************************************************************/

typedef struct query_downstream_pu
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  dspu_name[8];                 /* Downstream PU name filter  */
  unsigned char  dspu_services;                /* reserved                   */
} QUERY_DOWNSTREAM_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DOWNSTREAM_PU_DATA                                             */
/*                                                                           */
/* Description: Overlay returned on QUERY_DOWNSTREAM_PU                      */
/*****************************************************************************/

typedef struct downstream_pu_data
{
  AP_UINT16      overlay_size;              /* overlay size                  */
  unsigned char dspu_name[8];               /* PU name                       */
  unsigned char description[32];            /* resource description          */
  unsigned char reserv1[16];                /* reserved                      */
  unsigned char ls_name[8];                 /* Link name                     */
  unsigned char pu_sscp_sess_active;        /* Is the PU-SSCP session active */
  unsigned char dspu_services;              /* DSPU service type             */
  SESSION_STATS pu_sscp_stats;              /* SSCP-PU session statistics    */
  unsigned char reserva[20];                /* reserved                      */
} DOWNSTREAM_PU_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DOWNSTREAM_LU                                            */
/*                                                                           */
/* Description: Used to obtain information on one or more downstream LUs     */
/*****************************************************************************/

typedef struct query_downstream_lu
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  dspu_name[8];                 /* Downstream PU name filter  */
  unsigned char  dslu_name[8];                 /* Downstream LU name         */
  unsigned char  dspu_services;                /* reserved                   */
} QUERY_DOWNSTREAM_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DOWNSTREAM_LU_DET_DATA                                         */
/*                                                                           */
/* Description: Determined data for Downstream LUs                           */
/*****************************************************************************/

typedef struct downstream_lu_det_data
{
  unsigned char        lu_sscp_sess_active;    /* Is LU-SSCP session active  */
  unsigned char        plu_sess_active;        /* Is PLU-SLU session active  */
  unsigned char        dspu_services;          /* reserved                   */
  unsigned char        reserv1;                /* reserved                   */
  SESSION_STATS        lu_sscp_stats;          /* LU-SSCP session statistics */
  SESSION_STATS        ds_plu_stats;           /* Downstream PLU-SLU stats   */
  SESSION_STATS        us_plu_stats;           /* Upstream PLU-SLU stats     */
  unsigned char        host_lu_name[8];        /* host lu name               */
  unsigned char        host_pu_name[8];        /* host pu name               */
  unsigned char        reserva[4];             /* reserved                   */
} DOWNSTREAM_LU_DET_DATA;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DOWNSTREAM_LU_DETAIL                                         */
/*                                                                           */
/* Description: Overlay returned on QUERY_DOWNSTREAM_LU when detail          */
/*              information is requested                                     */
/*****************************************************************************/

typedef struct downstream_lu_detail
{
  AP_UINT16              overlay_size;        /* overlay size                */
  unsigned char          dslu_name[8];        /* LU name                     */
  unsigned char          reserv1[2];          /* reserved                    */
  DOWNSTREAM_LU_DET_DATA det_data;            /* Determined data             */
  DOWNSTREAM_LU_DEF_DATA def_data;            /* Defined data                */
} DOWNSTREAM_LU_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DOWNSTREAM_LU_SUMMARY                                        */
/*                                                                           */
/* Description: Overlay returned on QUERY_DOWNSTREAM_LU when summary         */
/*              information is requested                                     */
/*****************************************************************************/

typedef struct downstream_lu_summary
{
  AP_UINT16      overlay_size;                /* overlay size                */
  unsigned char dspu_name[8];                 /* PU name                     */
  unsigned char dslu_name[8];                 /* LU name                     */
  unsigned char description[32];              /* resource description        */
  unsigned char reserv1[16];                  /* reserved                    */
  unsigned char dspu_services;                /* Type of service provided to */
                                              /*  downstream node            */
  unsigned char nau_address;                  /* NAU address                 */
  unsigned char lu_sscp_sess_active;          /* Is LU-SSCP session active   */
  unsigned char plu_sess_active;              /* Is PLU-SLU session active   */
} DOWNSTREAM_LU_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: FQPCID                                                         */
/*                                                                           */
/* Description: Fully qualified procedure correlation id structure used      */
/*              in SESSION_INDICATIONS                                       */
/*****************************************************************************/

typedef struct fqpcid
{
  unsigned char pcid[8];                  /* procedure correlator identifier */
  unsigned char fqcp_name[17];            /* originator's network qualified  */
                                          /* CP name                         */
  unsigned char reserve3[3];              /* reserved                        */
} FQPCID;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SESSION_SUMMARY                                                */
/*                                                                           */
/* Description: Summary overlay used for QUERY_SESSION (above)               */
/*****************************************************************************/

typedef struct session_summary
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char plu_alias[8];             /* partner LU alias                */
  unsigned char fqplu_name[17];           /* fully qualified partner LU name */
  unsigned char reserv3[1];               /* reserved                        */
  unsigned char mode_name[8];             /* mode name                       */
  unsigned char session_id[8];            /* session ID                      */
  FQPCID        fqpcid;                   /* fully qualified procedure       */
                                          /* correlator ID                   */
} SESSION_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SESSION_DETAIL                                                 */
/*                                                                           */
/* Description: Detail overlay used for QUERY_SESSION (above)                */
/*****************************************************************************/

#define  AP_CONWINNER                             (0x00)
#define  AP_CONLOSER                              (0x01)

typedef struct session_detail
{
  AP_UINT16      overlay_size;           /* overlay size                     */
  unsigned char plu_alias[8];            /* partner LU alias                 */
  unsigned char fqplu_name[17];          /* fully qualified partner LU name  */
  unsigned char reserv3[1];              /* reserved                         */
  unsigned char mode_name[8];            /* mode name                        */
  unsigned char session_id[8];           /* session ID                       */
  FQPCID        fqpcid;                  /* fully qualified procedure        */
                                         /* correlator ID                    */
  unsigned char cos_name[8];             /* Class of Service name            */
  unsigned char trans_pri;               /* Transmission priority:           */
  unsigned char ltd_res;                 /* Session spans a limited resource */
  unsigned char polarity;                /* Session polarity                 */
  unsigned char contention;              /* Session contention               */
  SESSION_STATS sess_stats;              /* Session statistics               */
  unsigned char duplex_support;          /* duplex type supported            */
  unsigned char sscp_id[6];              /* sscp_id                          */
  unsigned char reserva;                 /* reserved                         */
  AP_UINT32     session_start_time;      /* Session start time               */
  AP_UINT16     session_timeout;         /* Session timeout                  */
  unsigned char cryptography;            /* reserved                         */
  unsigned char reservb[5];              /* reserved                         */
  unsigned char comp_in_series;          /* reserved                         */
  unsigned char plu_slu_comp_lvl;        /* PLU-SLU Compression              */
  unsigned char slu_plu_comp_lvl;        /* SLU-PLU Compression              */
  unsigned char rscv_len;                /* Length of following RSCV         */
} SESSION_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_ISR_SESSION                                              */
/*                                                                           */
/* Description: (NN only) Used for querying ISR information                  */
/*****************************************************************************/

typedef struct query_isr_session
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  session_type;                 /* type of session            */
  FQPCID         fqpcid;                       /* fully qualified procedure  */
                                               /* correlator ID              */
} QUERY_ISR_SESSION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ISR_SESSION_SUMMARY                                            */
/*                                                                           */
/* Description: Summary overlay used for QUERY_ISR_SESSION (above)           */
/*****************************************************************************/

typedef struct isr_session_summary
{
  AP_UINT16      overlay_size;                  /* overlay size              */
  FQPCID         fqpcid;                        /* fully qualified procedure */
                                                /* correlator ID             */
} ISR_SESSION_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ISR_SESSION_DETAIL                                             */
/*                                                                           */
/* Description: Detail overlay used for QUERY_ISR_SESSION (above)            */
/*****************************************************************************/

typedef struct isr_session_detail
{
  AP_UINT16      overlay_size;           /* overlay size                     */
  AP_UINT16      sub_overlay_size;       /* sub-overlay size                 */
  FQPCID        fqpcid;                  /* fully qualified procedure        */
                                         /* correlator ID                    */
  unsigned char trans_pri;               /* Transmission priority:           */
  unsigned char cos_name[8];             /* Class of Service name            */
  unsigned char ltd_res;                 /* Session spans a limited resource */
  unsigned char reserv1[2];              /* reserved                         */
  EXTENDED_SESSION_STATS pri_ext_sess_stats;
                                         /* primary hop session statistics   */
  EXTENDED_SESSION_STATS sec_ext_sess_stats;
                                         /* secondary hop session statistics */
  unsigned char sess_lu_type;            /* LU Type                          */
  unsigned char sess_lu_level;           /* LU Level                         */
  unsigned char pri_tg_number;           /* Primary TG number                */
  unsigned char sec_tg_number;           /* Secondary TG number              */
  AP_UINT32     rtp_tcid;                /* RTP TC ID                        */
  AP_UINT32     time_active;             /* Time active                      */
  unsigned char isr_state;               /* ISR state                        */
  unsigned char reserv2[11];             /* reserved                         */
  unsigned char mode_name[8];            /* Mode name                        */
  unsigned char pri_lu_name[17];         /* Primary LU name                  */
  unsigned char sec_lu_name[17];         /* Secondary LU name                */
  unsigned char pri_adj_cp_name[17];     /* Primary adjacent CP name         */
  unsigned char sec_adj_cp_name[17];     /* Secondary adjacent CP name       */
  unsigned char reserv3[3];              /* reserved                         */
  unsigned char rscv_len;                /* length of following RSCV         */
} ISR_SESSION_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DIRECTORY_STATS                                          */
/*                                                                           */
/* Description: Used for querying directory statistics                       */
/*****************************************************************************/

typedef struct query_directory_stats
{
  AP_UINT16      opcode;              /* verb operation code                 */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;          /* primary return code                 */
  AP_UINT32      secondary_rc;        /* secondary return code               */
  AP_UINT32      max_caches;          /* maximum number of cache entries     */
  AP_UINT32      cur_caches;          /* cache entry count                   */
  AP_UINT32      cur_home_entries;    /* home entry count                    */
  AP_UINT32      cur_reg_entries;     /* registered entry count              */
  AP_UINT32      cur_directory_entries; /* current number of directory ents  */
  AP_UINT32      cache_hits;          /* count of cache finds                */
  AP_UINT32      cache_misses;        /* count of resources found by         */
                                      /* broadcast search (not in cache)     */
  AP_UINT32      in_locates;          /* locates in                          */
  AP_UINT32      in_bcast_locates;    /* broadcast locates in                */
  AP_UINT32      out_locates;         /* locates out                         */
  AP_UINT32      out_bcast_locates;   /* broadcast locates out               */
  AP_UINT32      not_found_locates;   /* unsuccessful locates                */
  AP_UINT32      not_found_bcast_locates; /* unsuccessful broadcast locates  */
  AP_UINT32      locates_outstanding; /* total outstanding locates           */
  unsigned char  reserva[20];         /* reserved                            */
} QUERY_DIRECTORY_STATS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DIRECTORY_ENTRY                                          */
/*                                                                           */
/* Description: Used for extracting directory entries                        */
/*****************************************************************************/

typedef struct query_directory_entry
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  *buf_ptr;                /* pointer to buffer               */
  AP_UINT32      buf_size;                /* buffer size                     */
  AP_UINT32      total_buf_size;          /* total buffer size required      */
  AP_UINT16      num_entries;             /* number of entries               */
  AP_UINT16      total_num_entries;       /* total number of entries         */
  unsigned char  list_options;            /* listing options                 */
  unsigned char  reserv3;                 /* reserved                        */
  unsigned char  resource_name[17];       /* network qualified resource name */
  unsigned char  reserv4;                 /* reserved                        */
  AP_UINT16      resource_type;           /* Resource type                   */
  unsigned char  parent_name[17];         /* parent name filter              */
  unsigned char  reserv5;                 /* reserved                        */
  AP_UINT16      parent_type;             /* parent type                     */
  unsigned char  reserv6[24];             /* reserved                        */
} QUERY_DIRECTORY_ENTRY;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: DIRECTORY_ENTRY_SUMMARY                                        */
/*                                                                           */
/* Description: Summary overlay used for QUERY_DIRECTORY                     */
/*****************************************************************************/

typedef struct directory_entry_summary
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  resource_name[17];       /* network qualified resource name */
  unsigned char  reserve1;                /* reserved                        */
  AP_UINT16      resource_type;           /* Resource type                   */
                                          /*   AP_NNCP_RESOURCE              */
                                          /*   AP_ENCP_RESOURCE              */
                                          /*   AP_LU_RESOURCE                */
                                          /*   AP_WILDCARD_LU_RESOURCE       */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv1[16];             /* reserved                        */
  AP_UINT16      real_owning_cp_type;     /* CP type of real owner           */
  unsigned char  real_owning_cp_name[17]; /* CP name of real owner           */
  unsigned char  reserve2;                /* reserved                        */
} DIRECTORY_ENTRY_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DIRECTORY_ENTRY_DETAIL                                         */
/*                                                                           */
/* Description: Detail overlay used for QUERY_DIRECTORY (above)              */
/*****************************************************************************/

typedef struct directory_entry_detail
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  resource_name[17];       /* network qualified resource name */
  unsigned char  reserv1a;                /* reserved                        */
  AP_UINT16      resource_type;           /* Resource type                   */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv2[16];             /* reserved                        */
  unsigned char  parent_name[17];         /* Network qualified parent name   */
  unsigned char  reserv1b;                /* reserved                        */
  AP_UINT16      parent_type;             /* Parent resource type            */
  unsigned char  entry_type;              /* Type of the directory entry     */
  unsigned char  location;                /* Resource location               */
  AP_UINT16      real_owning_cp_type;
  unsigned char  real_owning_cp_name[17];
  unsigned char  reserv1c;
  AP_UINT16      supplier_cp_type;
  unsigned char  supplier_cp_name[17];
  unsigned char  reserva;                 /* reserved                        */
} DIRECTORY_ENTRY_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DIRECTORY_LU                                             */
/*                                                                           */
/* Description: Used for extracting directory entries about LUs              */
/*****************************************************************************/

typedef struct query_directory_lu
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  lu_name[17];                  /* network qualified lu name  */
} QUERY_DIRECTORY_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DIRECTORY_LU_DETAIL                                            */
/*                                                                           */
/* Description: Detail overlay used for QUERY_DIRECTORY_LU                   */
/*****************************************************************************/

typedef struct directory_lu_detail
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char lu_name[17];              /* network qualified lu name       */
  unsigned char description[32];          /* resource description            */
  unsigned char reserv1[16];              /* reserved                        */
  unsigned char server_name[17];          /* network qualified server name   */
  unsigned char lu_owner_name[17];        /* network qualified lu owner name */
  unsigned char location;                 /* Resource location               */
  unsigned char entry_type;               /* Type of the directory entry     */
  unsigned char wild_card;                /* type of wildcard entry          */
  unsigned char apparent_lu_owner_name[17]; /* name of apparent LU owner     */
  unsigned char reserva[3];               /* reserved                        */
} DIRECTORY_LU_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DIRECTORY_LU_SUMMARY                                           */
/*                                                                           */
/* Description: Summary overlay used for QUERY_DIRECTORY_LU                  */
/*****************************************************************************/

typedef struct directory_lu_summary
{
  AP_UINT16      overlay_size;                  /* overlay size              */
  unsigned char lu_name[17];                    /* network qualified lu name */
  unsigned char description[32];                /* resource description      */
  unsigned char reserv1[16];                    /* reserved                  */
} DIRECTORY_LU_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_NN_TOPOLOGY_STATS                                        */
/*                                                                           */
/* Description: Used for extracting Network Node topology database           */
/*              statistics                                                   */
/*****************************************************************************/

typedef struct query_nn_topology_stats
{
  AP_UINT16      opcode;              /* verb operation code                 */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;          /* primary return code                 */
  AP_UINT32      secondary_rc;        /* secondary return code               */
  AP_UINT32      max_nodes;           /* max number of nodes in database     */
  AP_UINT32      cur_num_nodes;       /* current number of nodes in          */
                                      /* database                            */
  AP_UINT32      node_in_tdus;        /* number of TDUs received             */
  AP_UINT32      node_out_tdus;       /* number of TDUs sent                 */
  AP_UINT32      node_low_rsns;       /* node updates received with low RSNs */
  AP_UINT32      node_equal_rsns;     /* node updates in with equal RSNs     */
  AP_UINT32      node_good_high_rsns; /* node updates in with high RSNs      */
  AP_UINT32      node_bad_high_rsns;  /* node updates in with high and       */
                                      /* odd RSNs                            */
  AP_UINT32      node_state_updates;  /* number of node updates sent         */
  AP_UINT32      node_errors;         /* number of node entry errors found   */
  AP_UINT32      node_timer_updates;  /* number of node records built due to */
                                      /* timer updates                       */
  AP_UINT32      node_purges;         /* number of node records purged       */
  AP_UINT32      tg_low_rsns;         /* TG updates received with low RSNs   */
  AP_UINT32      tg_equal_rsns;       /* TG updates in with equal RSNs       */
  AP_UINT32      tg_good_high_rsns;   /* TG updates in with high RSNs        */
  AP_UINT32      tg_bad_high_rsns;    /* TG updates in with high and         */
                                      /* odd RSNs                            */
  AP_UINT32      tg_state_updates;    /* number of TG updates sent           */
  AP_UINT32      tg_errors;           /* number of TG entry errors found     */
  AP_UINT32      tg_timer_updates;    /* number of node records built due to */
                                      /* timer updates                       */
  AP_UINT32      tg_purges;           /* number of node records purged       */
  AP_UINT32      total_route_calcs;   /* number of routes calculated for COS */
  AP_UINT32      total_route_rejs;    /* number of failed route calculations */
  AP_UINT32      total_tree_cache_hits; /* total number of tree cache hits   */
  AP_UINT32      total_tree_cache_misses; /* total number of tree cache miss */
  AP_UINT32      total_tdu_wars;      /* Total number of TDU wars            */
  unsigned char  reserva[16];         /* reserved                            */
} QUERY_NN_TOPOLOGY_STATS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_ADJACENT_NN                                              */
/*                                                                           */
/* Description: Used to access information about an adjacent Network Node    */
/*              (NN only)                                                    */
/*****************************************************************************/

typedef struct query_adjacent_nn
{
  AP_UINT16       opcode;                /* verb operation code              */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16       primary_rc;            /* primary return code              */
  AP_UINT32       secondary_rc;          /* secondary return code            */
  unsigned char   *buf_ptr;              /* pointer to buffer                */
  AP_UINT32       buf_size;              /* buffer size                      */
  AP_UINT32       total_buf_size;        /* total buffer size required       */
  AP_UINT16       num_entries;           /* number of entries                */
  AP_UINT16       total_num_entries;     /* total number of entries          */
  unsigned char   list_options;          /* listing options                  */
  unsigned char   reserv3;               /* reserved                         */
  unsigned char   adj_nncp_name[17];     /* CP name of adjacent Network Node */
} QUERY_ADJACENT_NN;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ADJ_NNCP_DATA                                                  */
/*                                                                           */
/* Description: Overlay used for QUERY_ADJACENT_NN                           */
/*****************************************************************************/

typedef struct adj_nncp_data
{
  AP_UINT16      overlay_size;           /* overlay size                     */
  unsigned char adj_nncp_name[17];       /* CP name of adjacent network node */
  unsigned char cp_cp_sess_status;       /* CP-CP session status             */
  AP_UINT32     out_of_seq_tdus;         /* out of sequence TDUs             */
  AP_UINT32     last_frsn_sent;          /* last FRSN sent                   */
  AP_UINT32     last_frsn_rcvd;          /* last FRSN received               */
  unsigned char reserva[20];             /* reserved                         */
} ADJ_NNCP_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_NN_TOPOLOGY_NODE                                         */
/*                                                                           */
/* Description: Returns Network Node entries for the Topology Database       */
/*****************************************************************************/

typedef struct query_nn_topology_node
{
  AP_UINT16      opcode;                   /* verb operation code            */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;               /* primary return code            */
  AP_UINT32      secondary_rc;             /* secondary return code          */
  unsigned char  *buf_ptr;                 /* pointer to buffer              */
  AP_UINT32      buf_size;                 /* buffer size                    */
  AP_UINT32      total_buf_size;           /* total buffer size required     */
  AP_UINT16      num_entries;              /* number of entries              */
  AP_UINT16      total_num_entries;        /* total number of entries        */
  unsigned char  list_options;             /* listing options                */
  unsigned char  reserv3;                  /* reserved                       */
  unsigned char  node_name[17];            /* network qualified node name    */
  unsigned char  node_type;                /* node type                      */
  AP_UINT32      frsn;                     /* flow reduction sequence number */
} QUERY_NN_TOPOLOGY_NODE;

/**STRUCT-********************************************************************/


/**STRUCT+****************************************************************** */
/* Structure: NN_TOPOLOGY_NODE_SUMMARY                                       */
/*                                                                           */
/* Description: Summary overlay used by QUERY_NN_TOPOLOGY_NODE (above)       */
/*****************************************************************************/

typedef struct nn_topology_node_summary
{
  AP_UINT16      overlay_size;                /* overlay size                */
  unsigned char node_name[17];                /* network qualified node name */
  unsigned char node_type;                    /* node type                   */
} NN_TOPOLOGY_NODE_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: NN_TOPOLOGY_NODE_DETAIL                                        */
/*                                                                           */
/* Description: Detail overlay used by QUERY_NN_TOPOLOGY_NODE                */
/*****************************************************************************/

typedef struct nn_topology_node_detail
{
  AP_UINT16      overlay_size;             /* overlay size                   */
  unsigned char  node_name[17];            /* network qualified node name    */
  unsigned char  node_type;                /* node type                      */
  AP_UINT16      days_left;                /* days left until entry purged   */
  unsigned char  reserv1[2];               /* reserved                       */
  AP_UINT32      frsn;                     /* flow reduction sequence number */
  AP_UINT32      rsn;                      /* resource sequence number       */
  unsigned char  rar;                      /* route additional resistance    */
  unsigned char  status;                   /* node status                    */
  unsigned char  function_support;         /* function support               */
  unsigned char  reserv2;                  /* reserved                       */
  unsigned char  branch_aware;             /* is node branch aware?          */
  unsigned char  reserva[19];              /* reserved                       */
} NN_TOPOLOGY_NODE_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_NN_TOPOLOGY_TG                                           */
/*                                                                           */
/* Description: Used to access TG topology information from NN topology      */
/*              database                                                     */
/*****************************************************************************/

typedef struct query_nn_topology_tg
{
  AP_UINT16      opcode;                   /* verb operation code            */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;               /* primary return code            */
  AP_UINT32      secondary_rc;             /* secondary return code          */
  unsigned char  *buf_ptr;                 /* pointer to buffer              */
  AP_UINT32      buf_size;                 /* buffer size                    */
  AP_UINT32      total_buf_size;           /* total buffer size required     */
  AP_UINT16      num_entries;              /* number of entries              */
  AP_UINT16      total_num_entries;        /* total number of entries        */
  unsigned char  list_options;             /* listing options                */
  unsigned char  reserv3;                  /* reserved                       */
  unsigned char  owner[17];                /* node that owns the TG          */
  unsigned char  owner_type;               /* type of node that owns the TG  */
  unsigned char  dest[17];                 /* TG destination node            */
  unsigned char  dest_type;                /* TG destination node type       */
  unsigned char  tg_num;                   /* TG number                      */
  unsigned char  reserv1;                  /* reserved                       */
  AP_UINT32      frsn;                     /* flow reduction sequence number */
} QUERY_NN_TOPOLOGY_TG;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: TOPOLOGY_TG_SUMMARY                                            */
/*                                                                           */
/* Description: Summary overlay used by QUERY_NN_TOPOLOGY_TG (above)         */
/*****************************************************************************/

typedef struct topology_tg_summary
{
  AP_UINT16      overlay_size;             /* overlay size                   */
  unsigned char owner[17];                 /* node that owns the TG          */
  unsigned char owner_type;                /* type of node that owns the TG  */
  unsigned char dest[17];                  /* TG destination node            */
  unsigned char dest_type;                 /* TG destination node type       */
  unsigned char tg_num;                    /* TG number                      */
  unsigned char reserv3[1];                /* reserved                       */
  AP_UINT32     frsn;                      /* flow reduction sequence number */
} TOPOLOGY_TG_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TOPOLOGY_TG_DETAIL                                             */
/*                                                                           */
/* Description: Detail overlay used for QUERY_NN_TOPOLOGY_TG (above)         */
/*****************************************************************************/

typedef struct topology_tg_detail
{
  AP_UINT16      overlay_size;             /* overlay size                   */
  unsigned char    owner[17];              /* node that owns the TG          */
  unsigned char    owner_type;             /* type of node that owns the TG  */
  unsigned char    dest[17];               /* TG destination node            */
  unsigned char    dest_type;              /* TG destination node type       */
  unsigned char    tg_num;                 /* TG number                      */
  unsigned char    reserv3[1];             /* reserved                       */
  AP_UINT32        frsn;                   /* flow reduction sequence number */
  AP_UINT16        days_left;              /* days left until entry purged   */
  LINK_ADDRESS     dlc_data;               /* DLC signalling data            */
  AP_UINT32        rsn;                    /* resource sequence number       */
  unsigned char    status;                 /* tg status                      */
  TG_DEFINED_CHARS tg_chars;               /* TG characteristics             */
  unsigned char    subarea_number[4];      /* Subarea number                 */
  unsigned char    tg_type;                /* TG type                        */
  unsigned char    intersubnet_tg;         /* TG between subnets             */
  unsigned char    cp_cp_session_active;   /* Are CP-CP sessions active?     */
  unsigned char    branch_tg;              /* TG branch aware?               */
  unsigned char    multilink_tg;           /* reserved                       */
  unsigned char    appended_data_format;   /* format of appended data        */
  unsigned char    appended_data_len;      /* length of appended data        */
  unsigned char    reserva[9];             /* reserved                       */
} TOPOLOGY_TG_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LOCAL_TOPOLOGY                                           */
/*                                                                           */
/* Description: Used for extracting local topology information for EN and NN */
/*              (uses local_topology overlays below but in EN case the rsn   */
/*               and frsn fields are reserved)                               */
/*****************************************************************************/

typedef struct query_local_topology
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  dest[17];                     /* TG destination node        */
  unsigned char  dest_type;                    /* TG destination node type   */
  unsigned char  tg_num;                       /* TG number                  */
} QUERY_LOCAL_TOPOLOGY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LOCAL_TOPOLOGY_SUMMARY                                         */
/*                                                                           */
/* Description: Summary overlay used by QUERY_LOCAL_TOPOLOGY (above)         */
/*****************************************************************************/

typedef struct local_topology_summary
{
  AP_UINT16      overlay_size;                   /* overlay size             */
  unsigned char dest[17];                        /* TG destination node      */
  unsigned char dest_type;                       /* TG destination node type */
  unsigned char tg_num;                          /* TG number                */
} LOCAL_TOPOLOGY_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LOCAL_TOPOLOGY_DETAIL                                          */
/*                                                                           */
/* Description: Detail overlay used for QUERY_LOCAL_TOPOLOGY (above)         */
/*****************************************************************************/

typedef struct local_topology_detail
{
  AP_UINT16        overlay_size;                 /* overlay size             */
  unsigned char    dest[17];                     /* TG destination node      */
  unsigned char    dest_type;                    /* TG destination node type */
  unsigned char    tg_num;                       /* TG number                */
  unsigned char    reserv1;                      /* reserved                 */
  LINK_ADDRESS     dlc_data;                     /* DLC signalling data      */
  AP_UINT32        rsn;                          /* resource sequence number */
  unsigned char    status;                       /* tg status                */
  TG_DEFINED_CHARS tg_chars;                     /* TG characteristics       */
  unsigned char    cp_cp_session_active;         /* CP-CP sessions active?   */
  unsigned char    branch_link_type;             /* Up or down link?         */
  unsigned char    branch_tg;                    /* Branch TG?               */
  unsigned char    appended_data_format;         /* Format of appended data  */
  unsigned char    appended_data_len;            /* Length of appended data  */
  unsigned char    reserva[11];                  /* reserved                 */
} LOCAL_TOPOLOGY_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_MODE_TO_COS_MAPPING                                      */
/*                                                                           */
/* Description: Extracts mode to COS mappings                                */
/*****************************************************************************/

typedef struct query_mode_to_cos_mapping
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  mode_name[8];                 /* mode name                  */
} QUERY_MODE_TO_COS_MAPPING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: MODE_TO_COS_MAPPING_DATA                                       */
/*                                                                           */
/* Description: Overlay used by QUERY_MODE_TO_COS_MAPPING                    */
/*****************************************************************************/

typedef struct mode_to_cos_mapping_data
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  unsigned char  mode_name[8];                 /* mode name                  */
  unsigned char  cos_name[8];                  /* cos name                   */
  unsigned char  reserva[20];                  /* reserved                   */
} MODE_TO_COS_MAPPING_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_CONVERSATION                                             */
/*                                                                           */
/* Description: Used to query the node for conversation information          */
/*****************************************************************************/

typedef struct query_conversation
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char *buf_ptr;                      /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  lu_name[8];                   /* LU Name                    */
  unsigned char  lu_alias[8];                  /* LU Alias                   */
  AP_UINT32      conv_id;                      /* Conversation ID            */
  unsigned char  session_id[8];                /* Session ID                 */
  unsigned char  reserv4[12];                  /* reserved                   */
} QUERY_CONVERSATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: CONV_SUMMARY                                                   */
/*                                                                           */
/* Description: Summary overlay used by QUERY_CONVERSATION                   */
/*****************************************************************************/

typedef struct conv_summary
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  AP_UINT32      conv_id;                      /* conversation ID            */
  unsigned char  local_tp_name[64];            /* local TP name              */
  unsigned char  partner_tp_name[64];          /* partner TP name            */
  unsigned char  tp_id[8];                     /* TP ID                      */
  unsigned char  sess_id[8];                   /* Session ID                 */
  AP_UINT32      conv_start_time;              /* Conversation start time    */
  AP_UINT32      bytes_sent;                   /* Number of bytes sent       */
  AP_UINT32      bytes_received;               /* Number of bytes received   */
  unsigned char  conv_state;                   /* conversation state         */
  unsigned char  duplex_type;                  /* duplex type                */
} CONV_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_COS                                                      */
/*                                                                           */
/* Description: Extracts Class Of Service specific information               */
/*****************************************************************************/

typedef struct query_cos
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  cos_name[8];                  /* cos name                   */
} QUERY_COS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: COS_DATA                                                       */
/*                                                                           */
/* Description: Overlay used for QUERY_COS (above)                           */
/*****************************************************************************/

typedef struct cos_data
{
  AP_UINT16      overlay_size;           /* overlay size                     */
  unsigned char  cos_name[8];            /* cos name                         */
  unsigned char  description[32];        /* resource description             */
  unsigned char  reserv1[16];            /* reserved                         */
  unsigned char  transmission_priority;  /* transmission priority            */
  AP_UINT16      num_of_node_rows;       /* number of node rows              */
  AP_UINT16      num_of_tg_rows;         /* number of tg rows                */
  AP_UINT32      trees;                  /* number of tree caches for COS    */
  AP_UINT32      calcs;                  /* number of route calculations for */
                                         /* this COS                         */
  AP_UINT32      rejs;                   /* number of route rejects for COS  */
  unsigned char  reserva[20];            /* reserved                         */
} COS_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_COS_NODE_ROW                                             */
/*                                                                           */
/* Description: Extracts Class Of Service node row information               */
/*****************************************************************************/

typedef struct query_cos_node_row
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  cos_name[8];                  /* cos name                   */
  AP_UINT16      node_row_index;               /* node row index             */
} QUERY_COS_NODE_ROW;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: COS_NODE_ROW_DATA                                              */
/*                                                                           */
/* Description: Overlay used for QUERY_COS_NODE_ROW                          */
/*****************************************************************************/

typedef struct cos_node_row_data
{
  AP_UINT16      overlay_size;                   /* overlay size             */
  unsigned char  cos_name[8];                    /* cos name                 */
  AP_UINT16      node_row_index;                 /* node row index           */
  COS_NODE_ROW   node_row;                       /* cos node row information */
} COS_NODE_ROW_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_COS_TG_ROW                                               */
/*                                                                           */
/* Description: Extracts COS TG row                                          */
/*****************************************************************************/

typedef struct query_cos_tg_row
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  cos_name[8];                  /* cos name                   */
  AP_UINT16      tg_row_index;                 /* TG row index               */
} QUERY_COS_TG_ROW;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: COS_TG_ROW_DATA                                                */
/*                                                                           */
/* Description: Overlay used for QUERY_COS_TG_ROW                            */
/*****************************************************************************/

typedef struct cos_tg_row_data
{
  AP_UINT16      overlay_size;                         /* overlay size       */
  unsigned char  cos_name[8];                          /* cos name           */
  AP_UINT16      tg_row_index;                         /* TG row index       */
  COS_TG_ROW     tg_row;                               /* TG row information */
} COS_TG_ROW_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DLC_SUMMARY                                                    */
/*                                                                           */
/* Description: Summary of DLC information used by QUERY_DLC (above)         */
/*****************************************************************************/

typedef struct dlc_summary
{
  AP_UINT16      overlay_size;                      /* overlay size          */
  unsigned char dlc_name[8];                        /* name of DLC           */
  unsigned char description[32];                    /* resource description  */
  unsigned char reserv1[16];                        /* reserved              */
  unsigned char state;                              /* State of the DLC      */
  unsigned char dlc_type;                           /* DLC type              */
}DLC_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DLC_DET_DATA                                                   */
/*                                                                           */
/* Description: Determined data for DLC                                      */
/*****************************************************************************/

typedef struct dlc_det_data
{
  unsigned char state;                                   /* State of the DLC */
  unsigned char reserv3[3];                              /* reserved         */
  unsigned char reserva[20];                             /* reserved         */
}DLC_DET_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLC_DETAIL                                                   */
/*                                                                           */
/* Description: Overlay returned for QUERY_DLC when DETAIL information       */
/*              is requested                                                 */
/*****************************************************************************/

typedef struct dlc_detail
{
  AP_UINT16      overlay_size;                            /* overlay size    */
  unsigned char dlc_name[8];                              /* name of DLC     */
  unsigned char reserv2[2];                               /* reserved        */
  DLC_DET_DATA  det_data;                                 /* Determined data */
  DLC_DEF_DATA  def_data;                                 /* Defined data    */
} DLC_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PORT_SUMMARY                                                   */
/*                                                                           */
/* Description: Summary overlay used for QUERY_PORT (above)                  */
/*****************************************************************************/

typedef struct port_summary
{
  AP_UINT16      overlay_size;                      /* overlay size          */
  unsigned char port_name[8];                       /* port name             */
  unsigned char description[32];                    /* resource description  */
  unsigned char reserv2[16];                        /* reserved              */
  unsigned char port_state;                         /* port state            */
  unsigned char reserv1[1];                         /* reserved              */
  unsigned char dlc_name[8];                        /* name of DLC           */
} PORT_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   PORT_DET_DATA                                                */
/*                                                                           */
/* Description: Determined data during life of port                          */
/*****************************************************************************/

typedef struct port_det_data
{
  unsigned char  port_state;          /* port state                          */
  unsigned char  dlc_type;            /* DLC type                            */
  unsigned char  port_sim_rim;        /* port initialization options         */
  unsigned char  reserv1;             /* reserved                            */
  AP_UINT16      def_ls_good_xids;    /* number of successful XIDs           */
  AP_UINT16      def_ls_bad_xids;     /* number of unsuccessful XIDs         */
  AP_UINT16      dyn_ls_good_xids;    /* successful XIDs on dynamic LS count */
  AP_UINT16      dyn_ls_bad_xids;     /* failed XIDs on dynamic LS count     */
  AP_UINT16      num_implicit_links;  /* number of implicit links            */
  unsigned char  neg_ls_supp;         /* negotiable?                         */
  unsigned char  abm_ls_supp;         /* ABM support?                        */
  AP_UINT32      start_time;          /* Start time of the port.             */
  unsigned char  reserva[12];         /* reserved                            */
} PORT_DET_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PORT_DETAIL                                                    */
/*                                                                           */
/* Description: Detailed overlay used for QUERY_PORT (above)                 */
/*****************************************************************************/

typedef struct port_detail
{
  AP_UINT16      overlay_size;                            /* overlay size    */
  unsigned char  port_name[8];                            /* port name       */
  unsigned char  reserv1[2];                              /* reserved        */
  PORT_DET_DATA  det_data;                                /* determined data */
  PORT_DEF_DATA  def_data;                                /* defined data    */
} PORT_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   ANR_LABEL                                                    */
/*                                                                           */
/* Description: A single ANR label, up to 8 bytes AP_INT32                   */
/* This structure is only used to store ANR labels assigned by nodes other   */
/* than this one, since all SNAP APPN ANR labels are 1 or two bytes          */
/*****************************************************************************/

typedef struct anr_label
{
  AP_UINT16       length;                   /* Length of label               */
  AP_UINT16       reserv;                   /* reserved                      */
  unsigned char   label[MAX_ANR_LABEL_SIZE]; /* Label data                   */
} ANR_LABEL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LS_SUMMARY                                                     */
/*                                                                           */
/* Description: Structure of summary info returned on a query ls             */
/*****************************************************************************/

typedef struct ls_summary
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  ls_name[8];              /* link station name               */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv1[16];             /* reserved                        */
  unsigned char  dlc_type;                /* DLC type                        */
  unsigned char  state;                   /* link station state              */
  AP_UINT16      act_sess_count;          /* currently active sessions count */
  unsigned char  det_adj_cp_name[17];     /* determined adjacent CP name     */
  unsigned char  det_adj_cp_type;         /* determined adjacent node type   */
  unsigned char  port_name[8];            /* port name                       */
  unsigned char  adj_cp_name[17];         /* adjacent CP name                */
  unsigned char  adj_cp_type;             /* adjacent node type              */
} LS_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LS_STATS                                                       */
/*                                                                           */
/* Description: LS statistics                                                */
/*****************************************************************************/

typedef struct ls_stats
{
  AP_UINT32     in_xid_bytes;           /* number of XID bytes received      */
  AP_UINT32     in_msg_bytes;           /* number of message bytes received  */
  AP_UINT32     in_xid_frames;          /* number of XID frames received     */
  AP_UINT32     in_msg_frames;          /* number of message frames received */
  AP_UINT32     out_xid_bytes;          /* number of XID bytes sent          */
  AP_UINT32     out_msg_bytes;          /* number of message bytes sent      */
  AP_UINT32     out_xid_frames;         /* number of XID frames sent         */
  AP_UINT32     out_msg_frames;         /* number of message frames sent     */
  AP_UINT32     in_invalid_sna_frames;  /* number of invalid frames received */
  AP_UINT32     in_session_control_frames; /* number of control frames rcvd  */
  AP_UINT32     out_session_control_frames; /* number of control frames sent */
  AP_UINT32     echo_rsps;              /* reserved                          */
  AP_UINT32     current_delay;          /* reserved                          */
  AP_UINT32     max_delay;              /* reserved                          */
  AP_UINT32     min_delay;              /* reserved                          */
  AP_UINT32     max_delay_time;         /* reserved                          */
  AP_UINT32     good_xids;              /* successful XID on LS count        */
  AP_UINT32     bad_xids;               /* unsuccessful XID on LS count      */
} LS_STATS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   LS_DET_DATA                                                  */
/*                                                                           */
/* Description: Data determined during life of LS                            */
/*****************************************************************************/

typedef struct ls_det_data
{
  AP_UINT16      act_sess_count;          /* currently active sessions count */
  unsigned char  dlc_type;                /* DLC type                        */
  unsigned char  state;                   /* link station state              */
  unsigned char  sub_state;               /* link station sub state          */
  unsigned char  det_adj_cp_name[17];     /* adjacent CP name                */
  unsigned char  det_adj_cp_type;         /* adjacent node type              */
  unsigned char  dlc_name[8];             /* name of DLC                     */
  unsigned char  dynamic;                 /* specifies whether LS is dynamic */
  unsigned char  migration;               /* supports migration partners     */
  unsigned char  tg_num;                  /* TG number                       */
  LS_STATS       ls_stats;                /* link station statistics         */
  AP_UINT32      start_time;              /* time LS started                 */
  AP_UINT32      stop_time;               /* time LS stopped                 */
  AP_UINT32      up_time;                 /* total time LS active            */
  AP_UINT32      current_state_time;      /* time in current state           */
  unsigned char  deact_cause;             /* deactivation cause              */
  unsigned char  hpr_support;             /* TG HPR support                  */
  unsigned char  anr_label[2];            /* local ANR label                 */
  unsigned char  hpr_link_lvl_error;      /* HPR link-level error            */
  unsigned char  auto_act;                /* is the LS auto-activatable?     */
  unsigned char  ls_role;                 /* LS role                         */
  unsigned char  ls_type;                 /* LS type                         */
  unsigned char  node_id[4];              /* Node id                         */
  AP_UINT16      active_isr_count;        /* active isr count                */
  AP_UINT16      active_lu_sess_count;    /* Count of active LU sessions     */
  AP_UINT16      active_sscp_sess_count;  /* Count of active SSCP sessions   */
  ANR_LABEL      reverse_anr_label;       /* Reverse ANR label               */
  LINK_ADDRESS   local_address;           /* Local address                   */
  AP_UINT16      max_send_btu_size;       /* Max send BTU size               */
  unsigned char  brnn_link_type;          /* type of branch link             */
  unsigned char  adj_cp_is_brnn;          /* is adjacent node a BrNN?        */
  unsigned char  mltg_member;             /* reserved                        */
  unsigned char  tg_sharing;              /* reserved                        */
  unsigned char  partner_cn_tg_number;    /* reserved                        */
  unsigned char  local_cn_tg_number;      /* reserved                        */
  unsigned char  reservb[60];             /* reserved                        */
} LS_DET_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   LS_DETAIL                                                    */
/*                                                                           */
/* Description: Overlay returned for a QUERY_LS when DETAIL information      */
/*              is requested                                                 */
/*****************************************************************************/

typedef struct ls_detail
{
  AP_UINT16      overlay_size;                          /* overlay size      */
  unsigned char ls_name[8];                             /* link station name */
  LS_DET_DATA   det_data;                               /* determined data   */
  LS_DEF_DATA   def_data;                               /* defined data      */
} LS_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_MODE_DEFINITION                                          */
/*                                                                           */
/* Description: Extracts mode definition information                         */
/*****************************************************************************/

typedef struct query_mode_definition
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  mode_name[8];                 /* mode name                  */
} QUERY_MODE_DEFINITION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: MODE_DEF_SUMMARY                                               */
/*                                                                           */
/* Description: Summary information returned on QUERY_MODE_DEF (above)       */
/*****************************************************************************/

typedef struct mode_def_summary
{
  AP_UINT16      overlay_size;                    /* overlay size            */
  unsigned char  mode_name[8];                    /* mode name               */
  unsigned char  description[32];                 /* resource description    */
  unsigned char  reserv1[16];                     /* reserved                */
} MODE_DEF_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: MODE_DEF_DETAIL                                                */
/*                                                                           */
/* Description: Detail information returned on QUERY_MODE_DEF (above)        */
/*****************************************************************************/

typedef struct mode_def_detail
{
  AP_UINT16      overlay_size;                       /* overlay size         */
  unsigned char mode_name[8];                        /* mode name            */
  MODE_CHARS    mode_chars;                          /* mode characteristics */
} MODE_DEF_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_PARTNER_LU_DEFINITION                                    */
/*                                                                           */
/* Description: Extracts partner LU definition details                       */
/*****************************************************************************/

typedef struct query_partner_lu_definition
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  *buf_ptr;                /* pointer to buffer               */
  AP_UINT32      buf_size;                /* buffer size                     */
  AP_UINT32      total_buf_size;          /* total buffer size required      */
  AP_UINT16      num_entries;             /* number of entries               */
  AP_UINT16      total_num_entries;       /* total number of entries         */
  unsigned char  list_options;            /* listing options                 */
  unsigned char  reserv3;                 /* reserved                        */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
} QUERY_PARTNER_LU_DEFINITION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PARTNER_LU_DEF_SUMMARY                                         */
/*                                                                           */
/* Description: Summary information used by QUERY_PARTNER_LU_DEF (above)     */
/*****************************************************************************/

typedef struct partner_lu_def_summary
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv1[16];             /* reserved                        */
} PARTNER_LU_DEF_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PARTNER_LU_DEF_DETAIL                                          */
/*                                                                           */
/* Description: Detail overlay used for QUERY_PARTNER_LU_DEF (above)         */
/*****************************************************************************/

typedef struct partner_lu_def_detail
{
  AP_UINT16      overlay_size;            /* overlay size                    */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  reserv1;                 /* reserved                        */
  PLU_CHARS      plu_chars;               /* partner LU characteristics      */
} PARTNER_LU_DEF_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TP_DEFINITION                                            */
/*                                                                           */
/* Description: Extracts TP definition information                           */
/*****************************************************************************/

typedef struct query_tp_definition
{
  AP_UINT16      opcode;                       /* Verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* Primary return code        */
  AP_UINT32      secondary_rc;                 /* Secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries *        */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  tp_name[64];                  /* TP name                    */
} QUERY_TP_DEFINITION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TP_DEF_SUMMARY                                                 */
/*                                                                           */
/* Description: Summary overlay used for QUERY_TP_DEF (above)                */
/*****************************************************************************/

typedef struct tp_def_summary
{
  AP_UINT16      overlay_size;                  /* overlay size              */
  unsigned char  tp_name[64];                   /* TP name                   */
  unsigned char  description[32];               /* resource description      */
  unsigned char  reserv1[16];                   /* reserved                  */
} TP_DEF_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TP_DEF_DETAIL                                                  */
/*                                                                           */
/* Description: Detail overlay used for QUERY_TP_DEF (above)                 */
/*****************************************************************************/

typedef struct tp_def_detail
{
  AP_UINT16      overlay_size;                         /* overlay size       */
  unsigned char tp_name[64];                           /* TP name            */
  TP_CHARS      tp_chars;                              /* TP characteristics */
} TP_DEF_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   START_DLC                                                    */
/*                                                                           */
/* Description: Start DLC                                                    */
/*****************************************************************************/

typedef struct start_dlc
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  dlc_name[8];                       /* name of DLC           */
} START_DLC;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   START_LS                                                     */
/*                                                                           */
/* Description: Start LS                                                     */
/*****************************************************************************/

typedef struct start_ls
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  ls_name[8];                        /* name of link station  */
  unsigned char  enable;                            /* enable for auto-act   */
  unsigned char  react_kicked;                      /* LR working on this LS?*/
  unsigned char  reserv3[2];                        /* reserved              */
} START_LS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   START_PORT                                                   */
/*                                                                           */
/* Description: Start port                                                   */
/*****************************************************************************/

typedef struct start_port
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  port_name[8];                      /* name of port          */
} START_PORT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   STOP_DLC                                                     */
/*                                                                           */
/* Description: Stop DLC                                                     */
/*****************************************************************************/

typedef struct stop_dlc
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  stop_type;                         /* stop type             */
  unsigned char  dlc_name[8];                       /* name of DLC           */
} STOP_DLC;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   STOP_LS                                                      */
/*                                                                           */
/* Description: Stop LS                                                      */
/*****************************************************************************/

typedef struct stop_ls
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  stop_type;                         /* stop type             */
  unsigned char  ls_name[8];                        /* name of link station  */
  unsigned char  disable;                           /* disable for auto-act  */
  unsigned char  reserved[3];                       /* reserved              */
} STOP_LS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   STOP_PORT                                                    */
/*                                                                           */
/* Description: Stop port                                                    */
/*****************************************************************************/

typedef struct stop_port
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  stop_type;                         /* Stop Type             */
  unsigned char  port_name[8];                      /* name of port          */
} STOP_PORT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_CN                                                     */
/*                                                                           */
/* Description: Query status of a CN.                                        */
/*****************************************************************************/

typedef struct query_cn
{
  AP_UINT16      opcode;                       /* Verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* Primary return code        */
  AP_UINT32      secondary_rc;                 /* Secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  fqcn_name[17];                /* Name of Connection Network */
} QUERY_CN;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   CN_DET_DATA                                                  */
/*                                                                           */
/* Description: Determined Data for a Connection Network                     */
/*****************************************************************************/

typedef struct cn_det_data
{
  AP_UINT16      num_act_ports;                /* number of active ports     */
  unsigned char  reserva[20];                  /* reserved                   */
} CN_DET_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   CN_DATA                                                      */
/*                                                                           */
/* Description: Overlay returned in response to a QUERY_CN                   */
/*****************************************************************************/

typedef struct cn_data
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  unsigned char fqcn_name[17];                 /* Name of Connection Network */
  unsigned char reserv1;                       /* reserved                   */
  CN_DET_DATA   det_data;                      /* Determined data            */
  CN_DEF_DATA   def_data;                      /* Defined data               */
} CN_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_CN_PORT                                                */
/*                                                                           */
/* Description: Query status of a port on a CN.                              */
/*****************************************************************************/

typedef struct query_cn_port
{
  AP_UINT16      opcode;                       /* Verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* Primary return code        */
  AP_UINT32      secondary_rc;                 /* Secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  fqcn_name[17];                /* Name of Connection Network */
  unsigned char  port_name[8];                 /* port name                  */
} QUERY_CN_PORT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   CN_PORT_DATA                                                 */
/*                                                                           */
/* Description: Data overlay returned in response to QUERY_CN_PORT           */
/*****************************************************************************/

typedef struct cn_port_data
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  unsigned char fqcn_name[17];                 /* Name of Connection Network */
  unsigned char port_name[8];                  /* name of port               */
  unsigned char tg_num;                        /* transmission group number  */
  unsigned char reserva[20];                   /* reserved                   */
} CN_PORT_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_DLC                                                    */
/*                                                                           */
/* Description: Query status of a DLC.                                       */
/*****************************************************************************/

typedef struct query_dlc
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  dlc_name[8];                  /* name of DLC                */
} QUERY_DLC;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_LS                                                     */
/*                                                                           */
/* Description: Query status of a LS.                                        */
/*****************************************************************************/

typedef struct query_ls
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* Primary return code        */
  AP_UINT32      secondary_rc;                 /* Secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  ls_name[8];                   /* name of link station       */
  unsigned char  port_name[8];                 /* port name filter           */
} QUERY_LS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_PORT                                                   */
/*                                                                           */
/* Description: Query status of a port.                                      */
/*****************************************************************************/

typedef struct query_port
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  port_name[8];                 /* port name                  */
  unsigned char  dlc_name[8];                  /* DLC name filter            */
} QUERY_PORT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: INITIALIZE_SESSION_LIMIT                                       */
/*                                                                           */
/* Description: INITIALIZE_SESSION_LIMIT verb control block                  */
/*****************************************************************************/

typedef struct initialize_session_limit
{
  AP_UINT16      opcode;                 /* verb operation code              */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;             /* primary return code              */
  AP_UINT32      secondary_rc;           /* secondary return code            */
  unsigned char  lu_name[8];             /* local LU name                    */
  unsigned char  lu_alias[8];            /* local LU alias                   */
  unsigned char  plu_alias[8];           /* partner                          */
  unsigned char  fqplu_name[17];         /* fully qualified partner LU name  */
  unsigned char  reserv3;                /* reserved                         */
  unsigned char  mode_name[8];           /* mode name                        */
  unsigned char  reserv3a;               /* reserved                         */
  unsigned char  set_negotiable;         /* set max negotiable limit?        */
  AP_UINT16      plu_mode_session_limit; /* session limit                    */
  AP_UINT16      min_conwinners_source;  /* minimum source contention winner */
                                         /* sessions                         */
  AP_UINT16      min_conwinners_target;  /* minimum target contention winner */
                                         /* sessions                         */
  AP_UINT16      auto_act;               /* auto activation limit            */
  unsigned char  reserv4[4];             /* reserved                         */
  AP_UINT32      sense_data;             /* sense data                       */
} INITIALIZE_SESSION_LIMIT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RESET_SESSION_LIMIT                                            */
/*                                                                           */
/* Description: RESET_SESSION_LIMIT verb control block                       */
/*****************************************************************************/

typedef struct reset_session_limit
{
  AP_UINT16      opcode;              /* verb operation code                 */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;          /* primary return code                 */
  AP_UINT32      secondary_rc;        /* secondary return code               */
  unsigned char  lu_name[8];          /* local LU name                       */
  unsigned char  lu_alias[8];         /* local LU alias                      */
  unsigned char  plu_alias[8];        /* partner LU alias                    */
  unsigned char  fqplu_name[17];      /* fully qualified partner LU name     */
  unsigned char  reserv3;             /* reserved                            */
  unsigned char  mode_name[8];        /* mode name                           */
  unsigned char  mode_name_select;    /* select mode name                    */
  unsigned char  set_negotiable;      /* set max negotiable limit?           */
  unsigned char  reserv4[8];          /* reserved                            */
  unsigned char  responsible;         /* who is responsible for deactivation */
  unsigned char  drain_source;        /* drain source                        */
  unsigned char  drain_target;        /* drain target                        */
  unsigned char  force;               /* force                               */
  AP_UINT32      sense_data;          /* sense data                          */
} RESET_SESSION_LIMIT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: CHANGE_SESSION_LIMIT                                           */
/*                                                                           */
/* Description: CHANGE_SESSION_LIMIT verb control block                      */
/*****************************************************************************/

typedef struct change_session_limit
{
  AP_UINT16      opcode;              /* verb operation code                 */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;          /* primary return code                 */
  AP_UINT32      secondary_rc;        /* secondary return code               */
  unsigned char  lu_name[8];          /* local LU name                       */
  unsigned char  lu_alias[8];         /* local LU alias                      */
  unsigned char  plu_alias[8];        /* partner LU alias                    */
  unsigned char  fqplu_name[17];      /* fully qualified partner LU name     */
  unsigned char  reserv3;             /* reserved                            */
  unsigned char  mode_name[8];        /* mode name                           */
  unsigned char  reserv3a;            /* reserved                            */
  unsigned char  set_negotiable;      /* set max negotiable limit?           */
  AP_UINT16      plu_mode_session_limit; /* session limit                    */
  AP_UINT16      min_conwinners_source;  /* minimum source contention        */
                                         /* winner sessions                  */
  AP_UINT16      min_conwinners_target;  /* minimum target contention        */
                                         /* winner sessions                  */
  AP_UINT16      auto_act;            /* auto activation limit               */
  unsigned char  responsible;         /* who is responsible for deactivating */
  unsigned char  reserv4[3];          /* reserved                            */
  AP_UINT32      sense_data;          /* sense data                          */
} CHANGE_SESSION_LIMIT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ACTIVATE_SESSION                                               */
/*                                                                           */
/* Description: ACTIVATE_SESSION verb control block                          */
/*****************************************************************************/

typedef struct activate_session
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  lu_name[8];              /* local LU name                   */
  unsigned char  lu_alias[8];             /* local LU alias                  */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  mode_name[8];            /* mode name                       */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  polarity;                /* polarity                        */
  unsigned char  session_id[8];           /* session id                      */
  unsigned char  cnos_permitted;          /* CNOS permitted?                 */
  unsigned char  reserv4[15];             /* reserved                        */
} ACTIVATE_SESSION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEACTIVATE_SESSION                                             */
/*                                                                           */
/* Description: DEACTIVATE_SESSION verb control block                        */
/*****************************************************************************/

typedef struct deactivate_session
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  lu_name[8];              /* local LU name                   */
  unsigned char  lu_alias[8];             /* local LU alias                  */
  unsigned char  session_id[8];           /* session identifier              */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  mode_name[8];            /* mode name                       */
  unsigned char  type;                    /* deactivation type               */
  unsigned char  reserv3[3];              /* reserved                        */
  AP_UINT32      sense_data;              /* deactivation sense data         */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  reserv4[20];             /* reserved                        */
} DEACTIVATE_SESSION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEACTIVATE_CONV_GROUP                                          */
/*                                                                           */
/* Description: DEACTIVATE_CONV_GROUP verb control block                     */
/*****************************************************************************/

typedef struct deactivate_conv_group
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  lu_name[8];                /* local LU name                 */
  unsigned char  lu_alias[8];               /* local LU alias                */
  AP_UINT32      conv_group_id;             /* conversation group identifier */
  unsigned char  type;                      /* deactivation type             */
  unsigned char  reserv3[3];                /* reserved                      */
  AP_UINT32      sense_data;                /* deactivation sense data       */
} DEACTIVATE_CONV_GROUP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEACTIVATE_LU_0_TO_3                                           */
/*                                                                           */
/* Description: NOF API verb sent to cause a TERM_SELF request for the       */
/*              PLU-SLU session to be sent to the host.                      */
/*****************************************************************************/
typedef struct deactivate_lu_0_to_3
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  lu_name[8];                /* LU Name                       */
} DEACTIVATE_LU_0_TO_3;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   INTERNAL_PU_DEF_DATA                                         */
/*                                                                           */
/* Description: Defined data for internal PU                                 */
/*****************************************************************************/
typedef struct internal_pu_def_data
{
  unsigned char    description[32];         /* resource description          */
  unsigned char    initially_active;        /* initially active              */
  unsigned char    reserv1[15];             /* reserved                      */
  unsigned char    dlus_name[17];           /* fully-qualified DLUS name     */
  unsigned char    bkup_dlus_name[17];      /* backup DLUS name              */
  unsigned char    pu_id[4];                /* PU identifier                 */
  AP_UINT16        dlus_retry_timeout;      /* time between attempts to      */
                                            /* recontact a DLUS              */
  AP_UINT16        dlus_retry_limit;        /* max number of attempts        */
  unsigned char    conventional_lu_compression; /* compression required?     */
  unsigned char    conventional_lu_cryptography; /* reserved                 */
  unsigned char    pu_can_send_dddlu_offline; /* send DDLU offline?          */
  unsigned char    reserv2[1];              /* reserved                      */
} INTERNAL_PU_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DEFINE_INTERNAL_PU                                           */
/*                                                                           */
/* Description: Define a DLUR-supported PU                                   */
/*****************************************************************************/
typedef struct define_internal_pu
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  pu_name[8];                /* PU name                       */
  INTERNAL_PU_DEF_DATA def_data;            /* defined data                  */

} DEFINE_INTERNAL_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   START_INTERNAL_PU                                            */
/*                                                                           */
/* Description: Start a DLUR-supported PU                                    */
/*****************************************************************************/
typedef struct start_internal_pu
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  pu_name[8];                /* PU name                       */
  unsigned char  dlus_name[17];             /* fully-qualified DLUS name     */
  unsigned char  bkup_dlus_name[17];        /* backup DLUS name              */

} START_INTERNAL_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   STOP_INTERNAL_PU                                             */
/*                                                                           */
/* Description: Stop a DLUR-supported PU                                     */
/*****************************************************************************/
typedef struct stop_internal_pu
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  pu_name[8];                /* PU name                       */
  unsigned char  stop_type;                 /* type of stop                  */

} STOP_INTERNAL_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DELETE_INTERNAL_PU                                           */
/*                                                                           */
/* Description: Delete a DLUR-supported PU                                   */
/*****************************************************************************/
typedef struct delete_internal_pu
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  pu_name[8];                /* PU name                       */

} DELETE_INTERNAL_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DEFINE_DLUR_DEFAULTS                                         */
/*                                                                           */
/* Description: Define DLUR defaults                                         */
/*****************************************************************************/
typedef struct define_dlur_defaults
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */

  unsigned char  description[32];           /* resource description          */
  unsigned char  reserv1[16];               /* reserved                      */
  unsigned char  dlus_name[17];             /* fully-qualified DLUS name     */
  unsigned char  bkup_dlus_name[17];        /* backup DLUS name              */
  unsigned char  reserv3;                   /* reserved                      */
  AP_UINT16      dlus_retry_timeout;        /* retry timeout                 */
  AP_UINT16      dlus_retry_limit;          /* retry limit                   */
  unsigned char  prefer_active_dlus;        /* retry using active DLUS       */
  unsigned char  persistent_pipe_support;   /* type of persistent pipe       */
  unsigned char  retry_cycles;              /* how long to retry             */
  unsigned char  reserv4[13];               /* reserved                      */

} DEFINE_DLUR_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_DLUR_DEFAULTS                                          */
/*                                                                           */
/* Description: Queries the default values of the DLUR.                      */
/*****************************************************************************/
typedef struct query_dlur_defaults
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */

  unsigned char  description[32];           /* resource description          */
  unsigned char  reserv1[16];               /* reserved                      */
  unsigned char  dlus_name[17];             /* Name of the DLUS              */
  unsigned char  bkup_dlus_name[17];        /* Name of the backup DLUS       */
  unsigned char  reserv3;                   /* reserved                      */
  AP_UINT16      dlus_retry_timeout;        /* Time in seconds between       */
                                            /* attempts to recontact a DLUS  */
  AP_UINT16      dlus_retry_limit;          /* Max number of attempts to     */
                                            /* recontact DLUS                */
  unsigned char  prefer_active_dlus;        /* retry using active DLUS       */
  unsigned char  persistent_pipe_support;   /* type of persistent pipe       */
  unsigned char  retry_cycles;              /* how long to retry             */
  unsigned char  reserv4[13];               /* reserved                      */

} QUERY_DLUR_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_DLUS                                                   */
/*                                                                           */
/* Description: Used to Query the state of DLUS nodes.                       */
/*****************************************************************************/

typedef struct query_dlus
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  *buf_ptr;                  /* pointer to buffer             */
  AP_UINT32      buf_size;                  /* buffer size                   */
  AP_UINT32      total_buf_size;            /* total buffer size required    */
  AP_UINT16      num_entries;               /* number of entries             */
  AP_UINT16      total_num_entries;         /* total number of entries       */
  unsigned char  list_options;              /* listing options               */
  unsigned char  reserv3;                   /* reserved                      */
  unsigned char  dlus_name[17];             /* fully-qualified DLUS name     */

} QUERY_DLUS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   PIPE_STATS                                                   */
/*                                                                           */
/* Description: DLUR Pipe Statistics                                         */
/*****************************************************************************/

typedef struct pipe_stats
{
  AP_UINT32     reqactpu_sent;              /* REQACTPUs sent to DLUs        */
  AP_UINT32     reqactpu_rsp_received;      /* RSP(REQACTPU)s rcvd from DLUS */
  AP_UINT32     actpu_received;             /* ACTPUs received from DLUS     */
  AP_UINT32     actpu_rsp_sent;             /* RSP(ACTPU)s sent to DLUS      */
  AP_UINT32     reqdactpu_sent;             /* REQDACTPUs sent to DLUS       */
  AP_UINT32     reqdactpu_rsp_received;     /* RSP(REQDACTPU)s from DLUS     */
  AP_UINT32     dactpu_received;            /* DACTPUs received from DLUS    */
  AP_UINT32     dactpu_rsp_sent;            /* RSP(DACTPU)s sent to DLUS     */
  AP_UINT32     actlu_received;             /* ACTLUs received from DLUS     */
  AP_UINT32     actlu_rsp_sent;             /* RSP(ACTLU)s sent to DLUS      */
  AP_UINT32     dactlu_received;            /* DACTLUs received from DLUS    */
  AP_UINT32     dactlu_rsp_sent;            /* RSP(DACTLU)s sent to DLUS     */
  AP_UINT32     sscp_pu_mus_rcvd;           /* MUs for SSCP-PU sessions rcvd */
  AP_UINT32     sscp_pu_mus_sent;           /* MUs for SSCP-PU sessions sent */
  AP_UINT32     sscp_lu_mus_rcvd;           /* MUs for SSCP-LU sessions rcvd */
  AP_UINT32     sscp_lu_mus_sent;           /* MUs for SSCP-LU sessions sent */

} PIPE_STATS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLUS_DATA                                                    */
/*                                                                           */
/* Description: Data overlay returned on QUERY_DLUS                          */
/*****************************************************************************/

typedef struct dlus_data
{
  AP_UINT16      overlay_size;              /* overlay size                  */
  unsigned char  dlus_name[17];             /* fully qualified DLUS name     */
  unsigned char  is_default;                /* is the DLUS the default       */
  unsigned char  is_backup_default;         /* is DLUS the backup default    */
  unsigned char  pipe_state;                /* state of CPSVRMGR pipe        */
  AP_UINT16      num_active_pus;            /* num of active PUs using pipe  */
  PIPE_STATS     pipe_stats;                /* pipe statistics               */
  unsigned char  persistent_pipe_support;   /* does DLUS support persistent  */
  unsigned char  persistent_pipe;           /* is the pipe persistent?       */
} DLUS_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLUS_INDICATION                                              */
/*                                                                           */
/* Description: Indication sent by DLUR when:                                */
/*                A DLUS CPSVRMGR pipe is activated                          */
/*                A DLUS CPSVRMGR pipe is deactivated                        */
/*****************************************************************************/

typedef struct dlus_indication
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  data_lost;                 /* previous indication lost      */
  unsigned char  deactivated;               /* has DLUS become inactive      */
  unsigned char  dlus_name[17];             /* DLUS name                     */
  unsigned char  reserv1;                   /* reserved                      */
  PIPE_STATS     pipe_stats;                /* pipe statistics               */
  unsigned char  persistent_pipe_support;   /* Does DLUS support persistent? */
  unsigned char  persistent_pipe;           /* Is pipe persistent?           */
  unsigned char  reserva[18];               /* reserved                      */
} DLUS_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_DLUR_PU                                                */
/*                                                                           */
/* Description: Used to Query the DLUR-specific PU information               */
/*****************************************************************************/

typedef struct query_dlur_pu
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  *buf_ptr;                  /* pointer to buffer             */
  AP_UINT32      buf_size;                  /* buffer size                   */
  AP_UINT32      total_buf_size;            /* total buffer size required    */
  AP_UINT16      num_entries;               /* number of entries             */
  AP_UINT16      total_num_entries;         /* total number of entries       */
  unsigned char  list_options;              /* listing options               */
  unsigned char  reserv3;                   /* reserved                      */
  unsigned char  pu_name[8];                /* PU name                       */
  unsigned char  dlus_name[17];             /* fully-qualified DLUS name     */
  unsigned char  filter;                    /* local/downstream filter       */

} QUERY_DLUR_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLUR_PU_DETAIL                                               */
/*                                                                           */
/* Description: Detailed data overlay returned on QUERY_DLUR_PU              */
/*****************************************************************************/

typedef struct dlur_pu_detail
{
  AP_UINT16      overlay_size;              /* overlay size                  */
  unsigned char  pu_name[8];                /* PU name                       */
  unsigned char  description[32];           /* resource description          */
  unsigned char  initially_active;          /* initially active              */
  unsigned char  reserv1[15];               /* reserved                      */
  unsigned char  defined_dlus_name[17];     /* defined DLUS name             */
  unsigned char  bkup_dlus_name[17];        /* backup DLUS name              */
  unsigned char  pu_id[4];                  /* PU identifier                 */
  unsigned char  pu_location;               /* downstream or local PU        */
  unsigned char  active_dlus_name[17];      /* active DLUS name              */
  unsigned char  ans_support;               /* auto network shutdown support */
  unsigned char  pu_status;                 /* status of the PU              */
  unsigned char  dlus_session_status;       /* status of the DLUS pipe       */
  unsigned char  reserv3;                   /* reserved                      */
  FQPCID         fqpcid;                    /* FQPCID used on pipe           */
  AP_UINT16      dlus_retry_timeout;        /* time in seconds between       */
  AP_UINT16      dlus_retry_limit;          /* max number of retries         */

} DLUR_PU_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLUR_PU_SUMMARY                                              */
/*                                                                           */
/* Description: Summary data overlay returned on QUERY_DLUR_PU               */
/*****************************************************************************/

typedef struct dlur_pu_summary
{
  AP_UINT16      overlay_size;              /* overlay size                  */
  unsigned char  pu_name[8];                /* PU name                       */
  unsigned char  description[32];           /* resource description          */
  unsigned char  initially_active;          /* initially active              */
  unsigned char  reserv1[15];               /* reserved                      */

} DLUR_PU_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_DLUR_LU                                                */
/*                                                                           */
/* Description: Used to Query the state of DLUR-served LUs                   */
/*****************************************************************************/

typedef struct query_dlur_lu
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  *buf_ptr;                  /* pointer to buffer             */
  AP_UINT32      buf_size;                  /* buffer size                   */
  AP_UINT32      total_buf_size;            /* total buffer size required    */
  AP_UINT16      num_entries;               /* number of entries             */
  AP_UINT16      total_num_entries;         /* total number of entries       */
  unsigned char  list_options;              /* listing options               */
  unsigned char  reserv3;                   /* reserved                      */
  unsigned char  lu_name[8];                /* LU name                       */
  unsigned char  pu_name[8];                /* PU name filter                */
  unsigned char  filter;                    /* local / downstream filter     */

} QUERY_DLUR_LU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLUR_LU_SUMMARY                                              */
/*                                                                           */
/* Description: Summary data overlay returned on QUERY_DLUR_LU               */
/*****************************************************************************/

typedef struct dlur_lu_summary
{
  AP_UINT16      overlay_size;              /* overlay size                  */
  unsigned char  lu_name[8];                /* LU name                       */

} DLUR_LU_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLUR_LU_DETAIL                                               */
/*                                                                           */
/* Description: Detailed data overlay returned on QUERY_DLUR_LU              */
/*****************************************************************************/

typedef struct dlur_lu_detail
{
  AP_UINT16      overlay_size;       /* overlay size                         */
  unsigned char  lu_name[8];         /* LU name                              */
  unsigned char  pu_name[8];         /* PU name of owning PU                 */
  unsigned char  dlus_name[17];      /* DLUS name if SSCP-LU session active  */
  unsigned char  lu_location;        /* downstream or local LU               */
  unsigned char  nau_address;        /* nau address of LU                    */
  unsigned char  plu_name[17];       /* PLU name if PLU-SLU session active   */
  unsigned char  reserv1[27];        /* reserved                             */
  unsigned char  rscv_len;           /* length of appended RSCV              */
} DLUR_LU_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DEFAULT_PU                                               */
/*                                                                           */
/* Description: Query verb for DEFAULT_PU                                    */
/*****************************************************************************/

typedef struct query_default_pu
{
  AP_UINT16      opcode;                /* verb operation code               */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;            /* primary return code               */
  AP_UINT32      secondary_rc;          /* secondary return code             */
  unsigned char  def_pu_name[8];        /* default PU name                   */
  unsigned char  description[32];       /* resource description              */
  unsigned char  reserv1[16];           /* reserved                          */
  unsigned char  def_pu_sess[8];        /* PU name of active default session */
  unsigned char  reserv3[16];           /* reserved                          */
} QUERY_DEFAULT_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_FOCAL_POINT                                              */
/*                                                                           */
/* Description: Used for querying focal points in MS                         */
/*****************************************************************************/

typedef struct query_focal_point
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  ms_category[8];               /* name of MS category        */
} QUERY_FOCAL_POINT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: FP_DATA                                                        */
/*                                                                           */
/* Description: Overlay returned for QUERY_FP (above)                        */
/*****************************************************************************/

typedef struct fp_data
{
  AP_UINT16      overlay_size;        /* overlay size                        */
  unsigned char  ms_appl_name[8];     /* focal point application name        */
  unsigned char  ms_category[8];      /* focal point category                */
  unsigned char  description[32];     /* resource description                */
  unsigned char  reserv1[16];         /* reserved                            */
  unsigned char  fp_fqcp_name[17];    /* focal point fully qualified cp name */
  unsigned char  bkup_appl_name[8];   /* backup focal point application name */
  unsigned char  bkup_fp_fqcp_name[17]; /* backup fp fully qualified cp name */
  unsigned char  implicit_appl_name[8]; /* implicit focal point appl name    */
  unsigned char  implicit_fp_fqcp_name[17]; /* implicit fp fq cp name        */
  unsigned char  fp_type;             /* focal point type                    */
  unsigned char  fp_status;           /* focal point status                  */
  unsigned char  fp_routing;          /* type of MDS routing to use          */
  unsigned char  reserva[20];         /* reserved                            */
  AP_UINT16      number_of_appls;     /* number of applications              */
}FP_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: APPLICATION_ID                                                 */
/*                                                                           */
/* Description: Application identifier returned with FP_DETAIL               */
/*****************************************************************************/
typedef struct application_id
{
  unsigned char  appl_name[8];        /* application name                    */
} APPLICATION_ID;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_NMVT_APPLICATION                                         */
/*                                                                           */
/* Description: Used for querying application registered for NMVT level      */
/*              information                                                  */
/*****************************************************************************/

typedef struct query_nmvt_application
{
  AP_UINT16      opcode;                       /* Verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  application[8];               /* application                */
} QUERY_NMVT_APPLICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: NMVT_APPLICATION_DATA                                          */
/*                                                                           */
/* Description: Information returned for QUERY_NMVT_APPLICATION              */
/*****************************************************************************/

typedef struct nmvt_application_data
{
  AP_UINT16      overlay_size;           /* overlay size                     */
  unsigned char  application[8];         /* application name                 */
  AP_UINT16      ms_vector_key_type;     /* MS vector key accepted by appl   */
  unsigned char  conversion_required;    /* is conversion to MDS_MU required */
  unsigned char  reserv[5];              /* reserved                         */
  unsigned char  reserva[20];            /* reserved                         */
}NMVT_APPLICATION_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_MDS_APPLICATION                                          */
/*                                                                           */
/* Description: Used for querying application registered for MDS level       */
/*              information                                                  */
/*****************************************************************************/

typedef struct query_mds_application
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  application[8];               /* application                */
} QUERY_MDS_APPLICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: MDS_APPLICATION_DATA                                           */
/*                                                                           */
/* Description: Information returned for QUERY_MDS_APPLICATION               */
/*****************************************************************************/

typedef struct mds_application_data
{
  AP_UINT16      overlay_size;             /* overlay size                   */
  unsigned char  application[8];           /* application name               */
  AP_UINT16      max_rcv_size;             /* max data size appl can receive */
  unsigned char  reserva[20];              /* reserved                       */
}MDS_APPLICATION_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_ACTIVE_TRANSACTION                                       */
/*                                                                           */
/* Description: Used for querying active transactions                        */
/*****************************************************************************/

typedef struct query_active_transaction
{
  AP_UINT16      opcode;              /* verb operation code                 */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;          /* primary return code                 */
  AP_UINT32      secondary_rc;        /* secondary return code               */
  unsigned char  *buf_ptr;            /* pointer to buffer                   */
  AP_UINT32      buf_size;            /* buffer size                         */
  AP_UINT32      total_buf_size;      /* total buffer size required          */
  AP_UINT16      num_entries;         /* number of entries                   */
  AP_UINT16      total_num_entries;   /* total number of entries             */
  unsigned char  list_options;        /* listing options                     */
  unsigned char  reserv3;             /* reserved                            */
  unsigned char  fq_req_loc_cp_name[17];        /* fq cp name of transaction */
                                                /* requestor                 */
  unsigned char  req_agent_appl_name[8];        /* appl name of transaction  */
                                                /* requestor                 */
  unsigned char  seq_num_dt[17];      /* sequence number date/time           */
} QUERY_ACTIVE_TRANSACTION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ACTIVE_TRANSACTION_DATA                                        */
/*                                                                           */
/* Description: Information returned for QUERY_ACTIVE_TRANSACTION            */
/*****************************************************************************/

typedef struct active_transaction_data
{
  AP_UINT16      overlay_size;             /* overlay size                   */
  unsigned char fq_origin_cp_name[17]; /* cp name of transaction origin      */
  unsigned char origin_ms_appl_name[8]; /* appl name of transaction origin   */
  unsigned char fq_dest_cp_name[17];  /* cp name of transaction destination  */
  unsigned char dest_ms_appl_name[8]; /* appl name of transaction dest       */
  unsigned char fq_req_loc_cp_name[17]; /* fq cp name of transaction requestr*/
  unsigned char req_agent_appl_name[8]; /* appl name of transaction requestor*/
  unsigned char seq_num_dt[17];       /* sequence number date/time           */
  unsigned char reserva[20];          /* reserved                            */
} ACTIVE_TRANSACTION_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_MDS_STATISTICS                                           */
/*                                                                           */
/* Description: Used for querying MDS statistics                             */
/*****************************************************************************/

typedef struct query_mds_statistics
{
  AP_UINT16      opcode;               /* verb operation code                */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;           /* primary return code                */
  AP_UINT32      secondary_rc;         /* secondary return code              */
  AP_UINT32      alerts_sent;          /* number of alert sends              */
  AP_UINT32      alert_errors_rcvd;    /* error messages received for alert  */
                                       /* sends                              */
  AP_UINT32      uncorrelated_alert_errors; /* uncorrelated alert errors rcvd*/
  AP_UINT32      mds_mus_rcvd_local;   /* number of MDS_MUs received from    */
                                       /* local applications                 */
  AP_UINT32      mds_mus_rcvd_remote;  /* number of MDS_MUs received from    */
                                       /* remote applications                */
  AP_UINT32      mds_mus_delivered_local; /* number of MDS_MUs delivered to  */
                                       /* local applications                 */
  AP_UINT32      mds_mus_delivered_remote; /* number of MDS_MUs delivered to */
                                       /* remote applications                */
  AP_UINT32      parse_errors;         /* number of MDS_MUs received with    */
                                       /* parse errors                       */
  AP_UINT32      failed_deliveries;    /* number of MDS_MUs where delivery   */
                                       /* failed                             */
  AP_UINT32      ds_searches_performed; /* number of DS searches performed   */
  AP_UINT32      unverified_errors;    /* number of unverified errors        */
  unsigned char  reserva[20];          /* reserved                           */
} QUERY_MDS_STATISTICS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   CPIC_SIDE_INFO_DATA                                          */
/*                                                                           */
/* Description: Used to hold a CPI-C side information entry                  */
/*****************************************************************************/

typedef struct cpic_side_info_data
{
  AP_UINT16      overlay_size;                  /* overlay size              */
  unsigned char  sym_dest_name[8];              /* Symbolic destination name */
  unsigned char  reserv1[2];                    /* reserved                  */
  CPIC_SIDE_INFO_DEF_DATA def_data;             /* defined data              */
} CPIC_SIDE_INFO_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   QUERY_CPIC_SIDE_INFO                                         */
/*                                                                           */
/* Description: Used to query a CPI-C side information entry                 */
/*****************************************************************************/

typedef struct query_cpic_side_info
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  sym_dest_name[8];             /* Symbolic destination name  */
} QUERY_CPIC_SIDE_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_DEFAULT_PU                                              */
/*                                                                           */
/* Description: Specifies a pu to serve as the default pu                    */
/*****************************************************************************/

typedef struct define_default_pu
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  pu_name[8];                        /* PU name               */
  unsigned char  description[32];                   /* resource description  */
  unsigned char  reserv1[16];                       /* reserved              */
  unsigned char  reserv3[16];                       /* reserved              */
} DEFINE_DEFAULT_PU;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_FOCAL_POINT                                             */
/*                                                                           */
/* Description: Allows the operator to define a focal point                  */
/*****************************************************************************/

typedef struct define_focal_point
{
  AP_UINT16      opcode;              /* verb operation code                 */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;          /* primary return code                 */
  AP_UINT32      secondary_rc;        /* secondary return code               */
  unsigned char  reserved;            /* reserved                            */
  unsigned char  ms_category[8];      /* management services category        */
  unsigned char  fp_fqcp_name[17];    /* Fully qualified focal point cp name */
  unsigned char  ms_appl_name[8];     /* Focal point application name        */
  unsigned char  description[32];     /* resource description                */
  unsigned char  reserv1[16];         /* reserved                            */
  unsigned char  backup;              /* is focal point a backup             */
  unsigned char  reserv3[16];         /* reserved                            */
} DEFINE_FOCAL_POINT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_FOCAL_POINT                                             */
/*                                                                           */
/* Description: DELETE_FOCAL_POINT verb control block                        */
/*****************************************************************************/

typedef struct delete_focal_point
{
  AP_UINT16      opcode;                     /* verb operation code          */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                 /* primary return code          */
  AP_UINT32      secondary_rc;               /* secondary return code        */
  unsigned char  reserved;                   /* reserved                     */
  unsigned char  ms_category[8];             /* management services category */
  unsigned char  type;                       /* type of focal point          */
} DELETE_FOCAL_POINT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: APING                                                          */
/*                                                                           */
/* Description: Used to initiate APING transaction                           */
/*****************************************************************************/

typedef struct aping
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  lu_name[8];              /* local LU name                   */
  unsigned char  lu_alias[8];             /* local LU alias                  */
  AP_UINT32      sense_data;              /* sense data                      */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  mode_name[8];            /* mode name                       */
  unsigned char  tp_name[64];             /* destination TP name             */
  unsigned char  security;                /* security level                  */
  unsigned char  reserv3a[3];             /* reserved                        */
  unsigned char  pwd[10];                 /* password                        */
  unsigned char  user_id[10];             /* user ID                         */
  AP_UINT16      dlen;                    /* length of data to send          */
  AP_UINT16      consec;                  /* number of consecutive sends     */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  echo;                    /* data echo flag                  */
  AP_UINT16      iterations;              /* number of iterations            */
  AP_UINT32      alloc_time;              /* time taken for ALLOCATE         */
  AP_UINT32      min_time;                /* minimum send/receive time       */
  AP_UINT32      avg_time;                /* average send/receive time       */
  AP_UINT32      max_time;                /* maximum send/receive time       */
  AP_UINT16      partner_ver_len;         /* size of string to receive       */
} APING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: REGISTER_INDICATION_SINK                                       */
/*                                                                           */
/* Description: Signal used to register to receive indication sinks          */
/*****************************************************************************/

typedef struct register_indication_sink
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  AP_UINT32      proc_id;                 /* reserved                        */
  AP_UINT16      queue_id;                /* reserved                        */
  AP_UINT16      indication_opcode;       /* opcode of indication to be sunk */
} REGISTER_INDICATION_SINK;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: UNREGISTER_INDICATION_SINK                                     */
/*                                                                           */
/* Description: Signal used to unregister for indication sinks               */
/*****************************************************************************/

typedef struct unregister_indication_sink
{
  AP_UINT16      opcode;                /* verb operation code               */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;            /* primary return code               */
  AP_UINT32      secondary_rc;          /* secondary return code             */
  AP_UINT32      proc_id;               /* reserved                          */
  AP_UINT16      queue_id;              /* reserved                          */
  AP_UINT16      indication_opcode;     /* opcode of indication to be unsunk */
} UNREGISTER_INDICATION_SINK;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: REGISTRATION_FAILURE                                           */
/*                                                                           */
/* Description: Verb indicating registration failure when registering        */
/*              resources to the Network Node server                         */
/*****************************************************************************/

typedef struct registration_failure
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  data_lost;               /* data lost ?                     */
  unsigned char  resource_name[17];       /* network qualified resource name */
  AP_UINT16      resource_type;           /* resource type                   */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv1[16];             /* reserved                        */
  unsigned char  reserv2b[2];             /* reserved                        */
  AP_UINT32      sense_data;              /* sense data                      */
  unsigned char  reserva[20];             /* reserved                        */
} REGISTRATION_FAILURE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PLU_INDICATION                                                 */
/*                                                                           */
/* Description: Used to indicate when a local LU first connects to a partner */
/*              LU.                                                          */
/*****************************************************************************/

typedef struct plu_indication
{
  AP_UINT16      opcode;               /* verb operation code                */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;           /* primary return code                */
  AP_UINT32      secondary_rc;         /* secondary return code              */
  unsigned char  data_lost;            /* has previous indication been lost? */
  unsigned char  removed;              /* is entry being removed?            */
  unsigned char  lu_alias[8];          /* LU alias                           */
  unsigned char  plu_alias[8];         /* partner LU alias                   */
  unsigned char  fqplu_name[17];       /* fully qualified partner LU name    */
  unsigned char  description[32];      /* resource description               */
  unsigned char  reserv1[16];          /* reserved                           */
  unsigned char  partner_cp_name[17];  /* partner control point name         */
  unsigned char  partner_lu_located;   /* partner LU located yet?            */
  unsigned char  reserva[20];          /* reserved                           */
} PLU_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: MODE_INDICATION                                                */
/*                                                                           */
/* Description: Used to indicate when a local LU/partner LU first start to   */
/*              use a particular mode                                        */
/*****************************************************************************/

typedef struct mode_indication
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  data_lost;               /* previous indication lost        */
  unsigned char  removed;                 /* is entry being removed?         */
  unsigned char  lu_alias[8];             /* LU alias                        */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  mode_name[8];            /* mode name                       */
  unsigned char  description[32];         /* resource description            */
  unsigned char  reserv1[16];             /* reserved                        */
  AP_UINT16      curr_sess_count;         /* current session count           */
  unsigned char  reserva[20];             /* reserved                        */
} MODE_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SESSION_INDICATION                                             */
/*                                                                           */
/* Description: Used to indicate when a session is activated or deactivated  */
/*****************************************************************************/

typedef struct session_indication
{
  AP_UINT16      opcode;                  /* verb operation code             */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;              /* primary return code             */
  AP_UINT32      secondary_rc;            /* secondary return code           */
  unsigned char  data_lost;               /* previous indication lost        */
  unsigned char  deactivated;             /* has session been deactivated?   */
  unsigned char  lu_name[8];              /* LU name                         */
  unsigned char  lu_alias[8];             /* LU alias                        */
  unsigned char  plu_alias[8];            /* partner LU alias                */
  unsigned char  fqplu_name[17];          /* fully qualified partner LU name */
  unsigned char  mode_name[8];            /* mode name                       */
  unsigned char  session_id[8];           /* session ID                      */
  FQPCID         fqpcid;                  /* fully qualified procedure       */
                                          /* correlator ID                   */
  AP_UINT32      sense_data;              /* sense data                      */
  unsigned char  reserv1;                 /* reserved                        */
  SESSION_STATS  sess_stats;              /* session statistics              */
  unsigned char  sscp_id[6];              /* SSCP id                         */
  unsigned char  plu_slu_comp_lvl;        /* PLU SLU compression             */
  unsigned char  slu_plu_comp_lvl;        /* SLU PLU compression             */
  unsigned char  comp_in_series;          /* reserved                        */
  unsigned char  reserva[11];             /* reserved                        */
} SESSION_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DIRECTORY_INDICATION                                           */
/*                                                                           */
/* Description: Used to indicate when an entry is added to or removed from   */
/*              the local directory database.                                */
/*****************************************************************************/

typedef struct directory_indication
{
  AP_UINT16      opcode;                         /* verb operation code      */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                     /* primary return code      */
  AP_UINT32      secondary_rc;                   /* secondary return code    */
  unsigned char  data_lost;                      /* previous indication lost */
  unsigned char  removed;                        /* is entry being removed?  */
  unsigned char  resource_name[17];              /* resource name            */
  unsigned char  invalid;                        /* removed because wrong?   */
  AP_UINT16      resource_type;                  /* resource type            */
  unsigned char  parent_name[17];                /* parent resource name     */
  unsigned char  entry_type;                     /* type of entry            */
  AP_UINT16      parent_type;                    /* parent resource type     */
  unsigned char  description[32];                /* resource description     */
  unsigned char  reserv3[16];                    /* reserved                 */
  AP_UINT16      real_owning_cp_type;
  unsigned char  real_owning_cp_name[17];
  AP_UINT16      supplier_cp_type;
  unsigned char  supplier_cp_name[17];
  unsigned char  reserva;                        /* reserved                 */
} DIRECTORY_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DLC_INDICATION                                                 */
/*                                                                           */
/* Description: Used to indicate when a DLC is stopped or started            */
/*****************************************************************************/

typedef struct dlc_indication
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  data_lost;                 /* previous indication lost      */
  unsigned char  deactivated;               /* has session been deactivated? */
  unsigned char  dlc_name[8];               /* link station name             */
  unsigned char  description[32];           /* resource description          */
  unsigned char  reserv1[16];               /* reserved                      */
  unsigned char  reserva[20];               /* reserved                      */
} DLC_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PORT_INDICATION                                                */
/*                                                                           */
/* Description: Used to indicate when a port is stopped or started           */
/*****************************************************************************/

typedef struct port_indication
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  data_lost;                 /* previous indication lost      */
  unsigned char  deactivated;               /* has session been deactivated? */
  unsigned char  port_name[8];              /* port name                     */
  unsigned char  description[32];           /* resource description          */
  unsigned char  reserv1[16];               /* reserved                      */
  unsigned char  reserva[20];               /* reserved                      */
} PORT_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LS_INDICATION                                                  */
/*                                                                           */
/* Description: Used to indicate when the number of active sessions using    */
/*              changes or when the link station goes from active to         */
/*              inactive (or vice-versa)                                     */
/*****************************************************************************/

typedef struct ls_indication
{
  AP_UINT16      opcode;               /* verb operation code                */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;           /* primary return code                */
  AP_UINT32      secondary_rc;         /* secondary return code              */
  unsigned char  data_lost;            /* previous indication lost           */
  unsigned char  deactivated;          /* has session been deactivated?      */
  unsigned char  ls_name[8];           /* link station name                  */
  unsigned char  description[32];      /* resource description               */
  unsigned char  reserv1[16];          /* reserved                           */
  unsigned char  adj_cp_name[17];      /* network qualified Adjacent CP name */
  unsigned char  adj_node_type;        /* adjacent node type                 */
  AP_UINT16      act_sess_count;       /* active session count on link       */
  unsigned char  indication_cause;     /* cause of indication                */
  LS_STATS       ls_stats;             /* link station statistics            */
  unsigned char  tg_num;               /* tg number                          */
  AP_UINT32      sense_data;           /* sense data                         */
  unsigned char  brnn_link_type;       /* uplink or downlink?                */
  unsigned char  adj_cp_is_brnn;       /* type of adjacent CP                */
  unsigned char  det_mltg_member;      /* reserved                           */
  unsigned char  tg_sharing;           /* reserved                           */
  unsigned char  ls_type;              /* type of the link station           */
  unsigned char  reserva[14];          /* reserved                           */
} LS_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: NN_TOPOLOGY_NODE_INDICATION                                    */
/*                                                                           */
/* Description: Used to indicate a change in the state of a node in the      */
/*              network topology                                             */
/*****************************************************************************/

typedef struct nn_topology_node_indication
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  data_lost;                 /* previous indication lost      */
  unsigned char  deactivated;               /* has the node become inactive? */
  unsigned char  node_name[17];             /* node name                     */
  unsigned char  node_type;                 /* node type                     */
  unsigned char  branch_aware;
  unsigned char  reserva[19];               /* reserved                      */
} NN_TOPOLOGY_NODE_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: NN_TOPOLOGY_TG_INDICATION                                      */
/*                                                                           */
/* Description: Used to indicate a change in the state of a TG in the        */
/*              network topology                                             */
/*****************************************************************************/

typedef struct nn_topology_tg_indication
{
  AP_UINT16      opcode;                      /* verb operation code         */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                  /* primary return code         */
  AP_UINT32      secondary_rc;                /* secondary return code       */
  unsigned char  data_lost;                   /* previous indication lost    */
  unsigned char  status;                      /* TG status                   */
  unsigned char  owner[17];                   /* name of TG owner node       */
  unsigned char  dest[17];                    /* name of TG destination node */
  unsigned char  tg_num;                      /* TG number                   */
  unsigned char  owner_type;                  /* type of TG owner node       */
  unsigned char  dest_type;                   /* type of TG destination node */
  unsigned char  cp_cp_session_active;        /* are CP-CP sessions active?  */
  unsigned char  branch_tg;                   /* is this a branch link?      */
  unsigned char  multilink_tg;                /* reserved                    */
  unsigned char  reserva[15];                 /* reserved                    */
} NN_TOPOLOGY_TG_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LOCAL_TOPOLOGY_INDICATION                                      */
/*                                                                           */
/* Description: Used to indicate a change in the state of a TG in the local  */
/*              topology                                                     */
/*****************************************************************************/

typedef struct local_topology_indication
{
  AP_UINT16      opcode;                      /* verb operation code         */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                  /* primary return code         */
  AP_UINT32      secondary_rc;                /* secondary return code       */
  unsigned char  data_lost;                   /* previous indication lost    */
  unsigned char  status;                      /* TG status                   */
  unsigned char  dest[17];                    /* name of TG destination node */
  unsigned char  dest_type;                   /* TG destination node type    */
  unsigned char  tg_num;                      /* TG number                   */
  unsigned char  cp_cp_session_active;
  unsigned char  branch_link_type;
  unsigned char  branch_tg;
  unsigned char  reserva[17];                 /* reserved                    */
} LOCAL_TOPOLOGY_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ISR_INDICATION                                                 */
/*                                                                           */
/* Description: Used to indicate a change in the state of an intermediate    */
/*              session                                                      */
/*****************************************************************************/

typedef struct isr_indication
{
  AP_UINT16      opcode;                /* verb operation code               */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;            /* primary return code               */
  AP_UINT32      secondary_rc;          /* secondary return code             */
  unsigned char  data_lost;             /* previous indication lost          */
  unsigned char  deactivated;           /* has ISR session been deactivated? */
  FQPCID         fqpcid;                /* FQPCID for ISR session            */
  unsigned char  fqplu_name[17];        /* fully qualified primary LU name   */
  unsigned char  fqslu_name[17];        /* fully qualified secondary LU name */
  unsigned char  mode_name[8];          /* mode name                         */
  unsigned char  cos_name[8];           /* class of service name             */
  unsigned char  transmission_priority; /* transmission priority             */
  AP_UINT32      sense_data;            /* sense data                        */
  unsigned char  reserv2a[2];           /* reserved                          */
  SESSION_STATS  pri_sess_stats;        /* Primary hop session statistics    */
  SESSION_STATS  sec_sess_stats;        /* Secondary hop session statistics  */
  unsigned char  reserva[20];           /* reserved                          */
} ISR_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: FOCAL_POINT_INDICATION                                         */
/*                                                                           */
/* Description: Used to indicate when a focal point is acquired, changed or  */
/*              revoked.                                                     */
/*****************************************************************************/

typedef struct focal_point_indication
{
  AP_UINT16      opcode;              /* verb operation code                 */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;          /* primary return code                 */
  AP_UINT32      secondary_rc;        /* secondary return code               */
  unsigned char  data_lost;           /* previous indication lost            */
  unsigned char  ms_category[8];      /* Focal point category                */
  unsigned char  fp_fqcp_name[17];    /* Fully qualified focal point cp name */
  unsigned char  ms_appl_name[8];     /* Focal point application name        */
  unsigned char  fp_type;             /* type of current focal point         */
  unsigned char  fp_status;           /* status of focal point               */
  unsigned char  fp_routing;          /* type of MDS routing to reach FP     */
  unsigned char  reserva[20];         /* reserved                            */
} FOCAL_POINT_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LU_0_TO_3_INDICATION                                           */
/*                                                                           */
/* Description: Used to indicate when the state of a local LU type 0 to 3    */
/*              changes.                                                     */
/*****************************************************************************/
typedef struct lu_0_to_3_indication
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  data_lost;                    /* previous indication lost   */
  unsigned char  pu_name[8];                   /* PU Name                    */
  unsigned char  lu_name[8];                   /* LU Name                    */
  unsigned char  description[32];              /* resource description       */
  unsigned char  reserv1[16];                  /* reserved                   */
  unsigned char  nau_address;                  /* NAU address                */
  unsigned char  lu_sscp_sess_active;          /* Is SSCP session active?    */
  unsigned char  appl_conn_active;             /* Is application using LU?   */
  unsigned char  plu_sess_active;              /* Is PLU-SLU session active? */
  unsigned char  host_attachment;              /* Host attachment            */
  SESSION_STATS  lu_sscp_stats;                /* reserved                   */
  SESSION_STATS  plu_stats;                    /* reserved                   */
  unsigned char  sscp_id[6];                   /* SSCP id                    */
} LU_0_TO_3_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PU_INDICATION                                                  */
/*                                                                           */
/* Description: Used to indicate when the state of a local PU changes        */
/*****************************************************************************/

typedef struct pu_indication
{
  AP_UINT16      opcode;                         /* verb operation code      */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                     /* primary return code      */
  AP_UINT32      secondary_rc;                   /* secondary return code    */
  unsigned char  data_lost;                      /* previous indication lost */
  unsigned char  pu_name[8];                     /* PU Name                  */
  unsigned char  description[32];                /* resource description     */
  unsigned char  reserv3[16];                    /* reserved                 */
  unsigned char  pu_sscp_sess_active;            /* Is SSCP session active?  */
  unsigned char  host_attachment;                /* Host attachment          */
  unsigned char  reserv1[2];                     /* reserved                 */
  SESSION_STATS  pu_sscp_stats;                  /* reserved                 */
  unsigned char  sscp_id[6];                   /* SSCP id                    */
} PU_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DOWNSTREAM_LU_INDICATION                                       */
/*                                                                           */
/* Description: Indication generated when state of downstream LU changes     */
/*****************************************************************************/

typedef struct downstream_lu_indication
{
  AP_UINT16      opcode;                 /* verb operation code              */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;             /* primary return code              */
  AP_UINT32      secondary_rc;           /* secondary return code            */
  unsigned char  data_lost;              /* previous indication lost         */
  unsigned char  dspu_name[8];           /* PU name                          */
  unsigned char  ls_name[8];             /* Link station name                */
  unsigned char  dslu_name[8];           /* LU name                          */
  unsigned char  description[32];        /* resource description             */
  unsigned char  reserv3[16];            /* reserved                         */
  unsigned char  nau_address;            /* NAU address                      */
  unsigned char  lu_sscp_sess_active;    /* Is LU-SSCP session active        */
  unsigned char  plu_sess_active;        /* Is PLU-SLU session active        */
  unsigned char  dspu_services;          /* DSPU services                    */
  unsigned char  reserv1;                /* reserved                         */
  SESSION_STATS  lu_sscp_stats;          /* LU-SSCP session statistics       */
  SESSION_STATS  ds_plu_stats;           /* Downstream PLU-SLU session stats */
  SESSION_STATS  us_plu_stats;           /* Upstream PLU-SLU session stats   */
} DOWNSTREAM_LU_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DOWNSTREAM_PU_INDICATION                                       */
/* Description: Indication generated when state of downstream PU changes     */
/*****************************************************************************/

typedef struct downstream_pu_indication
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  data_lost;                    /* previous indication lost   */
  unsigned char  dspu_name[8];                 /* PU name                    */
  unsigned char  description[32];              /* resource description       */
  unsigned char  reserv3[16];                  /* reserved                   */
  unsigned char  ls_name[8];                   /* Link station name          */
  unsigned char  pu_sscp_sess_active;          /* Is PU-SSCP session active  */
  unsigned char  dspu_services;                /* DSPU services              */
  unsigned char  reserv1[2];                   /* reserved                   */
  SESSION_STATS  pu_sscp_stats;                /* PU-SSCP session statistics */
} DOWNSTREAM_PU_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DLUR_PU_INDICATION                                             */
/*                                                                           */
/* Description: Indication generated when state of DLUR PU changes           */
/*****************************************************************************/

typedef struct dlur_pu_indication
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  data_lost;                    /* data lost                  */
  unsigned char  reason;                       /* reason                     */
  unsigned char  pu_name[8];                   /* PU name                    */
  unsigned char  pu_id[4];                     /* PU id                      */
  unsigned char  pu_location;                  /* PU location                */
  unsigned char  pu_status;                    /* PU status                  */
  unsigned char  dlus_name[17];                /* DLUS name                  */
  unsigned char  dlus_session_status;          /* DLUS session status        */
  unsigned char  reserv5[2];                   /* reserved                   */
} DLUR_PU_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LOCAL_LU_INDICATION                                            */
/* Description: Indication generated when a local LU is added or removed     */
/*****************************************************************************/

typedef struct local_lu_indication
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  data_lost;                    /* previous indication lost   */
  unsigned char  reason;                       /* LU added or removed?       */
  unsigned char  lu_name[8];                   /* LU name                    */
  unsigned char  description[32];              /* resource description       */
  unsigned char  lu_alias[8];                  /* LU alias                   */
  unsigned char  nau_address;                  /* NAU address                */
  unsigned char  reserv4;                      /* reserved                   */
  unsigned char  pu_name[8];                   /* PU name                    */
  unsigned char  lu_sscp_active;               /* Is LU-SSCP session active? */
  unsigned char  reserv5;                      /* reserved                   */
  SESSION_STATS  lu_sscp_stats;                /* LU_SSCP session statistics */
} LOCAL_LU_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DLUR_LU_INDICATION                                             */
/* Description: Indication generated when a DLUR LU is added or removed      */
/*****************************************************************************/

typedef struct dlur_lu_indication
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  data_lost;                    /* previous indication lost   */
  unsigned char  reason;                       /* LU added or removed?       */
  unsigned char  lu_name[8];                   /* LU name                    */
  unsigned char  pu_name[8];                   /* PU name                    */
  unsigned char  nau_address;                  /* NAU address                */
  unsigned char  reserv5[7];                   /* reserved                   */
} DLUR_LU_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SESSION_FAILURE_INDICATION                                     */
/*                                                                           */
/* Description: Used to indicate when a session fails                        */
/*****************************************************************************/

typedef struct session_failure_indication
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  reserv3[3];                   /* reserved                   */
  unsigned char  lu_name[8];                   /* LU name                    */
  unsigned char  lu_alias[8];                  /* LU alias                   */
  unsigned char  plu_alias[8];                 /* PLU alias                  */
  unsigned char  fqplu_name[17];               /* FQPLU name                 */
  unsigned char  mode_name[8];                 /* Mode name                  */
  unsigned char  session_id[8];                /* Session id                 */
  AP_UINT32      sense_data;                   /* Sense data                 */
} SESSION_FAILURE_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SERVER_INDICATION                                              */
/*                                                                           */
/* Description: Indication sent when an SNA Server on the LAN starts/stops   */
/*****************************************************************************/

typedef struct server_indication
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  data_lost;                    /* previous indication lost?  */
  AP_UINT32      status;                       /* SNA Server active?         */
  AP_UINT32      flags;                        /* Service flags              */
  unsigned char  server_name[128];             /* Name of Server             */
} SERVER_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   SNA_LFSID                                                    */
/*                                                                           */
/* Description: Local Form Session Identifier                                */
/*****************************************************************************/

typedef struct sna_lfsid
{
  union
  {
    AP_UINT16      session_id;                 /* session identifier         */
    struct
    {
      unsigned char sidh;                      /* session ID high byte       */
      unsigned char sidl;                      /* session ID low byte        */
    } s;
  } uu;
  AP_UINT16      odai;                         /* ODAI bit                   */
} SNA_LFSID;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLC_TRACE_FILTER                                             */
/*                                                                           */
/* Description: Specifies a DLC trace filter                                 */
/*****************************************************************************/
typedef struct dlc_trace_filter
{
  unsigned char  resource_type;            /* type of filter                 */
  unsigned char  resource_name[8];         /* resource name                  */
  SNA_LFSID      lfsid;                    /* LFSID - used for LS resources  */
  unsigned char  message_type;             /* message types to be filtered   */

} DLC_TRACE_FILTER;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ADD_DLC_TRACE                                                  */
/*                                                                           */
/* Description: Add DLC trace filter                                         */
/*****************************************************************************/
typedef struct add_dlc_trace
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */

  DLC_TRACE_FILTER  filter;                    /* additional trace filter    */

} ADD_DLC_TRACE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: REMOVE_DLC_TRACE                                               */
/*                                                                           */
/* Description: Remove DLC trace filter                                      */
/*****************************************************************************/
typedef struct remove_dlc_trace
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */

  DLC_TRACE_FILTER  filter;                    /* additional trace filter    */

} REMOVE_DLC_TRACE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLC_TRACE_DATA                                               */
/*                                                                           */
/* Description: Overlay returned by QUERY_DLC_TRACE                          */
/*****************************************************************************/

typedef struct dlc_trace_data
{
  AP_UINT16        overlay_size;               /* overlay size               */
  DLC_TRACE_FILTER filter;                     /* trace filter               */
} DLC_TRACE_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DLC_TRACE                                                */
/*                                                                           */
/* Description: Query the DLC TRACE filters                                  */
/*****************************************************************************/
typedef struct query_dlc_trace
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  list_type;                    /* reserved                   */

  DLC_TRACE_FILTER  filter_entry;              /* trace filter               */

} QUERY_DLC_TRACE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: NOF_STATUS_INDICATION                                          */
/*****************************************************************************/

typedef struct nof_status_indication
{
  AP_UINT16      opcode;                      /* verb operation code         */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                  /* primary return code         */
  AP_UINT32      secondary_rc;                /* secondary return code       */
  AP_UINT32      status;                      /* status being reported       */
  AP_UINT32      dead_target_handle;          /* Handle of dead connection   */
                                              /* NULL for system termination */
  unsigned char  reserv1[32];                 /* reserved                    */
} NOF_STATUS_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: CONFIG_FILE                                                    */
/*                                                                           */
/* Description: Data structure used to carry the config file definition.     */
/*              The file name is a null-terminated string.                   */
/*****************************************************************************/

typedef struct config_file
{
  unsigned char  requested_role;         /* config file requested            */
  unsigned char  role_supplied;          /* config file returned             */
  unsigned char  system_name[128];       /* computer name where file located */
  unsigned char  file_name[81];          /* file name                        */
} CONFIG_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: CONNECT_NODE                                                   */
/*                                                                           */
/* Description: Connect to a Node and get a target handle which is then      */
/*              used on all subsequent verbs.  The Node can either be a      */
/*              running Node or a particular Node within a config file.      */
/*****************************************************************************/

typedef struct connect_node
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  node_type;                 /* specified node or central log */
  unsigned char  node_name[128];            /* name of Node                  */
  AP_UINT32      target_handle;             /* handle for subsequent verbs   */
  unsigned char  node_status;               /* node status                   */
  unsigned char  reserv3[12];               /* reserved                      */
} CONNECT_NODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DISCONNECT_NODE                                                */
/*                                                                           */
/* Description: Disconnect from Node and free resources used for connection  */
/*****************************************************************************/

typedef struct disconnect_node
{
  AP_UINT16      opcode;                            /* Verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* Primary return code   */
  AP_UINT32      secondary_rc;                      /* Secondary return code */
} DISCONNECT_NODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_PROCESSING_MODE                                            */
/*                                                                           */
/* Description: Sets specified processing mode for this target handle        */
/*****************************************************************************/

typedef struct set_processing_mode
{
  AP_UINT16      opcode;               /* verb operation code                */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;           /* primary return code                */
  AP_UINT32      secondary_rc;         /* secondary return code              */
  unsigned char  mode;                 /* new mode to be set for this handle */
  AP_UINT16      reserv1;              /* reserved - MUST BE SET TO ZERO     */
} SET_PROCESSING_MODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: OPEN_FILE                                                      */
/*                                                                           */
/* Description: Open a configuration file                                    */
/*****************************************************************************/

typedef struct open_file
{
  AP_UINT16       opcode;                    /* verb operation code          */
  unsigned char   reserv2;
  unsigned char   format;
  AP_UINT16       primary_rc;                /* primary return code          */
  AP_UINT32       secondary_rc;              /* secondary return code        */
  CONFIG_FILE     file_info;                 /* definition of file requested */
  AP_UINT32       target_handle;             /* handle for subsequent verbs  */
  unsigned char  reserv3[4];                /* reserved                      */
} OPEN_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: CLOSE_FILE                                                     */
/*                                                                           */
/* Description: Close a configuration file                                   */
/*****************************************************************************/

typedef struct close_file
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
} CLOSE_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_NODE                                                    */
/*                                                                           */
/* Description: Define a new node within a config file and set up a          */
/*              connection for it                                            */
/*****************************************************************************/

typedef struct define_node
{
  AP_UINT16       opcode;                     /* verb operation code         */
  unsigned char   reserv2;
  unsigned char   format;
  AP_UINT16       primary_rc;                 /* primary return code         */
  AP_UINT32       secondary_rc;               /* secondary return code       */
  unsigned char   node_name[128];             /* name of Node                */
  AP_UINT32       target_handle;              /* handle for subsequent verbs */
  CP_CREATE_PARMS cp_create_parms;            /* CP create parameters        */
} DEFINE_NODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: INIT_NODE                                                      */
/*                                                                           */
/* Description:  VCB to start a configured node                              */
/*****************************************************************************/

typedef struct init_node
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
} INIT_NODE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_NODE_ALL                                                 */
/*                                                                           */
/* Description: Used to obtain information on one or more nodes              */
/*****************************************************************************/

typedef struct query_node_all
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  unsigned char  node_name[128];               /* node name                  */
} QUERY_NODE_ALL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: NODE_SUMMARY                                                   */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_NODE_ALL (above)          */
/*****************************************************************************/

typedef struct node_summary
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  unsigned char  node_name[128];               /* node name                  */
  unsigned char  config_role;                  /* configuration role         */
  unsigned char  reserv3[12];                  /* reserved                   */
} NODE_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: NODE_RESOURCE_LIMITS                                           */
/*                                                                           */
/* Description: Node Resource Limits                                         */
/*              for each resource, -1 means unlimited                        */
/*****************************************************************************/
typedef struct node_resource_limits
{
  AP_INT32     lu62_tps;                      /* APPC/CPI-C applications     */
  AP_INT32     lua_tps;                       /* LUA applications            */
  AP_INT32     fmapi_tps;                     /* FMAPI applications          */
  AP_INT32     link_stations;                 /* Active link stations        */
  AP_INT32     tn3270_connections;            /* TN3270 server connections   */
  AP_INT32     tn_redirector_connections;     /* TN redirector connections   */
  AP_INT32     dunlap_sna_channels;           /* Dunlap LU6.2/LU1-3 channels */
  AP_INT32     dunlap_gsna_channels;          /* Dunlap G-SNA channels       */
  AP_INT32     data_sessions;                 /* Active PLU-SLU sessions,    */
  AP_INT32     lu_0_to_3_sessions;            /* Active Old LU sessions      */
  AP_INT32     appc_sessions;                 /* Active APPC sessions        */
  AP_INT32     sna_users;                     /* # of 3270, 5250 & RJE users */
  AP_INT32     reserv1[8];                    /* Reserved                    */

} NODE_RESOURCE_LIMITS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: NODE_OPTIONS                                                   */
/*                                                                           */
/* Description: Node Options - compile-time, licensing or presence of        */
/*              external libraries.  For each option set to AP_YES or AP_NO  */
/*****************************************************************************/
typedef struct node_options
{
  unsigned char    network_node;            /* network node capable          */
  unsigned char    end_node;                /* end node capable              */
  unsigned char    len_node;                /* LEN node capable              */
  unsigned char    dlur_support;            /* DLUR supported                */
  unsigned char    pu_conc_support;         /* PU Concentration supported    */
  unsigned char    tn_server_support;       /* TN Server supported           */
  unsigned char    hpr_support;             /* HPR supported                 */

  unsigned char    back_level_client;       /* back-level client support     */
  unsigned char    reserv2;
  unsigned char    ssl_support;             /* SKIT library installed        */
  unsigned char    branch_network_node;     /* BrNN capable                  */
  unsigned char    reserv1[21];

} NODE_OPTIONS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_NODE_LIMITS                                              */
/*                                                                           */
/* Description: Queries the licensing limits and options for the Node        */
/*****************************************************************************/
typedef struct query_node_limits
{
  AP_UINT16        opcode;                  /* verb operation code           */
  unsigned char    reserv2;                 /* reserved                      */
  unsigned char    format;                  /* format indicator              */
  AP_UINT16        primary_rc;              /* primary return code           */
  AP_UINT32        secondary_rc;            /* secondary return code         */

  NODE_RESOURCE_LIMITS  max_limits;         /* maximum limits                */
  NODE_RESOURCE_LIMITS  curr_usage;         /* current usage                 */

  NODE_OPTIONS     node_options;            /* node options                  */

  unsigned char    reserv4[4];              /* reserved                      */
  NODE_RESOURCE_LIMITS  max_usage;          /* highest usage                 */

} QUERY_NODE_LIMITS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_CENTRAL_LOGGING                                            */
/*                                                                           */
/* Description: Switch central logging on/off.  Must be issued on the handle */
/*              associated with the machine on which the central logger      */
/*              resides.                                                     */
/*****************************************************************************/

typedef struct set_central_logging
{
  AP_UINT16        opcode;                  /* verb operation code           */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;              /* primary return code           */
  AP_UINT32        secondary_rc;            /* secondary return code         */

  unsigned char    enabled;                 /* AP_YES or AP_NO               */
  unsigned char    reserv3[3];              /* reserved                      */

} SET_CENTRAL_LOGGING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_CENTRAL_LOGGING                                          */
/*                                                                           */
/* Description: Discover the state of central logging (on/off).              */
/*****************************************************************************/

typedef struct query_central_logging
{
  AP_UINT16        opcode;                  /* verb operation code           */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;              /* primary return code           */
  AP_UINT32        secondary_rc;            /* secondary return code         */

  unsigned char    enabled;                 /* AP_YES or AP_NO               */
  unsigned char    reserv3[3];              /* reserved                      */

} QUERY_CENTRAL_LOGGING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_CENTRAL_LOGGER                                           */
/*                                                                           */
/* Description: Query location of central logger.                            */
/*****************************************************************************/
typedef struct query_central_logger
{
  AP_UINT16        opcode;                  /* verb operation code           */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;              /* primary return code           */
  AP_UINT32        secondary_rc;            /* secondary return code         */
  unsigned char    reserv3[4];              /* reserved                      */

  unsigned char    node_name[128];          /* name of server                */

} QUERY_CENTRAL_LOGGER;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_GLOBAL_LOG_TYPE                                          */
/*                                                                           */
/* Description: Query global log type.  Must be sent on a target handle      */
/*              corresponding to the current global logger.                  */
/*****************************************************************************/
typedef struct query_global_log_type
{
  AP_UINT16        opcode;                  /* verb operation code           */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;              /* primary return code           */
  AP_UINT32        secondary_rc;            /* secondary return code         */

  unsigned char    audit;                   /* audit logging on or off       */
  unsigned char    exception;               /* exception logging on or off   */
  unsigned char    succinct_audits;         /* succinst display of audits    */
  unsigned char    succinct_errors;         /* succinst display of errors    */
  unsigned char    reserv3[4];

} QUERY_GLOBAL_LOG_TYPE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_LOG_FILE                                                   */
/*                                                                           */
/* Description: Set log file on a particular server or globally.             */
/*****************************************************************************/

typedef struct set_log_file
{
  AP_UINT16      opcode;                     /* verb operation code          */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                 /* primary return code          */
  AP_UINT32      secondary_rc;               /* secondary return code        */
  unsigned char  log_file_type;              /* type of log file             */
  unsigned char  action;                     /* reset and/or backup existing */
  unsigned char  file_name[81];              /* file name                    */
  unsigned char  backup_file_name[81];       /* backup file                  */
  AP_UINT32      file_size;                  /* log file size                */
  unsigned char  succinct;                   /* succinct logging on or off   */
  unsigned char  reserv3[3];                /* reserved                      */
} SET_LOG_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LOG_FILE                                                 */
/*                                                                           */
/* Description: Query log file on a particular server.                       */
/*****************************************************************************/
typedef struct query_log_file
{
  AP_UINT16      opcode;                      /* verb operation code         */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                  /* primary return code         */
  AP_UINT32      secondary_rc;                /* secondary return code       */

  unsigned char  log_file_type;               /* type of log file            */
  unsigned char  file_name[81];               /* file name                   */
  unsigned char  backup_file_name[81];        /* backup file                 */
  AP_UINT32      file_size;                   /* log file size               */
  unsigned char  succinct;                    /* succinct logging on or off  */
  unsigned char  reserv3[3];                /* reserved                      */
} QUERY_LOG_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_DIAGNOSTICS_DIRECTORY                                      */
/*                                                                           */
/* Description: Set the directory to contain log and trace files             */
/*****************************************************************************/

typedef struct set_diagnostics_directory
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */

  unsigned char  directory_name[81];        /* directory name                */
  unsigned char  reserv3[4];                /* reserved                      */
} SET_DIAGNOSTICS_DIRECTORY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DIAGNOSTICS_DIRECTORY                                    */
/*                                                                           */
/* Description: Query the directory containing log and trace files           */
/*****************************************************************************/

typedef struct query_diagnostics_directory
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */

  unsigned char  directory_name[81];        /* directory name                */
  unsigned char  reserv3[4];                /* reserved                      */
} QUERY_DIAGNOSTICS_DIRECTORY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_LOG_TYPE                                                   */
/*                                                                           */
/* Description: SET_LOG_TYPE verb control block                              */
/*****************************************************************************/

typedef struct set_log_type
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */

  unsigned char  override;                  /* override global settings?     */
  unsigned char  audit;                     /* audit logging on or off       */
  unsigned char  exception;                 /* exception logging on or off   */
  unsigned char  succinct_audits;           /* succinst display of audits    */
  unsigned char  succinct_errors;           /* succinst display of errors    */
  unsigned char  reserv3[3];                /* reserved                      */

} SET_LOG_TYPE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_GLOBAL_LOG_TYPE                                            */
/*                                                                           */
/* Description: Set log type globally.  Must be issued on the target_handle  */
/*              corresponding to the current central logger.                 */
/*****************************************************************************/

typedef struct set_global_log_type
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */

  unsigned char  audit;                     /* audit logging on or off       */
  unsigned char  exception;                 /* exception logging on or off   */
  unsigned char  succinct_audits;           /* succinst display of audits    */
  unsigned char  succinct_errors;           /* succinst display of errors    */
  unsigned char  reserv3[4];                /* reserved                      */

} SET_GLOBAL_LOG_TYPE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_TRACE_FILE                                                 */
/*                                                                           */
/* Description: SET_TRACE_FILE verb control block                            */
/*****************************************************************************/

typedef struct set_trace_file
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  trace_file_type;                   /* type of trace file    */
  unsigned char  dual_files;                        /* dual trace files      */
  AP_UINT32      trace_file_size;                   /* trace file size       */
  unsigned char  reserv3[4];                        /* reserved              */
  unsigned char  file_name[81];                     /* file name             */
  unsigned char  file_name_2[81];                   /* second file name      */
} SET_TRACE_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_TRACE_TYPE                                                 */
/*                                                                           */
/* Description:  SET_TRACE_TYPE verb control block                           */
/*****************************************************************************/

typedef struct set_trace_type
{
  AP_UINT16      opcode;                   /* verb operation code            */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;               /* primary return code            */
  AP_UINT32      secondary_rc;             /* secondary return code          */
  AP_UINT32      trace_flags;              /* trace flags                    */
  AP_UINT32      truncation_length;        /* truncate each msg to this size */
  unsigned char  init_flags;               /* TRUE if initializing flags     */
  unsigned char  set_flags;                /* TRUE if setting flags          */
                                           /* FALSE if unsetting flags       */
  unsigned char  set_internal;             /* reserved                       */
  AP_UINT16      internal_level;           /* reserved                       */
  unsigned char  reserv3[4];                /* reserved                      */
} SET_TRACE_TYPE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LOG_TYPE                                                 */
/*                                                                           */
/* Description: QUERY_LOG_TYPE verb control block                            */
/*****************************************************************************/

typedef struct query_log_type
{
  AP_UINT16      opcode;                      /* verb operation code         */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                  /* primary return code         */
  AP_UINT32      secondary_rc;                /* secondary return code       */

  unsigned char  override;                    /* override global values?     */
  unsigned char  audit;                       /* audit logging on or off     */
  unsigned char  exception;                   /* exception logging on or off */
  unsigned char  succinct_audits;           /* succinst display of audits    */
  unsigned char  succinct_errors;           /* succinst display of errors    */
  unsigned char  reserv3[3];                /* reserved                      */

} QUERY_LOG_TYPE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TRACE_FILE                                               */
/*                                                                           */
/* Description: QUERY_TRACE_FILE verb control block                          */
/*****************************************************************************/

typedef struct query_trace_file
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  trace_file_type;                   /* type of trace file    */
  unsigned char  dual_files;                        /* dual trace files      */
  AP_UINT32      trace_file_size;                   /* trace file size       */
  unsigned char  reserv3[4];                        /* reserved              */
  unsigned char  file_name[81];                     /* file name             */
  unsigned char  file_name_2[81];                   /* second file name      */
} QUERY_TRACE_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TRACE_TYPE                                               */
/*                                                                           */
/* Description: QUERY_TRACE_TYPE verb control block                          */
/*****************************************************************************/

typedef struct query_trace_type
{
  AP_UINT16      opcode;                   /* verb operation code            */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;               /* primary return code            */
  AP_UINT32      secondary_rc;             /* secondary return code          */
  AP_UINT32      trace_flags;              /* trace flags                    */
  AP_UINT32      truncation_length;        /* truncate each msg to this size */
  AP_UINT16      internal_level;           /* reserved                       */
  unsigned char  reserv3[4];                /* reserved                      */
} QUERY_TRACE_TYPE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_AVAILABLE_TP                                             */
/*                                                                           */
/* Description: Query Transaction Program information from the Service Mgr.  */
/*****************************************************************************/
typedef struct query_available_tp
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  *buf_ptr;                  /* pointer to buffer             */
  AP_UINT32      buf_size;                  /* buffer size                   */
  AP_UINT32      total_buf_size;            /* total buffer size required    */
  AP_UINT16      num_entries;               /* number of entries             */
  AP_UINT16      total_num_entries;         /* total number of entries       */
  unsigned char  list_options;              /* listing options               */
  unsigned char  reserv3[3];                /* reserved                      */
  unsigned char  tp_name[64];               /* Transaction Program name      */
  unsigned char  system_name[128];          /* The system the TP is at       */
} QUERY_AVAILABLE_TP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: AVAILABLE_TP_DATA                                              */
/*                                                                           */
/* Description: Detail overlay returned for QUERY_AVAILABLE_TP               */
/*****************************************************************************/
typedef struct available_tp_data
{
  AP_UINT16      overlay_size;
  unsigned char  tp_name[64];               /* Transaction Program name      */
  unsigned char  reserv4[4];                /* reserved                      */
  unsigned char  system_name[128];          /* The system the TP is at       */

} AVAILABLE_TP_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: EMULATOR_USER_DEF_DATA                                         */
/*                                                                           */
/* Description: Defined data for an emulator user.  The session data         */
/*              (if any) is appended to the main VCB.                        */
/*****************************************************************************/

typedef struct emulator_user_def_data
{
  unsigned char     description[32];  /* Description - null terminated       */
  unsigned char     reserv3[16];      /* reserved                            */
  unsigned char     style_file[9];    /* Initial style file name - null      */
                                      /* terminated                          */
  unsigned char     reserv1[3];       /* reserved                            */
  AP_UINT32         num_sessions;     /* Number of sessions being added      */
  AP_UINT32         max_act_sessions; /* Max number of active 3270 sessions  */
  unsigned char     view_rtm;         /* Can user view RTM info?             */
  unsigned char     alert_permission; /* Can user send alerts?               */
  unsigned char     change_lu;        /* Can use change LU/Pool accessed?    */
  unsigned char     change_style;     /* Can user modify initial style file? */
  unsigned char     user_is_group;    /* Does user_name specify a UNIX group?*/
  unsigned char     status;           /* reserved                            */
  unsigned char     reserv2[14];      /* reserved                            */
} EMULATOR_USER_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SESSION_3270_DEF_DATA                                          */
/*                                                                           */
/* Description: Defined data for a 3270 session                              */
/*****************************************************************************/

typedef struct session_3270_def_data
{
  unsigned char     lu_name[8];       /* LU/pool name accessed               */
  unsigned char     session_type;     /* Session type (2-5 or printer)       */
  unsigned char     model_override;   /* Can user override model?            */
} SESSION_3270_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SESSION_5250_DEF_DATA                                          */
/*                                                                           */
/* Description: Defined data for a 5250 session                              */
/*****************************************************************************/

typedef struct session_5250_def_data
{
  unsigned char     local_lu_alias[8];  /* Local LU alias                    */
  unsigned char     plu_alias[8];       /* Partner LU alias                  */
  unsigned char     fqplu_name[17];     /* FQPLU name                        */
  unsigned char     mode_name[8];       /* Mode name                         */
  unsigned char     session_type;       /* session type - display or printer */
} SESSION_5250_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SESSION_DEF_DATA                                               */
/*                                                                           */
/* Description: Defined data for a 3270 or 5250 session                      */
/*****************************************************************************/

typedef struct session_def_data
{
  AP_UINT16        sub_overlay_size; /* size of this session structure       */
  unsigned char    session_name[8];  /* AP_INT32 session name                    */
  unsigned char    emulator_type;    /* Emulator type - 3270 or 5250         */
  unsigned char    reserv1;          /* reserved                             */
  unsigned char    description[32];  /* session description                  */
  unsigned char    reserv2[16];      /* reserved                             */
  union
  {
    SESSION_3270_DEF_DATA  def_data_3270;  /* definition of 3270 session     */
    SESSION_5250_DEF_DATA  def_data_5250;  /* definition of 5250 session     */
  } session_variant;
  unsigned char    reserv3[4];       /* reserved                             */
  AP_UINT32        reserv4;          /* reserved                             */
} SESSION_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_EMULATOR_USER                                           */
/*                                                                           */
/* Description: Define characteristics of an emulator user.                  */
/*              The session data is appended to the main VCB.                */
/*****************************************************************************/

typedef struct define_emulator_user
{
  AP_UINT16        opcode;               /* verb operation code              */
  unsigned char    reserv2;              /* reserved                         */
  unsigned char    format;               /* reserved                         */
  AP_UINT16        primary_rc;           /* primary return code              */
  AP_UINT32        secondary_rc;         /* secondary return code            */
  unsigned char    user_name[32];        /* emulator user name               */
  EMULATOR_USER_DEF_DATA def_data;       /* defined data                     */
} DEFINE_EMULATOR_USER;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_EMULATOR_USER                                           */
/*                                                                           */
/* Description: Delete a user or user's sessions                             */
/*****************************************************************************/

typedef struct delete_emulator_user
{
  AP_UINT16          opcode;            /* verb operation code               */
  unsigned char      reserv2;           /* reserved                          */
  unsigned char      format;            /* reserved                          */
  AP_UINT16          primary_rc;        /* primary return code               */
  AP_UINT32          secondary_rc;      /* secondary return code             */
  unsigned char      user_name[32];     /* user name                         */
  AP_UINT32          num_sessions;      /* number of sessions to delete      */
  AP_UINT16          delete_options;    /* delete all sessions/delete user?  */
  unsigned char      session_names[10][8]; /* specified session names        */
} DELETE_EMULATOR_USER;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: EMULATOR_USER_SUMMARY                                          */
/*                                                                           */
/* Description: Summary overlay returned by QUERY_EMULATOR_USER_DEF          */
/*****************************************************************************/

typedef struct emulator_user_summary
{
  AP_UINT16        overlay_size;        /* overlay size                      */
  unsigned char    user_name[32];       /* user name                         */
  unsigned char    reserv4[4];          /* reserved                          */
} EMULATOR_USER_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: EMULATOR_USER_DETAIL                                           */
/*                                                                           */
/* Description: Detailed overlay returned by QUERY_EMULATOR_USER_DEF         */
/*****************************************************************************/

typedef struct emulator_user_detail
{
  AP_UINT16        overlay_size;        /* overlay size                      */
  AP_UINT16        sub_overlay_offset;  /* offset to first session structure */
  unsigned char    user_name[32];       /* user name                         */
  AP_UINT32        num_filtered_sessions; /* total number of sessions of the */
                                          /* requested type                  */
  unsigned char    reserv4[4];            /* reserved                        */
  EMULATOR_USER_DEF_DATA def_data;        /* user definition                 */
} EMULATOR_USER_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_EMULATOR_USER_DEF                                        */
/*                                                                           */
/* Description: Query configured users/sessions                              */
/*****************************************************************************/

typedef struct query_emulator_user_def
{
  AP_UINT16          opcode;               /* verb operation code            */
  unsigned char      reserv2;              /* reserved                       */
  unsigned char      format;               /* reserved                       */
  AP_UINT16          primary_rc;           /* primary return code            */
  AP_UINT32          secondary_rc;         /* secondary return code          */
  unsigned char     *buf_ptr;              /* pointer to buffer              */
  AP_UINT32          buf_size;             /* buffer size                    */
  AP_UINT32          total_buf_size;       /* total buffer size required     */
  AP_UINT16          num_entries;          /* number of entries              */
  AP_UINT16          total_num_entries;    /* total number of entries        */
  unsigned char      list_options;         /* listing options                */
  unsigned char      reserv3;              /* reserved                       */
  unsigned char      user_name[32];        /* emulator user name             */
  unsigned char      session_name[8];      /* session name                   */
  unsigned char      emulator_type;        /* type of entries to return      */
  unsigned char      reserv4;              /* reserved                       */
  AP_UINT32          num_init_sessions;    /* number of sessions for first   */
                                           /* user when starting in middle   */
  AP_UINT32          num_last_sessions;    /* number of sessions on last     */
                                           /* detail overlay if last user is */
                                           /* incomplete                     */
  unsigned char      last_user_incomplete; /* set to AP_YES if session data  */
                                           /* for last user is incomplete    */
} QUERY_EMULATOR_USER_DEF;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_3270_USER                                                */
/*                                                                           */
/* Description: Query 3270 user UNIX ID, System name etc.                    */
/*              The overlay buffer always contains complete information,     */
/*              however - users are not split across verbs.                  */
/*****************************************************************************/
typedef struct query_3270_user
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  *buf_ptr;                  /* pointer to buffer             */
  AP_UINT32      buf_size;                  /* buffer size                   */
  AP_UINT32      total_buf_size;            /* total buffer size required    */
  AP_UINT16      num_entries;               /* number of entries             */
  AP_UINT16      total_num_entries;         /* total number of entries       */
  unsigned char  list_options;              /* listing options               */
  unsigned char  reserv3[5];                /* reserved                      */
  AP_UINT16      scope;                     /* What parameters to match with */
  unsigned char  user_name[32];             /* 3270 user name                */
  unsigned char  system_name[128];          /* System name                   */
  AP_UINT32      user_pid;                  /* 3270 user PID                 */

} QUERY_3270_USER;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: USER_3270_DETAIL                                               */
/*                                                                           */
/* Description: Detail overlay returned for QUERY_3270_USER                  */
/*****************************************************************************/
typedef struct user_3270_detail
{
  AP_UINT16      overlay_size;
  unsigned char  user_name[32];                /* 3270 user name             */
  unsigned char  system_name[128];             /* System name                */
  AP_UINT32      user_pid;                     /* 3270 user PID              */
  AP_UINT32      reserv1;                      /* reserved                   */
  AP_UINT32      user_uid;                     /* The User's User ID         */
  AP_UINT32      user_gid;                     /* The User's Group ID        */
  unsigned char  user_gname[32];               /* The User's group name      */
  AP_UINT32      user_session_count;           /* Count of LU session names  */
  AP_UINT32      user_start_time;              /* Time at which User Started */
  unsigned char  reserv3[4];                   /* reserved                   */

} USER_3270_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: USER_3270_SUMMARY                                              */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_3270_USER                 */
/*****************************************************************************/
typedef struct user_3270_summary
{
  AP_UINT16      overlay_size;
  unsigned char  user_name[32];                /* 3270 user name             */
  unsigned char  system_name[128];             /* System name                */
  AP_UINT32      user_pid;                     /* 3270 user PID              */
  unsigned char  reserv3[4];                   /* reserved                   */

} USER_3270_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_3270_USER_SESSIONS                                       */
/*                                                                           */
/* Description: Query 3270 user session details for a particular user.       */
/*              Can be split over multiple QUERY verbs.  The overlay buffer  */
/*              always contains complete information, however - sessions     */
/*              are not split across verbs.                                  */
/*****************************************************************************/
typedef struct query_3270_user_sessions
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  *buf_ptr;                  /* pointer to buffer             */
  AP_UINT32      buf_size;                  /* buffer size                   */
  AP_UINT32      total_buf_size;            /* total buffer size required    */
  AP_UINT16      num_entries;               /* number of entries             */
  AP_UINT16      total_num_entries;         /* total number of entries       */
  unsigned char  list_options;              /* listing options               */
  unsigned char  reserv3[5];                /* reserved                      */
  unsigned char  user_name[32];             /* 3270 user name                */
  unsigned char  system_name[128];          /* System name                   */
  AP_UINT32      user_pid;                  /* 3270 user PID                 */
  unsigned char  lu_name[8];                /* Session to start from         */

} QUERY_3270_USER_SESSIONS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: USER_3270_SESSION_DETAIL                                       */
/*                                                                           */
/* Description: Detail overlay returned for QUERY_3270_USER_SESSIONS         */
/*****************************************************************************/
typedef struct user_3270_session_detail
{
  AP_UINT16      overlay_size;
  unsigned char  lu_name[8];                /* LU name                       */
  unsigned char  nau_address;               /* NAU address                   */
  unsigned char  sess_type;                 /* 3270 session type             */
  unsigned char  reserv1[6];                /* Reserved                      */

} USER_3270_SESSION_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: USER_3270_SESSION_SUMMARY                                      */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_3270_USER_SESSIONS        */
/*****************************************************************************/
typedef struct user_3270_session_summary
{
  AP_UINT16      overlay_size;
  unsigned char  lu_name[8];                /* LU name                       */
  unsigned char  reserv1[4];                /* Reserved                      */

} USER_3270_SESSION_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DIAG_3270_DATA                                                 */
/*                                                                           */
/* Description: 3270 diagnostic data                                         */
/*              The user alert data is appended to the main VCB.             */
/*****************************************************************************/

typedef struct diag_3270_data
{
  unsigned char  rtm_overflow;          /* Send RTM data at counter overflow */
  unsigned char  rtm_unbind;            /* Send RTM data at UNBIND           */
  unsigned char  rtm_timer_option;      /* RTM timers option                 */
  unsigned char  reserv1;               /* reserved                          */
  AP_UINT16      rtm_thresh1;           /* RTM threshold #1                  */
  AP_UINT16      rtm_thresh2;           /* RTM threshold #2                  */
  AP_UINT16      rtm_thresh3;           /* RTM threshold #3                  */
  AP_UINT16      rtm_thresh4;           /* RTM threshold #4                  */
  AP_UINT16      num_alerts;            /* Number of user alerts             */
  unsigned char  reserv2[6];            /* Reserved                          */
} DIAG_3270_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ALERT_3270_DATA                                                */
/*                                                                           */
/* Description: 3270 User Alert data                                         */
/*****************************************************************************/

typedef struct alert_3270_data
{
  AP_UINT16      overlay_size;
  unsigned char description[53];                              /* description */
  unsigned char parameter1[33];                               /* parameter 1 */
  unsigned char parameter2[33];                               /* parameter 2 */
  unsigned char parameter3[33];                               /* parameter 3 */
  unsigned char reserv1[8];                                   /* Reserved    */
} ALERT_3270_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_3270_DIAG                                               */
/*                                                                           */
/* Description: Define 3270 diagnostic data.                                 */
/*              The user alerts are appended to the main VCB.                */
/*****************************************************************************/

typedef struct define_3270_diag
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  DIAG_3270_DATA def_data;                          /* defined data          */
} DEFINE_3270_DIAG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_3270_DIAG                                                */
/*                                                                           */
/* Description: Query 3270 diagnostic data.                                  */
/*              Returns buffer containing ALERT_3270_DATA overlays.          */
/*****************************************************************************/

typedef struct query_3270_diag
{
  AP_UINT16      opcode;                       /* verb operation code        */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                   /* primary return code        */
  AP_UINT32      secondary_rc;                 /* secondary return code      */
  unsigned char  *buf_ptr;                     /* pointer to buffer          */
  AP_UINT32      buf_size;                     /* buffer size                */
  AP_UINT32      total_buf_size;               /* total buffer size required */
  AP_UINT16      num_entries;                  /* number of entries          */
  AP_UINT16      total_num_entries;            /* total number of entries    */
  unsigned char  list_options;                 /* listing options            */
  unsigned char  reserv3;                      /* reserved                   */
  AP_UINT16      alert_number;                 /* index into alerts          */
  DIAG_3270_DATA def_data;
} QUERY_3270_DIAG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RJE_WKSTN_DEF_DATA                                             */
/*                                                                           */
/* Description: Defined data for an RJE workstation.                         */
/*****************************************************************************/

typedef struct rje_wkstn_def_data
{
  unsigned char  description[32];     /* Description - null terminated       */
  unsigned char  reserv1[16];         /* reserved                            */
  unsigned char  primary_user[32];    /* The primary user name               */
  unsigned char  group_name[32];      /* The user's group name               */
  unsigned char  system_name[128];    /* computer where workstation runs     */
  AP_UINT16      num_lus;             /* count field for the number of LUs   */
  unsigned char  reserv4[18];         /* Reserved                            */
  unsigned char  lu_name[5][8];       /* LU or LU Group used by workstation  */
                                      /* (up to 5)                           */
} RJE_WKSTN_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_RJE_WKSTN                                               */
/*                                                                           */
/* Description: Define characteristics of an rje workstation.                */
/*****************************************************************************/

typedef struct define_rje_wkstn
{
  AP_UINT16          opcode;                        /* verb operation code   */
  unsigned char      reserv2;                       /* reserved              */
  unsigned char      format;
  AP_UINT16          primary_rc;                    /* primary return code   */
  AP_UINT32          secondary_rc;                  /* secondary return code */
  unsigned char      workstation_name[4];           /* RJE workstation name  */
  RJE_WKSTN_DEF_DATA def_data;                      /* defined data          */
} DEFINE_RJE_WKSTN;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_RJE_WKSTN                                               */
/*                                                                           */
/* Description: Delete an RJE workstation.                                   */
/*****************************************************************************/

typedef struct delete_rje_wkstn
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;                           /* reserved              */
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  workstation_name[4];               /* RJE workstation name  */
} DELETE_RJE_WKSTN;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_RJE_WKSTN_DEF                                            */
/*                                                                           */
/* Description: Query RJE workstation definition.                            */
/*****************************************************************************/

typedef struct query_rje_wkstn_def
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;                   /* reserved                      */
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  *buf_ptr;                  /* pointer to buffer             */
  AP_UINT32      buf_size;                  /* buffer size                   */
  AP_UINT32      total_buf_size;            /* total buffer size required    */
  AP_UINT16      num_entries;               /* number of entries             */
  AP_UINT16      total_num_entries;         /* total number of entries       */
                                            /* - includes partial entries    */
  unsigned char  list_options;              /* listing options               */
  unsigned char  reserv3[3];                /* reserved                      */
  unsigned char  workstation_name[4];       /* RJE workstation name          */
} QUERY_RJE_WKSTN_DEF;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RJE_WKSTN_DEF_SUMMARY                                          */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_RJE_WKSTN_DEF             */
/*****************************************************************************/

typedef struct rje_wkstn_def_summary
{
  AP_UINT16      overlay_size;              /* overlay size                  */
  unsigned char workstation_name[4];        /* RJE workstation name          */
  unsigned char  reserv3[8];                /* Reserved                      */
} RJE_WKSTN_DEF_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RJE_WKSTN_DEF_DETAIL                                           */
/*                                                                           */
/* Description: Detail overlay returned for QUERY_RJE_WKSTN_DEF              */
/*****************************************************************************/

typedef struct rje_wkstn_def_detail
{
  AP_UINT16      overlay_size;              /* overlay size                  */
  unsigned char workstation_name[4];        /* RJE workstation name          */
  unsigned char  reserv3[8];                /* Reserved                      */
  RJE_WKSTN_DEF_DATA   def_data;            /* defined data                  */
} RJE_WKSTN_DEF_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_RJE_WKSTN                                                */
/*                                                                           */
/* Description: Query RJE workstation UNIX ID, System name etc.              */
/*              The overlay buffer always contains complete information,     */
/*              however - users are not split across verbs.                  */
/*****************************************************************************/
typedef struct query_rje_wkstn
{
  AP_UINT16      opcode;                    /* verb operation code           */
  unsigned char  reserv2;                   /* reserved                      */
  unsigned char  format;
  AP_UINT16      primary_rc;                /* primary return code           */
  AP_UINT32      secondary_rc;              /* secondary return code         */
  unsigned char  *buf_ptr;                  /* pointer to buffer             */
  AP_UINT32      buf_size;                  /* buffer size                   */
  AP_UINT32      total_buf_size;            /* total buffer size required    */
  AP_UINT16      num_entries;               /* number of entries             */
  AP_UINT16      total_num_entries;         /* total number of entries       */
  unsigned char  list_options;              /* listing options               */
  unsigned char  reserv3;                   /* reserved                      */
  AP_UINT16      scope;                     /* What parameters to match with */
  AP_UINT16      reserv1;                   /* reserved                      */
  unsigned char  workstation_name[4];       /* RJE workstation name          */
  unsigned char  system_name[128];          /* System name                   */
  AP_UINT32      wkstn_pid;                 /* RJE workstation PID           */
  unsigned char  reserv4[8];                /* Reserved                      */

} QUERY_RJE_WKSTN;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RJE_WKSTN_DETAIL                                               */
/*                                                                           */
/* Description: Detail overlay returned for QUERY_RJE_WKSTN                  */
/*****************************************************************************/
typedef struct rje_wkstn_detail
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  unsigned char  workstation_name[4];          /* RJE workstation name       */
  unsigned char  system_name[128];             /* System name                */
  AP_UINT32      wkstn_pid;                    /* RJE workstation PID        */
  AP_UINT32      reserv1;                      /* reserved                   */
  AP_UINT32      wkstn_uid;                    /* The User's User ID         */
  AP_UINT32      wkstn_gid;                    /* The User's Group ID        */
  unsigned char  wkstn_gname[32];              /* The User's group name      */
  AP_UINT32      wkstn_session_count;          /* Count of LU session names  */
  AP_UINT32      wkstn_start_time;             /* Time at which User Started */
  unsigned char  reserv3[8];                   /* Reserved                   */

} RJE_WKSTN_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RJE_WKSTN_SUMMARY                                              */
/*                                                                           */
/* Description: Summary overlay returned for QUERY_RJE_WKSTN                 */
/*****************************************************************************/
typedef struct rje_wkstn_summary
{
  AP_UINT16      overlay_size;                 /* overlay size               */
  unsigned char  workstation_name[4];          /* RJE workstation name       */
  unsigned char  system_name[128];             /* System name                */
  AP_UINT32      wkstn_pid;                    /* RJE workstation PID        */
  unsigned char  reserv3[8];                   /* Reserved                   */

} RJE_WKSTN_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: CONFIG_FILE_HEADER                                             */
/*                                                                           */
/* Description: Header record for configuration file                         */
/*****************************************************************************/

typedef struct config_file_header
{
  AP_UINT16      major_version;                   /* major version           */
  AP_UINT16      minor_version;                   /* minor version           */
  AP_UINT16      update_release;                  /* update release          */
  AP_UINT32      revision_level;                  /* revision level          */
  unsigned char  comment[100];                    /* optional comment string */
  AP_UINT16      updating;                        /* reserved                */
} CONFIG_FILE_HEADER;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_DOMAIN_CONFIG_FILE                                      */
/*                                                                           */
/* Description: Define config file header                                    */
/*****************************************************************************/

typedef struct define_domain_config_file
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  reserv3[8];                        /* Reserved              */
  CONFIG_FILE_HEADER hdr;                           /* defined data          */
} DEFINE_DOMAIN_CONFIG_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DOMAIN_CONFIG_FILE                                       */
/*                                                                           */
/* Description: Query config file header                                     */
/*****************************************************************************/

typedef struct query_domain_config_file
{
  AP_UINT16      opcode;                            /* verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  reserv3[8];                        /* Reserved              */
  CONFIG_FILE_HEADER   hdr;
} QUERY_DOMAIN_CONFIG_FILE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_RCF_ACCESS                                              */
/*                                                                           */
/* Description: Defines access to UCF and SPCF                               */
/*****************************************************************************/

typedef struct define_rcf_access
{
  AP_UINT16      opcode;                            /* Verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  ucf_username[32];                  /* UCF username          */
  AP_UINT32      spcf_permissions;                  /* SPCF permissions      */
  unsigned char  reserv3[8];                        /* Reserved              */
} DEFINE_RCF_ACCESS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_RCF_ACCESS                                              */
/*                                                                           */
/* Description: Removes access to UCF and SPCF                               */
/*****************************************************************************/

typedef struct delete_rcf_access
{
  AP_UINT16      opcode;                            /* Verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
} DELETE_RCF_ACCESS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_RCF_ACCESS                                               */
/*                                                                           */
/* Description:  Queries defined access to UCF and SPCF                      */
/*****************************************************************************/

typedef struct query_rcf_access
{
  AP_UINT16      opcode;                            /* Verb operation code   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;                        /* primary return code   */
  AP_UINT32      secondary_rc;                      /* secondary return code */
  unsigned char  ucf_username[32];                  /* UCF username          */
  AP_UINT32      spcf_permissions;                  /* SPCF permissions      */
  unsigned char  reserv3[8];                        /* Reserved              */
} QUERY_RCF_ACCESS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_KERNEL_MEMORY_LIMIT                                        */
/*                                                                           */
/* Description: SET_KERNEL_MEMORY_LIMIT verb control block                   */
/*****************************************************************************/

typedef struct set_kernel_memory_limit
{
  AP_UINT16      opcode;               /* verb operation code                */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;           /* primary return code                */
  AP_UINT32      secondary_rc;         /* secondary return code              */
  AP_UINT32      limit;                /* kernel memory limit, 0 => no limit */
  unsigned char  reserv3[8];           /* Reserved                           */
} SET_KERNEL_MEMORY_LIMIT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_KERNEL_MEMORY_LIMIT                                      */
/*                                                                           */
/* Description:  QUERY_KERNEL_MEMORY_LIMIT verb control block                */
/*****************************************************************************/

typedef struct query_kernel_memory_limit
{
  AP_UINT16      opcode;               /* verb operation code                */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;           /* primary return code                */
  AP_UINT32      secondary_rc;         /* secondary return code              */
  AP_UINT32      limit;                /* kernel memory limit, 0 => no limit */
  AP_UINT32      actual;               /* current amount of memory allocated */
  AP_UINT32      max_used;             /* maximum amount of memory allocated */
  unsigned char  reset_max_used;       /* set max_used = actual              */
  unsigned char  reserv3[8];           /* Reserved                           */
} QUERY_KERNEL_MEMORY_LIMIT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_BUFFER_AVAILABILITY                                        */
/*                                                                           */
/* Description: Set STREAMS buffer availability - this is used               */
/*              as part of the APPN adaptive pacing algorithm.               */
/*****************************************************************************/

typedef struct set_buffer_availability
{
  AP_UINT16      opcode;            /* verb operation code                   */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;        /* primary return code                   */
  AP_UINT32      secondary_rc;      /* secondary return code                 */
  AP_UINT32      buf_avail;         /* buffer availability                   */
  unsigned char  reserv3[8];        /* Reserved                              */
} SET_BUFFER_AVAILABILITY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_BUFFER_AVAILABILITY                                      */
/*                                                                           */
/* Description: Query STREAMS buffer availability and other                  */
/*              parameters used in the adaptive pacing algorithm.            */
/*****************************************************************************/

typedef struct query_buffer_availability
{
  AP_UINT16      opcode;               /* verb operation code                */
  unsigned char  reserv2;
  unsigned char  format;
  AP_UINT16      primary_rc;           /* primary return code                */
  AP_UINT32      secondary_rc;         /* secondary return code              */
  unsigned char  reset_max_values;     /* reset maximum values               */
  AP_UINT32      buf_avail;            /* buffer availablity                 */
  AP_UINT32      buf_total_count;      /* current buffer usage - count       */
  AP_UINT32      buf_total_bytes;      /*                      - bytes       */
  AP_UINT32      buf_rsrv_count;       /* buffers reserved     - count       */
  AP_UINT32      buf_rsrv_bytes[2];    /*                      - byte        */
  AP_UINT32      buf_res_use_count;    /* usage of reserved buffers - count  */
  AP_UINT32      buf_res_use_bytes;    /*                           - bytes  */
  AP_UINT32      peak_usage;           /* peak usage                         */
  AP_UINT32      peak_decay;           /* peak decay                         */
  unsigned char  throttle_status;      /* throttle status                    */
  unsigned char  buf_use_status;       /* congestion status                  */
  AP_UINT32      max_buf_total_count;  /* maximum buffer usage - count       */
  AP_UINT32      max_buf_total_bytes;  /*                      - bytes       */
  AP_UINT32      max_buf_rsrv_count;   /* max buffers reserved - count       */
  AP_UINT32      max_buf_rsrv_bytes[2];/*                      - bytes       */
  AP_UINT32      max_buf_res_use_count;/* max usage of resvd buffers - count */
  AP_UINT32      max_buf_res_use_bytes;/*                            - bytes */
  AP_UINT32      max_peak_usage;       /* maximum peak usage                 */
  unsigned char  max_throttle_status;  /* maximum throttle status            */
  unsigned char  max_buf_use_status;   /* maximum congestion status          */
  unsigned char  debug_param[32];      /* array of debug parameters          */
                                       /* - for future use                   */
  unsigned char  reserv3[8];           /* Reserved                           */

} QUERY_BUFFER_AVAILABILITY;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* Start of DLC-specific data.                                               */
/* On any system, only some of the following structures are relevant.        */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure: V0_MUX_INFO                                                    */
/*                                                                           */
/* Description: Streams information for DLCs.                                */
/*****************************************************************************/

typedef struct v0_mux_info
{
  AP_UINT16      dlc_type;                  /* DLC implementation type       */
  unsigned char  need_vrfy_fixup;           /* LS needs verification with    */
                                            /* corresponding DLC             */
  unsigned char  num_mux_ids;               /* number of mux IDs in array    */
  AP_UINT32      card_type;                 /* adapter type used by the DLC  */
  AP_UINT32      adapter_number;            /* DLC adapter number            */
  AP_UINT32      oem_data_length;           /* len of data from OEM exit     */
  AP_INT32            mux_ids[5];                /* array of mux IDs              */

} V0_MUX_INFO;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* DLC-specific data structures for SDLC                                     */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SDH_CREATE_PARMS                                               */
/*                                                                           */
/* Description: Creation parameters used in SDLC                             */
/*              - used on DEFINE_DLC for SDLC                                */
/*****************************************************************************/

typedef struct sdh_create_parms
{
  AP_UINT16      length;             /* Length of this structure             */
  AP_UINT16      num_ports;          /* max number of ports DLC can support  */
  AP_UINT32      creators_pid;       /* process ID of DLC                    */
  V0_MUX_INFO    mux_info;           /* streams information                  */
} SDH_CREATE_PARMS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SDL_SPEC_DATA                                                  */
/*                                                                           */
/* Description: DLC-specific data for SDLC - used on DEFINE_DLC              */
/*****************************************************************************/

typedef struct sdl_spec_data
{
  V0_MUX_INFO      mux_info;         /* streams information                  */
  AP_UINT16        mu_credit;        /* amount of credit to allow PC to send */
  unsigned char    stats_support;    /* activate statistics gathering?       */
  unsigned char    reserve1;         /* pad                                  */
  AP_UINT16        sdh_parms_len;    /* Length of sdh_parms structure        */
  SDH_CREATE_PARMS sdh_parms;        /* HMOD stub create_parms structure     */
} SDL_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   STUB_SPEC_DATA                                               */
/*                                                                           */
/* Description: HMOD specific data for SDLC - used on DEFINE_PORT and        */
/*                                            DEFINE_LS                      */
/*****************************************************************************/

typedef struct stub_spec_data
{
  AP_INT32       mux_id;            /* Stream info for STREAMS-based HMODs   */
  unsigned char  opt1;              /* options flag 1                        */
  unsigned char  opt2;              /* options flag 2                        */
  unsigned char  pad[2];
  AP_UINT32      linesp;            /* line speed in bps                     */
                                    /*  ignored if opt1 says clocking is     */
                                    /*  external                             */
  AP_UINT16      rcv_pool_size;     /* initial number of buffs for rcv pool  */
  AP_UINT16      poll_wait;         /* secs between polling HMOD for errors  */
  AP_UINT16      hmod_data_len;     /* Count of valid bytes in hmod_data     */
  unsigned char  hmod_data[80];     /* ASCII string of dial digits           */
  char           x21_sequence[255]; /* X21 selection sequence                */
  unsigned char  x21_retry_count;   /* X21 max call retries                  */
  AP_UINT16      x21_retry_delay;   /* X21 delay between retries             */
  AP_UINT16      v25_tx_delay;      /* V25 pause before dialling             */
  unsigned char  cdstl;             /* Wait for RI before DTR                */
  unsigned char  hex_hmod_data;     /* Flag that the hmod_data was           */
                                    /* originally specified in hex by the    */
                                    /* user                                  */
  unsigned char  reserve1[2];       /* reserved                              */
} STUB_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   SDL_PORT_SPEC_DATA                                           */
/*                                                                           */
/* Description: Port-specific data for SDLC - used on DEFINE_PORT            */
/*****************************************************************************/

typedef struct sdl_port_spec_data
{
  V0_MUX_INFO    mux_info;           /* streams information                  */
  AP_UINT32      idle_timer;         /* idle timer (in ms)                   */
  AP_UINT16      idle_timer_retry;   /* idle timer retry                     */
  AP_UINT16      reserve1;           /* reserved field                       */
  AP_UINT32      np_rcv_timer;       /* non-productive receive timer( in ms) */
  AP_UINT16      np_rcv_timer_retry; /* non-productive receive timer retry   */
  unsigned char  np_rcv_timer_fail;  /* NPR timer fail secondary link flag   */
  unsigned char  reserve2;
  AP_UINT32      write_timer;        /* write timer (in ms)                  */
  AP_UINT16      write_timer_retry;  /* write timer retry                    */
  AP_UINT16      reserve3;           /* reserved field                       */
  AP_UINT32      link_conn_timer;    /* link connection timer (in ms)        */
  AP_UINT16      link_conn_timer_retry; /* link connection timer retry       */
  AP_UINT16      reserve4;           /* reserved field                       */
  AP_UINT16      pri_fdplx;          /* Is primary on this link full-duplex  */
  AP_UINT16      sec_fdplx;          /* Is secondary on link full-duplex     */
  AP_UINT16      use_rej;            /* Can REJ command be used on this port */
  AP_UINT16      port_type;          /* Port type                            */
  AP_UINT16      max_xid_size;       /* max size of XIDs in MU_SEND_XID      */
  AP_UINT16      max_retry_count;    /* max number of times to retransmit    */
  AP_UINT16      physical_link;      /* physical link type (for GDLC)        */
  AP_UINT16      stub_spec_data_len; /* length of next field                 */
  STUB_SPEC_DATA stub_spec_data;     /* data specific to HMOD stub           */
} SDL_PORT_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   SDL_LINK_SPEC_DATA                                           */
/*                                                                           */
/* Description: Link-specific data for SDLC - used on DEFINE_PORT and        */
/*                                            DEFINE_LS                      */
/*****************************************************************************/

typedef struct sdl_link_spec_data
{
  V0_MUX_INFO    mux_info;              /* streams information               */
  AP_UINT16      reserve8;              /* reserved field                    */
  AP_UINT16      reserve9;              /* reserved field                    */
  AP_UINT32      contact_timer;         /* contact timer (fast poll, in ms)  */
  AP_UINT16      contact_timer_retry;   /* contact timer retry               */
  AP_UINT16      reserve1;              /* reserved field                    */
  AP_UINT32      contact_timer2;        /* contact timer (slow poll, in ms)  */
  AP_UINT16      contact_timer_retry2;  /* contact timer 2 retry             */
  AP_UINT16      reserve2;              /* reserved field                    */
  AP_UINT32      disc_timer;            /* disconnect timer (in ms)          */
  AP_UINT16      disc_timer_retry;      /* disconnect timer retry            */
  AP_UINT16      reserve3;              /* reserved field                    */
  AP_UINT32      nve_poll_timer;        /* negative poll timer (fast poll)   */
  AP_UINT16      nve_poll_timer_retry;  /* negative poll timer retry         */
  AP_UINT16      reserve4;              /* reserved field                    */
  AP_UINT32      nve_poll_timer2;       /* negative poll timer (slow poll)   */
  AP_UINT16      nve_poll_timer_retry2; /* negative poll timer 2 retry       */
  AP_UINT16      reserve5;              /* reserved field                    */
  AP_UINT32      no_resp_timer;         /* No response timer (T1 timer) in ms*/
  AP_UINT16      no_resp_timer_retry;   /* No response timer retry           */
  AP_UINT16      reserve6;              /* reserved field                    */
  AP_UINT32      rem_busy_timer;        /* Remote busy timer (in ms)         */
  AP_UINT16      rem_busy_timer_retry;  /* Remote busy timer retry           */
  unsigned char  re_tx_threshold;       /* I-frame retransmission threshold  */
  unsigned char  repoll_threshold;      /* Poll retransmission threshold     */
  AP_UINT32      rr_timer;              /* RR turnaround timer (in ms)       */
  unsigned char  group_address;         /* reserved field                    */
  unsigned char  poll_frame;            /* Poll frame to use when Primary    */
                                        /*   and contact polling secondary   */
                                        /*   XID, DISC, SNRM, SNRME, TESTT   */
  AP_UINT16      poll_on_iframe;        /* Can LS send poll bit on I-frame   */
  AP_UINT16      stub_spec_data_len;    /* length of HMOD specific data      */
  STUB_SPEC_DATA stub_spec_data;        /* HMOD specific data                */
} SDL_LINK_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SDL_PORT_OPER_TABLE                                            */
/*                                                                           */
/* Description: SDLC Port operational statistics.                            */
/*****************************************************************************/
typedef struct sdl_port_oper_table
{
  V0_MUX_INFO    mux_info;          /* streams information                   */
  AP_UINT32      index;             /* index of port                         */
  unsigned char  name[8];           /* reserved                              */
  unsigned char  role;              /* current role of link station(s)       */
                                    /*   using port                          */
  unsigned char  type;              /* line type - leased or switched        */
  unsigned char  topology;          /* can port be point-to-point            */
                                    /*   or multipoint                       */
  unsigned char  reserve;           /* reserved                              */
  AP_UINT32      activto;           /* how long switched line can be         */
                                    /*  inactive before port disconects      */
  AP_UINT32      pause;             /* time between poll cycles              */
  unsigned char  slow_poll_method;  /* slow poll method                      */
  unsigned char  reserv2[3];        /* reserved                              */
  AP_UINT32      slow_poll_timer;   /* slow poll timer                       */
  unsigned char  last_fail_cause;   /* cause of last failure of this port    */

} SDL_PORT_OPER_TABLE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SDL_PORT_STATS_TABLE                                           */
/*                                                                           */
/* Description: SDLC Port main statistics                                    */
/*****************************************************************************/
typedef struct sdl_port_stats_table
{
  V0_MUX_INFO    mux_info;          /* streams information                   */
  AP_UINT32      index;             /* index of port                         */
  AP_UINT32      dwarf_frames;      /* frames rcvd too AP_INT16 to be valid     */
  AP_UINT32      polls_out;         /* polls sent to adjacent link stations  */
  AP_UINT32      poll_rsps_out;     /* polls responded to by adjacent link   */
                                    /*   stations                            */
  AP_UINT32      local_busies;      /* number of times local link stations   */
                                    /*   have entered busy state (RNR)       */
  AP_UINT32      remote_busies;     /* number of times remote link stations  */
                                    /*   have entered busy state (RNR)       */
  AP_UINT32      iframes_in;        /* I-frames rcvd from adjacent link      */
                                    /*   stations                            */
  AP_UINT32      iframes_out;       /* I-frames sent to adjacent link        */
                                    /*   stations                            */
  AP_UINT32      octets_in;         /* bytes in I-Frames received            */
  AP_UINT32      octets_out;        /* bytes in I-Frames sent                */
  AP_UINT32      protocol_errs;     /* number of times that links have been  */
                                    /*   taken down due to protocol errors   */
  AP_UINT32      activity_to_s;     /* number of times that links have been  */
                                    /*   taken down as a result of no        */
                                    /*   activity on the link                */
  AP_UINT32      rnrlimit_s;        /* number of times that links have been  */
                                    /*   taken down due to RNR timer         */
                                    /*   expiring                            */
  AP_UINT32      retries_exps;      /* number of times that links have been  */
                                    /*   taken down due to a retry limit     */
                                    /*   being exhausted                     */
  AP_UINT32      retransmits_in;    /* rcvd I-frames retransmitted           */
  AP_UINT32      retransmits_out;   /* sent I-frames retransmitted           */
} SDL_PORT_STATS_TABLE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SDL_LS_OPER_TABLE                                              */
/*                                                                           */
/* Description: SDLC Link Station operational statistics.                    */
/*****************************************************************************/
typedef struct sdl_ls_oper_table
{
  V0_MUX_INFO    mux_info;           /* streams information                  */
  AP_UINT32      index;              /* index of port that owns LS           */
  unsigned char  address;            /* poll address of secondary LS         */
  unsigned char  reserve;            /* reserved                             */
  AP_UINT16      role;               /* current role of link station         */
  unsigned char  name[8];            /* reserved                             */
  AP_UINT16      state;              /* operational state of LS              */
  AP_UINT16      maxdata;            /* current max PDU size for logical link*/
  AP_UINT32      replyto;            /* current reply timeout                */
  AP_UINT32      maxin;              /* current max unack'd frames that LS   */
                                     /*   can receive                        */
  AP_UINT32      maxout;             /* current max unack'd frames that LS   */
                                     /*   can send                           */
  unsigned char  modulo;             /* sequence number modulus              */
  unsigned char  reserv2[3];         /* reserved                             */
  AP_UINT32      retries_m;          /* number of retries in retry sequence  */
  AP_UINT32      retries_t;          /* interval between retries             */
  AP_UINT32      retries_n;          /* retry limit                          */
  AP_UINT32      rnrlimit;           /* how AP_INT32 adjacent LS can be in RNR   */
                                     /*   state before lnk is dropped        */
  unsigned char  datmode;            /* data mode - half or full duplex      */
  unsigned char  last_fail_cause;    /* cause of last failure                */
  unsigned char  last_fail_ctrl_in[2];   /* control field of last frame      */
                                         /*   received before failure        */
  unsigned char  last_fail_ctrl_out[2];  /* control field of last frame      */
                                         /*   sent before failure            */
  unsigned char  last_fail_frmr_info[5]; /* info field of FRMR frame if last */
                                         /*   failure was caused by invalid  */
                                         /*   frame                          */
  unsigned char  sdoppad1;               /* reserved                         */
  AP_UINT32      last_fail_replyto_s;    /* number of REPLYTO timeouts at    */
                                         /*   time of last failure           */
  unsigned char  g_poll;                 /* group poll address               */
  unsigned char  sim_rim;                /* are SIM / RIM supported          */
  unsigned char  xmit_rcv_cap;           /* transmit-receive capabilities    */
                                         /*   of remote LS                   */

} SDL_LS_OPER_TABLE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SDL_LS_STATS_TABLE                                             */
/*                                                                           */
/* Description: SDLC Link Station main statistics                            */
/*****************************************************************************/

typedef struct sdl_ls_stats_table
{
  V0_MUX_INFO    mux_info;          /* streams information                   */
  AP_UINT32      index;             /* index of port that owns LS            */
  unsigned char  address;           /* poll address of secondary LS          */
  unsigned char  reserv[3];         /* reserved                              */
  AP_UINT32      blus_in;           /* frames received from adjacent LS      */
  AP_UINT32      blus_out;          /* frames sent to adjacent LS            */
  AP_UINT32      octets_in;         /* bytes received from adjacent LS       */
  AP_UINT32      octets_out;        /* bytes sent to adjacent LS             */
  AP_UINT32      polls_out;         /* polls sent to adjacent LS             */
  AP_UINT32      poll_rsps_out;     /* polls responded to by adjacent LS     */
  AP_UINT32      local_busies;      /* number of times local LS has          */
                                    /*  entered busy state (RNR)             */
  AP_UINT32      remote_busies;     /* number of times remote LS has         */
                                    /*  entered busy state (RNR)             */
  AP_UINT32      iframes_in;        /* I-frames rcvd from adjacent LS        */
  AP_UINT32      iframes_out;       /* I-frames sent to adjacent LS          */
  AP_UINT32      retransmits_in;    /* rcvd I-frames retransmitted since LS  */
                                    /*   was started                         */
  AP_UINT32      retransmits_out;   /* sent I-frames retransmitted since LS  */
                                    /*   was started                         */
  AP_UINT32      ioctets_in;        /* bytes in I-frames received            */
  AP_UINT32      ioctets_out;       /* bytes in I-frames sent                */
  AP_UINT32      uiframes_in;       /* reserved                              */
  AP_UINT32      uiframes_out;      /* reserved                              */
  AP_UINT32      xids_in;           /* XIDs rcvd from adjacent LS            */
  AP_UINT32      xids_out;          /* XIDs sent to adjacent LS              */
  AP_UINT32      tests_in;          /* TEST frames received                  */
  AP_UINT32      tests_out;         /* TEST frames sent                      */
  AP_UINT32      rejs_in;           /* REJ frames received                   */
  AP_UINT32      rejs_out;          /* REJ frames sent                       */
  AP_UINT32      frmrs_in;          /* FRMR frames received                  */
  AP_UINT32      frmrs_out;         /* FRMR frames sent                      */
  AP_UINT32      sims_in;           /* SIM frames received                   */
  AP_UINT32      sims_out;          /* SIM frames sent                       */
  AP_UINT32      rims_in;           /* RIM frames received                   */
  AP_UINT32      rims_out;          /* RIM frames sent                       */
  AP_UINT32      disc_in;           /* DISC/RD frames received               */
  AP_UINT32      disc_out;          /* DISC/RD frames sent                   */
  AP_UINT32      ua_in;             /* UA frames received                    */
  AP_UINT32      ua_out;            /* UA frames sent                        */
  AP_UINT32      dm_in;             /* DM frames received                    */
  AP_UINT32      dm_out;            /* DM frames sent                        */
  AP_UINT32      snrm_in;           /* SNRM frames received                  */
  AP_UINT32      snrm_out;          /* SNRM frames sent                      */

} SDL_LS_STATS_TABLE;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* DLC-specific data structures for DLPI links                               */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure: vdl_dlc_cfg                                                    */
/*                                                                           */
/* Description: Configuration data for a DLPI User DLC Process               */
/*****************************************************************************/

typedef struct vdl_dlc_cfg
{
  V0_MUX_INFO    mux_info;           /* streams information                  */
} VDL_DLC_CFG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: vdl_sap_cfg                                                    */
/*                                                                           */
/* Description: Configuration data for a DLPI User SAP                       */
/*****************************************************************************/

typedef struct vdl_sap_cfg
{
  V0_MUX_INFO    mux_info;           /* streams information                  */
#ifdef ATTGIS
  unsigned short dlci_number;        /* DLCI number (frame relay only)       */
#endif
} VDL_SAP_CFG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: vdl_ls_cfg                                                     */
/*                                                                           */
/* Description: Configuration data for a DLPI User Link Station              */
/*****************************************************************************/

typedef struct vdl_ls_cfg
{
  V0_MUX_INFO    mux_info;                  /* streams information           */
  AP_UINT16      reserve1;                  /* reserved field                */
  AP_UINT16      reserve2;                  /* reserved field                */
  AP_UINT16      test_timeout;              /* TEST timeout value in seconds */
  AP_UINT16      test_retry_limit;          /* TEST retransmission limit     */
  AP_UINT16      xid_timeout;               /* XID timeout value in seconds  */
  AP_UINT16      xid_retry_limit;           /* XID retransmission limit      */
  AP_UINT16      t1_timeout;                /* T1 timeout value in seconds   */
  AP_UINT16      t1_retry_limit;            /* I-frame retransmission limit  */
} VDL_LS_CFG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: vdl_ls_statistics                                              */
/*                                                                           */
/* Description: Statistics structure for DLPI LAN-based DLC Link Stns        */
/*****************************************************************************/

typedef struct vdl_ls_statistics
{
  V0_MUX_INFO    mux_info;           /* streams information                  */
  AP_UINT32      ls_st_mus_sent;     /* Frames sent from this Link Station   */
  AP_UINT32      ls_st_mus_received; /* Frames received at this Link Station */
  AP_UINT32      ls_st_bytes_sent;   /* Bytes sent from this Link Station    */
  AP_UINT32      ls_st_bytes_received; /* Bytes received at this Link Station*/
} VDL_LS_STATISTICS;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* DLC-specific data structures for LLC2.                                    */
/* These are used when the LLC2 support is provided by SNAP-LLC2, which is   */
/* delivered as part of SNAP-IX, as opposed to a DLPI driver.  For a         */
/* particular operating system, only one of these is used.                   */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure: VMC_DLC_CFG                                                    */
/*                                                                           */
/* Description: DLC-specific data appended to DEFINE_DLC                     */
/*****************************************************************************/
typedef struct vmc_dlc_cfg
{
  V0_MUX_INFO    mux_info;          /* streams information                   */
  AP_UINT16      lan_type;          /* type of LAN                           */
  AP_UINT16      min_rcv_dsf;       /* reserved                              */
} VMC_DLC_CFG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: VMC_SAP_CFG                                                    */
/*                                                                           */
/* Description: Adapter-specific data appended to DEFINE_PORT                */
/*****************************************************************************/
typedef struct vmc_sap_cfg
{
  V0_MUX_INFO    mux_info;          /* streams information                   */
  AP_UINT16      lan_type;          /* type of LAN                           */
  AP_UINT16      min_rcv_dsf;       /* reserved                              */
} VMC_SAP_CFG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LLC_SAP_SPEC_DATA                                              */
/*                                                                           */
/* Description: Port/Link-specific data appended to DEFINE_PORT + DEFINE_LS  */
/*****************************************************************************/
typedef struct llc_sap_spec_data
{
  AP_UINT16      ack_timeout;             /* acknowledgement timeout in ms   */
  AP_UINT16      p_bit_timeout;           /* Poll response timeout in ms     */
  AP_UINT16      t2_timeout;              /* acknowledgement delay in ms     */
  AP_UINT16      rej_timeout;             /* REJ response timeout in seconds */
  AP_UINT16      busy_state_timeout;      /* remote busy timeout in seconds  */
  AP_UINT16      idle_timeout;            /* idle RR interval in seconds     */
  AP_UINT16      max_retry;               /* retry limit for any response    */
  AP_UINT16      upward_cred_q_threshold; /* upward credit threshold         */
  AP_UINT16      window_inc_threshold;    /* window increment threshold      */
  AP_UINT16      pad;                     /* reserved                        */
} LLC_SAP_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PROM_MODE_DATA                                                 */
/*                                                                           */
/* Description: Data which control the setting of promiscous mode            */
/*              - appended to DEFINE_PORT                                    */
/*****************************************************************************/
typedef struct prom_mode_data
{
  AP_UINT16      port_spec_data_size;     /* reserved                        */
  unsigned char  promiscuous;             /* promiscuous mode supported      */
                                          /*   - set to NO                   */
  unsigned char  dlsw_flag;               /* reserved                        */
  AP_UINT16      vrn;                     /* reserved                        */
  AP_UINT16      bridge_num;              /* reserved                        */

} PROM_MODE_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LLC_PORT_SPEC_DATA                                             */
/*                                                                           */
/* Description: PORT specific data - appended to DEFINE_PORT                 */
/*****************************************************************************/
typedef struct llc_port_spec_data
{
  V0_MUX_INFO    mux_info;                /* streams information             */
  PROM_MODE_DATA prom_data;               /* promiscuous mode data           */
  LLC_SAP_SPEC_DATA sap_spec_data;        /* SAP specific data               */
  unsigned char  adapter_id[8];           /* adapter identifier              */
#define NOF_API_MAX(X,Y)   (((X)>(Y)) ? (X) : (Y))
#define SIZEOF_ADAPT_SPEC_DATA         \
                       (NOF_API_MAX(sizeof(VMC_DLC_CFG), sizeof(VMC_SAP_CFG)))
  unsigned char  adapt_spec_data[SIZEOF_ADAPT_SPEC_DATA]; /* adapter specific*/
                                                          /* data            */
  AP_UINT32      max_rcv_pool_kb;         /* maximum size of receive buf pool*/
  unsigned char  throttle_back_pcent;     /* throttle back threshold         */
  unsigned char  pad[3];                  /* pad to round up to dword        */

} LLC_PORT_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   LLC_LINK_SPEC_DATA                                           */
/*                                                                           */
/* Description: Link specific data - appended to DEFINE_LS                   */
/*****************************************************************************/
typedef struct llc_link_spec_data
{
  V0_MUX_INFO    mux_info;                /* streams information             */
  AP_UINT16      reserve1;                /* reserved field                  */
  AP_UINT16      reserve2;                /* reserved field                  */
  AP_UINT16      length;                  /* length of this structure        */
  AP_UINT16      xid_timer;               /* XID timeout in seconds          */
  AP_UINT16      xid_timer_retry;         /* XID retry limit                 */
  AP_UINT16      test_timer;              /* TEST timeout in seconds         */
  AP_UINT16      test_timer_retry;        /* TEST retry limit                */
  AP_UINT16      ack_timeout;             /* acknowledgement timeout in ms   */
  AP_UINT16      p_bit_timeout;           /* Poll response timeout in ms     */
  AP_UINT16      t2_timeout;              /* acknowledgement delay in ms     */
  AP_UINT16      rej_timeout;             /* REJ response timeout in seconds */
  AP_UINT16      busy_state_timeout;      /* remote busy timeout in seconds  */
  AP_UINT16      idle_timeout;            /* idle RR interval in seconds     */
  AP_UINT16      max_retry;               /* retry limit for any response    */

} LLC_LINK_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LLC2_DEV_STATS                                                 */
/*                                                                           */
/* Description: LLC2 adapter statistics - part of the port statistics        */
/*****************************************************************************/

typedef struct llc2_dev_stats
{
  unsigned char  adapter_number;          /* reserved                        */
  unsigned char  res1;                    /* reserved                        */
  unsigned char  line_error;              /* line error count                */
  unsigned char  internal_error;          /* internal error count            */
  unsigned char  burst_error;             /* burst error count               */
  unsigned char  ari_fci_error;           /* ARI/FCI error count             */
  unsigned char  end_delim;               /* end delimiter                   */
  unsigned char  res2;                    /* reserved                        */
  unsigned char  lost_frame;              /* lost frame count                */
  unsigned char  rcv_cngstn;              /* receive congestion count        */
  unsigned char  frm_cpy_err;             /* Frame Copied error count        */
  unsigned char  freq_err;                /* frequency error count           */
  unsigned char  token_err;               /* token error count               */
  unsigned char  crc_err;                 /* CRC error count                 */
  unsigned char  res3;                    /* reserved                        */
  unsigned char  xmit_err;                /* transmit error count            */
  unsigned char  res4;                    /* reserved                        */
  unsigned char  collision_err;           /* collision error counts          */
  unsigned char  res5[7];                 /* reserved                        */

} LLC2_DEV_STATS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LLC2_PORT_STATS                                                */
/*                                                                           */
/* Description: LLC2 port statistics                                         */
/*****************************************************************************/

typedef struct llc2_port_stats
{
  V0_MUX_INFO    mux_info;                /* streams information             */
  AP_UINT32      time_secs;               /* system time when port activated */
  AP_UINT16      time_ms;                 /* system time when port activated */
  unsigned char  mac_addr[6];             /* MAC address of port             */
  AP_UINT16      ack_timeout;             /* reserved                        */
  AP_UINT16      p_bit_timeout;           /* reserved                        */
  AP_UINT16      t2_timeout;              /* reserved                        */
  AP_UINT16      rej_timeout;             /* reserved                        */
  AP_UINT16      busy_state_timeout;      /* reserved                        */
  AP_UINT16      idle_timeout;            /* reserved                        */
  AP_UINT16      max_retry;               /* reserved                        */
  AP_UINT16      max_btu_size;            /* max BTU size for port           */
  AP_UINT16      ls_count;                /* count of link stations          */
                                          /*   currently using this port     */
  AP_UINT16      reserve1;                /* reserved                        */
  AP_UINT32      ui_frames_sent;          /* count of UI frames sent         */
  AP_UINT32      ui_frames_rcvd;          /* count of UI frames received     */
  LLC2_DEV_STATS device_stats;            /* adapter statistics              */

} LLC2_PORT_STATS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LLC2_LS_STATS                                                  */
/*                                                                           */
/* Description: LLC2 link station statistics                                 */
/*****************************************************************************/

typedef struct llc2_ls_stats
{
  V0_MUX_INFO    mux_info;                /* streams information             */
  unsigned char  local_mac[6];            /* MAC address of local port       */
  unsigned char  local_sap;               /* SAP address of local port       */
  unsigned char  reserve1;                /* reserved                        */
  unsigned char  remote_mac[6];           /* MAC address of remote port      */
  unsigned char  remote_sap;              /* SAP address of remote port      */
  unsigned char  reserve2;                /* reserved                        */
  AP_UINT16      rif_len;                 /* length of routing information   */
  AP_UINT16      rif[8];                  /* routing information field       */
                                          /*   -  Token Ring only            */
  unsigned char  ls_fsm;                  /* LLC2 FSM state                  */
  unsigned char  reserve3;                /* reserved                        */
  AP_UINT16      mac_type;                /* MAC type                        */
  AP_UINT16      max_btu_size;            /* maximum BTU size                */
  AP_UINT16      send_window;             /* send window                     */
  AP_UINT16      receive_window;          /* receive window                  */
  AP_UINT32      t1_expiry_count;         /* T1 expiry count                 */
  AP_UINT32      t2_expiry_count;         /* T2 expiry count                 */
  AP_UINT32      remote_busy;             /* remote busy state count         */
  AP_UINT32      local_busy;              /* local busy state count          */
  AP_UINT32      i_frames_sent;           /* I frames sent                   */
  AP_UINT32      i_bytes_sent;            /* bytes in I frames sent          */
  AP_UINT32      i_frames_rcvd;           /* I frames received               */
  AP_UINT32      i_bytes_rcvd;            /* bytes in I frames received      */
  AP_UINT32      i_frames_rjctd;          /* I frames rejected               */
  AP_UINT32      i_bytes_rjctd;           /* bytes in I frames rejected      */
  AP_UINT32      i_frames_rexmit;         /* I frames retransmitted          */
  AP_UINT32      i_bytes_rexmit;          /* bytes in I frames retransmitted */
  AP_UINT32      rej_frames_sent;         /* REJ frames sent                 */
  AP_UINT32      rej_frames_rcvd;         /* REJ frames received             */
  AP_UINT32      xid_frames_sent;         /* XID frames sent                 */
  AP_UINT32      xid_frames_rcvd;         /* XID frames received             */
  AP_UINT16      ack_timeout;             /* acknowledgement timeout         */
  AP_UINT16      p_bit_timeout;           /* Poll response timeout           */
  AP_UINT16      t2_timeout;              /* T2 timeout                      */
  AP_UINT16      rej_timeout;             /* REJ timeout                     */
  AP_UINT16      busy_state_timeout;      /* busy state timeout              */
  AP_UINT16      idle_timeout;            /* idle timeout                    */
  AP_UINT16      max_retry;               /* max retry count                 */

} LLC2_LS_STATS;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* DLC-specific data structures for QLLC                                     */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure: vql_dlc_spec_data                                              */
/*                                                                           */
/* Description: Configuration data for a QLLC DLC Process                    */
/*****************************************************************************/

typedef struct vql_dlc_spec_data
{
  V0_MUX_INFO   mux_info;       /* streams config info                       */
} VQL_DLC_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: vql_port_spec_data                                             */
/*                                                                           */
/* Description: Configuration data for a QLLC Port                           */
/*****************************************************************************/
typedef struct vql_port_spec_data
{
  V0_MUX_INFO   mux_info;       /* streams config info                       */
  unsigned char driver_name[13];  /* driver name                             */
  unsigned char cud_mode;       /* type of matching required on CUD          */
                                /* X25_DONTCARE no CUD                       */
                                /* X25_IDENTITY match if all CUD match this  */
                                /* X25_STARTSWITH match if CUD starts with   */
  AP_UINT16      cud_len;       /* length of Call User Data in octets        */
  unsigned char cud[128];       /* Call User Data                            */
  unsigned char add_mode;       /* type of matching required on called addres*/
                                /* X25_DONTCARE no addr                      */
                                /* X25_IDENTITY match if address matches this*/
                                /* X25_STARTSWITH match if address startswith*/
  AP_UINT16      add_len;       /* len of called address to match local X.25 */
                                /* in main message if X25_STARTSWITH         */
  AP_UINT32      xtras;         /* flags field indicating mux or extra data  */
  AP_UINT32      xtra_len;      /* length of extra spec data following       */

} VQL_PORT_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: vql_ls_spec_data                                               */
/*                                                                           */
/* Description: Configuration data for a QLLC LS                             */
/*****************************************************************************/
typedef struct vql_ls_spec_data
{
  V0_MUX_INFO     mux_info;         /* streams config info                   */
  AP_UINT16       reserve1;         /* reserved field                        */
  AP_UINT16       reserve2;         /* reserved field                        */
  unsigned char   vc_type;          /* Virtual Circuit type                  */
  unsigned char   req_rev_charge;   /* request reverse charge if non-zero    */
  unsigned char   loc_packet;       /* loc->rem packet size 2**locpacket     */
  unsigned char   rem_packet;       /* rem->loc packet size 2**rempacket     */
  unsigned char   loc_wsize;        /* loc->rem window size                  */
  unsigned char   rem_wsize;        /* rem->loc window size                  */
  AP_UINT16       fac_len;          /* X.25 facilities length                */
  unsigned char   fac[128];         /* X.25 facilities                       */
  AP_UINT16       retry_limit;      /* times to retry sending QXID,QSM,QDISC */
  AP_UINT16       retry_timeout;    /* timeout for each of the above tries   */
  AP_UINT16       idle_timeout;     /* timeout for no Q messages during init */

  AP_UINT16       pvc_id;           /* PVC logical channel identifier        */
  AP_UINT16       sn_id_len;        /* Length of the subnet identifier       */
  unsigned char   sn_id[4];         /* Subnet identifier                     */
  AP_UINT16       cud_len;          /* length of any call user data to send  */
  unsigned char   cud[128];         /* actual call user data                 */
  AP_UINT32       xtras;            /* flags field indicating mux/extra data */
  AP_UINT32       xtra_len;         /* length of extra spec data following   */

  unsigned char   rx_thruput_class; /* Max Rx speed of calling DTE           */
  unsigned char   tx_thruput_class; /* Max Tx speed of calling DTE.  Values  */
                                    /* for these fields are:                 */
                                    /*   0     - Default                     */
                                    /*   0x07  - 1200                        */
                                    /*   0x08  - 2400                        */
                                    /*   0x09  - 4800                        */
                                    /*   0x0a  - 9600                        */
                                    /*   0x0b  - 19200                       */
                                    /*   0x0c  - 48000                       */
  unsigned char   cugo;             /* Closed User Group (outgoing)          */
  unsigned char   cug;              /* LS used by Closed User Group          */
  AP_UINT16       cug_index;        /* Index of CUG using this LS            */
  AP_UINT16       nuid_length;      /* Length of Network User ID             */
  unsigned char   nuid_data[109];   /* Network User ID                       */
  unsigned char   reserve3[2];      /* Reserved field                        */
  unsigned char   rpoa_count;       /* Count of RPOA codes                   */
  AP_UINT16       rpoa_ids[30];     /* Array of RPOA codes                   */

} VQL_LS_SPEC_DATA;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* DLC-specific configuration data for AIX GDLC support                      */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   gdlc_dlc_cfg                                                 */
/*                                                                           */
/* Description: Configuration data for a GDLC DLC                            */
/*****************************************************************************/

typedef struct gdlc_dlc_cfg
{
  V0_MUX_INFO      mux_info;           /* Streams config info                */
  AP_UINT16        max_saps;           /* Maximum number of SAPS supported   */
                                       /* by this DLC                        */
  union
  {
    struct
    {
      AP_UINT16      support_level;    /* CCITT level (1980/1984) of adapter */
#define GDLC_X25_1980       1980
#define GDLC_X25_1984       1984
#define GDLC_X25_1988       1988
    } x25;
    struct
    {
      char           ethernet_type;    /* Standard or IEEE 802.3             */
#define GDLC_ETHER_STANDARD     0
#define GDLC_ETHER_8023         1
    } ether;
  } uu;
}
GDLC_DLC_CFG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   gdlc_sap_cfg                                                 */
/*                                                                           */
/* Description: Configuration data for a GDLC SAP                            */
/*****************************************************************************/

typedef struct gdlc_sap_cfg
{
  V0_MUX_INFO      mux_info;           /* Streams config info                */
  AP_UINT16        rem_wsize;          /* Remote window size                 */
  char             local_name[8];      /* Unique name of SAP on network      */
}
GDLC_SAP_CFG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   gdlc_ls_cfg                                                  */
/*                                                                           */
/* Description: Configuration data for a GDLC LS                             */
/*****************************************************************************/

typedef struct gdlc_ls_cfg
{
  V0_MUX_INFO    mux_info;             /* Streams config info                */
  AP_UINT16      reserve1;             /* reserved field                     */
  AP_UINT16      reserve2;             /* reserved field                     */
  AP_UINT16      loc_wsize;            /* Local window size                  */
  AP_UINT16      rem_wsize;            /* Remote window size                 */
  AP_UINT16      xid_timeout;          /* XID timeout value                  */
  AP_UINT16      xid_retry_limit;      /* XID retransmission limit           */
  AP_UINT16      t1_timeout;           /* T1 timeout value                   */
  AP_UINT16      t1_retry_limit;       /* I-frame retransmission limit       */
  AP_UINT16      ack_time;             /* Acknowledgement timeout value      */
  AP_UINT16      inact_time;           /* Inactivity timeout value           */
  AP_UINT16      force_time;           /* Force halt time-out value          */
  union
  {
    struct
    {
      AP_UINT16      pkt_prty;         /* Ring access priority for transmit  */
                                       /* packets                            */
      AP_UINT16      dyna_wnd;         /* Dynamic window increment for       */
                                       /* transmit window                    */
    } tr;                              /* TOKEN RING & FDDI                  */
  } uu;
}
GDLC_LS_CFG;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   gdlc_ls_stats                                                */
/*                                                                           */
/* Description: Statistics structure for a GDLC LS                           */
/*****************************************************************************/

typedef struct gdlc_ls_stats
{
  V0_MUX_INFO    mux_info;             /* Streams config info                */
  AP_UINT32      ls_st_mus_sent;       /* Frames sent from this Link Stn     */
  AP_UINT32      ls_st_mus_received;   /* Frames received at this Link Stn   */
  AP_UINT32      ls_st_bytes_sent;     /* Bytes sent from this Link Stn      */
  AP_UINT32      ls_st_bytes_received; /* Bytes received at this Link Stn    */
}
GDLC_LS_STATS;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* DLC-specific configuration data for Channel support                       */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   chnl_dlc_spec_data                                           */
/*                                                                           */
/* Description: Configuration data for a Channel DLC                         */
/*****************************************************************************/

typedef struct chnl_dlc_spec_data
{
  V0_MUX_INFO      mux_info;           /* Streams config info                */
  AP_UINT16        mu_credit;          /* Amount of credit to allow PC to    */
                                       /* send                               */
  unsigned char    stats_support;      /* Activate statistics gathering?     */
  unsigned char    reserv1[31];        /* Pad and future expansion           */
}
CHNL_DLC_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   chnl_port_spec_data                                          */
/*                                                                           */
/* Description: Configuration data for a Channel SAP                         */
/*****************************************************************************/

typedef struct chnl_port_spec_data
{
  V0_MUX_INFO      mux_info;           /* Streams config info                */
  AP_UINT16        tx_buffers;         /* Transmit buffers to use (def 64)   */
  AP_UINT16        rx_buffers;         /* Receive buffers to use (def 64)    */
  AP_UINT32        speed;              /* Channel streaming speed            */
  unsigned char    reserv1[32];        /* Pad and future expansion           */
}
CHNL_PORT_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   chnl_link_spec_data                                          */
/*                                                                           */
/* Description: Configuration data for a Channel LS                          */
/*****************************************************************************/

typedef struct chnl_link_spec_data
{
  V0_MUX_INFO    mux_info;             /* Streams config info                */
  AP_UINT16      device_end;           /* BlkMux protocol flag               */
  unsigned char  escd_port;            /* ESCON director port (if used)      */
  unsigned char  cuadd;                /* IOCP CNTL UNIT CUADD field (or 0)  */
  unsigned char  local_name[8];        /* Local name (LS name is default)    */
  unsigned char  remote_name[8];       /* Host name (from VTAM PU defn)      */
  unsigned char  reserv1[32];          /* Pad and future expansion           */
}
CHNL_LINK_SPEC_DATA;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* DLC-specific data structures for HPR/IP                                   */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   IPDLC_DLC_SPEC_DATA                                          */
/*                                                                           */
/* Description: DLC-specific data for IPDLC                                  */
/*****************************************************************************/
typedef struct ipdlc_dlc_spec_data
{
  V0_MUX_INFO   mux_info;       /* streams config info                       */

  AP_UINT16 udp_port[5];        /* UDP port numbers for traffic priorities   */
                                /* LLC,Network,High,Medium,Low               */

  unsigned char ip_precedence[5];       /* IP precedence (0-7) for traffic   */
                                /* priotities LLC,Network,High,Medium,Low    */
  unsigned char no_dns_lookup;  /* Set to AP_YES if all remote host          */
                                /* addresses are specified as IPv4 dotted    */
                                /* decimal addresses for this port and no    */
                                /* DNS lookup is to be performed to get the  */
                                /* remote hostname for incoming calls.       */

} IPDLC_DLC_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: IPDLC_PORT_SPEC_DATA                                           */
/*                                                                           */
/* Description: Port-specific data for IPDLC                                 */
/*****************************************************************************/

typedef struct ipdlc_port_spec_data
{
  V0_MUX_INFO  mux_info;        /* streams config info                       */
  char         if_name[46];     /* Local interface (eth0 etc) or IP addr     */
} IPDLC_PORT_SPEC_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   IPDLC_LINK_SPEC_DATA                                         */
/*                                                                           */
/* Description: Link-specific data for IPDLC                                 */
/*****************************************************************************/

typedef struct ipdlc_link_spec_data
{
  V0_MUX_INFO   mux_info;       /* streams config info                       */
  AP_UINT16 length;             /* Length of this structure                  */
  AP_UINT16 ack_timeout;         /* ACK timer for command frames             */
  AP_UINT16 max_retry;          /* Retry limit for command frames            */
  AP_UINT16 liveness_timeout;   /* Liveness timer                            */
  unsigned char short_hold_mode;      /* Run in short-hold mode              */
  unsigned char remote_hostname[255]; /* Name of remote host to contact      */

} IPDLC_LINK_SPEC_DATA;

/**STRUCT+********************************************************************/
/* Structure: UDP_LS_STATS_TABLE                                             */
/*                                                                           */
/* Description: LS statistics table returned on QUERY_LS_STATISTICS.         */
/*****************************************************************************/

typedef struct udp_ls_stats_table
{
   V0_MUX_INFO      mux_info;           /* streams config info               */
   AP_UINT32        udp_low_out;        /* Count of UDP datagrams sent       */
                                        /* containing low priority data      */
   AP_UINT32        udp_med_out;        /* Count of UDP datagrams sent       */
                                        /* containing medium priority data   */
   AP_UINT32        udp_high_out;       /* Count of UDP datagrams sent       */
                                        /* containing high priority data     */
   AP_UINT32        udp_network_out;    /* Count of UDP datagrams sent       */
                                        /* containing network priority data  */
   AP_UINT32        udp_llc_out;        /* Count of UDP datagrams sent       */
                                        /* containing LLC commands           */
} UDP_LS_STATS_TABLE;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* end of DLC specific data                                                  */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure: backup_summary                                                 */
/*                                                                           */
/* Description: Structure which holds summary information about a backup     */
/*              machine                                                      */
/*****************************************************************************/

typedef struct backup_summary
{
  AP_UINT16           overlay_size;         /* overlay size                  */
  unsigned char       reserv1[2];           /* reserved                      */
  unsigned char       server_name[128];     /* server name                   */
  unsigned char       reserv2[4];           /* reserved                      */
} BACKUP_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: add_backup                                                     */
/*                                                                           */
/* Description: Add a backup machine                                         */
/*****************************************************************************/

typedef struct add_backup
{
  AP_UINT16              opcode;            /* verb operation code           */
  unsigned char          reserv2;
  unsigned char          format;
  AP_UINT16              primary_rc;        /* primary return code           */
  AP_UINT32              secondary_rc;      /* secondary return code         */
  unsigned char          backup_name[128];  /* server name of backup server  */
  unsigned char          reserv3[4];        /* reserved                      */
} ADD_BACKUP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: delete_backup                                                  */
/*                                                                           */
/* Description: Remove an unused backup machine                              */
/*****************************************************************************/

typedef struct delete_backup
{
  AP_UINT16              opcode;            /* verb operation code           */
  unsigned char          reserv2;
  unsigned char          format;
  AP_UINT16              primary_rc;        /* primary return code           */
  AP_UINT32              secondary_rc;      /* secondary return code         */
  unsigned char          backup_name[128];  /* server name of backup server  */
  unsigned char          reserv3[4];        /* reserved                      */
} DELETE_BACKUP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: query_sna_net                                                  */
/*                                                                           */
/* Description: Query the list of backup machines                            */
/*****************************************************************************/

typedef struct query_sna_net
{
  AP_UINT16              opcode;            /* verb operation code           */
  unsigned char          reserv2;
  unsigned char          format;
  AP_UINT16              primary_rc;        /* primary return code           */
  AP_UINT32              secondary_rc;      /* secondary return code         */
  unsigned char         *buf_ptr;           /* pointer to buffer             */
  AP_UINT32              buf_size;          /* buffer size                   */
  AP_UINT32              total_buf_size;    /* total buffer size required    */
  AP_UINT16              num_entries;       /* number of entries             */
  AP_UINT16              total_num_entries; /* total number of entries       */
  unsigned char          list_options;      /* listing options               */
  unsigned char          security;          /* security level                */
  unsigned char          domain_name[64];   /* domain name                   */
  unsigned char          server_name[128];  /* server name                   */
  unsigned char          reserv4[4];        /* reserved                      */
} QUERY_SNA_NET;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_CS_TRACE                                                   */
/*                                                                           */
/* Description: Set the client server tracing level                          */
/*****************************************************************************/

typedef struct set_cs_trace
{
  AP_UINT16              opcode;           /* verb operation code            */
  unsigned char          reserv2;
  unsigned char          format;
  AP_UINT16              primary_rc;       /* primary return code            */
  AP_UINT32              secondary_rc;     /* secondary return code          */
  unsigned char          dest_sys[128];    /* destination system name        */
  unsigned char          reserv4[4];       /* reserved                       */
  AP_UINT16              trace_flags;      /* trace flags                    */
  AP_UINT16              trace_direction;  /* trace direction                */
  unsigned char          reserv3[8];       /* Reserved                       */
} SET_CS_TRACE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_CS_TRACE                                                 */
/*                                                                           */
/* Description: Query the client server tracing level                        */
/*****************************************************************************/

typedef struct query_cs_trace
{
  AP_UINT16              opcode;           /* verb operation code            */
  unsigned char          reserv2;
  unsigned char          format;
  AP_UINT16              primary_rc;       /* primary return code            */
  AP_UINT32              secondary_rc;     /* secondary return code          */
  unsigned char          dest_sys[128];    /* destination system name        */
  unsigned char          reserv4[4];       /* reserved                       */
  AP_UINT16              trace_flags;      /* trace flags                    */
  AP_UINT16              trace_direction;  /* trace direction                */
  unsigned char          reserv3[8];       /* Reserved                       */
} QUERY_CS_TRACE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_ACCESS_DEF_DATA                                         */
/*                                                                           */
/* Description: Defined data for a TN3270 Client accessing a TN Server. The  */
/*              session data (if any) is appended to the main VCB.           */
/*****************************************************************************/

typedef struct tn3270_access_def_data
{
  unsigned char     description[32];  /* Description - null terminated       */
  unsigned char     reserv1[16];      /* reserved                            */
  AP_UINT16         address_format;   /* Format of client address            */
  AP_UINT32         num_sessions;     /* Number of sessions being added      */
  unsigned char     reserv3[64];      /* Reserved                            */
} TN3270_ACCESS_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_SESSION_DEF_DATA                                        */
/*                                                                           */
/* Description: Defined data for a TN3270 session                            */
/*****************************************************************************/

typedef struct tn3270_session_def_data
{
  AP_UINT16         sub_overlay_size;   /* size of this session structure    */
  unsigned char     description[32];    /* Description - null terminated     */
  unsigned char     tn3270_support;     /* Level of TN3270 support           */
  unsigned char     allow_specific_lu;  /* Allow access to specific LUs      */
  unsigned char     printer_lu_name[8]; /* Printer LU/pool name accessed     */
  unsigned char     reserv1[6];         /* reserved                          */
  AP_UINT16         port_number;        /* TCP/IP port used to access TN     */
                                        /* Server                            */
  unsigned char     lu_name[8];         /* LU/pool name accessed             */
  unsigned char     session_type;       /* Session type - Model 2-5          */
  unsigned char     model_override;     /* Can user override model?          */
  unsigned char     ssl_enabled;        /* Is this an SSL session?           */
  unsigned char     security_level;     /* SSL encryption strength           */
  unsigned char     cert_key_label[80]; /* Certificate key label             */
  unsigned char     listen_local_address[46]; /* Local addr cli connects to  */
  unsigned char     allow_ssl_timeout_to_nonssl;
  unsigned char     reserv3[17];
  AP_UINT32         reserv4;            /* To force ULONG alignment          */
} TN3270_SESSION_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TN3270_ACCESS                                           */
/*                                                                           */
/* Description: Define characteristics of a TN3270 Client access to TN       */
/*              Server. The session data is appended to the main VCB.        */
/*****************************************************************************/

typedef struct define_tn3270_access
{
  AP_UINT16        opcode;               /* verb operation code              */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;           /* primary return code              */
  AP_UINT32        secondary_rc;         /* secondary return code            */
  AP_UINT16        default_record;       /* Is this the default record ?     */
  unsigned char    client_address[256];  /* address of TN3270 client         */
  TN3270_ACCESS_DEF_DATA def_data;       /* defined data                     */
} DEFINE_TN3270_ACCESS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_SESSION_NAME                                            */
/*                                                                           */
/* Description: Name of TN3270 Client session to be deleted.                 */
/*****************************************************************************/

typedef struct tn3270_session_name
{
  AP_UINT16         port_number;      /* TCP/IP port used to access TN Server*/
  unsigned char     listen_local_address[46];  /* local addr cli connects to */
} TN3270_SESSION_NAME;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_TN3270_ACCESS                                           */
/*                                                                           */
/* Description: Delete a TN3270 Client access record or sessions. The        */
/*              session names are appended to the main vcb.                  */
/*****************************************************************************/

typedef struct delete_tn3270_access
{
  AP_UINT16          opcode;            /* verb operation code               */
  unsigned char      reserv2;
  unsigned char      format;
  AP_UINT16          primary_rc;        /* primary return code               */
  AP_UINT32          secondary_rc;      /* secondary return code             */
  AP_UINT16          default_record;    /* delete the default record ?       */
  unsigned char      client_address[256];/* address of TN3270 client          */
  AP_UINT32          num_sessions;      /* number of sessions to delete      */
  AP_UINT16          delete_options;    /* delete all sessions/delete user?  */
} DELETE_TN3270_ACCESS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_ACCESS_SUMMARY                                          */
/*                                                                           */
/* Description: Summary overlay returned by QUERY_TN3270_ACCESS_DEF          */
/*****************************************************************************/

typedef struct tn3270_access_summary
{
  AP_UINT16          overlay_size;        /* overlay size                    */
  AP_UINT16          default_record;      /* Is this the default record ?    */
  unsigned char      client_address[256]; /* address of TN3270 client        */
  AP_UINT16          address_format;      /* format of client address        */
  unsigned char      reserv3[6];          /* Reserved                        */
} TN3270_ACCESS_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_ACCESS_DETAIL                                           */
/*                                                                           */
/* Description: Detailed overlay returned by QUERY_TN3270_ACCESS_DEF         */
/*****************************************************************************/

typedef struct tn3270_access_detail
{
  AP_UINT16        overlay_size;          /* overlay size                    */
  AP_UINT16        sub_overlay_offset;    /* offset to first session struct  */
  AP_UINT16        default_record;        /* Is this the default record ?    */
  unsigned char    client_address[256];   /* address of TN3270 client        */
  AP_UINT32        num_filtered_sessions; /* number of sessions returned     */
  unsigned char    reserv3[4];            /* Reserved                        */
  TN3270_ACCESS_DEF_DATA def_data;        /* user definition                 */
} TN3270_ACCESS_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TN3270_ACCESS_DEF                                        */
/*                                                                           */
/* Description: Query configured clients/sessions                            */
/*****************************************************************************/

typedef struct query_tn3270_access_def
{
  AP_UINT16          opcode;               /* verb operation code            */
  unsigned char      reserv2;
  unsigned char      format;
  AP_UINT16          primary_rc;           /* primary return code            */
  AP_UINT32          secondary_rc;         /* secondary return code          */
  unsigned char     *buf_ptr;              /* pointer to buffer              */
  AP_UINT32          buf_size;             /* buffer size                    */
  AP_UINT32          total_buf_size;       /* total buffer size required     */
  AP_UINT16          num_entries;          /* number of entries              */
  AP_UINT16          total_num_entries;    /* total number of entries        */
  unsigned char      list_options;         /* listing options                */
  unsigned char      reserv3;              /* reserved                       */
  AP_UINT16          default_record;       /* start with default record ?    */
  unsigned char      client_address[256];  /* address of TN3270 client       */
  AP_UINT16          port_number;          /* TCP/IP port used to access TN  */
                                           /* Server                         */
  AP_UINT32          num_init_sessions;    /* number of sessions for first   */
                                           /* user when starting in middle   */
  AP_UINT32          num_last_sessions;    /* number of sessions on last     */
                                           /* detail overlay if last user is */
                                           /* incomplete                     */
  unsigned char      last_user_incomplete; /* set to AP_YES if session data  */
                                           /* for last user is incomplete    */
  unsigned char      reserv4[11];          /* Reserved                       */
} QUERY_TN3270_ACCESS_DEF;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_ASSOCIATION_DEF_DATA                                    */
/*                                                                           */
/* Description: Defined data for an association between a display LU and     */
/*              printer LU.                                                  */
/*****************************************************************************/

typedef struct tn3270_association_def_data
{
  unsigned char   description[32];    /* resource description                */
  unsigned char   reserve0[16];       /* reserved                            */
  unsigned char   printer_lu_name[8]; /* name of printer LU/pool             */
  unsigned char   reserv2[8];         /* reserved                            */
} TN3270_ASSOCIATION_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TN3270_ASSOCIATION                                      */
/*                                                                           */
/* Description: Define an association between a display LU and a pinter      */
/*              LU.                                                          */
/*****************************************************************************/

typedef struct define_tn3270_association
{
  AP_UINT16        opcode;               /* verb operation code              */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;           /* primary return code              */
  AP_UINT32        secondary_rc;         /* secondary return code            */
  unsigned char    display_lu_name[8];   /* name of display LU/pool          */
  TN3270_ASSOCIATION_DEF_DATA def_data;  /* defined data                     */
} DEFINE_TN3270_ASSOCIATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_TN3270_ASSOCIATION                                      */
/*                                                                           */
/* Description: Delete a TN3270 association record.                          */
/*****************************************************************************/

typedef struct delete_tn3270_association
{
  AP_UINT16         opcode;             /* verb operation code               */
  unsigned char     reserv2;
  unsigned char     format;
  AP_UINT16         primary_rc;         /* primary return code               */
  AP_UINT32         secondary_rc;       /* secondary return code             */
  unsigned char     display_lu_name[8]; /* display LU name                   */
} DELETE_TN3270_ASSOCIATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_ASSOCIATION                                             */
/*                                                                           */
/* Description: Overlay returned by QUERY_TN3270_ASSOCIATION                 */
/*****************************************************************************/

typedef struct tn3270_association
{
  AP_UINT16        overlay_size;          /* overlay size                    */
  unsigned char    reserv2[2];            /* reserved                        */
  unsigned char    display_lu_name[8];    /* display LU name                 */
  TN3270_ASSOCIATION_DEF_DATA def_data;   /* association definition          */
} TN3270_ASSOCIATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TN3270_ASSOCIATION                                       */
/*                                                                           */
/* Description: Query configured LU associations                             */
/*****************************************************************************/

typedef struct query_tn3270_association
{
  AP_UINT16          opcode;               /* verb operation code            */
  unsigned char      reserv2;
  unsigned char      format;
  AP_UINT16          primary_rc;           /* primary return code            */
  AP_UINT32          secondary_rc;         /* secondary return code          */
  unsigned char     *buf_ptr;              /* pointer to buffer              */
  AP_UINT32          buf_size;             /* buffer size                    */
  AP_UINT32          total_buf_size;       /* total buffer size required     */
  AP_UINT16          num_entries;          /* number of entries              */
  AP_UINT16          total_num_entries;    /* total number of entries        */
  unsigned char      list_options;         /* listing options                */
  unsigned char      reserv3;              /* reserved                       */
  unsigned char      display_lu_name[8];   /* display LU name                */
} QUERY_TN3270_ASSOCIATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_DEFAULTS_DEF_DATA                                       */
/*                                                                           */
/* Description: Defined data for TN3270 Server defaults                      */
/*****************************************************************************/

typedef struct tn3270_defaults_def_data
{
  AP_UINT16       force_responses;    /* force printer responses?            */
  AP_UINT16       keepalive_method;   /* method for sending keepalives       */
  AP_UINT32       keepalive_interval; /* interval between keepalives         */
  unsigned char   reserv2[32];        /* reserved                            */
} TN3270_DEFAULTS_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TN3270_DEFAULTS                                         */
/*                                                                           */
/* Description: Define TN3270 Server defaults                                */
/*****************************************************************************/

typedef struct define_tn3270_defaults
{
  AP_UINT16        opcode;               /* verb operation code              */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;           /* primary return code              */
  AP_UINT32        secondary_rc;         /* secondary return code            */
  TN3270_DEFAULTS_DEF_DATA def_data;     /* defined data                     */
} DEFINE_TN3270_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TN3270_DEFAULTS                                          */
/*                                                                           */
/* Description: Query configured TN3270 Server defaults                      */
/*****************************************************************************/

typedef struct query_tn3270_defaults
{
  AP_UINT16          opcode;               /* verb operation code            */
  unsigned char      reserv2;
  unsigned char      format;
  AP_UINT16          primary_rc;           /* primary return code            */
  AP_UINT32          secondary_rc;         /* secondary return code          */
  TN3270_DEFAULTS_DEF_DATA def_data;       /* defined defaults               */
} QUERY_TN3270_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN3270_SSL_LDAP_DEF_DATA                                       */
/*                                                                           */
/* Description: Defined data overlay for TN3270 SSL client authentication    */
/* LDAP server                                                               */
/*****************************************************************************/

typedef struct tn3270_ssl_ldap_def_data
{
  AP_UINT16       overlay_size;       /* overlay size                        */
  unsigned char   auth_type;          /* takes values NAP_LOCAL_ONLY,        */
                                      /* NAP_LOCAL_X500 Reserved for later   */
  unsigned char   reserv1;            /* reserved                            */
  unsigned char   ldap_addr[256];     /* IP address of LDAP server           */
  AP_UINT16       ldap_port;          /* Port number on LDAP server          */
  unsigned char   ldap_user[1024];    /* Distinguished name of certificate   */
                                      /* revocation list on LDAP server.     */
  unsigned char   ldap_password[128]; /* Password of LDAP server             */
  unsigned char   reserv2[256];       /* reserved                            */
} TN3270_SSL_LDAP_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TN3270_SSL_LDAP                                         */
/*                                                                           */
/* Description: Define TN3270 SSL client authentication LDAP server          */
/*****************************************************************************/

typedef struct define_tn3270_ssl_ldap
{
  AP_UINT16        opcode;               /* verb operation code              */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;           /* primary return code              */
  AP_UINT32        secondary_rc;         /* secondary return code            */
} DEFINE_TN3270_SSL_LDAP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TN3270_SSL_LDAP                                          */
/*                                                                           */
/* Description: Query configured TN3270 SSL client authentication LDAP       */
/* server                                                                    */
/*****************************************************************************/

typedef struct query_tn3270_ssl_ldap
{
  AP_UINT16          opcode;               /* verb operation code            */
  unsigned char      reserv2;
  unsigned char      format;
  AP_UINT16          primary_rc;           /* primary return code            */
  AP_UINT32          secondary_rc;         /* secondary return code          */
  unsigned char     *buf_ptr;              /* pointer to buffer              */
  AP_UINT32          buf_size;             /* buffer size                    */
  AP_UINT32          total_buf_size;       /* total buffer size required     */
  AP_UINT16          num_entries;          /* number of entries              */
  AP_UINT16          total_num_entries;    /* total number of entries        */
  unsigned char      list_options;         /* listing options                */
  unsigned char      reserv3;              /* reserved                       */
} QUERY_TN3270_SSL_LDAP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TN3270_EXPRESS_LOGON                                    */
/*                                                                           */
/* Description: Defines the configuration for TN3270 Express Logon           */
/*****************************************************************************/
typedef struct define_tn3270_express_logon
{
   AP_UINT16        opcode;                /* verb operation code            */
   unsigned char    reserv2;               /* reserved                       */
   unsigned char    format;                /* reserved                       */
   AP_UINT16        primary_rc;            /* primary return code            */
   AP_UINT32        secondary_rc;          /* secondary return code          */
   unsigned char    dcas_server[256];      /* NULL terminated hostname of    */
                                           /*    host DCAS server            */
   AP_UINT16        dcas_port;             /* IP Port of host DCAS server    */
   unsigned char    enabled;               /* Is express logon enabled?      */
   unsigned char    reserv3[33];           /* reserved                       */
} DEFINE_TN3270_EXPRESS_LOGON;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TN3270_EXPRESS_LOGON                                     */
/*                                                                           */
/* Description: Queries the configuration for TN3270 Express Logon           */
/*****************************************************************************/
typedef struct query_tn3270_express_logon
{
   AP_UINT16        opcode;                /* verb operation code            */
   unsigned char    reserv2;               /* reserved                       */
   unsigned char    format;                /* reserved                       */
   AP_UINT16        primary_rc;            /* primary return code            */
   AP_UINT32        secondary_rc;          /* secondary return code          */
   unsigned char    dcas_server[256];      /* NULL terminated hostname of    */
                                           /*    host DCAS server            */
   AP_UINT16        dcas_port;             /* IP Port of host DCAS server    */
   unsigned char    enabled;               /* Is express logon enabled?      */
   unsigned char    reserv3[33];           /* reserved                       */
} QUERY_TN3270_EXPRESS_LOGON;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_TN_SERVER_TRACE                                            */
/*                                                                           */
/* Description: Set the tracing level in TN Server.                          */
/*****************************************************************************/

typedef struct set_tn_server_trace
{
  AP_UINT16        opcode;                 /* verb operation code            */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;             /* primary return code            */
  AP_UINT32        secondary_rc;           /* secondary return code          */
  AP_UINT16        trace_flags;            /* trace flags                    */
  unsigned char    reserv3[6];             /* Reserved                       */
} SET_TN_SERVER_TRACE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TN_SERVER_TRACE                                          */
/*                                                                           */
/* Description: Query the tracing level in TN Server.                        */
/*****************************************************************************/

typedef struct query_tn_server_trace
{
  AP_UINT16        opcode;                 /* verb operation code            */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;             /* primary return code            */
  AP_UINT32        secondary_rc;           /* secondary return code          */
  AP_UINT16        trace_flags;            /* trace flags                    */
  unsigned char    reserv3[6];             /* Reserved                       */
} QUERY_TN_SERVER_TRACE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN_SERVER_SESSION_USER_INFO                                    */
/*                                                                           */
/* Description: Information about the user of a LU of type 0 to 3            */
/*              This field is mapped over the app_spec_det_data of           */
/*              LU_0_TO_3_DET_DATA on QUERY_LU_0_TO_3                        */
/*****************************************************************************/

typedef struct tn_server_session_user_info
{
  unsigned char   user_ip_address[40];     /* IP address of user             */
  AP_UINT16       port_number;             /* TCP/IP port used to access TN  */
                                           /* Server                         */
  AP_UINT16       cb_number;               /* TN Server control block number */
                                           /* allocated to user              */
  AP_UINT16       cfg_default;             /* Is this the default record ?   */
  unsigned char   cfg_address[68];         /* Name of configuration record   */
                                           /* used                           */
  AP_UINT16       cfg_format;              /* Format of record used          */
  unsigned char   tn3270_level;            /* TN3270 level used              */
  unsigned char   lu_select;               /* Method of LU selection         */
  unsigned char   request_lu_name[8];      /* Requested LU name (or          */
                                           /* associated display LU name)    */
  unsigned char   cipher_spec;             /* SSL cipher spec used           */
  unsigned char   reserv3[21];             /* Reserved                       */
} TN_SERVER_SESSION_USER_INFO;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: TN_REDIRECT_ADDRESS                                            */
/*                                                                           */
/* Description: Address field for a TN redirector session. This is used to   */
/* identify records for DEFINE/DELETE and QUERY verbs.                       */
/*****************************************************************************/

typedef struct tn_redirect_address
{
  AP_UINT16         default_record;         /* Is this the default record?   */
  unsigned char     address_format;         /* IP address or FQN?            */
  unsigned char     client_address[256];    /* IP address, FQN or alias of   */
                                            /* the client.                   */
  AP_UINT16         port_number;            /* port the client connects on.  */
  unsigned char     listen_local_address[46]; /* Local addr cli connects to  */
  unsigned char     reserved[34];           /* reserved.                     */
} TN_REDIRECT_ADDRESS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN_REDIRECT_DEF_DATA                                           */
/*                                                                           */
/* Description: Defined data for a TN redirector record.                     */
/*****************************************************************************/

typedef struct tn_redirect_def_data
{
  unsigned char     description[32];         /* resource description         */
  unsigned char     reserve0[16];            /* reserved                     */
  unsigned char     cli_conn_ssl_enabled;    /* Is SSL enabled inbound?      */
  unsigned char     serv_conn_ssl_enabled;   /* Use SSL on outbound?         */
  unsigned char     host_address_format;     /* IP address or FQN?           */
  unsigned char     reserv1;
  unsigned char     host_address[256];       /* IP address, FQN or alias of  */
                                             /* the host.                    */
  AP_UINT16         host_port_number;        /* Port number to use to connec */
                                             /* to the host.                 */
  unsigned char     cli_conn_security_level; /* SSL encryption strengh       */
  unsigned char     serv_conn_security_level;
  unsigned char     cli_conn_cert_key_label[80];
                                             /* Key label for cert. to use   */
                                             /* on client-side connection    */
  unsigned char     serv_conn_cert_key_label[80];  /* Key label for cert. to */
                                             /* use on server-side           */
                                             /* connection                   */
  unsigned char     reserved[46];            /* Reserved.                    */
} TN_REDIRECT_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TN_REDIRECT                                             */
/*                                                                           */
/* Description: Verb to define a TN redirector record.                       */
/*****************************************************************************/

typedef struct define_tn_redirect
{
  AP_UINT16         opcode;                  /* NAP_DEFINE_TN_REDIRECT       */
  unsigned char     reserv2;                 /* reserved                     */
  unsigned char     format;                  /* reserved                     */
  AP_UINT16         primary_rc;              /* Primary return code          */
  AP_UINT32         secondary_rc;            /* Secondary return code        */
  TN_REDIRECT_ADDRESS addr;                  /* Client address info          */
  TN_REDIRECT_DEF_DATA def_data;             /* Destination host/security    */
} DEFINE_TN_REDIRECT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_TN_REDIRECT                                             */
/*                                                                           */
/* Description: Verb to delete a TN redirector record.                       */
/*****************************************************************************/

typedef struct delete_tn_redirect
{
  AP_UINT16         opcode;                  /* NAP_DELETE_TN_REDIRECT       */
  unsigned char     reserv2;                 /* reserved                     */
  unsigned char     format;                  /* reserved                     */
  AP_UINT16         primary_rc;              /* Primary return code          */
  AP_UINT32         secondary_rc;            /* Secondary return code        */
  TN_REDIRECT_ADDRESS addr;                  /* Client address info          */
} DELETE_TN_REDIRECT;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TN_REDIRECT                                              */
/*                                                                           */
/* Description: Verb to query TN redirector records.                         */
/*****************************************************************************/

typedef struct query_tn_redirect_def
{
  AP_UINT16         opcode;                  /* NAP_QUERY_TN_REDIRECT_DEF    */
  unsigned char     reserv2;                 /* reserved                     */
  unsigned char     format;                  /* reserved                     */
  AP_UINT16         primary_rc;              /* Primary return code          */
  AP_UINT32         secondary_rc;            /* Secondary return code        */
  unsigned char     *buf_ptr;                /* pointer to buffer            */
  AP_UINT32         buf_size;                /* buffer size                  */
  AP_UINT32         total_buf_size;          /* total buffer size required   */
  AP_UINT16         num_entries;             /* number of entries            */
  AP_UINT16         total_num_entries;       /* total number of entries      */
  unsigned char     list_options;            /* listing options              */
  unsigned char     reserv3[3];              /* reserved                     */
  TN_REDIRECT_ADDRESS addr;                  /* Client address info          */
} QUERY_TN_REDIRECT_DEF;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN_REDIRECT_DATA                                               */
/*                                                                           */
/* Description: Returned overlay for QUERY_TN_REDIRECT_DEF                   */
/*****************************************************************************/

typedef struct tn_redirect_data
{
  AP_UINT16         overlay_size;            /* size of returned overlay     */
  unsigned char     reserv1[2];              /* reserved                     */
  TN_REDIRECT_ADDRESS addr;                  /* Client address info          */
  TN_REDIRECT_DEF_DATA def_data;             /* Destination host/security    */
} TN_REDIRECT_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SET_BCK_CS_TRACE                                               */
/*                                                                           */
/* Description: Turn off tracing for back level clients                      */
/*****************************************************************************/

typedef struct set_bck_cs_trace
{
  AP_UINT16        opcode;                 /* verb operation code            */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;             /* primary return code            */
  AP_UINT32        secondary_rc;           /* secondary return code          */
  AP_UINT32        trace_flags;            /* trace flags                    */
  unsigned char    init_flags;             /* YES if initialising flags      */
  unsigned char    set_flags;              /* YES if setting flags           */
                                           /* NO if clearing flags           */
  unsigned char    reserv3[6];             /* Reserved                       */
} SET_BCK_CS_TRACE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_BCK_CS_TRACE                                             */
/*                                                                           */
/* Description: Query tracing of back level clients                          */
/*****************************************************************************/

typedef struct query_bck_cs_trace
{
  AP_UINT16        opcode;                 /* verb operation code            */
  unsigned char    reserv2;
  unsigned char    format;
  AP_UINT16        primary_rc;             /* primary return code            */
  AP_UINT32        secondary_rc;           /* secondary return code          */
  AP_UINT32        trace_flags;            /* trace flags                    */
  unsigned char    reserv3[8];             /* Reserved                       */
} QUERY_BCK_CS_TRACE;

/*****************************************************************************/

/*****************************************************************************/
/* Constants for extra data key name lengths.                                */
/*****************************************************************************/
#define NAP_PRIMKEY_LEN   64
#define NAP_SECNKEY_LEN   14

/**STRUCT+********************************************************************/
/* Structure: COMMON_EXTRA_DATA                                              */
/*                                                                           */
/* Description: Common structure for extra data.                             */
/*****************************************************************************/
typedef struct common_extra_data
{
  AP_UINT16         length;                /* length of extra data           */
  unsigned char     secondary_key[14];     /* Extra data name                */
} COMMON_EXTRA_DATA;
/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: XDATA_LS                                                       */
/*                                                                           */
/* Description: Extra data for a link station.                               */
/*****************************************************************************/
typedef struct xdata_ls
{
  AP_UINT16         length;                /* length of extra data           */
  unsigned char     secondary_key[14];     /* Extra data name                */
  unsigned char     port_name[14];         /* Secondary key port name        */
} XDATA_LS;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: XDATA_LU_POOL                                                  */
/*                                                                           */
/* Description: Extra data for an LU pool.                                   */
/*****************************************************************************/
typedef struct xdata_lu_pool
{
  unsigned char     length;                /* length of extra data           */
  unsigned char     secondary_key[14];     /* Extra data name                */
  unsigned char     host_group_name[14];   /* Parent profile                 */
} XDATA_LU_POOL;
/**STRUCT-********************************************************************/

/*****************************************************************************/
/* Constants used for G-SNA and secondary LU0 LUs.                           */
/*****************************************************************************/
#define AP_GSNA_LU     0x01
#define AP_SEC_LU0_LU  0x02

#define AP_LU0_TERM_SELF  0x00
#define AP_LU0_RSHUTD     0x01

/**STRUCT+********************************************************************/
/* Structure: XDATA_LU_0_TO_3                                                */
/*                                                                           */
/* Description: Extra data for a lu_0_to_3                                   */
/*****************************************************************************/
typedef struct xdata_lu_0_to_3
{
  AP_UINT16         length;                /* length of extra data           */
  unsigned char     secondary_key[14];     /* Extra data name                */
  unsigned char     remote_lu_name[17];    /* LU1-3 API FQ remote LU name    */
  unsigned char     lu_use;                /* generic SNA/LU0 support        */
  unsigned char     host_app[8];           /* Sec LU0 host app. name         */
  unsigned char     log_mode[8];           /* Sec LU0 log mode table entry   */
  unsigned char     plu_partner[8];        /* Sec LU0 passthru partner PLU   */
  unsigned char     init_self;             /* Sec LU0 Send INIT_SELF?        */
  unsigned char     session_term;          /* Sec LU0 termination option     */
                                           /* AP_LU0_TERM_SELF/AP_LU0_RSHUTD */
  unsigned char     api_trace;             /* Sec LU0 API tracing?           */
} XDATA_LU_0_TO_3;
/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: DEFINE_EXTRA_DATA                                              */
/*                                                                           */
/* Description: Define extra data for a given SNAP-IX resource.              */
/*****************************************************************************/
typedef struct define_extra_data
{
  AP_UINT16         opcode;                /* verb operation code            */
  unsigned char     reserv2;               /* reserved                       */
  unsigned char     format;                /* format indicator               */
  AP_UINT16         primary_rc;            /* primary return code            */
  AP_UINT32         secondary_rc;          /* secondary return code          */
  AP_UINT16         resource_type;         /* Type of resource               */
  unsigned char     primary_key[64];       /* SNAP-IX resource name          */
  COMMON_EXTRA_DATA xdata;                 /* Extra data for resource     s  */
} DEFINE_EXTRA_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_EXTRA_DATA                                              */
/*                                                                           */
/* Description: Deletes extra data for a given SNAP-IX resource.             */
/*****************************************************************************/
typedef struct delete_extra_data
{
  AP_UINT16         opcode;                /* verb operation code            */
  unsigned char     reserv2;               /* reserved                       */
  unsigned char     format;                /* format indicator               */
  AP_UINT16         primary_rc;            /* primary return code            */
  AP_UINT32         secondary_rc;          /* secondary return code          */
  AP_UINT16         resource_type;         /* Type of resource               */
  unsigned char     primary_key[64];       /* SNAP-IX resource name          */
} DELETE_EXTRA_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_EXTRA_DATA                                               */
/*                                                                           */
/* Description: Queries extra data for a given SNAP-IX resource.             */
/*****************************************************************************/
typedef struct query_extra_data
{
  AP_UINT16         opcode;                /* verb operation code            */
  unsigned char     reserv2;               /* reserved                       */
  unsigned char     format;                /* format indicator               */
  AP_UINT16         primary_rc;            /* primary return code            */
  AP_UINT32         secondary_rc;          /* secondary return code          */
  unsigned char     *buf_ptr;              /* pointer to buffer              */
  AP_UINT32         buf_size;              /* buffer size                    */
  AP_UINT32         total_buf_size;        /* total buffer size required     */
  AP_UINT16         num_entries;           /* number of entries              */
  AP_UINT16         total_num_entries;     /* total number of entries        */
  unsigned char     list_options;          /* listing options                */
  unsigned char     reserv3;               /* reserved                       */
  AP_UINT16         resource_type;         /* Type of resource               */
  AP_UINT16         supplied_key;          /* What kind of supplied key      */
  unsigned char     primary_key[64];       /* SNAP-IX resource name          */
  unsigned char     secondary_key[14];     /* Extra data name                */
  unsigned char     reserv4[2];            /* Reserved                       */
} QUERY_EXTRA_DATA;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: EXTRA_DATA_DETAIL                                              */
/*                                                                           */
/* Description: Returned overlay for extra data.                             */
/*****************************************************************************/
typedef struct extra_data_detail
{
  AP_UINT16         overlay_size;          /* length of extra data           */
  unsigned char     primary_key[64];       /* SNAP-IX resource name          */
  COMMON_EXTRA_DATA xdata;                 /* Actual extra data              */
} EXTRA_DATA_DETAIL;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DSLU_TEMPLATE                                                */
/*                                                                           */
/* Description: Data used to define a DSPU_TEMPLATE                          */
/*****************************************************************************/
typedef struct dslu_template
{
  unsigned char   min_nau;                 /* Minimum NAU address            */
  unsigned char   max_nau;                 /* Maximum NAU address            */
  unsigned char   allow_timeout;           /* Allow timeout?                 */
  unsigned char   delayed_logon;           /* Delay logon?                   */
  unsigned char   reserv1[8];              /* reserved                       */
  unsigned char   host_lu[8];              /* Host LU name                   */
} DSLU_TEMPLATE;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_DSPU_TEMPLATE                                           */
/*                                                                           */
/* Description: DEFINE_DSPU_TEMPLATE verb control block                      */
/*****************************************************************************/
typedef struct define_dspu_template
{
  AP_UINT16       opcode;                  /* verb operation code            */
  unsigned char   reserv3;                 /* reserved                       */
  unsigned char   format;                  /* format indicator               */
  AP_UINT16       primary_rc;              /* primary return code            */
  AP_UINT32       secondary_rc;            /* secondary return code          */
  unsigned char   template_name[8];        /* name of template               */
  unsigned char   description[32];         /* description of template        */
  unsigned char   reserv2[16];             /* reserved                       */
  unsigned char   modify_template;         /* modify or replace template?    */
  unsigned char   reserv1[11];             /* reserved                       */
  AP_UINT16       max_instance;            /* maximum instances              */
  AP_UINT16       num_of_dslu_templates;   /* number of DSLU templates       */
} DEFINE_DSPU_TEMPLATE;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_DSPU_TEMPLATE                                           */
/*                                                                           */
/* Description: DELETE_DSPU_TEMPLATE verb control block                      */
/*****************************************************************************/
typedef struct delete_dspu_template
{
  AP_UINT16       opcode;                  /* verb operation code            */
  unsigned char   reserv2;                 /* reserved                       */
  unsigned char   format;                  /* format indicator               */
  AP_UINT16       primary_rc;              /* primary return code            */
  AP_UINT32       secondary_rc;            /* secondary return code          */
  unsigned char   template_name[8];        /* name of template               */
  AP_UINT16       num_of_dslu_templates;   /* number of DSLU templates       */
  unsigned char   reserv1[10];             /* reserved                       */
} DELETE_DSPU_TEMPLATE;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_DSPU_TEMPLATE                                            */
/*                                                                           */
/* Description: Used for querying downstream PU templates                    */
/*****************************************************************************/

typedef struct query_dspu_template
{
  AP_UINT16       opcode;                  /* verb operation code            */
  unsigned char   reserv1;                 /* reserved                       */
  unsigned char   format;                  /* format indicator               */
  AP_UINT16       primary_rc;              /* primary return code            */
  AP_UINT32       secondary_rc;            /* secondary return code          */
  unsigned char  *buf_ptr;                 /* buffer pointer                 */
  AP_UINT32       buf_size;                /* buffer size                    */
  AP_UINT32       total_buf_size;          /* total buffer size              */
  AP_UINT16       num_entries;             /* number of entries              */
  AP_UINT16       total_num_entries;       /* total number of entries        */
  unsigned char   list_options;            /* list options                   */
  unsigned char   reserv3;                 /* reserved                       */
  unsigned char   template_name[8];        /* Name of DSPU template          */
} QUERY_DSPU_TEMPLATE;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DSPU_TEMPLATE_DATA                                             */
/*                                                                           */
/* Description: Overlay returned for QUERY_DSPU_TEMPLATE (above)             */
/*****************************************************************************/

typedef struct dspu_template_data
{
  AP_UINT16      overlay_size;           /* Size of this overlay             */
  unsigned char  template_name[8];       /* Name of template                 */
  unsigned char  description[32];        /* Description of template          */
  unsigned char  reserv2[16];            /* reserved                         */
  unsigned char  reserv1[12];            /* reserved                         */
  AP_UINT16      max_instance;           /* Maximum instances allowed        */
  AP_UINT16      active_instance;        /* Active instance                  */
  AP_UINT16      num_of_dslu_templates;  /* Number of DSLU templates         */
} DSPU_TEMPLATE_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DSLU_TEMPLATE_DATA                                             */
/*                                                                           */
/* Description: Overlay returned for QUERY_DSPU_TEMPLATE (above)             */
/*****************************************************************************/

typedef struct dslu_template_data
{
  AP_UINT16        overlay_size;         /* Size of this overlay             */
  unsigned char    reserv1[2];           /* reserved                         */
  DSLU_TEMPLATE    dslu_template;        /* DSLU template details            */
} DSLU_TEMPLATE_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_LS_ROUTING                                               */
/*                                                                           */
/* Description: Used to obtain information on the location of a partner LU   */
/*              using a link station.                                        */
/*****************************************************************************/

typedef struct query_ls_routing
{
  AP_UINT16         opcode;                /* verb operation code            */
  unsigned char     reserv2;               /* reserved                       */
  unsigned char     format;                /* format indicator               */
  AP_UINT16         primary_rc;            /* primary return code            */
  AP_UINT32         secondary_rc;          /* secondary return code          */
  unsigned char    *buf_ptr;               /* buffer pointer                 */
  AP_UINT32         buf_size;              /* buffer size                    */
  AP_UINT32         total_buf_size;        /* total buffer size              */
  AP_UINT16         num_entries;           /* number of entries              */
  AP_UINT16         total_num_entries;     /* total number of entries        */
  unsigned char     list_options;          /* list options                   */
  unsigned char     reserv3;               /* reserved                       */
  unsigned char     lu_name[8];            /* LU Name                        */
  unsigned char     lu_alias[8];           /* reserved                       */
  unsigned char     fq_partner_lu[17];     /* partner lu name                */
  unsigned char     wildcard_fqplu;        /* wildcard partner LU flag       */
  unsigned char     reserv4[2];            /* reserved                       */
} QUERY_LS_ROUTING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: LS_ROUTING_DATA                                                */
/*                                                                           */
/* Description:  Overlay returned for QUERY_LS_ROUTING                       */
/*****************************************************************************/

typedef struct ls_routing_data
{
  AP_UINT16        overlay_size;
  unsigned char    lu_name[8];              /* local LU name                 */
  unsigned char    lu_alias[8];             /* reserved                      */
  unsigned char    fq_partner_lu[17];       /* partner lu                    */
  unsigned char    wildcard_fqplu;          /* wildcard partner LU flag      */
  unsigned char    description[32];         /* resource description          */
  unsigned char    reserv1[16];             /* reserved                      */
  unsigned char    ls_name[8];              /* link to use                   */
  unsigned char    reserv3[2];              /* reserved                      */
} LS_ROUTING_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_LS_ROUTING                                              */
/*                                                                           */
/* Description: Defines the location of a partner LU using a link station    */
/*****************************************************************************/

typedef struct define_ls_routing
{
  AP_UINT16         opcode;                /* verb operation code            */
  unsigned char     reserv2;               /* reserved                       */
  unsigned char     format;                /* format indicator               */
  AP_UINT16         primary_rc;            /* primary return code            */
  AP_UINT32         secondary_rc;          /* secondary return code          */
  unsigned char     lu_name[8];            /* LU Name                        */
  unsigned char     lu_alias[8];           /* reserved                       */
  unsigned char     fq_partner_lu[17];     /* partner lu name                */
  unsigned char     wildcard_fqplu;        /* wildcard partner LU flag       */
  unsigned char     description[32];       /* resource description           */
  unsigned char     reserv5[16];           /* reserved                       */
  unsigned char     ls_name[8];            /* link to use                    */
  unsigned char     reserv4[2];            /* reserved                       */
} DEFINE_LS_ROUTING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_LS_ROUTING                                              */
/*                                                                           */
/* Description: Deletes the partner LU location                              */
/*****************************************************************************/

typedef struct delete_ls_routing
{
  AP_UINT16         opcode;                /* verb operation code            */
  unsigned char     reserv2;               /* reserved                       */
  unsigned char     format;                /* format indicator               */
  AP_UINT16         primary_rc;            /* primary return code            */
  AP_UINT32         secondary_rc;          /* secondary return code          */
  unsigned char     lu_name[8];            /* LU Name                        */
  unsigned char     lu_alias[8];           /* reserved                       */
  unsigned char     fq_partner_lu[17];     /* partner lu name                */
  unsigned char     wildcard_fqplu;        /* wildcard partner LU flag       */
  unsigned char     reserv4[2];            /* reserved                       */
} DELETE_LS_ROUTING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RTP_STATISTICS                                                 */
/*                                                                           */
/* Description: RTP connection statistics                                    */
/*****************************************************************************/

typedef struct rtp_statistics
{
  AP_UINT32       bytes_sent;          /* total num of bytes sent            */
  AP_UINT32       bytes_received;      /* total num bytes received           */
  AP_UINT32       bytes_resent;        /* total num of bytes resent          */
  AP_UINT32       bytes_discarded;     /* total num bytes discarded          */
  AP_UINT32       packets_sent;        /* total num of packets sent          */
  AP_UINT32       packets_received;    /* total num packets received         */
  AP_UINT32       packets_resent;      /* total num of packets resent        */
  AP_UINT32       packets_discarded;   /* total num packets discarded        */
  AP_UINT32       gaps_detected;       /* gaps detected                      */
  AP_UINT32       send_rate;           /* current send rate                  */
  AP_UINT32       max_send_rate;       /* maximum send rate                  */
  AP_UINT32       min_send_rate;       /* minimum send rate                  */
  AP_UINT32       receive_rate;        /* current receive rate               */
  AP_UINT32       max_receive_rate;    /* maximum receive rate               */
  AP_UINT32       min_receive_rate;    /* minimum receive rate               */
  AP_UINT32       burst_size;          /* current burst size                 */
  AP_UINT32       up_time;             /* total uptime of connection         */
  AP_UINT32       smooth_rtt;          /* smoothed round-trip time           */
  AP_UINT32       last_rtt;            /* last round-trip time               */
  AP_UINT32       short_req_timer;     /* SHORT_REQ timer duration           */
  AP_UINT32       short_req_timeouts;  /* number of SHORT_REQ timeouts       */
  AP_UINT32       liveness_timeouts;   /* number of liveness timeouts        */
  AP_UINT32       in_invalid_sna_frames;
                                       /* number of invalid SNA frames       */
                                       /* received                           */
  AP_UINT32       in_sc_frames;        /* number of SC frames received       */
  AP_UINT32       out_sc_frames;       /* number of SC frames sent           */
  AP_INT32        delay_change_sum;    /* delay change sum                   */
  AP_UINT32       current_receiver_threshold;
                                       /* current ARB-R receiver threshold   */
  AP_UINT32       minimum_receiver_threshold;
                                       /* minimum ARB-R receiver threshold   */
  AP_UINT32       maximum_receiver_threshold;
                                       /* maximum ARB-R receiver threshold   */
  AP_UINT32       sent_normals_count;  /* number of NORMALs sent             */
  AP_UINT32       sent_slowdowns_count;
                                       /* number of SLOWDOWNs sent           */
  AP_UINT32       rcvd_normals_count;  /* number of NORMALs received         */
  AP_UINT32       rcvd_slowdowns_count;
                                       /* number of SLOWDOWNs received       */
  AP_UINT32       dcs_reset_count_non_heal;
                                       /* Number of non-healing resets       */
  AP_UINT16       dcs_reset_count_healing;
                                       /* Number of self-healing DCS resets  */
  unsigned char   arb_mode;            /* ARB mode (GREEN, YELLOW, RED)      */
  unsigned char   reserve[1];          /* reserved                           */
} RTP_STATISTICS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RTP_INDICATION                                                 */
/*                                                                           */
/* Description: Sent from RTP to HPR, and from HPR to NOF (for fanning out)  */
/*              when the state of an RTP connection changes                  */
/*****************************************************************************/

typedef struct rtp_indication
{
  AP_UINT16       opcode;               /* verb operation code               */
  unsigned char   reserv2;              /* reserved                          */
  unsigned char   format;               /* format                            */
  AP_UINT16       primary_rc;           /* primary return code               */
  AP_UINT32       secondary_rc;         /* secondary return code             */
  unsigned char   data_lost;            /* previous indication(s) lost       */
  unsigned char   connection_state;     /* the current state of the RTP      */
                                        /* connection                        */
  unsigned char   rtp_name[8];          /* name of the RTP connection        */
  AP_UINT16       num_sess_active;      /* number of active sessions         */
  unsigned char   indication_cause;     /* reason for this indication        */
  unsigned char   connection_type;      /* usage of RTP connection           */
  unsigned char   reserv3[2];           /* reserved                          */
  RTP_STATISTICS  rtp_stats;            /* RTP statistics                    */
} RTP_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_RTP_CONNECTION                                           */
/*                                                                           */
/* Description: NOF verb to query information about one or more RTP          */
/*              connections                                                  */
/*****************************************************************************/

typedef struct query_rtp_connection
{
  AP_UINT16        opcode;              /* verb operation code               */
  unsigned char    reserv2;             /* reserved                          */
  unsigned char    format;              /* format                            */
  AP_UINT16        primary_rc;          /* Primary return code               */
  AP_UINT32        secondary_rc;        /* Secondary return code             */
  unsigned char   *buf_ptr;             /* pointer to buffer                 */
  AP_UINT32        buf_size;            /* buffer size                       */
  AP_UINT32        total_buf_size;      /* total buffer size require         */
  AP_UINT16        num_entries;         /* number of entries                 */
  AP_UINT16        total_num_entries;   /* total number of entries           */
  unsigned char    list_options;        /* listing options                   */
  unsigned char    reserv3;             /* reserved                          */
  unsigned char    rtp_name[8];         /* name of RTP connection            */
} QUERY_RTP_CONNECTION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RTP_CONNECTION_SUMMARY                                         */
/*                                                                           */
/* Description: Summary information returned for QUERY_RTP_CONNECTION        */
/*****************************************************************************/

typedef struct rtp_connection_summary
{
  AP_UINT16       overlay_size;          /* size of this entry               */
  unsigned char   rtp_name[8];           /* RTP connection name              */
  unsigned char   first_hop_ls_name[8];  /* LS name of first hop             */
  unsigned char   dest_node_name[17];    /* fully-qualified name of          */
                                         /* destination node                 */
  unsigned char   connection_type;       /* LU-LU or CP-CP connection?       */
  unsigned char   cos_name[8];           /* class-of-service name            */
  AP_UINT16       num_sess_active;       /* number of active sessions        */
} RTP_CONNECTION_SUMMARY;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RTP_CONNECTION_DETAIL                                          */
/*                                                                           */
/* Description: Detailed information returned for QUERY_RTP_CONNECTION       */
/*****************************************************************************/

typedef struct rtp_connection_detail
{
  AP_UINT16       overlay_size;          /* size of this entry               */
  unsigned char   rtp_name[8];           /* RTP connection name              */
  unsigned char   first_hop_ls_name[8];  /* LS name of first hop             */
  unsigned char   dest_node_name[17];    /* fully-qualified name of          */
                                         /* destination node                 */
  unsigned char   isr_boundary_fn;
  unsigned char   connection_type;       /* LU-LU or CP-CP connection?       */
  unsigned char   reserv1;               /* reserved                         */
  unsigned char   cos_name[8];           /* class-of-service name            */
  AP_UINT16       max_btu_size;          /* max btu size                     */
  AP_UINT32       liveness_timer;        /* liveness timer                   */
  unsigned char   local_tcid[8];         /* local tcid                       */
  unsigned char   remote_tcid[8];        /* remote tcid                      */
  RTP_STATISTICS  rtp_stats;             /* RTP statistics                   */
  AP_UINT16       num_sess_active;       /* number of active sessions        */
  unsigned char   arb_mode;              /* ARB-S or ARB-R?                  */
  unsigned char   reserv2[15];           /* reserved                         */
  AP_UINT16       rscv_len;              /* length of appended RSCV          */
} RTP_CONNECTION_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:    DEFINE_RTP_TUNING                                           */
/*                                                                           */
/* Description:  Defines values to modify the default RTP settings, to       */
/*               change the delay between path switch attempts               */
/*****************************************************************************/
typedef struct define_rtp_tuning
{
  /***************************************************************************/
  /* Standard VCB header.                                                    */
  /***************************************************************************/
  AP_UINT16        opcode;                  /* verb operation code           */
  unsigned char    reserv2;                 /* reserved                      */
  unsigned char    format;                  /* format                        */
  AP_UINT16        primary_rc;              /* Primary return code           */
  AP_UINT32        secondary_rc;            /* Secondary return code         */
  unsigned char    path_switch_attempts;    /* number of path switch attempts*/
  unsigned char    short_req_retry_limit;   /* short request retry limit     */
  AP_UINT16        path_switch_times[4];    /* Path switch timeouts          */
  AP_UINT32        refifo_cap;              /* cap for refifo timer          */
  AP_UINT32        srt_cap;                 /* cap for short request timer   */
  unsigned char    reserved[80];            /* reserved                      */
} DEFINE_RTP_TUNING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:    QUERY_RTP_TUNING                                            */
/*                                                                           */
/* Description:  Queries the RTP_TUNING parameters, which can be used to     */
/*               change the delay between path switch attempts               */
/*****************************************************************************/
typedef struct query_rtp_tuning
{
  /***************************************************************************/
  /* Standard VCB header.                                                    */
  /***************************************************************************/
  AP_UINT16        opcode;                  /* verb operation code           */
  unsigned char    reserv2;                 /* reserved                      */
  unsigned char    format;                  /* format                        */
  AP_UINT16        primary_rc;              /* Primary return code           */
  AP_UINT32        secondary_rc;            /* Secondary return code         */
  unsigned char    path_switch_attempts;    /* number of path switch attempts*/
  unsigned char    short_req_retry_limit;   /* short request retry limit     */
  AP_UINT16        path_switch_times[4];    /* path switch timeouts          */
  AP_UINT32        refifo_cap;              /* cap for refifo timer          */
  AP_UINT32        srt_cap;                 /* cap for short request timer   */
  unsigned char    reserved[80];            /* reserved                      */
} QUERY_RTP_TUNING;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: PATH_SWITCH                                                    */
/*                                                                           */
/* Description: NOF verb requesting the optimal path is recalculated for     */
/*              a given RTP connection                                       */
/*****************************************************************************/

typedef struct path_switch
{
  AP_UINT16       opcode;                 /* verb operation code             */
  unsigned char   reserv2;                /* reserved                        */
  unsigned char   format;                 /* format                          */
  AP_UINT16       primary_rc;             /* primary return code             */
  AP_UINT32       secondary_rc;           /* secondary return code           */
  unsigned char   rtp_connection_name[8]; /* RTP connection name             */
} PATH_SWITCH;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TP_LOAD_INFO_DEF_DATA                                          */
/*                                                                           */
/* Description: Load Information about a TP                                  */
/*****************************************************************************/
typedef struct tp_load_info_def_data
{
  unsigned char   description[32];      /* Description                       */
  unsigned char   reserv1[16];          /* reserved                          */
  unsigned char   user_id[64];          /* User ID                           */
  unsigned char   group_id[64];         /* Group ID                          */
  AP_INT32        timeout;              /* Timeout value                     */
  unsigned char   type;                 /* TP type                           */
  unsigned char   style;                /* TP type                           */
  AP_UINT16       ltv_length;           /* Length of LTV data                */

} TP_LOAD_INFO_DEF_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TP_LOAD_INFO                                            */
/*                                                                           */
/* Description: NOF Verb to define the Load Information about a TP           */
/*****************************************************************************/
typedef struct define_tp_load_info
{
  AP_UINT16             opcode;         /* verb operation code               */
  unsigned char         reserv2;        /* reserved                          */
  unsigned char         format;         /* reserved                          */
  AP_UINT16             primary_rc;     /* primary return code               */
  AP_UINT32             secondary_rc;   /* secondary return code             */
  unsigned char         tp_name[64];    /* TP name                           */
  unsigned char         lu_alias[8];    /* LU alias                          */
  TP_LOAD_INFO_DEF_DATA def_data;       /* defined data                      */

} DEFINE_TP_LOAD_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_TP_LOAD_INFO                                            */
/*                                                                           */
/* Description: NOF verb to delete the Load Information about a TP           */
/*****************************************************************************/
typedef struct delete_tp_load_info
{
  AP_UINT16       opcode;               /* verb operation code               */
  unsigned char   reserv2;              /* reserved                          */
  unsigned char   format;               /* reserved                          */
  AP_UINT16       primary_rc;           /* primary return code               */
  AP_UINT32       secondary_rc;         /* secondary return code             */
  unsigned char   tp_name[64];          /* TP name                           */
  unsigned char   lu_alias[8];          /* LU alias                          */

} DELETE_TP_LOAD_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TP_LOAD_INFO                                                   */
/*                                                                           */
/* Description: Information returned on the QUERY_TP_LOAD_INFO verb          */
/*****************************************************************************/
typedef struct tp_load_info
{
  AP_UINT16             overlay_size;   /* overlay size                      */
  unsigned char         tp_name[64];    /* TP name                           */
  unsigned char         lu_alias[8];    /* LU alias                          */
  TP_LOAD_INFO_DEF_DATA def_data;       /* defined dada                      */

} TP_LOAD_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TP_LOAD_INFO                                             */
/*                                                                           */
/* Description: NOF verb to query the configured Load Information about a TP */
/*****************************************************************************/
typedef struct query_tp_load_info
{
  AP_UINT16        opcode;              /* verb operation code               */
  unsigned char    reserv2;             /* reserved                          */
  unsigned char    format;              /* reserved                          */
  AP_UINT16        primary_rc;          /* primary return code               */
  AP_UINT32        secondary_rc;        /* secondary return code             */
  unsigned char   *buf_ptr;             /* pointer to buffer                 */
  AP_UINT32        buf_size;            /* buffer size                       */
  AP_UINT32        total_buf_size;      /* total buffer size required        */
  AP_UINT16        num_entries;         /* number of entries                 */
  AP_UINT16        total_num_entries;   /* total number of entries           */
  unsigned char    list_options;        /* listing options                   */
  unsigned char    reserv3[3];          /* reserved                          */
  unsigned char    tp_name[64];         /* TP name                           */
  unsigned char    lu_alias[8];         /* LU alias                          */

} QUERY_TP_LOAD_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_TRUSTED_GROUPS                                          */
/*                                                                           */
/* Description: Verb to define trusted groups - note that this structure is  */
/*              currently exactly the same as the query_trusted_groups       */
/*              structure                                                    */
/*****************************************************************************/
typedef struct define_trusted_groups
{
  AP_UINT16       opcode;                 /* verb operation code             */
  unsigned char   reserv2;                /* reserved                        */
  unsigned char   format;                 /* reserved                        */
  AP_UINT16       primary_rc;             /* primary return code             */
  AP_UINT32       secondary_rc;           /* secondary return code           */
  unsigned char   num_groups;             /* number of groups specified      */
  unsigned char   reserv3[3];             /* reserved                        */
  AP_INT32        group_ids[10];          /* array of AIX group ids          */
} DEFINE_TRUSTED_GROUPS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_TRUSTED_GROUPS                                           */
/*                                                                           */
/* Description: Verb to query trusted groups - note that this structure is   */
/*              currently exactly the same as the define_trusted_groups      */
/*              structure                                                    */
/*****************************************************************************/
typedef struct query_trusted_groups
{
  AP_UINT16       opcode;                 /* verb operation code             */
  unsigned char   reserv2;                /* reserved                        */
  unsigned char   format;                 /* reserved                        */
  AP_UINT16       primary_rc;             /* primary return code             */
  AP_UINT32       secondary_rc;           /* secondary rturn code            */
  unsigned char   num_groups;             /* number of trusted groups        */
  unsigned char   reserv3[3];             /* reserved                        */
  AP_INT32        group_ids[10];          /* array of trusted AIX groups     */
} QUERY_TRUSTED_GROUPS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SECURITY_USER_DATA                                             */
/*                                                                           */
/* Description: user data entry in security list                             */
/*****************************************************************************/
typedef struct security_user_data
{
  AP_UINT16         sub_overlay_size;     /* reserved                        */
  unsigned char     user_name[10];        /* user name                       */
} SECURITY_USER_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SECURITY_LIST_DEF                                              */
/*                                                                           */
/* Description: Security list definition                                     */
/*****************************************************************************/
typedef struct security_list_def
{
  unsigned char     description[32];      /* description                     */
  unsigned char     reserv3[16];          /* reserved                        */
  AP_UINT32         num_users;            /* number of users in list         */
  unsigned char     reserv2[16];          /* reserved                        */
} SECURITY_LIST_DEF;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_SECURITY_ACCESS_LIST                                    */
/*                                                                           */
/* Description: Verb to define a security access list                        */
/*****************************************************************************/
typedef struct define_security_access_list
{
  AP_UINT16         opcode;               /* verb operation code             */
  unsigned char     reserv2;              /* reserved                        */
  unsigned char     format;               /* reserved                        */
  AP_UINT16         primary_rc;           /* primary return code             */
  AP_UINT32         secondary_rc;         /* secondary return code           */
  unsigned char     list_name[14];        /* name of this list               */
  unsigned char     reserv3[2];           /* reserved                        */
  SECURITY_LIST_DEF def_data;             /* list itself                     */
} DEFINE_SECURITY_ACCESS_LIST;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SECURITY_ACCESS_DETAIL                                         */
/*                                                                           */
/* Description: Returned information applicable to whole security access list*/
/*****************************************************************************/
typedef struct security_access_detail
{
  AP_UINT16         overlay_size;         /* size of returned entry          */
  unsigned char     list_name[14];        /* list name                       */
  unsigned char     reserv1[2];           /* reserved                        */
  AP_UINT32         num_filtered_users;   /* number of users returned        */
  SECURITY_LIST_DEF def_data;             /* list definition                 */
} SECURITY_ACCESS_DETAIL;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_SECURITY_ACCESS_LISTT                                    */
/*                                                                           */
/* Description: Verb to query a security access list                         */
/*****************************************************************************/
typedef struct query_security_access_list
{
  AP_UINT16         opcode;                 /* verb operation code           */
  unsigned char     reserv2;                /* reserved                      */
  unsigned char     format;                 /* reserved                      */
  AP_UINT16         primary_rc;             /* primary return code           */
  AP_UINT32         secondary_rc;           /* secondary return code         */
  unsigned char     *buf_ptr;               /* pointer to buffer             */
  AP_UINT32         buf_size;               /* buffer size                   */
  AP_UINT32         total_buf_size;         /* total buffer size required    */
  AP_UINT16         num_entries;            /* number of entries             */
  AP_UINT16         total_num_entries;      /* total number of entries       */
  unsigned char     list_options;           /* listing options               */
  unsigned char     reserv3;                /* reserved                      */
  unsigned char     list_name[14];          /* Security Access List name     */
  unsigned char     user_name[10];          /* user name                     */
  AP_UINT32         num_init_users;         /* number of users for first     */
                                            /* list when starting in middle  */
  AP_UINT32         num_last_users;         /* number of users on last       */
                                            /* overlay if last list is       */
                                            /* incomplete                    */
  unsigned char     last_list_incomplete;   /* set to AP_YES if user data    */
                                            /* for last list is incomplete   */
} QUERY_SECURITY_ACCESS_LIST;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SECURITY_USER_NAME                                             */
/*                                                                           */
/* Description: User name definition                                         */
/*****************************************************************************/
typedef struct security_user_name
{
  unsigned char     user_name[10];        /* user name to delete             */
} SECURITY_USER_NAME;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_SECURITY_ACCESS_LIST                                    */
/*                                                                           */
/* Description: Verb to delete a security access list                        */
/*****************************************************************************/
typedef struct delete_security_access_list
{
  AP_UINT16         opcode;                 /* verb operation code           */
  unsigned char     reserv2;                /* reserved                      */
  unsigned char     format;                 /* reserved                      */
  AP_UINT16         primary_rc;             /* primary return code           */
  AP_UINT32         secondary_rc;           /* secondary return code         */
  unsigned char     list_name[14];          /* name of this list             */
  unsigned char     reserv3[2];             /* reserved                      */
  AP_UINT32         num_users;              /* number of users to delete     */
} DELETE_SECURITY_ACCESS_LIST;

/**STRUCT-********************************************************************/

/*****************************************************************************/
/* AnyNet verbs                                                              */
/*****************************************************************************/

/**STRUCT+********************************************************************/
/* Structure: IP_ADDRESS                                                     */
/*                                                                           */
/* Description: IP_ADDRESS structure used by AnyNet verbs                    */
/*****************************************************************************/
typedef struct ip_address
{
  unsigned char     family;              /* IP address family                */
  unsigned char     address[16];         /* IP address                       */
  unsigned char     reserv1[7];          /* reserved                         */
} IP_ADDRESS;
/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ANYNET_APPCIP_DEFAULTS                                         */
/*                                                                           */
/* Description: Defaults for use by ANYNET_APPCIP_DEFAULTS verbs             */
/*****************************************************************************/
typedef struct anynet_appcip_defaults
{
  unsigned char     default_routing;     /* Default routing preference       */
  unsigned char     domain_name[238];    /* Domain suffix added to LU name   */
  AP_UINT32         connection_retry;    /* Maximum setup time for MPTN conn */
  AP_UINT32         connection_wait;     /* Maximum conn wait time           */
  AP_UINT16         port_number;         /* IP port used                     */
  AP_UINT32         unacked_retry_time;  /* Retry time for unacked datagram  */
  AP_UINT32         unsent_retry_time;   /* Retry time for unsent data       */
  AP_UINT32         inactivity_sweep;    /* Inactivity sweep time            */
  unsigned char     description[32];     /* Description                      */
  unsigned char     reserved[16];        /* reserved                         */
  unsigned char     reserv2[32];         /* reserved                         */
} ANYNET_APPCIP_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DEFINE_ANYNET_APPCIP_DEFAULTS                                  */
/*                                                                           */
/* Description: Verb used to define the defaults for APPC over IP            */
/*****************************************************************************/
typedef struct define_anynet_appcip_defaults
{
  AP_UINT16         opcode;              /* verb operation code              */
  unsigned char     reserv2;             /* reserved                         */
  unsigned char     format;              /* format                           */
  AP_UINT16         primary_rc;          /* primary return code              */
  AP_UINT32         secondary_rc;        /* secondary return code            */
  ANYNET_APPCIP_DEFAULTS
                    appcip_defaults;     /* defaults for APPC over IP        */
} DEFINE_ANYNET_APPCIP_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: DELETE_ANYNET_APPCIP_DEFAULTS                                  */
/*                                                                           */
/* Description: Verb used to delete the default values for APPC over IP,     */
/*              effectively removing support for APPCI/IP from SNAP-IX       */
/*****************************************************************************/
typedef struct delete_anynet_appcip_defaults
{
  AP_UINT16         opcode;              /* verb operation code              */
  unsigned char     reserv2;             /* reserved                         */
  unsigned char     format;              /* format                           */
  AP_UINT16         primary_rc;          /* primary return code              */
  AP_UINT32         secondary_rc;        /* secondary return code            */
} DELETE_ANYNET_APPCIP_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_ANYNET_APPCIP_DEFAULTS                                   */
/*                                                                           */
/* Description: Verb used to query the defaults for APPC over IP             */
/*****************************************************************************/
typedef struct query_anynet_appcip_defaults
{
  AP_UINT16         opcode;              /* verb operation code              */
  unsigned char     reserv2;             /* reserved                         */
  unsigned char     format;              /* format                           */
  AP_UINT16         primary_rc;          /* primary return code              */
  AP_UINT32         secondary_rc;        /* secondary return code            */
  unsigned char     status;              /* Is APPC/IP active?               */
  unsigned char     reserv3[31];         /* reserved                         */
  ANYNET_APPCIP_DEFAULTS
                    appcip_defaults;     /* Information for APPC over IP     */
} QUERY_ANYNET_APPCIP_DEFAULTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ANYNET_APPCIP_DATA                                             */
/*                                                                           */
/* Description: Overlay for use by the QUERY_ANYNET_APPCIP verb              */
/*****************************************************************************/
typedef struct anynet_appcip_data
{
  AP_UINT16         overlay_size;        /* size of returned overlay         */
  AP_UINT32         anynet_id;           /* anynet ID                        */
  unsigned char     gateway;             /* is connection a gateway?         */
  unsigned char     fqlu_name[17];       /* local LU name                    */
  unsigned char     fqplu_name[17];      /* partner LU name                  */
  IP_ADDRESS        local_ip_address;    /* local IP address                 */
  AP_UINT16         local_port_number;   /* local IP port number             */
  IP_ADDRESS        remote_ip_address;   /* remote IP address                */
  AP_UINT16         remote_port_number;  /* remote IP port number            */
  unsigned char     state;               /* state of session                 */
  unsigned char     pcid[16];            /* procedure correlation ID         */
  unsigned char     correlator[10];      /* anynet correlator                */
} ANYNET_APPCIP_DATA;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: QUERY_ANYNET_APPCIP                                            */
/*                                                                           */
/* Description: Verb used to Query the status of APPC over IP                */
/*****************************************************************************/
typedef struct query_anynet_appcip
{
  AP_UINT16         opcode;              /* verb operation code              */
  unsigned char     reserv2;             /* reserved                         */
  unsigned char     format;              /* format                           */
  AP_UINT16         primary_rc;          /* primary return code              */
  AP_UINT32         secondary_rc;        /* secondary return code            */
  unsigned char    *buf_ptr;             /* pointer to supplied buffer       */
  AP_UINT32         buf_size;            /* size of supplied buffer          */
  AP_UINT32         total_buf_size;      /* required size of buffer          */
  AP_UINT16         num_entries;         /* number of entries required       */
  AP_UINT16         total_num_entries;   /* total number of entries          */
  unsigned char     list_options;        /* listing options                  */
  AP_UINT32         anynet_id;           /* anynet ID                        */
  unsigned char     anynet_type;         /* type of anynet connection filter */
  unsigned char     fqlu_name[17];       /* local LU name filter             */
  unsigned char     fqplu_name[17];      /* partner LU filter                */
  IP_ADDRESS        local_ip_address;    /* local IP address filter          */
  IP_ADDRESS        remote_ip_address;   /* remote IP address filter         */
  unsigned char     reserv3[64];         /* reserved                         */
} QUERY_ANYNET_APPCIP;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: ANYNET_INDICATION                                              */
/*                                                                           */
/* Description: An indication sent by AnyNet upon the starting or stopping   */
/*              of an AnyNet connection                                      */
/*****************************************************************************/
typedef struct anynet_indication
{
  AP_UINT16         opcode;              /* verb operation code              */
  unsigned char     reserv2;             /* reserved                         */
  unsigned char     format;              /* format                           */
  AP_UINT16         primary_rc;          /* primary return code              */
  AP_UINT32         secondary_rc;        /* secondary return code            */
  unsigned char     data_lost;           /* Previous indication lost?        */
  unsigned char     connection_state;    /* State of connection              */
  unsigned char     connection_type;     /* APPC/IP or Snackets              */
  AP_UINT32         anynet_id;           /* anynet identification number     */
  unsigned char     gateway;             /* is connection a gateway?         */
  unsigned char     fqlu_name[17];       /* local LU name                    */
  unsigned char     fqplu_name[17];      /* partner LU name                  */
  IP_ADDRESS        local_ip_address;    /* local IP address                 */
  AP_UINT16         local_port_number;   /* local IP port number             */
  IP_ADDRESS        remote_ip_address;   /* remote IP address                */
  AP_UINT16         remote_port_number;  /* remote IP port number            */
  unsigned char     mode_name[8];        /* mode name                        */
  unsigned char     correlator[10];      /* anynet correlator                */
} ANYNET_INDICATION;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: SNA_IP_ADDR                                                    */
/*                                                                           */
/* Description: Used to store a protocol independent IP address (v4 or v6)   */
/*****************************************************************************/
typedef struct sna_ip_addr
{
  AP_UINT16        family; /* AF_INET or AF_INET6 */
  union
  {
    unsigned char  ipv4_addr[4];
    unsigned char  ipv6_addr[16];
  } ip_addr;
} SNA_IP_ADDR;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: TN_REDIRECTION_INDICATION                                      */
/*                                                                           */
/* Description: Used to indicate information about an IP redirection         */
/*            connection.                                                    */
/*****************************************************************************/

typedef struct tn_redirection_indication
{
   AP_UINT16        opcode;                  /* verb operation code          */
   unsigned char    reserv2;                 /* reserved                     */
   unsigned char    format;                  /* reserved                     */
   AP_UINT16        primary_rc;              /* primary return code          */
   AP_UINT32        secondary_rc;            /* secondary return code        */
   unsigned char    data_lost;               /* previous indication lost     */
   unsigned char    reason;                  /* reason for indication        */
   SNA_IP_ADDR      client_ip_addr;          /* client IP address            */
   AP_UINT16        client_port;             /* client port number           */
   SNA_IP_ADDR      host_ip_addr;            /* host IP address              */
   AP_UINT16        host_port;               /* host port number             */
   AP_UINT16        client_number;           /* client number                */
   unsigned char    listen_local_address[46]; /* Local addr cli connects to  */
   unsigned char    reserva[16];             /* reserved                     */
} TN_REDIRECTION_INDICATION;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: RAPI_CLIENT_INDICATION                                         */
/*                                                                           */
/* Description: Used to indicate information about an Remote API Client      */
/*              connecting or disconnecting                                  */
/*****************************************************************************/

typedef struct rapi_client_indication
{
   AP_UINT16        opcode;                  /* verb operation code          */
   unsigned char    reserv2;                 /* reserved                     */
   unsigned char    format;                  /* reserved                     */
   AP_UINT16        primary_rc;              /* primary return code          */
   AP_UINT32        secondary_rc;            /* secondary return code        */
   unsigned char    data_lost;               /* previous indication lost     */
   unsigned char    reason;                  /* reason for indication        */
   unsigned char    sys_name[128];           /* system name client sends us  */
   SNA_IP_ADDR      rapi_client_origin_ip_addr; /* IP addr client sends us   */
   SNA_IP_ADDR      rapi_client_adj_ip_addr; /* IP addr client comes in on   */
   AP_UINT16        rapi_client_adj_port;    /* port IP client comes in on   */
   unsigned char    reserva[16];             /* reserved                     */
} RAPI_CLIENT_INDICATION;

/**STRUCT-********************************************************************/


/**STRUCT+********************************************************************/
/* Structure: QUERY_RAPI_CLIENTS                                             */
/*                                                                           */
/* Description: Query the remote API Clients                                 */
/*****************************************************************************/

typedef struct query_rapi_clients
{
  AP_UINT16              opcode;           /* verb operation code            */
  unsigned char          reserv2;
  unsigned char          format;
  AP_UINT16              primary_rc;       /* primary return code            */
  AP_UINT32              secondary_rc;     /* secondary return code          */
  unsigned char         *buf_ptr;          /* pointer to buffer              */
  AP_UINT32              buf_size;         /* buffer size                    */
  AP_UINT32              total_buf_size;   /* total buffer size required     */
  AP_UINT16              num_entries;      /* number of entries              */
  AP_UINT16              total_num_entries;/* total number of entries        */
  unsigned char          list_options;     /* listing options                */
  AP_UINT16              max_clients;      /* maximum number of clients      */
  unsigned char          sys_name[128];    /* RAPI Client to start query     */
} QUERY_RAPI_CLIENTS;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure: RAPI_CLIENT_INFO                                               */
/*                                                                           */
/* Description: Summary overlay used for QUERY_RAPI_CLIENTS above)           */
/*****************************************************************************/

typedef struct rapi_client_info
{
  AP_UINT16        overlay_size;            /* overlay size                  */
  unsigned char    sys_name[128];           /* RAPI Client System name       */
  SNA_IP_ADDR      rapi_client_origin_ip_addr; /* IP addr client sends us    */
  SNA_IP_ADDR      rapi_client_adj_ip_addr; /* IP addr client comes in on    */
  AP_UINT16        rapi_client_adj_port;    /* port IP client comes in on    */
} RAPI_CLIENT_INFO;

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   DLC_SPEC_DATA                                                */
/*                                                                           */
/* Description: Contains data related to a specific DLC type                 */
/*****************************************************************************/

typedef struct dlc_spec_data
{

  union
  {
    SDL_SPEC_DATA       sdlc;
    VMC_DLC_CFG         llc;
    VQL_DLC_SPEC_DATA   qllc;
    GDLC_DLC_CFG        gdlc;
    CHNL_DLC_SPEC_DATA  chnl;
    IPDLC_DLC_SPEC_DATA ip;
    VDL_DLC_CFG         ndg;
    unsigned char       generic[16];
  } dlc_data;

} DLC_SPEC_DATA;
#define SIZEOF_DLC_SPEC_DATA (sizeof(DLC_SPEC_DATA))

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   PORT_SPEC_DATA                                               */
/*                                                                           */
/* Description: Contains data related to a specific Port                     */
/*****************************************************************************/

typedef struct port_spec_data
{
  union
  {
    SDL_PORT_SPEC_DATA       sdlc;
    LLC_PORT_SPEC_DATA       llc;
    VQL_PORT_SPEC_DATA       qllc;
    GDLC_SAP_CFG             gdlc;
    CHNL_PORT_SPEC_DATA      chnl;
    IPDLC_PORT_SPEC_DATA     ip;
    VDL_SAP_CFG              ndg;
    unsigned char            generic[256];
  } port_data;
} PORT_SPEC_DATA;
#define SIZEOF_PORT_SPEC_DATA (sizeof(PORT_SPEC_DATA))

/**STRUCT-********************************************************************/

/**STRUCT+********************************************************************/
/* Structure:   LINK_SPEC_DATA                                               */
/*                                                                           */
/* Description: Consists of information specific to this link station -      */
/*              format is specific to DLC type                               */
/*****************************************************************************/

typedef struct link_spec_data
{
  union
  {
    SDL_LINK_SPEC_DATA       sdlc;
    LLC_LINK_SPEC_DATA       llc;
    VQL_LS_SPEC_DATA         qllc;
    GDLC_LS_CFG              gdlc;
    CHNL_LINK_SPEC_DATA      chnl;
    IPDLC_LINK_SPEC_DATA     ip;
    VDL_LS_CFG               ndg;
    unsigned char            generic[256];
  } link_data;
} LINK_SPEC_DATA;
#define SIZEOF_LINK_SPEC_DATA (sizeof(LINK_SPEC_DATA))

/**STRUCT-********************************************************************/

#ifdef __cplusplus

}

#endif

#endif

