/********************************************************/
/*                                                      */
/* RSERVERH.CMD - IBM REXX Sample Program               */
/*                                                      */
/* Server handler for remote control server. Commands   */
/* from client are executed as long as QUIT is reached. */
/*                                                      */
/* Parameters:                                          */
/*   Socket: Socket for client communication            */
/*                                                      */
/* ---------------------------------------------------- */
/* (C) Copyright IBM Corp. 1996 - All Rights Reserved.  */
/*                                                      */
/* DISCLAIMER OF WARRANTIES.  The following [enclosed]  */
/* code is sample code created by IBM Corporation. This */
/* sample code is not part of any standard or IBM       */
/* product and is provided to you solely for the        */
/* purpose of assisting you in the development of your  */
/* applications.  The code is provided "AS IS", without */
/* warranty of any kind.  IBM shall not be liable for   */
/* any damages arising out of your use of the sample    */
/* code,  even if they have been advised of the         */
/* possibility of such damages.                         */
/*                                                      */
/********************************************************/
Parse Arg Socket

/* Load REXX Socket library if not already loaded       */
If RxFuncQuery("SockLoadFuncs") Then
 Do
   Call RxFuncAdd "SockLoadFuncs","RXSOCK","SockLoadFuncs"
   Call SockLoadFuncs
 End

/* Show some information about connected client         */
Call SockGetPeerName Socket, "ClientAddr.!"
Call SockGetHostByAddr ClientAddr.!addr, "ClientAddr.!"
Say "Established connection with client '" ||,
    ClientAddr.!name || "'."

/* Process commands until QUIT is reached               */
Do Until Command = "QUIT"
  Command = ReceiveRequest(Socket)
End

/* Close socket and OS/2 session                        */
Call Close Socket
"Exit"


/********************************************************/
/*                                                      */
/* Function:  ReceiveRequest                            */
/* Purpose:   Wait for a command from the client and    */
/*            execute it. Return the identifier of the  */
/*            command to the caller.                    */
/* Arguments: Socket - active socket number             */
/* Returns:   command identifier                        */
/*                                                      */
/********************************************************/
ReceiveRequest: Procedure
  Parse Arg Socket

  /* Wait for the command from the client               */
  BytesRcvd = SockRecv(Socket, "CommandLine", 1024)
  Say "Command line from client:" CommandLine

  Parse Var CommandLine Command Option
  Command = Translate(Command)

  Select
    When Command = "DIR" Then
      Call ProcessDirCommand Socket, Option
    When Command = "TYPE" Then
      Call ProcessTypeCommand Socket, Option
    When Command = "QUIT" Then
      Nop
    Otherwise
      Call Answer Socket, "Invalid command."
  End

  /* send end of answer marker back to client           */
  Call SockSend Socket, ">>>End_of_transmission<<<"

  Return Command


/********************************************************/
/*                                                      */
/* Procedure: Close                                     */
/* Purpose:   Close the specified socket.               */
/* Arguments: Socket - active socket number             */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
Close: Procedure
  Parse Arg Socket

  Call SockShutDown Socket, 2
  Call SockClose Socket
  Return


/********************************************************/
/*                                                      */
/* Procedure: Answer                                    */
/* Purpose:   Send one answer line back to the client   */
/*            and wait for acknowledgement from client. */
/* Arguments: Socket - active socket number             */
/*            AnswerString - line to send to client     */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
Answer: Procedure
  Parse Arg Socket, AnswerString

  Call SockSend Socket, AnswerString
  Call SockRecv Socket, "Ack", 256
  Return


/********************************************************/
/*                                                      */
/* Procedure: AnswerQueue                               */
/* Purpose:   Send all lines from the session queue     */
/*            back to the client as the answer of the   */
/*            previous executed command.                */
/* Arguments: Socket - active socket number             */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
AnswerQueue: Procedure
  Parse Arg Socket

  /* send answer lines until session queue is empty     */
  Do While Queued() > 0
    Parse Pull Line

    /* empty lines will be sent as a space              */
    If Line = "" Then
      Line = " "

    Call Answer Socket, Line
  End

  Return


/********************************************************/
/*                                                      */
/* Procedure: ProcessDirCommand                         */
/* Purpose:   Process the DIR command that was received */
/*            from the client and send back the result. */
/* Arguments: Socket - active socket number             */
/*            FileMask - optional file mask for DIR     */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
ProcessDirCommand: Procedure
  Parse Arg Socket, FileMask

  /* redirect output from DIR command to session queue  */
  "DIR" FileMask " | RXQUEUE"
  Call AnswerQueue Socket
  Return


/********************************************************/
/*                                                      */
/* Procedure: ProcessTypeCommand                        */
/* Purpose:   Process the TYPE command that was received*/
/*            from the client and send back the result. */
/* Arguments: Socket - active socket number             */
/*            FileName - required filename to type      */
/* Returns:   nothing                                   */
/*                                                      */
/********************************************************/
ProcessTypeCommand: Procedure
  Parse Arg Socket, FileName

  /* TYPE needs a filename as argument                  */
  If FileName = "" Then
   Do
     Call Answer Socket, "You have to specify a filename",
          " with the TYPE command."
     Return
   End

  /* Check if the file really exists                    */
  If Stream(FileName, "C", "QUERY EXISTS") = "" Then
   Do
     Call Answer Socket, "The specified file '" ||,
          FileName || "' does not exist."
     Return
   End

  /* redirect output from TYPE command to session queue */
  "TYPE" FileName " | RXQUEUE"
  Call AnswerQueue Socket
  Return
