/*REXX*****************************************************************
 *
 * (C) Copyright IBM Corp. 2000 - All Rights Reserved.
 *
 * DISCLAIMER OF WARRANTIES.  The following [enclosed] code is sample
 * code created by IBM Corporation. This sample code is not part of
 * any standard or IBM product and is provided to you solely for the
 * purpose of assisting you in the development of your applications.
 * The code is provided "AS IS", without warranty of any kind.
 * IBM shall not be liable for any damages arising out of your use of
 * the sample code,  even if they have been advised of the possibility
 * of such damages.
 *
 *______________________________________________________________about__
 *
 *  purpose:
 *    - build JCL batch job from a given makefile that compiles and
 *      links a main module and its external functions into one single
 *      big load module.
 *      This improves performance not only because of optimizations
 *      introduced by the compiler, it saves also a lot each time an
 *      external function is called during runtime.
 *    - write generated job to a dataset.
 *    - on 'SUB' option submit the generated job.
 *    - on 'EDIT' option open editor loaded with the generated job.
 *
 *  systems : mvs, aix, linux, win32, os/2
 *
 *  syntax  : RXMAKE makeFileName [SUB|EDIT]
 *
 *  requirements:
 *    - valid makefile (name is the 1.st command line parameter)
 *
 *  limitations:
 *    - names of modules must not differ only in their eighth
 *      character (ie. their first 7 characters must definitely
 *      be matched by only one single modules name)
 *    - EDIT option only available under TSO
 *    - SUB  option only available under TSO and MVS
 *    - as makefile nearly every valid REXX program that sets the
 *      required attributes is thinkable - BUT line continuations
 *      with commas or omitting the end-of-statement semicolon
 *      will lead to an error situation.
 *
 *  known problems:
 *    - none
 *
 *  author            : Thorsten Schaper, IBM
 *                      thorsten.schaper@de.ibm.com
 *  created           : 06/03/2000
 *  last modfication : 07/05/2000
 *
 *  change history:
 *    mm/dd/yyyy ididid
 *      descriptive text
 */

/*_____________________________________________________code_structure__
 *
 *  MAIN
 *    main
 *
 *  MAKE UTILITY FUNCTIONS
 *    readMakeFile
 *    writeJobFile
 *    submitJob
 *    editJob
 *
 *  REXX UTILITY FUNCTIONS
 *    readFile
 *    writeFile
 *    stemCopy
 *    countChar
 *
 *  JCL CARDS BUILDING
 *    addStmt
 *    addComm
 *    buildJOBcard
 *    buildJOBLIBcard
 *    buildAllocateStep
 *    buildCopyStep
 *    buildCompilerCards
 *    buildMainRoutineCompileStep
 *    buildExternalFunctionsCompileSteps
 *    buildMainRoutineLinkStep
 *    buildExternalFunctionsLinkSteps
 *    buildTargetModuleLinkStep
 *
 *  SOME ERROR HANDLING
 *    saySyntax
 *    testArgs
 *    exitWithError
 *    noValue
 *    syntax
 */

/*=====================================================================
 *                                MAIN
 *=====================================================================
 */

SIGNAL ON NOVALUE
SIGNAL ON SYNTAX

parse arg makeFileName subOrEdit

/*_______________________________________________________________main__
 */
main:

  recuFuncs = 0  /* # of recursive external functions used           */
  compCalls = 0  /* # of calls to the REXX/370 Compiler  ("REXXCOMP")*/
  linkCalls = 0  /* # of calls to the Linkage Editor     ("HEWL")    */
  jclOut.0  = 0  /* # of lines in string array for job-code-output   */
  cMain     = 0  /* indicates if the main program is in C language   */

  call testArgs
  call readAndTestMakeFile

  call buildJOBcard
  call buildJOBLIBcard
  call buildAllocateStep
  call buildCopyStep
  call buildExternalFunctionsCompileSteps
  call buildExternalFunctionsLinkSteps
  call buildMainRoutineCompileStep
  call buildTargetModuleLinkStep

  call writeJobFile

  if subOrEdit = 'SUB' then
    call submitJob
  else
    if subOrEdit = 'EDIT' then
      call editJob

  exit 0

/*=====================================================================
 *                        MAKE UTILITY FUNCTIONS
 *=====================================================================
 */

/*________________________________________________readAndTestMakeFile__
 * purpose: - read the makefile
 *          - test if all required attributes are set
 *          - count number of recursive external functions
 */
readAndTestMakeFile:

  if \readFile(makeFileName) then
    exitWithError(1, "could not read the file '"makeFileName"'")

  makeFileExec = ''
  do i = 1 to fileContent.0
    mLine = strip(fileContent.i)
    if mLine = '' then
      iterate
    makeFileExec = makeFileExec || mLine
  end

  INTERPRET makeFileExec

  SIGNAL ON NOVALUE NAME requiredAttributeNotSet

  errOcc = 0

testStart:
  test = mainRoutine.ExecDSN              /* test         */
  test = mainRoutine.Language             /* the          */
  test = mainRoutine.Stub                 /* ##REQUIRED## */
                                          /* attributes.  */
  do i = 1 to extFunc.0                   /* Raise        */
    test = extFunc.i.ExecDSN              /* NOVALUE      */
    if extFunc.i.Recursive = 1 then       /* condition    */
      recuFuncs = recuFuncs + 1
  end

  test = dsn.RexxPrefix                   /* if           */
  test = dsn.C370Prefix                   /* not          */
  test = dsn.FinalMod                     /* set.         */

  test = job.OutputDSN
  test = job.User
  test = job.Name
  test = job.Class
  test = job.MsgClass

  SIGNAL ON NOVALUE name noValue

  if errOcc then
    exitWithError(2, "Set all ##REQUIRED## attributes in the makefile",
                     "and try again.")

  /* set the "ExecName" and "OrgName" attributes  for main routine
   */
  parse var mainRoutine.ExecDSN '(' mainRoutine.ExecName ')'
  if mainRoutine.ExecName = '' then
    mainRoutine.ExecName = substr(mainRoutine.ExecDSN,,
                                  lastPos('.', mainRoutine.ExecDSN)+1)
    mainRoutine.OrgName = mainRoutine.ExecName

  /* set the "ExecName" and "OrgName" attributes for external functions
   */
  do i = 1 to extFunc.0
    parse var extFunc.i.ExecDSN '(' extFunc.i.ExecName ')'
    if extFunc.i.ExecName = '' then
      extFunc.i.ExecName = substr(extFunc.i.ExecDSN,,
                                  lastPos('.', extFunc.i.ExecDSN)+1)
    extFunc.i.OrgName = extFunc.i.ExecName
  end

  return

requiredAttributeNotSet:
  say "You didn't set the ##REQUIRED## attribute named",
                                                "'"condition('d')"' !"
  call VALUE condition('d'), '**DUMMY**'
  errOcc = 1
  SIGNAL ON NOVALUE NAME requiredAttributeNotSet
  SIGNAL testStart

/*________________________________________________________writeJobFile__
 */
writeJobFile:

  call stemCopy 'fileContent.' 'jclOut.'
  call writeFile job.OutputDSN

  return

/*__________________________________________________________submitJob__
 */
submitJob:

  ADDRESS TSO "SUB '"job.OutputDSN"'"

  return

/*____________________________________________________________editJob__
 */
editJob:

  ADDRESS TSO "ISPEXEC EDIT DATASET('"job.OutputDSN"')"

  return

/*=====================================================================
 *                      REXX UTILITY FUNCTIONS
 *=====================================================================
 */

/*___________________________________________________________readFile__
 * purpose: read a file.
 * returns: 1 - ok
 *          0 - error
 */
readFile: procedure expose fileContent.
  parse arg fileName

  parse source os .
  os = translate(left(os, 3))
  select
    when os = 'TSO' then do
      ADDRESS TSO
      "ALLOCATE DSNAME('"fileName"') DDNAME(MAKEFILE) SHR REUSE"
      'EXECIO * DISKR MAKEFILE (STEM FILECONTENT. FINIS'
      ADDRESS
      return RC = 0
    end
    when os = 'WIN' then do
      streamRC = stream(fileName, 'C', 'OPEN READ SHARED')
      if abbrev(streamRC, 'READY') then do
        do i = 1 while lines(fileName) > 0
          fileContent.i = lineIn(fileName)
        end
        fileContent.0 = i - 1
        call stream fileName, 'C', 'CLOSE'
        return 1
      end
    end
    otherwise
      exitWithError(3, "Your OS '"os"' is not supported.")
  end

  return 0

/*__________________________________________________________writeFile__
 * purpose: write a file.
 * returns : 1 - ok
 *           0 - error
 */
writeFile: procedure expose fileContent.
  parse arg fileName

  parse source os .
  os = translate(left(os, 3))
  select
    when os = 'TSO' then do
      do until RC = 0
        ADDRESS TSO
        "ALLOCATE DSNAME('"fileName"') DDNAME(OUTPUT) OLD REUSE"
        if RC <> 0 then do
          say '--> try to remove the problem and press ENTER...'
          say '    ...or cancel the program using the ATTENTION key'
          parse pull .
        end
      end
      'EXECIO' fileContent.0 'DISKW OUTPUT (STEM FILECONTENT. FINIS'
      ADDRESS
      return RC = 0
    end
    when os = 'WIN' then do
      'del' fileName
      streamRC = stream(fileName, 'C', 'OPEN WRITE REPLACE')
      if abbrev(streamRC, 'READY') then do
        do i = 1 to fileContent.0
          call lineOut fileName, fileContent.i
        end
        call stream fileName, 'C', 'CLOSE'
        return 1
      end
    end
    otherwise
      exitWithError(3, "Your OS '"os"' is not supported.")
  end

  return 0

/*___________________________________________________________stemCopy__
 */
stemCopy:
  parse arg dest src

  INTERPRET,
    'do !i! = 0 to 'src'0;',
        dest'!i! = 'src'!i!;',
    'end;'

  return

/*__________________________________________________________countChar__
 */
countChar: procedure
  parse arg searchChar, string

  c = 0
  p = 0
  do until p = 0
    p = pos(searchChar, string, p + 1)
    if p > 0 then
      c = c + 1
  end

  return c

/*=====================================================================
 *                          JCL CARDS BUILDING
 *=====================================================================
 */

/*____________________________________________________________addStmt__
 * purpose: add a JCL statement (EXEC, DD, ...)
 * param:   - name: should be max. 8 characters!
 *          - statement: the JCL statement and its parameters
 */
addStmt: procedure expose jclOut.
  parse arg name, statement
  parse var statement stmt options

  c = jclOut.0                         /* get current line in output */

  jclLine = '//'left(name, 8) right(stmt ,4) options

  do while length(jclLine) > 71
    parenthesisMatch = 0
    cutPos = 72
    do until parenthesisMatch
      cutPos = lastPos(',', jclLine, cutPos - 1)
      cutStr = subStr(jclLine, 1, cutPos)
      if countChar('(', cutStr) = countChar(')', cutStr) then
        parenthesisMatch = 1
    end
    cutPos = cutPos + 1
    parse var jclLine 0 cutStr =(cutPos) jclLine
    c = c + 1
    jclOut.c = cutStr
    jclLine = '//'copies(' ',13)jclLine
  end

  if jclLine \= '' then do
    c = c + 1
    jclOut.c = jclLine
  end

  jclOut.0 = c                         /* set current line in output */

  return

/*____________________________________________________________addInln__
 * purpose: add an inline statement
 * param:   the inline statement
 */
addInln: procedure expose jclOut.
  parse arg inline

  c = jclOut.0                         /* get current line in output */

  do until inline = ''
    if length(inline) > 69 then do
      cutPos = lastPos(' ', inline, 70)
      parse var inline 0 cutInln =(cutPos) ' ' inline
      cutInln = left(cutInln, 69)'+'
    end
    else do
      cutInln = inline
      inline = ''
    end
    c = c + 1
    if cutInln = '/*' then
      jclOut.c = '/*'
    else
      jclOut.c = '  'cutInln
  end

  jclOut.0 = c                         /* set current line in output */

  return

/*____________________________________________________________addComm__
 * purpose: add an inline comment
 * param:   the comment text
 */
addComm: procedure expose jclOut.
  parse arg comment

  c = jclOut.0                         /* get current line in output */

  c = c + 1
  jclOut.c = '//*'
  c = c + 1
  jclOut.c = '//*'copies('-',68)
  do until comment = ''
    if length(comment) > 67 then
      cutPos = lastPos(' ', comment, 68)
    else
      cutPos = 68
    parse var comment 0 cutComm =(cutPos) ' ' comment
    c = c + 1
    jclOut.c = '//*' translate(cutComm)
  end
  c = c + 1
  jclOut.c = '//*'

  jclOut.0 = c                         /* set current line in output */

  return

/*_______________________________________________________buildJOBcard__
 */
buildJOBcard:

  /* positional parameters
   */
  select
    when job.AccountNumber \= '' & job.AccountingInfo \= '' then
      jobCardOptions = '('job.AccountNumber','job.AccountingInfo'),'
    when job.AccountNumber \= '' then
      jobCardOptions = '('job.AccountNumber'),'
    when job.AccountingInfo \= '' then
      jobCardOptions = '(,'job.AccountingInfo'),'
    otherwise
      jobCardOptions = ''
  end
  if job.ProgrammersName \= '' then
    jobCardOptions = jobCardOptions || "'"job.ProgrammersName"',"

  /* keyword parameters
   */
  if job.Class \= '' then
    jobCardOptions = jobCardOptions || 'CLASS='    || job.Class','
  if job.MsgClass \= '' then
    jobCardOptions = jobCardOptions || 'MSGCLASS=' || job.MsgClass','
  if job.Notify \= '' then
    jobCardOptions = jobCardOptions || 'NOTIFY='   || job.Notify','
  if job.Password \= '' then
    jobCardOptions = jobCardOptions || 'PASSWORD=' || job.Password','
  if job.Priority \= '' then
    jobCardOptions = jobCardOptions || 'PRTY='     || job.Priority','
  if job.Region \= '' then
    jobCardOptions = jobCardOptions || 'REGION='   || job.Region','
  if job.User \= '' then
    jobCardOptions = jobCardOptions || 'USER='     || job.User','

  jobCardOptions = left(jobCardOptions, length(jobCardOptions) - 1)
  call addStmt job.User || job.Name, 'JOB' jobCardOptions

  /* now the job comment (must be after jobcard)
   */
  call addComm,
    'JOB TO COMPILE AND LINK A MAIN EXEC AND SEVERAL EXTERNAL',
    'FUNCTION EXECS INTO LOAD MODULES - AND FINALLY LINK ALL THESE',
    'SEPERATE LOAD MODULES INTO ONE SINGLE BIG LOAD MODULE.'

  return

/*____________________________________________________buildJOBLIBcard__
 */
buildJOBLIBcard:

  if mainRoutine.Language = 'REXX' then
    call addComm 'JOBLIB, NECESSARY FOR FINDING THE REXX COMPILER'
  else
    call addComm 'JOBLIB, NECESSARY FOR FINDING THE REXX AND C/370',
                                                         'COMPILERS'
  call addStmt 'JOBLIB', 'DD DSN='dsn.rexxPrefix'SFANLMD,DISP=SHR'
  if mainRoutine.Language = 'C' then do
    call addStmt '', 'DD DSN='dsn.c370prefix'SCBC3CMP,DISP=SHR'
    call addStmt '', 'DD DSN=SYS1.SCEERUN,DISP=SHR'
  end

  return

/*__________________________________________________buildAllocateStep__
 */
buildAllocateStep:

  call addComm 'ALLOCATE PARTITIONED DATASETS FOR TEMPORARY DATA'
  call addStmt 'ALLOC', 'EXEC PGM=IEFBR14'

  if recuFuncs > 0 then
    call addStmt 'EXTFUNCS',,
                 'DD DSN=&&EXTFUNCS,DISP=(NEW,PASS),' ||,
                    'DCB=(LRECL=80,RECFM=FB),SPACE=(3120,(50,50,10))'

  if dsn.objects  = '' then do
    call addStmt 'OBJDECKS',,
                 'DD DSN=&&OBJDECKS,DISP=(NEW,PASS),' ||,
                    'DCB=(LRECL=80,RECFM=FB),SPACE=(3120,(50,50,10))'
    dsn.objects = '&&OBJDECKS'
  end

  if dsn.cPrelink = '' & mainRoutine.Language = 'C' then do
    call addStmt 'PLKOBJS',,
                 'DD DSN=&&PLKOBJS,DISP=(NEW,PASS),' ||,
                    'DCB=(LRECL=80,RECFM=FB),SPACE=(3120,(50,50,10))'
    dsn.cPrelink = '&&PLKOBJS'
  end

  if dsn.loadMods = '' then do
    call addStmt 'LOADMODS',,
                 'DD DSN=&&LOADMODS,DISP=(NEW,PASS),' ||,
                    'DCB=(RECFM=U),SPACE=(32200,(20,20,10))'
    dsn.loadMods = '&&LOADMODS'
  end

  return

/*______________________________________________________buildCopyStep__
 */
buildCopyStep:

  if recuFuncs = 0 then
    return

  recuMembers = 0
  recuSeqFiles = 0
  allSrcDSNames = ''
  do i = 1 to extFunc.0
    /* get the names of all involved datasets.
     */
    if pos('(', extFunc.i.ExecDSN) > 0 then do
      parse var extFunc.i.ExecDSN 0 srcDSName '('
      if wordPos(srcDSName, allSrcDSNames) = 0 then
        allSrcDSNames = allSrcDSNames srcDSName
    end
    /* count recursive members and sequential datatsets.
     */
    if extFunc.i.Recursive = 1 then do
      if pos('(', extFunc.i.ExecDSN) > 0 then
        recuMembers = recuMembers + 1
      else
        recuSeqFiles = recuSeqFiles + 1
    end
  end

  call addComm 'COPY AND RENAME RECURSIVE EXTERNAL REXX FUNCTIONS'

  if recuMembers > 0 then do
    call addStmt 'MCOPY',     'EXEC PGM=IEBCOPY'
    call addStmt 'SYSPRINT',  'DD SYSOUT=*'
    do i = 1 to words(allSrcDSNames)
      srcDSName = word(allSrcDSNames, i)
      call addStmt 'SRC'left(i, 6), 'DD DSN='srcDSName',DISP=SHR'
    end
    call addStmt 'TRG',   'DD DSN=&&EXTFUNCS,DISP=(SHR,PASS)'
    call addStmt 'SYSIN', 'DD *'
    do i = 1 to extFunc.0
      if extFunc.i.Recursive = 1 then do
        parse var extFunc.i.ExecDSN srcDSName '(' srcMemberName ')'
        targetMemberName = strip(left('@'srcMemberName,8))
        if srcMemberName \= '' then do
          inDDnr = wordPos(srcDSName, allSrcDSNames)
          call addInln 'COPY     OUTDD=TRG,INDD=SRC'inDDnr
          call addInln 'SELECT   MEMBER=(('srcMemberName',' ||,
                                           targetMemberName'))'
          extFunc.i.ExecDSN = '&&EXTFUNCS('targetMemberName')'
          extFunc.i.ExecName= targetMemberName
        end
      end
    end
    call addInln '/*'
  end

  if recuSeqFiles > 0 then do
    c = 0
    do i = 1 to extFunc.0
      if extFunc.i.Recursive = 1 then do
        if pos('(', extFunc.i.ExecDSN) = 0 then do
          targetMemberName = strip(left('@'extFunc.i.ExecName, 8))
          c = c + 1
          call addStmt 'SCOPY'left(c, 2), 'EXEC PGM=IEBGENER'
          call addStmt 'SYSPRINT','DD SYSOUT=*'
          call addStmt 'SYSUT1',  'DD DSN='extFunc.i.ExecDSN',DISP=SHR'
          call addStmt 'SYSUT2',  'DD DSN=&&EXTFUNCS(' ||,
                                    targetMemberName'),DISP=(SHR,PASS)'
          call addStmt 'SYSIN',   'DD DUMMY'

          extFunc.i.ExecDSN = '&&EXTFUNCS('targetMemberName')'
          extFunc.i.ExecName= targetMemberName
        end
      end
    end
  end

  return

/*_________________________________________________buildRexxCompCards__
 */
buildRexxCompCards:
  parse arg stemPrefix

  compCalls = compCalls + 1

  /* check if DLINK compiler option is needed:
   * - for each main unit, if there are any external functions.
   * - for each ext. fu. that wants to call another statically bound
   *   external function. To keep it simple DLINK is used for ALL ext.
   *   functions - it doesn't matter, if they don't make any external
   *   calls...
   */
  if (abbrev(stemPrefix, 'mainRoutine') & extFunc.0 > 0) |,
     (abbrev(stemPrefix, 'extFunc'))   then
    parms = "'NOCEXEC OBJECT DLINK'"
  else
    parms = "'NOCEXEC OBJECT'"

  /* build complete compile options with optional user given options.
   */
  if VALUE(stemPrefix'CompOpt') \= '' then
    parms = insert(' 'translate(VALUE(stemPrefix'CompOpt')),,
                   parms,length(parms)-1)
  call addStmt left('C'compCalls, 8), 'EXEC PGM=REXXCOMP,PARM='parms

  call addStmt 'SYSTERM', 'DD SYSOUT=*'

  if dsn.listings \= '' then
    sysp = 'DSN='dsn.listings'('VALUE(stemPrefix'OrgName')'),DISP=SHR'
  else
    sysp = 'SYSOUT=*'
  call addStmt 'SYSPRINT', 'DD' sysp

  if abbrev(VALUE(stemPrefix'ExecDSN'), '&&') then
    disp = 'DISP=(SHR,PASS)'
  else
    disp = 'DISP=SHR'
  call addStmt 'SYSIN', 'DD DSN='VALUE(stemPrefix'ExecDSN')','disp

  if abbrev(dsn.objects, '&&') then
    disp = 'DISP=(SHR,PASS)'
  else
    disp = 'DISP=SHR'
  call addStmt 'SYSPUNCH', 'DD DSN='dsn.objects ||,
                                 '('VALUE(stemPrefix'ExecName')'),'disp

  return

/*_________________________________________________buildC370CompCards__
 */
buildC370CompCards:

  compCalls = compCalls + 1

  parms = "'/OBJECT,XREF,RENT'"
  if mainRoutine.CompOpt \= '' then
    parms = insert(','translate(mainRoutine.CompOpt), parms,,
                                                      length(parms)-1)
  call addStmt left('C'compCalls, 8), 'EXEC PGM=CBC320PP,PARM='parms

  call addStmt 'SYSMSGS', 'DD DUMMY,DSN=SYS1.CMVS.V320.SCBC3MSG' ||,
                                                   '(EDCMSGE),DISP=SHR'
  call addStmt 'SYSLIB',  'DD DSNAME=SYS1.SCEEH.H,DISP=SHR'
  call addStmt '',        'DD DSNAME=SYS1.SCEEH.SYS.H,DISP=SHR'
  if dsn.cHeaderFiles \= '' then
    call addStmt '',      'DD DSNAME='dsn.cHeaderFiles',DISP=SHR'

  call addStmt 'SYSIN',   'DD DSN='mainRoutine.ExecDSN',DISP=SHR'
  call addStmt 'SYSLIN',  'DD DSN=&&OBJDECKS('mainRoutine.OrgName ||,
                                             '),DISP=(SHR,PASS)'

  if dsn.listings \= '' then
    sysp = 'DSN='dsn.listings'('mainRoutine.OrgName'),DISP=SHR'
  else
    sysp = 'SYSOUT=*'
  call addStmt 'SYSPRINT','DD' sysp
  call addStmt 'SYSOUT',  'DD SYSOUT=*'
  call addStmt 'SYSCPRT', 'DD SYSOUT=*'
  call addStmt 'SYSUT1',  'DD UNIT=VIO,SPACE=(32000,(30,30)),' ||,
                              'DCB=(RECFM=FB,LRECL=80,BLKSIZE=3200)'
  call addStmt 'SYSUT4',  'DD UNIT=VIO,SPACE=(32000,(30,30)),' ||,
                              'DCB=(RECFM=FB,LRECL=80,BLKSIZE=3200)'
  call addStmt 'SYSUT5',  'DD UNIT=VIO,SPACE=(32000,(30,30)),' ||,
                              'DCB=(RECFM=FB,LRECL=3200,BLKSIZE=12800)'
  call addStmt 'SYSUT6',  'DD UNIT=VIO,SPACE=(32000,(30,30)),' ||,
                              'DCB=(RECFM=FB,LRECL=3200,BLKSIZE=12800)'
  call addStmt 'SYSUT7',  'DD UNIT=VIO,SPACE=(32000,(30,30)),' ||,
                              'DCB=(RECFM=FB,LRECL=3200,BLKSIZE=12800)'
  call addStmt 'SYSUT8',  'DD UNIT=VIO,SPACE=(32000,(30,30)),' ||,
                              'DCB=(RECFM=FB,LRECL=3200,BLKSIZE=12800)'
  call addStmt 'SYSUT9',  'DD UNIT=VIO,SPACE=(32000,(30,30)),' ||,
                              'DCB=(RECFM=VB,LRECL=137,BLKSIZE=882)'
  call addStmt 'SYSUT10', 'DD SYSOUT=*'
  call addStmt 'SYSUT14', 'DD UNIT=VIO,SPACE=(32000,(30,30)),' ||,
                              'DCB=(RECFM=FB,LRECL=3200,BLKSIZE=12800)'

  cMain = 1

  return

/*______________________________________________buildC370PrelinkCards__
 */
buildC370PrelinkCards:

  call addcomm 'PRE-LINKEDIT THE MAIN C/370 MODULE'

  parms = "'MAP'"
  if mainRoutine.PreLinkOpt \= '' then
    parms = insert(','translate(mainRoutine.PreLinkOpt), parms,,
                                                      length(parms)-1)
  call addStmt 'PLK',     'EXEC PGM=EDCPRLK,PARM='parms

  call addStmt 'SYSMSGS', 'DD DSNAME=SYS1.SCEEMSGP(EDCPMSGE),DISP=SHR'
  call addStmt 'SYSLIB',  'DD DSN=SYS1.SCEERUN,DISP=SHR'

  if abbrev(dsn.objects, '&&') then
    disp = 'DISP=(SHR,PASS)'
  else
    disp = 'DISP=SHR'
  call addStmt 'SYSIN',   'DD DSN='dsn.objects'('mainRoutine.OrgName ||,
                                                '),DISP=(SHR,PASS)'

  call addStmt '',        'DD DDNAME=SYSIN2'
  call addStmt 'SYSMOD',  'DD DSN='dsn.cPrelink'('mainRoutine.OrgName ||,
                                                '),DISP=(SHR,PASS)'
  call addStmt 'SYSOUT',  'DD SYSOUT=*'
  call addStmt 'SYSPRINT','DD SYSOUT=*'
  call addStmt 'SYSIN2',  'DD DUMMY'

  return

/*________________________________________buildMainRoutineCompileStep__
 */
buildMainRoutineCompileStep:

  select
    when mainRoutine.Language = 'REXX' then do
      call addComm 'COMPILE THE MAIN REXX MODULE'
      call buildRexxCompCards 'mainRoutine.'
      call buildMainRoutineLinkStep
    end
    when mainRoutine.Language = 'C' then do
      call addComm 'COMPILE THE MAIN C/370 MODULE'
      call buildC370CompCards
      call buildC370PrelinkCards
    end
    otherwise
      exitWithError(4,"'"mainRoutine.Language"'",
                      "is not a valid LANGUAGE !")
  end

  return

/*_________________________________buildExternalFunctionsCompileSteps__
 */
buildExternalFunctionsCompileSteps:

  call addComm 'COMPILE THE EXTERNAL REXX FUNCTIONS'

  do i = 1 to extFunc.0
    call buildRexxCompCards 'extFunc.'i'.'
    if i < extFunc.0 then
      call addStmt '*'
  end

  return

/*_________________________________________________buildLinkerOptions__
 */
buildLinkerOptions: procedure
  parse arg extraOptions

  parms = "'LIST,AMODE=31,RMODE=ANY,RENT,MAP'"
  if extraOptions \= '' then do
    do until extraOptions = ''
      parse var extraOptions 0 parm ',' extraOptions
      parse var parm 0 parmName '='
      parmNamePos = pos(parmName, parms)
      if parmNamePos > 0 then
        parms = delstr(parms, parmNamePos,,
                       pos(',', parms, parmNamePos) - parmNamePos + 1)
      parms = insert(','translate(parm), parms, length(parms) - 1)
    end
  end

  return parms

/*___________________________________________buildMainRoutineLinkStep__
 */
buildMainRoutineLinkStep:

  linkCalls = linkCalls + 1

  call addComm 'LINK THE MAIN MODULE'

  call addStmt left('L'linkCalls, 8), 'EXEC PGM=HEWL,' ||,
                        'PARM='buildLinkerOptions(mainRoutine.LinkOpt)
  call addStmt 'SYSUT1',  'DD UNIT=SYSDA,SPACE=(1024,(200,20))'
  call addStmt 'SYSPRINT','DD SYSOUT=*'

  if dsn.stubs = '' then
    syslib = 'DD DSN='dsn.rexxPrefix'SEAGLMD,DISP=SHR'
  else
    syslib = 'DD DSN='dsn.stubs',DISP=SHR'
  call addStmt 'SYSLIB',  syslib

  if abbrev(dsn.loadMods, '&&') then
    disp = 'DISP=(SHR,PASS)'
  else
    disp = 'DISP=SHR'
  call addStmt 'SYSLMOD', 'DD DSN='dsn.loadMods','disp

  if abbrev(dsn.objects, '&&') then
    disp = 'DISP=(SHR,PASS)'
  else
    disp = 'DISP=SHR'
  call addStmt 'OBJS',    'DD DSN='dsn.objects','disp

  call addStmt 'SYSLIN',  'DD *'
  objName = '@'strip(left(mainRoutine.ExecName,7))
  call addInln 'CHANGE   EAGSTUB('mainRoutine.ExecName'),' ||,
                        'EAGOBJ('objName')'

  st = translate(mainRoutine.Stub)
  select
    when st = 'CPPLEFPL' then
      stubName = 'EAGSTCE'
    when st = 'CALLCMD' then
      stubName = 'EAGSTCALL'
    when st = 'EFPL' then
      stubName = 'EAGSTEFP'
    when st = 'CPPL' then
      stubName = 'EAGSTCPP'
    when st = 'MVS' then
      stubName = 'EAGSTMVS'
    otherwise
      exitWithError(5, "'"mainRoutine.stub"'" 'is not a valid STUB!')
  end
  call addInln 'INCLUDE  SYSLIB('stubName')'
  call addInln 'CHANGE   'mainRoutine.ExecName'('objName')'
  call addInln 'INCLUDE  OBJS('mainRoutine.ExecName')'
  call addInln 'ENTRY    'mainRoutine.ExecName
  call addInln 'NAME     'mainRoutine.ExecName'(R)'
  call addInln '/*'

  return

/*____________________________________buildExternalFunctionsLinkSteps__
 */
buildExternalFunctionsLinkSteps:

  call addComm 'LINK THE EXTERNAL REXX FUNCTIONS'

  do i = 1 to extFunc.0

    linkCalls = linkCalls + 1

    call addStmt left('L'linkCalls, 8), 'EXEC PGM=HEWL,' ||,
                           'PARM='buildLinkerOptions(extFunc.i.LinkOpt)
    call addStmt 'SYSUT1',    'DD UNIT=SYSDA,SPACE=(1024,(200,20))'
    call addStmt 'SYSPRINT',  'DD SYSOUT=*'

    if dsn.stubs = '' then
      syslib = 'DD DSN='dsn.rexxPrefix'SEAGLMD,DISP=SHR'
    else
      syslib = 'DD DSN='dsn.stubs',DISP=SHR'
    call addStmt 'SYSLIB',  syslib

    if abbrev(dsn.loadMods, '&&') then
      disp = 'DISP=(SHR,PASS)'
    else
      disp = 'DISP=SHR'
    call addStmt 'SYSLMOD',   'DD DSN='dsn.loadMods','disp

    if abbrev(dsn.objects, '&&') then
      disp = 'DISP=(SHR,PASS)'
    else
      disp = 'DISP=SHR'
    call addStmt 'OBJS',      'DD DSN='dsn.objects','disp

    call addStmt 'SYSLIN',    'DD *'

    if extFunc.i.Recursive = 1 then do
      call addInln 'CHANGE   EAGSTUB('extFunc.i.OrgName'),' ||,
                            'EAGOBJ('extFunc.i.ExecName')'
      if mainRoutine.Language = 'C' then
        stubName = 'EAGSTCR'
      else
        stubName = 'EAGSTEFP'
      call addInln 'INCLUDE  SYSLIB('stubName')'
    end
    else do
      objName = strip(left('@'extFunc.i.ExecName, 8))
      call addInln 'CHANGE   EAGSTUB('extFunc.i.ExecName'),' ||,
                            'EAGOBJ('objName')'
      if mainRoutine.Language = 'C' then
        stubName = 'EAGSTC'
      else
        stubName = 'EAGSTCE'
      call addInln 'INCLUDE  SYSLIB('stubName')'
      call addInln 'CHANGE   'extFunc.i.ExecName'('objName')'
    end
    call addInln 'INCLUDE  OBJS('extFunc.i.ExecName')'
    call addInln 'ENTRY    'extFunc.i.OrgName
    call addInln 'NAME     'extFunc.i.OrgName'(R)'
    call addInln '/*'

    if i < extFunc.0 then
      call addStmt '*'
  end

  return

/*__________________________________________buildTargetModuleLinkStep__
 */
buildTargetModuleLinkStep:

  call addComm 'NOW LINK IT ALL TOGETHER INTO ONE BIG LOAD MODULE'
  call addStmt 'LFINAL',,
               "EXEC PGM=HEWL,PARM='LIST,RENT,MAP,AMODE=31,RMODE=ANY'"

  if mainRoutine.Language = 'REXX' then
    call addStmt 'SYSLIB',  'DD DUMMY'
  else do
    call addStmt 'SYSLIB',  'DD DSN=SYS1.SCEELKED,DISP=SHR'
    if cMain then
      if dsn.stubs = '' then
        call addStmt '',    'DD DSN='dsn.rexxPrefix'SEAGLMD,DISP=SHR'
      else
        call addStmt '',    'DD DSN='dsn.stubs',DISP=SHR'
  end

  if mainRoutine.Language = 'REXX' then
    sysut1 = 'DD UNIT=SYSDA,SPACE=(1024,(200,20))'
  else
    sysut1 = 'DD UNIT=VIO,SPACE=(32000,(30,30))'
  call addStmt 'SYSUT1',  sysut1

  call addStmt 'SYSPRINT','DD SYSOUT=*'
  call addStmt 'SYSLMOD', 'DD DSN='dsn.finalMod',DISP=SHR'

  if abbrev(dsn.loadMods, '&&') then
    disp = 'DISP=(SHR,PASS)'
  else
    disp = 'DISP=SHR'
  call addStmt 'LMODS',   'DD DSN='dsn.loadMods','disp

  if mainRoutine.Language = 'REXX' then do
    call addStmt 'SYSLIN',  'DD *'
    call addInln 'INCLUDE  LMODS('mainRoutine.ExecName')'
  end
  else do
    call addStmt 'SYSLIN',  'DD DSN='dsn.cPrelink ||,
                              '('mainRoutine.OrgName'),DISP=(SHR,PASS)'
    call addStmt '',        'DD *'
    call addInln 'INCLUDE  SYSLIB(EVALFREE)'
  end

  do i = 1 to extFunc.0
    call addInln 'INCLUDE  LMODS('extFunc.i.OrgName')'
  end

  if mainRoutine.Language = 'REXX' then
    call addInln 'ENTRY    'mainRoutine.ExecName

  if pos('(', dsn.finalMod) > 0 then do
    p1 = pos('(', dsn.finalMod)
    p2 = pos(')', dsn.finalMod)
    modName = substr(dsn.finalMod, p1 + 1, p2 - p1 - 1)
  end
  else
    modName = mainRoutine.ExecName
  call addInln 'NAME     'modName'(R)'
  call addInln '/*'

  return

/*=====================================================================
 *                        SOME ERROR HANDLING
 *=====================================================================
 */

/*__________________________________________________________saySyntax__
 */
saySyntax:

  say 'the syntax is:'
  say '  RXMAKMOD <makefile_name> [SUB|EDIT]'

  return

/*___________________________________________________________testArgs__
 */
testArgs:

  errOcc = 0

  if makeFileName = '' then do
    errOcc = 1
    say 'You must specify the Makefile-DD-name'
  end
  if subOrEdit \= '' & subOrEdit \= 'SUB' & subOrEdit \= 'EDIT' then do
    errOcc = 1
    say 'The only valid values for the second parm are SUB and EDIT'
  end

  if errOcc then do
    call saySyntax
    exitWithError(6, 'correct the command line and retry.')
  end

  return

/*______________________________________________________exitWithError__
 */
exitWithError:
  parse arg errRC, errTxt

  say copies('*-', 36)
  say 'Sorrrry, but some  ..$%&/#!%.. error must have occured:'
  say 'the errRC is:' errRC
  say 'description: ' errTxt
  say copies('*-', 36)
  say '...exiting with' errRC

  exit errRC

/*____________________________________________________________noValue__
 */
noValue:

  say copies('-', 72)
  say 'Runtime condition raised:' condition('c')
  say 'Variable    :' condition('d')
  say copies('-', 72)

  return

/*_____________________________________________________________syntax__
 */
syntax:

  say copies('-', 72)
  say 'Runtime condition raised:'
  say '  kind         :' condition('C')
  say '  description  :' condition('D')
  say '  Source line# :' SIGL
  say '  Source line  :' sourceLine(SIGL)
  say '  Return code  :' RC
  say '  Error text   :' errorText(RC)
  say copies('-', 72)

  exit RC
