/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    IBM Corporation 1998                                                  */
/*                                                                          */
/*  Command_Sx.cmd  -  Extended TCP/IP Socket Server with sessions          */
/*                                                                          */
/*  ---> This server requires the asynchronous client command_cx!           */
/*                                                                          */
/*    Parameters:                                                           */
/*      Port:   server port number (optional)                               */
/*                                                                          */
/*  Documentation: see cmmnd.doc in directory docs                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

Parse Arg Port 
if Port = '' then Port = 1922          /* Default server port is 1922       */

aCmdServer = .CommandServer~new(Port)  
aCmdServer~startAccepting              /* method inherited from "tcpServer" */    
aCmdServer~terminate                   /* terminate all client sessions     */    
aCmdServer~shutdown                    /* shutdown the server               */                    

/****************************************************************************/

::REQUIRES "servers.frm"               /* Load the servers framework        */   
::REQUIRES "security.frm"              /* Load the security framework       */   

/*--------------------------------------------------------------------------*/
/* CommandServer Class definition                                           */
/*--------------------------------------------------------------------------*/
::CLASS CommandServer SUBCLASS tcpServer PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD init
  expose Sessions 
  use arg  Port

  self~init:super(Port)                /* Run the superclass init           */
  Sessions = .bag~new                  /* Create Sessions with no sessions  */                                          

/*--------------------------------------------------------------------------*/
::METHOD NewClient UNGUARDED           /* Over-write superclass method      */    
  expose Sessions 
  use arg cSocket, Prefix              /* parm: client socket and prefix    */
                                       /* Create a session at server        */ 
  session = .Session~new(self, cSocket, Prefix)
  session~logon(Sessions)              /* and start it                      */
  Sessions~put(session)                /* Put new client socket in Sessions */
                                       /* NewClient must return NOTHING!    */                                   
  return                               /* otherwise it blocks accepting     */
                                       /* new clients!                      */

/*--------------------------------------------------------------------------*/
::METHOD terminate                     /* Terminate the whole server        */
  expose Sessions

  do session over Sessions             /* Logoff all sessions               */      
    session~logoff(Sessions)
  end

/*--------------------------------------------------------------------------*/
/* Session Class definition                                                 */
/*--------------------------------------------------------------------------*/
::CLASS Session

/*--------------------------------------------------------------------------*/
::METHOD init
  expose server cSocket Platform NL
  use arg server, cSocket, prefix      /* Parameters                        */  

  self~name = prefix~word(1)           /* Get the client name from prefix   */
  platform = prefix~word(2)            /* Get the platform id from prefix   */

  if platform = 'LINUX' | platform = 'AIX' then
    NL = x2c('0a')                     /* 'new line' character for UNIX     */   
  else
    NL = x2c('0d0a')                   /* 'new line' characters for nonUNIX */   
                                       
/*--------------------------------------------------------------------------*/
::METHOD name ATTRIBUTE

/*--------------------------------------------------------------------------*/
::METHOD logon 
  expose server cSocket Platform NL 
  use arg Sessions           
                                       /* create a command processor        */
  processor = .CommandProcessor~new(cSocket, Platform, self~name, NL)
  running = .TRUE                      /* set session active                */
  reply

  do while running                     /* Session handling loop             */ 
    CommandLine = cSocket~ReceiveData  /* Receive command line from client  */
    if \cSocket~stillOpen then leave   /* if problems, end client session   */ 
                                       /* parse into command and options    */
    Parse Var CommandLine Command Option
    Command = Command~translate        /* translate command to upper and    */
    Option = option~strip              /* strip off leading/trailing blanks */

    if Option = '' then                /* if no option specified ...        */  
      optstring = '<none>'             /* define substitute                 */
    else
      optstring = Option               /* or use real option otherwise      */

    say "Processing" Command "with option:" optstring "(client:" self~name")"
    cSocket~SendData("---Begin_of_transmission---"NL)
    cSocket~SendData("Processing" Command "with option:" optstring || NL ) 

                                       /* Create message for the processor  */
    Message = .message~new(Processor, Command, 'I', Option)
    Message~send                       /* now send it to him                */
                                       /* Send end of answer back to client */
    if Message~result = "#>>End_of_session<<#" then do
      self~logoff(Sessions)                /* Logoff this session               */
      leave
    end
                                        /* Initiate server shutdown          */ 
    if Message~result = "#>>Shutdown<<#" then do
                                       /* determine no of sessions running  */
      say "Shutting down" Sessions~makearray~last "sessions!" 
      server~close                     /* close server request queue        */  
      running = .FALSE                 /* switch off session running        */
    end
    cSocket~SendData("---End_of_transmission---"NL)
  end

/*--------------------------------------------------------------------------*/
::METHOD logoff UNGUARDED
  expose cSocket Platform NL 
  use arg Sessions           

  sessions~remove(self)                /* Remove the session from sessions  */
  csocket~SendData("#>>End_of_session<<#"NL)
  csocket~Close                        /* Close client's socket             */  

/*--------------------------------------------------------------------------*/
/* CommandProcessor Class definition                                        */
/*--------------------------------------------------------------------------*/
::CLASS CommandProcessor

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Command processor initialisation  */
  expose cSocket Platform Name NL 
  use arg cSocket, Platform, Name, NL

/*--------------------------------------------------------------------------*/
::METHOD ?                             /* Method handling query             */
  expose cSocket NL

  cSocket~SendData("Valid commands:" cmdlist() || NL ) 
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Dir                           /* Method handling the DIR command   */ 
  expose cSocket NL Platform
  use arg option

  if Platform = "LINUX" then           /* determine platform specific list  */
    "ls -l" option "| rxqueue"
  else
    "DIR" option "| rxqueue"           /* route the result to RXqueue       */

  do while QUEUED()>0                  /* read from RXqueue until it's empty*/
    parse pull line
    cSocket~SendData(line || NL)       /* send each line to the client      */
  end 
  return ""                            /* the session can continue          */

/*--------------------------------------------------------------------------*/
::METHOD show                          /* Method handling the show command  */  
  expose cSocket NL                    
  use arg file

  if file = "" then                    /* send message to client            */
    cSocket~SendData("You have to specify a filename with SHOW" NL) 

  else do
    stream = .stream~new(file)         /* create a stream object for file   */

    if stream~query("EXISTS") = "" then/* does file exist?                  */ 
      cSocket~SendData("The file '"file"' does not exist"NL) 
    else do line over stream           /* Walk through the stream by line   */
      cSocket~SendData(line || NL)     /* and send each line to the client  */
    end 
  end 
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Hello                         /* Method handling hello command     */ 
  expose cSocket Name NL                     
                                       /* Do whatever is necessary for this */
  thisHost = .tcpHost~new
  if thisHost~name = cSocket~Hostname then 
    cSocket~SendData("It's me," thisHost~name || NL)
  else 
    cSocket~SendData("Hello" Name"! My name is" thisHost~name || NL)
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Quit                          /* Method handling the quit command  */
  expose cSocket NL                     
  return "#>>End_of_session<<#"

/*--------------------------------------------------------------------------*/
::METHOD Shutdown                      /* Method handling shutdown command  */  
  expose cSocket NL                    

  thisHost = .tcpHost~new
  if thisHost~name = cSocket~Hostname then do 
    cSocket~SendData("Server" thisHost~name "is shutting down!"NL)
    return "#>>Shutdown<<#"            /* tell the session to shutdown      */
  end          
  cSocket~SendData("You are not authorized to shut down server" thisHost~name"!"NL)
  return ""                            /* the session can continue          */ 

/*--------------------------------------------------------------------------*/
::METHOD Unknown                       /* Method handling all unknown cases */
  expose cSocket NL
  use arg Command
                                       /* Were there socket problems?       */ 
  if Command < 0 & command~length > 0 then do  
    Say "Lost client on socket" cSocket "; session closed"
    return "#>>End_of_session<<#"      /* tell the session to end           */ 
  end
  else do                              /* Signal command is unsupported     */
    if arg() = 1 then                  /* Immediate Unknown Command         */
      Command = "UNKNOWN"
    else                               /* Unsup'd command routed to unknown */
      Say "--> Actually processing UNKNOWN command"

    cSocket~SendData("Invalid command:" Command || NL) 
    cSocket~SendData("Valid commands are:" cmdlist() || NL ) 
    return ""                          /* the session can continue          */ 
  end

/*--------------------------------------------------------------------------*/
::ROUTINE cmdlist                      /* Generic fct providing cmdList     */
                                       /* define set of hidden commands     */
  hiddenCmds = .set~of("UNKNOWN", "INIT") 
                                       /* get instance methods of class     */
  commandSupplier = .CommandProcessor~methods(.nil)
  cmds = commandSupplier~index         /* get first command from supplier   */ 
  commandSupplier~next                 /* move to the next supplier entry   */ 

  do while commandSupplier~available   /* process all supplier entries      */
    cmd = commandSupplier~index        /* get command from the supplier     */
    if \hiddenCmds~hasindex(cmd) then  /* if a hidden command, skip         */
      cmds = cmds"," cmd               /* include this command in the list  */
    commandSupplier~next               /* move to the next supplier entry   */ 
  end
  return cmds                          /* return the list of commands       */
