/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  NetCentric Computing with Object Rexx                                   */
/*  Programming Example                                                     */
/*                                                                          */
/*    IBM Corporation 1998                                                  */
/*                                                                          */
/*  Clients.frm  -  Classes for TCP/IP Clients                              */
/*                                                                          */
/*    Client program for the mirror server. Ask the user for a string,      */ 
/*    send the string to the mirror server and print the mirrored string.   */
/*                                                                          */
/*    Parameters:                                                           */
/*      Server: alias name of server                                        */
/*      Port:   server port number                                          */
/*      Client: alias for client                                            */
/*      Option: option data (sent with the client prefix to the server)     */
/*                                                                          */
/*  History:                                                                */
/*    06/30/97  added client alias option for multiple clients per host     */
/*              made prompt message on promptinput method optional          */
/*    07/20/97  added client prefix (client alias followed by system id)    */
/*    07/22/97  added option passed to the server with message              */
/*    08/26/97  added option '!' to suppress client prefix                  */
/*    08/29/97  added alternative init option to create unconnected client  */
/*                                                                          */
/*--------------------------------------------------------------------------*/

::REQUIRES "sockets.frm"               /* Load the sockets framework        */   

/*--------------------------------------------------------------------------*/
/* tcpClient Class definition                                               */
/*--------------------------------------------------------------------------*/
::CLASS tcpClient PUBLIC

/*--------------------------------------------------------------------------*/
::METHOD init                          /* Get sockets and binds port        */ 
  expose input output socket port      /* Attributes of a tcpClient object  */
  use arg server, port, client, option /* Arguments                         */        

  parse source sysinfo .               /* get information on the platform   */ 
  if sysinfo = 'LINUX' | sysinfo = 'AIX' then
    self~NL= '0d'x                     /* and define NL control             */
  else
    self~NL= '0d0a'x 
  self~sysInfo= sysinfo                /* Set system ID                     */
  self~connected = 0                   /* Initialize connect status         */ 

  if arg() = 0 then do                 /* unconnected client...             */
    input = ''                         /* Initialize input and output       */  
    output = ''
  end 
  else do                              /* connected client...               */ 
    if client = '' then client = '.'   /* Set default client name           */
    socket = .tcpSocket~new            /* Get a stream socket               */
  
    host = .tcpHost~new(server)        /* Create a host instance            */
    address = host~address             /* Get server address from host      */
    if address \= "" then do           /* Valid address obtained?           */
      rc = socket~connect(port, address) /* Connect to port of server       */
  
      if client \= '!' then do         /* if no shriek client               */
        prefix = client sysinfo        /* and generate client prefix        */
      end                              /* client prefix: <alias systemid>   */
      else prefix = ''                 /* use no client prefix otherwise    */
                                       /* now send this message to server   */
      self~connected = (rc = 0 & socket~SendData(prefix option)>0)
    end
    server = host~Name                 /* Get the server name               */
    if self~connected then             /* Is connection ok?                 */
      say "Client connected to server" server "at port" port
    else 
      say "Client could not establish connection to server" server "at port" port   
  end

/*--------------------------------------------------------------------------*/
::METHOD connected ATTRIBUTE           /* Show connection status            */   

/*--------------------------------------------------------------------------*/
::METHOD sysInfo ATTRIBUTE             /* System information                */   

/*--------------------------------------------------------------------------*/
::METHOD NL ATTRIBUTE                  /* New Line control                  */   

/*--------------------------------------------------------------------------*/
::METHOD send UNGUARDED                /* Send message to server            */
  expose socket 
  use arg message
  
  return socket~SendData(message)

/*--------------------------------------------------------------------------*/
::METHOD receive UNGUARDED             /* Receive return message from server*/
  expose socket output
  
  output = socket~ReceiveData
  if socket~stillOpen then             /* Socket still open?                */          
    rc = output~length
  else do                              /* Received data ok, return # bytes  */
    rc = output
    output = ""                        /* Indicate error                    */ 
  end
  return rc

/*--------------------------------------------------------------------------*/
::METHOD getInput                      /* get input from user               */
  expose Input

  signal on syntax                     /* Intercept ctrl-c interrupts       */
  parse pull Input                     /* read input from STDIN             */
  if Input == '' then Input = ' '
  return 0

syntax:                                /* process when ctrl-c was pressed   */
  say "Client interrupted by user"
  return -1

/*--------------------------------------------------------------------------*/
::METHOD promptInput                   /* prompt input from user            */
  expose input

  if arg() > 0 then do
    say ""
    say arg(1)                         /* display prompt message            */
  end
  rc = self~getInput()                 /* get input from user               */
  if rc = 0 then 
    return self~sendInput              /* send input string to server       */
  return rc

/*--------------------------------------------------------------------------*/
::METHOD sendInput                     /* provide input string              */
  expose socket port input

  rc = socket~SendData(input)          /* send input string to server       */
  if rc < 0 then                     
    say "Lost connection to server on port" port  
  return rc

/*--------------------------------------------------------------------------*/
::METHOD sendFile UNGUARDED            /* provide input string              */
  expose socket port 
  use arg file

  rc = socket~SendFile(file)           /* send file content to server       */
  if rc < 0 then                     
    say "Lost connection to server on port" port  
  return rc

/*--------------------------------------------------------------------------*/
::METHOD receiveFile UNGUARDED         /* Receive file (with or w/o header) */
  expose socket                        /* arg(2): 'HEADER'   - with header  */
                                       /*         'NOHEADER' - w/o header   */    
  if arg() = 0 then                    /* if no argument...                 */
    return socket~ReceiveFile()        /*   receiveFile()      -> transient */
  else if arg() = 1 then               /* if one argument...                */
    return socket~ReceiveFile(arg(1))  /*   receiveFile(fname) -> file      */
  else                                 /* else receiveFile(fname, mode)     */ 
    return socket~ReceiveFile(arg(1), arg(2))

/*--------------------------------------------------------------------------*/
::METHOD Input UNGUARDED               /* provide input string              */
  expose input
  return input                         /* return what has been entered      */

/*--------------------------------------------------------------------------*/
::METHOD Output UNGUARDED              /* provide output string             */
  expose output
  return output                        /* return what has been received     */

/*--------------------------------------------------------------------------*/
::METHOD shutdown UNGUARDED            /* Shut down client                  */ 
  expose socket port

  if socket~HostName \= -1 then do
    if self~connected then do
      say "Client disconnected from server" socket~HostName "at port" port
      socket~Shutdown                  /* Shutdown the socket               */
    end
  end
  socket~Close                         /* Return the socket resource        */
  self~connected = 0                   /* Set socket status unconnected     */ 
  
