/*---------------------------------------------------------------------------
 * nsc_accl.c
 *
 * This file contains the graphics acceleration routines.  It makes use 
 * of the Durango graphics software support package from National.
 *
 * It is ported from the Cyrix driver developed by Annius Groenink.
 *
 * Copyright (c) 1999-2000 National Semicondictor.
 *---------------------------------------------------------------------------
 */

/* Xfree86 header files */

#include "vga256.h"
#include "xf86.h"
#include "vga.h"
#include "xf86xaa.h"
#include "xf86_Config.h"
#include "miline.h"

/* National Semiconductor header files */

#include "gfx_rtns.h"

extern unsigned long geodeOffscreenOffset;
extern unsigned long geodeOffscreenSize;

/* The following ROPs only use pattern and destination data. */
/* They are used when the planemask specifies all planes (no mask). */

static const int windowsROPpat[16] = { 
	0x00, 0xA0, 0x50, 0xF0, 0x0A, 0xAA, 0x5A, 0xFA, 
	0x05, 0xA5, 0x55, 0xF5, 0x0F, 0xAF, 0x5F, 0xFF
};

/* The following ROPs use source data to specify a planemask. */
/* If the planemask (src) is one, then the result is the appropriate */
/* combination of pattern and destination data.  If the planemask (src) */
/* is zero, then the result is always just destination data. */

static const int windowsROPsrcMask[16] = { 
	0x22, 0xA2, 0x62, 0xE2, 0x2A, 0xAA, 0x6A, 0xEA,
    0x26, 0xA6, 0x66, 0xE6, 0x2E, 0xAE, 0x6E, 0xEE
};

/* The following ROPs use pattern data to specify a planemask. */
/* If the planemask (pat) is one, then the result is the appropriate */
/* combination of source and destination data.  If the planemask (pat) */
/* is zero, then the result is always just destination data. */

static const int windowsROPpatMask[16] = { 
	0x0A, 0x8A, 0x4A, 0xCA, 0x2A, 0xAA, 0x6A, 0xEA,
    0x1A, 0x9A, 0x5A, 0xDA, 0x3A, 0xBA, 0x7A, 0xFA 
};

/* STATIC VARIABLES FOR THIS FILE */
/* Used to maintain state between setup and rendering calls. */

static int GeodeTransparent;
static int GeodeTransColor;

/* DECLARATION OF ROUTINES THAT ARE HOOKED */

void GeodeAccelSync();
void GeodeAccelInit();
void GeodeSetupForFillRectSolid();
void GeodeSubsequentFillRectSolid();
void GeodeSetupFor8x8PatternColorExpand();
void GeodeSubsequent8x8PatternColorExpand();
void GeodeSetupForScreenToScreenCopy();
void GeodeSubsequentScreenToScreenCopy();
void GeodeSubsequentBresenhamLine();

/*---------------------------------------------------------------------------
 * GeodeAccelInit
 *
 * This routine hooks the acceleration routines and sets appropriate flags.
 *---------------------------------------------------------------------------
 */
void GeodeAccelInit() 
{
	/* SET ACCELERATION FLAGS */

	xf86AccelInfoRec.Flags = PIXMAP_CACHE
	                       | BACKGROUND_OPERATIONS
	                       | HARDWARE_PATTERN_SCREEN_ORIGIN
	                       | HARDWARE_PATTERN_BIT_ORDER_MSBFIRST
	                       | HARDWARE_PATTERN_PROGRAMMED_BITS
	                       | HARDWARE_PATTERN_MONO_TRANSPARENCY;

	/* HOOK SYNCRONIZARION ROUTINE */

	xf86AccelInfoRec.Sync = GeodeAccelSync;

	/* HOOK FILLED RECTANGLES */

	xf86AccelInfoRec.SetupForFillRectSolid = 
		GeodeSetupForFillRectSolid;
	xf86AccelInfoRec.SubsequentFillRectSolid = 
		GeodeSubsequentFillRectSolid;
	xf86GCInfoRec.PolyFillRectSolidFlags = 0;

	/* HOOK 8x8 COLOR EXPAND PATTERNS */

	xf86AccelInfoRec.SetupFor8x8PatternColorExpand = 
		GeodeSetupFor8x8PatternColorExpand;
	xf86AccelInfoRec.Subsequent8x8PatternColorExpand =
	    GeodeSubsequent8x8PatternColorExpand;

	/* HOOK SCREEN TO SCREEN COPIES */
	/* Set flag to only allow copy if transparency is enabled. */

	xf86AccelInfoRec.SetupForScreenToScreenCopy = 
		GeodeSetupForScreenToScreenCopy;
	xf86AccelInfoRec.SubsequentScreenToScreenCopy = 
		GeodeSubsequentScreenToScreenCopy;
	xf86GCInfoRec.CopyAreaFlags = TRANSPARENCY_GXCOPY;

	/* HOOK BRESENHAM LINES */
	/* Do not hook unless flag can be set preventing use of planemask. */

#if 0
	xf86AccelInfoRec.SubsequentBresenhamLine = 
		GeodeSubsequentBresenhamLine;
	xf86AccelInfoRec.ErrorTermBits = 15;
#endif

	/* SET UP GRAPHICS MEMORY AVAILABLE FOR PIXMAP CACHE */

	xf86InitPixmapCache(&vga256InfoRec, geodeOffscreenOffset, 
		geodeOffscreenOffset + geodeOffscreenSize);

	/* TEXT IS NOT HOOKED */
	/* Actually faster to not accelerate it.  To accelerate, the data */
	/* is copied from the font data to a buffer by the OS, then the */
	/* driver copies it from the buffer to the BLT buffer, then the */
	/* rendering engine reads it from the BLT buffer.  May be worthwhile */
	/* to hook text later with the second generation graphics unit that */
	/* can take the data directly. */
} 

/*---------------------------------------------------------------------------
 * GeodeAccelSync
 *
 * This routine is called before accessing the frame buffer directly to 
 * make sure that the graphics pipeline is idle. 
 *---------------------------------------------------------------------------
 */
void GeodeAccelSync()
{	
	/* WAIT UNTIL GRAPHICS ENGINE IS IDLE */

	gfx_wait_until_idle();
}

/*---------------------------------------------------------------------------
 * GeodeSetupForFillRectSolid
 *
 * This routine is calles to setup the solid pattern color for future
 * rectangular fills or vectors.
 *---------------------------------------------------------------------------
 */
void GeodeSetupForFillRectSolid(color, rop, planemask)
int color, rop;
unsigned int planemask;
{
	/* LOAD THE SOLID PATTERN COLOR */

	gfx_set_solid_pattern((unsigned long) color);

	/* CHECK IF PLANEMASK IS NOT USED (ALL PLANES ENABLED) */

	if (planemask == (unsigned int) -1)
	{
		/* USE NORMAL PATTERN ROPs IF ALL PLANES ARE ENABLED */

		gfx_set_raster_operation(windowsROPpat[rop & 0x0F]);
	}
	else
	{
		/* SELECT ROP THAT USES SOURCE DATA FOR PLANEMASK */

		gfx_set_solid_source((unsigned long) planemask);
		gfx_set_raster_operation(windowsROPsrcMask[rop & 0x0F]);
	}
}
    
/*---------------------------------------------------------------------------
 * GeodeSubsequentFillRectSolid
 *
 * This routine is called to fill a rectangular region using the previously
 * specified solid pattern and raster operation.  
 *
 * Sample application uses:
 *   - Window backgrounds. 
 *   - x11perf: rectangle tests (-rect500).
 *   - x11perf: fill trapezoid tests (-trap100).
 *   - x11perf: horizontal line segments (-hseg500).
 *---------------------------------------------------------------------------
 */
void GeodeSubsequentFillRectSolid(x, y, w, h)
int x, y, w, h;
{
	/* SIMPLY PASS THE PARAMETERS TO THE DURANGO ROUTINE */

	gfx_pattern_fill((unsigned short) x, (unsigned short) y, 
		(unsigned short) w, (unsigned short) h);
}

/*---------------------------------------------------------------------------
 * GeodeSetupFor8x8PatternColorExpand
 *
 * This routine is called to setup the monochrome pattern (8x8) and raster 
 * operation for future rectangular fills.
 *---------------------------------------------------------------------------
 */
void GeodeSetupFor8x8PatternColorExpand(patternx, patterny, bg, fg, rop, planemask)
int patternx, patterny;
int bg, fg, rop;
unsigned int planemask;
{	
	int trans = (bg == -1);

	/* LOAD PATTERN COLORS AND DATA */

	gfx_set_mono_pattern((unsigned long) bg, (unsigned long) fg,
		(unsigned long) patternx, (unsigned long) patterny, 
		(unsigned char) trans);

	/* CHECK IF PLANEMASK IS NOT USED (ALL PLANES ENABLED) */

	if (planemask == (unsigned int) -1)
	{
		/* USE NORMAL PATTERN ROPs IF ALL PLANES ARE ENABLED */

		gfx_set_raster_operation(windowsROPpat[rop & 0x0F]);
	}
	else
	{
		/* SELECT ROP THAT USES SOURCE DATA FOR PLANEMASK */

		gfx_set_solid_source((unsigned long) planemask);
		gfx_set_raster_operation(windowsROPsrcMask[rop & 0x0F]);
	}
}

/*---------------------------------------------------------------------------
 * GeodeSubsequent8x8PatternColorExpand
 *
 * This routine is called to fill a rectangular region using the previously
 * specified monochrome pattern (8x8) and raster operation.
 *
 * Sample application uses:
 *   - Patterned desktops
 *   - x11perf: stippled rectangle tests (-srect500).
 *   - x11perf: opaque stippled rectangle tests (-osrect500).
 *---------------------------------------------------------------------------
 */
void GeodeSubsequent8x8PatternColorExpand(patternx, patterny, x, y, w, h)
int patternx, patterny;
int x, y, w, h;
{
	/* SIMPLY PASS THE PARAMETERS TO THE DURANGO ROUTINE */
	/* Ignores specified pattern. */

	gfx_pattern_fill((unsigned short) x, (unsigned short) y, 
		(unsigned short) w, (unsigned short) h);
}

/*---------------------------------------------------------------------------
 * GeodeSetupForScreenToScreenCopy
 *
 * This routine is called to setup the planemask and raster operation 
 * for future screen to screen BLTs.
 *---------------------------------------------------------------------------
 */
void GeodeSetupForScreenToScreenCopy(xdir, ydir, rop, planemask, transparency_color)
int xdir, ydir;
int rop;
unsigned int planemask;
int transparency_color;
{
	/* LOAD PLANEMASK INTO PATTERN DATA */
	
	gfx_set_solid_pattern((unsigned long) planemask);
	
	/* SET RASTER OPERATION FOR USING PATTERN AS PLANE MASK */

	gfx_set_raster_operation(windowsROPpatMask[rop & 0x0F]);
	
	/* SAVE TRANSPARENCY FLAG */

	GeodeTransparent = (transparency_color == -1) ? 0 : 1;
	GeodeTransColor = transparency_color;
}

/*---------------------------------------------------------------------------
 * GeodeSubsequentScreenToScreenCopy
 *
 * This routine is called to perform a screen to screen BLT using the 
 * previously specified planemask, raster operation, and transparency flag.
 *
 * Sample application uses (non-transparent):
 *   - Moving windows.
 *   - x11perf: scroll tests (-scroll500).
 *   - x11perf: copy from window to window (-copywinwin500).
 *
 * No application found using transparency.
 *---------------------------------------------------------------------------
 */
void GeodeSubsequentScreenToScreenCopy(x1, y1, x2, y2, w, h)
int x1, y1, x2, y2, w, h;
{
	if (GeodeTransparent)
	{
		/* CALL ROUTINE FOR TRANSPARENT SCREEN TO SCREEN BLT */
		/* Should only be called for the "copy" raster operation. */

		gfx_screen_to_screen_xblt(
			(unsigned short) x1, (unsigned short) y1, 
			(unsigned short) x2, (unsigned short) y2, 
			(unsigned short) w, (unsigned short) h, 
			(unsigned short) GeodeTransColor);
	}
	else
	{
		/* CALL ROUTINE FOR NORMAL SCREEN TO SCREEN BLT */

		gfx_screen_to_screen_blt(
			(unsigned short) x1, (unsigned short) y1, 
			(unsigned short) x2, (unsigned short) y2, 
			(unsigned short) w, (unsigned short) h);
	}
}

/*---------------------------------------------------------------------------
 * GeodeSubsequentBresenhamLine
 *
 * This routine is called to render a vector using the specified Bresenham
 * parameters.  
 *
 * Sample application uses:
 *   - Window outlines on window move.
 *   - x11perf: line segments (-seg500).
 *---------------------------------------------------------------------------
 */
void GeodeSubsequentBresenhamLine(x1, y1, octant, err, e1, e2, length)
int x1, y1, octant, err, e1, e2, length;
{	
	unsigned short flags;

	/* DETERMINE YMAJOR AND DIRECTION FLAGS */

	if (octant & YMAJOR)
	{	flags = 0x01; /* ymajor */
		if (!(octant & XDECREASING)) flags |= 0x04; /* minor inc */
		if (!(octant & YDECREASING)) flags |= 0x02; /* major inc */
	}
	else
	{	flags = 0;
		if (!(octant & XDECREASING)) flags |= 0x02; /* major inc */
		if (!(octant & YDECREASING)) flags |= 0x04; /* minor inc */
	}

	/* CALL ROUTINE TO DRAW VECTOR */

	gfx_bresenham_line((unsigned short) x1, (unsigned short) y1, 
		(unsigned short) length, (unsigned short) err, 
		(unsigned short) e1, (unsigned short) e2, (unsigned short) flags);	
}

/* END OF FILE */



