/*
 * target_STM32F4.h				header file specific to the Stm32F4 Discovery target compiler
 */

#ifndef  TARGET_STM32F4_H
#define  TARGET_STM32F4_H

#include  <stdio.h>
#include  <stdint.h>
#include  <limits.h>

#include "stm32f4_discovery.h"
#include "stm32f4xx_adc.h"
#include "stm32f4xx_rcc.h"
#include "stm32f4xx_gpio.h"
#include "stm32f4xx_usart.h"
#include "stm32f4xx_tim.h"
#include "misc.h"



enum  usart_txrx_pins {
	USART1_PA9_PA10, 	USART1_PB6_PB7,
	USART2_PA2_PA3, 	USART2_PD5_PD6,
	USART3_PB10_PB11, 	USART3_PC10_PC11, 	USART3_PD8_PD9,
	UART4_PA0_PA1, 		UART4_PC10_PC11, 
	UART5_PC12_PD2,
	USART6_PC6_PC7};


/*
 * Define the characteristics of the flash storage area.  This is specific
 * to the STM32F407; change these definitions if you are targeting a different
 * STM32F4 device.
 * 
 * Uncomment only one of the following blocks of lines.
 */
#define  FLASH_SECTOR_STORAGE				4		/* ID of flash sector reserved for storage */
#define  FLASH_SECTOR_STORAGE_ADDR			(0x08010000)
#define  FLASH_SECTOR_STORAGE_SIZE			(0x10000)

//#define  FLASH_SECTOR_STORAGE				3		/* ID of flash sector reserved for storage */
//#define  FLASH_SECTOR_STORAGE_ADDR		(0x0800c000)
//#define  FLASH_SECTOR_STORAGE_SIZE		(0x4000)


/*
 * Define the file layout, including the number of files that can be stored in the
 * flash storage area.
 */
#define  NUM_FLASH_FILES			8
#define  SIZEOF_FLASH_FILE			(FLASH_SECTOR_STORAGE_SIZE / NUM_FLASH_FILES)
#define  SIZEOF_FLASH_FILE_HEADER	16				/* header is null-terminated file name */

#define  MAX_LEN_FILENAME			(SIZEOF_FLASH_FILE_HEADER - 1)




/*
 * Define the amount of RAM available for the user's program.  This should be the
 * size of a single flash file.
 */
#define  T_PROGSPACE_SIZE			SIZEOF_FLASH_FILE





/*
 * Define the size (in bits) of various data elements for the STM32F4 target.
 * These definitions apply to the size of stack elements and variables, among
 * other things.
 */
#define		DATA_SIZE			int32_t
#define		DATA_SIZE_MAX		INT32_MAX
#define		UDATA_SIZE			uint32_t
#define		UDATA_SIZE_MAX		UINT32_MAX

/*
 * Define target characteristics and resources of interest to the core program.
 */
 
/*
 * Define available timers
 * 
 * T_NUM_TIMERS     n					number of down-counting timers target can support
 */
#define		T_NUM_TIMERS		4


/*
 * Define available tone channels
 * 
 * T_NUM_TONE_CHS	n					number of TONE channels
 */
#define		T_NUM_TONE_CHS      1



/*
 * Define available character I/O streams and file support
 */
#define		T_STREAM_SERIAL		0
#define		T_STREAM_FILE		1
#define		T_SUPPORTS_FILES	TRUE		/* no file support (yet!) */





extern const unsigned char				ports_tab[];



void			t_ColdBoot(void);				// system prep following reset
void			t_WarmBoot(void);				// system prep following crash
void			t_Shutdown(void);				// clean system shutdown at end of program

void			t_OutChar(int  c);				// output a char to active stream
int				t_GetChar(void);				// waits (blocks) for char from active stream, returns char
int				t_ConsoleCharAvailable(void);	// returns TRUE if user has entered a char at console; does not lock
int				t_GetCharFromConsole(void);		// always waits (blocks) for char from console, ignoring InputStream

void			t_SetTimerRate(unsigned int  usecs);	// sets down-counting timer tic rate in usecs; 0 disables timer
int				t_AddTimer(DATA_SIZE  *t);		// adds a variable at address t to the timer list
int				t_DeleteTimer(DATA_SIZE  *t);	// removes a variable at address to from the timer list

void			t_SetTone(DATA_SIZE  freq, DATA_SIZE  duration);	// starts tone at freq Hz for duration msecs
int				t_CheckToneStatus(void);		// return TRUE if tone still active, else FALSE

int				t_SetOutputStream(int  s);		// assigns the output device for outchar() data from core
int				t_SetInputStream(int  s);		// assigns the input device for inchar() data to core

int				t_FileExistsQ(char  *name);		// returns TRUE if named file exists in buffer
FILE			*t_OpenFile(char  *name, char  *mode);	// open a file for text I/O; closes any open files first
int				t_CloseFile(FILE  *fp);			// close file if open
char			*t_GetFirstFileName(void);		// starts list of file names
char			*t_GetNextFileName(char *ptr);		// gets next file name from list
int				t_DeleteFile(char  *name);

DATA_SIZE		t_ReadPort(unsigned int  index);			// reads a port
void			t_WritePort(unsigned int  index, DATA_SIZE  value);		// writes value to indexed port

int				t_CheckForHWBreak(void);		// returns TRUE if USER pushbutton is pressed

void			t_Test(void);					// triggers a target-specific test routine from TBL								




#endif				// TARGET_LINUX_H
