/*
 *  targetavr.c      AVR-specific routines
 */

#include  <stdio.h>
#include  <string.h>
#include  <time.h>
#include  <avr/sfr_defs.h>
#include  <avr/eeprom.h>
#include  <avr/pgmspace.h>
#include  <avr/io.h>
#include  <avr/interrupt.h>
#include  <avr/boot.h>
#include  <ctype.h>




/*
 *  To build the Basic interpreter for use in the AVR environment, you
 *  must:
 *
 *  1.  Define the literal AVR in your project settings.
 *  2.  Add the directory containing the core Basic interpreter files
 *      to your project's Include search path; this directory is normally
 *      c:\projects\basiccore.
 *  3.  Define the literal OWNER in THIS file and no other!  This definition
 *      causes this file to own all variables defined by the defines.h
 *      file.
 */


#define  OWNER			/* this file defines all variables */


#include  "defines.h"
#include  "funcs.h"
#include  "targetavr.h"



/*
 *  As of 28 Sep 08, the definition for SPM_PAGESIZE for the atmega1284p
 *  is wrong; it is 128 (number of words in a page) rather than 256
 *  (number of bytes in a page).  The following statements correct for this
 *  discrepancy until WinAVR fixes it.
 */
#ifdef  __AVR_ATmega1284P__
#undef  SPM_PAGESIZE
#define SPM_PAGESIZE  256
#endif



/*
 *  These routines are specific to the target hardware.  Here is where
 *  you define routines for non-volatile memory access, serial I/O,
 *  and other elements used by KLBasic in support of the compiler
 *  itself.
 */



#define  USART0RCVQLEN			32				/* number of chars in USART0 rcv queue */


#define  NUM_DCTIMERS			4				/* number of down-counting timers */

#define  UPTIME_FLAG			0xffff			/* special flag for uptime "port" */
#define  TIMER0_FLAG			0xfff0			/* special flag for down-counter 0 */
#define  TIMER1_FLAG			0xfff1			/* special flag for down-counter 1 */
#define  TIMER2_FLAG			0xfff2			/* special flag for down-counter 2 */
#define  TIMER3_FLAG			0xfff3			/* special flag for down-counter 3 */

#define  FCPU_VALUE				0xfff4			/* special flag for F_CPU value */



/*
 *  I/O port table
 *
 *  The AVR version uses named SFRs as I/O ports.  The .addr field is the
 *  standard name of the I/O port, cast as a U16 address.
 */

IOPORT	porttable[]  PROGMEM  =
{
//    name            addr		  size
//----------------------------------------

// Down-counting timers
	{"timer0",		TIMER0_FLAG,	4},
	{"timer1",		TIMER1_FLAG,	4},
	{"timer2",		TIMER2_FLAG,	4},
	{"timer3",		TIMER3_FLAG,	4},
// NV system values, read from EEPROM at boot
	{"f_cpu",		FCPU_VALUE,		4},
// General-purpose I/O ports
	{"porta",		(U16)&PORTA,	1},
	{"portb",		(U16)&PORTB,	1},
	{"portc",		(U16)&PORTC,	1},
	{"portd",		(U16)&PORTD,	1},
	{"ddra",		(U16)&DDRA,		1},
	{"ddrb",		(U16)&DDRB,		1},
	{"ddrc",		(U16)&DDRC,		1},
	{"ddrd",		(U16)&DDRD,		1},
	{"pina",		(U16)&PINA,		1},
	{"pinb",		(U16)&PINB,		1},
	{"pinc",		(U16)&PINC,		1},
	{"pind",		(U16)&PIND,		1},
// ADC registers
	{"adcsra",		(U16)&ADCSRA,	1},
	{"admux",		(U16)&ADMUX,	1},
	{"adcl",		(U16)&ADCL,		1},		// low byte of A/D result; read first!
	{"adch",		(U16)&ADCH,		1},
	{"adc",			(U16)&ADCL,		2},		// reads adcl and adch in one operation
// SPI
	{"spsr",		(U16)&SPSR,		1},
	{"spdr",		(U16)&SPDR,		1},
	{"spcr",		(U16)&SPCR,		1},
// Uptime timer
	{"uptime",		UPTIME_FLAG,	4},


#ifdef  __AVR_AT90CAN128__					// if building for 'can128...
	{"porte",		(U16)&PORTE,	1},
	{"portf",		(U16)&PORTF,	1},
	{"portg",		(U16)&PORTG,	1},
	{"ddre",		(U16)&DDRE,		1},
	{"ddrf",		(U16)&DDRF,		1},
	{"ddrg",		(U16)&DDRG,		1},
	{"pine",		(U16)&PINE,		1},
	{"pinf",		(U16)&PINF,		1},
	{"ping",		(U16)&PING,		1},
#endif
#ifdef  __AVR_ATmega128__					// if building for 'mega128...
	{"porte",		(U16)&PORTE,	1},
	{"portf",		(U16)&PORTF,	1},
	{"portg",		(U16)&PORTG,	1},
	{"ddre",		(U16)&DDRE,		1},
	{"ddrf",		(U16)&DDRF,		1},
	{"ddrg",		(U16)&DDRG,		1},
	{"pine",		(U16)&PINE,		1},
	{"pinf",		(U16)&PINF,		1},
	{"ping",		(U16)&PING,		1},
#endif
#ifdef  __AVR_ATmega1284P__
	{"adcsrb",		(U16)&ADCSRB,	1},
	{"didr0",		(U16)&DIDR0,	1},
#endif

// Null entry, marks end of table
	{"",			0,				0}		// last entry MUST be blank!
};



/*
 *  Defines used for support of flash save/load.
 */

typedef struct
{
	U16					hdr;
	U16					prog;
}  FLASHBLK;


#define  FLASH_DIR_PAGE				256L
#define  FLASH_PROGRAM_NUM_PAGES	(16*4)		/* 16K at 256 bytes per K */

#define  FLASH_PROGRAM_SIZE			(FLASH_PROGRAM_NUM_PAGES*SPM_PAGESIZE)

#define  FLASH_HDR0_PAGE			(FLASH_DIR_PAGE+1L)
#define  FLASH_PROG0_START_PAGE		(FLASH_HDR0_PAGE+1L)

#define  FLASH_HDR1_PAGE			(FLASH_PROG0_START_PAGE+FLASH_PROGRAM_NUM_PAGES)
#define  FLASH_PROG1_START_PAGE		(FLASH_HDR1_PAGE+1L)

#define  FLASH_HDR2_PAGE			(FLASH_PROG1_START_PAGE+FLASH_PROGRAM_NUM_PAGES)
#define  FLASH_PROG2_START_PAGE		(FLASH_HDR2_PAGE+1L)



/*
 *  Local functions
 */
static unsigned char  				EEReadB(unsigned int  addr);
static unsigned int  				EEReadW(unsigned int  addr);
static void  						EEWriteB(unsigned int  addr, unsigned char  b);
static void  						EEWriteW(unsigned int  addr, unsigned int  w);
static BOOTLOADER_SECTION void 		bootldr_write_flash_page(U32 address, U8 *buf);
static U16  						getunum(void);
static I16  						getflashfilenum(void);
static void  						cesave(void);
static void  						cflsave(void);
static void  						ceload(void);
static void  						cflload(I16  fnum);
static void							GetNVSysValues(void);



/*
 *  Local variables
 */
static unsigned long int			tics;			// up-time in msecs
static U32							timers[NUM_DCTIMERS];	// holds four down-counting timers
static volatile unsigned char		usart0rcvq[USART0RCVQLEN];
static volatile unsigned char		usart0rcvin;
static volatile unsigned char		usart0rcvout;
static unsigned char				ans[20];
static U32							fcpu;			// loaded from EEPROM upon boot
static U32							baud;			// loaded from EEPROM upon boot

static U8							flashbuff[SPM_PAGESIZE];

static FLASHBLK						flpagetable[NUM_FLASH_FILES] =
{
	{FLASH_HDR0_PAGE, FLASH_PROG0_START_PAGE},
	{FLASH_HDR1_PAGE, FLASH_PROG1_START_PAGE},
	{FLASH_HDR2_PAGE, FLASH_PROG2_START_PAGE}
};



/*
 *  Table of target-specific constants
 *
 *  You can add constants to this table as needed.  Each constant
 *  consists of a string holding the name as it will be typed by the
 *  user and a 32-bit value for that constant.
 *
 *  The last entry in the table must contain an empty string.
 */

CONST  consttable[]  PROGMEM  =
{
//    name            value
//-------------------------
//	{"f_cpu",			(I32)F_CPU},
	{"",				0}				// last entry must be blank
};



/*
 *  Macros for creating a table of NV system values in EEPROM.
 *
 *  These macros store a value in EEPROM as MSB first, which is the
 *  same format used by KLBasic's eep32() and eep16() functions.  This
 *  allows you to update the NV system values from within a KLB program
 *  and have the system behave properly on the next reboot.
 */
#define  NVSYSVALUE_16(w)  (((w>>8) & 0xff)), (w & 0xff)
#define  NVSYSVALUE_32(x)  NVSYSVALUE_16((x>>16)), NVSYSVALUE_16(x)



/*
 *  EEPROM table of NV system values.  This is an example of an EEPROM
 *  table that can be used to set up the target.
 *
 *  You can also create a separate, stand-alone EEPROM file, using this
 *  format, to create a configuration file for a specific target.  If you
 *  then write that EEPROM file to the target and reboot, the target will
 *  use the valued from the EEPROM file, rather than the default values
 *  defined in the target.h file, for setting up the hardware.  This lets
 *  you create custom target setups without having to edit and rebuild the
 *  KLBasic core or target files.
 *
 *  To be consistent with the way KLBasic writes multi-byte values to
 *  EEPROM, multi-byte data in this table must always be written MSB first!
 */
const U8  eesysvalues[]  EEMEM =
{
	NVSYSVALUE_32(F_CPU),
	NVSYSVALUE_32(38400),
	NVSYSVALUE_32((F_CPU + 38400))					// calc 32-bit checksum of all values saved
};






/*
 *  The main entry point for the KLBasic interpreter.  This routine consists
 *  of a call to basic_main().  Control never leaves basic_main();
 */

int  main(void)
{
	basic_main();
	return  1;						// should never happen!
}




/*
 *  targetinitvars      set up target-specific variables
 *
 *  This routine is called from the KLBasic core and sets up global buffers
 *  and pointers, based on the amount of RAM available.
 */
void  targetinitvars(void)
{
	unsigned char				*x;

	varbegin = varram;									// point to start of variable RAM
	varend = varram;									// show no variables defined yet
	varmend = &varram[MAX_VARRAM-1];					// point to end of variable RAM
	for (x=varbegin; x<=varmend; x++) *x=0;				// zero all of variable memory

	basbeg = prgram;									// point to start of program RAM
	basend = prgram;									// show no program stored yet
	basmend = &prgram[MAX_PRGRAM-1];					// point to end of program RAM
}



/*
 *  getconst      support for target-specific constants
 *
 *  Upon entry,the global input buffer pointer (ibufptr) must be pointing
 *  to a string.  The argument pval must point to a variable that will
 *  be written with the value of the constant, if that constant exists.
 *
 *  This routine returns false if the token was not found, and the variable
 *  pointed to by pval will be unchanged.  This routine returns the number
 *  of chars in the token if the token was found, and the variable pointed
 *  to by pval will be overwritten with the constant's value.
 */
int  getconst(I32  *pval)
{
	unsigned char			n;						// index into table of constant names
	char					tname[MAX_NAME_LEN+1];	// temp copy for the name search

	for (n=0; ; n++)								// for all entries in constant table...
	{
		if (strlen_P(consttable[n].name) == 0)  break;	// if hit the end, outta here
		strcpy_P(tname, consttable[n].name);			// get a RAM copy of the name
		if (match(tname))							// if found the desired port name...
		{
			*pval = pgm_read_dword((U32 *)&consttable[n].value);			// update variable
			return  strlen(tname);					// show we're good
		}
	}
	return 0;									// if no luck with name, all done
}




/*
 *  targetdisplayheader      display target-specific header on console
 */
void  targetdisplayheader(void)
{
	pl_P(PSTR("\n\r"  TARGET_INFO  " v"  TARGET_VERSION));
}




/*
 *  targetgetportaddr      return address of selected port or timer
 *
 *  This routine returns the address of the indexed entry in the
 *  target-specific I/O port table.
 *
 *  In general, the indexed entry will point to a value that can be
 *  modified.  This works for ports such as the timers or other
 *  ports with an associated variable.
 *
 *  However, the "port" for target clock frequency, F_CPU, can not
 *  be modified.  In this case, this routine returns 0.
 */
U32  targetgetportaddr(U16  index)
{
	U16								taddr;			// temp copy of the port address
	unsigned char					n;				// holds temp index into timer array

	taddr = pgm_read_word(&porttable[index].addr);	// get temp copy of address

	if ((taddr >= TIMER0_FLAG) && (taddr <= TIMER3_FLAG))	// if a down-counting timer...
	{
		n = taddr & 0x000f;							// leave index into timer table
		return  (U32)&timers[n];
	}
	else if (taddr == UPTIME_FLAG)					// if trying to read uptime...
	{
		return  (U32)&tics;
	}
	else if (taddr == FCPU_VALUE)					// if trying to read system clock frequency...
	{
		return  (U32)0;								// this is bad, user not allowed to write to F_CPU port address!
	}
	else    return  (U32)taddr;						// default is the address of a port
}




/*
 *  targetgetvarptr      return pointer to a variable in the varram array
 *
 *  This routine uses the contents of addr as an offset into the variable
 *  array varram[], and returns a pointer to the requested variable.
 */
U32  *targetgetvarptr(U16  addr)
{
	return  (U32 *)(varram + addr);
}




/*
 *  targetgetdynmeminfo      set up dynamic memory pointers
 *
 *  This routine sets up the global pointer for the start of dynamic memory
 *  and the variable holding the size of the dynamic memory buffer, dynmem[].
 */
void  targetgetdynmeminfo(U8 **start, U16  *numbytes)
{
	*start = dynmem;						// save addr of dynamic memory array
	*numbytes = MAX_DYN_MEM;				// save number of bytes in array
}




/*
 *  targetcheckautorun      prepare for auto-running an app, if possible
 *
 *  This routine tests the EEPROM-based autorun flag to determine which, if
 *  any, stored program to execute upon reset.
 *
 *  If a program is to be autorun, this routine loads the program from
 *  the selected file, then returns TRUE.  If no program is to be autorun,
 *  this routine returns FALSE.
 */
int  targetcheckautorun(void)
{
	U8						flag;			// holds autostart flag

	flag = targetreadautostart();			// find out where to autostart from

	switch  (flag)							// based on type of autostart...
	{
		case  AUTOST_EE:					// start from EEPROM
		ceload();							// do the load
		return  TRUE;						// all OK

		case  AUTOST_OFF:					// nothing to run...
		return  FALSE;						// should not happen!

		case  AUTOST_FL0:					// flash file 0
		cflload(0);							// load and run
		return  TRUE;						// all OK

		case  AUTOST_FL1:					// flash file 1
		cflload(1);							// load and run
		return  TRUE;						// all OK

		case  AUTOST_FL2:					// flash file 2
		cflload(2);							// load and run
		return  TRUE;						// all OK

		default:							// should never happen
		return  FALSE;						// not OK!
	}
}




/*
 *  targetreadautostart      return the AUTOST value now in EEPROM
 *
 *  This routine reports the current state of the autostart flag in
 *  EEPROM.  This allows the core routines, notably the code for
 *  handling the AUTOST command, to query the current autostart 
 *  device.
 */

U8  targetreadautostart(void)
{
	return  EEReadB(EEOFF_AUTOST);
}




/*
 *  targetmarkautostart      set/clear the non-volatile autostart flag
 *
 *  This routine clears the autostart flag in EEPROM if turning off
 *  the autostart flag, else it sets the autostart flag to show the
 *  file to use for autostart on subsequent reboots.
 *
 *  This routine also checks to see if the requested file actually
 *  has a program stored in it; if not, the request is ignored with
 *  a message to the console.
 */
void  targetmarkautostart(int  flag)
{
	unsigned int			tstart;
	U8						fnum;
	U16						n;
	U32						fladdr;					// addr to access in flash
	
	if (flag == AUTOST_OFF)				// if turning off autostart...
	{
		EEWriteB(EEOFF_AUTOST, 0xff);	// that always works
		pl_P(PSTR("Autostart flag removed."));
		return;
	}
	else if (flag == AUTOST_EE)
	{
		tstart = EEReadW(EEOFF_BASBEG);	// test the recorded BASBEG value
		if (tstart == 0xffff)			// if nothing there...
		{
			pl_P(PSTR("No program saved to EEPROM!"));
		}
		else
		{
			EEWriteB(EEOFF_AUTOST, flag);	// write the autostart flag
			pl_P(PSTR("Program in EEPROM marked for autostart."));
		}
		return;
	}
	else if (flag == AUTOST_FL0)  fnum = 0;
	else if (flag == AUTOST_FL1)  fnum = 1;
	else if (flag == AUTOST_FL2)  fnum = 2;
	else  return;						// internal error, bad autostart flag (should never happen!)

	fladdr = (U32)flpagetable[fnum].hdr * SPM_PAGESIZE;	// set up to read header for this flash file
	n = pgm_read_word_far((U32)(fladdr+EEOFF_BASBEG));	// get start addr of basic program
	if (n == 0xffff)									// if nothing there...
	{
		pl_P(PSTR("\n\rNo program saved in flash file "));
		outdeci(fnum);
		return;
	}
	EEWriteB(EEOFF_AUTOST, flag);						// write the autostart flag
	pl_P(PSTR("Program in flash file "));
	outdeci(fnum);
	pl_P(PSTR(" marked for autostart."));
}



/*
 *  targetwriteeeprom      		writes one byte to EEPROM
 */
void  targetwriteeeprom(U32  addr, U8  val)
{
	eeprom_write_byte((U8 *)(U16)addr, val);
}


/*
 *  targetwriteeeprom16      writes one word to EEPROM, MSB first
 */
void  targetwriteeeprom16(U32  addr, U16  val)
{
	eeprom_write_byte((U8 *)(U16)addr, (val >> 8));
	eeprom_write_byte((U8 *)(U16)addr+1, val);
}


/*
 *  targetwriteeeprom32      writes one double word to EEPROM, MSB first
 */
void  targetwriteeeprom32(U32  addr, U32  val)
{
	targetwriteeeprom16(addr, (val>>16));
	targetwriteeeprom16(addr+2, (val&0xffff));

}


/*
 *  targetreadeeprom      reads one byte from EEPROM, returns as U32
 */
U32  targetreadeeprom(U32  addr)
{
	return  eeprom_read_byte((U8 *)(U16)addr);
}


									
/*
 *  targetreadeeprom16      reads one word from EEPROM, returns U32
 *
 *  Value is assumed to be stored MSB first.  For example, $1234 would be
 *  stored as:
 *  addr:  xx xx 12 34 xx xx xx xx 
 *
 */
U32  targetreadeeprom16(U32  addr)
{
	U32					res;

	res = eeprom_read_byte((U8 *)(U16)addr);
	res = (res << 8) + eeprom_read_byte((U8 *)(U16)(addr+1));

	return  res;
}


/*
 *  targetreadeeprom32      reads one double word from EEPROM, returns U32
 *
 *  Value is assumed to be stored MSB first.  For example, $12345678 would be
 *  stored as:
 *  addr:  xx xx 12 34 56 78 xx xx
 *
 */
U32  targetreadeeprom32(U32  addr)
{
	U32					res;

	res = targetreadeeprom16(addr);
	res = (res << 16) + targetreadeeprom16(addr+2);

	return  res;
}




/*
 *  Routines for processing system time
 *
 *  These are legacy routines from an early attempt to provide time
 *  support for the AVR targets.  I was not happy with the result and
 *  stubbed out the routines in the core.  The AVR code you see
 *  here was left in so you could see where I was going with it.
 *
 *  Essentially, you need to develop a generic timing device and
 *  a set of routines to support it.  I couldn't come up with a RTC
 *  that met all my needs for all AVR systems I was supporting, so
 *  I just got rid of RTC support.   (KEL  9 May 2012)
 */


/*
 *  targetgetsystime      returns time (only!) in broken format
 */
U32  targetgetsystime(struct tm  *pt)
{
	pt->tm_sec = 0;						// stubbed until RTC support is added
	pt->tm_min = 0;
	pt->tm_hour = 0;

	return  0;
}


/*
 * targetsetsystime      updates time (only!), using broken format
 */
void  targetsetsystime(struct tm  *pt)
{
}




/*
 *  targetgetsysdate      returns date (only!) in broken format
 */
U32  targetgetsysdate(struct tm  *pt)
{
	pt->tm_wday = 1;					// 1 Jan 12; stubbed until RTC support is added
	pt->tm_mday = 1;
	pt->tm_yday = 1;
	pt->tm_mon = 0;
	pt->tm_year = 112;

	return  0;
}





/*
 *  targetsetsysdate      updates date (only!), using broken format
 */
void  targetsetsysdate(struct tm  *pt)
{
}




		
	

/*
 *  EEPROM storage support routines
 *
 *  For the AVR version of KLBasic, non-volatile storage is kept in
 *  the EEPROM memory and in flash memory above the interpreter code.
 *
 */

static unsigned char  EEReadB(unsigned int  addr)
{
	return  eeprom_read_byte((void *)addr);
}



static unsigned int  EEReadW(unsigned int  addr)
{
	unsigned int			r;

	r = EEReadB(addr);
	r = (r<<8) + EEReadB(addr+1);
	return  r;
}




static void  EEWriteB(unsigned int  addr, unsigned char  b)
{
	if (b != EEReadB(addr))				// if need to change the target addr...
	{
		eeprom_write_byte((void *)addr, b);
	}
}




static void  EEWriteW(unsigned int  addr, unsigned int  w)
{
	EEWriteB(addr, (unsigned char)(w>>8));
	EEWriteB(addr+1, (unsigned char)(w&0xff));
}






/*
 *  csave      save program to either flash or EEPROM
 *
 *  This routine performs the actions for the SAVE command,
 *  entered at the console.  I moved this entire command out
 *  of the core, since its internal design is so target-
 *  dependent.
 *
 *  This routine parses a string from the console, using the
 *  global pointer ibufptr.  It expects an argument defining
 *  either EEPROM or a flash file.  Command is of the form:
 *
 *  save			; no argument means EEPROM
 *  save  ee		; ee specifies EEPROM
 *  save  fl<n>		; fl and number (0-n) specifies a flash file
 *
 */
void  csave(void)
{
	if (basbeg == basend)							// if no program in RAM...
	{
		pl_P(PSTR("No program in RAM; nothing saved."));
		return;
	}

	skipspcs();										// move to file type, if any
	if (*ibufptr == CR)								// if nothing specified...
	{
		cesave();									// save to EEPROM
	}
	else if ((tolower(*ibufptr) == 'e') &&			// else if explicitly EEPROM...
			 (tolower(*(ibufptr+1)) == 'e'))
	{
		cesave();									// save to EEPROM
	}
	else if ((tolower(*ibufptr) == 'f') &&			// else if flash...
			 (tolower(*(ibufptr+1)) == 'l'))
	{	
		ibufptr++;									// step over file type
		ibufptr++;
		cflsave();									// save to flash
	}
	else
	{
		pl_P(PSTR("Unknown argument to SAVE; use 'fl<n>' to save to FLASH"));
		pl_P(PSTR("\n\ror 'ee' to save to EEPROM."));
	}
}





/*
 *  cesave()      save program and variables to EEPROM
 *
 *  This is a helper function for csave() and saves the current program
 *  in RAM to EEPROM.  It should never be called directly by the core.
 */
static void  cesave(void)
{
	unsigned int				count;
	unsigned int				progbytes;
	unsigned char				*bptr;
	unsigned int				n;
	unsigned char				ans[20];

	n = EEReadW(EEOFF_BASBEG);						// test the recorded BASBEG value
	if (n != 0xffff)								// if something already there...
	{
		pl_P(PSTR("This will erase the program now in EEPROM!  Are you sure? "));
		targetgets(ans);
		if (tolower(*ans) != 'y')  return;
	}

	count = EEOFF_EEPROG;							// compute number of EEPROM bytes needed...
	count += (basend - basbeg);						// add in bytes of program space
	count += (varend - varbegin);					// add in bytes of variable space
	if (count > EESIZE)								// if total RAM used exceeds available program EEPROM...
	{
		errcode = EETOSMAL;							// show not enough room in EEPROM
		rpterr();									// tell the world
		return;
	}
	EEWriteB(EEOFF_AUTOST, 0xff);					// smudge the autostart flag
	EEWriteW(EEOFF_BASBEG, 0);						// Basic program ALWAYS starts at index 0 in PRGRAM
	EEWriteW(EEOFF_BASEND, (unsigned int)(basend-basbeg));	// save index to end of program in PRGRAM
	EEWriteW(EEOFF_VARBEGIN, 0);					// variables ALWAYS start at index 0 in VARRAM
	EEWriteW(EEOFF_VAREND, (unsigned int)(varend-varbegin));	// save index to end of variables in VARRAM
	progbytes = basend - basbeg;					// compute number of program bytes to save
	bptr = basbeg;									// start at beginning of program in RAM
	for (n=0; n<progbytes; n++)						// for all bytes in program...
	{
		EEWriteB(EEOFF_EEPROG+n, *bptr);			// write a program byte
		bptr++;										// move to next cell
	}
	count = varend - varbegin;						// compute number of bytes in variable area
	bptr = varbegin;								// start at beginning of variable area in RAM
	for (n=0; n<count; n++)							// for all bytes in variable area
	{
		EEWriteB(EEOFF_EEPROG+progbytes+n, *bptr);	// write a variable byte
		bptr++;										// move to next cell
	}

	pl_P(PSTR("Saved "));
	outdeci(progbytes);
	pl_P(PSTR(" program bytes and "));
	outdeci(count);
	pl_P(PSTR(" data bytes to EEPROM."));
}

/*
 * CESAVE:   EQU    *
 *           LDD    BASBEG       ; GET POINTER TO THE START OF THE BASIC PROGRAM.
 *           CPD    BASEND       ; IS THERE A PROGRAM IN MEMORY?
 *           BNE    CESAVE1      ; YES. GO SAVE IT.
 *           RTS                 ; NO. RETURN.
 * CESAVE1:  LDD       VAREND
 *           SUBD      BASBEG
 *           CPD       EESIZE
 *           BLS       CESAVE5
 *           LDAA      #EETOSMAL
 *           JMP       RPTERR
 * CESAVE5:  LDX       EEStart   ; point to the start of the EEPROM.
 *           LDY       #BASBEG
 *           LDAB      #4
 *           STAB      COUNT
 * CESAVE3:  LDD       0,Y
 *           SUBD      RAMSTART
 *           STAA      0,X
 *           JSR       DLY10MS
 *           INX
 *           TBA
 *           STAA      0,X
 *           JSR       DLY10MS
 *           INX
 *           INY
 *           INY
 *           DEC       COUNT
 *           BNE       CESAVE3
 * *
 *           LDD       0,Y
 *           STAA      0,X
 *           JSR       DLY10MS
 *           INX
 *           TBA
 *           STAA      0,X
 *           JSR       DLY10MS
 * 
 * *
 *           LDX       EEStart
 *           LDY       BASBEG
 * CESAVE4:  LDAA      0,Y
 *           STAA      SSTART,X
 *           JSR       DLY10MS
 *           INX
 *           INY
 *           CPY       VAREND
 *           BLS       CESAVE4
 *           RTS                    ;RETURN.
 */




/*
 *  cload      load program from EEPROM or a flash file.
 *
 *  This routine performs the actions for the LOAD command,
 *  entered at the console.  I moved this entire command out
 *  of the core, since its internal design is so target-
 *  dependent.
 *
 *  This routine parses a string from the console, pointed to
 *  by the global pointer ibufptr.  Command is of the form:
 *
 *  load					; no arg implies EEPROM
 *  load  ee				; explicitly specifies EEPROM
 *  load  fl<n>				; specifies flash file n (0-2)
 *
 */
void  cload(void)
{
	I16							fnum;

	skipspcs();										// move to file type, if any
	if (*ibufptr == CR)								// if nothing specified...
	{
		ceload();									// load from EEPROM
	}
	else if ((tolower(*ibufptr) == 'e') &&			// else if explicitly EEPROM...
			 (tolower(*(ibufptr+1)) == 'e'))
	{
		ceload();									// load from EEPROM
	}
	else if ((tolower(*ibufptr) == 'f') &&			// else if flash...
			 (tolower(*(ibufptr+1)) == 'l'))
	{	
		ibufptr++;									// step over file type
		ibufptr++;
		fnum = getflashfilenum();
		if (fnum == -1)  return;					// if file number is out of range, done
		cflload(fnum);								// load from flash
	}
	else
	{
		pl_P(PSTR("Unknown argument to LOAD; use 'fl<n>' to load from FLASH"));
		pl_P(PSTR("\n\ror 'ee' to load from EEPROM."));
	}
}






/*
 *  ceload      load program and variables from EEPROM to RAM
 *
 *  This is a helper function for cload(); it should never be called
 *  directly by the core.
 */
static void  ceload(void)
{
	unsigned int				count;
	unsigned int				progbytes;
	unsigned char				*bptr;
	unsigned int				n;

	n = EEReadW(EEOFF_BASBEG);						// test the recorded BASBEG value
	if (n == 0xffff)								// if nothing there...
	{
		pl_P(PSTR("\n\rNo program saved in EEPROM!"));
		return;
	}
	if (basbeg != basend)							// if something in program area...
	{
		pl_P(PSTR("This will overwrite the current program in RAM!  Are you sure? "));
		targetgets(ans);
		if (tolower(*ans) != 'y')  return;
	}

	basbeg = prgram + EEReadW(EEOFF_BASBEG);		// Basic program ALWAYS starts at beginning of program RAM
	basend = basbeg + EEReadW(EEOFF_BASEND);		// calc end addr of Basic program in RAM
	varbegin = varram + EEReadW(EEOFF_VARBEGIN);	// variables ALWAYS start at beginning of variable RAM
	varend = varbegin + EEReadW(EEOFF_VAREND);		// get end addr of variable storage in RAM

	progbytes = basend - basbeg;					// compute number of program bytes to read
	bptr = basbeg;									// start at beginning of program in RAM
	for (n=0; n<progbytes; n++)						// for all bytes in program...
	{
		*bptr = EEReadB(EEOFF_EEPROG+n);			// read a program byte
		bptr++;										// move to next cell
	}
	count = varend - varbegin;						// compute number of bytes in variable area
	bptr = varbegin;								// start at beginning of variable area in RAM
	for (n=0; n<count; n++)							// for all bytes in variable area
	{
		*bptr = EEReadB(EEOFF_EEPROG+progbytes+n);	// get a variable byte
		bptr++;										// move to next cell
	}

	hiline = findhlin();
	pl_P(PSTR("Loaded "));
	outdeci(progbytes);
	pl_P(PSTR(" program bytes and "));
	outdeci(count);
	pl_P(PSTR(" data bytes from EEPROM."));
}


/*  
 * CELOAD:   EQU    *
 *           LDX       EEStart   ; point to the start of the program storage EEPROM.
 *           LDY       #BASBEG   ; point to the start of the program pointer storage area.
 *           LDAB      #4        ; number of words to move.
 *           STAB      COUNT     ; save the count.
 * CELOAD3:  LDD       0,X       ; get the offset that was saved.
 *           ADDD      RAMSTART  ; add the starting address of the RAM to it.
 *           STD       0,Y       ; save the resulting pointer
 *           INX                 ; point to the next offset.
 *           INX
 *           INY                 ; point to the next pointer in RAM
 *           INY
 *           DEC       COUNT     ; have we gotten all the pointers yet?
 *           BNE       CELOAD3   ; no. keep going.
 * *
 *           LDD       0,X       ; yes. get the high line number.
 *           STD       0,Y       ; save it in RAM.
 * 
 * *
 * *         now load the actual program from EEPROM
 * *
 *           LDX       EEStart   ; point to the start of the EEPROM
 *           LDY       BASBEG    ; point to the start of the BASIC program buffer.
 * CELOAD4:  LDAA      SSTART,X  ; get a byte of the program.
 *           STAA      0,Y       ; put it in the program buffer.
 *           INX                 ; point to the next program byte
 *           INY                 ; point to the next buffer location.
 *           CPY       VAREND    ; have we finished loading the program.
 *           BLS       CELOAD4   ; no. keep loading.
 *           STY       STRASTG   ; yes. initialize the array storage area.
 *           RTS                 ; RETURN.
 *           
 */



/*
 *  Flash support (program save and load)
 */


/*
 *  cflsave      save program in RAM to selected flash file
 *
 *  This is a helper function for csave(); it should never be called
 *  directly by the core.
 */
static void  cflsave(void)
{
	U32						tpage;					// holds page of interest
	U32						fladdr;					// addr to write in flash
	U16						flashcnt;				// counts bytes written to flash buffer
	U16						n;
	I16						fnum;					// holds flash file number
	U16						pagecnt;				// counts flash pages written
//	char					tbuff[40];				// debug

	fnum = getflashfilenum();
	if (fnum == -1)  return;						// if file number is out of range, done

	fladdr = (U32)flpagetable[fnum].hdr * SPM_PAGESIZE;
	n = pgm_read_word_far((U32)(fladdr+EEOFF_BASBEG));	// get start of basic program
	if (n != 0xffff)								// if something already there...
	{
		pl_P(PSTR("This will erase the program now in flash file "));
		outdeci(fnum);
		pl_P(PSTR("!  Are you sure? "));
		targetgets(ans);
		if (tolower(*ans) != 'y')  return;
	}

	count = EEOFF_EEPROG;							// compute number of flash bytes needed...
	count += (basend - basbeg);						// add in bytes of program space
	count += (varend - varbegin);					// add in bytes of variable space
	if (count > FLASH_PROGRAM_SIZE)					// if total RAM used exceeds available program flash...
	{
		errcode = FLTOSMAL;							// show not enough room in flash
		rpterr();									// tell the world
		return;
	}

//	sprintf(tbuff, "\n\rSPM_PAGESIZE=%u\n\r", SPM_PAGESIZE);
//	pl(tbuff);

	flashbuff[EEOFF_AUTOST] = 0xff;					// smudge the autostart flag
	flashbuff[EEOFF_BASBEG] = 0;					// Basic program ALWAYS starts at index 0 in PRGRAM
	flashbuff[EEOFF_BASBEG+1] = 0;
	flashbuff[EEOFF_BASEND] = ((U16)(basend-basbeg)) & 0xff;	// save end of program
	flashbuff[EEOFF_BASEND+1] = ((U16)(basend-basbeg)) >> 8;
	flashbuff[EEOFF_VARBEGIN] = 0;					// variables always start at index 0 in VARRAM
	flashbuff[EEOFF_VARBEGIN+1] = 0;
	flashbuff[EEOFF_VAREND] = ((U16)(varend-varbegin)) & 0xff;
	flashbuff[EEOFF_VAREND+1] = ((U16)(varend-varbegin)) >> 8;
	flashbuff[EEOFF_DYNBEG] = 0;						// dynamic pool always starts at index 0 in DYNMEM
	flashbuff[EEOFF_DYNBEG+1] = 0;
	flashbuff[EEOFF_DYNEND] = (strastg - dynmem) & 0xff;	// save amount of memory used in dynamic pool
	flashbuff[EEOFF_DYNEND+1] = (strastg - dynmem) >> 8;

	fladdr = (U32)flpagetable[fnum].hdr * SPM_PAGESIZE;	// calc addr of desired header
	bootldr_write_flash_page(fladdr, flashbuff);	// write new header
//			outhexbyte(fladdr>>8);
//			outhexbyte(fladdr&0xff);
//			outbyte(' ');

//	nl();
//	for (n=0; n< SPM_PAGESIZE; n++)
//	{
//		outhexbyte(flashbuff[n]);
//		outbyte(' ');
//		if ((n % 16) == 15)  nl();
//	}


	pagecnt = 0;									// restart written page count
	tpage = flpagetable[fnum].prog;					// start at first page in program area
	flashcnt = 0;									// tracks bytes written to flash buffer
	for (n=0; n<basend-basbeg; n++)					// for all bytes in program...
	{
		flashbuff[flashcnt++] = prgram[n];			// move a byte into flash buffer
		if (flashcnt == SPM_PAGESIZE)				// if filled the buffer...
		{
			fladdr = tpage*SPM_PAGESIZE;			// calc addr of start of page
			bootldr_write_flash_page(fladdr, flashbuff);	// write the page
			tpage++;								// move to next page
			pagecnt++;								// count this page
			flashcnt = 0;							// rewind flash byte counter
//			outhexbyte(fladdr>>8);
//			outhexbyte(fladdr&0xff);
//			outbyte(' ');
		}
	}
	if (flashcnt)									// if did a partial page...
	{
		fladdr = tpage*SPM_PAGESIZE;				// calc addr of start of page
		bootldr_write_flash_page(fladdr, flashbuff);	// write the page
//			outhexbyte(fladdr>>8);
//			outhexbyte(fladdr&0xff);
//			outbyte(' ');
	}

	tpage++;										// start variable on next page
	pagecnt++;										// count this page
	flashcnt = 0;									// tracks bytes written to flash buffer
	for (n=0; n<varend-varbegin; n++)				// for all bytes in variable area...
	{
		flashbuff[flashcnt++] = varram[n];			// move a byte into flash buffer
		if (flashcnt == SPM_PAGESIZE)				// if filled the buffer...
		{
			fladdr = tpage*SPM_PAGESIZE;			// calc addr of start of page
			bootldr_write_flash_page(fladdr, flashbuff);	// write the page
//			outhexbyte(fladdr>>8);
//			outhexbyte(fladdr&0xff);
//			outbyte(' ');
			tpage++;								// move to next page
			pagecnt++;								// count this page
			flashcnt = 0;							// rewind flash byte counter
		}
	}
	if (flashcnt)									// if did a partial page...
	{
		fladdr = tpage*SPM_PAGESIZE;				// calc addr of start of page
		bootldr_write_flash_page(fladdr, flashbuff);	// write the page
//			outhexbyte(fladdr>>8);
//			outhexbyte(fladdr&0xff);
//			outbyte(' ');
	}

	tpage++;										// start dynamic memory on next page
	pagecnt++;										// count this page
	flashcnt = 0;									// tracks bytes written to flash buffer
	for (n=0; n<(strastg-dynmem); n++)				// for all bytes in dynamic area...
	{
		flashbuff[flashcnt++] = *(dynmem+n);		// move a byte into flash buffer
		if (flashcnt == SPM_PAGESIZE)				// if filled the buffer...
		{
			fladdr = tpage*SPM_PAGESIZE;			// calc addr of start of page
			bootldr_write_flash_page(fladdr, flashbuff);	// write the page
			tpage++;								// move to next page
			pagecnt++;								// count this page
			flashcnt = 0;							// rewind flash byte counter
		}
	}
	if (flashcnt)									// if did a partial page...
	{
		fladdr = tpage*SPM_PAGESIZE;				// calc addr of start of page
		bootldr_write_flash_page(fladdr, flashbuff);	// write the page
	}

	pl_P(PSTR("Saved "));
	outdeci(basend-basbeg);
	pl_P(PSTR(" program bytes and "));
	outdeci(varend-varbegin);
	pl_P(PSTR(" data bytes to flash file "));
	outdeci(fnum);
	outbyte('.');
}
		 



/*
 *  cflload      load program from specified flash file
 *
 *  This is a helper function for cload(); it should never be called directly
 *  by the core.
 */
static void  cflload(I16  fnum)
{
	U32							fladdr;					// addr to read in flash
	U32							tpage;					// holds page of interest
	U32							t;
	U16							count;
	U16							progbytes;
	U8							*bptr;
	U16							n;
	U16							pagecnt;

	fladdr = (U32)flpagetable[fnum].hdr * SPM_PAGESIZE;	// calc location of header block
	n = pgm_read_word_far((U32)(fladdr+EEOFF_BASBEG));		// test the recorded BASBEG value
	if (n == 0xffff)								// if nothing there...
	{
		pl_P(PSTR("\n\rNo program saved in flash file "));
		outdeci(fnum);
		return;
	}
	if (basbeg != basend)							// if something in program area...
	{
		pl_P(PSTR("This will overwrite the current program!  Are you sure? "));
		targetgets(ans);
		if (tolower(*ans) != 'y')  return;
	}

	basbeg = prgram + pgm_read_word_far((U32)(fladdr+EEOFF_BASBEG));	// Basic program ALWAYS starts at beginning of program RAM
	basend = basbeg + pgm_read_word_far((U32)(fladdr+EEOFF_BASEND));	// calc end addr of Basic program in RAM
	varbegin = varram + pgm_read_word_far((U32)(fladdr+EEOFF_VARBEGIN));	// variables ALWAYS start at beginning of variable RAM
	varend = varbegin + pgm_read_word_far((U32)(fladdr+EEOFF_VAREND));		// get end addr of variable storage in RAM
	t = pgm_read_word_far((U32)(fladdr+EEOFF_DYNEND));			// get end of dynamic pool
	strastg = dynmem + t;

//	sprintf(tbuff, "\n\rbasbeg=%u  basend=%u  varbegin=%u  varend=%u",
//		(U16)basbeg, (U16)basend, (U16)varbegin, (U16)varend);
//	pl(tbuff);

	progbytes = basend - basbeg;					// compute number of program bytes to read
	bptr = basbeg;									// start at beginning of program in RAM
	tpage = flpagetable[fnum].prog;					// select starting page of program
	fladdr = tpage * SPM_PAGESIZE;				// compute starting read address	
	pagecnt = 0;									// tracks pages read

//	pl_P(PSTR("\n\rReading program area..."));
//	sprintf(tbuff, "\n\rtpage=%lu  fladdr=%lu", tpage, fladdr);
//	pl(tbuff);

	for (n=0; n<progbytes; n++)						// for all bytes in program...
	{
		*bptr = pgm_read_byte_far((U32)(fladdr));	// read a program byte
//		outhexbyte(*bptr);								// debug
//		outbyte(' ');
		bptr++;										// move to next cell
		fladdr++;									// update read addr
		if ((fladdr % SPM_PAGESIZE) == 0)			// if finished reading a page...
		{
			tpage++;								// move to next page
			pagecnt++;								// count this page
			fladdr = tpage * SPM_PAGESIZE;			// compute new addr

//			sprintf(tbuff, "\n\rtpage=%lu  fladdr=%lu", tpage, fladdr);
//			pl(tbuff);

		}
	}
	tpage++;										// vars always start in next page
	pagecnt++;										// count this page
	fladdr = tpage * SPM_PAGESIZE;					// compute starting read address
	count = varend - varbegin;						// compute number of bytes in variable area
	bptr = varbegin;								// start at beginning of variable area in RAM

//	pl_P(PSTR("\n\rReading variable area..."));
//	sprintf(tbuff, "\n\rcount=%u  bptr=%u", count, (unsigned int)bptr);
//	pl(tbuff);
//	sprintf(tbuff, "\n\rtpage=%lu  fladdr=%lu", tpage, fladdr);
//	pl(tbuff);

	for (n=0; n<count; n++)							// for all bytes in variable area
	{
		*bptr = pgm_read_byte_far((U32)(fladdr));	// get a variable byte
		bptr++;										// move to next cell
		fladdr++;									// update read addr
		if ((fladdr % SPM_PAGESIZE) == 0)			// if finished reading a page...
		{
			tpage++;								// move to next page
			pagecnt++;								// count this page
			fladdr = tpage * SPM_PAGESIZE;			// compute new addr (should be same!)
		}
	}

	tpage++;										// dynamic mem always starts in next page
	pagecnt++;										// count this page
	fladdr = tpage * SPM_PAGESIZE;					// compute starting read address
	bptr = dynmem;									// start at beginning of dynamic pool in RAM
	for (n=0; n<(strastg-dynmem); n++)				// for all bytes in dynamic pool...
	{
		*bptr = pgm_read_byte_far((U32)(fladdr));	// get a variable byte
		bptr++;										// move to next cell
		fladdr++;									// move to next flash addr
		if ((fladdr % SPM_PAGESIZE) == 0)			// if finished reading a page...
		{
			tpage++;								// move to next page
			pagecnt++;								// count this page
			fladdr = tpage * SPM_PAGESIZE;			// compute new addr
		}
	}

//	pl_P(PSTR("\n\rFinished loading, look for hiline..."));
	hiline = findhlin();
	pl_P(PSTR("Loaded "));
	outdeci(progbytes);
	pl_P(PSTR(" program bytes and "));
	outdeci(count);
	pl_P(PSTR(" data bytes from flash file "));
	outdeci(fnum);
	pl_P(PSTR("."));
}








/*
 *  Character I/O routines
 */

/*
 *  targetgets      get a string, with editing, from the input device
 *
 *  This routine accepts characters from the console device, until
 *  the user enters a CR.  This routine provides support for backspace
 *  so you can edit the input line.
 *
 *  WARNING!  This routine does NOT check for buffer overflow!
 *
 *  The correct way to implement this is to accept a max character
 *  count from the calling routine.  This in turn requires that the
 *  core routine calling this one must have some way of determining
 *  the size of the target-specific input buffer *OR* the core design
 *  must assume that the input buffer is a standard size (say, 80 chars).
 *
 *  I never came up with a solution I liked, so I ignored the problem.
 */
void  targetgets(unsigned char  *ptr)
{
	unsigned char				cnt;
	unsigned char				c;

	cnt = 0;
	while (1)
	{
		c = inbyte();
		if (c == EOL)				// if got CR...
		{
			*ptr = EOL;
			outbyte('\n');
			outbyte('\r');
			return;
		}
		else if (c == '\b')			// if backing up...
		{
			if (cnt)
			{
				outbyte('\b');
				outbyte(' ');
				outbyte('\b');
				ptr = ptr - 1;
				cnt--;
			}
		}
		else if (c == LF)  ;		// never accept a LF!
		else						// just a char
		{
			outbyte(c);
			*ptr = c;
			ptr++;
			cnt++;
		}
	}
}
		



/*
 *  nl      send CR/LF to output device, usually to the console
 */
void  nl(void)
{
	outbyte(CR);
	outbyte(LF);
	return;
}

/*
 * NL2:      BSR    NL
 * NL:       EQU    *
 *           LDX    #CRLFSTR
 *           BSR    PL
 *           CLR    PRINTPOS     ; SET THE CURRENT PRINT POSITION TO 0.
 *           RTS
 *
 * CRLFSTR:  FCB    $0A,$0D,$00
 */




/*
 *  pl      display a null-terminated string stored in RAM, usually to the console
 */
void  pl(char  *ptr)
{
	unsigned char				c;

	while ((c = *ptr++))					// assignment, not test!
	{
		outbyte_xlate(c);					// output byte with ESCAPE translation
	}
}




/*
 *  pl_P      display a null-terminated string stored in program memory
 */
void  pl_P(char  *ptr)
{
	unsigned char				c;

	while ((c = pgm_read_byte(ptr++)))		// assignment, not test!
	{
		outbyte_xlate(c);					// output byte with ESCAPE translation
	}
}





/*
 *  outbyte_xlate      provide C-style escape sequences for strings
 */
void  outbyte_xlate(unsigned char  c)
{
	static char				esc = FALSE;

   	if (c)									// don't do nulls
   	{
		if (esc)							// if escape sequence is active...
		{
			switch  (c)						// need to escape the character
			{
				case  'e':					// \e = ESCAPE
				c = 0x1b;					// get the ASCII ESCAPE char
				break;

				case  '\\':					// 0x3c = single backslash
				break;

				case  't':					// \t = tab char
				c = 0x09;					// get the TAB char
				break;

				case  'b':					// \b = BKSP char
				c = 0x08;					// get the BKSP char
				break;

				case  'a':					// \a = ALARM (bell) char
				c = 0x07;					// get the BELL char
				break;

				case  'n':					// \n = newline
				c = 0x0a;					// get LF char
				break;

				case  'r':					// \r = CR
				c = 0x0d;					// get CR char
				break;

				case  'f':					// \f = formfeed
				c = 0x0c;					// get FF char
				break;

				default:					// no clue, discard the '\' char
				break;
			}
			esc = FALSE;					// always clears the ESC code
	    	outbyte(c);						// send translated char
		}
		else if (c == '\\')					// if just got backslash...
		{
			esc = TRUE;
		}
		else
		{
	    	outbyte(c);						// send actual char
		}
	}
}




/*
 *  outbyte      send character to active output device
 *
 *  This implementation simply polls until the console (UART0) is
 *  ready to send a character.
 */
void  outbyte(U8  c)
{
	while ((UCSR0A & (1<<UDRE0)) == 0)  ;				// wait until OK to send
	UDR0 = c;											// send the char
}

/*
 * OUTBYTE:  EQU    *
 *           INC    PRINTPOS     ; INCREMENT THE CURRENT PRINT POSITION.
 *           PSHB                ; SAVE THE B-REG.
 *           PSHX                ; SAVE THE X-REG.
 *           LDX    #OUTABLE     ; POINT TO THE OUTPUT VECTOR TABLE.
 * OUTBYTE1: LDAB   DEVNUM       ; GET THE CURRENT DEVICE NUMBER.
 *           ASLB                ; MULT BY 2.
 *           ABX                 ; POINT TO THE ADDRESS OF THE OUTPUT ROUTINE.
 *           LDX    0,X          ; GET THE ADDRESS. HAS THE VECTOR BEEN INITALIZED?
 *           BNE    OUTBYTE2     ; YES. GO OUTPUT THE CHARACTER.
 *           CLR    DEVNUM       ; NO. RESET TO DEVICE #0.
 *           LDAA   #UNINIERR    ; GO REPORT AN UNINITALIZED I/O VECTOR ERROR.
 *           JMP    RPTRERR
 * OUTBYTE2: JSR    0,X          ; GO OUTPUT THE CHARACTER.
 *           PULX                ; RESTORE X.
 *           PULB                ; RESTORE B.
 *           RTS                 ; RETURN.
 */



/*
 *  inbyte      read a character from the active input device, blocks until available
 *
 *  This version blocks until there is a character available in the UART receive
 *  queue.
 */
unsigned char  inbyte(void)
{
	unsigned char			c;

	while (usart0rcvin == usart0rcvout)  ;		// wait while nothing is available...
	cli();										// shut off interrupts for a moment
	c = usart0rcvq[usart0rcvout];				// get char from queue
	usart0rcvout = (usart0rcvout + 1) % USART0RCVQLEN;	// adjust output index
	sei();										// allow interrupts again (forced!!)
	return  c;
}

/*
 * INBYTE:   EQU    *
 *           PSHB                ; SAVE THE B-REG.
 *           PSHX                ; SAVE THE X-REG.
 *           LDX    #INTABLE     ; POINT TO THE INPUT VECTOR TABLE.
 *           BRA    OUTBYTE1     ; GO USE THE SAME CODE AS OUTBYTE.
 * *
 * *
 *           $if       * > $FF00
 *           $fatal    "BASIC Is Too Large"
 *           $endif
 * *
 */



/*
 *  getunum      read unsigned number from input buffer (ibuff)
 *
 *  This is very similar to getlinum() but does not update any global
 *  variables.
 */
static U16  getunum(void)
{
	U16				val;

	val = 0;
	while (isdigit(*ibufptr))
	{
		val = val * 10 + (*ibufptr - '0');			// update the running total
		ibufptr++;									// move to next char
	}
	return  val;
}




/*
 *  getflashfilenum      read flash file number from input buffer (ibuff)
 *
 *  This routine returns -1 if the flash file number is illegal.
 */
static I16  getflashfilenum(void)
{
	I16					fnum;

	skipspcs();										// step over spaces to argument
	if (isdigit(*ibufptr))							// if there is a flash file specified...
	{
		fnum = getunum();							// get flash file number (0-2)
		skipspcs();									// move past arg
	}
	else											// not a numeric argument
	{
		fnum = 0;									// assume file number 0
	}

	if (fnum >= NUM_FLASH_FILES)					// if file number is out of range...
	{
		pl_P(PSTR("No such flash file; must be 0 to "));
		outdeci(NUM_FLASH_FILES-1);
		pl_P(PSTR("."));
		fnum = -1;									// get the illegal flag
	}
	return  fnum;									// report the file number or -1
}




/*
 *  The following routines support use of target-specific I/O ports.
 *  The table provides the name of the port and the target-specific
 *  means of accessing the port.
 *
 *  For the AVR, these routines support table-driven access to the
 *  on-chip I/O registers.
 */




/*
 *  getioport      look up a string in the target port table
 *
 *  This is a translate-time routine.
 *
 *  This routine scans the string currently pointed to by
 *  ibufptr and tries to find a match between that string and
 *  any name entry in the port table.
 *
 *  If a match is found, this routine adds the port variable
 *  token (PVARTOK) to the token buffer cell pointed to by
 *  variable tbufptr, writes the address of the port to the
 *  token buffer, then advances both ibufptr and tbufptr.
 *
 *  If a match is not found, this routine returns 0 and does
 *  not alter either ibufptr or tbufptr.
 */
int  getioport(void)
{
	unsigned char			n;						// index into name buffer
	char					foundit;				// true if found name in table
	char					tname[MAX_NAME_LEN+1];	// temp copy for the name search

	foundit = 0;									// show haven't found name yet
	for (n=0; ; n++)								// for all entries in port name table...
	{
		if (strlen_P(porttable[n].name) == 0)  break;	// if hit the end, outta here
		strcpy_P(tname, porttable[n].name);			// get a RAM copy of the name
		if (match(tname))							// if found the desired port name...
		{
			foundit = 1;							// show we found the name
			break;									// stop looking
		}
	}
	if (!foundit)  return 0;						// if no luck with name, all done
	*tbufptr++ = PVARTOK;							// show we found a port name
	putint16(n);									// save index into port table after token
	return  NUM;									// show all is well
}




/*
 *  targetwriteport      modify the contents of a port address
 *
 *  This is a run-time routine.
 *
 *  This routine writes the value in variable val to a I/O port indexed
 *  by the variable index.
 *
 *  Upon entry, variable index holds the offset of the selected port
 *  entry in the port table.  This routine reads the address of the
 *  port and the size of the port from port table, then writes the
 *  currect number of bytes to that address.
 *
 *  WARNING!  The AVR does not have any 32-bit ports, and the following
 *  code does not support such.  "Ports" that are really variables,
 *  such as the timers, are handled properly.
 */
void  targetwriteport(U16  index, U32  val)
{
	unsigned char					tsize;			// temp copy of the port size
	U16								taddr;			// temp copy of the port address
	unsigned char					n;				// holds temp index into timer array

	tsize = pgm_read_byte(&porttable[index].size);	// get temp copy of size
	taddr = pgm_read_word(&porttable[index].addr);	// get temp copy of address

	if (tsize == 1)									// for byte-wide port...
	{
		*(volatile unsigned char *)(taddr) = val;
	}
	else if (tsize == 2)							// for word-wide port...
	{
		*(volatile unsigned int *)(taddr) = val;
	}
	else if ((taddr >= TIMER0_FLAG) && (taddr <= TIMER3_FLAG))	// if a down-counting timer...
	{
		n = taddr & 0x000f;							// create index into timer table
		cli();										// can't let interrupts happen
		timers[n] = val;							// load the timer
		sei();										// assumes interrupts were enabled!!
	}
	else if (taddr == FCPU_VALUE)					// if trying to write to system clock frequency...
	{
	}
	else	;										// uh oh, something is badly wrong!
}
	



/*
 *  targetreadport      read the contents of a port address
 *
 *  This is a run-time routine.
 *
 *  This routine uses the value in variable index to select an entry in the
 *  port table.  This routine then returns the contents of the associated
 *  address, trimmed to the size of that port as shown in the entry's
 *  .size field.
 *
 *  WARNING!  The AVR does not have any 32-bit ports and this routine
 *  does not support such.  Variable-based "ports", such as the timers,
 *  are handled correctly.
 */
U32  targetreadport(U16  index)
{
	unsigned char					tsize;			// temp copy of the port size
	U16								taddr;			// temp copy of the port address
	U32								tval;			// temp copy of tics
	unsigned char					n;				// holds temp index into timer array

	tsize = pgm_read_byte(&porttable[index].size);	// get temp copy of size
	taddr = pgm_read_word(&porttable[index].addr);	// get temp copy of address

	if (tsize == 1)									// if reading a byte...
	{
		return  (*(volatile unsigned char *)(taddr)) & 0xff;
	}
	else if (tsize == 2)							// if reading a word...
	{
		return  (*(volatile unsigned int *)(taddr)) & 0xffff;
	}
	else if ((taddr >= TIMER0_FLAG) && (taddr <= TIMER3_FLAG))	// if a down-counting timer...
	{
		n = taddr & 0x000f;							// leave index into timer table
		tval = timers[n];							// get value of selected timer
		if (tval != timers[n])  tval = timers[n];	// if interrupted in middle of read, try again
		if (tval != timers[n])  tval = timers[n];	// this one has to work
		return  tval;
	}
	else if (taddr == UPTIME_FLAG)					// if trying to read uptime...
	{
		tval = tics;								// get time since reset
		if (tval != tics)  tval = tics;				// if interrupted in middle of read, try again
		if (tval != tics)  tval = tics;				// this one has to work
		return  tval;
	}
	else if (taddr == FCPU_VALUE)					// if trying to read sytem clock frequency...
	{
		tval = fcpu;
		return  tval;
	}
	else    return 0;								// this is really wrong, just return 0
}




/*
 *  targetgetportname      return user-friendly name of indexed port
 *
 *  This routine copies the name field of an entry in the port table to
 *  the buffer passed in variable namebuf.  The index of the selected
 *  entry is passed in variable index.
 */
void  targetgetportname(U16  index, char  *namebuf)
{
	strcpy_P(namebuf, (char *)&porttable[index].name);
}




/*
 *  targetflashprogramaddr      return address in flash memory of start of flash file
 *
 *  This routine returns the absolute address in flash of the start of the selected
 *  file.
 *
 *  This routine allows the core to reference the first line of a stored
 *  program in one of the flash files.
 */
U32  targetgetflashprogramaddr(U8  filenum)
{
	if ((filenum >= 0) && (filenum < NUM_FLASH_FILES))  return  (U32) ((U32)flpagetable[filenum].prog * SPM_PAGESIZE);
	else												return  (U32) 0;
}





/*
 *  targetcopyfromflash      copies bytes from a flash file to RAM
 *
 *  This routine copies len bytes of data from the absolute flash address
 *  (passed in fladdr) to the RAM buffer pointed to by variable ramptr.
 *
 *  This routine allows the core to read a few bytes from the beginning of
 *  a flash file.
 */
void  targetcopyfromflash(U8  *ramptr, U32  fladdr, U16  len)
{
	U16						n;

	for (n=0; n<len; n++)
	{
		*ramptr = pgm_read_byte_far(fladdr);
		ramptr++;
		fladdr++;
	}
}




/*
 *  The following routines support character I/O via an ACIA.
 *  They were left in as examples of interfacing to the Basic11
 *  character I/O routines.
 */
/*
 *           ORG       $FF00
 * *
 * *
 * ACIAIN:   BSR    ACIAINNE     ; GO GET CHARACTER FROM ACIA, NO ECHO.
 * *                             ; NOW, FALL THROUGH TO ACIAOUT TO ECHO CHARACTER.
 * *
 * *
 * ACIAOUT:  PSHA                ; SAVE THE CHARACTER TO OUTPUT.
 * ACIAOUT1: LDAA   ACIAST       ; GET THE ACIA STATUS.
 *           BITA   #$02         ; IS THE XMIT DATA REGISTER EMPTY?
 *           BEQ    ACIAOUT1     ; NO. WAIT TILL IT IS.
 *           PULA                ; YES. GET BYTE TO SEND.
 *           STAA   ACIADT       ; SEND IT.
 *           RTS                 ; RETURN.
 */

/*
 * ACIAINNE: LDAA   ACIAST       ; GET THE ACIA STATUS.
 *           BITA   #$01         ; HAS A CHARACTER BEEN RECIEVED?
 *           BEQ    ACIAINNE     ; NO. WAIT TILL WE HAVE ONE.
 *           LDAA   ACIADT       ; YES. GET THE CHARACTER.
 *           RTS                 ; RETURN.
 * *
 * ACIASTAT: EQU    *
 *           PSHA                ; SAVE THE A-REG.
 *           LDAA   ACIAST       ; GET THE ACIA STATUS.
 *           BITA   #$01         ; CHECK FOR A CHARACTER.
 *           PULA                ; RESTORE A.
 *           RTS                 ; RETURN.
 */


/*
 * SCIIN:    EQU    *
 *           PSHX                ; Save the index register.
 *           LDX    IOBaseV
 * SCIIN1:   LDAA   SCSR,X       ; GET SCI STATUS.
 *           ANDA   #$20         ; HAS A CHARACTER BEEN RECIEVED?
 *           BEQ    SCIIN1       ; NO. WAIT FOR CHARACTER TO BE RECIEVED.
 *           LDAA   SCDR,X       ; GET THE CHARACTER.
 *           PULX                ; Restore X.
 *           RTS                 ; RETURN.
 * *
 * *
 * SCIOUT:   EQU    *
 *           PSHX                ; Save the index register.
 *           LDX    IOBaseV
 *           PSHA               ; SAVE THE CHARACTER TO SEND.
 * SCIOUT1:  LDAA   SCSR,X      ; GET THE SCI STATUS.
 *           BITA   #$80        ; HAS THE LAST CHARACTER BEEN SHIFTED OUT?
 *           BEQ    SCIOUT1     ; NO. WAIT TILL IT HAS.
 *           PULA               ; RESTORE CHARACTER TO SEND.
 *           STAA   SCDR,X      ; SEND THE CHARACTER.
 *           PULX                ; Restore X.
 *           RTS                ; RETURN.
 * *
 * *
 * SCISTAT:  EQU    *
 *           PSHX                ; Save the index register.
 *           LDX    IOBaseV
 *           PSHA                ; SAVE THE A-REG.
 *           LDAA   SCSR,X       ; GET THE SCI STATUS.
 *           BITA   #$20         ; CHECK TO SEE IF A CHARACTER HAS BEEN RECIEVED.
 *           PULA                ; RESTORE STATUS.
 *           PULX                ; Restore X.
 *           RTS                 ; RETURN W/ STATUS.
 */





/*
 *  iodevinit      set up the target MCU and I/O device tables
 *
 *  This routine configures the console (usually a UART) and
 *  sets up global variables, such as the timers and the
 *  break flag.
 */
void  iodevinit(void)
{
	unsigned char					n;

/*
 *  Set up the receive queue for the console (USART0).
 */
 	usart0rcvin = 0;					// rewind the input pointer
	usart0rcvout = 0;					// rewind the output pointer
	breakflag = 0;						// show no break yet

/*
 *  Retrieve the device's F_CPU and default baud rate from EEPROM, if
 *  it is available.
 */
	GetNVSysValues();

/*
 *  Set up USART0 as the console, interrupt enabled on RcvChar event
 */
    UCSR0A = (1<<U2X0);									// double baud-rate clock
    UCSR0B = (1<<TXEN0) + (1<<RXEN0) + (1<<RXCIE0);	    // enable rx and tx
    UCSR0C = (1<<UCSZ01) + (1<<UCSZ00);                 // 8-bit, no parity, one stop bit
    UBRR0H = (unsigned char)  ((fcpu/(baud*8))-1) >> 8; // baud rate is written as two bytes
    UBRR0L = (unsigned char)  ((fcpu/(baud*8))-1);      // write to this reg triggers change

/*
 *  Set up OCR0A as a real-time clock, generating an interrupt
 *  every one millisecond.  Refer to the block of named literals
 *  above that define the timing of this interrupt.
 */
#ifdef  __AVR_ATmega1284P__				// if building for 'mega1284p...
	TCCR0A = (1<<WGM01);				// CTC operation, OC0A,B not used
	TCCR0B = RTC_PRESCALER_MASK;		// prescale for real-time clock interrupts
	OCR0A = ((fcpu/(1000L*RTC_PRESCALER))-1);	
//	OCR0A = OC_COUNTS_PER_MSEC;			// load count for next interrupt
	TIMSK0 = TIMSK0 | (1<<OCIE0A);		// interrupt on OC
#endif
#ifdef  __AVR_AT90CAN128__				// if building for 'can128...
	TCCR0A = (RTC_PRESCALER_MASK | (1<<WGM01));		// prescale for real-time clock interrupts
	OCR0A = ((fcpu/(1000L*RTC_PRESCALER))-1);	
//	OCR0A = OC_COUNTS_PER_MSEC;			// load count for next interrupt
	TIMSK0 = TIMSK0 | (1<<OCIE0A);		// interrupt on OC
#endif
#ifdef  __AVR_ATmega128__				// if building for 'mega128...
	TCCR0 = RTC_PRESCALER_MASK;			// prescale for real-time clock interrupts
	OCR0 = ((fcpu/(1000L*RTC_PRESCALER))-1);	
//	OCR0 = OC_COUNTS_PER_MSEC;			// load count for next interrupt
	TIMSK = TIMSK | (1<<OCIE0);			// interrupt on OC
#endif

	for (n=0; n<NUM_DCTIMERS; n++)
	{
		timers[n] = 0;
	}
	sei();								// all set up, allow interrupts
}

/*
 * IODevInit:
 *           BSR       InitACIA
 *           BSR       InitSCI
 *           LDAA      #JMPOP
 *           STAA      CONSTAT     ; INITIALIZE THE CONSOLE STATUS VECTOR.
 *           STAA      INCONNE     ; INITIALIZE THE INPUT FROM CONSOLE NO ECHO VECT.
 *           LDD        #ACIASTAT  ; CONSOLE IS INITIALLY THE ACIA.
 *           STD       CONSTAT+1
 *           LDD       #ACIAINNE   ; GET BYTE FROM ACIA, DON'T ECHO IT.
 *           STD       INCONNE+1
 *           RTS
 * *
 * *
 * INITSCI:  EQU    *
 *           PSHX                ; Save the index register.
 *           LDX    IOBaseV
 *           LDAA   #$30        ; SET BAUD RATE TO 9600.
 *           STAA   BAUD,X
 *           CLR    SCCR1,X     ; SET FOR 8 BIT OPERATION, DISABLE WAKEUP.
 *           LDAA   #$0C        ; ENABLE THE TRANSMITER & RECEIVER.
 *           STAA   SCCR2,X
 *           LDAA   #$11        ; GET THE XON CHARACTER (CONTROL-Q).
 *           STAA   XONCH       ; INITALIZE THE XON REGISTER.
 *           LDAA   #$13        ; GET THE XOFF CHARACTER (CONTROL-S).
 *           STAA   XOFFCH      ; INITALIZE THE XOFF CHARACTER.
 *           PULX
 *           RTS                ; RETURN.
 * *
 * *
 * INITACIA: LDAA   #$13         ; VALUE TO RESET THE ACIA.
 *           STAA   ACIAST       ; RESET IT.
 *           LDAA   #$56         ; SET /64, RTS=HI, 8-DATA/1 STOP
 *           STAA   ACIAST
 *           RTS                 ; RETURN.
 * *
 * *
 * *
 * PROUT:    EQU    *            ; SEND A CHARACTER TO THE PRINTER.
 *           BSR    SCISTAT      ; WAS AN "X-OFF" RECIEVED?
 *           BEQ    PROUT1       ; NO. GO SEND THE CHARACTER.
 *           PSHA                ; SAVE THE CHARACTER TO SEND.
 *           BSR    SCIIN        ; YES. GO RESET THE SCI RECEIVER STATUS.
 *           CMPA   XOFFCH       ; WAS IT AN XOFF?
 *           BNE    PROUT2       ; NO. SO GO SEND THE CHARACTER.
 * PROUT3:   BSR    SCIIN        ; GO WAIT FOR AN "X-ON" CHARACTER.
 *           CMPA   XONCH        ; IS IT AN X-ON CHARACTER?
 *           BNE    PROUT3       ; NO. GO WAIT FOR AN X-ON CHARACTER.
 * PROUT2:   PULA                ; GET THE CHARACTER TO SEND.
 * PROUT1:   BRA    SCIOUT       ; SEND THE CHARACTER TO THE PRINTER & RETURN.
 *
 */






/*
 *  GetNVSysValues    load system values from EEPROM, use defaults if not available or valid
 *
 *  This routine sets up global variables for MCU frequency and console baud rate.  If these
 *  values can not be read from non-volatile storage, this routine uses compile-time
 *  defaults instead. 
 */
static void  GetNVSysValues(void)
{
	U32								chk;

	chk = 0;
	fcpu = targetreadeeprom32(EEOFF_FCPU);
	chk += fcpu;

	baud = targetreadeeprom32(EEOFF_BAUDRATE);
	chk += baud;

	if (chk != targetreadeeprom32(EEOFF_CHKSUM))
	{
		fcpu = F_CPU;
		baud = BAUDRATE_DEFAULT;
	}
}




/*
 *  ISR for receiving characters on USART0 (console)
 */
ISR(USART0_RX_vect)
{
	unsigned char					c;

	c = UDR0;

	if (c == BREAK_CHAR)				// if user wants attention...
	{
		breakflag = TRUE;				// show globally
	}
	else
	{
		usart0rcvq[usart0rcvin] = c;
		usart0rcvin = (usart0rcvin + 1) % USART0RCVQLEN;
	}
}




/*
 *  ISR for OC0A output-compare match
 *
 *  This ISR acts as a real-time clock.  The ISR is triggered
 *  each time an output-compare match occurs between TCNT0 and
 *  OCR0A.  This value set by initialization code based on the
 *  target's operating frequency (fcpu) and is nominally one millisecond.
 */
#ifdef  __AVR_ATmega1284P__				// if building for 'mega1284p...
ISR(TIMER0_COMPA_vect)
{
#else									// for all others
ISR(TIMER0_COMP_vect)
{
#endif

	tics++;								// count this tic

	if (timers[0])  --timers[0];
	if (timers[1])  --timers[1];
	if (timers[2])  --timers[2];
	if (timers[3])  --timers[3];
}


/*
 *  ISR for EE_READY (should not need this!)
 */
ISR(EE_READY_vect)
{
//	pl("!!");
}




/*
 *  Flash-write support
 *
 *  This routine erases, then writes, one full page of data to flash
 *  memory.  Upon entry, address holds the address of the first word in
 *  the selected flash page, and buf points to a RAM buffer holding
 *  the data to write.
 *
 *  This routine must reside in the bootloader section of flash so it can
 *  modify the lower pages of flash.
 *
 *  Note that your linker instructions must assign a BYTE address (not
 *  a WORD address) for BOOTLOADER_SECTION.  For 128K flash devices, such
 *  as AT90CAN128, this is normally done with:
 *
 *  LDFLAGS = -Wl,-section-start=.bootloader=0x1fc00
 */

static BOOTLOADER_SECTION void bootldr_write_flash_page(U32 address, U8 *buf)
{
U16 				i;
U8 					sreg;
U16					word;

	sreg = SREG;						// record interrupt settings
	cli();								// shut off interrupts for now

	eeprom_busy_wait();					// let any EEPROM writes finish

	boot_spm_busy_wait(); 				// let any previous flash operation finish
	boot_page_erase(address);			// erase the flash page
	boot_spm_busy_wait(); 				// wait until the memory is erased

	for(i=0;i<SPM_PAGESIZE; i+=2)		// for all bytes in the page...
	{
		word = *buf++;					// create a little-endian word
		word += (*buf++) << 8;			// add in the MSB
		boot_page_fill(address+i, word);	// write to page buffer
	}

	boot_page_write(address);			// write page buffer to flash
	boot_spm_busy_wait(); 				// wait until the memory is written

	boot_rww_enable();					// reenable RWW section

	SREG = sreg;						// restore interrupts to previous state
}
