//--------------------------------------------------------------------------------
//
//                    CX5530 - Audio functions
//
//     This is a group of hardware specific functions that define a general 
//  interface to be portable to other Operating systems and to run on the 
//  National Semiconductor/Cyrix MediaGX CPU and 5530. 
//
//     These functions include interfaces to an AC'97 Codec, Hardware 
//  Initialization and DMA buffer input/Output and currently run on 
//  Unicorn and Scorpius platforms using LM4548 Codecs.
//
//  5530_def.h - Contains the defines and typedefs  
//  5530_glb.h - Contains the Global variables and structures 
//
//  OS_inc.h - Contains the OS-Specific definitions/Macros
//             The driver developer has to define these macros according to the O/S
//             to which the driver is being developed.
//--------------------------------------------------------------------------------

#include "5530_def.h"
#include "OS_inc.h"
#include "5530_glb.h"

//--------------------------------------------------------------------------------
//               Function prototypes
//--------------------------------------------------------------------------------

unsigned char DURAUDIO_Initialize   (unsigned int, unsigned int, unsigned char);
void          DURAUDIO_Deinitialize (void);
void          DURAUDIO_Standby      (void);
unsigned long DURAUDIO_GetVolume    (void);
void          DURAUDIO_SetVolume    (unsigned long);
void          DURAUDIO_WaveStop     (void);

//---------------------------
//                    PCI and IO Port
//---------------------------
unsigned long  DURAUDIO_PortRead32		(unsigned short reg);
void           DURAUDIO_PortWrite32		(unsigned short reg, unsigned long val);
void           DURAUDIO_IO_PCI_Write16  (unsigned long pdev, unsigned short port, unsigned short data);
void           DURAUDIO_IO_PCI_Write32  (unsigned long pdev, unsigned short port, unsigned long data);
unsigned short DURAUDIO_IO_PCI_Read16	(unsigned long pdev, unsigned short port);
unsigned long  DURAUDIO_IO_PCI_Read32	(unsigned long pdev, unsigned short port);


//---------------------------
//                    CODEC Control
//---------------------------
unsigned short DURAUDIO_CodecDirectRegRead	(unsigned short codec_port);
void           DURAUDIO_CodecWrite			(unsigned short codec_port, unsigned short codec_data);
void           DURAUDIO_SetCodecRate		(unsigned long);      
void           DURAUDIO_SetCodecControl		(unsigned long val);
unsigned long  DURAUDIO_CodecStatus			(void );  
unsigned char  DURAUDIO_ClearStatus			(void);  

//---------------------------
//                    DMA Buffers
//---------------------------
PWAVEHDR DURAUDIO_FillBuffer 	(PWAVEHDR pwh, short * pBuffer);
PWAVEHDR DURAUDIO_FillDMABuffer (PWAVEHDR pwh, signed short  * pBuffer);
PWAVEHDR DURAUDIO_FillUserBuffer(PWAVEHDR pwh, signed short  * pBuffer);

//--------------------------------------------------------------------------------

unsigned long DMA_buffers_phys_addr;

static int AD1819A = 0;

//-----------------------------------------------------------------------------
//
//  unsigned char DURAUDIO_Initialize (unsigned char Irq, unsigned short IoBase, unsigned char DmaChan)
//
//  Initializes the 5530 hardware.
//  It has to be called with the resources that were assigned by the OS
//
//  This function:
//
//  - Sets VSA audio irq component to create SMI generated interrupts
//  - Maps the Irg to a system interrupt number
//  - Allocates the DMA Buffers
//  - Sets up the buffer pointers
//  - Checks that the 5530 or 5540 Bridge Device is found
//  - Reset and initialize the CODEC
//
//  Resources assigned by the system:
//
//  unsigned char   Irq, 
//  unsigned short  IoBase, 
//  unsigned char   DmaChan
//
//  In the case of windows drivers, the resources are specified on the registry 
//  and are assigned by the plug-and-play system.
//
//--------------------------------------------------------------------------------
unsigned char DURAUDIO_Initialize (unsigned int Irq, unsigned int IoBase, unsigned char DmaChan)
{
	unsigned long   config_data;       
	unsigned short  i;      
	unsigned long   phys_map_adr; 
	unsigned short  pci_cmd;   
	unsigned long	SetIRQData;

	DURAUDIO_MSG("DURAUDIO_Initialize:\n");
	
	//
	// VSA2 IRQ config method
	//
	OUTP_USHORT( 0xAC1C, 0xFC53 ); //unlock virtual registers
	OUTP_USHORT( 0xAC1C, 0x108 );  //ID the audio config virtual regs
	OUTP_USHORT( 0xAC1E, Irq );  //set the value

	//
	// VSA1 IRQ config method
	//
	//  Get the Irq from the registry and OR it with the command
	//
	SetIRQData = ((((unsigned long) Irq) << 16) | 0xA00A);
	DBG_MSG("SetIRQData = %08X \n", SetIRQData);

	OUTP_ULONG(PCI_CADDR, 0x800090D0);		//  Set the address
	OUTP_ULONG(PCI_CDATA, SetIRQData);		//  Send the command


    //
    //  Maps the Irg to a system interrupt number
    //
	gIntrAudio = MAP_IRQ_TO_SYS_INT(Irq);   
	
    // Allow 0.1 Second for stabilization
	SLEEP(100);             
	
    //
    // Setup Flags for input and output buffers
    //
    //   v_nNextPage:  Points to the next buffer to be filled with data.
    //   v_fMoreData:  Flag that signals if there is more data coming.
    //
	v_nNextPage[WAPI_OUT] = 0;
	v_fMoreData[WAPI_OUT] = FALSE;
	
	v_nNextPage[WAPI_IN]  = 2;
	v_fMoreData[WAPI_IN]  = FALSE;
	
    //
    //        PCI Audio/DMA  init
    //
    //   Go probe for PCI device reg spaces...
    //   NOTE: This loop does not find the PCI Audio Header in the VSA code.
    //         It just checks that the 5530 Bridge Header is found (Need to see
    //         if all the other 'things' are just functions... not devices)
    //
	for(i=0;i<32;i++) 
	{
        // Iterate over all dev numbers in PCI COnfig addr reg.
		pci_config_addr = 0x80000000 | (i<<11) ;  
		
        // Write Reg.
		OUTP_ULONG( (unsigned long *) PCI_CADDR, (unsigned long)pci_config_addr );  
		
        // Get back Vendor_ID+dev_id reg (1st word in PCI Header). 1078h is Cyrix. 100Bh is National.
		config_data = INP_ULONG((unsigned long *) PCI_CDATA ); 
		
        // If device ID is CX5530 device ID or CX5540 (Scorpius)
		if( (UNICORN_BRIDGE == config_data) 
            || (SCORPIUS_BRIDGE == config_data) 
			|| (CARMEL_BRIDGE == config_data))
			goto Found_Device;
	}
	
    //
    //  Didn't find the Device ID.  Return.
    //
    DURAUDIO_MSG("Cant find PCI Audio dev!\n");
	return ( FALSE );
	
    //    
    //   Found the device ID.  Now initialize the device.
    //
	
    Found_Device:;
	
	DURAUDIO_MSG("Found Device! \n");
	
    //
    // Allocates the 4 DMA Buffers
    //
	dma_page[0]=ALLOC_DMA_BUFFER;
	DMA_buffers_phys_addr=PHYSICAL_ADDRESS (dma_page[0]);
    
    if (!dma_page[0])  
    {
      DURAUDIO_MSG("dma_buffers Allocation Failed!!!\n");
      return FALSE;
    }

    //
    // Map the DMA pages into the local address space
    //
    MAP_DMA_PAGES ((void *) dma_page[0],
                   (void *)(AUDIO_DMA_BUFFER_BASE_PA | 0x80000000),
                    DMA_BUFFER_SIZE * 4);
		
    DURAUDIO_MSG("dma_buffers Allocation Worked...\n");

    // 10-13h. BASE REGISTER --> VSA Audio Regs are mem mapped at this loc.
	pci_audio_func_reg = pci_config_addr | PCI_FUNC3_AUDIO;
	
    //
    // PCI Audio and DMA stuff.
    //
	
    // Create an address window to map to the regs.
	
	audio_regs_adr = (unsigned long) ALLOC_SYS_ADDRESS(128);
    //
    //   Fails if system cannot Allocate
    //
	if (!audio_regs_adr)  
	{
        DURAUDIO_MSG("VSA regs mem-map Failed \n");
		return( FALSE );
	}

    //
    //  phys_map_adr is the base of mem-mapped registers.
    //
	
	DURAUDIO_IO_PCI_Write32( pci_audio_func_reg, 0x10, AUDIO_REGS_BUFFER );   
	phys_map_adr = DURAUDIO_IO_PCI_Read32( pci_audio_func_reg, 0x10 );
	
	if( AUDIO_REGS_BUFFER != phys_map_adr ) 
        DURAUDIO_MSG( "Bad AUDIO_REGS_BUFFER \n" );
	
    //
    // Turn on mem-map of PCI audio regs (Enable 5530 audio regs memory).
    //
	pci_cmd = DURAUDIO_IO_PCI_Read16( pci_audio_func_reg, PCI_AUDIO_CMD_REG );
	
    //
    // Enable mem-map. (Boot loader code).
    //
	pci_cmd   |= 0x06;   
	DURAUDIO_IO_PCI_Write16( pci_audio_func_reg, PCI_AUDIO_CMD_REG, pci_cmd );   
	
	// Mem-map the audio regs.
    MAP_IO_ADDRESS ((unsigned long) audio_regs_adr,				 // Logical address.
                    (unsigned long)( (phys_map_adr) | 0x80000000 ), // Physical address.
					 IoBase,								 // IoBase.
                     256);									 // Size of space to map.

	if (!audio_regs_adr)  
	{
        DURAUDIO_MSG("VSA regs mem-map access Failed\n");
		return ( FALSE );
	}    

    //
    // CODEC - RESET and volumes initalization.
    //
    DURAUDIO_MSG("Sending CODEC commands\n");
	
	DURAUDIO_PortWrite32( CODEC_CMD_REG, 0L );  

	//
	//  Check which codec is being used 
	//
	if (DURAUDIO_CodecDirectRegRead(AD1819A_VENDORID1>>8) == 0x4144 &&
	    DURAUDIO_CodecDirectRegRead(AD1819A_VENDORID2>>8) == 0x5303) 
		AD1819A = 1;
	else
		AD1819A = 0;

	if (AD1819A) 
	{
		//  Enable non-48kHz sample rates. 
		DURAUDIO_CodecWrite(AD1819A_SER_CONF, (unsigned short) (DURAUDIO_CodecDirectRegRead(AD1819A_SER_CONF>>8) | AD1819A_SER_CONF_DRQEN));
	}
	else 
		DURAUDIO_CodecWrite(LM4548_EXT_AUDIO_CTRL_STAT, 0x01);	// set the VRA bit to ON


	DURAUDIO_CodecWrite ( LM4548_MASTER_VOLUME,       0x00);	// left/right vol
	DURAUDIO_CodecWrite ( LM4548_PCM_OUT_VOL,         0x00);	// left/right vol
	DURAUDIO_CodecWrite ( LM4548_VIDEO_VOLUME,        0x8000);	// Mute DVD Volume
	DURAUDIO_CodecWrite ( LM4548_TV_VOLUME,           0x8000);	// Mute TV Volume
	DURAUDIO_CodecWrite ( 0x1000,                     0x00);	// left/right vol
	
	DURAUDIO_MSG("DURAUDIO_Initialize:  FINISHED\n");

	return ( TRUE );
}

//--------------------------------------------------------------------------
// 
//  MMRESULT DURAUDIO_WaveOpen (WAPI_INOUT apidir,
//                     LPWAVEFORMATEX lpFormat, 
//                     unsigned char fQueryFormatOnly)
//
//  Opens or query the wave stream.  
//  It sets the g_pcmtype variable with 
//  the right value to be used by the DURAUDIO_FillBuffer routine 
//  and sets the sample rate.
//
//  PCM_TYPE_M8  - PCM Mono 8bit
//  PCM_TYPE_S8  - PCM Stereo 8bit
//  PCM_TYPE_M16 - PCM Mono 16bit
//  PCM_TYPE_S16 - PCM Stereo 16bit
//
//  It also can be used to query the format to see if it's a valid one. 
//  To do so, the caller has to set fQueryFormatOnly to TRUE.
//
//  WAPI_INOUT apidir
//      Direction of the stream:  Playback=1, Record=0
//      
//  LPWAVEFORMATEX lpFormat
//      Pointer to a structure that defines the format of the wave.
//      Based on Microsoft definition.
//
//  unsigned char fQueryFormatOnly)
//      If the caller only want to check on the format, 
//      this value should be set to TRUE
//
//---------------------------------------------------------------------------
MMRESULT DURAUDIO_WaveOpen (WAPI_INOUT apidir,
	LPWAVEFORMATEX lpFormat, 
	unsigned char fQueryFormatOnly)
{
	MMRESULT mmRet = MMSYSERR_NOERROR;

    DURAUDIO_MSG("+DURAUDIO_WaveOpen");

    //  
    //  Save the direction of the stream on a global variable
    //
	GlobalApidir=apidir;
	
    //
    // Allow only PCM, mono or stereo, 8 or 16 bit
    //
	if ((lpFormat->wFormatTag != WAVE_FORMAT_PCM)  ||
		( lpFormat->nChannels  !=1 && lpFormat->nChannels != 2) ||
		( lpFormat->wBitsPerSample != 16 && lpFormat->wBitsPerSample != 8) )
		    mmRet = WAVERR_BADFORMAT;

    //
    //  If the caller only wants to check on that format, return with result
    //
	if (fQueryFormatOnly || mmRet != MMSYSERR_NOERROR) 
		return(mmRet);
	
    //  
    //  If the device is already being use, return with MMSYSERR_ALLOCATED
    //
	if (fInUse) 
	{
		SLEEP(200);
		return(MMSYSERR_ALLOCATED);
	}
	fInUse = TRUE;
	
    //
    // Sets the sample rate based on the Format structure
    //
	DURAUDIO_SetCodecRate (lpFormat->nSamplesPerSec);
	
    //
    //  Sets the g_pcmtype variable with the right value to be 
    //  used by the DURAUDIO_FillBuffer routine:
    //
	if (lpFormat->wBitsPerSample == 8) 
	{
		if (lpFormat->nChannels == 1) 
			g_pcmtype[GlobalApidir] = PCM_TYPE_M8;
		else
			g_pcmtype[GlobalApidir] = PCM_TYPE_S8;
	} 
	else 
	{
		if (lpFormat->nChannels == 1)
			g_pcmtype[GlobalApidir] = PCM_TYPE_M16;
		else
			g_pcmtype[GlobalApidir] = PCM_TYPE_S16;
	}

    DURAUDIO_MSG("-DURAUDIO_WaveOpen");

    return(MMSYSERR_NOERROR);
} 

// -----------------------------------------------------------------------------
//   void DURAUDIO_WaveClose(void)
//
//   Closes the Wave Stream.
//   Sets the "In use" flag to FALSE;
// 
// -----------------------------------------------------------------------------
void DURAUDIO_WaveClose(void)
{
    DURAUDIO_MSG("+DURAUDIO_WaveClose");

    // Reset the digital volume to zero
    v_nVolume = 0;

	fInUse =FALSE;

    DURAUDIO_MSG("-DURAUDIO_WaveClose");
};

// -----------------------------------------------------------------------------
//
//  void DURAUDIO_WaveStart (PWAVEHDR pwh)
//
// -----------------------------------------------------------------------------
PWAVEHDR DURAUDIO_WaveStart (PWAVEHDR pwh, unsigned long dma_page_size)
{
    unsigned char   SmallBuffer;
    unsigned char   *dma_cmd;
    unsigned char   *dma_status;
    unsigned long   *dma_prd_adr;
    unsigned char   status;
    PWAVEHDR         pwhRet;

    DURAUDIO_MSG("+DURAUDIO_WaveStart");

    GLOBAL_DMAPageSize=dma_page_size;
    pwhRet=NULL;
    Underflow = FALSE;
    tot_input_samples     = 0;
    tot_output_samples    = 0;

    //
    //  If the digital volume is 0 (nobody set it), set it to MAX
    //
    if(!v_nVolume)
        v_nVolume = 0xFFFFFFFF;

    if (Running)
	    SLEEP(200);

    v_fMoreData[GlobalApidir] = TRUE;        // we expect to have more data coming...
    v_nNextPage[GlobalApidir] = 0;
		
   //
   //	Setup the DMA pages
   //
    dma_page[1] = dma_page[0] + GLOBAL_DMAPageSize; 
    dma_page[2] = dma_page[1] + GLOBAL_DMAPageSize; 
    dma_page[3] = dma_page[2] + GLOBAL_DMAPageSize; 

    //
    //  The 5530/5540 contain an "audio buffered PCI bus master" that performs DMA-like
    //  functions.
    //  The Physical Region Descriptor (PRD) specifies a buffer to be "DMAed" by the
    //  PCI bus master. The prd_array_phys_adr in the mem-mapped 5530 audio regs points
    //  to the start of an array of PRD descriptors. 
    //
    //  Each entry of this array consists of 2 32-bit words:
    //
    //   32-bit word 0:   Address of the DMA buffer  .
    //               1:   bit-31:   End of Table (EOT)
    //                    bit-30:   End of Page  (EOP)
    //                    bit-29:   Loop         (JMP)
    //                    bit-15:0  byte count.
    //
    //
    //  Here, the PRD table is initialized as following:
    //
    //                          +----------------------------------------+
    //  prd_array[0]  !       Address of the 1st buffer        +
    //                         +----------------------------------------+
    //  prd_array[1]  !  EOP bit set  !  Length of 1st Buffer  +
    //                +----------------------------------------+
    //  prd_array[2]  !       Address of the 2nd Buffer        +
    //                +----------------------------------------+
    //  prd_array[3]  !  EOP bit set  !  Length of 2nd Buffer  +
    //                +----------------------------------------+
    //  prd_array[4]  !       Pointer to 1st PRD entry         +
    //                +----------------------------------------+
    //  prd_array[5]  !  JMP bit set  !                        +
    //                +----------------------------------------+
    //
	
    //
    //  Logical address of the PRD 
    //
    prd_array = (unsigned long *)( dma_page[2] );     
	
    //
    //  Physical address of the PRD 
    //
    prd_array_phys_adr = DMA_buffers_phys_addr + (2 * GLOBAL_DMAPageSize);  
	
    //
    // Address of the 1st buffer 
    //
    prd_array[0]  = DMA_buffers_phys_addr;
	
    //
    // EOP flag  | Length of 1st Buffer
    //
    prd_array[1]  = (0x40000000 | (GLOBAL_DMAPageSize) );    
	
    //
    // Address of the 2nd Buffer
    //
    prd_array[2]  = DMA_buffers_phys_addr + GLOBAL_DMAPageSize;
	
    //
    // EOP flag  | Length of 1st Buffer
    //
    prd_array[3]  = (0x40000000 | (GLOBAL_DMAPageSize) );      
	
    //
    // Pointer to the 1st PRD entry
    //
    prd_array[4]  = prd_array_phys_adr;
	
    //
    //  JMP flag - Loop to the beggining of the PRD
    //
    prd_array[5]  = 0x20000000;   
	
	FILL_MEMORY (dma_page[0], 0, GLOBAL_DMAPageSize);
	FILL_MEMORY (dma_page[1], 0, GLOBAL_DMAPageSize);
	
	if ((pwhRet=DURAUDIO_FillBuffer( pwh, (short*) dma_page[0] )))
	{
		DURAUDIO_MSG("More data \n");
		SmallBuffer=FALSE;
		pwhRet=DURAUDIO_FillBuffer( pwhRet, (short*) dma_page[1] );
	}
	else  // No more data ?
	{
		DURAUDIO_MSG("No more data \n");
        prd_array = (unsigned long *)( dma_page[2] ); 
		prd_array[1]  =  0x40000000 | tot_output_samples*4;     // Fill PRD#0 with EOP and size 
		prd_array[3]  =  0x40000000 | tot_output_samples*4;     // Fill PRD#1 with EOP and same size as PRD#0
		FILL_MEMORY (dma_page[1], 0, GLOBAL_DMAPageSize);
	}

    //
    //  Start DMAing.
    //  (See p.109 of the 5530 documentation).
    //  If it's Playback...
    //
	
	if (GlobalApidir)
	{
        // Use DMA 0 for Playback, 
		dma_cmd     = (unsigned char *)(audio_regs_adr + 0x20 );    // CMD.
		dma_status  = (unsigned char *)(audio_regs_adr + 0x21 );    // DStatus.
		dma_prd_adr = (unsigned long *)(audio_regs_adr + 0x24 );    // DPRD.
	}
    //
    //  If it's Recording...
    //
	else
	{
        // Use DMA 1 for Record, see p.109 of the 5530 documentation.
		dma_cmd     = (unsigned char *)(audio_regs_adr + 0x28 );    // CMD.
		dma_status  = (unsigned char *)(audio_regs_adr + 0x29 );    // Status.
		dma_prd_adr = (unsigned long *)(audio_regs_adr + 0x2C );    // PRD.
	}
	
    //
    //  Clear the EOP bits
    //
	status = *dma_status; 
	
    //
    //  Points to PRD array's physical address.
    //
    *dma_prd_adr = DMA_buffers_phys_addr + (2 * GLOBAL_DMAPageSize);  
	
    //  Starts DMA
	if (GlobalApidir)
		*dma_cmd      = 0x01;     // Bus Master ON. Direction bit is 0 (Playback).
	else
		*dma_cmd      = 0x09;     // Bus Master ON. Direction bit is 0 (Record)

    Running=TRUE; 
	
    DURAUDIO_MSG("-DURAUDIO_WaveStart");

	return pwhRet;
}

//-------------------------------------------------------------
//
//   PWAVEHDR DURAUDIO_FillBuffer (PWAVEHDR pwh, short * pBuffer)
//
//   Fills the Audio buffers according to the global  
//   variable GlobalApidir. 
//
//   GlobalApidir indicates the direction of the stream:
//
//   Playback: GlobalApidir is equal to 1.
//   Record:   GlobalApidir is equal to 0.
//
//   pwh - Pointer to the WAVEHDR structure of the stream
//
//   pBuffer - Pointer to the DMA buffer to be Played/Recorded
//
//-------------------------------------------------------------
PWAVEHDR DURAUDIO_FillBuffer (PWAVEHDR pwh, signed short * pBuffer)
{
    DURAUDIO_MSG("+DURAUDIO_FillBuffer");

    // If it's Playback...
	if (GlobalApidir)
		return(DURAUDIO_FillDMABuffer(pwh,pBuffer));
	else
		return(DURAUDIO_FillUserBuffer(pwh,pBuffer));
}

//--------------------------------------------------------------------
//
//   PWAVEHDR DURAUDIO_FillDMABuffer (PWAVEHDR pwh, signed short * pBuffer)
//
//   Fills the hardware DMA buffer with the data from the user's 
//   buffer to be played.
//
//   pwh - Pointer to the WAVEHDR structure of the stream
//
//   pBuffer - Pointer to the user buffer to be played.  
//             It points to the data that will fill the Hardware buffer 
//
//--------------------------------------------------------------------
PWAVEHDR DURAUDIO_FillDMABuffer (PWAVEHDR pwh, signed short * pBuffer)
{
	int             bits_to_shift;
	unsigned long   i;
	unsigned long   samples_to_mix;
	unsigned long   nVolume = v_nVolume & 0xFFFF;
	
	signed short    mult;
	signed short    div;
	PCM_SAMPLE      pcmsample;
	PPCM_SAMPLE     pSampleSource;
	
	DURAUDIO_MSG("DURAUDIO_FillDMABuffer \n");
	
    //
    //  How many samples from this stream are needed to fill the DMA buffer?
    //
    //	samples_to_mix = v_dma_num_samples;
	samples_to_mix = GLOBAL_DMAPageSize/2;
	i = (nVolume >> 12) & 0x0F;
	
	mult = VOL[i].multiplier;
	div  = VOL[i].divisor; 
	
	bits_to_shift = div;
	
    //
    // For each sample in the buffer...
    //
	
	for (i = 0; i < samples_to_mix; i++) 
	{
		if ((pwh != NULL) && (pwh->dwBytesRecorded >= pwh->dwBufferLength))
		{
			DBG_MSG("pwh: %08X ", pwh);
			DBG_MSG("Length: %d ", pwh->dwBufferLength);
			DBG_MSG("BytesRecorded: %d ", pwh->dwBytesRecorded);
			DBG_MSG("BytesOnDMA: %d \n", i*2);
            pwh = pwh->lpNext;

		    if ((pwh != NULL) && (pwh!=(struct wavehdr_tag *)1))
            {
			    DBG_MSG("NEW pwh: %08X ", pwh);
			    DBG_MSG("Length: %d ", pwh->dwBufferLength);
			    DBG_MSG("BytesRecorded: %d ", pwh->dwBytesRecorded);
			    DBG_MSG("BytesOnDMA: %d \n", i*2);
            }
            else
            {
                pwh=NULL;
                DURAUDIO_MSG("pwh IS NULL!\n");			
            }

		}
		
		if (pwh == NULL)
		{
			if (g_pcmtype[GlobalApidir]==PCM_TYPE_M8 || g_pcmtype[GlobalApidir]==PCM_TYPE_S8)
				pBuffer[i]=0x80;
			else
				pBuffer[i]=0x00;
		} 
		else  
		{
			pSampleSource = (PPCM_SAMPLE) (pwh->lpData + pwh->dwBytesRecorded);
			
			switch (g_pcmtype[GlobalApidir]) 
			{
				case PCM_TYPE_M8:
					pBuffer[i] = (signed short) BITS_8_TO_16(pSampleSource->m8.sample);
					pBuffer[i] = (pBuffer[i] * mult) >> bits_to_shift; // software volume control
					i++;
					pBuffer[i] = (signed short) BITS_8_TO_16(pSampleSource->m8.sample);
					pwh->dwBytesRecorded+=1;
					break;
					
				case PCM_TYPE_S8:
					pcmsample.s8.sample_left  = pSampleSource->s8.sample_left;
					pcmsample.s8.sample_right = pSampleSource->s8.sample_right;
					pBuffer[i] = (signed short) BITS_8_TO_16(pcmsample.s8.sample_left);
					pBuffer[i] = (pBuffer[i] * mult) >> bits_to_shift; // software volume control
					++i;
					pBuffer[i] = (signed short) BITS_8_TO_16(pcmsample.s8.sample_right);
					pwh->dwBytesRecorded+=2;
					break;
					
				case PCM_TYPE_M16:
					pBuffer[i] = pSampleSource->m16.sample;
					pBuffer[i] = (pBuffer[i] * mult) >> bits_to_shift; // software volume control
					i++;
					pBuffer[i] = pSampleSource->m16.sample;
					pwh->dwBytesRecorded += 2;
					break;
					
				case PCM_TYPE_S16:
					pcmsample.s16.sample_left  = pSampleSource->s16.sample_left;
					pcmsample.s16.sample_right = pSampleSource->s16.sample_right;
					pwh->dwBytesRecorded += 4;
					pBuffer[i] = pcmsample.s16.sample_left;
					pBuffer[i] = (pBuffer[i] * mult) >> bits_to_shift; // software volume control
					++i;
					pBuffer[i] = pcmsample.s16.sample_right;
					break;
			}
			
			tot_output_samples++;
			
			pBuffer[i] = (pBuffer[i] * mult) >> bits_to_shift; // software volume control
		}
		
	} // for each sample
	
    DURAUDIO_MSG("-DURAUDIO_FillDMABuffer");
	
	return (pwh);
}

//--------------------------------------------------------------------
//
//   PWAVEHDR DURAUDIO_FillUserBuffer (PWAVEHDR pwh, signed short  * pBuffer)
//
//   Fills the user buffer with the data recorded 
//   (from the hardware DMA buffer).
//
//   pwh - Pointer to the WAVEHDR structure of the stream
//
//   pBuffer - Pointer to the user buffer to save the recorded data.
//             It points to the buffer that will be filled with data 
//             from the Hardware buffer 
//
//--------------------------------------------------------------------
PWAVEHDR DURAUDIO_FillUserBuffer (PWAVEHDR pwh, signed short * pDMABuffer)
{
	unsigned int      i;
	unsigned long   Samples_Recorded;
	
	PCM_SAMPLE      pcmsample;
	PPCM_SAMPLE     pSampleRecorded;

    DURAUDIO_MSG("+DURAUDIO_FillUserBuffer");

    //
    // What's the maximum number of samples from this stream needed 
    // to fill the DMA buffer?
    //
	Samples_Recorded = GLOBAL_DMAPageSize/2;
	
	for (i = 0; i < Samples_Recorded; i++) 
	{
		if ((pwh != NULL) && (pwh->dwBytesRecorded >= pwh->dwBufferLength))
			pwh = pwh->lpNext;
		
		if (pwh == NULL)
		{
            DURAUDIO_WaveStop();
            v_fMoreData[GlobalApidir]=FALSE;
			DURAUDIO_MSG("** END OF RECORDING ** \n");
			return pwh;
		}
		else  
		{
			pSampleRecorded = (PPCM_SAMPLE) (pwh->lpData + pwh->dwBytesRecorded);
			
			switch (g_pcmtype[GlobalApidir]) 
			{
				case PCM_TYPE_M8:
					pcmsample.m8.sample=BITS_16_TO_8(pDMABuffer[i]);
					pSampleRecorded->m8=pcmsample.m8;
					++i;
					pwh->dwBytesRecorded+=1;
					break;
					
				case PCM_TYPE_S8:
					pcmsample.s8.sample_left  = BITS_16_TO_8(pDMABuffer[i]);
					++i;
					pcmsample.s8.sample_right = BITS_16_TO_8(pDMABuffer[i]);
					pSampleRecorded->s8=pcmsample.s8;
					pwh->dwBytesRecorded+=2;
					break;
					
				case PCM_TYPE_M16:
					pSampleRecorded->m16.sample = pDMABuffer[i];
					i++;
					pwh->dwBytesRecorded += 2;
					break;
					
				case PCM_TYPE_S16:
					pSampleRecorded->s16.sample_left = pDMABuffer[i];
					++i;
					pSampleRecorded->s16.sample_right = pDMABuffer[i];
					pwh->dwBytesRecorded += 4;
					break;
			}
		}
		tot_output_samples++;
	} // for each sample

    DURAUDIO_MSG("-DURAUDIO_FillUserBuffer");

	return (pwh);
}

// -----------------------------------------------------------------------------
//
//  void DURAUDIO_WaveContinue (PWAVEHDR pwh)
//
//  This function is called with v_nNextPage[WAPI_OUT] containing
//  the buffer that has just finished DMAing, and thus the buffer that is to
//  be filled with data. The _other_ buffer is the buffer that needs to have
//  its DMA activated.
//
// -----------------------------------------------------------------------------
PWAVEHDR DURAUDIO_WaveContinue (PWAVEHDR pwh)
{
	PWAVEHDR      Retpwh;
	
    DURAUDIO_MSG("+DURAUDIO_WaveContinue");

    DURAUDIO_ClearStatus ();
	Retpwh=DURAUDIO_FillBuffer( pwh, (signed short*) dma_page[v_nNextPage[GlobalApidir]] );
	v_nNextPage[GlobalApidir] = (!v_nNextPage[GlobalApidir] ? 1 : 0);
	
    DURAUDIO_MSG("-DURAUDIO_WaveContinue");

	return (Retpwh);
}

//-------------------------------------------------------
//  void WaveStop()
//
//  Stops the wave stream.
//
//  It zeroes the next buffer to be played and put a EOT
//  (End of table flag) on next PRD.
//
//-------------------------------------------------------
void DURAUDIO_WaveStop(void)
{
    DURAUDIO_MSG("+DURAUDIO_WaveStop");

    //
    //  Fills the buffer with silence (zeroes)
    //
	FILL_MEMORY (dma_page[v_nNextPage[GlobalApidir]], AUDIO_SILENCE, GLOBAL_DMAPageSize);
	
    // Logical address: DMA input buffer (unused).
	prd_array = (unsigned long *)( dma_page[2] );   
	
    //
    // sets EOT (End of table flag) on next PRD
    //
	if (v_nNextPage[GlobalApidir])
		prd_array[3]  =  0x80000000;  
	else
		prd_array[5]  =  0x80000000;  
	
    //  Clear the Audio Bus Master status bit
	DURAUDIO_ClearStatus();
	
    //
    // There is no more data coming. It goes to the InterruptType 
    // function and Returns a STOPPED flag.
    //
	v_fMoreData[GlobalApidir] = FALSE;

    DURAUDIO_MSG("-DURAUDIO_WaveStop");
}


void DURAUDIO_Standby(void)
{
	DURAUDIO_ClearStatus();
	Running =FALSE;    
	fInUse =FALSE;   
}

//---------------------------------------------------------
//  void DURAUDIO_Deinitialize(void)
//
//  This routine is called when the driver 
//  is being unloaded. Free up memory, etc.
//---------------------------------------------------------
void DURAUDIO_Deinitialize(void)
{
	FREE_DMA_BUFFER(dma_page[0]);
}

//---------------------------------------------------------
//   unsigned long DURAUDIO_GetVolume(void)
//  
//   Returns the digital volume to the caller
//---------------------------------------------------------
unsigned long DURAUDIO_GetVolume(void)
{
	return v_nVolume;
}

//---------------------------------------------------------
//   unsigned long DURAUDIO_SetVolume(void)
//  
//   Sets the digital volume based on the value sent by the caller
//---------------------------------------------------------
void DURAUDIO_SetVolume(unsigned long Vol)
{
	v_nVolume=Vol;
}

//---------------------------------------------------------
//
//  void DURAUDIO_CodecWrite( unsigned short codec_port, unsigned short codec_data  )
//
//  Writes data to the CODEC
//
//---------------------------------------------------------
void DURAUDIO_CodecWrite( unsigned short codec_port, unsigned short codec_data  )
{
	unsigned long codec_port_data = 0;
	
	codec_port_data = ((unsigned long)codec_port)<<16;
	codec_port_data |= (unsigned long)codec_data;
	
    // write specific codec register
	DURAUDIO_SetCodecControl( codec_port_data );
	
	DURAUDIO_CodecStatus();
	
    // This prevents multiple volume writes to a codec register 
	DURAUDIO_SetCodecControl( 0x80000000|codec_port_data );
}

//------------------------------------------------------------------------------
//
//  unsigned short DURAUDIO_CodecDirectRegRead( unsigned short codec_port )
//
//  Reads data from the CODEC
//
//------------------------------------------------------------------------------

unsigned short DURAUDIO_CodecDirectRegRead( unsigned short codec_port )
{
	unsigned long codec_port_data = 0;
	unsigned long val;
	
	codec_port_data  = ((unsigned long)codec_port)<<24;
	codec_port_data |= 0x80000000;   // High-bit set (p.106) is a CODEC reg READ.
	
	do 
	{
		DURAUDIO_SetCodecControl( codec_port_data );
		val = DURAUDIO_CodecStatus();
	} while ( ( 0x000000FF & ((unsigned long)(codec_port)) ) !=
		( 0x000000FF & (val >> 24) ) ); // kludge, new bit 
	
	return( (unsigned short)val );
}

//------------------------------------------------------------------------------
//
//  void DURAUDIO_SetCodecControl(unsigned long val)
//
//  Waits for next avail pos in the CODEC's serial frame and sends the data.
//
//  While the CODEC_CMD_VALID bit is set, it should not be overwritten.
//  i.e., the codec cycles around to ship the command out at the next 
//  avail pos in the serial frame.
//
//------------------------------------------------------------------------------

void DURAUDIO_SetCodecControl(unsigned long val)
{
	long             timeout;
	unsigned long    cmd_val;
	unsigned long    status;
	
	timeout = 30000;
	
	while ((DURAUDIO_PortRead32( CODEC_CMD_REG ) & CODEC_CMD_VALID) && (--timeout));
	
	
	status = DURAUDIO_PortRead32( CODEC_CMD_REG );
	cmd_val = (val & KAHLUA_CODEC_COMMAND_MASK) | KAHLUA_CODEC_NUM;
	
	DURAUDIO_PortWrite32( CODEC_CMD_REG, cmd_val );
}

//---------------------------------------------------------
//
// unsigned long DURAUDIO_CodecStatus( void )
//
// This routine is the "core" of the codec read capability.
// It is actually only called about 9 times when the driver
// is initialized and sets the codec operational registers.
//
//---------------------------------------------------------

unsigned long DURAUDIO_CodecStatus( void )
{
	unsigned short  i;
	unsigned long   result;
	
    // Its a synchronous interface, so it had better be here by then,
    //  or the codec just doesnt do it...
	
	for( i=0; i<=30000; i++) 
	{
		result = DURAUDIO_PortRead32( CODEC_STATUS_REG );
		
		if( ( result & CODEC_STATUS_VALID ) ) 
			return( result );
	} 
	
    //
    // Timeout waiting for CODEC_STATUS_VALID.  Return what we have.
    //
	return result;
}

//------------------------------------------
//   void DURAUDIO_SetCodecRate (unsigned long Rate)
//  
//   Sets the CODEC Sample Rate
//------------------------------------------
void DURAUDIO_SetCodecRate(unsigned long Rate)
{
	if (AD1819A) 
	{
		DURAUDIO_CodecWrite(AD1819A_PCM_SR0,(unsigned short) Rate);
		DURAUDIO_CodecWrite(AD1819A_PCM_SR1,(unsigned short) Rate);
	} 
	else 
	{
		DURAUDIO_CodecWrite(LM4548_PCM_FRONT_DAC_RATE,(unsigned short) Rate);
		DURAUDIO_CodecWrite(LM4548_PCM_LR_ADC_RATE,   (unsigned short) Rate);
	}
}

//----------------------------------------------------------
//  unsigned char DURAUDIO_ClearStatus()
//
//  Clear the status bit of the respective Audio Bus Master
//
//  0x21 if it's playback  (Audio Bus Master 0)
//  0x29 if it's recording (Audio Bus Master 1)
//----------------------------------------------------------
unsigned char DURAUDIO_ClearStatus(void)
{
	unsigned char *dma_status;
	volatile unsigned char  status;     // Volatile to force read-to-clear.
	
	if (GlobalApidir)
        // If it's Playback, point to status of Audio Bus Master 0 
		dma_status  = (unsigned char *)(audio_regs_adr + 0x21 );
	else
        // If it's Recording, point to status of Audio Bus Master 1
		dma_status  = (unsigned char *)(audio_regs_adr + 0x29 );
	
	status  = *dma_status;          // Read to clear
	
	return status;
}

// -----------------------------------------------------------------------------
//
//   AUDIO_STATE DURAUDIO_GetInterruptType(VOID)
//
//   Called by the driver's ISR.
//   Returns the state of the stream being Played/Recorded.
// -----------------------------------------------------------------------------
AUDIO_STATE DURAUDIO_GetInterruptType(void)
{ 

    DURAUDIO_MSG("+DURAUDIO_GetInterruptType\n");

	if (v_fMoreData[GlobalApidir] == TRUE)    // Cause following to be called:
	{        
		if (GlobalApidir)
		{
			if (Underflow)
				return AUDIO_STATE_OUT_UNDERFLOW; // or Return Underflow, synchronize and call WaveoutStart
			else
				return AUDIO_STATE_OUT_PLAYING;   // private_WaveOutContinue() or
		}
		else
		{
			if (Overflow)
				return AUDIO_STATE_IN_OVERFLOW;     // or Return Overflow, synchronize and call WaveoutStart
			else
				return AUDIO_STATE_IN_RECORDING;    // private_WaveInContinue() or
		} 
	}                                     
	else 
	{                                     
		if (GlobalApidir)
			return AUDIO_STATE_OUT_STOPPED;   
		else
			return AUDIO_STATE_IN_STOPPED;   
	}                                         
}

//-------------------------------------------------------------------------
//   unsigned short DURAUDIO_IO_PCI_Read16(unsigned long pdev, unsigned short port)
//
//   Reads 16 bits to the PCI I/O space
//-------------------------------------------------------------------------
unsigned short DURAUDIO_IO_PCI_Read16(unsigned long pdev, unsigned short port)
{
	OUTP_ULONG( PCI_CADDR, pdev + (port & 0xFFC) );
	
	return INP_USHORT( PCI_CDATA + (port & 3) );
}

//------------------------------------------------------------------------------------
//   void DURAUDIO_IO_PCI_Write16(unsigned long pdev, unsigned short port, unsigned short data )
//
//   Writes 16 bits to the PCI I/O space
//------------------------------------------------------------------------------------
void DURAUDIO_IO_PCI_Write16(unsigned long pdev, unsigned short port, unsigned short data )
{
	OUTP_ULONG( PCI_CADDR, pdev + (port & 0xFFC) );
	
	OUTP_USHORT( PCI_CDATA + (port & 3), data );
}

//-----------------------------------------------------------------------------------
//    void DURAUDIO_IO_PCI_Write32(unsigned long pdev, unsigned short port, unsigned long data)
//
//   Writes 32 bits to the PCI I/O space
//-----------------------------------------------------------------------------------
void DURAUDIO_IO_PCI_Write32(unsigned long pdev, unsigned short port, unsigned long data)
{
	OUTP_ULONG( PCI_CADDR, pdev + (port & 0xFFC) );
	
	OUTP_ULONG( PCI_CDATA + (port & 3), data );
}

//-------------------------------------------------------------------------
//    unsigned long DURAUDIO_IO_PCI_Read32(unsigned long pdev, unsigned short port)
//
//   Reads 32 bits to the PCI I/O space
//-------------------------------------------------------------------------
unsigned long DURAUDIO_IO_PCI_Read32(unsigned long pdev, unsigned short port)
{
	OUTP_ULONG( PCI_CADDR, pdev + (port & 0xFFC) );
	
	return INP_ULONG( PCI_CDATA + (port & 3) );
}

//-----------------------------------------------------------------------
//   void DURAUDIO_PortWrite32 (unsigned short reg, unsigned long val)
//
//   Writes 32 bits to the 5530/5540 I/O port
//------------------------------------------------------------------------
void DURAUDIO_PortWrite32 (unsigned short reg, unsigned long val)
{
	unsigned long *reg_loc;
	
	reg_loc  = (unsigned long *)(audio_regs_adr + reg);
	*reg_loc = val;
}

//-----------------------------------------------------------------------
//   unsigned long DURAUDIO_PortRead32 (unsigned short reg)
//
//   Reads 32 bits from the 5530/5540 I/O port
//------------------------------------------------------------------------
unsigned long DURAUDIO_PortRead32 (unsigned short reg)
{
	unsigned long *reg_loc;
	unsigned long  val;
	
	reg_loc  = (unsigned long *)(audio_regs_adr + reg);
	val      = *reg_loc;
	
	return( val );
}


