; EMS41.ASM
;
; This is an LIM EMS 4.1 driver for Micro Mainframe's 5150T expanded memory
; card(s).  The driver supports up to 4 EMS cards with up to 2MB of memory
; apiece.  This driver is based on information provided by a disassembly of
; Micro Mainframe's LIM EMS 3.2 driver, by the EMS40.SYS driver that appeared
; in _PC Magazine_ in 1989, and by the LIM 4.1 standard.
;
; When assembled with A86, this program becomes EMS41.BIN, a device driver
; for the CONFIG.SYS file.  This driver is compatible with Micro Mainframe's
; driver and recognizes the same /Q and /C parameters.  Reading from the 
; EMMXXXX0 device returns the same information, and writing to the device has 
; the same effect.
;
; Following the terminology in the standard, we refer to those pages allocated
; by a handle as "logical pages."  The 64kB segment of memory at 0D000h is
; referred to as the "page frame."  The page frame is divided into 4 16kB
; parts, called "physical pages," where logical pages may be made to appear
; ("mapped").  The physical memory on the card is also divided into 16kB parts, 
; which I call "hardware pages."  
;
; The standard specifies that logical pages may be either "raw" or "standard," 
; where a standard 16kB page may consist of several raw pages.  This driver 
; makes no distinction between raw and standard pages.  The specification also 
; provides that a handle may be either "volatile" or "nonvolatile," where pages 
; allocated to a nonvolatile handle remain allocated across a warm boot.  This 
; driver does not support the nonvolatile attribute (note:  the RAM disk driver 
; EMSRAMDK.BIN provided by Micro Mainframe does not rely on this attribute,
; which did not exist in LIM version 3.2).  This driver also does not support
; alternate map register sets or DMA register sets (alternate map register sets
; are emulated per the standard).
;
; Note on I/O ports:
;
; There are four EMS page mapping registers for each of up to four 2-meg EMS
; cards in the system, one register for each 16k physical page:
;
;			card 0	card 1	card 2	card 3
; physical page 0	 208	 20A	 20C	 20E
; physical page 1	4208	420A	420C	420E
; physical page 2	8208	820A	820C	820E
; physical page 3	C208	C20A	C20C	C20E
;
; To map hardware page x on card y into the page frame, write AL = 1xxxxxxx to 
; the indicated I/O port, where xxxxxxx is the number of hardware page x (zero-
; based, up to 127) on the card.
;
; To unmap the hardware page on card y that is currently mapped into the page 
; frame (so that another card can use that physical page), write AL = 00h to 
; the indicated port.
;

		ORG 0
;
; Device header.
;  
		DD	-1		; link to next driver in chain
DEVICEATTRIB	DW	8000h		; device attribute
		;
		; Pointers to strategy and interrupt routines.
		;
		DW	OFFSET STRATEGY	
		DW	OFFSET INTERRUPT
		DB	"EMMXXXX0"	; device name
;
; Doubleword pointer to request header (stored by strategy routine, used by
; interrupt routine).
;
		EVEN
REQUESTPTR	LABEL	DWORD
REQUESTOFFS	DW	0
REQUESTSEG	DW	0
;
; Generic request structure, for INTERRUPT and other routines.
;
GENREQ		STRUC 	[BX]
		DB	2 DUP (?)
COMMANDCODE	DB	?
REQSTATUS	DW	?
		ENDS
;
; Request structure for READ routine.
;
READREQ		STRUC	[BX]
		DB	2 DUP (?)
COMMANDCODE	DB	?
REQSTATUS	DW	?
		DB	9 DUP (?)
DATAPTR		DD	?
NUMBYTES	DW	?
		ENDS
;
; Request structure for NONDEST routine.
;
NONDESTREQ	STRUC	[BX]
		DB	2 DUP (?)
COMMANDCODE	DB	?
REQSTATUS	DW	?
		DB	8 DUP (?)
NONDESTCHAR	DB	?
		ENDS
;
; Jump table for INTERRUPT routine.
;
		EVEN
INTERJUMPS	EQU	$
		DW	OFFSET INIT		; Init
		DW	OFFSET NULLFUNC		; Media Check
		DW	OFFSET NULLFUNC		; Build BPB
		DW	OFFSET UNSUPP		; IOCTL Read
		DW	OFFSET READ		; Read
		DW	OFFSET NONDEST		; Nondestructive Read
		DW	OFFSET NULLFUNC		; Input Status
		DW	OFFSET WRITE		; Flush Input Buffers
		DW	OFFSET WRITE		; Write
		DW	OFFSET WRITE		; Write with Verify
		DW	OFFSET NULLFUNC		; Output Status
		DW	OFFSET NULLFUNC		; Flush Output Buffers
		DW	OFFSET UNSUPP		; IOCTL Write
;
; Jump table for INT67HDLR (main jump table).
;
FUNCTIONJUMPS	EQU	$
		DW	OFFSET FUNC1	; Get Status
		DW	OFFSET FUNC2	; Get Page Frame Address
		DW	OFFSET FUNC3	; Get Unallocated Page Count
		DW	OFFSET FUNC4	; Allocate Pages
		DW	OFFSET FUNC5	; Map/Unmap Handle Pages
		DW	OFFSET FUNC6	; Deallocate Pages
		DW	OFFSET FUNC7	; Get Version
		DW	OFFSET FUNC8	; Save Page Map
		DW	OFFSET FUNC9	; Restore Page Map
		DW	OFFSET FUNCBAD	; (invalid function code)
		DW	OFFSET FUNCBAD	; (invalid function code)
		DW	OFFSET FUNC12	; Get Handle Count
		DW	OFFSET FUNC13	; Get Handle Pages
		DW	OFFSET FUNC14	; Get All Handle Pages
		DW	OFFSET FUNC15	; Get/Set Page Map
		DW	OFFSET FUNC16	; Get/Set Partial Page Map
		DW	OFFSET FUNC17	; Map/Unmap Multiple Handle Pages
		DW	OFFSET FUNC18	; Reallocate Pages
		DW	OFFSET FUNC19	; Get/Set Handle Attribute
		DW	OFFSET FUNC20	; Get/Set Handle Name
		DW	OFFSET FUNC21	; Get Handle Directory
		DW	OFFSET FUNC22	; Alter Page Map & Jump
		DW	OFFSET FUNC23	; Alter Page Map & Call
		DW	OFFSET FUNC24	; Move/Exchange Memory Region
		DW	OFFSET FUNC25	; Get Mappable Physical Address Array
		DW	OFFSET FUNC26	; Get Expanded Memory Hardware Info
		DW	OFFSET FUNC27	; Allocate Standard/Raw Pages
		DW	OFFSET FUNC28	; Alternate Map Register Set
		DW	OFFSET FUNC29	; Prepare Expanded Memory for Warm Boot
		DW	OFFSET FUNC30	; Enable/Disable OS/E Function Set
		DW	OFFSET FUNCBAD	; (AH = 5Eh)
		DW	OFFSET FUNCBAD	; (AH = 5Fh)
		DW	OFFSET MMFA	; Get Page Owner Table (nonstandard)
		DW	OFFSET MMFB	; Get Context Save Area Contents (ns.)
		DW	OFFSET MMFC	; Get I/O Port and Value Array (ns.)
;
; Jump table for function 15.
;
FUNC15JUMPS	EQU	$
		DW	OFFSET FUNC15S0		; Get Page Map
		DW	OFFSET FUNC15S1		; Set Page Map
		DW	OFFSET FUNC15S2		; Get & Set Page Map
		DW	OFFSET FUNC15S3		; Get Size of Page Map Save Ar.
;
; Jump table for function 16.
;
FUNC16JUMPS	EQU	$
		DW	OFFSET FUNC16S0		; Get Partial Page Map
		DW	OFFSET FUNC16S1		; Set Partial Page Map
		DW	OFFSET FUNC16S2		; Get Size of Partial Page Map
						;   Save Array
;
; Jump table for function 28.
;
FUNC28JUMPS	EQU	$
		DW	OFFSET FUNC28S0		; Get Alt Map Register Set
		DW	OFFSET FUNC28S1		; Set Alt Map Register Set
		DW	OFFSET FUNC28S2		; Get Alt Map Save Array Size
		DW	OFFSET FUNC28S3		; Alloc Alt Map Register Set
		DW	OFFSET FUNC28S4		; Dealloc Alt Map Register Set
		DW	OFFSET FUNC28S3		; Alloc DMA Register Set
		DW	OFFSET FUNC28S4		; Enable DMA on Alt Map Reg Set
		DW	OFFSET FUNC28S4		; Disable DMA on Alt Map Reg Set
		DW	OFFSET FUNC28S4		; Dealloc DMA Register Set
;
; Data for Read:  pointer for Read function.  This pointer addresses the 
; next byte returned by the read function.
;
READPTR		DW	OFFSET DRIVERSEG
		;
		; Count of calls to the read function.
		;
READCOUNT	DB	0
		EVEN
		;
		; Segment address of this driver.  (Set by Init.)
		;
DRIVERSEG	DW	0
		DW	OFFSET NUMBOARDS
		DW	OFFSET TRANSLATEARRAY
		DW	OFFSET CONTEXTSAVE
		DB	"B"
		DB	"W"
;
; Variables common to version 3.2.
;
; Number of unallocated EMS pages available.  (Initialized by Init.)
;
UNALLOCPAGES	DW	0	; initially none (will detect)
		;
		; Total number of EMS pages in the system.  (Set by Init.)
		;	
TOTALPAGES	DW	0	; initially none (will detect)
		;
		; Compatibility mode flag.  0FFh = compatibility mode.  EMM 
		; functions 60h through 62h are invalid in compatibility mode.  
		; (Set by Init.)
		;
COMPATIBILITY	DB	0	; initially not compatibility mode
		;
		; The number of EMS boards installed in the system.  (Set 
		; by Init.)
		;
NUMBOARDS	DW	0	; initially none installed (will detect)
;
; The following two data structures from Micro Mainframe's driver must remain
; the same for compatibility.
;
; Page translation array.  For each of 512 pages, the word entry is 0FFFFh if 
; the page is unavailable, 80xxh if the page is allocated to handle xx, 0 if 
; the page is available and unallocated.  (Initialized by Init.)
;
		EVEN
TRANSLATEARRAY	DW	512 DUP (0FFFFh)	; initially all unavailable
;
; The following is the save area for mapping contexts.  There is room for 20 
; contexts, each 19 bytes long, formatted as follows:
;    offset:  length:  meaning:
;	0	byte	0 if this save region is in use, 0FFh if not
;	1	word	handle to which this context is allocated
;	3      byte*16	page mapping register contents, in the following order:
;
; 	208, 4208, 8208, C208, 20A, 420A, 820A, C20A,
;	20C, 420C, 820C, C20C, 20E, 420E, 820E, C20E
;
; See note on I/O ports, top of file.  All contexts initially free.
;
CONTEXTSAVE	DB	20 DUP (0FFh, 18 DUP (0))
;
; The following are new data structures for this version.
;
; Hardware page allocation array.  Entry is 0 if the hardware page is
; allocated, bad or not installed, 0FFh if hardware page is available.
;
PAGEALLOC	DB	512 DUP (0)	; initially all not installed
		;
		; Hardware page to logical page array.  Entry is logical
		; page number for hardware page if allocated, don't care
		; if not allocated.
		;
LOGPAGES	DW	512 DUP (0)		; don't care initially
		;
		; Handle allocation array.  Entry is 0 if handle is active,
		; 0FFh if inactive.
		;
HNDLALLOC	DB	0, 254 DUP (0FFh)	; initially only 0 is active
		;
		; Handle names array.
		;
HANDLENAMES	DB	255 DUP (8 DUP (0))	; initially all names null
		;
		; Installation status of each board.  0 = installed, 0FFh = 
		; not installed.  MMF says boards must be installed in order,
		; 0 to 3.  This driver tries to be more tolerant.
		;
INSTALLED	DB	4 DUP (0FFh)		; initially none installed
		;
		; Permission flag, for OS/E functions.  1 = access permitted,
		; 0 = access denied.
		;
PERMISSION	DB	1			; access initially permitted
		;
		; Flag, indicates whether an access key has been issued.
		; 1 = access key is active, 0 = no key has been issued.
		;
KEYACTIVE	DB	0			; initially no key
		;
		; Access key to enable/disable OS/E functions.  A 32-bit
		; random key.
		;
ACCESS_LO	DW	0			; CX value
ACCESS_HI	DW	0			; BX value
		;
		; Alternate map register set context restore area pointer
		; (never did think of a shorter name for this).
		;
FUNC28PTR	DD	0			; initially null

;***************************** RESIDENT DEVICE DRIVER CODE
;
; Strategy routine:  stores address of request header.
;
STRATEGY:	MOV	CS:REQUESTOFFS,BX
		MOV	CS:REQUESTSEG,ES
		RETF	
;
; Interrupt routine:  processes request.
;
; Save registers and flags on stack.
;
INTERRUPT:	PUSHF	
		PUSH	AX
		PUSH	BX
		PUSH	CX
		PUSH	DX
		PUSH	SI
		PUSH	DI
		PUSH	BP
		PUSH	DS
		PUSH	ES
		;
		; Get function code and check if valid.
		;
		LDS	BX,CS:REQUESTPTR
		MOV	REQSTATUS,0		; success by default
		MOV	BL,COMMANDCODE
		CMP	BL,12
		JNA	>L0
		CALL	UNSUPP
		JMP	>L1
		;
		; Function code valid, call function.
		;
L0:		XOR	BH,BH
		SHL	BX,1
		CALL	CS:[BX+INTERJUMPS]
		;
		; Set done bit.
		;
L1:		LDS	BX,CS:REQUESTPTR
		OR	REQSTATUS,100h
		;
		; Pop registers and flags and return.
		;
		POP	ES
		POP	DS
		POP	BP
		POP	DI
		POP	SI
		POP	DX
		POP	CX
		POP	BX
		POP	AX
		POPF
		RETF	
;
; Null function, does nothing.
;
NULLFUNC:	RET
;
; Unsupported function, sets error.
;
UNSUPP:		LDS	BX,CS:REQUESTPTR
		MOV	REQSTATUS,8003h
		RET		
;
; Read function routine.  This routine can be called up to 10 times in
; succession and will return a different byte each time.  A call to the 
; write function (below) resets the read pointer; the read pointer will 
; also be reset after the 10th call to this function.  The following data 
; is returned by successive calls:
;     call    data returned
;	1,2	segment address of this driver
;	3,4	offset of the word giving the number of boards installed
;	5,6	offset of the page translation array
;	7,8	offset of the context save area
;	9,10	word with "B" in the low byte, "W" in the high byte
;
READ: 		LDS	BX,CS:REQUESTPTR
		LES	DI,DATAPTR
		MOV	CX,NUMBYTES
		JCXZ	>L1
L0:		CALL	READONE
		LOOP	L0
L1:		RET
;
; Subroutine to read one byte into ES:[DI], incrementing DI.
;
READONE:	PUSH	DS
		PUSH	SI
		PUSH	CS
		POP	DS
		MOV	SI,READPTR
		CLD
		MOVSB
		MOV	READPTR,SI
		INC	READCOUNT
		CMP	READCOUNT,10
		JB	>L0
		MOV	READCOUNT,0
		MOV	READPTR,OFFSET DRIVERSEG
L0:		POP	SI
		POP	DS
		RET
;
; Nondestructive read.  Returns next character to read, does not update
; read pointer.
;
NONDEST:	LDS	BX,CS:REQUESTPTR
		MOV	SI,READPTR
		MOV	AL,CS:[SI]
		MOV	NONDESTCHAR,AL
		RET
;
; Flush input buffers, write, or write with verify.  Resets the read pointer.
;
WRITE:		MOV	CS:READPTR,OFFSET DRIVERSEG
		MOV	CS:READCOUNT,0
		RET

;********************** INT 67H CODE
;
; Register save area on stack.
;
REGS		STRUC	[BP-14]
_ES		DW	?
_DS		DW	?
_DI		DW	?
_SI		DW	?
_DX		DW	?
_CX		DW	?
_BX		DW	?
_BP		DW	?
_IP		DW	?
_CS		DW	?
_FLAGS		DW	?
		ENDS
;
; Interrupt 67h handler.  Main entry point to memory manager.  When a
; memory manager function gets control, interrupts are enabled, direction
; is set to increment, BP addresses the registers on the stack (above),
; DS addresses the driver segment, AH is a meaningless value, and other
; registers are as the caller left them (including AL).  The function is
; expected to return a return code in AX.  Other registers, if used to
; return values, must be modified on the stack.
;
INT67HDLR:	PUSH	BP
		MOV	BP,SP
		CLD
		STI
		PUSH	BX
		PUSH	CX
		PUSH	DX
		PUSH	SI
		PUSH	DI
		PUSH	DS
		PUSH	ES
		PUSH	CS
		POP	DS
		CMP	AH,62h
		JA	INT67HDLR_BADFUNC
		SUB	AH,40h
		JB	INT67HDLR_BADFUNC
		MOV	BX,OFFSET INT67HDLR_RETADDR
		PUSH	BX
		MOV	BL,AH
		XOR	BH,BH
		SHL	BX,1
		PUSH	[BX+FUNCTIONJUMPS]
		MOV	BX,_BX
		RET
INT67HDLR_BADFUNC:
		MOV	AH,84h
INT67HDLR_RETADDR:
		POP	ES
		POP	DS
		POP	DI
		POP	SI
		POP	DX
		POP	CX
		POP	BX
		POP	BP
		IRET
;
; Invalid or unsupported function code.
;
FUNCBAD:	MOV	AH,84h
		RET
;
; Function 1:  Get Status.  Returns "OK."
;
FUNC1:		MOV	AH,0
		RET
;
; Function 2:  Get Page Frame Address.
;
FUNC2:		MOV	_BX,0D000h
		MOV	AH,0
		RET
;
; Function 3:  Get Unallocated Page Count.
;
FUNC3:		MOV	AX,UNALLOCPAGES
		MOV	_BX,AX
		MOV	AX,TOTALPAGES
		MOV	_DX,AX
		MOV	AH,0
		RET
;
; Function 4:  Allocate Pages.  This function lets function 27 do most of
; the work.
;
FUNC4:		OR	BX,BX
		JNZ	>L0
		MOV	AH,89h		; can't allocate 0 pages
		RET
L0:		MOV	AL,0
		JMP	FUNC27
;
; Function 5:  Map/Unmap Handle Pages.
;
FUNC5:		CMP	AL,4
		JB	>L0
		MOV	AH,8Bh		; physical page out of range
		RET
L0:		MOV	SI,BX
		CMP	BX,0FFFFh	; if request to unmap, go to it
		JE	>L3
		CALL	VERIFYHNDL	; otherwise, verify handle/logical pg
		JNC	>L2
		MOV	AH,83h		; invalid or inactive handle
		RET
L2:		CALL	LOG2HARD
		JNC	>L3
		MOV	AH,8Ah		; logical page invalid for handle
		RET
L3:		CALL	SETPHYSPAGE
		MOV	AH,0
		RET
;
; Function 6:  Deallocate Pages.  Deallocate all pages allocated to a handle
; and deactivate the handle (if not handle 0).
;
; First, verify the handle.
;
FUNC6:		CALL	VERIFYHNDL
		JNC	>L0
		MOV	AH,83h		; invalid or inactive handle
		RET
		;
		; Check that the handle does not have a context saved in the
		; internal save area (illegal to deactivate if it does).
		;
L0:		MOV	CX,20
		XOR	SI,SI
L1:		CMP	CONTEXTSAVE[SI],0
		JNE	>L2
		CMP	WORD PTR CONTEXTSAVE[SI+1],DX
		JNE	>L2
		MOV	AH,86h		; context saved, can't deallocate
		RET
L2:		ADD	SI,19
		LOOP	L1
		;
		; Context not saved, OK to deallocate.  First, save the
		; current number of pages in SI.
		;
		CALL	COUNTPAGES
		MOV	SI,BX
		;
		; Deallocate all pages.
		;
		XOR	BX,BX
		CALL	DEALLOCPAGES
		;
		; Return pages to free pool.
		;
		ADD	UNALLOCPAGES,SI
		;
		; If not handle 0, mark the handle free and return.
		;
		OR	DX,DX
		JZ	>L3
		MOV	SI,DX
		MOV	HNDLALLOC[SI],0FFh
L3:		MOV	AH,0
		RET
;
; Function 7:  Get Version.
;
FUNC7:		MOV	AX,0041h
		RET
;
; Function 8:  Save Page Map.  First, verify the handle.
;
FUNC8:		CALL	VERIFYHNDL
		JNC	>L0
		MOV	AH,83h		; invalid or inactive handle
		RET
		;
		; Handle OK.  Check for a context already saved by this
		; handle.
		;
L0:		MOV	BX,OFFSET CONTEXTSAVE
		MOV	CX,20
L1:		CMP	BYTE PTR [BX],0
		JNE	>L2
		CMP	[BX+1],DX
		JNE	>L2
		MOV	AH,8Dh		; context already saved for handle
		RET
L2:		ADD	BX,19
		LOOP	L1
		;
		; OK to save context.  Find a place.
		;
		MOV	BX,OFFSET CONTEXTSAVE
		MOV	CX,20
L3:		MOV	AL,0
		XCHG	AL,[BX]		; test and set
		OR	AL,AL
		JNZ	>L4
		ADD	BX,19
		LOOP	L3
		;
		; No free context save areas.
		;
		MOV	AH,8Ch		; context save area full
		RET
		;
		; Place found and reserved.  Save context.
		;
L4:		MOV	[BX+1],DX
		LEA	DI,[BX+3]
		PUSH	CS
		POP	ES
		CALL	GETCONTEXT
		MOV	AH,0
		RET
;
; Function 9:  Restore Page Map.  First, check the handle.
;
FUNC9:		CALL	VERIFYHNDL
		JNC	>L0
		MOV	AH,83h		; invalid or inactive handle
		RET
		;
		; Search for a context saved by this handle.
		;
L0:		MOV	BX,OFFSET CONTEXTSAVE
		MOV	CX,20
L1:		CMP	BYTE PTR [BX],0
		JNE	>L2
		CMP	[BX+1],DX
		JE	>L3
L2:		ADD	BX,19
		LOOP	L1
		;
		; Saved context not found.
		;
		MOV	AH,8Eh		; context not saved for handle
		RET
		;
		; Context found; restore, then free context save area.
		;
L3:		LEA	SI,[BX+3]
		CALL	SETCONTEXT
		LEA	DI,[BX+1]
		PUSH	CS
		POP	ES
		XOR	AX,AX
		MOV	CX,9
		CLD
		REP	STOSW
		MOV	BYTE PTR [BX],0FFh
		RET				; return code already 0
;
; Function 12:  Get Handle Count.
;
FUNC12:		MOV	CX,255
		XOR	AX,AX
		MOV	BX,AX
L0:		CMP	HNDLALLOC[BX],1
		ADC	AX,0
		INC	BX
		LOOP	L0
		MOV	_BX,AX
		MOV	AH,0
		RET
;
; Function 13:  Get Handle Pages.
;
FUNC13:		CALL	VERIFYHNDL
		JNC	>L0
		MOV	AH,83h		; invalid or inactive handle
		RET
L0:		CALL	COUNTPAGES
		MOV	_BX,BX
		MOV	AH,0
		RET
;
; Function 14:  Get All Handle Pages.
;
FUNC14:		MOV	CX,255
		XOR	AX,AX
		MOV	SI,AX
L0:		CMP	HNDLALLOC[SI],0
		JNE	>L1
		MOV	ES:[DI],SI
		ADD	DI,2
		MOV	DX,SI
		CALL	COUNTPAGES
		MOV	ES:[DI],BX
		ADD	DI,2
		INC	AX	
L1:		INC	SI
		LOOP	L0
		MOV	_BX,AX
		RET		; AH = 0 still (max handles is 255)
;
; Function 15:  Get/Set Page Map.
;
FUNC15:		CMP	AL,4
		JB	>L0
		MOV	AH,8Fh		; invalid subfunction code
		RET
L0:		MOV	DS,_DS		; DS:SI -> input structure
		MOV	BL,AL
		MOV	BH,0
		SHL	BX,1
		PUSH	CS:[BX+FUNC15JUMPS]
		MOV	BX,_BX
		RET
;
; Function 15, subfunction 0:  Get Page Map.
;
FUNC15S0:	CALL	GETCONTEXT
		MOV	AH,0
		RET
;
; Function 15, subfunction 1:  Set Page Map.
;
FUNC15S1:	CALL	VERIFYCONTEXT
		JNC	>L0
		MOV	AH,0A3h		; invalid mapping context
		RET
L0:		CALL	SETCONTEXT
		MOV	AH,0
		RET
;
; Function 15, subfunction 2:  Get & Set Page Map.
;
FUNC15S2:	CALL	VERIFYCONTEXT
		JNC	>L0
		MOV	AH,0A3h		; invalid mapping context
		RET
L0:		CALL	GETCONTEXT
		CALL	SETCONTEXT
		MOV	AH,0
		RET
;
; Function 15, subfunction 3:  Get Size of Page Map Array.  Return 16 bytes.
;
FUNC15S3:	MOV	AX,0010h
		RET
;
; Function 16:  Get/Set Partial Page Map.
;
FUNC16:		MOV	DS,_DS		; restore caller's DS from stack
		CMP	AL,2
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
L0:		MOV	BL,AL
		XOR	BH,BH
		ADD	BX,BX
		PUSH	CS:[BX+FUNC16JUMPS]
		MOV	BX,_BX
		RET
;
; Function 16, subfunction 0:  Get Partial Page Map.  If the input structure
; addressed by DS:SI is invalid, the output structure at ES:DI is undefined.
;
FUNC16S0:	CLD
		LODSW
		CMP	AX,4
		JBE	>L0
		MOV	AH,0A3h		; too many physical pages specified
		RET
L0:		STOSB			; put number of pages in output struct
		MOV	CX,AX
		JCXZ	>L3
L1:		LODSW			; get segment address
		CALL	SEG2PHYSPAGE	; convert to physical page number
		JNC	>L2
		MOV	AH,8Bh		; bad segment address specified
		RET
L2:		STOSB			; save physical page number
		CALL	GETPHYSPAGE
		STOSW			; save hardware page mapped in
		LOOP	L1
L3:		MOV	AH,0
		RET
;
; Function 16, subfunction 1:  Set Partial Page Map.  If the partial page
; map addressed by DS:SI is invalid, an error is returned, and the mapping
; context is not changed.
;
FUNC16S1:	CLD			; increment
		LODSB			; verify number of physical pages
		MOV	BX,SI		; BX saves address for setting step
		CMP	AL,4		; (invalid if more than 4)
		JA	FUNC16S1_INVALID
		MOV	CL,AL		; loop over entries in array
		XOR	CH,CH
		MOV	DX,CX		; DX saves CX value for setting step
		JCXZ	FUNC16S1_DONE	; if 0 physical pages, done
FUNC16S1_VERIFYLOOP:
		LODSB			; verify physical page number
		CMP	AL,4
		JAE	FUNC16S1_INVALID
		LODSW			; verify hardware page number
		CMP	AX,511
		JBE	>L0
		CMP	AX,0FFFFh
		JNE	FUNC16S1_INVALID
L0:		LOOP	FUNC16S1_VERIFYLOOP
		;
		; Partial context has been verified.  Set it now.
		;
		MOV	CX,DX
FUNC16S1_SETLOOP:
		MOV	AL,[BX]		; AL = physical page number
		MOV	SI,[BX+1]	; SI = hardware page number
		CALL	SETPHYSPAGE	; map hardware page into physical page
		ADD	BX,3		; go to next entry
		LOOP	FUNC16S1_SETLOOP
		;
		; Partial context has been set.
		;
FUNC16S1_DONE:	MOV	AH,0
		RET
		;
		; Partial context invalid, return error.
		;
FUNC16S1_INVALID:
		MOV	AH,0A3h
		RET
;
; Function 16, subfunction 2:  Get Size of Partial Page Map Save Array.
;
FUNC16S2:	CMP	BX,4
		JBE	>L0
		MOV	AH,8Bh		; too many physical pages specified
		RET
L0:		MOV	AL,BL		; return 3*(# physical pages) + 1
		ADD	AL,AL
		ADD	AL,BL
		INC	AL
		MOV	AH,0
		RET
;
; Function 17:  Map/Unmap Multiple Handle Pages.  Note:  this function, and
; its internal part, map pages up to the point at which an error occurs, as
; the specification requires.  That is, the array of pages to map is not
; verified before mapping begins.
;
FUNC17:		MOV	DS,_DS		; DS addresses caller's data
		;
		; Internal part.  This code is called by functions 22 and 23.
		; Modifies no registers except AH (i.e., preserves AL).
		;
FUNC17_INTERNAL:
		PUSH	BX
		PUSH	CX
		PUSH	SI
		PUSH	DI
		;
		; Verify physical page/segment address code.
		;
		CMP	AL,1
		JBE	>L0
		MOV	AH,8Fh		; invalid physical page/segment address
					;   code
		JMP	FUNC17_END
		;
		; Verify handle.
		;
L0:		CALL	VERIFYHNDL
		JNC	>L1
		MOV	AH,83h		; invalid or inactive handle
		JMP	FUNC17_END
		;
		; Verify number of pages to map/unmap.
		;
L1:		CMP	CX,4
		JBE	>L2
		MOV	AH,8Bh		; too many physical pages specified
		JMP	FUNC17_END
		;
		; If request to map no pages, do nothing and return.
		;
L2:		MOV	AH,0
		JCXZ	FUNC17_END	; if 0 pages specified, stop now
		;
		; Register use:
		;   AH = return code (0 unless changed)
		;   AL = physical page/segment address code
		;   BX = logical page (later, to save AX)
		;   CX = loop counter
		;   DX = handle
		;   SI = hardware page
		;   DS:DI -> caller's array
		;   BP not used
		;   ES not used
		;
		MOV	DI,SI		; DS:DI -> caller's array
FUNC17_LOOP:	MOV	BX,[DI]		; BX = logical page number
		ADD	DI,2
		MOV	SI,0FFFFh	; SI = hardware page number
		CMP	BX,SI
		JE	>L5
		CALL	LOG2HARD
		JNC	>L5
		MOV	AH,8Ah		; logical page out of range for handle
		JMP	FUNC17_END
L5:		MOV	BX,AX		; BH = 0, BL = log pg/seg code
		MOV	AX,[DI]		; AX = logical page/segment address
		ADD	DI,2
		OR	BL,BL
		JNZ	>L6
		;
		; Verify physical page number (if AL = 0).
		;
		CMP	AX,4
		JBE	>L7
		MOV	AX,8B00h	; physical page number out of range
		JMP	FUNC17_END
		;
		; Verify and convert segment address (if AL = 1).
		;
L6:		CALL	SEG2PHYSPAGE
		JNC	>L7
		MOV	AX,8B01h	; segment not a physical page
		JMP	FUNC17_END
		;
		; Physical page in AL, hardware page in SI.  Map (or unmap
		; if SI = 0FFFFh).
		;
L7:		CALL	SETPHYSPAGE
		MOV	AX,BX		; AH = 0, AL = phys pg/seg code again
		LOOP	FUNC17_LOOP
		;
		; Restore registers and return.
		;
FUNC17_END:	POP	DI
		POP	SI
		POP	CX
		POP	BX
		RET
;
; Function 18:  Reallocate Pages.  First, check the handle.
;
FUNC18:		CALL	VERIFYHNDL
		JNC	>L0
		MOV	AH,83h		; invalid or inactive handle
		RET
		;
		; Well, the handle is OK.  Save the number of pages requested 
		; in AX, determine how many pages the handle has now, and 
		; prepare to return that value to the caller in case of error.
		;
L0:		MOV	AX,BX
		CALL	COUNTPAGES
		MOV	_BX,BX
		;
		; Check if the request exceeds the total number of pages
		; installed.
		;
		CMP	AX,TOTALPAGES
		JBE	>L1
		MOV	AH,87h		; request exceeds total installed
		RET
		;
		; Is this a request for additional pages, a request to release
		; pages, or a request to do nothing (i.e., keep the same number
		; of pages)?
		;
L1:		CMP	AX,BX
		JNE	>L2
		;
		; Case 1:  same number of pages.  Do nothing and return
		; success.
		;
		MOV	AH,0
		RET
		;
		; Case 2:  smaller number of pages.  Call DEALLOCPAGES to
		; release some.
		;
L2:		JA	>L3
		XCHG	AX,BX		; BX = # to keep, AX = # there was
		CALL	DEALLOCPAGES
		MOV	_BX,BX		; return changed value to caller
		SUB	AX,BX		; AX = number deallocated
		ADD	UNALLOCPAGES,AX		; return pages to free pool
		MOV	AH,0		; return success
		RET
		;
		; Case 3:  larger number of pages.  AX = # you want, BX =
		; # you have.  Set AX = # additional to try to get.
		;
L3:		SUB	AX,BX
		;
		; Try to reserve the needed pages.  Do this with interrupts
		; disabled.
		;
		CLI
		SUB	UNALLOCPAGES,AX
		JNB	>L4
		ADD	UNALLOCPAGES,AX	; request exceeds number available
		STI
		MOV	AH,88h
		RET
		;
		; Pages reserved.  Allocate them for the handle.  Set AX =
		; the number of pages we have, BX = the number additional we
		; want, then call ALLOCPAGES.
		;
L4:		STI
		XCHG	AX,BX
		CALL	ALLOCPAGES
		;
		; Return changed number of pages to caller.
		;
		ADD	_BX,BX
		MOV	AH,0
		RET
;
; Function 19:  Get/Set Handle Attribute.  This does little.  Nonvolatile
; attribute is not supported.
;
FUNC19:		CMP	AL,2
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
		;
		; Subfunction 2:  Get Attribute Capability.  Return
		; "nonvolatile not supported."
		;
L0:		JNE	>L1
		XOR	AX,AX
		RET
		;
		; Other subfunctions:  return "Unsupported feature."
		;
L1:		MOV	AH,91h		; nonvolatile handles not supported
		RET
;
; Function 20:  Get/Set Handle Name.
;
FUNC20:		CMP	AL,1
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
L0:		CALL	VERIFYHNDL
		JNC	>L1
		MOV	AH,83h		; invalid or inactive handle
		RET
		;
		; Subfunction 0:  Get Handle Name.
		;
L1:		OR	AL,AL
		JNZ	>L2
		MOV	SI,DX
		MOV	CL,3
		SHL	SI,CL
		ADD	SI,OFFSET HANDLENAMES
		MOV	CX,4
		CLD
		REP	MOVSW
		MOV	AH,0
		RET
		;
		; Subfunction 1:  Set Handle Name.
		;
L2:		MOV	DS,_DS
		CALL	CHKNULLNAME		
		JC	>L3
		CALL	FINDNAME
		JC	>L3
		MOV	AH,0A1h			; name already exists
		RET
L3:		MOV	DI,DX
		MOV	CL,3
		SHL	DI,CL
		ADD	DI,OFFSET HANDLENAMES
		PUSH	CS
		POP	ES
		CLD
		MOV	CX,4
		REP	MOVSW
		MOV	AH,0
		RET
;
; Function 21:  Get Handle Directory.
;
FUNC21:		CMP	AL,2
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
		;
		; Subfunction 2:  Get Total Handles.  (Return constant 255.)
		;
L0:		JB	>L1
		MOV	_BX,255
		MOV	AH,0
		RET
		;
		; Subfunction 1:  Search for Named Handle.
		;
L1:		CMP	AL,1
		JNE	>L4
		;
		; Check if the name is null.
		;
		MOV	DS,_DS
		CALL	CHKNULLNAME
		JNC	>L2
		MOV	AH,0A1h			; specified name is null
		RET
		;
		; Name is not null.  Search for it.
		;
L2:		CALL	FINDNAME
		JNC	>L3
		MOV	AH,0A0h			; name not found
		RET
		;
		; Return name.
		;
L3:		MOV	_DX,AX
		MOV	AH,0
		RET
		;
		; Subfunction 0:  Get Handle Directory.
		;
L4:		XOR	AX,AX			; AL counts active handles
		MOV	BX,AX			; BX is handle number
		MOV	CX,255			; loop counter
		CLD				; increment
L5:		PUSH	CX
		CMP	HNDLALLOC[BX],0		; if handle inactive, skip
		JNZ	>L6
		MOV	ES:[DI],BX		; save handle number
		INC	DI
		INC	DI
		MOV	SI,BX			; get address of name in SI
		MOV	CL,3
		SHL	SI,CL
		ADD	SI,OFFSET HANDLENAMES
		MOV	CX,4			; copy name
		REP	MOVSW
		INC	AL			; bump the count of handles
L6:		INC	BX			; go to next
		POP	CX
		LOOP	L5
		RET				; (return code already set)
;
; Function 22:  Alter Page Map & Jump.  In case of error, we return to the
; caller without jumping.
;
; Call FUNC17_INTERNAL to map in requested pages.
;
FUNC22:		MOV	DS,_DS		; DS:SI -> caller's structure
		MOV	CL,[SI+4]
		MOV	CH,0
		LDS	SI,[SI+5]
		CALL	FUNC17_INTERNAL
		OR	AH,AH		; in case of error, return to caller
		JZ	>L1		; (FUNC17_INTERNAL sets the error code)
		RET
		;
		; Pages mapped in, OK to jump.  Do this by modifying the
		; return address for INT67HDLR.
		;
L1:		MOV	DS,_DS
		MOV	SI,_SI
		MOV	AX,[SI]
		MOV	_IP,AX
		MOV	AX,[SI+2]
		MOV	_CS,AX
		;
		; Set return code for success.  Caller's registers and flags
		; will be restored by INT67HDLR before the "jump."
		;
		MOV	AH,0
		RET
;
; Function 23:  Alter Page Map & Call.  An error may occur either before
; or after the call is made.  If before, we return immediately without calling.
; If an error code is returned, the application must determine whether the
; requested subroutine was called or not.
;
FUNC23:		CMP	AL,2
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
		;
		; Subfunction 2:  Get Page Map Stack Space Size.  Return
		; 28 bytes; see note below.  If subfunctions 0 and 1 are
		; modified, this value may need to be changed.
		;
L0:		JB	FUNC23_SUBF_0_1
		MOV	AH,0
		MOV	_BX,28
		RET
		;
		; Subfunctions 0 and 1:  Alter Page Map & Call.  First,
		; call FUNC17_INTERNAL to map in requested pages.
		;
FUNC23_SUBF_0_1:
		MOV	DS,_DS		; DS:SI -> caller's structure
		MOV	CL,[SI+4]
		MOV	CH,0
		LDS	SI,[SI+5]
		CALL	FUNC17_INTERNAL
		OR	AH,AH		; if error, return
		JZ	>L1		; (return code already in AH)
		RET
		;
		; Pages mapped in OK.  We can call.  We need to save our own
		; BP so that we will be able to get original register values
		; back and save registers returned for the caller, after the
		; called subroutine returns.  We also need to save AX so that 
		; we will have the physical page/segment address code in AL 
		; when the subroutine returns.  Note that FUNC17_INTERNAL
		; must *not* modify AL, or we're sunk here.
		;
L1:		PUSH	AX
		PUSH	BP
		;
		; Restore caller's flags for the subroutine.
		;
		PUSH	_FLAGS
		POPF
		;
		; Restore registers to caller's values.
		;
		MOV	CX,_CX
		MOV	DS,_DS
		MOV	SI,_SI
		MOV	BP,_BP
		;
		; Call the subroutine.  Note:  subfunction 2 returns the
		; number of bytes pushed on the stack from the time the
		; caller invoked INT 67h to here, including the flags and
		; return address pushed by the INT instruction.  Note also
		; that AH was set to 0 by FUNC17_INTERNAL, so we pass a
		; "success" code to the subroutine.
		;
		CALL	DWORD PTR [SI]
		;
		; The caller's subroutine returns here.  We need to return
		; flags and all registers except AX, which will contain a
		; return code.
		;
		; First, get our BP back, then save BP returned and flags
		; returned.
		;
		MOV	AX,BP
		POP	BP
		MOV	_BP,AX
		PUSHF
		POP	_FLAGS
		;
		; Get old DS back and save DS returned.
		;
		MOV	AX,_DS
		MOV	_DS,DS
		MOV	DS,AX
		;
		; Get old DX and SI back, and save DX and SI returned.
		;
		XCHG	DX,_DX
		XCHG	SI,_SI
		;
		; Save the rest of the registers returned.
		;
		MOV	_BX,BX
		MOV	_CX,CX
		MOV	_DI,DI
		MOV	_ES,ES
		;
		; Get old AX back.
		;
		POP	AX
		;
		; Call FUNC17_INTERNAL to map in pages for the return.
		;
		MOV	CL,[SI+9]
		MOV	CH,0
		LDS	SI,[SI+10]
		CALL	FUNC17_INTERNAL
		;
		; That's it.  Return code is from FUNC17_INTERNAL.  INT67HDLR
		; will return the registers (they're stored on the stack).
		;
		RET
;
; Function 24:  Move/Exchange Memory Region.  Memory is moved in units of
; up to 16k (i.e., up to the next 16k boundary on either the source or
; destination region).  Physical page 0 is used for the source if expanded,
; and physical page 1 is used for the destination if expanded.
;
; Stack frame:
;
FUNC24LOCAL	STRUC	[BP-22]
		;
		; Various flags:
		;   bit   meaning
		;    0    0 = source is conventional, 1 = source is expanded
		;    1    0 = dest is conventional, 1 = dest is expanded
		;    2    0 = move, 1 = exchange
		;    3    0 = no overlap, 1 = overlap
		;    4    0 = move/exchange bottom-up, 1 = move top-down
		;    5    1 = source count was used in last block move/exchange
		;    6    1 = dest count was used in last block move/exchange
		;    7    unused
		;
FLAGS24		DB	?	
BYTESLEFT	DB	3 DUP (?)	; total bytes left to move
		;
		; Source and destination pointers.  If conventional, these
		; are in segment:offset form, with an offset of less than
		; 16k.  If expanded, the low word is an offset into an
		; expanded memory page, and the high word is a logical page
		; number.
		;
SOURCEPTR	DD	?
DESTPTR		DD	?

SOURCEENDPTR	LABEL	DWORD	; pointer to last byte of source
PG0SAVE		DW	?	; saved context for page 0
PG1SAVE		DW	?	; saved context for page 1

DESTENDPTR	LABEL	DWORD	; pointer to last byte of dest
SOURCECOUNT	DW	?	; count to next 16k boundary on source
DESTCOUNT	DW	?	; count to next 16k boundary on dest

SOURCEHANDLE	DB	?	; low byte of source handle, if expanded
DESTHANDLE	DB	?	; low byte of dest handle, if expanded
		ENDS
;
; Main routine for function 24:
;
; Set DS back to caller's value.  Save BP, and allocate a 22-byte stack
; frame for local variables.  Note:  DS *never* addresses the driver
; segment in this function.
;
FUNC24:		MOV 	DS,_DS
		PUSH	BP
		MOV	BP,SP
		SUB	SP,22
		;
		; Check subfunction code and store.  Flags other than the
		; subfunction code are initially zero.
		;
		CMP	AL,1
		JNA	>L00
		MOV 	AH,8Fh		; invalid subfunction code
		JMP	FUNC24EXIT
L00:		SHL	AL,1		
		SHL	AL,1
		MOV	FLAGS24,AL
		;
		; Check length of move/exchange and store.
		;
		MOV	AX,[SI]
		MOV	DX,[SI+2]
		CMP	DX,10h			; count >= 1M?
		JB	>L01
		JA	>L03
		OR	AX,AX
		JZ	>L02
		JMP	>L03
L01:		OR	DX,DX		; count < 1M.  Count = 0?
		JNZ	>L02
		OR	AX,AX
		JNZ	>L02
		JMP	FUNC24EXIT	; count = 0, exit (successful)
L02:		MOV	WORD PTR BYTESLEFT,AX	; 0 < count <= 1M, save count
		MOV	BYTESLEFT+2,DL
		JMP	>L04
L03:		MOV	AH,96h		; count > 1M, exit (error)
		JMP	FUNC24EXIT
		;
		; Set variables for source region.  Is the source conventional?
		; 
L04:		CMP	BYTE PTR [SI+4],1
		JBE	>L041
		MOV	AH,98h		; invalid memory type
		JMP	FUNC24EXIT
L041:		JE	>L08
		;
		; Source is conventional.  Save starting segment:offset.
		;
		MOV	AX,[SI+7]
		MOV	DX,[SI+9]
		CALL	CONV16K		; adjust starting offset to < 16k
		JNC	>L05
		MOV	AH,0A2h		; source starting address >= 1M
		JMP	FUNC24EXIT
L05:		MOV	WORD PTR SOURCEPTR,AX	; starting address OK, save
		MOV	WORD PTR SOURCEPTR+2,DX
		CALL	CONVSEG2LIN		; convert to linear address
		MOV	BX,DX		; save linear address
		MOV	CX,AX
		ADD	AX,[SI]		; get linear address of last byte
		ADC	DX,[SI+2]
		SUB	AX,1
		SBB	DX,0
		PUSH	AX		; save linear address on stack
		PUSH	DX
		CALL	CONVLIN2SEG	; convert back to segment:offset
		JNC	>L06
		ADD	SP,4		; attempt to wrap around 1M boundary
		MOV	AH,0A2h
		JMP	FUNC24EXIT
L06:		MOV	WORD PTR [BP]SOURCEENDPTR,AX	; save
		MOV	WORD PTR [BP]SOURCEENDPTR+2,DX
		POP	DX		; BX:CX = linear address of 1st byte
		POP	AX		; DX:AX = linear address of last byte
		CALL	VERIFYPGFRAME	; overlaps EMS page frame?
		JNC	>L07
		MOV	AH,94h		; conv region overlaps page frame
		JMP	FUNC24EXIT
L07:		JMP	FUNC24GETDEST
		;
		; Source is expanded.  Set flag in flag byte.
		;
L08:		OR	FLAGS24,1
		MOV	DX,[SI+5]		; get source handle
		CALL	VERIFYHNDL
		JNC	>L09
		MOV	AH,83h		; source handle invalid or inactive
		JMP	FUNC24EXIT
L09:		MOV	SOURCEHANDLE,DL		; source handle OK, save
		MOV	AX,[SI+7]		; get source offset
		CMP	AX,16384
		JB	>L10	
		MOV	AH,95h			; source offset >= 16k
		JMP	FUNC24EXIT
L10:		MOV	WORD PTR SOURCEPTR,AX	; save source offset
		MOV	BX,[SI+9]		; get logical page
		CALL	VERIFYLOGICAL
		JNC	>L11
		MOV	AH,8Ah		; starting logical page invalid
		JMP	FUNC24EXIT
L11:		MOV	WORD PTR SOURCEPTR+2,BX	; save logical page
		;
		; Get ending source logical page.  Convert starting logical 
		; page/offset into a linear address.
		;
		XCHG	DX,BX		; save handle in BX
		CALL	CONVEXP2LIN
		ADD	AX,[SI]		; add count to move/exchange
		ADC	DX,[SI+2]
		SUB	AX,1		; subtract one to address last byte
		SBB	DX,0
		CALL	CONVLIN2EXP	; convert back to logical page/offset
		XCHG	DX,BX		; get handle back
		CALL	VERIFYLOGICAL
		JNC	>L12
		MOV	AH,93h		; expanded memory region too long
		JMP	FUNC24EXIT
L12:		MOV	WORD PTR [BP]SOURCEENDPTR,AX	; save ending address
		MOV	WORD PTR [BP]SOURCEENDPTR+2,BX	
		;
		; The source region has been verified.  Set variables for 
		; destination region, and verify it too.
		;
FUNC24GETDEST:	CMP	BYTE PTR [SI+11],1
		JBE	>L121
		MOV	AH,98h		; invalid memory type
		JMP	FUNC24EXIT
L121:		JE	>L16
		;
		; Dest is conventional.  Save starting segment:offset.
		;
		MOV	AX,[SI+14]
		MOV	DX,[SI+16]
		CALL	CONV16K		; adjust starting offset to < 16k
		JNC	>L13
		MOV	AH,0A2h		; dest starting address >= 1M
		JMP	FUNC24EXIT
L13:		MOV	WORD PTR DESTPTR,AX	; starting address OK, save
		MOV	WORD PTR DESTPTR+2,DX
		CALL	CONVSEG2LIN		; convert to linear address
		MOV	BX,DX		; save linear address
		MOV	CX,AX
		ADD	AX,[SI]		; get linear address of last byte
		ADC	DX,[SI+2]
		SUB	AX,1
		SBB	DX,0
		PUSH	AX		; save linear address on stack
		PUSH	DX
		CALL	CONVLIN2SEG	; convert back to segment:offset
		JNC	>L14
		ADD	SP,4		; attempt to wrap around 1M boundary
		MOV	AH,0A2h
		JMP	FUNC24EXIT
L14:		MOV	WORD PTR [BP]DESTENDPTR,AX		; save
		MOV	WORD PTR [BP]DESTENDPTR+2,DX
		POP	DX		; BX:CX = linear address of 1st byte
		POP	AX		; DX:AX = linear address of last byte
		CALL	VERIFYPGFRAME	; overlaps EMS page frame?
		JNC	>L15
		MOV	AH,94h		; conv region overlaps page frame
		JMP	FUNC24EXIT
L15:		JMP	FUNC24CHKOVRLAP
		;
		; Dest is expanded.  Set flag in flag byte.
		;
L16:		OR	FLAGS24,2
		MOV	DX,[SI+12]		; get dest handle
		CALL	VERIFYHNDL
		JNC	>L17
		MOV	AH,83h		; dest handle invalid or inactive
		JMP	FUNC24EXIT
L17:		MOV	DESTHANDLE,DL		; dest handle OK, save
		MOV	AX,[SI+14]		; get dest offset
		CMP	AX,16384
		JB	>L18	
		MOV	AH,95h			; dest offset >= 16k
		JMP	FUNC24EXIT
L18:		MOV	WORD PTR DESTPTR,AX	; save dest offset
		MOV	BX,[SI+16]		; get logical page
		CALL	VERIFYLOGICAL
		JNC	>L19
		MOV	AH,8Ah		; starting logical page invalid
		JMP	FUNC24EXIT
L19:		MOV	WORD PTR DESTPTR+2,BX	; save logical page
		;
		; Get ending dest logical page.  Convert starting logical 
		; page/offset into a linear address.
		;
		XCHG	DX,BX		; save handle in BX
		CALL	CONVEXP2LIN
		ADD	AX,[SI]		; add count to move/exchange
		ADC	DX,[SI+2]
		SUB	AX,1		; subtract one to address last byte
		SBB	DX,0
		CALL	CONVLIN2EXP	; convert back to logical page/offset
		XCHG	DX,BX		; get handle back
		CALL	VERIFYLOGICAL
		JNC	>L20
		MOV	AH,93h		; expanded memory region too long
		JMP	FUNC24EXIT
L20:		MOV	WORD PTR [BP]DESTENDPTR,AX	; save ending address
		MOV	WORD PTR [BP]DESTENDPTR+2,BX	
		;
		; Source and destination regions both verified, length of
		; move/exchange verified, subfunction code verified.  Now
		; check for overlap.  DI is a code, 0 = both conventional,
		; 1 = both expanded.
		;
FUNC24CHKOVRLAP:
		CLD			; assume bottom-up
		TEST	FLAGS24,3
		JPE	>L21
		JMP	FUNC24SAVCONT	; one expanded, one conventional
L21:		MOV	DI,0		; assume both conventional
		JZ	>L22
		MOV	DI,1		; both expanded
		MOV	AL,SOURCEHANDLE	; if not same handle, no overlap
		CMP	AL,DESTHANDLE
		JE	>L22
		JMP	FUNC24SAVCONT
		;
		; If source.hi < dest.lo, no overlap.
		;
L22:		MOV	AX,WORD PTR DESTPTR
		MOV	DX,WORD PTR DESTPTR+2
		CALL	CONVBOTH
		MOV	CX,AX
		MOV	BX,DX
		MOV	AX,WORD PTR [BP]SOURCEENDPTR
		MOV	DX,WORD PTR [BP]SOURCEENDPTR+2
		CALL	CONVBOTH
		CALL	CMP32
		JB	FUNC24SAVCONT
		;
		; Else if dest.hi < source.lo, no overlap.
		;
		MOV	AX,WORD PTR SOURCEPTR
		MOV	DX,WORD PTR SOURCEPTR+2
		CALL	CONVBOTH
		MOV	CX,AX
		MOV	BX,DX
		MOV	AX,WORD PTR [BP]DESTENDPTR
		MOV	DX,WORD PTR [BP]DESTENDPTR+2
		CALL	CONVBOTH
		CALL	CMP32
		JB	FUNC24SAVCONT
		;
		; Else overlap.  If this is an exchange, error.
		;
		TEST	FLAGS24,4
		JZ	>L23
		MOV	AH,97h		; overlap on exchange
		JMP	FUNC24EXIT
L23:		OR	FLAGS24,8	; set overlap bit in flag byte
		;
		; If dest.lo > source.lo, copy top-down.
		;
		MOV	AX,WORD PTR DESTPTR
		MOV	DX,WORD PTR DESTPTR+2
		CALL	CONVBOTH
		CALL	CMP32
		JNA	FUNC24SAVCONT
		OR	FLAGS24,10h	; set top-down bit in flag byte
		;
		; Direction flag set here.  Note that it is important to
		; ensure that none of the subroutines called below clears
		; the flag.
		;
		STD			; set direction flag (top-down)
		;
		; Going top-down.  Set SOURCEPTR, DESTPTR to point to last
		; byte.  We're through with SOURCEENDPTR and DESTENDPTR;
		; their memory can be used to save context now.
		;
		MOV	AX,WORD PTR [BP]SOURCEENDPTR
		MOV	WORD PTR SOURCEPTR,AX
		MOV	AX,WORD PTR [BP]SOURCEENDPTR+2
		MOV	WORD PTR SOURCEPTR+2,AX
		MOV	AX,WORD PTR [BP]DESTENDPTR
		MOV	WORD PTR DESTPTR,AX
		MOV	AX,WORD PTR [BP]DESTENDPTR+2
		MOV	WORD PTR DESTPTR+2,AX
		;
		; Overlap taken care of, direction flag set.  Save partial 
		; context for physical pages 0 and 1.
		;
FUNC24SAVCONT:	MOV	AL,0
		CALL	GETPHYSPAGE
		MOV	PG0SAVE,AX
		MOV	AL,1
		CALL	GETPHYSPAGE
		MOV	PG1SAVE,AX
		;
		; Initialize source.  DS:SI -> source data, source mapped 
		; into page 0 if expanded.
		;
		TEST	FLAGS24,1
		JZ	>L24
		MOV	DL,SOURCEHANDLE		; source is expanded
		MOV	DH,0
		MOV	BX,WORD PTR SOURCEPTR+2
		CALL	LOG2HARD		; map first logical page
		MOV	AL,0
		CALL	SETPHYSPAGE
		MOV	AX,0D000h		; set DS -> physical page 0
		MOV	DS,AX
		JMP	>L25
L24:		MOV	DS,WORD PTR SOURCEPTR+2	; source is conventional
		;
		; Initialize dest.  ES:DI -> dest data, dest mapped into
		; page 1 if expanded.
		;
L25:		TEST	FLAGS24,2
		JZ	>L26
		MOV	DL,DESTHANDLE		; dest is expanded
		MOV	DH,0
		MOV	BX,WORD PTR DESTPTR+2
		CALL 	LOG2HARD
		MOV	AL,1
		CALL	SETPHYSPAGE
		MOV	AX,0D400h		; set ES -> physical page 1
		MOV	ES,AX
		JMP	FUNC24LOOP
L26:		MOV	ES,WORD PTR DESTPTR+2	; dest is conventional
		;
		; Main loop begins here.  Set SI, DI here to avoid conflict 
		; with LOG2HARD and SETPHYSPAGE.
		;
FUNC24LOOP:	MOV	SI,WORD PTR SOURCEPTR
		MOV	DI,WORD PTR DESTPTR
		;
		; Compute SOURCECOUNT, DESTCOUNT.
		;
		TEST	FLAGS24,10h
		JNZ	>L28
		MOV	AX,16384	; going bottom-up
		MOV	BX,AX
		SUB	AX,SI
		SUB	BX,DI
		MOV	SOURCECOUNT,AX
		MOV	DESTCOUNT,BX
		JMP	>L29
L28:		MOV	SOURCECOUNT,SI	; going top-down
		MOV	DESTCOUNT,DI
		INC	SOURCECOUNT
		INC	DESTCOUNT
		;
		; Find the least of BYTESLEFT, SOURCECOUNT and DESTCOUNT.
		; Place the result in CX.  While you're at it, set the 
		; relevant bits in the flag byte.
		;
L29:		AND	FLAGS24,9Fh	; both bits initially off
		;
		; We first eliminate BYTESLEFT (if this is least, we don't
		; bother to set bits for the other two).
		;
		CMP	BYTESLEFT+2,0
		JNE	>L30
		MOV	CX,WORD PTR BYTESLEFT
		CMP	CX,SOURCECOUNT
		JA	>L30
		CMP	CX,DESTCOUNT
		JNA	>L33		; jump if BYTESLEFT is least
		;
		; BYTESLEFT is not least.  Must be SOURCECOUNT or DESTCOUNT.
		;
L30:		MOV	CX,SOURCECOUNT
		CMP	CX,DESTCOUNT
		JA	>L31
		JE	>L32
		OR	FLAGS24,20h	; SOURCECOUNT is unique least
		JMP	>L33
L31:		MOV	CX,DESTCOUNT	; DESTCOUNT is unique least
		OR	FLAGS24,40h
		JMP	>L33
L32:		OR	FLAGS24,60h	; both SOURCECOUNT and DESTCOUNT least
		;
		; CX is the amount we will move/exchange this time.  Subtract
		; from BYTESLEFT.
		;
L33:		SUB	WORD PTR BYTESLEFT,CX
		SBB	BYTESLEFT+2,0
		;
		; Shift CX right to move words (for speed).  Save carry.
		;
		SHR	CX,1
		PUSHF
		;
		; Move if moving ...
		;
		TEST	FLAGS24,4
		JNZ	>L332
		JCXZ	>L331		; first move words
		REP	MOVSW
L331:		POPF			; move extra byte if present
		JNC	FUNC24UPDATE
		MOVSB
		JMP	FUNC24UPDATE
		;
		; Or exchange if exchanging ...  (Note:  we will never
		; exchange top-down.)
		;
L332:		JCXZ	>L341		; first exchange words
L34:		MOV	AX,ES:[DI]
		MOVSW
		MOV	[SI-2],AX
		LOOP	L34
L341:		POPF			; exchange extra byte if present
		JNC	FUNC24UPDATE
		MOV	AL,ES:[DI]
		MOVSB
		MOV	[SI-1],AL
		;
		; Save SI, DI.
		;
FUNC24UPDATE:	MOV	WORD PTR SOURCEPTR,SI
		MOV	WORD PTR DESTPTR,DI
		;
		; Update source, if SOURCECOUNT was least.
		;
		TEST	FLAGS24,20h
		JZ	FUNC24UPDDEST
		TEST	FLAGS24,10h
		JNZ	>L36				; bottom-up
		MOV	WORD PTR SOURCEPTR,0
		TEST	FLAGS24,1
		JNZ	>L35
		MOV	AX,DS			; bottom-up, conventional
		ADD	AX,1024
		JMP	>L38
L35:		INC	WORD PTR SOURCEPTR+2	; bottom-up, expanded
		JMP	>L38
L36:		MOV	WORD PTR SOURCEPTR,16383	; top-down
		TEST	FLAGS24,1
		JNZ	>L37
		MOV	AX,DS			; top-down, conventional
		SUB	AX,1024
		JMP	>L38
L37:		DEC	WORD PTR SOURCEPTR+2	; top-down, expanded
L38:		TEST	FLAGS24,1		; all cases
		JNZ	>L39
		MOV	DS,AX			; source is conventional
		MOV	WORD PTR SOURCEPTR+2,AX
		JMP	FUNC24UPDDEST
L39:		MOV	BX,WORD PTR SOURCEPTR+2	; source is expanded
		MOV	DL,SOURCEHANDLE
		XOR	DH,DH
		CALL	LOG2HARD
		MOV	AL,0
		CALL	SETPHYSPAGE
		;
		; Update dest, if DESTCOUNT was least.
		;
FUNC24UPDDEST:	TEST	FLAGS24,40h
		JZ	FUNC24LOOPTEST
		TEST	FLAGS24,10h
		JNZ	>L42				; bottom-up
		MOV	WORD PTR DESTPTR,0
		TEST	FLAGS24,2
		JNZ	>L41
		MOV	AX,ES			; bottom-up, conventional
		ADD	AX,1024
		JMP	>L44
L41:		INC	WORD PTR DESTPTR+2	; bottom-up, expanded
		JMP	>L44
L42:		MOV	WORD PTR DESTPTR,16383	; top-down
		TEST	FLAGS24,2
		JNZ	>L43
		MOV	AX,ES			; top-down, conventional
		SUB	AX,1024
		JMP	>L44
L43:		DEC	WORD PTR DESTPTR+2	; top-down, expanded
L44:		TEST	FLAGS24,2		; all cases
		JNZ	>L45
		MOV	ES,AX			; dest is conventional
		MOV	WORD PTR DESTPTR+2,AX
		JMP	FUNC24LOOPTEST
L45:		MOV	BX,WORD PTR DESTPTR+2	; dest is expanded
		MOV	DL,DESTHANDLE
		XOR	DH,DH
		CALL	LOG2HARD
		MOV	AL,1
		CALL	SETPHYSPAGE
		;
		; Go again if more to move/exchange.
		;
FUNC24LOOPTEST:	CMP	WORD PTR BYTESLEFT,0
		JNE	>L47
		CMP	BYTESLEFT+2,0
		JNE	>L47
		JMP	FUNC24LOOPEND
L47:		JMP	FUNC24LOOP
		;
		; Restore partial context on physical pages 0 and 1.
		;
FUNC24LOOPEND:	MOV	AL,0
		MOV	SI,PG0SAVE
		CALL	SETPHYSPAGE
		MOV	AL,1
		MOV	SI,PG1SAVE
		CALL	SETPHYSPAGE
		;
		; If overlap, return code is 92h, else 0.
		;
		TEST	FLAGS24,8
		JZ	>L48
		MOV	AH,92h
		JMP	FUNC24EXIT
L48:		MOV	AH,0
FUNC24EXIT:	ADD	SP,22
		POP	BP
		RET		; end of FUNC24
;
; Function 25:  Get Mappable Physical Address Array.
;
FUNC25:		CMP	AL,1
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
		;
		; Subfunction 1:  Get Mappable Physical Address Array Entries.
		;
L0:		JNE	>L1
		MOV	_CX,4		; four physical pages available
		MOV	AH,0
		RET
		; 
		; Subfunction 0:  Get Mappable Physical Address Array.
		;
L1:		MOV	CX,4
		MOV	_CX,CX
		MOV	AX,0D000h
		XOR	BX,BX
L2:		MOV	ES:[DI],AX
		MOV	ES:[DI+2],BX
		ADD	DI,4
		ADD	AX,400h
		INC	BX
		LOOP	L2
		MOV	AH,0
		RET
;
; Function 26:  Get Expanded Memory Hardware Information.
;
FUNC26:		CMP	AL,1
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
		;
		; Subfunction 1:  Get Unallocated Raw Page Count.
		;
L0:		JNE	>L1
		JMP	FUNC3		; save info returned as Function 3
		;
		; Subfunction 0:  Get Hardware Configuration Array.
		;
L1:		CMP	PERMISSION,1
		JE	>L2
		MOV	AH,0A4h			; access denied
		RET
L2:		CLD
		MOV	AX,1024
		STOSW
		XOR	AX,AX
		STOSW
		MOV	AX,16
		STOSW
		XOR	AX,AX
		STOSW
		STOSW
		RET
;
; Function 27:  Allocate Standard/Raw Pages.  Makes no distinction between
; the two; our "raw" pages are 16k.
;
FUNC27:		CMP	AL,1	; check subfunction code anyway
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
		;
		; Subfunction code OK.  See if a handle can be had.
		;
L0:		MOV	CX,255
		XOR	SI,SI
		;
		; Loop over the possible handles.  If one is free, allocate
		; it.
		;
L1:		MOV	AL,0
		XCHG	AL,HNDLALLOC[SI]	; test and set
		OR	AL,AL
		JNZ	>L2
		INC	SI
		LOOP	L1
		MOV	AH,85h		; no free handles
		RET
		;
		; Handle found.  Reserve the requested number of pages, if
		; possible.
		;
L2:		MOV	_DX,SI		; save handle into caller's DX
		CMP	TOTALPAGES,BX
		JNB	>L3
		MOV	AH,87h		; request exceeds total installed
		JMP	>L4
L3:		CLI
		SUB	UNALLOCPAGES,BX
		JNB	>L5
		ADD	UNALLOCPAGES,BX	; request exceeds currently available
		STI
		MOV	AH,88h
L4:		MOV	HNDLALLOC[SI],0FFh
		RET
		;
		; Requested number of pages reserved.  Allocate them.
		;
L5:		STI
		MOV	DX,SI
		XOR	AX,AX		; start at logical page 0
		CALL	ALLOCPAGES
		;
		; Return success.  (Code already set.)
		;
		RET
;
; Function 28:  Alternate Map Register Set.  Most of these subfunctions are
; not supported and perform no operation (other than perhaps checking 
; parameters).  These are OS/E functions.
;
FUNC28:		CMP	AL,8
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		RET
L0:		CMP	PERMISSION,1
		JE	>L1
		MOV	AH,0A4h		; access denied
		RET
L1:		MOV	BL,AL
		MOV	BH,0
		SHL	BX,1
		PUSH	[BX+FUNC28JUMPS]
		MOV	BX,_BX
		RET
;
; Function 28, subfunction 0:  Get Alternate Map Register Set.
;
FUNC28S0:	MOV	BL,0		; BL = 0 (constant)
		LES	DI,FUNC28PTR	; ES:DI = alt map reg set pointer
		OR	DI,DI		; check if pointer is null
		JNE	>L0
		MOV	AX,ES
		OR	AX,AX
		JE	>L1
L0:		CALL	GETCONTEXT	; save context if pointer is not null
L1:		MOV	AH,0
		MOV	_BX,BX		; save values returned
		MOV	_ES,ES
		MOV	_DI,DI
		RET
;
; Function 28, subfunction 1:  Set Alternate Map Register Set.
;
FUNC28S1:	CMP	BL,0
		JE	>L0
		MOV	AH,9Ch		; alt map reg set specified not zero
		RET
		;
		; Check whether the context save area pointer passed in
		; ES:DI is null.
		;
L0:		MOV	AX,ES
		OR	AX,AX
		JNZ	>L1
		OR	DI,DI
		JZ	>L3
		;
		; Pointer is not null.  Verify and set context.
		;
L1:		MOV	DS,AX
		MOV	SI,DI
		CALL	VERIFYCONTEXT	
		JNC	>L2
		MOV	AH,0A3h			; invalid mapping context
		RET
L2:		CALL	SETCONTEXT
		PUSH	CS
		POP	DS
		;
		; Jump to here if pointer is null.  Save the pointer.
		;
L3:		MOV	WORD PTR FUNC28PTR,DI
		MOV	WORD PTR FUNC28PTR+2,ES
		MOV	AH,0
		RET
;
; Function 28, subfunction 2:  Get Alternate Map Save Array Size.  Return
; constant 16.
;
FUNC28S2:	MOV	_DX,16
		MOV	AH,0
		RET
;
; Function 28, subfunctions 3 and 5:  Allocate Alternate Map Register Set
; and Allocate DMA Register Set.  These functions are unsupported and do
; nothing but return 0 in BL.
;
FUNC28S3:	MOV	BL,0
		MOV	_BX,BX
		MOV	AH,0
		RET
;
; Function 28, subfunctions 4, 6, 7, and 8:  Deallocate Alternate Map Register
; Set, Enable DMA on Alternate Map Register Set, Disable DMA on Alternate Map
; Register Set, and Deallocate DMA Register Set.  These functions are unsup-
; ported and do nothing but verify that BL = 0 on entry.
;
FUNC28S4:	CMP	BL,0
		JE	>L0
		MOV	AH,9Ch		; register set not zero
		RET
L0:		MOV	AH,0
		RET
;
; Function 29:  Prepare Expanded Memory for Warm Boot.  A no-op.
;
FUNC29:		MOV	AH,0
		RET
;
; Function 30:  Enable/Disable OS/E Function Set Functions.  Interrupts
; disabled for this function, based on the vague impression that it has
; something to do with security.
;
FUNC30:		CLI
		CMP	AL,2
		JBE	>L0
		MOV	AH,8Fh		; invalid subfunction
		JMP	FUNC30_END
		;
		; Subfunction 2:  Return Access Key.  Key must be active
		; for this function.
		;
L0:		JB	>L1
		CMP	KEYACTIVE,1
		JNE	FUNC30_ACCESS_DENIED
		CMP	ACCESS_LO,CX
		JNE	FUNC30_ACCESS_DENIED
		CMP	ACCESS_HI,BX
		JNE	FUNC30_ACCESS_DENIED
		;
		; Access OK.  Inactivate key and enable OS/E functions.
		;
		MOV	PERMISSION,1
		MOV	KEYACTIVE,0
		MOV	AH,0
		JMP	FUNC30_END
		;
		; Subfunctions 0 and 1:  Enable or Disable OS/E Function Set 
		; Functions.  If the key is active, verify it.
		;
L1:		CMP	KEYACTIVE,1
		JNE	>L2
		CMP	ACCESS_LO,CX
		JNE	FUNC30_ACCESS_DENIED
		CMP	ACCESS_HI,BX
		JNE	FUNC30_ACCESS_DENIED
		JMP	>L3
		;
		; If the key is inactive, generate, store, and return a new 
		; random key, and activate the key.  The high 16 bits of the
		; key is the low word of the system clock count.
		;
L2:		MOV	BX,40h
		MOV	ES,BX
		MOV	BX,6Ch
		MOV	BX,ES:[BX]
		MOV	ACCESS_HI,BX
		MOV	_BX,BX
		;
		; The low 16 bits of the key is the current timer channel 0
		; counter value.
		;
		MOV	AH,AL		; AH saves subfunction code
		MOV	AL,0
		OUT	43h,AL
		JMP	$+2
		IN	AL,40h
		JMP	$+2
		MOV	CL,AL
		IN	AL,40h
		JMP	$+2
		MOV	CH,AL
		MOV	ACCESS_LO,CX
		MOV	_CX,CX
		MOV	KEYACTIVE,1
		MOV	AL,AH		; AL is subfunction again
		;
		; If subfunction 0, enable OS/E functions.  If subfunction 1,
		; disable them.
		;
L3:		XOR	AL,1
		MOV	PERMISSION,AL
		MOV	AH,0
		JMP	FUNC30_END
FUNC30_ACCESS_DENIED:
		MOV	AH,0A4h			; access denied
FUNC30_END:	STI
		RET
;
; Nonstandard Micro Mainframe function A:  Get Page Owner Table.
;
MMFA:		CMP	COMPATIBILITY,0
		JE	>L0
		MOV	AH,84h		; function invalid in compatibility 
		RET			;   mode
L0:		MOV	SI,OFFSET TRANSLATEARRAY
		MOV	CX,512
		CLD
		REP	MOVSW
		MOV	AH,0
		RET
;
; Nonstandard Micro Mainframe function B:  Get Context Save Area Contents.
;
MMFB:		CMP	COMPATIBILITY,0
		JE	>L0
		MOV	AH,84h		; function invalid in compatibility
		RET			;   mode
L0:		CLD
		MOV	AX,NUMBOARDS
		STOSB
		MOV	SI,OFFSET CONTEXTSAVE
		MOV	CX,190
		REP	MOVSW
		MOV	AH,0
		RET
;
; Nonstandard Micro Mainframe function C:  Get I/O Port and Value Array.
; Called with a handle in DX, a physical page number in AL, and the address
; of a buffer in ES:DI.  Returns an array of 3-byte quantities, one for each 
; logical page associated with the handle.  The entries in the array consist
; of a word I/O port number (page mapping register address) followed by a
; data byte (the value to write to the register).  The data byte and I/O 
; port are those that would be used to map the associated logical page in at
; the specified physical page.  This information can be used to determine the
; hardware page number (i.e., location on the EMS card) of the logical page.
;
MMFC:		CMP	COMPATIBILITY,0
		JE	>L0
		MOV	AH,84h		; function invalid in compatibility
		RET			;   mode
L0:		CMP	AL,4
		JB	>L1
		MOV	AH,8Bh		; invalid physical page number
		RET
L1:		CALL	VERIFYHNDL
		JNC	>L2
		MOV	AH,83h		; invalid or inactive handle
		RET
L2:		CALL	COUNTPAGES
		MOV	CX,BX
		JCXZ	>L4
		CLD
		ROR	AL,1		; AH = xx00_0010, xx = physical page
		ROR	AL,1
		INC	AL
		INC	AL
		MOV	AH,AL
		XOR	BX,BX		; BX = logical page
L3:		CALL	LOG2HARD
		PUSH	CX		; AL = 0000_01yy0, yy = card number
		MOV	CX,SI
		SHL	CX,1
		MOV	AL,CH
		SHL	AL,1
		ADD	AL,8
		STOSW			; save port number (AX)
		SHR	CL,1		; AL = 1zzz_zzzz, zzzzzzz = hardware
		MOV	AL,CL		;   page on card
		OR	AL,80h
		STOSB			; save data byte (AL)
		INC	BX
		POP	CX
		LOOP	L3
L4:		MOV	AH,0
		RET
