// ==============================================================
//
//  Copyright (c) 2002-2003 by Alex Vinokur.
//
//  For conditions of distribution and use, see
//  copyright notice in version.h
//
// ==============================================================


// ##############################################################
//
//  SOFTWARE : C/C++ Program Perfometer
//  FILE     : pfmeter.cpp
//
//  DESCRIPTION :
//         Implementation of perfometer's common (non-template) classes
//
// ##############################################################



// ===================
#include "pfmeter2.h"
// ===================

// -----------------------------------------
unsigned long meas_counter_exn = 0;
unsigned long curr_meas_exn = 0;
// -----------------------------------------

// -----------------------------------------
// -
// -  Constants for IO Manipulation	
// -
// -----------------------------------------
static const string	space1	(1, ' ');
static const string	space2	(2, ' ');
static const string	space3	(3, ' ');
static const string	space4	(4, ' ');
static const string	space5	(5, ' ');
static const string	space9	(9, ' ');

static const string	slash2	(2, '/');

static const string	colon1 (1, ':');

static const string	sulamit1 (1, '#');

static const string	pref1	("# ");

static const size_t	size12	= 12;
static const size_t	size25	= 25;
static const size_t	size27	= 27;
static const size_t	size38	= 38;
static const size_t	size40	= 40;
static const size_t	size43	= 43;
static const size_t	size45	= 45;
static const size_t	setw16	= 16;

// #########################################
// #
// #  Global (Non-Template) Functions 
// #  	IMPLEMENTATION
// #
// #########################################
// =============================
void MeasureFunc (TypeMeasuredFunc * const func_i)
{
  mng.add_measured_func (func_i);
}


// =============================
void Compare_Func(
		const string&	group_name_i, 
		const string&	file_name_i, 
		size_t		line_no_i, 
		TypeMeasuredFunc* const	func_i
		)
{
  mng.add_compared_func (
		GroupDetailedName (
			group_name_i, 
			file_name_i, 
			line_no_i
			),
		func_i
		);
}

// =============================
void SetTotalTests(ulong tests_i)
{
  mng.set_tests (tests_i);
}


// =============================
void SetDiscardingThreshold(float discarding_threshold_i)
{
  mng.set_discarding_threshold (discarding_threshold_i);
}


// =============================
void SetScaleAndTotalIterations (ulong per_calls_i, ulong iterations_i)
{
  mng.set_per_calls (per_calls_i);	// Must be before set_iterations()
  mng.set_iterations (iterations_i);
}

// =============================
void SetDetailedMeasurementReportFlag(bool flag_i)
{
  mng.set_detailed_measurement_report_flag(flag_i);
}

// =============================
void SetMeasurementReportFlag(bool flag_i)
{
  mng.set_measurement_report_flag(flag_i);
}

// =============================
void SetTracingReportFlag(bool flag_i)
{
  mng.set_tracing_report_flag (flag_i);
}

// =============================
bool ParseCommandLine(int argc, char** argv)
{
  return mng.parse_command_line(argc, argv); 
}

     
// =============================
bool RunDemo()
{
  return mng.get_demo_flag(); 
}

     
// =============================
void Measure()
{
  RunDemo() ? MeasureDemo() : MeasureIt();
}

     
// =============================
void Compare()
{
  RunDemo() ? CompareDemo() : CompareIt();
}



// #########################################
// #
// #  AuxClass-0 : ClassResultsFunc
// #  	METHODS
// #
// #########################################
// =============================
bool operator< (
		const GroupDetailedName& ins1, 
		const GroupDetailedName& ins2
		)
{
  // DON'T CHANGE
  return (ins1.group_name_ < ins2.group_name_);

  // -------------------------------------------------------
  // IMPORTANT. file_lines__ must not take part in operator<
  // -------------------------------------------------------


}


// =============================
// Constructor-1
GroupDetailedName::GroupDetailedName(
		const string&	group_name_i,
		const string&	file_name_i,
		size_t		line_no_i
		)
		:
		group_name_ (group_name_i)
		// file_name_ (file_name_i),
		// line_no_ (line_no_i)
{
  if (file_lines_.count(file_name_i) == 0)
  {
    file_lines_[file_name_i] = vector<size_t> ();
  }
  assert (file_lines_.count(file_name_i) == 1);

  file_lines_[file_name_i].push_back (line_no_i);

  assert (file_lines_[file_name_i].size() == 1);

}


// =============================
// Constructor-2
GroupDetailedName::GroupDetailedName(
		const string&	group_name_i,
		const map<string, vector<size_t> >&	file_lines_i
		)
		:
		group_name_ (group_name_i),
		file_lines_ (file_lines_i)
{
}
  


// =============================
// Destructor
GroupDetailedName::~GroupDetailedName()
{
}

// =============================
bool GroupDetailedName::empty () const
{
  return group_name_.empty();
}


// =============================
string GroupDetailedName::get_group_name () const
{
  return group_name_;
}


// =============================
map<string, vector<size_t> > GroupDetailedName::get_file_lines () const
{
  return file_lines_;
}


// =============================
string GroupDetailedName::get_group_detailed_name (size_t setw_size_i) const
{
ostringstream oss;
  oss << ""
      << setw (setw_size_i)
      << left
      << group_name_.c_str()
      << "  [";

map<string, vector<size_t> >::const_iterator iter;
size_t id1;
  for (iter = file_lines_.begin(), id1 = 0;
       iter != file_lines_.end();
       iter++, id1++
      )
  {
    if (id1 > 0) oss << "; ";
    oss << iter->first;
    oss << " : ";
    
    for (size_t id2 = 0; id2 < iter->second.size(); id2++)
    {
      if (id2 > 0) oss << ", ";
      oss << iter->second[id2];
    }
  }
  oss << "]";

  return oss.str();   
}


// #########################################
// #
// #  Class-1 : ClassResultsFunc
// #  	METHODS
// #
// #########################################

// -----------------------------
size_t		ClassResultsFunc::max_prettyfuncname_len_s(0);
size_t		ClassResultsFunc::max_funcname_len_s(0);
size_t		ClassResultsFunc::max_filename_len_s(0);
size_t		ClassResultsFunc::max_descrmsg_len_s(0);
// -----------------------------

// =============================
bool operator< (
		const ClassResultsFunc&	ins1, 
		const ClassResultsFunc& ins2
		)
// =============================
{
  if (ins1.prettyfuncname_ < ins2.prettyfuncname_) return true;
  if (ins1.prettyfuncname_ > ins2.prettyfuncname_) return false;

  if (ins1.funcname_ < ins2.funcname_) return true;
  if (ins1.funcname_ > ins2.funcname_) return false;

  if (ins1.filename_ < ins2.filename_) return true;
  if (ins1.filename_ > ins2.filename_) return false;

  if (ins1.lineno_ < ins2.lineno_) return true;
  if (ins1.lineno_ > ins2.lineno_) return false;

  if (ins1.descrmsg_ < ins2.descrmsg_) return true;
  if (ins1.descrmsg_ > ins2.descrmsg_) return false;
 
  // ----------------------------------------------------
  // IMPORTANT. funcptr_ must not take part in operator<
  // ----------------------------------------------------


  return false;

} // operator<


// =============================
bool operator== (
		const ClassResultsFunc&	ins1, 
		const ClassResultsFunc& ins2
		)
// =============================
{
  if (!(ins1.prettyfuncname_ == ins2.prettyfuncname_))	return false;
  if (!(ins1.funcname_ == ins2.funcname_))		return false;
  if (!(ins1.filename_ == ins2.filename_))		return false;
  if (!(ins1.lineno_ == ins2.lineno_))			return false;
  if (!(ins1.descrmsg_ == ins2.descrmsg_))		return false;

  // -----------------------------------------------------
  // IMPORTANT. funcptr_ must not take part in operator==
  // -----------------------------------------------------

  return true;

} // operator==


// =============================
// Constructor-0
ClassResultsFunc::ClassResultsFunc() :
      		funcptr_ (NULL),		
      		prettyfuncname_ (""),		
      		funcname_ (""),		
      		filename_ (""),		
      		lineno_ (0),		
      		descrmsg_ ("")		
// =============================
{
  assert (0);
} // ClassResultsFunc


// =============================
// Constructor-1
ClassResultsFunc::ClassResultsFunc(
      	TypeMeasuredFunc*	const	funcptr_i,		
	const string&			prettyfuncname_i,
	const string&			funcname_i,
	const string&			filename_i,
	size_t				lineno_i,
	const string&			descrmsg_i
	) :
      		funcptr_ (funcptr_i),		
      		prettyfuncname_ (prettyfuncname_i),		
      		funcname_ (funcname_i),		
      		filename_ (filename_i),		
      		lineno_ (lineno_i),		
      		descrmsg_ (descrmsg_i)		
// =============================
{

  max_prettyfuncname_len_s = MAX_VALUE (
				max_prettyfuncname_len_s, 
				prettyfuncname_.size()
				);

  max_funcname_len_s = MAX_VALUE (
				max_funcname_len_s, 
				funcname_.size()
				);

  max_filename_len_s = MAX_VALUE (
				max_filename_len_s, 
				filename_.size()
				);
  max_descrmsg_len_s = MAX_VALUE (
				max_descrmsg_len_s, 
				descrmsg_.size()
				);

} // ClassResultsFunc



// =============================
// Destructor
ClassResultsFunc::~ClassResultsFunc()
// =============================
{
} // ~ClassResultsFunc


// =============================
string ClassResultsFunc::get_prettyfuncname () const
// =============================
{
  return prettyfuncname_;
}


// =============================
string ClassResultsFunc::get_funcname () const
// =============================
{
  return funcname_;
}

// =============================
string ClassResultsFunc::get_filename () const
// =============================
{
  return filename_;
}


// =============================
size_t ClassResultsFunc::get_lineno () const
// =============================
{
  return lineno_;
}


// =============================
string ClassResultsFunc::get_descr () const
// =============================
{
  return descrmsg_;
}

// =============================
TypeMeasuredFunc* ClassResultsFunc::get_funcptr () const
// =============================
{
  return funcptr_;
}

// =============================
string ClassResultsFunc::getstr_func_detailed_descr (
		vector<string>	resource_descr_i,
		count_t		func_serial_no_i, 
		count_t		total_funcs_i, 
		count_t		total_tests_i, 
		bool		detailed_report_flag_i, 
		const string&	msg_i
		) const

// =============================
{
ostringstream osstr;

  osstr	<< endl;

const string word_Resource ("Resource");

#ifdef OPTIMIZATION_LEVEL
  osstr << space3 << OPTIMIZATION_LEVEL << endl;
#endif

  osstr	<< space3
	<< string (size45, '#') 
	<< endl;

  osstr	<< space3
        << pref1
        << string (word_Resource.size(), '-')
	<< endl;

  osstr	<< space3
        << pref1
        << word_Resource
	<< endl;

  osstr	<< space3
        << pref1
        << string (word_Resource.size(), '-')
	<< endl;

  for (count_t i = 0; i < resource_descr_i.size(); i++)
  {
    osstr	<< space3
    		<< pref1
    		<< "\t"
    		<< resource_descr_i[i]
    		<< endl;
 }

  osstr	<< space3
	<< pref1
	<< string (size43, '=') 
	<< endl;

  if (!msg_i.empty())
  {
    osstr << space3 << msg_i << " ---> "; 
  }


  osstr	<< space3
	<< pref1 
	<< setw (setw16) 
	<< string ("Function-" + to_string (func_serial_no_i)).c_str()
	<< " : " 
	<< "of "
        << total_funcs_i
	<< endl;

  osstr	<< space3
	<< pref1 
	<< setw (setw16) 
	<< "PRETTY FUNCTION" 
	<< " : " 
	<< prettyfuncname_ 
	<< endl;

  osstr	<< space3
	<< pref1 
	<< setw (setw16) 
	<< "FUNCTION" 
	<< " : " 
	<< funcname_ 
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< "FILE" 
	<< " : " 
	<< filename_ 
	<< ", line#" 
	<< lineno_ 
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< "DESCRIPTION" 
	<< " : " 
	<< descrmsg_ 
	<< endl;


  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< "PER-CALLS" 
	<< " : " 
	<< mng.get_scale_per_calls() 
	<< endl;


  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< "TOTAL TESTS" 
	<< " : " 
	<< total_tests_i 
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< string (9, '-').c_str() 
	<< string (3, '-') 
	<< string (9, '-') 
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< "Specific Performance is Runcost expended per " 
	<< mng.get_scale_per_calls() 
	<< " iterations"
	<< endl;

  osstr	<< space3 
	<< pref1 
	<< setw (setw16) 
	<< string (9, '-').c_str() 
	<< string (3, '-') 
	<< string (9, '-') 
	<< endl;

  osstr	<< space3
	<< pref1
	<< space5;

  if (detailed_report_flag_i)
  { 
    osstr << "Detailed measurement report";
  }
  else
  {
    osstr << "Brief measurement report";
  }
  osstr	<< endl;

  osstr	<< space3
	<< string (size45, '#') 
	<< endl;

 return osstr.str();   

} // ClassResultsFunc::getstr_func_detailed_descr



// =============================
size_t ClassResultsFunc::get_max_prettyfuncname_len_S () 
// =============================
{
  return max_prettyfuncname_len_s;
}


// =============================
size_t ClassResultsFunc::get_max_funcname_len_S () 
// =============================
{
  return max_funcname_len_s;
}


// =============================
size_t ClassResultsFunc::get_max_filename_len_S () 
// =============================
{
  return max_filename_len_s;
}



// =============================
size_t ClassResultsFunc::get_max_lineno_len_S () 
// =============================
{
  return 3;
}

// =============================
size_t ClassResultsFunc::get_max_descrmsg_len_S () 
// =============================
{
  return max_descrmsg_len_s;
}



// #########################################
// #
// #  Class-7 : BasicClassMeasurementCollection
// #  	METHODS
// #
// #########################################
// -----------------------------
vector<BasicClassMeasurementCollection*>	BasicClassMeasurementCollection::all_measurement_collections_s;
// -----------------------------

// =============================
// Constructor-0
BasicClassMeasurementCollection::BasicClassMeasurementCollection () 
			:
			typeid_name_(string()),
			total_measured_funcs_(0),
    			resource_description_ (vector<string>())
// =============================
{
}

// =============================
// Copy Constructor
BasicClassMeasurementCollection::BasicClassMeasurementCollection (const BasicClassMeasurementCollection&) 
// =============================
{
  assert (0);
}

// =============================
// Destructor
BasicClassMeasurementCollection::~BasicClassMeasurementCollection () 
// =============================
{
}



// =============================
void BasicClassMeasurementCollection::add_resource_type_if_need () 
// =============================
{

  if (find_if(
		all_measurement_collections_s.begin(), 
		all_measurement_collections_s.end(), 
		bind1st(
			mem_fun(&BasicClassMeasurementCollection::typeid_predicat), 
			this
			)
		)
      ==
      all_measurement_collections_s.end()
     )
  {
    all_measurement_collections_s.push_back(this);
  }
  
  
  assert (count_if(
		all_measurement_collections_s.begin(), 
		all_measurement_collections_s.end(), 
		bind1st(
			mem_fun(&BasicClassMeasurementCollection::typeid_predicat), 
			this
			)
		)
          == 1
         );

} // BasicClassMeasurementCollection::add_resource_type_if_need



// =============================
bool BasicClassMeasurementCollection::typeid_predicat (
		const BasicClassMeasurementCollection* const p_inst_i
		) const 
// =============================
{
  assert (p_inst_i);  
  return (typeid_name_ == p_inst_i->typeid_name_);
}



// =============================
size_t BasicClassMeasurementCollection::get_total_resource_types_s ()
// =============================
{
  return all_measurement_collections_s.size();
}


// =============================
void BasicClassMeasurementCollection::show_list_of_resources_s ()
// =============================
{

  // -----------------------------------------------
  if (all_measurement_collections_s.empty()) return;
  // -----------------------------------------------

const char fill0 = '%';
const char fill1 = '=';
ostringstream osstr;
  

  sout << endl;
  sout << endl;
  sout << endl;
  sout << "\t" << string (42, fill1) << endl;
  sout << "\t" << string (2, fill0) << string (40, fill1) << endl;
  sout << "\t" << string (2, fill0) << endl;

  sout << "\t" << string (2, fill0) << space2 
       << "Total Measured Resources : " 
       << all_measurement_collections_s.size() 
       << endl;

  for (count_t i = 0; i < all_measurement_collections_s.size(); i++)
  {

    sout << "\t" 
         << string (2, fill0) 
         << space1 
         << endl; 

    sout << "\t" 
         << string (2, fill0) 
         << space2 
         << string (10, '-')
         << space1 
         << "Resource#"
         << (i + 1)
         << space1 
         << string (10, '-')
         << endl; 

    for (
	count_t j = 0; 
	        j < all_measurement_collections_s[i]->resource_description_.size(); 
	        j++
	        )
    {
      sout << "\t" 
           << string (2, fill0) 
           << space1; 
      sout << all_measurement_collections_s[i]->resource_description_[j];
      sout << endl;

    }

    sout << "\t" 
         << string (2, fill0) 
         << space1
         << "---> Total measured functions : "
         << all_measurement_collections_s[i]->total_measured_funcs_
         << endl; 
  }
  sout << "\t" << string (2, fill0) << "\t" << string(osstr.str().size(), '-') << endl;

  sout << "\t" << string (2, fill0) << endl;
  sout << "\t" << string (2, fill0) << string (40, fill1) << endl;
  sout << "\t" << string (42, fill1) << endl;
  sout << endl;


} // BasicClassMeasurementCollection::show_list_of_resources_s



// #########################################
// #
// #  Class-9 : ClassManager	
// #  	METHODS
// #
// #########################################

// -----------------------------
ostringstream		sout;
ClassManager		mng;
count_t			ClassManager::inst_counter_s(0);
TypeMeasuredFunc*	ClassManager::cur_func_ptr_s(NULL);

// -----------------------------

// =============================
// Constructor-0
ClassManager::ClassManager () 
		: 
		performing_flag_ (true), 
		demo_flag_ (false)
// =============================
{

  // ---------------------
  // show_logo ("");
  // ---------------------

  inst_counter_s++;
  assert (inst_counter_s == 1);
  // -----------------------
  set_measurement_report_flag ();
  set_detailed_measurement_report_flag ();
  set_tracing_report_flag ();

  set_discarding_threshold (); 
  set_tests (); 
  set_per_calls (); 	// Must be before  set_iterations()
  set_iterations (); 
  
  assert (scale_per_calls_ > 0);
  assert (total_iterations_ > 0);
  assert (scale_per_calls_ <= total_iterations_);

  assert (total_tests_ > 0);


} // ClassManager::ClassManager


// =============================
// Copy Constructor
ClassManager::ClassManager (const ClassManager&) 
// =============================
{
  assert (0);
} // ClassManager::ClassManager

// =============================
// Destructor
ClassManager::~ClassManager () 
// =============================
{
  // -----------
  if (performing_flag_)
  {
    perform_measurement ();
    report_measurement (); 
  }
  else
  {
    show_comparison_group ();
    // MAYBE TBD : show_testcases ();
  }
  delete_it (); 

const string logfile_info ("Log file  " + (log_filename_.empty() ? string("has not been created") : log_filename_));
  show_logo ("FINISH : " + logfile_info);

  close_log ();

ostringstream oss;
  
  oss << logfile_info;

  cout << endl;
  cout << "\t" << string (oss.str().size(), '-') << endl;
  cout << "\t" << oss.str() << endl;
  cout << "\t" << string (oss.str().size(), '-') << endl;
  cout << endl;
  
}


// =============================
void ClassManager::open_log ()
// =============================
{
  assert (fout_);

  fout_.open (log_filename_.c_str());

  if ((!fout_) || (fout_ && !fout_.is_open()))
  {
    cerr << endl;
    cerr << "\t--- WARNING! Unable to open log-file " << log_filename_ << " ---" << endl;
    cerr << endl;
 
    log_filename_.clear();

    return;
  }
  assert (fout_);
  assert (fout_.is_open());

}


// =============================
void ClassManager::close_log ()
// =============================
{

  if (fout_ && fout_.is_open()) 
  {
    fout_ << sout.str();

    assert (fout_);
    assert (fout_.is_open());

    fout_.close();

  }
  else
  {
    cout << sout.str();
  }


  assert (!fout_ || (fout_ && !fout_.is_open()));

}

// =============================
void ClassManager::flush_log ()
// =============================
{

  assert (!(fout_ && fout_.is_open()));
  fout_ << sout.str();
  sout.str("");
}


// =============================
void ClassManager::set_log_filename (const string& log_filename_i)
// =============================
{
  log_filename_ = log_filename_i;  
}


// =============================
void ClassManager::perform_measurement () const
// =============================
{

  for (count_t i = 0; i < total_tests_; i++)
  {
    curr_meas_exn = i + 1;
    perform_a_measurement (); 
  }

} // perform_measurement

// =============================
void ClassManager::perform_a_measurement () const
// =============================
{

  /*
  for_each(
	measured_funcs_.begin(), 
	measured_funcs_.end(), 
	self_caller<TypeMeasuredFunc*>
	);
  */


  cur_func_ptr_s = NULL;
  for (count_t cur_func_no = 0; 
	cur_func_no < measured_funcs_.size(); 
	cur_func_no++
	)
  {
    cur_func_ptr_s = measured_funcs_[cur_func_no];
    assert (cur_func_ptr_s);

    // -----------------------------
    measured_funcs_[cur_func_no] ();
    // -----------------------------

    cur_func_ptr_s = NULL;
  }


} // perform_a_measurement


// =============================
void ClassManager::report_measurement () const
// =============================
{

vector<BasicClassMeasurementCollection*> &all_measurement_collections = 
		BasicClassMeasurementCollection::all_measurement_collections_s;

map<GroupDetailedName, vector<TypeMeasuredFunc*>, less<string> >::const_iterator	map_iter;

  BasicClassMeasurementCollection::show_list_of_resources_s ();

  for (count_t i = 0;
               i < all_measurement_collections.size();
               i++
      )
  {
   
    if (mng.get_measurement_report_flag ())
    {
      // Measurement Report
      all_measurement_collections[i]->show_resource_detailed_descr(i + 1, "Performance Measurement", false);  
      all_measurement_collections[i]->show_measurement_collection();
    }


    // -----------------------
    // Comparison Report
    all_measurement_collections[i]->show_resource_detailed_descr(i + 1, "Performance Comparison", true);  

    count_t group_no = 0;
    for (
	  map_iter = comparison_groups_.begin();
	  map_iter != comparison_groups_.end();
	  map_iter++
	  )
    {
      group_no++;
      ostringstream osstr1;
      ostringstream osstr2;
      osstr1 << "Comparison Group#"
             << group_no
             << " of "
             << comparison_groups_.size()
             << " : "
             << map_iter->first.get_group_name();

      for (
	   count_t j = 0; 
	           j < all_measurement_collections[i]->resource_description_.size(); 
	           j++
	           )
      {
        osstr2 << "\t\t" 
               << "#" 
               << string (2, ' ') 
               << space1; 
        osstr2 << all_measurement_collections[i]->resource_description_[j];
        osstr2 << endl;
      }

      // -----------------------
      sout << endl;
      sout << endl;

#ifdef OPTIMIZATION_LEVEL
  sout << "\t\t" << OPTIMIZATION_LEVEL << endl;
#endif

      sout << "\t\t" 
	   << "#" 
	   << string (osstr1.str().size() + 1, '=') 
	   << endl;

      sout << "\t\t" 
	   << "#" 
	   << " " 
	   << osstr1.str() 
	   << endl;

      sout << "\t\t" 
	   << "#" 
           << string (osstr1.str().size() + 1, '-')
	   << endl;

      sout << osstr2.str(); 

      sout << "\t\t" 
	   << "#" 
	   << string (osstr1.str().size() + 1, '=') 
	   << endl;
      
      osstr1.str(string());
    
      osstr1 << string (3, ' ') 
             << ": " 
             << string (
		ClassResultsFunc::get_max_funcname_len_S() + 3
		+ 
		ClassResultsFunc::get_max_filename_len_S() + 3
		+ 
		ClassResultsFunc::get_max_lineno_len_S() + 3
		+ 
		ClassResultsFunc::get_max_descrmsg_len_S() + 4
		+
		12
		, '-'); 

      sout << osstr1.str() << endl;
      all_measurement_collections[i]->show_comparison_collection(map_iter->second);
      sout << osstr1.str() << endl;
    }
    sout << endl;
    sout << endl;

  }  // for (count_t i = 0; ...

} // ClassManager::report_measurement


// =============================
void ClassManager::add_measured_func (TypeMeasuredFunc * const func_i)
// =============================
{
  assert (func_i);
  if (find (
	     measured_funcs_.begin(), 
	     measured_funcs_.end(),
	     func_i
	     )
	     == measured_funcs_.end()
     )
  {
    measured_funcs_.push_back(func_i);
  }
} 


// =============================
void ClassManager::add_comparison_group (
			const GroupDetailedName& group_detailed_name_i
			)
// =============================
{
  if (group_detailed_name_i.empty())	
  {
    sout << "Unable to add comparison group : group name is empty" << endl;
    return;
  }

  assert (!group_detailed_name_i.empty());
  assert (
	  (comparison_groups_.count(group_detailed_name_i) == 0)
	  ||
	  (comparison_groups_.count(group_detailed_name_i) == 1)
	 );


  if (comparison_groups_.count(group_detailed_name_i) == 0)
  {
    comparison_groups_ [group_detailed_name_i] = vector<TypeMeasuredFunc*> ();
  }
  else
  {
    GroupDetailedName tmp_group_detailed_name (group_detailed_name_i); // Default Copy Constructor

    map<GroupDetailedName, vector<TypeMeasuredFunc*>, less<GroupDetailedName> >::iterator find_iter;
    find_iter = comparison_groups_.find (group_detailed_name_i);
    assert (find_iter != comparison_groups_.end());

    vector<TypeMeasuredFunc*> tmp_vect (find_iter->second);

    map<string, vector<size_t> > map_to_be_updated (find_iter->first.get_file_lines());
    map<string, vector<size_t> > new_map (group_detailed_name_i.get_file_lines());
    assert (new_map.size() == 1);

    assert (new_map[new_map.begin()->first].size() == 1);
    const string group_new_file_name (new_map.begin()->first);
    const size_t group_new_line_no (new_map.begin()->second.front());

    assert (map_to_be_updated.size() > 0);

    map<string, vector<size_t> >::const_iterator find2_iter;

    find2_iter = map_to_be_updated.find(group_new_file_name);

    if (find2_iter == map_to_be_updated.end())
    {
      map_to_be_updated[group_new_file_name] = vector<size_t> ();
    }
    map_to_be_updated[group_new_file_name].push_back (group_new_line_no);

    assert (!comparison_groups_.empty());
    comparison_groups_.erase(find_iter);

    comparison_groups_[GroupDetailedName (
				group_detailed_name_i.get_group_name(), 
				map_to_be_updated
				)
		      ] = tmp_vect; 
  }
  assert (comparison_groups_.count(group_detailed_name_i) == 1);



} 

// =============================
void ClassManager::add_compared_func (
			const GroupDetailedName& group_detailed_name_i,
			TypeMeasuredFunc*	 const func_i
			)
// =============================
{
  // if (comparison_groups_.count(group_detailed_name_i) == 0)
  // {
  // -----------------------------------------
  add_comparison_group (group_detailed_name_i);
  // -----------------------------------------
  // }

  assert (!comparison_groups_.empty());
  assert (comparison_groups_.count(group_detailed_name_i) == 1);
  assert (func_i);


  comparison_groups_[group_detailed_name_i].push_back(func_i);

  if (find (
	     comparison_groups_[group_detailed_name_i].begin(), 
	     comparison_groups_[group_detailed_name_i].end(),
	     func_i
	     )
	     == comparison_groups_[group_detailed_name_i].end()
     )
  {
    comparison_groups_[group_detailed_name_i].push_back(func_i);
  }

  add_measured_func (func_i);
} 


// =============================
void ClassManager::show_comparison_group () const
// =============================
{
map<GroupDetailedName, vector<TypeMeasuredFunc*>, less<string> >::const_iterator iter;
const string text_all_groups ("Comparison groups");
const string text_testsuites ("Testsuites-" + to_string (sw_testsuites_version));

ostringstream oss;
  
  oss << endl
       << "\t " << text_testsuites
       << endl
       << "\t " << string (text_testsuites.size(), '=')
       << endl;

  oss << ""
       << "\t" << text_all_groups
       << endl
       << "\t" << string (text_all_groups.size(), '-')
       << endl;

size_t setw_size = 0;
  for (iter = comparison_groups_.begin();
       iter != comparison_groups_.end();
       iter++
       )
  {
    setw_size = MAX_VALUE (setw_size, iter->first.get_group_name().size());
  }

  for (iter = comparison_groups_.begin();
       iter != comparison_groups_.end();
       iter++
       )
  {
    oss << "\t* " << iter->first.get_group_detailed_name(setw_size) << endl;
  }
  oss << endl;
  oss << endl;

  // ----------
  sout << oss.str();
  cerr << oss.str();

}

// =============================
void ClassManager::show_testcases () const
// =============================
{
// TO BE CHANGED
vector<BasicClassMeasurementCollection*> &all_measurement_collections = 
		BasicClassMeasurementCollection::all_measurement_collections_s;

map<GroupDetailedName, vector<TypeMeasuredFunc*>, less<string> >::const_iterator	map_iter;

  BasicClassMeasurementCollection::show_list_of_resources_s ();

  for (count_t i = 0;
               i < all_measurement_collections.size();
               i++
      )
  {
   

    // -----------------------
    // Comparison Report
    all_measurement_collections[i]->show_resource_detailed_descr(i + 1, "Performance Comparison", true);  

    count_t group_no = 0;
    for (
	  map_iter = comparison_groups_.begin();
	  map_iter != comparison_groups_.end();
	  map_iter++
	  )
    {
      group_no++;
      ostringstream osstr1;
      ostringstream osstr2;
      osstr1 << "Comparison Group#"
             << group_no
             << " of "
             << comparison_groups_.size()
             << " : "
             << map_iter->first.get_group_name();

      for (
	   count_t j = 0; 
	           j < all_measurement_collections[i]->resource_description_.size(); 
	           j++
	           )
      {
        osstr2 << "\t\t" 
               << "#" 
               << string (2, ' ') 
               << space1; 
        osstr2 << all_measurement_collections[i]->resource_description_[j];
        osstr2 << endl;
      }

      // -----------------------
      sout << endl;
      sout << endl;

#ifdef OPTIMIZATION_LEVEL
  sout << "\t\t" << OPTIMIZATION_LEVEL << endl;
#endif

      sout << "\t\t" 
	   << "#" 
	   << string (osstr1.str().size() + 1, '=') 
	   << endl;

      sout << "\t\t" 
	   << "#" 
	   << " " 
	   << osstr1.str() 
	   << endl;

      sout << "\t\t" 
	   << "#" 
           << string (osstr1.str().size() + 1, '-')
	   << endl;

      sout << osstr2.str(); 

      sout << "\t\t" 
	   << "#" 
	   << string (osstr1.str().size() + 1, '=') 
	   << endl;
      
      osstr1.str(string());
    
      osstr1 << string (3, ' ') 
             << ": " 
             << string (
		ClassResultsFunc::get_max_funcname_len_S() + 3
		+ 
		ClassResultsFunc::get_max_filename_len_S() + 3
		+ 
		ClassResultsFunc::get_max_lineno_len_S() + 3
		+ 
		ClassResultsFunc::get_max_descrmsg_len_S() + 4
		+
		12
		, '-'); 

      sout << osstr1.str() << endl;
      all_measurement_collections[i]->show_comparison_collection(map_iter->second);
      sout << osstr1.str() << endl;
    }
    sout << endl;
    sout << endl;

  }  // for (count_t i = 0; ...

}



// =============================
void ClassManager::set_tests (ulong tests_i) 
// =============================
{
  total_tests_ = tests_i;
}


// =============================
void ClassManager::set_discarding_threshold (float discarding_threshold_i) 
// =============================
{
  discarding_threshold_ = discarding_threshold_i;
}

// =============================
void ClassManager::set_detailed_measurement_report_flag (bool type_i) 
// =============================
{
  detailed_measurement_report_flag_ = type_i;
}


// =============================
void ClassManager::set_measurement_report_flag (bool type_i) 
// =============================
{
  measurement_report_flag_ = type_i;
} 

// =============================
void ClassManager::set_iterations (ulong iterations_i) 
// =============================
{
  total_iterations_ = iterations_i;
  if (total_iterations_ == 0)
  {
    total_iterations_ = TOTAL_ITERATIONS_PER_TEST_default;
  }

  while (scale_per_calls_ > total_iterations_)
  {
    total_iterations_ *= 10;
  }

  assert (scale_per_calls_ <= total_iterations_);


  total_iterations_ = (total_iterations_/scale_per_calls_)*scale_per_calls_;

  if (iterations_i != total_iterations_)
  {
     string pref0 = "\t ";
     string pref1 = pref0 + ": ";
     size_t the_size = 52;
     ostringstream oss;
     oss << pref0 << string (the_size, '-') << endl;
     oss << pref1 << "\t--- WARNING! ---" << endl;
     oss << pref1 << "Requested number-of-iterations   = " << iterations_i << endl;
     oss << pref1 << "---> Actual number-of-iterations = " << total_iterations_ << endl;
     oss << pref1 << "   Info! scale-per-calls         = " << scale_per_calls_ << endl;
     oss << pref1 << "   Note! Must be : scale-per-calls <= number-of-iterations"<< endl;
     oss << pref1 << "                 : total_iterations_%scale_per_calls_ == 0"<< endl;
     oss << pref0 << string (the_size, '-') << endl;
     notes_.push_back (oss.str());
  }

} // ClassManager::set_iterations



// =============================
void ClassManager::set_per_calls (ulong per_calls_i) 
// =============================
{
  scale_per_calls_ = per_calls_i;
  if (scale_per_calls_ == 0)
  {
    scale_per_calls_ = PER_CALLS_default;
  }
    
} // ClassManager::set_per_calls



// =============================
ulong ClassManager::get_scale_per_calls () const
// =============================
{
  return scale_per_calls_;
} 


// =============================
ulong ClassManager::get_total_tests () const
// =============================
{
  return total_tests_;
} 

// =============================
float ClassManager::get_discarding_threshold ()	const
// =============================
{
  return discarding_threshold_;
}


// =============================
float ClassManager::get_discarding_threshold_S ()
// =============================
{
  assert (inst_counter_s == 1);
  return mng.get_discarding_threshold();
}


// =============================
bool ClassManager::get_tracing_report_flag_S ()
// =============================
{
  assert (inst_counter_s == 1);
  return mng.get_tracing_report_flag();
}


// =============================
bool ClassManager::get_demo_flag_S ()
// =============================
{
  assert (inst_counter_s == 1);
  return mng.get_demo_flag();
}



// =============================
ulong ClassManager::get_current_total_iterations () const
// =============================
{
  return total_iterations_;
} 

// =============================
bool ClassManager::get_detailed_measurement_report_flag () const
// =============================
{
  return detailed_measurement_report_flag_;
} 


// =============================
bool ClassManager::get_measurement_report_flag () const
// =============================
{
  return measurement_report_flag_;
} 

// =============================
TypeMeasuredFunc* ClassManager::get_cur_func_ptr () const
// =============================
{
  return cur_func_ptr_s;
} 


// -----------------------------------------
// -
// -  Constants for Parsing command line	
// -
// -----------------------------------------
static const char	ch_minus_CNS	= '-';	
static const char	ch_plus_CNS	= '+';	

static const string	log_filename_option_CNS		("-l");	
static const string	getting_groups_option_CNS	("-g");	
static const string	skipping_actual_run_option_CNS	("-k");	
static const string	demo_option_CNS			("-x");	
static const string	help_option_CNS			("-h");	
static const string	settings_option_CNS		("-s");	
static const string	version_option_CNS		("-v");	

static const string	report_option_enabled_CNS	("+");	
static const string	report_option_disabled_CNS	("-");	

static const string	full_measuring_report_option_CNS	("m");	
static const string	detailed_measuring_report_option_CNS	("d");	
static const string	tracing_report_option_CNS		("t");	

static const string	set_iterations_option_CNS		("-i");	
static const string	set_tests_option_CNS			("-e");	
static const string	set_per_calls_option_CNS		("-p");	
static const string	set_confidence_threshold_option_CNS	("-f");	


// =============================
void ClassManager::show_brief_man_description() const
// =============================
{
ostringstream oss;
const string proj_name ("--- C/C++ Program Perfometer ---");

  oss << endl
      << endl
      << space2 << string (proj_name.size(), '=') << endl
      << space2 << proj_name << endl
      << space2 << string (proj_name.size(), '=') << endl
      << endl
      << space2 << "BRIEF DESCRIPTION" << endl
      << endl
      << space4 << string (proj_name.size() - (space4.size() - space2.size()), '-')  << endl
      << space4 << "The program enables to get performance" << endl
      << space4 << "of C/C++ program and separated pieces of code for any metrics" << endl
      << space4 << "(for instance : clocks, uclocks, rusage-metrics, metrics defined by user etc.)." << endl
      << space4 << endl
      << space4 << "The measurement results are represented in detailed/summary reports." << endl
      << space4 << "The detailed report holds results of individual tests," << endl
      << space4 << "the summary report holds average cost and its analysis." << endl
      << space4 << "The comparison results are represented in comparative tables" << endl
      << space4 << "for individual comparison groups." << endl
      << space4 << endl
      << space4 << "The user may set various parameters" << endl
      << space4 << "in order to control the measurement/comparison process :" << endl
      << space4 << "* measurement report and detailed measurement report flags," << endl
      << space4 << "* total tests," << endl
      << space4 << "* total iterations," << endl
      << space4 << "* measurement scale," << endl
      << space4 << "* confidence threshold." << endl
      << endl
      << endl

      << space4 << "User-Defined Program Files." << endl
      << space4 << "* Metrics related program files (adapt.h, adapt.cpp) :" << endl
      << space4 << "  The files contain definition and implementation of specific performance metrics (IF ANY)." << endl
      << space4 << "* Testsuite related program files (tests.h, tests.cpp, t_*.cpp) :" << endl
      << space4 << "  The files contain declaration and implementation of measured/compared functions." << endl
      << endl
      << endl

      << space4 << "Download :" << endl
      << space4 << "- http://sourceforge.net/projects/cpp-perfometer" << endl
      << space4 << "- http://alexvn.freeservers.com/s1/perfometer.html" << endl
      << endl
      << endl

      << space4 << "Some C/C++ Performance Tests performed using the Perfometer :" << endl
      << space4 << "- http://groups.google.com/groups?th=466338d8028d2fc6" << endl
      << space4 << "- http://article.gmane.org/gmane.comp.gcc.g++.general/101" << endl
      << space4 << "- http://mail.gnu.org/archive/html/help-gplusplus/2003-06/msg00003.html" << endl
      << space4 << "- news://news.gmane.org/bd13qj$s86$1@main.gmane.org (via NNTP newsreader)" << endl
      << space4 << "- news://msnnews.msn.com/uI#FPx8NDHA.1908@TK2MSFTNGP11.phx.gbl (via NNTP newsreader)" << endl

      << space2 << string (proj_name.size(), '=') << endl
      << endl
      << endl
      << endl;

  sout << oss.str();
  cerr << oss.str();
}


// =============================
void ClassManager::show_help_usage(const string& exe_name_i) const
// =============================
{
ostringstream osstr;

  osstr << ""
        << "--- "
        << "USAGE : "
        << exe_name_i

        << " ["
        << log_filename_option_CNS
        << " "
        << "<log-file-name>"

        << "]"
        << " ["
        << help_option_CNS
        << "]"

        << " ["
        << version_option_CNS
        << "]"

        << " ["
        << demo_option_CNS
        << "]"

        << " ["
        << getting_groups_option_CNS
        << "]"

        << " ["
        << skipping_actual_run_option_CNS
        << "]"

        << " ["
        << settings_option_CNS
        << "]"

        << " ["
        << (report_option_disabled_CNS + full_measuring_report_option_CNS)
        << "]"

        << " ["
        << (report_option_enabled_CNS + full_measuring_report_option_CNS)
        << "]"

        << " ["
        << (report_option_disabled_CNS + detailed_measuring_report_option_CNS)
        << "]"

        << " ["
        << (report_option_enabled_CNS + detailed_measuring_report_option_CNS)
        << "]"

        << " ["
        << (report_option_disabled_CNS + tracing_report_option_CNS)
        << "]"

        << " ["
        << (report_option_enabled_CNS + tracing_report_option_CNS)
        << "]"

        << " ["
        << set_iterations_option_CNS
        << " "
        << "<number>"
        << "]"

        << " ["
        << set_tests_option_CNS
        << " "
        << "<number>"
        << "]"

        << " ["
        << set_per_calls_option_CNS
        << " "
        << "<number>"
        << "]"

        << " ["
        << set_confidence_threshold_option_CNS
        << " "
        << "<float-number>"
        << "]";

const string help_msg = osstr.str();

ostringstream oss;

  oss  << endl
       << string (help_msg.size(), '-')
       << endl
       << help_msg
       << endl
       << string (help_msg.size(), '-')
       << endl;


  sout << oss.str();
  cerr << oss.str();

} // show_help_usage



// =============================
void ClassManager::show_help_example(const string& exe_name_i) const
// =============================
{
#define	HELP_EXAMPLE_LINES	14
ostringstream	osstr [HELP_EXAMPLE_LINES];
size_t	pos = 0;

const string pref1 (4, ' ');
const string str_examples ("USAGE EXAMPLES");

  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << str_examples;


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << string (str_examples.size(), '^');



  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << help_option_CNS;


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << version_option_CNS;


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << demo_option_CNS;


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << getting_groups_option_CNS;


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << skipping_actual_run_option_CNS;


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << settings_option_CNS;


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i;


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << (report_option_disabled_CNS + full_measuring_report_option_CNS);


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << (report_option_enabled_CNS + detailed_measuring_report_option_CNS);


  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << (report_option_enabled_CNS + tracing_report_option_CNS);


const int set_iterations_example = 75000;
const int set_tests_example = 15;
const int set_per_calls_example = 7500;
const float set_confidence_threshold_example = 0.17;
  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << set_iterations_option_CNS
               << " "
               << set_iterations_example
               << " "
               << set_tests_option_CNS
               << " "
               << set_tests_example
               << " "
               << set_per_calls_option_CNS
               << " "
               << set_per_calls_example
               << " "
               << set_confidence_threshold_option_CNS
               << " "
               << set_confidence_threshold_example;



const string log_filename_example ("my.log");
  assert (pos < HELP_EXAMPLE_LINES);
  osstr[pos++] << ""
               << pref1
               << exe_name_i
               << " "
               << log_filename_option_CNS
               << " "
               << log_filename_example;


size_t wsize = 0;

  for (size_t i = 0; i < pos; i++)
  {
    wsize = MAX_VALUE (wsize, osstr[i].str().size());
  }

ostringstream oss;
  oss  << endl
       << string (wsize, '-')
       << endl;

  for (size_t i = 0; i < pos; i++)
  {
    oss << osstr[i].str() << endl;
  }

  oss << ""
       << string (wsize, '-')
       << endl;


  sout << oss.str();
  cerr << oss.str();

} // show_help_example



// =============================
void ClassManager::show_help_info(const string& exe_name_i) const
// =============================
{
ostringstream	osstr;

const string pref1 ("\t===> ");
const string str_infos1 ("HELP : ");
const string str_infos2 ("DEMO : ");


  osstr << ""
        << pref1
        << str_infos1
        << exe_name_i
        << " "
        << help_option_CNS;

  osstr << endl;

  osstr << ""
        << pref1
        << str_infos2
        << exe_name_i
        << " "
        << demo_option_CNS;


ostringstream oss;

  oss << osstr.str() << endl;

  sout << oss.str();
  cerr << oss.str();

} // show_help_info



// =============================
void ClassManager::show_help_options() const
// =============================
{
ostringstream osstr;
const string shift1 = "   #   ";
const string delim1 = "   : ";
const string text1 = "OPTIONS DESCRIPTION";
const string text2 = " <number>";
const string text3 = " <name>";
const string text_default = "Default";
const string text_run_time_default = "Run-time Defined Default";
const string show_delim_CNS = "/";
const string text_enabled = "ENABLED";
const string text_disabled = "DISABLED";

const size_t line1_len = 70;

  osstr.setf (ios::left, ios::adjustfield);

  osstr << ""
        << shift1
        << space9
        << text1
        << endl
        << shift1
        << space9
        << string (text1.size(), '-')
        << endl

        << shift1
        << setw (size12)
        << help_option_CNS.c_str()
        << delim1
        << " get help information"
        << endl

        << shift1
        << setw (size12)
        << version_option_CNS.c_str()
        << delim1
        << " get version information"
        << endl


        << shift1
        << setw (size12)
        << demo_option_CNS.c_str()
        << delim1
        << " run demo tests"
        << endl

        << shift1
        << setw (size12)
        << settings_option_CNS.c_str()
        << delim1
        << " get environment variables values"
        << endl


        << shift1
        << setw (size12)
        << getting_groups_option_CNS.c_str()
        << delim1
        << " get list of compared groups"
        << endl

        << shift1
        << setw (size12)
        << skipping_actual_run_option_CNS.c_str()
        << delim1
        << " skip an actual run (get list of test cases)"
        << endl


        << shift1
        << setw (size12)
        << (report_option_disabled_CNS + show_delim_CNS + report_option_enabled_CNS + full_measuring_report_option_CNS).c_str()
        << delim1
        << " measurement report "  
        << text_disabled
        << show_delim_CNS
        << text_enabled 
        << " ("
        << text_default
        << " - "
        << text_enabled 
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (report_option_disabled_CNS + show_delim_CNS + report_option_enabled_CNS + detailed_measuring_report_option_CNS).c_str()
        << delim1
        << " detailed measurement report "  
        << text_disabled
        << show_delim_CNS
        << text_enabled 
        << " ("
        << text_default
        << " - "
        << text_disabled 
        << ")"
        << endl

        << shift1
        << setw (size12)
        << ""
        << delim1
        << "   Note! If measurement report " 
        <<  text_disabled 
        << " this option ignored"
        << endl

        << shift1
        << setw (size12)
        << (report_option_disabled_CNS + show_delim_CNS + report_option_enabled_CNS + tracing_report_option_CNS).c_str()
        << delim1
        << " tracing report "  
        << text_disabled
        << show_delim_CNS
        << text_enabled 
        << " ("
        << text_default
        << " - "
        << text_disabled 
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (set_iterations_option_CNS + text2).c_str()
        << delim1
        << " set number of iterations"
        << " ("
        << text_default
        << " = "
        << TOTAL_ITERATIONS_PER_TEST_default
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (set_confidence_threshold_option_CNS + text2).c_str()
        << delim1
        << " set confidence threshold"
        << " ("
        << text_default
        << " = "
        << DISCARDING_THRESHOLD_default
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (set_tests_option_CNS + text2).c_str()
        << delim1
        << " set number of tests"
        << " ("
        << text_default
        << " = "
        << TOTAL_TESTS_default
        << ")"
        << endl

        << shift1
        << setw (size12)
        << (set_per_calls_option_CNS + text2).c_str()
        << delim1
        << " set scale per calls"
        << " ("
        << text_default
        << " = "
        << PER_CALLS_default
        << ")"
        << endl


        << shift1
        << setw (size12)
        << (log_filename_option_CNS + text3).c_str()
        << delim1
        << " set log file name"
        << " ("
        << text_run_time_default
        << " = "
        << (get_exe_basename() + ".log")
        << ")"
        << endl;

const string help_msg = osstr.str();

ostringstream oss;

  oss  << endl
       << string (line1_len, '=')
       << endl
       << help_msg
       << string (line1_len, '=')
       << endl;

  sout << oss.str();
  cerr << oss.str();

} // show_help_options


// =============================
void ClassManager::show_settings(string msg_i) const
// =============================
{
ostringstream osstr;

const string shift_1 = "   *   ";
const string delim_1 = "   : ";
string text_1 = "ENVIRONMENT VARIABLES";
  text_1 += " ";  
  text_1 += "(";  
  text_1 += msg_i;  
  text_1 += ")";  

const string msg_info1 = "Measurement Report";
const string msg_info2 = "Detailed Measurement Report";
const string msg_info3 = "Tracing Report";

const string msg_info4 = "Total Iterations";
const string msg_info5 = "Total Tests";
const string msg_info6 = "Scale per Calls Value";
const string msg_info7 = "Confidence Threshold";

const string msg_enable = "Enable";
const string msg_disable = "Disable";

const size_t line_1_len = 70;

  osstr.setf (ios::left, ios::adjustfield);

  osstr << ""
        << shift_1
        << space9
        << text_1
        << endl
        << shift_1
        << space9
        << string (text_1.size(), '-')
        << endl

        << shift_1
        << setw (size27)
        << msg_info1.c_str()
        << delim_1
        << (measurement_report_flag_ ? msg_enable : msg_disable)
        << endl;

        
        if (measurement_report_flag_)
        {
          osstr << ""
                << shift_1
                << setw (size27)
                << msg_info2.c_str()
                << delim_1
                << (detailed_measurement_report_flag_ ? msg_enable : msg_disable)
                << endl;
        }

  osstr << ""
        << shift_1
        << setw (size27)
        << msg_info3.c_str()
        << delim_1
        << (tracing_report_flag_ ? msg_enable : msg_disable)
        << endl

        << shift_1
        << setw (size27)
        << msg_info4.c_str()
        << delim_1
        << total_iterations_
        << endl

        << shift_1
        << setw (size27)
        << msg_info5.c_str()
        << delim_1
        << total_tests_
        << endl

        << shift_1
        << setw (size27)
        << msg_info6.c_str()
        << delim_1
        << scale_per_calls_
        << endl

        << shift_1
        << setw (size27)
        << msg_info7.c_str()
        << delim_1
        << discarding_threshold_
        << endl;

const string help_msg = osstr.str();
ostringstream oss;

  oss  << endl
       << string (line_1_len, '=')
       << endl
       << help_msg
       << string (line_1_len, '=')
       << endl
       << endl
       << endl;

  sout << oss.str();
  cerr << oss.str();


} // show_settings




// =============================
void ClassManager::show_notes(string msg_i) const
// =============================
{
  sout << endl;
  if (!msg_i.empty())
  {
     sout << "\t" << space3 << msg_i << endl;
     sout << "\t" << string (space3.size() + msg_i.size(), '-') << endl;
  }

  for (count_t i = 0; i < notes_.size(); i++)
  {
    sout << notes_[i] << endl;
  }
  sout << endl;

} // show_notes



// =============================
bool ClassManager::parse_command_line (int argc, char** argv)
// =============================
{
bool ret_value = false;

  assert (command_line_.empty());
  read_command_line (argc, argv);

  // -----------------------------
  parse0_command_line();

  // -----------------
  open_log ();
const string logfile_info ("Log file  " + (log_filename_.empty() ? string("has not been created") : log_filename_));
  show_logo ("START : " + logfile_info);
  // -----------------

  assert (!command_line_.empty());
  show_help_info (command_line_[0]);


  // -----------------------------
  ret_value |= parse1_command_line();
  ret_value |= parse5_command_line();
  ret_value |= parse4_command_line();

  // -----------------------------
  //show_settings("Default Values");
  // -----------------------------

  ret_value |= parse2_command_line();
  ret_value |= parse3_command_line();
  parse6_command_line();

  // -----------------------------
  show_command_line();
  show_notes ();
  show_settings ("Actual Values");
  // -----------------------------

  return ret_value;

} // ClassManager::parse_command_line()


// =============================
string ClassManager::get_exe_basename () const
// =============================
{
  assert (!command_line_[0].empty());

string name (command_line_[0]);

const size_t ind1 = name.find_last_of('/');
  if (ind1 != string::npos) name = name.substr (ind1 + 1);

const size_t ind2 = name.find_last_of('.');
  if (ind2 != string::npos) name = name.substr (0, ind2);

  return name;
}


// =============================
void ClassManager::parse0_command_line ()
// =============================
{
bool flag = false;
const string pref1 ("\t---> ");
const string str1_logfile ("Missing Log File Name");
const string str2_logfile ("Default Log File Name Used");

  for (count_t i = 0; i < command_line_.size(); i++)
  {
    // --------------------------------
    if (command_line_[i] == log_filename_option_CNS)
    {
      if (command_line_.size() > (i + 1))
      {
        if (!(
             (command_line_[i + 1][0] == ch_minus_CNS) 
             || 
             (command_line_[i + 1][0] == ch_plus_CNS) 
           ))
        {
          set_log_filename (command_line_[i + 1]);
          flag = true;
        }
      }
      
      if (!flag)
      {
        sout << endl;
        sout << endl
             << pref1
             << str1_logfile
             << endl
             << pref1
             << str2_logfile
             << endl;
      }

      break;

    }  // if (command_line_[i] == log_filename_option_CNS)

  } // for (count_t i = 0; i < command_line_.size(); i++)

  // --------------------
  if (!flag)
  {
    set_log_filename (get_exe_basename() + ".log");
  }


} // ClassManager::parse0_command_line()


// =============================
bool ClassManager::parse1_command_line ()
// =============================
{
bool	ret_value = false;
  
 ret_value = (find (
			command_line_.begin(), 
			command_line_.end(), 
			help_option_CNS
			) 
			!= command_line_.end()
		);

  if (ret_value)
  {
    show_brief_man_description();
    show_help_usage(command_line_[0]);
    show_help_example(command_line_[0]);
    show_help_options();
  }

  return ret_value;

} // ClassManager::parse1_command_line()



// =============================
bool ClassManager::parse2_command_line ()
// =============================
{
  
  for (count_t i = 0; i < command_line_.size(); i++)
  {
    // --------------------------------
    if (command_line_[i] == (report_option_disabled_CNS + full_measuring_report_option_CNS))
    {
      set_measurement_report_flag (false);
    }

    // --------------------------------
    if (command_line_[i] == (report_option_enabled_CNS + full_measuring_report_option_CNS))
    {
      set_measurement_report_flag (true);
    }


    // --------------------------------
    if (command_line_[i] == (report_option_disabled_CNS + detailed_measuring_report_option_CNS))
    {
      set_detailed_measurement_report_flag (false);
    }

    // --------------------------------
    if (command_line_[i] == (report_option_enabled_CNS + detailed_measuring_report_option_CNS))
    {
      set_detailed_measurement_report_flag (true);
    }


    // --------------------------------
    if (command_line_[i] == (report_option_disabled_CNS + tracing_report_option_CNS))
    {
      set_tracing_report_flag (false);
    }

    // --------------------------------
    if (command_line_[i] == (report_option_enabled_CNS + tracing_report_option_CNS))
    {
      set_tracing_report_flag (true);
    }


    // --------------------------------
    if (command_line_[i] == set_per_calls_option_CNS)
    {
      bool flag = false;
      if (command_line_.size() > (i + 1))
      {
        ulong entered_value;
	if (string_to (command_line_[i + 1], entered_value))
        {
          set_per_calls (entered_value);
          flag = true;
        }
      }
      
      if (!flag)
      {
        sout << "---> Desired Number of Scale-Per-Calls Value is Illegal or Not Defined "
             << endl
             << "---> Option "
             << set_per_calls_option_CNS
             << " irnored"
             << endl
             << "---> Used Default Value =  "
             << get_scale_per_calls () 
             << endl;
      }
    }  // if (command_line_[i] == set_per_calls_option_CNS)



    // --------------------------------
    if (command_line_[i] == set_tests_option_CNS)
    {
      bool flag = false;
      if (command_line_.size() > (i + 1))
      {
        ulong entered_value;
	if (string_to (command_line_[i + 1], entered_value))
        {
          set_tests (entered_value);
          flag = true;
        }
      }
      
      if (!flag)
      {
        sout << "---> Desired Number of Tests is Illegal or Not Defined "
             << endl
             << "---> Option "
             << set_tests_option_CNS
             << " irnored"
             << endl
             << "---> Used Default Value =  "
             << get_total_tests ()
             << endl;
      }

    } // if (command_line_[i] == set_tests_option_CNS)

    // --------------------------------
    if (command_line_[i] == set_confidence_threshold_option_CNS)
    {
      bool flag = false;
      if (command_line_.size() > (i + 1))
      {
        float entered_value;
	if (string_to (command_line_[i + 1], entered_value))
        {
          set_discarding_threshold (entered_value);
          flag = true;
        }
      }
      
      if (!flag)
      {
        sout << "---> Desired Number of Tests is Illegal or Not Defined "
             << endl
             << "---> Option "
             << set_confidence_threshold_option_CNS
             << " irnored"
             << endl
             << "---> Used Default Value =  "
             << get_discarding_threshold ()
             << endl;
      }


    } // if (command_line_[i] == set_confidence_threshold_option_CNS)


  } // for (count_t i = 0; i < command_line_.size(); i++)


  return false;

} // ClassManager::parse2_command_line()



// =============================
bool ClassManager::parse3_command_line ()
// =============================
{
  
  for (count_t i = 0; i < command_line_.size(); i++)
  {
    // --------------------------------
    if (command_line_[i] == set_iterations_option_CNS)
    {
      bool flag = false;
      if (command_line_.size() > (i + 1))
      {
        ulong entered_value;
	if (string_to (command_line_[i + 1], entered_value))
        {
          set_iterations (entered_value);
          flag = true;
        }
      }
      
      if (!flag)
      {
        sout << "---> Desired Number of Iterations is Illegal or Not Defined "
             << endl
             << "---> Option "
             << set_iterations_option_CNS
             << " irnored"
             << endl
             << "---> Used Default Value =  "
             << get_current_total_iterations ()
             << endl;
      }
    } // if (command_line_[i] == set_iterations_option_CNS)

  } // for (count_t i = 0; i < command_line_.size(); i++)


  return false;

} // ClassManager::parse3_command_line()


// =============================
bool ClassManager::parse4_command_line () const
// =============================
{
bool	ret_value = false;
  
 ret_value = (find (
			command_line_.begin(), 
			command_line_.end(), 
			settings_option_CNS
			) 
			!= command_line_.end()
		);

  return ret_value;

} // ClassManager::parse4_command_line()


// =============================
bool ClassManager::parse5_command_line () const
// =============================
{
bool	ret_value = false;
  			
 ret_value = (find (
			command_line_.begin(), 
			command_line_.end(), 
			version_option_CNS
			) 
			!= command_line_.end()
		);

  if (ret_value)
  {
    show_version();
  }

  return ret_value;

} // ClassManager::parse5_command_line()



// =============================
void ClassManager::parse6_command_line ()
// =============================
{
bool flag;

  // -----------------------------
  flag = (find (
		command_line_.begin(), 
		command_line_.end(), 
		demo_option_CNS
		) 
		!= command_line_.end()
	 );

  if (flag) demo_flag_ = true;

  // -----------------------------
  flag = (find (
		command_line_.begin(), 
		command_line_.end(), 
		getting_groups_option_CNS
		) 
		!= command_line_.end()
	 );

  if (flag) performing_flag_ = false;

  // -----------------------------
  flag = (find (
		command_line_.begin(), 
		command_line_.end(), 
		skipping_actual_run_option_CNS
		) 
		!= command_line_.end()
	 );

  if (flag)
  {
    set_per_calls (1);	// Must be before set_iterations()
    set_iterations (1);
    set_tests (1);
    set_measurement_report_flag (false);
  }


} // ClassManager::parse6_command_line()


// =============================
void ClassManager::show_command_line()
// =============================
{
  sout << endl;
  sout << endl;

#ifdef OPTIMIZATION_LEVEL
  sout << "\t---> COMPILATION       : " << OPTIMIZATION_LEVEL << endl;
#endif

  sout << "\t---> YOUR COMMAND LINE : ";
  copy (
	command_line_.begin(), 
	command_line_.end(), 
	ostream_iterator<string> (sout, " ")
	);

  sout << endl;
  sout << endl;
} // ClassManager::show_command_line



// =============================
void ClassManager::set_tracing_report_flag (bool format_i)
// =============================
{
  tracing_report_flag_ = format_i;
} 

// =============================
bool ClassManager::get_tracing_report_flag () const
// =============================
{
  return tracing_report_flag_;
} 


// =============================
bool ClassManager::get_demo_flag () const
// =============================
{
  return demo_flag_;
} 



// =============================
void ClassManager::read_command_line (int argc, char** argv)
// =============================
{
  assert (command_line_.empty());
  for (int i = 0; i < argc; i++)
  {
    command_line_.push_back (argv[i]);
  }

const string	slash ("/\\");
const string	the_exe_name (command_line_[0]);
string::size_type	pos = the_exe_name.find_last_of (slash);

  if (!(pos == string::npos))
  {
    command_line_[0] = the_exe_name.substr(pos + 1);
  }


} 

#if (defined UNIX_ENV) 
#include <sys/utsname.h>
#endif

// =============================
void ClassManager::show_logo (string msg_i)
// =============================
{
time_t timer;
	timer	= time(NULL);

const string pref1 ("\t#");
const string pref2 (pref1 + " ");

const string sw_version (string ("Version ") + string (sw_tool_version) + "-" + string(sw_testsuites_version));

vector<string>	product_info;
  product_info.push_back (product_name);
  product_info.push_back (string (2, ' ') + product_url);
  product_info.push_back (string (2, ' ') + sw_version);
  product_info.push_back (string (2, ' ') + string (sw_version.size(), '-'));
  product_info.push_back (string (2, ' ') + author_name);
  product_info.push_back (string (2, ' ') + author_url);
  product_info.push_back (string (2, ' ') + author_email);

vector<vector<string> >	system_info;

#if (__DJGPP__)
  system_info.push_back(vector<string>());
  system_info.rbegin()->push_back (pref2);
  system_info.rbegin()->push_back ("DJGPP ");
  system_info.rbegin()->push_back (to_string(__DJGPP__));
#ifdef __DJGPP_MINOR__ 
  system_info.rbegin()->push_back (".");
  system_info.rbegin()->push_back (to_string(__DJGPP_MINOR__));
#endif
#endif  

#if (__MINGW32__ || __MINGW__ )
  system_info.push_back(vector<string>());
  system_info.rbegin()->push_back (pref2);
  system_info.rbegin()->push_back ("MINGW");
#endif  

#if (__CYGWIN32__ || __CYGWIN__)
  system_info.push_back(vector<string>());
  system_info.rbegin()->push_back (pref2);
  system_info.rbegin()->push_back ("CYGWIN");
#endif  

#if (defined UNIX_ENV) 
struct utsname inst_utsname;
  uname (&inst_utsname);
  system_info.push_back(vector<string>());
  system_info.rbegin()->push_back (pref2);
  system_info.rbegin()->push_back (to_string(inst_utsname.sysname));
  system_info.rbegin()->push_back (" ");
  system_info.rbegin()->push_back (to_string(inst_utsname.release));
  system_info.rbegin()->push_back (" ");
  system_info.rbegin()->push_back (to_string(inst_utsname.machine));
#endif  
 
#ifdef __GNUC__  
  system_info.push_back(vector<string>());
  system_info.rbegin()->push_back (pref2);
  system_info.rbegin()->push_back ("GNU gcc version ");
  system_info.rbegin()->push_back (to_string(__GNUC__));
#ifdef __GNUC_MINOR__  
  system_info.rbegin()->push_back (".");
  system_info.rbegin()->push_back (to_string(__GNUC_MINOR__));
#ifdef __GNUC_PATCHLEVEL__   
#if (__GNUC_PATCHLEVEL__)
  system_info.rbegin()->push_back (".");
  system_info.rbegin()->push_back (to_string(__GNUC_PATCHLEVEL__));
#endif  
#endif
#endif
#endif  


size_t the_width = 0;
  for (size_t i = 0; i < product_info.size(); i++)
  {
    the_width = MAX_VALUE (the_width, product_info[i].size());
  }
  
  sout << endl;
  sout << pref1 << string (the_width + 1, '=') << endl;
  for (size_t i = 0; i < product_info.size(); i++)
  {
    sout << pref2 << product_info[i] << endl;
  }

  if (!system_info.empty())
  {
    sout << pref1 << string (the_width + 1, '-') << endl;
  }
  for (size_t i = 0; i < system_info.size(); i++)
  {
    copy (system_info[i].begin(), system_info[i].end(), ostream_iterator<string> (sout, ""));
    sout << endl;
  }
  sout << pref1 << string (the_width + 1, '-') << endl;

  sout << pref2 << msg_i << endl;
  sout << pref2 << asctime (localtime(&timer));
  sout << pref1 << string (the_width + 1, '=') << endl;
  sout << endl;

} // show_logo



// =============================
void ClassManager::show_version(string msg_i) const
// =============================
{
const string	pref1 ("\t===> ");
size_t		size1 = 0;		
  sout << endl;
  if (!msg_i.empty())
  {
     sout << "\t" << space3 << msg_i << endl;
     sout << "\t" << string (space3.size() + msg_i.size(), '-') << endl;
  }
  

const string sw_version (string ("Version ") + string (sw_tool_version) + "-" + string(sw_testsuites_version));
  size1 = MAX_VALUE (size1, sw_version.size());
  sout << pref1 << string (size1, '#') << endl;
  sout << pref1 << sw_version << endl;
  sout << pref1 << string (size1, '#') << endl;

  sout << endl;
  sout << endl;

} // show_version




// =============================
void ClassManager::delete_it ()
// =============================
{
vector<BasicClassMeasurementCollection*> &all_that = 
		BasicClassMeasurementCollection::all_measurement_collections_s;

  for (count_t i = 0; i < all_that.size(); i++)
  {
    delete all_that[i];
  }
  all_that.clear();
  assert (all_that.empty());
}




