// ==============================================================
//
//  Copyright (c) 2003 by Alex Vinokur.
//
//  For conditions of distribution and use, see
//  copyright notice in version.h
//
// ==============================================================


// ##############################################################
//
//  SOFTWARE : Turing Machine with faults, failures and recovery
//             C++ Simulator
//
//  FILE     : t-tapes.cpp
//
//  DESCRIPTION :
//         Class TuringMachine : handling tape methods (Implementation)
//
// ##############################################################


// =================
#include "turing-s.h"
// =================



// --------------------------
// =========
void TuringMachine::remote_master_and_masterbackup_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_markers_i
			)
{

  // --- Returning MASTER_COMPUTATION_TAPE and BACKUP_COMPUTATION_TAPE to left side

  assert (!built_transitions_.empty());

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // MASTER_COMPUTATION_TAPE, BACKUP_COMPUTATION_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_master_tape_alphabet_symbols ()
			));

const symbol_mt	user_defined_marker (get_user_defined_marker ());

  for (size_t i = 0; i < all_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[MASTER_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[BACKUP_COMPUTATION_TAPE] = per_tape_kind_permut[1];


    state_t next_state;
    bool homogeneous_flag = is_homogeneous_permut (all_permuts[i], user_defined_marker);
    if (homogeneous_flag)
    {
      next_state = right_state_for_all_master_markers_i;
    }
    else
    {
      next_state = left_state_i;
    }				


    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case MASTER_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = RIGHT_SHIFT;
              }
              else
              {
                tape_shift = ((tape_symbol == user_defined_marker) ? NO_SHIFT : LEFT_SHIFT);
              }
              cur_vect_rule.push_back (tape_symbol);
              next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;
  
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t
    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation   cur_situation_rule (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

    const NextSituation  next_situation_rule (
				next_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_rule
				);
    pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
    assert (couple_rule.second);

    assert (info_user_transitions_.count (cur_situation_rule) == 0);
    info_user_transitions_[cur_situation_rule] = RULE_NEW;

  } // for (size_t i


} // remote_master_and_masterbackup_tape
    

// =========
void TuringMachine::remote_master_and_tester_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_markers_i
			)
{
  // --- Returning MASTER_COMPUTATION_TAPE and TESTER_TAPE to left side

  assert (!built_transitions_.empty());

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // MASTER_COMPUTATION_TAPE, TESTER_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_master_tape_alphabet_symbols ()
			));

const symbol_mt	user_defined_marker (get_user_defined_marker ());

  for (size_t i = 0; i < all_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[MASTER_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[TESTER_TAPE] = per_tape_kind_permut[1];


    state_t next_state;
    bool homogeneous_flag = is_homogeneous_permut (all_permuts[i], user_defined_marker);
    if (homogeneous_flag)
    {
      next_state = right_state_for_all_master_markers_i;
    }
    else
    {
      next_state = left_state_i;
    }

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case MASTER_COMPUTATION_TAPE :
          case TESTER_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = RIGHT_SHIFT;
              }
              else
              {
                tape_shift = ((tape_symbol == user_defined_marker) ? NO_SHIFT : LEFT_SHIFT);
              }
              cur_vect_rule.push_back (tape_symbol);
              next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;
  
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t
    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation   cur_situation_rule (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

    const NextSituation  next_situation_rule (
				next_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_rule
				);
    pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
    assert (couple_rule.second);

    assert (info_user_transitions_.count (cur_situation_rule) == 0);
    info_user_transitions_[cur_situation_rule] = RULE_NEW;

  } // for (size_t i

} // remote_master_and_tester_tape

// =========
void TuringMachine::remote_synchro_and_synchrobackup_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_synchro_markers_i
			)
{
  assert (!built_transitions_.empty());

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // SYNCHRONOUS_COMPUTATION_TAPE, SYNCHRONOUS_BACKUP_COMPUTATION_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_synchronous_tape_alphabet_symbols_s ()
			));

const symbol_mt	synchro_marker (get_embedded_synchronous_tape_marker_s());

  for (size_t i = 0; i < all_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[SYNCHRONOUS_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE] = per_tape_kind_permut[1];


    state_t next_state;
    bool homogeneous_flag = is_homogeneous_permut (all_permuts[i], synchro_marker);
    if (homogeneous_flag)
    {
      next_state = right_state_for_all_synchro_markers_i;
    }
    else
    {
      next_state = left_state_i;
    }

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = RIGHT_SHIFT;
              }
              else
              {
                tape_shift = ((tape_symbol == synchro_marker) ? NO_SHIFT : LEFT_SHIFT);
              }
              cur_vect_rule.push_back (tape_symbol);
              next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;
  
          case MASTER_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t
    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation   cur_situation_rule (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

    const NextSituation  next_situation_rule (
				next_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_rule
				);
    pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
    assert (couple_rule.second);

    assert (info_user_transitions_.count (cur_situation_rule) == 0);
    info_user_transitions_[cur_situation_rule] = RULE_NEW;

  } // for (size_t i


} // remote_synchro_and_synchrobackup_tape
    

// =========
void TuringMachine::remote_master_and_synchro_and_synchrobackup_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_markers_i
			)
{


  assert (!built_transitions_.empty());

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_master_tape_kinds = 1; // MASTER_COMPUTATION_TAPE
const size_t no_of_active_synchro_tape_kinds = 2; // SYNCHRONOUS_COMPUTATION_TAPE, SYNCHRONOUS_BACKUP_COMPUTATION_TAPE
const size_t no_of_active_multi_tape_kinds = no_of_active_master_tape_kinds + no_of_active_synchro_tape_kinds;

const vector<vector<symbol_mt> > all_master_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_master_tape_kinds, 
			get_all_master_tape_alphabet_symbols ()
			));

const vector<vector<symbol_mt> > all_synchro_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_synchro_tape_kinds, 
			get_all_synchronous_tape_alphabet_symbols_s ()
			));

const symbol_mt user_defined_marker (get_user_defined_marker ());
const symbol_mt	synchro_marker (get_embedded_synchronous_tape_marker_s());

const vector<vector<symbol_mt> > all_multi_permuts (multi_permut (all_master_permuts, all_synchro_permuts));

  for (size_t i = 0; i < all_multi_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_floor4_3;
    vector<SymbolAndShift>	next_vect_floor4_3;


    assert (all_multi_permuts[i].size() == (number_of_tapes * no_of_active_multi_tape_kinds));
    const vector<vector<symbol_mt> >	per_multi_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_multi_tape_kinds,
				all_multi_permuts[i]
				));
    assert (per_multi_tape_kind_permut.size() == no_of_active_multi_tape_kinds);


    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[MASTER_COMPUTATION_TAPE] = per_multi_tape_kind_permut[0];
    tape_data[SYNCHRONOUS_COMPUTATION_TAPE] = per_multi_tape_kind_permut[1];
    tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE] = per_multi_tape_kind_permut[2];

    vector<symbol_mt> tmp_master_permut;
    for (size_t f = 0; f < tape_data[MASTER_COMPUTATION_TAPE].size(); f++)
    {
      tmp_master_permut.push_back (tape_data[MASTER_COMPUTATION_TAPE][f]);
    }
    assert (tmp_master_permut.size() == (number_of_tapes * no_of_active_master_tape_kinds));
  
    vector<symbol_mt> tmp_synchro_permut;
    for (size_t f = 0; f < tape_data[SYNCHRONOUS_COMPUTATION_TAPE].size(); f++)
    {
      tmp_synchro_permut.push_back (tape_data[SYNCHRONOUS_COMPUTATION_TAPE][f]);
    }
    for (size_t f = 0; f < tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE].size(); f++)
    {
      tmp_synchro_permut.push_back (tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE][f]);
    }
    assert (tmp_synchro_permut.size() == (number_of_tapes * no_of_active_synchro_tape_kinds));


    state_t next_state;

    bool homogeneous_flag =
       (
         (is_homogeneous_permut (tmp_master_permut, user_defined_marker))
         &&
         (is_homogeneous_permut (tmp_synchro_permut, synchro_marker))
       );

    if (homogeneous_flag)
    {
      next_state = right_state_for_all_markers_i;
    }
    else
    {
      next_state = left_state_i;
    }

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case MASTER_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = RIGHT_SHIFT;
              }
              else
              {
                tape_shift = ((tape_symbol == user_defined_marker) ? NO_SHIFT : LEFT_SHIFT);
              }
              cur_vect_floor4_3.push_back (tape_symbol);
              next_vect_floor4_3.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;

          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = RIGHT_SHIFT;
              }
              else
              {
                tape_shift = ((tape_symbol == synchro_marker) ? NO_SHIFT : LEFT_SHIFT);
              }
              cur_vect_floor4_3.push_back (tape_symbol);
              next_vect_floor4_3.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;
  
          case BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
            assert (tape_data[s].empty());
            cur_vect_floor4_3.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_floor4_3.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t
    } // for (size_t s = 0

    assert (cur_vect_floor4_3.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_floor4_3.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation   cur_situation_floor4_3 (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_floor4_3, 
				get_symbol_mtypes (cur_vect_floor4_3), 
				false
				);

    const NextSituation  next_situation_floor4_3 (
				next_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_floor4_3
				);
    pair<Transitions_t::iterator, bool> couple_floor4_3 = built_transitions_.insert (
			value_type (cur_situation_floor4_3, next_situation_floor4_3)
			);
    assert (couple_floor4_3.second);

    assert (info_user_transitions_.count (cur_situation_floor4_3) == 0);
    info_user_transitions_[cur_situation_floor4_3] = RULE_NEW;

  } // for (size_t i

} // remote_master_and_synchro_and_synchrobackup_tape

// =========
void TuringMachine::compare_master_and_tester_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_blanks_i,
			const state_t&	right_state_for_not_identical_i
			)
{

  assert (!built_transitions_.empty());

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // MASTER_COMPUTATION_TAPE, TESTER_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_not_marker_master_tape_alphabet_symbols ()
			));

  assert (user_defined_blank_alphabet_.size() == 1);
const symbol_mt user_defined_blank = user_defined_blank_alphabet_.front();

  for (size_t i = 0; i < all_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[MASTER_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[TESTER_TAPE] = per_tape_kind_permut[1];

    assert (tape_data[MASTER_COMPUTATION_TAPE].size() == tape_data[TESTER_TAPE].size());
    const vector<bool> tape_compare (vcompare (tape_data[MASTER_COMPUTATION_TAPE], tape_data[TESTER_TAPE]));
    const bool tape_bool_compare = get_true_vector (tape_compare);

    state_t next_state;
    bool homogeneous_flag = is_homogeneous_permut (all_permuts[i], user_defined_blank);
    if (homogeneous_flag)
    {
      assert (tape_bool_compare);
      next_state = right_state_for_all_master_blanks_i;
    }
    else
    {
      if (tape_bool_compare)
      {
        next_state = left_state_i;
      }
      else
      {
        next_state = right_state_for_not_identical_i;
      }
    }

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case MASTER_COMPUTATION_TAPE :
          case TESTER_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = LEFT_SHIFT;
              }
              else
              {
                if (tape_compare[t])
                {
                  if (tape_data[s][t] == user_defined_blank)
                  {
                    tape_shift = NO_SHIFT;
                  }
                  else
                  {
                    tape_shift = RIGHT_SHIFT;
                  }
                }
                else
                {
                  tape_shift = LEFT_SHIFT;
                }
              }
              cur_vect_rule.push_back (tape_symbol);
              next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;
  
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t
    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation	cur_situation_rule (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

    const NextSituation	next_situation_rule (
				next_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_rule
				);
    pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
    assert (couple_rule.second);

    assert (info_user_transitions_.count (cur_situation_rule) == 0);
    info_user_transitions_[cur_situation_rule] = RULE_NEW;

  } // for (size_t i


} // compare_master_and_tester_tape


// =========
void TuringMachine::compare_master_and_masterbackup_tape (
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_blanks_i,
			const state_t&	right_state_for_not_identical_i
			)
{

  assert (!built_transitions_.empty());

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // MASTER_COMPUTATION_TAPE, BACKUP_COMPUTATION_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_not_marker_master_tape_alphabet_symbols ()
			));

  assert (user_defined_blank_alphabet_.size() == 1);
const symbol_mt user_defined_blank = user_defined_blank_alphabet_.front();

  for (size_t i = 0; i < all_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[MASTER_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[BACKUP_COMPUTATION_TAPE] = per_tape_kind_permut[1];

    assert (tape_data[MASTER_COMPUTATION_TAPE].size() == tape_data[BACKUP_COMPUTATION_TAPE].size());
    const vector<bool> tape_compare (vcompare (tape_data[MASTER_COMPUTATION_TAPE], tape_data[BACKUP_COMPUTATION_TAPE]));
    const bool tape_bool_compare = get_true_vector (tape_compare);

    state_t next_state;
    bool homogeneous_flag = is_homogeneous_permut (all_permuts[i], user_defined_blank);
    if (homogeneous_flag)
    {
      assert (tape_bool_compare);
      next_state = right_state_for_all_master_blanks_i;
    }
    else
    {
      if (tape_bool_compare)
      {
        next_state = left_state_i;
      }
      else
      {
        next_state = right_state_for_not_identical_i;
      }
    }

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case MASTER_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = LEFT_SHIFT;
              }
              else
              {
                if (tape_compare[t])
                {
                  if (tape_data[s][t] == user_defined_blank)
                  {
                    tape_shift = NO_SHIFT;
                  }
                  else
                  {
                    tape_shift = RIGHT_SHIFT;
                  }
                }
                else
                {
                  tape_shift = LEFT_SHIFT;
                }
              }
              cur_vect_rule.push_back (tape_symbol);
              next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;
  
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case TESTER_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t
    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation	cur_situation_rule (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

    const NextSituation	next_situation_rule (
				next_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_rule
				);
    pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
    assert (couple_rule.second);

    assert (info_user_transitions_.count (cur_situation_rule) == 0);
    info_user_transitions_[cur_situation_rule] = RULE_NEW;

  } // for (size_t i

} // compare_master_and_masterbackup_tape


// =========
void TuringMachine::compare_synchro_and_synchrobackup_tape_and_setting_master_tape (
			const state_t&	left_state_i, 
			const vector<state_t>&	right_states_for_all_synchro_blanks_i,
			const state_t&	right_state_for_not_identical_i
			)
{

  assert (!built_transitions_.empty());

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // SYNCHRONOUS_COMPUTATION_TAPE, SYNCHRONOUS_BACKUP_COMPUTATION_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_not_marker_synchronous_tape_alphabet_symbols_s ()
			));


  assert (embedded_blank_synchronous_tape_alphabet_s.size() == 1);
const symbol_mt synchro_blank (embedded_blank_synchronous_tape_alphabet_s.front());
const symbol_mt	synchro_marker (get_embedded_synchronous_tape_marker_s());

  for (size_t i = 0; i < all_permuts.size(); i++)
  {

    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[SYNCHRONOUS_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE] = per_tape_kind_permut[1];

    assert (tape_data[SYNCHRONOUS_COMPUTATION_TAPE].size() == tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE].size());
    const vector<bool> tape_compare (vcompare (tape_data[SYNCHRONOUS_COMPUTATION_TAPE], tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE]));
    const bool tape_bool_compare = get_true_vector (tape_compare);


    vector<state_t> states_vect;
    bool homogeneous_flag = is_homogeneous_permut (all_permuts[i], synchro_blank);
    if (homogeneous_flag)
    {
      assert (tape_bool_compare);
      states_vect = right_states_for_all_synchro_blanks_i;  // TO BE TREATED in roof-function
    }
    else
    {
      if (tape_bool_compare)
      {	
        states_vect.push_back (left_state_i);
      }
      else
      {
        states_vect.push_back (right_state_for_not_identical_i);
      }
    }

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = NO_SHIFT;
              }
              else
              {
                if (tape_compare[t])
                {
                  if (tape_data[s][t] == synchro_blank)
                  {
                    tape_shift = NO_SHIFT;
                  }
                  else
                  {
                    tape_shift = RIGHT_SHIFT;
                  }
                }
                else
                {
                  tape_shift = LEFT_SHIFT;
                }
              }
              cur_vect_rule.push_back (tape_symbol);
              next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;
  
          case MASTER_COMPUTATION_TAPE :
            assert (tape_data[s].empty());
            {
              shift_t tape_shift;
              if (homogeneous_flag)
              {
                tape_shift = NO_SHIFT;
              }
              else
              {
                if (tape_compare[t])
                {
                  if (tape_data[SYNCHRONOUS_COMPUTATION_TAPE][t] == synchro_blank)
                  {
                    tape_shift = NO_SHIFT;
                  }
                  else
                  {
                    tape_shift = RIGHT_SHIFT;
                  }
                }
                else
                {
                  tape_shift = NO_SHIFT;
                }
              }
              cur_vect_rule.push_back (pseudo_not_marker_symbols_alphabet_.front());
              next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
            }
            break;

          case TESTER_TAPE :
          case BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t
    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    for (size_t d = 0; d < states_vect.size(); d++)
    {

      const CurSituation  cur_situation_rule (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
			        ((homogeneous_flag) ? states_vect[d] : get_neutral_cur_substate_alphabet_symbol ()),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

      const NextSituation next_situation_rule (
				states_vect[d],
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
			        // old ((homogeneous_flag) ? get_neutral_next_not_the_same_substate_alphabet_symbol() : get_neutral_next_the_same_substate_alphabet_symbol()),
				next_vect_rule
				);
      pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
      assert (couple_rule.second);

      assert (info_user_transitions_.count (cur_situation_rule) == 0);
      info_user_transitions_[cur_situation_rule] = RULE_NEW;
    }

  } // for (size_t i

} // compare_synchro_and_synchrobackup_tape_and_setting_master_tape


// =========
void TuringMachine::backup_master_and_masterbackup_tape (
			TapeKinds	from_tape_i, 
			TapeKinds	to_tape_i, 
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_master_blanks_i
			)
{

  assert (!built_transitions_.empty());

  assert (from_tape_i != to_tape_i);
  assert ((from_tape_i == MASTER_COMPUTATION_TAPE) || (from_tape_i == BACKUP_COMPUTATION_TAPE));
  assert ((to_tape_i == MASTER_COMPUTATION_TAPE) || (to_tape_i == BACKUP_COMPUTATION_TAPE));

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // MASTER_COMPUTATION_TAPE, BACKUP_COMPUTATION_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_not_marker_master_tape_alphabet_symbols ()
			));

  assert (user_defined_blank_alphabet_.size() == 1);
const symbol_mt user_defined_blank (user_defined_blank_alphabet_.front());
const symbol_mt user_defined_marker (get_user_defined_marker ());

  // ------ Loop#1 ------
  for (size_t i = 0; i < all_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[MASTER_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[BACKUP_COMPUTATION_TAPE] = per_tape_kind_permut[1];

    assert (tape_data[MASTER_COMPUTATION_TAPE].size() == tape_data[BACKUP_COMPUTATION_TAPE].size());
    const vector<bool> tape_compare (vcompare (tape_data[MASTER_COMPUTATION_TAPE], tape_data[BACKUP_COMPUTATION_TAPE]));
    const bool tape_bool_compare = get_true_vector (tape_compare);

    state_t next_state;
    bool homogeneous_flag = is_homogeneous_permut (all_permuts[i], user_defined_blank);
    if (homogeneous_flag)
    {
      assert (tape_bool_compare);
      next_state = right_state_for_all_master_blanks_i;
    }
    else
    {
      next_state = left_state_i;
    }

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case MASTER_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = LEFT_SHIFT;
              }
              else
              {
                tape_shift = RIGHT_SHIFT;
              }

              cur_vect_rule.push_back (tape_symbol);
              if (s == static_cast<size_t>(from_tape_i))
              {
                next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
              }
              else
              {
                assert (s == static_cast<size_t>(to_tape_i));
                next_vect_rule.push_back (SymbolAndShift (tape_data[from_tape_i][t], tape_shift));     
              } 
            }
            break;
  
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t

    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation	cur_situation_rule (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

    const NextSituation	next_situation_rule (
				next_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_rule
				);
    pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
    assert (couple_rule.second);

    assert (info_user_transitions_.count (cur_situation_rule) == 0);
    info_user_transitions_[cur_situation_rule] = RULE_NEW;

  } // for (size_t i

} // backup_master_and_masterbackup_tape

    
// =========
void TuringMachine::backup_synchro_and_synchrobackup_tape (
			TapeKinds	from_tape_i, 
			TapeKinds	to_tape_i, 
			const state_t&	left_state_i, 
			const state_t&	right_state_for_all_synchro_blanks_i
			)
{
  assert (!built_transitions_.empty());

  assert (from_tape_i != to_tape_i);
  assert ((from_tape_i == SYNCHRONOUS_COMPUTATION_TAPE) || (from_tape_i == SYNCHRONOUS_BACKUP_COMPUTATION_TAPE));
  assert ((to_tape_i == SYNCHRONOUS_COMPUTATION_TAPE) || (to_tape_i == SYNCHRONOUS_BACKUP_COMPUTATION_TAPE));

typedef Transitions_t::value_type		value_type;

const size_t	number_of_tapes = tapes_[MASTER_COMPUTATION_TAPE].size();

  assert (!user_defined_blank_alphabet_.empty());
  assert (!pseudo_never_mind_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_symbols_alphabet_.empty());
  assert (!pseudo_not_marker_and_not_blank_symbols_alphabet_.empty());
  assert (!pseudo_write_nothing_symbols_alphabet_.empty());

const size_t no_of_active_tape_kinds = 2; // SYNCHRONOUS_COMPUTATION_TAPE, SYNCHRONOUS_BACKUP_COMPUTATION_TAPE
const vector<vector<symbol_mt> > all_permuts (
		get_all_taped_permuts (
			number_of_tapes, 
			no_of_active_tape_kinds, 
			get_all_synchronous_tape_alphabet_symbols_s ()
			));


  assert (embedded_blank_synchronous_tape_alphabet_s.size() == 1);
const symbol_mt synchro_blank (embedded_blank_synchronous_tape_alphabet_s.front());
const symbol_mt	synchro_marker (get_embedded_synchronous_tape_marker_s());

  // ------ Loop#1 ------
  for (size_t i = 0; i < all_permuts.size(); i++)
  {
    vector<symbol_mt>		cur_vect_rule;
    vector<SymbolAndShift>	next_vect_rule;

    assert (all_permuts[i].size() == (number_of_tapes * no_of_active_tape_kinds));
    const vector<vector<symbol_mt> >	per_tape_kind_permut (
			get_one_per_tape_kind_permut (
				number_of_tapes, 
				no_of_active_tape_kinds,
				all_permuts[i]
				));
    assert (per_tape_kind_permut.size() == no_of_active_tape_kinds);

    vector<vector<symbol_mt> >	tape_data (NUMBER_OF_TAPE_KINDS);
    tape_data[SYNCHRONOUS_COMPUTATION_TAPE] = per_tape_kind_permut[0];
    tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE] = per_tape_kind_permut[1];

    assert (tape_data[SYNCHRONOUS_COMPUTATION_TAPE].size() == tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE].size());
    const vector<bool> tape_compare (vcompare (tape_data[SYNCHRONOUS_COMPUTATION_TAPE], tape_data[SYNCHRONOUS_BACKUP_COMPUTATION_TAPE]));
    const bool tape_bool_compare = get_true_vector (tape_compare);

    state_t next_state;
    bool homogeneous_flag = is_homogeneous_permut (all_permuts[i], synchro_blank);
    if (homogeneous_flag)
    {
      assert (tape_bool_compare);
      next_state = right_state_for_all_synchro_blanks_i;
    }
    else
    {
      next_state = left_state_i;
    }

    for (size_t s = 0; s < NUMBER_OF_TAPE_KINDS; s++)
    {
      for (size_t t = 0; t < number_of_tapes; t++)
      {
        switch (s)
        {
          case SYNCHRONOUS_COMPUTATION_TAPE :
          case SYNCHRONOUS_BACKUP_COMPUTATION_TAPE :
            assert (tape_data[s].size() == number_of_tapes);
            {
              shift_t tape_shift;
              const symbol_mt tape_symbol (tape_data[s][t]);
              if (homogeneous_flag)
              {
                tape_shift = LEFT_SHIFT;
              }
              else
              {
                tape_shift = RIGHT_SHIFT;
              }

              cur_vect_rule.push_back (tape_symbol);
              if (s == static_cast<size_t>(from_tape_i))
              {
                next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), tape_shift));     
              }
              else
              {
                assert (s == static_cast<size_t>(to_tape_i));
                next_vect_rule.push_back (SymbolAndShift (tape_data[from_tape_i][t], tape_shift));     
              } 
            }
            break;
  
          case MASTER_COMPUTATION_TAPE :
          case BACKUP_COMPUTATION_TAPE :
          case TESTER_TAPE :
            assert (tape_data[s].empty());
            cur_vect_rule.push_back (pseudo_never_mind_symbols_alphabet_.front());
            next_vect_rule.push_back (SymbolAndShift (pseudo_write_nothing_symbols_alphabet_.front(), NO_SHIFT));     
            break;
  
          default :    
            assert (0);
            break;
        }
      }	// for (size_t t

    } // for (size_t s = 0

    assert (cur_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));
    assert (next_vect_rule.size() == (number_of_tapes * NUMBER_OF_TAPE_KINDS));


    const CurSituation	cur_situation_rule (
  				left_state_i,
				get_neutral_cur_substate_alphabet_symbol (),
				get_neutral_cur_substate_alphabet_symbol (),
				cur_vect_rule, 
				get_symbol_mtypes (cur_vect_rule), 
				false
				);

    const NextSituation	next_situation_rule (
				next_state,
				get_neutral_next_the_same_substate_alphabet_symbol(),
				get_neutral_next_the_same_substate_alphabet_symbol(),
				next_vect_rule
				);
    pair<Transitions_t::iterator, bool> couple_rule = built_transitions_.insert (
			value_type (cur_situation_rule, next_situation_rule)
			);
    assert (couple_rule.second);

    assert (info_user_transitions_.count (cur_situation_rule) == 0);
    info_user_transitions_[cur_situation_rule] = RULE_NEW;

  } // for (size_t i


} // backup_synchro_and_synchrobackup_tape


