/*===========================================================================
SOLAR v0.95.3 solar :: Module ..\solar\config.c
Original Author: Kevin Houle <kjhoule@iowegia.dsm.ia.us>

This software module has been placed in the public domain.
===========================================================================*/

/* Header Files */
#include <stdio.h>
#include <string.h>
#include <dir.h>
#include <stdlib.h>
#include "config.h"

/*
 * Function: int load_static(char username[10])
 * Purpose : Load parameters from Waffle's static file.
 * Return  : 0 on success, non-zero on error and set _slrerr
*/

int load_static(char username[10])
{
	FILE *static_file = NULL;

	char buffer[128];
	char delimiter = ':';
	char *path = NULL;

	/* Get path to static file and open it */

	if ((path = getenv("WAFFLE")) == NULL)
	{
		strcpy(_slrerr,"load_static(): environment variable WAFFLE not set.");
		goto ErrorExit;
	}
	strcpy(static_path,path);
  if (verbose == YES)
    RPrintf("Waffle static file: %s\n",static_path);

	if ((static_file = fopen(static_path,"rt")) == NULL)
	{
		sprintf(_slrerr,"load_static(): error opening %s for reading.",static_path);
		goto ErrorExit;
	}

	/* Read static file and pull out what we need. */

	while (fgets(buffer,128,static_file) != NULL)
	{
		/* Build path to user's directory */

		if (strnicmp(buffer,"USER",4) == 0)
		{
			strcpy(user_path,extract_parm(buffer,delimiter));
			strcat(user_path, "\\");
			strcat(user_path, username);
      if (verbose == YES)
        RPrintf("User Path: %s\n",user_path);
		}

    /* Set path to Solar's configuration file.
       Give 'solar95' priority. */

    if (strnicmp(buffer,"SOLAR95",7) == 0)
		{
			strcpy(config_path,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("Solar config path: %s\n",config_path);
		}

    if ((strnicmp(buffer,"SOLAR",5) == 0) && (strcmp(config_path,"NONE") == 0))
		{
			strcpy(config_path,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("Solar config path: %s\n",config_path);
		}

		/* Get path to temporary directory. This only gets used if
			 'solarwork' is not set in Solar's configuration file. */

		if (strnicmp(buffer,"TEMPORARY",9) == 0)
		{
			if (strcmp(temp_path,"NONE") == 0)
			{
				strcpy(temp_path,extract_parm(buffer,delimiter));
        if (verbose == YES)
          RPrintf("Temp path: %s\n",temp_path);
			}
		}

    /* Store system's UUCP name */

    if (strnicmp(buffer,"UUCPNAME",8) == 0)
    {
      strcpy(uucpname,extract_parm(buffer,delimiter));
      if (verbose == YES)
        RPrintf("System UUCP name: %s\n",uucpname);
    }

		/* Store path to Waffle root directory */

		if (strnicmp(buffer,"WAFFLE",6) == 0)
		{
			strcpy(waffle_path,extract_parm(buffer,delimiter));

			/* Set default path to logfile */

			strcpy(log_path,waffle_path);
			strcat(log_path,"\\admin\\solar");
			if (verbose == YES)
				RPrintf("Log file: %s\n",log_path);

			/* Set path to Waffle's system directory */

			strcpy(system_path,waffle_path);
			strcat(system_path,"\\system");
      if (verbose == YES)
        RPrintf("System path: %s\n",system_path);

      /* Set path to Waffle's extern directory */

			strcpy(extern_path,waffle_path);
			strcat(extern_path,"\\extern");
      if (verbose == YES)
        RPrintf("Extern path: %s\n",extern_path);
		}

		/* Check the driver, use fossil only if selected on command line
			 and in static file 'driver' parameter. */

		if (strnicmp(buffer,"DRIVER",6) == 0)
		{
			if (use_fossil == YES)
			{
				if (strnicmp(extract_parm(buffer,delimiter),"FOSSIL",6) == 0)
				{
					use_fossil = YES;
          if (verbose == YES)
            RPrintf("Fossil support enabled.\n");
				}
				else
				{
					use_fossil = NO;
          if (verbose == YES)
            RPrintf("Fossil support disabled.\n");
				}
			}
			else
			{
        if (verbose == YES)
          RPrintf("Fossil support disabled.\n");
			}
		}

		/* Load comm device, in case we are using a fossil driver. */

		if (strnicmp(buffer,"DEVICE",6) == 0)
		{
			port = atoi(extract_parm(buffer,delimiter)) - 1;
      if (verbose == YES)
        RPrintf("COM Port: %u\n",port);
		}
	}
	fclose(static_file);

	if (verbose == YES)
    RPrintf("Finished reading static, validating...\n");

	/* Check to make sure we got all that we need. */

	if (strcmp(temp_path,"NONE") == 0)
	{
		strcpy(_slrerr,"load_static(): temporary: parameter not found in static file");
		goto ErrorExit;
	}
	if (strcmp(user_path,"NONE") == 0)
	{
		strcpy(_slrerr,"load_static(): user: parameter not found in static file.");
		goto ErrorExit;
	}
	if (strcmp(waffle_path,"NONE") == 0)
	{
		strcpy(_slrerr,"load_static(): waffle: parameter not found in static file.");
		goto ErrorExit;
	}
  if (strcmp(uucpname,"NONE") == 0)
  {
    strcpy(_slrerr,"load_static(): uucpname: parameter not found in static file.");
    goto ErrorExit;
  }
  if ((use_fossil == YES) && (port == 99))
	{
		strcpy(_slrerr,"load_static(): device: parameter not found in static file.");
		goto ErrorExit;
	}

	if (verbose == YES)
    RPrintf("Static file load OK\n\n");

GoodExit:
	return 0;
ErrorExit:
	if (static_file) fclose(static_file);
	return 1;
}

/*
 * Function: load_config()
 * Purpose : Load Solar's configuration file.
 * Return  : 0 on success, non-zero on error and set _slrerr.
*/

int load_config()
{
	FILE *config_file = NULL;
	char buf[128];
	char delimiter = '=';
	char *path = NULL;

	/* Get path to config file if not already set */

	if (strcmp(config_path,"NONE") == 0)
	{
    if ((path = getenv("SOLAR95")) == NULL)
    {
      if ((path = getenv("SOLAR")) == NULL)
      {
        strcpy(_slrerr,"load_config(): environment variable SOLAR not set.");
        goto ErrorExit;
      }
    }
    strcpy(config_path,path);
	}

  if (verbose == YES)
    RPrintf("Solar config file: %s\n",config_path);

	if ((config_file = fopen(config_path,"rt")) == NULL)
	{
		sprintf(_slrerr,"load_config(): could not open %s for reading.",config_path);
		goto ErrorExit;
	}

	while (fgets(buf,128,config_file) != NULL)
	{
		/* Override the default temporary work directory */

    if (strnicmp(buf,"SOLARWORK",9) == 0)
		{
			strcpy(temp_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Temp path set to: %s\n",temp_path);
		}

    /* Set path to swap directory for SPAWNO */

    if (strnicmp(buf,"SOLARSWAP",9) == 0)
		{
      strcpy(swap_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Swap path set to: %s\n",swap_path);
    }

    /* Set swap option for solar.exe */

    if (strnicmp(buf,"SWAP-SOLAR",10) == 0)
    {
      diskswap = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
        if (diskswap == YES)
          RPrintf("Disk Swap set to YES\n");
				else
          RPrintf("Disk Swap set to NO\n");
      }
    }

    /* Set Solar System home path */

		if (strnicmp(buf,"SOLARPATH",9) == 0)
		{
			strcpy(solar_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Solar path set to: %s\n",solar_path);
		}

    /* Set path to menu file */

    if (strnicmp(buf,"MENUPATH",8) == 0)
    {
      strcpy(menu_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Menu path set to: %s\n",menu_path);
    }

    /* Optional name for root extern filename */

    if (strnicmp(buf,"EXTERN-ROOT",11) == 0)
    {
      strcpy(extern_root,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Extern root set to: %s\n",extern_root);
    }

    /* Override the default log file path */

		if (strnicmp(buf,"LOGFILE",7) == 0)
		{
			strcpy(log_path,extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Log file set to: %s\n",log_path);
		}

		/* Override the default SOUP index type for news */

		if (strnicmp(buf,"NEWS-INDEX",10) == 0)
		{
			news_index = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (news_index == n) RPrintf("News index: n (None)\n");
				if (news_index == C) RPrintf("News index: C (Partial C News)\n");
				if (news_index == c) RPrintf("News index: c (Full C News)\n");
			}
		}

		/* Override the default SOUP message type for news */

		if (strnicmp(buf,"NEWS-TYPE",9) == 0)
		{
			news_type = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (news_type == u) RPrintf("news type: u (USENET Format)\n");
        if (news_type == B) RPrintf("news type: B (8-bit Binary)\n");
        if (news_type == i) RPrintf("news type: i (Index only)\n");
			}
		}

		/* Set the global limit of bytes per message area */

		if (strnicmp(buf,"NEWS-AREA-BYTES",15) == 0)
		{
			news_area_bytes = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Max. bytes per area: %ul\n",news_area_bytes);
		}

		/* Set the global limit of messages per message area */

		if (strnicmp(buf,"NEWS-AREA-MESSAGES",18) == 0)
		{
			news_area_messages = atol(extract_parm(buf,delimiter));
      if (verbose == YES)
        RPrintf("Max. msgs per area: %ul\n",news_area_messages);
		}

		/* Set the global limit of total bytes */

		if (strnicmp(buf,"NEWS-TOTAL-BYTES",16) == 0)
		{
			news_total_bytes = atol(extract_parm(buf,delimiter));
			if (verbose == YES)
				RPrintf("Max. bytes per area: %ul\n",news_total_bytes);
		}

		/* Set the global limit of total messages */

		if (strnicmp(buf,"NEWS-TOTAL-MESSAGES",18) == 0)
		{
			news_total_messages = atol(extract_parm(buf,delimiter));
			if (verbose == YES)
				RPrintf("Max. msgs per area: %ul\n",news_total_messages);
		}

		/* Override the default SOUP index type for mail */

		if (strnicmp(buf,"MAIL-INDEX",10) == 0)
		{
			mail_index = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (mail_index == n) RPrintf("Mail index: n (None)\n");
				if (mail_index == C) RPrintf("Mail index: C (Partial C News)\n");
				if (mail_index == c) RPrintf("Mail index: c (Full C News)\n");
				if (mail_index == i) RPrintf("Mail index: i (Offset/length)\n");
			}
		}

		/* Override the default SOUP message type for mail */

		if (strnicmp(buf,"MAIL-TYPE",9) == 0)
		{
			mail_type = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (mail_type == m) RPrintf("Mail type: m (Unix Format)\n");
				if (mail_type == M) RPrintf("Mail type: M (MMDF Format)\n");
				if (mail_type == b) RPrintf("Mail type: b (Binary 8-bit)\n");
				if (mail_type == u) RPrintf("Mail type: u (USENET news)\n");
				if (mail_type == i) RPrintf("Mail type: i (Index only)\n");
			}
		}

		/* Set the default file transfer protocol */

		if ((strnicmp(buf,"RECEIVE",7)) == 0)
		{
			if (strcmp(protocol,"NONE") == 0)
			{
				if (get_protocol_type(buf) != 0) goto ErrorExit;
				if (verbose == YES)
					RPrintf("Default protocol: %s\n",protocol);
			}
		}

		if (strnicmp(buf,"SEND-MAIL",9) == 0)
		{
			send_mail = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (send_mail == YES)
					RPrintf("Send mail set to YES\n");
				else
					RPrintf("Send mail set to NO\n");
			}
		}
		if ((strnicmp(buf,"DELETE-MAIL",11)) == 0)
		{
			delete_mail = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (delete_mail == YES)
					RPrintf("Delete mail set to YES\n");
				else
					RPrintf("Delete mail set to NO\n");
			}
		}
		if ((strnicmp(buf,"SEND-LIST",9)) == 0)
		{
			send_list = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (send_list == YES)
					RPrintf("Send list set to YES\n");
				else
          RPrintf("Send list set to NO\n");
			}
		}
		if (strnicmp(buf,"COMPRESS",8) == 0)
		{
			if (strcmp(compress,"NONE") == 0)
			{
				if (get_compress_type(buf) != 0) goto ErrorExit;
				if (verbose == YES)
          RPrintf("Compression Method: %s\n",compress);
			}
		}
    if (strnicmp(buf,"STUPID-USER",11) == 0)
    {
      stupid_user = atoi(extract_parm(buf,delimiter));
      if (verbose)
        RPrintf("Stupid user threshold: %u\n",stupid_user);
    }
    if (strnicmp(buf,"DISKMINIMUM",11) == 0)
    {
      disk_minimum = atol(extract_parm(buf,delimiter));
      if (verbose)
        RPrintf("Minimum disk space: %ul\n",disk_minimum);
    }
  }
	fclose(config_file);

	if (verbose == YES)
		RPrintf("Solar config file loaded. Validating...\n");

	/* Make sure the SOUP index and message types for news are supported */

	switch (news_index) {
		case n	: break;
		case c	: break;
		case C	: break;
		default : sprintf(_slrerr,"load_config(): unsupported news-index in %s\n",config_path);
							goto ErrorExit;
	}
	switch (news_type) {
		case u	: break;
    case B  : break;
    case i  : break;
		default : sprintf(_slrerr,"load_config(): unsupported news-type in %s\n",config_path);
							goto ErrorExit;
	}

	/* Make sure the SOUP index and message types for mail are supported */

	switch (mail_index) {
		case n	: break;
		default : sprintf(_slrerr,"load_config(): unsupported mail-index in %s\n",config_path);
							goto ErrorExit;
	}

	switch (mail_type) {
		case m	: break;
		case M	: break;
    case b  : break;
    default : sprintf(_slrerr,"load_config(): unsupported mail-type in %s\n",config_path);
							goto ErrorExit;
	}

	/* The protocol must be set in the config file, there are no defaults */

	if (strcmp(protocol,"NONE") == 0)
	{
		sprintf(_slrerr,"load_config(): no transfer protocols defined.");
		goto ErrorExit;
	}

	/* The compression method must be set, there are no defaults */

	if (strcmp(compress,"NONE") == 0)
	{
		sprintf(_slrerr,"load_config(): no compression methods defined.");
		goto ErrorExit;
	}

  /* If menupath is not set in Solar's config file, build from
     Waffle's directory tree. */

  if (strcmp(menu_path,"NONE") == 0)
  {
    strcpy(menu_path,waffle_path);
    strcat(menu_path,"\\menus\\solar");
  }

	if (verbose == YES)
    RPrintf("Solar config file load OK\n\n");

GoodExit:
	return 0;
ErrorExit:
	if (config_file) fclose(config_file);
	return 1;
}

/*
 * Function: user_defaults()
 * Purpose : Check for a user config file and load values if present.
 * Return  : zero on success, non-zero on error and set _slrerr.
*/

int user_defaults()
{
	FILE *user_config = NULL;

	char buf[128];
	char path[MAXPATH];
	char delimiter = '=';

	strcpy(path,user_path);
	strcat(path,"\\");
	strcat(path,OPTIONS_NAME);

	if (verbose == YES)
		RPrintf("User config file: %s\n",path);

	if ((user_config = fopen(path,"rt")) == NULL)
	{
		if (verbose == YES)
			RPrintf("No user config file. Using defaults.\n");
		goto GoodExit;		/* For some reason, user has no defaults */
	}

	while (fgets(buf,128,user_config) != NULL)
	{
		/* Override the default SOUP message type for mail */

		if (strnicmp(buf,"MAIL-TYPE",9) == 0)
		{
			mail_type = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (mail_type == m) RPrintf("Mail type: m (Unix Format)\n");
				if (mail_type == M) RPrintf("Mail type: M (MMDF Format)\n");
				if (mail_type == b) RPrintf("Mail type: b (Binary 8-bit)\n");
				if (mail_type == u) RPrintf("Mail type: u (USENET news)\n");
				if (mail_type == i) RPrintf("Mail type: i (Index only)\n");
			}
		}

		/* Override the default protocol */

		if ((strnicmp(buf,"PROTOCOL",8)) == 0)
		{
			strcpy(protocol,extract_parm(buf,delimiter));
			if (verbose == YES)
				RPrintf("Protocol set to %s\n",protocol);
		}

		/* Override the SOUP index type for news */

		if (strnicmp(buf,"NEWS-INDEX",10) == 0)
		{
			news_index = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (news_index == n) RPrintf("News index: n (None)\n");
				if (news_index == C) RPrintf("News index: C (Partial C News)\n");
				if (news_index == c) RPrintf("News index: c (Full C News)\n");
			}
		}

		/* Override the SOUP message type for news */

		if (strnicmp(buf,"NEWS-TYPE",9) == 0)
		{
			news_type = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (news_type == u) RPrintf("news type: u (USENET Format)\n");
        if (news_type == B) RPrintf("news type: B (8-bit Binary)\n");
        if (news_type == i) RPrintf("news type: i (Index only)\n");
			}
		}

		if (strnicmp(buf,"SEND-MAIL",9) == 0)
		{
			send_mail = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (send_mail == YES)
					RPrintf("Send mail set to YES\n");
				else
					RPrintf("Send mail set to NO\n");
			}
		}
		if ((strnicmp(buf,"DELETE-MAIL",11)) == 0)
		{
			delete_mail = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (delete_mail == YES)
					RPrintf("Delete mail set to YES\n");
				else
					RPrintf("Delete mail set to NO\n");
			}
		}
		if ((strnicmp(buf,"SEND-LIST",9)) == 0)
		{
			send_list = convert_to_number(extract_parm(buf,delimiter));
			if (verbose == YES)
			{
				if (send_list == YES)
					RPrintf("Send list set to YES\n");
				else
					RPrintf("Send list set to NO\n");
			}
		}
		if (strnicmp(buf,"COMPRESS",8) == 0)
		{
			strcpy(compress,extract_parm(buf,delimiter));
			if (verbose == YES)
				RPrintf("Compression Method: %s\n",compress);
		}
    if (strnicmp(buf,"USE-WAFJOIN",11) == 0)
    {
      use_wafjoin = convert_to_number(extract_parm(buf,delimiter));
      if (verbose)
      {
        if (use_wafjoin)
          RPrintf("Subscription file: JOIN\n");
        else
          RPrintf("Subscription file: NEWSRC.SLR\n");
      }
    }
  }
	fclose(user_config);

	if (verbose == YES)
		RPrintf("User config file loaded. Validating...\n");

	switch (mail_type) {
		case m	: break;
    case b  : break;
    case M  : break;
    default : sprintf(_slrerr,"unsupported mail-type in %s\n",path);
							goto ErrorExit;
	}

	/* Make sure SOUP index and message types for news are supported */

	switch (news_index) {
		case n	: break;
		case c	: break;
		case C	: break;
    default : sprintf(_slrerr,"unsupported news-index in %s\n",path);
							goto ErrorExit;
	}
	switch (news_type) {
		case u	: break;
    case B  : break;
    case i  : break;
    default : sprintf(_slrerr,"unsupported news-type in %s\n",path);
							goto ErrorExit;
	}

	if (verbose == YES)
    RPrintf("User config file load OK\n\n");

GoodExit:
	return 0;
ErrorExit:
	if (user_config) fclose(user_config);
	return 1;
}

