(*****************************************************************************

  Pointer_
    version 1.0

  This unit contains an interface for the use of the pointer device.  The
    standard pointer device is accessed through the operating system and the
    device driver.

  Purpose:
    To make adding the pointer to any program simple.

  How it works:
    The system handles the pointer operations, or the program can call the
      procedures directly.
    This unit works through the keyboard in a somewhat indirect manner.
      Note that nothing is generated from the pointer when neither buttons
        are active. (Down)  All action takes place when the button is held
        down or the last button condition had changed.
      The following occurs when either of the buttons are pressed.
        First, the position of the pointer is calculated relative to the
          position of the keyboard supplied cursor.
        Next, the interface generates cursor movement commands as it waits
          for the keyboard supplied cursor to move to the pointer's location.
        When or if the cursor and pointer are in alignment, a command is
          generated that indicates the button is in the down position.
        When the button is released, a command is generated that indicates
          the button was released.
        If the pointer is moved while the button is held down, the
          appropriate movement commands are again generated between the up and
          down signals until the pointer and keyboard cursor are again in
          alignment. (For dragging operations)
      Double-clicks are generated separately and may override the up or down
        signal.  Double clicks are recognized in a better manner than before.

  Features:
    This pointer interface is designed to allow mice, trackballs and light
      pens to be used within the program.
    When a program uses the KeyBoard unit, Pointer will link up with it
      automatically and include commands as keystrokes.
    Supports screen manipulation through the KeyBoard unit.
    New version improves the interface for the keyboard by increased the
      possible keyboard alterable commands and providing better movement
      using the secondary button.

  Limitations:
    The system service is assumed to interlink with the standard basic
      input output system pointer device link.
    While both Pointer and JoyStick can work nicely together with Keyboard,
      it isn't recommended.
    To avoid messing the screen up, the pointer should first be turned off
      while writing to the screen, then should be turned back on.
    The keyboard interface routines are rather complex and should not be
      altered unless they are completely understood.

  Versions
    1.0 - Adapted for OS/2 from the Pointer unit.

  Compilers:
    Speed Pascal/2 version 1.5

  Systems:
    OS/2

*****************************************************************************)

Unit Pointer_;

  Interface

    Uses
      DOS,
      Core,
      OS2Supp,
      KeyBoard;

    Const
     { These variable constants are alterable by the program to alter the commands generated. }
      Button1_Up: Byte = Pointer_Button1_Up;
      Button2_Up: Byte = Pointer_Button2_Up;
      Button1_Down: Byte = Pointer_Button1_Down;
      Button2_Down: Byte = Pointer_Button2_Down;
      Button1_Double: Byte = Pointer_Button1_Double;
      Button2_Double: Byte = Pointer_Button2_Double;
     { Amount of hundredths of a second in which to register a double click. }
      Time_Delay: Word = 75;
     { Used to simulate the delay time for a keystroke from the keyboard. }
      Wait_Time: Word = 150;

    Type
     { This is the basic data return record. }
      Pointer_Data_Type = Record
                            X,
                            Y: Word;
                            Row,
                            Column: Word;
                            Button1,
                            Button2,
                            Button3: Boolean;
                          End;

    Var
     { These values determine the last location of the cursor. }
      Last_Row,
      Last_Column: Word;

(***********************************************************

  Flag: Find pointer.

    This flag is set to true if the pointer was detected,
    otherwise it returns false.

***********************************************************)

      Find_Pointer: Boolean;

(*************************************************

  Procedure: Read position and buttons.

    This procedure reads the values of the
    pointer's X and Y screen coordinates and the
    buttons using the pointer device driver
    interface.

*************************************************)

    Procedure Read_Position_And_Buttons( Var X, Y: Word; Var Button1, Button2: Boolean );

(*************************************************

  Procedure: Display pointer.

    This procedure puts the pointer on the screen.

*************************************************)

    Procedure Display_Pointer;

(*************************************************

  Procedure: Hide pointer.

    This procedure removes the graphical pointer
    from the screen.

*************************************************)

    Procedure Hide_Pointer;

(***********************************************************

  Procedure: Set for pointer.

    This procedure sets up the system for the pointer.

***********************************************************)

    Procedure Set_For_Pointer;

(***********************************************************

  Function: Reset pointer.

    This function performs a pointer reset.  It returns true
    if everything is fine, false if it isn't.

***********************************************************)

    Function Reset_Pointer: Boolean;

(***********************************************************

  Function: Check for pointer.

    This function returns true if the pointer is detected.
    It works much quicker than using Reset_Pointer.

***********************************************************)

    Function Check_For_Pointer: Boolean;

(***********************************************************

  Procedure: Read the pointer.

    This procedure reads the status of the pointer and
    returns the results in the supplied data structures.
    X and Y return the value of the pointer in screen pixels
    while Row and Column returns the location in terms of
    cursor coordinates.

***********************************************************)

    Procedure Read_Pointer( Var Data: Pointer_Data_Type );

{-----------------------------------------------------------------------------}

(*****************************************************************************

    Useful Information:
      The pointer device driver is accessed through the interrupt 33h.
        Initialize the pointer - set AX to 0, returns AX <> 0 if successful.
        Turn the pointer on - set AX to 1
        Turn the pointer off - set AX to 2
        Get the pointer position - set AX to 3, returns CX with horizontal
                                   position, DX with the vertical position,
                                   and the buttons in BX
        Move the pointer - set AX to 4, CX to horizontal position, DX to
                           vertical position
        Redefine the cursor - set Ax to 9, BX,CX define the hotspot, ES:DX
                              points to bitmapped data of 64 bytes in length.
                              The first 32 bytes define the mask  ( the cursor
                              is a 16x16 bit map )

*****************************************************************************)

  Implementation

    Var
     { Used to watch the status of the two pointer buttons. }
      Down_Button1,
      Down_Button2: Boolean;
     { Used to store the last two events to watch for double clicking. }
      Previous_Event,
      Previous_Event2: Byte;
     { Used to time a double pointer device click. }
      Old_Time_Last_Event: LongInt;
     { Used for keeping track of the on screen pointer. }
      Last_Attribute,
      Last_New_Attribute: Byte;
      Last_Character: Char;
      Old_Interface: Function: Byte;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read position and buttons.
    As previously defined.

*************************************************)

    Procedure Read_Position_And_Buttons( Var X, Y: Word; Var Button1, Button2: Boolean );
      Var
        Button3: Boolean;
      Begin
        Case Mouse_Buttons of
          1: Mouse_Poll_Queue( Y, X, Button1, Button2, Button3 );
          2: Mouse_Poll_Queue( Y, X, Button1, Button2, Button3 );
          3: Mouse_Poll_Queue( Y, X, Button1, Button3, Button2 );
        End; { Case }
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display_Pointer.
    As previously defined.

*************************************************)

    Procedure Display_Pointer;
      Begin
        Mouse_Display_Pointer;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Hide_Pointer.
    As previously defined.

*************************************************)

    Procedure Hide_Pointer;
      Begin
        Mouse_Hide_Pointer;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Reset pointer.
    As previously defined.

*************************************************)

    Function Reset_Pointer: Boolean;
      Begin
        Reset_Pointer := True;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Check for pointer.
    This function uses the pointer interrupt to
    check and see if the pointer driver is
    installed and working.

*************************************************)

    Function Check_For_Pointer: Boolean;
      Begin
        Check_For_Pointer := Mouse_Available;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Read pointer.
    As previously defined.

*************************************************)

    Procedure Read_Pointer( Var Data: Pointer_Data_Type );
      Begin
        Mouse_Poll_Queue( Data.Row, Data.Column, Data.Button1, Data.Button2, Data.Button3 );
        Data.X := Data.Column;
        Data.Y := Data.Row;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: Move.
    This function sets the Data byte to the
    appropriate value depending on the location
    of the selector cursor.  It returns false if
    the nothing is moved.

*************************************************)

    Function Move( Point: Pointer_Data_Type; Var Data: Byte ): Boolean;
      Var
        Adjust_Row,
        Adjust_Column: Byte;
      Begin
        Inc( Point.Row );
        Inc( Point.Column );
        Adjust_Row := ( Point.Row - Top_Of_Window^ );
        Adjust_Column := ( Point.Column - Left_Of_Window^ );
        If ( Adjust_Row > Cursor_Row )
          then
            Begin
              Data := Pointer_Down;
              Adjust_Amount := ( Adjust_Row - Cursor_Row );
              Move := True;
            End
          else
            If ( Adjust_Row < Cursor_Row )
              then
                Begin
                  Data := Pointer_Up;
                  Adjust_Amount := ( Cursor_Row - Adjust_Row );
                  Move := True;
                End
              else
                If ( Adjust_Column > Cursor_Column_Finish )
                  then
                    Begin
                      Data := Pointer_Right;
                      Adjust_Amount := ( Adjust_Column - Cursor_Column_Finish );
                      Move := True;
                    End
                  else
                    If ( Adjust_Column < Cursor_Column_Start )
                      then
                        Begin
                          Data := Pointer_Left;
                          Adjust_Amount := ( Cursor_Column_Start - Adjust_Column );
                          Move := True;
                        End
                      else
                        Move := False;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Outside.
    This procedure sets the Data byte to the
    appropriate value depending on the location
    outside the current window.

*************************************************)

    Procedure OutSide( Point: Pointer_Data_Type; Var Data: Byte );
      Begin
        If ( Point.Row > Bottom_Of_Window^ )
          then
            Begin
              If ( Point.Row > Succ( Bottom_Of_Window^ ) )
                then
                  Data := Outside_Down
                else
                  Data := On_Frame_Bottom;
            End
          else
            If ( Point.Row < Top_Of_Window^ )
              then
                Begin
                  If ( Point.Row < Pred( Top_Of_Window^ ) )
                    then
                      Data := Outside_Up
                    else
                      Data := On_Frame_Top;
                End
              else
                If ( Point.Column > Right_Of_Window^ )
                  then
                    Begin
                      If ( Point.Column > Succ( Right_Of_Window^ ) )
                        then
                          Data := Outside_Right
                        else
                          Data := On_Frame_Right;
                    End
                  else
                    If ( Point.Column < Left_Of_Window^ )
                      then
                        Begin
                          If ( Point.Column < Pred( Left_Of_Window^ ) )
                            then
                              Data := Outside_Left
                            else
                              Data := On_Frame_Left;
                        End
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Display text pointer.
    While in text mode, this procedure puts a
    block on the screen to simulate the display
    pointer.  It works better than the default
    system because it allows data to be written
    to the screen without turning off the cursor
    first.

*************************************************)

    Procedure Display_Text_Pointer( Row, Column: Byte );
      Var
        Character: Char;
        Attribute: Byte;
      Begin
        Inc( Row );
        Inc( Column );
        Get_Character_From_Screen( Last_Column, Last_Row, Character, Attribute );
        If ( Character = Last_Character ) and ( Attribute = Last_New_Attribute )
          then
            Begin
              If ( Row = Last_Row ) and ( Column = Last_Column )
                then
                  Exit;
              Put_Character_On_Screen( Last_Column, Last_Row, Character, Last_Attribute );
            End;
        Last_Row := Row;
        Last_Column := Column;
        Get_Character_From_Screen( Last_Column, Last_Row, Last_Character, Last_Attribute );
        Last_New_Attribute := Reverse( Last_Attribute );
        Put_Character_On_Screen( Last_Column, Last_Row, Last_Character, Last_New_Attribute );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Remove text pointer.
    This procedure removes the text mode pointer
    from the screen.

*************************************************)

    Procedure Remove_Text_Pointer;
      Var
        Character: Char;
        Attribute: Byte;
      Begin
        Get_Character_From_Screen( Last_Column, Last_Row, Character, Attribute );
        If ( Character = Last_Character )
          then
            Put_Character_On_Screen( Last_Column, Last_Row, Character, Last_Attribute );
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Evaluate pointer.
    This procedure determines if the pointer is
    within the current screen window or outside
    of it.  In either case, it takes the
    appropriate measures.

*************************************************)

    Procedure Evaluate_Pointer( Var Point: Pointer_Data_Type; Var Data: Byte );
      Begin
        Display_Text_Pointer( Point.Row, Point.Column );
        If ( Point.Row >= Top_Of_Window^ ) and ( Point.Row <= Bottom_Of_Window^ ) and
           ( Point.Column >= Left_Of_Window^ ) and ( Point.Column <= Right_Of_Window^ )
          then
            Begin
              If ( Point.Button1 or Point.Button2 )
                then
                  Begin
                    If not Move( Point, Data )
                      then
                        Begin
                          If Point.Button1 and ( not Down_Button1 )
                            then
                              Begin
                                Data := Button1_Down;
                                Down_Button1 := True;
                              End;
                          If Point.Button2 and ( not Down_Button2 )
                            then
                              Begin
                                Data := Button2_Down;
                                Down_Button2 := True;
                              End;
                        End;
                  End
                else
                  Begin
                    If ( Down_Button1 )
                      then
                        Data := Button1_Up;
                    Down_Button1 := False;
                    If ( Down_Button2 )
                      then
                        Data := Button2_Up;
                    Down_Button2 := False;
                  End;
            End
          else
            Begin
              Down_Button1 := False;
              Down_Button2 := False;
              If ( Point.Button1 or Point.Button2 )
                then
                  OutSide( Point, Data );
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Look for clicks.
    This procedure checks for when double clicks

*************************************************)

    Procedure Look_For_Clicks( Var Data: Byte );
      Var
        Hours,
        Minutes,
        Seconds,
        Hundredths: Word;
        New_Time: LongInt;
      Begin
        If ( Data in [ Button1_Up, Button2_Up ] )
          then
            Begin
              GetTime( Hours, Minutes, Seconds, Hundredths );
              New_Time := ( ( Minutes * 6000 ) + ( Seconds * 100 ) + Hundredths );
              If ( Data = Previous_Event2 ) and ( Previous_Event in [ Button1_Down, Button2_Down ] )
                then
                  If ( ( New_Time - Old_Time_Last_Event ) < Time_Delay )
                    then
                      If ( Data = Button1_Up )
                        then
                          Data := Button1_Double
                        else
                          Data := Button2_Double;
              Old_Time_Last_Event := New_Time;
            End;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Function: New interface.
    This function is the routine that allows the
    pointer to generate keystrokes for the
    Keyboard unit.

*************************************************)

    Function New_Interface: Byte;
      Var
        Data: Byte;
        Point: Pointer_Data_Type;
      Begin
        Data := Old_Interface;
        If ( Data = 0 )
          then
            Begin
              Read_Pointer( Point );
              Evaluate_Pointer( Point, Data );
              If ( Data <> 0 )
                then
                  Begin
                    Look_For_Clicks( Data );
                    Previous_Event2 := Previous_Event;
                    Previous_Event := Data;
                  End;
            End;
        New_Interface := Data;
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Procedure: Set for pointer.
    This function is defined for compatibility
    with the DOS version of this unit.

*************************************************)

    Procedure Set_For_Pointer;
      Begin
      End;

{-----------------------------------------------------------------------------}

(*************************************************

  Main initialization section.
    Set the pointer for default settings.
    Hook up the new interface.
    Initialize the click parameters.

*************************************************)

  Begin
    Find_Pointer := Check_For_Pointer;
    If Find_Pointer
      then
        Begin
          Old_Interface := Alternative_Input;
          Alternative_Input := New_Interface;
          Erase_Pointer := Remove_Text_Pointer;
          Previous_Event := 0;
          Old_Time_Last_Event := 0;
          Down_Button1 := False;
          Down_Button2 := False;
        End;
  End.

