{ MATLIB.PAS : Matrix handling routines

  title   : MATLIB
  version : 4.0
  date    : May 15,1994
  author  : J R Ferguson
  language: Turbo Pascal v7.0 (all Targets)
  usage   : unit
}

UNIT MatLib;

INTERFACE

const
  MatMAX    = 8;           { max numer of rows/columns }
  MatEPS    = 1.0E-10;     { computational precision }

type
  MatErrTyp = (MatERROK,   { ok }
               MatERRDIM,  { dimension error   }
               MatERREXP   { negative exponent }
              );
  MatElmTyp = real;

  MatInd    = 0..MatMAX;
  MatTyp    = record
                Nrow,Ncol: MatInd;
                elm      : array[1..MatMAX,1..MatMAX] of MatElmTyp
              end;
  MatPtr    = ^MatTyp;

const
  MatErrCod : MatErrTyp = MatERROK;
  MatErrMsg : array[MatErrTyp] of String[20] =
              ('',
               'Dimension error',
               'Negative exponent');

function MatError: MatErrTyp;
{ Retrieves the current value of MatErrCod,
  then resets MatErrCod to MatERROK.
}

procedure MatDim(var A: MatTyp; n,m: MatInd);
{ Sets dimensions for matrix A to n rows, m columns.
}

procedure MatInsert(var A: MatTyp; row,col: MatInd; x: MatElmTyp);
{ Sets A[row,col] to x.

  The row and col values are range checked against the current dimensions
  of matrix A. If the check fails, MatErrCod will be set to MatERRDIM,
  and matrix A is left unchanged.
}

procedure MatRetrieve(var A: MatTyp; row,col: MatInd; var x: MatElmTyp);
{ Retrieves A[row,col] into x.

  The row and col values are range checked against the current dimensions
  of matrix A. If the check fails, MatErrCod will be set to MatERRDIM,
  and variable x is left unchanged.
}

procedure MatCopy(var source,dest: MatTyp);
{ Copies the source matrix to dest (both dimensions and elements).
}

procedure MatZero(var A: MatTyp);
{ Sets all elements for the current dimensions of A to zero.
}

procedure MatUnify(var A: MatTyp);
{ Unifies the contents of matrix A for the current dimensions.
  A must be a square matrix (Nrow=Ncol).
  The main diagonal elements (having equal indices for row and column)
  are set to 1. All other elements are set to zero.

  If A is not a square matrix, MatErrCod will be set to MatERRDIM, and
  matrix A is left unchanged.
}

procedure MatAdd(var A,B: MatTyp; var result: MatTyp);
{ Matrix addition: sets result = A + B

  If A and B do not have the same dimensions, MatErrCod will be set to
  MatERRDIM and the result matrix is left unchanged.
}

procedure MatSubtract(var A,B: MatTyp; var result: MatTyp);
{ Matrix subtraction: sets result = A - B
  If A and B do not have the same dimensions, MatErrCod will be set to
  MatERRDIM and the result matrix is left unchanged.
}

procedure MatScalarProd(factor: MatElmTyp; var A: MatTyp; var result: MatTyp);
{ Scalar product: result = factor . A
}

procedure MatMatrixProd(var A,B: MatTyp; var result: MatTyp);
{ Matrix product: result = A * B
  A.Ncol must be equal to B.Nrow. The result dimensions will be
  result.Nrow=A.Nrow, result.Ncol=B.NCol.

  If A.Ncol <> B.Nrow, MatErrCod will be set to MatERRDIM and the result
  matrix is left unchanged.
}

procedure MatTranspose(var A: MatTyp; var result: MatTyp);
{ Constructs a result matrix that is the transpose of matrix A
  (rows and columns exchanged).
}

procedure MatPower(var A: MatTyp; exponent: integer; var result: MatTyp);
{ result = A * A * ... A  (exponent times)
  Constructs a result matrix with the same dimensions of square matrix A
  which holds the result of exponent matrix multiplications of A with
  itself. The exponent may not be negative.
  If the exponent is 0, the result matrix will be a unified matrix.
  If the exponent is 1, the result matrix will be the same of A.

  If A is not a square matrix, MatErrCod will be set to MatERRDIM, and
  the result matrix is left unchanged.
  If the exponent is negative, MatErrCod will be set to MatERREXP, and
  the result matrix is left unchanged.
}

procedure MatSsq(var A: MatTyp; var ssq: MatElmTyp);
{ Computes ssq to be the sum of squares of all elements of matrix A,
  which must be a one-row or a one-column matrix.

  If A is not a one-row or a one-column matrix, MatErrCod is set
  to MarERRDIM, and ssq is left unchanged.
}

procedure MatGaussJordan(var A,A1: MatTyp; var det: MatElmTyp);
{ Performs the Gauss Jordan elimination process on square matrix A,
  which sets A to its normalized state. Matrix A1 will be set to the
  inverse of A, and det to the determinant of A.

  If A is not a square matrix, MatErrCod will be set to MatERRDIM, and
  A1 and det are left unchanged.
}

procedure MatInvert(var A:MatTyp; var result:MatTyp; var singular: boolean);
{ Constructs the result matrix the inverse of square matrix A, if
  possible (if which case singular is set to false). If A does not have
  an inverse, singular wil be set to true and the result matrix is left
  unchanged.
  Matrix A is not changed by this procedure.

  If A is not a square matrix, MatErrCod will be set to MatERRDIM, and
  result and singular are left unchanged.
}

procedure MatDet(var A: MatTyp; var det: MatElmTyp);
{ Sets det to be the determinant of square matrix A.
  Matrix A is not changed by this procedure.

  If A is not a square matrix, MatErrCod will be set to MatERRDIM, and
  det is left unchanged.
}


procedure MatSolve(var A,b,x: MatTyp; var singular: boolean);
{ Solves a set of linear equations represented by the matrix
  formula A * b = x.
  If there is a solution, it will be represented by a one-column
  result matrix x and singular will be set to false.
  If there is no solution, singular is set to true, and matrix
  x is left unchanged.

  A must be a square matrix (Nrow=Ncol) and b must be a one-column
  matrix (Ncol=1) with b.Nrow=A.Nrow.
  If one or more of these conditions is not met, MatErrCod is set
  to MarERRDIM, and the result matrix is left unchanged.
}


IMPLEMENTATION

function MatError: MatErrTyp;
begin
  MatError := MatErrCod;
  MatErrCod:= MatERROK;
end;

procedure MatDim(var A: MatTyp; n,m: MatInd);
begin A.Nrow:=n; A.Ncol:=m; end;

procedure MatInsert(var A: MatTyp; row,col: MatInd; x: MatElmTyp);
begin with A do begin
  if (row<1) or (row>Nrow) or (col<1) or (col>Ncol)
  then MatErrCod:= MatERRDIM
  else elm[row,col]:=x;
end end;

procedure MatRetrieve(var A: MatTyp; row,col: MatInd; var x: MatElmTyp);
begin with A do begin
  if (row<1) or (row>Nrow) or (col<1) or (col>Ncol)
  then MatErrCod:= MatERRDIM else x:=elm[row,col];
end end;

procedure MatCopy(var source,dest: MatTyp);
begin dest:=source end;

procedure MatZero(var A: MatTyp);
var i,j: MatInd;
begin with A do begin
  for i:=1 to Nrow do for j:=1 to Ncol do elm[i,j]:=0.0;
end end;

procedure MatUnify(var A: MatTyp);
var i: MatInd;
begin with A do begin
  if Nrow<>Ncol then MatErrCod:= MatERRDIM
  else begin MatZero(A); for i:=1 to Nrow do elm[i,i]:=1.0; end;
end end;

procedure MatAdd(var A,B: MatTyp; var result: MatTyp);
var i,j: MatInd;
begin with A do begin
  if (Nrow<>B.Nrow) or (Ncol<>B.Ncol) then MatErrCod:= MatERRDIM
  else begin
    MatDim(result,Nrow,Ncol);
    for i:=1 to Nrow do for j:=1 to Ncol do
      result.elm[i,j]:=elm[i,j]+B.elm[i,j];
  end
end end;

procedure MatSubtract(var A,B: MatTyp; var result: MatTyp);
var i,j: MatInd;
begin with A do begin
  if (Nrow<>B.Nrow) or (Ncol<>B.Ncol) then MatErrCod:= MatERRDIM
  else begin
    MatDim(result,Nrow,Ncol);
    for i:=1 to Nrow do for j:=1 to Ncol do
      result.elm[i,j]:=elm[i,j]-B.elm[i,j];
  end
end end;

procedure MatScalarProd(factor: MatElmTyp; var A: MatTyp; var result: MatTyp);
var i,j: MatInd;
begin with A do begin
  MatDim(result,Nrow,Ncol);
  for i:=1 to Nrow do for j:=1 to Ncol do result.elm[i,j]:=factor*elm[i,j];
end end;

procedure MatMatrixProd(var A,B: MatTyp; var result: MatTyp);
var i,j,k: MatInd; Tmp: MatPtr;
begin
  if A.Ncol<>B.Nrow then MatErrCod:= MatERRDIM
  else begin
    new(Tmp); MatDim(Tmp^,A.Nrow,B.Ncol); MatZero(Tmp^);
    for i:=1 to A.Nrow do for j:=1 to B.Ncol do for k:=1 to A.Ncol do
      with Tmp^ do elm[i,j]:=elm[i,j]+A.elm[i,k]*B.elm[k,j];
    result:=Tmp^; dispose(Tmp);
  end
end;

procedure MatTranspose(var A: MatTyp; var result: MatTyp);
var i,j: MatInd; Tmp: MatPtr;
begin with A do begin
  new(Tmp); MatDim(Tmp^,Ncol,Nrow);
  for i:=1 to Nrow do for j:=1 to Ncol do Tmp^.elm[j,i]:=elm[i,j];
  result:= Tmp^; dispose(Tmp);
end end;

procedure MatPower(var A: MatTyp; exponent: integer; var result: MatTyp);
var i: integer;
begin with A do begin
  if Nrow<>Ncol then MatErrCod:= MatERRDIM
  else if exponent<0 then MatErrCod:= MatERREXP
  else begin
    MatDim(result,Nrow,Ncol); MatUnify(result);
    for i:=1 to exponent do MatMatrixProd(result,A,result);
  end
end end;

procedure MatSsq(var A: MatTyp; var ssq: MatElmTyp);
var AT, Prod: MatPtr;
begin with A do begin
  new(AT); new(Prod);
  MatTranspose(A,AT^);
  if      Ncol = 1 then MatMatrixProd(AT^,A,Prod^)
  else if Nrow = 1 then MatMatrixProd(A,AT^,Prod^)
  else MatErrCod:= MatERRDIM;
  MatRetrieve(Prod^,1,1,ssq);
  dispose(AT); dispose(Prod);
end end;

procedure MatGaussJordan(var A,A1: MatTyp; var det: MatElmTyp);
var i,r,k,m: MatInd; e: MatElmTyp;
begin with A do begin
  if Nrow<>Ncol then MatErrCod:= MatERRDIM
  else begin
    MatDim(A1,Nrow,Ncol); MatUnify(A1); i:=0; det:=1.0;
    while (i<Ncol) and (abs(det)>=MatEPS) do begin
      i:=i+1;
      m:=i; e:=abs(elm[i,i]);
      for r:=i+1 to Nrow do
        if abs(elm[r,i])>e then begin e:=abs(elm[r,i]); m:=r end;
      if m<>i then begin
        det:=-det;
        for k:=i to Ncol do begin
          e:=elm[i,k]; elm[i,k]:=elm[m,k]; elm[m,k]:=e;
        end;
        for k:=1 to A1.Ncol do begin
          e:=A1.elm[i,k]; A1.elm[i,k]:=A1.elm[m,k]; A1.elm[m,k]:=e;
        end;
      end;
      e:=elm[i,i];
      if abs(e)<MatEPS then det:=0.0
      else begin
        det:=det*e;
        for k:=i to Ncol do elm[i,k] :=elm[i,k]/e;
        for k:=1 to Ncol do A1.elm[i,k]:=A1.elm[i,k]/e;
        for r:=1 to Nrow do if r<>i then begin
          e:=elm[r,i];
          for k:=i to Ncol do elm[r,k]:=elm[r,k]-e*elm[i,k];
          for k:=1 to Ncol do A1.elm[r,k]:=A1.elm[r,k]-e*A1.elm[i,k];
        end
      end
    end
  end
end end;

procedure MatInvert(var A:MatTyp; var result:MatTyp; var singular: boolean);
var det: MatElmTyp; A1: MatPtr;
begin
  new(A1); MatCopy(A,A1^);
  MatGaussJordan(A1^,result,det);
  singular:=abs(det)<MatEPS;
  dispose(A1);
end;

procedure MatDet(var A: MatTyp; var det: MatElmTyp);
var A1,A2: MatPtr;
begin
  new(A1); new(A2); MatCopy(A,A1^);
  MatGaussJordan(A1^,A2^,det);
  dispose(A2); dispose(A1);
end;

procedure MatSolve(var A,b,x: MatTyp; var singular: boolean);
var Ainv: MatPtr;
begin
  new(Ainv); MatInvert(A,Ainv^,singular);
  if (MatErrCod = MatERROK) and not singular then begin
    MatDim(x,b.Nrow,1);
    MatMatrixProd(Ainv^,b,x);
  end;
  dispose(Ainv);
end;

END.
